/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DSM-CC API for client requests
 * @file    dsm_client.h
 * @date    11/12/2013
 * @author  Adam Sturtridge
 */
#ifndef _DSM_CLIENT_H
#define _DSM_CLIENT_H


#include "fs_types.h"
#include "dsm_types.h"


/*---Constant and macro definitions for public use-----------------------------*/

/*
 * Defines for 'loadflags' parameter in DSMCC_ClientLoadObject function
 */
#define LOAD_FLAGS_DEFAULT                      0x0000007F

/* LOAD_FLAGS_CACHE_PRIORITY_MASK is mask bits that specify cache priority
 * With these bits set to zero, the object is loaded from the stream.
 */
#define LOAD_FLAGS_CACHE_PRIORITY_MASK          0x000000FF
#define LOAD_FLAGS_REQUEST_FROM_STREAM          0x00000000

/* When LOAD_FLAGS_REQUEST_FILE_ONLY_BIT is set, the load request will only
 * succeed for objects with OBJECT_TYPE_FILE
 */
#define LOAD_FLAGS_REQUEST_FILE_ONLY_BIT        0x00000100

/* When LOAD_FLAGS_REQUEST_EXISTANCE_BIT is set, content data is NOT provided
 * and only E_FsStatus value is relevant on the return
 */
#define LOAD_FLAGS_REQUEST_EXISTANCE_BIT        0x00000400

/* For stream object type (with events), and LOAD_FLAGS_REQUEST_STREAM_OBJ_XML_BIT set,
 * but without LOAD_FLAGS_REQUEST_EVENT_LIST_BIT being set,
 * the content data is XML format as defined by 'TS 102 809' v1.2.1, section 8.2
 */
#define LOAD_FLAGS_REQUEST_STREAM_OBJ_XML_BIT   0x00002000

/* For stream object type, and LOAD_FLAGS_REQUEST_DEFERRED_SERVICE_BIT set,
 * the content data is S_DvbLocator of the deferred service for that stream
 * Note: ignored if LOAD_FLAGS_REQUEST_EVENT_LIST_BIT is set and stream has events
 */
#define LOAD_FLAGS_REQUEST_DEFERRED_SERVICE_BIT 0x00004000

/* For stream object type (with events), and LOAD_FLAGS_REQUEST_EVENT_LIST_BIT set,
 * the content data is a comma separated list of events on that stream
 */
#define LOAD_FLAGS_REQUEST_EVENT_LIST_BIT       0x00008000

/* LOAD_FLAGS_REQUEST_DIR_FILE_LIST_BIT is depreciated, as this is the default.
 * For a directory object type, the content data is given as a comma separated list
 * of names for each entry in the directory.
 * Note: LOAD_FLAGS_REQUEST_DIRECTORY_LIST_BIT modifies this behaviour
 * and LOAD_FLAGS_REQUEST_SERVICE_CONTEXT_BIT overrides it.
 */
#define LOAD_FLAGS_REQUEST_DIR_FILE_LIST_BIT    0x00000000

/* For a directory object type, LOAD_FLAGS_REQUEST_DIRECTORY_LIST_BIT specifies that
 * each entry in the directory list should start with a byte that gives its type/kind,
 * followed by the name and then a separating comma or null or last entry.
 * For the type/kind byte see values in E_DsmObjectType.
 * The size of this content data is does not include the terminating null byte.
 * Note: ignored if LOAD_FLAGS_REQUEST_SERVICE_CONTEXT_BIT is set!
 */
#define LOAD_FLAGS_REQUEST_DIRECTORY_LIST_BIT   0x00040000

/* For a directory object type, and LOAD_FLAGS_REQUEST_SERVICE_CONTEXT_BIT set,
 * the content data points to service context data for service ID
 * Service ID value must be given in content 'user_data' field.
 */
#define LOAD_FLAGS_REQUEST_SERVICE_CONTEXT_BIT  0x00080000

/*---Enumerations for public use-----------------------------------------------*/

typedef enum
{
   OBJECT_TYPE_UNKNOWN = 0x30,   /* ascii '0' */
   OBJECT_TYPE_FILE,
   OBJECT_TYPE_DIRECTORY,
   OBJECT_TYPE_ROOT_DIR,
   OBJECT_TYPE_STREAM,
   OBJECT_TYPE_STREAM_WITH_EVENTS
} E_DsmObjectType;


/*---Global type defs for public use-------------------------------------------*/

typedef F_FS_CALLBACK F_DSM_CBLOAD;

/**
 * @brief Callback function to notify that a DSM-CC stream event has triggered
 *        The event name is given, because DSMCC_ClientEventSubscribeName can
 *        subscribe to all events on a URL, and then this specifies which event
 *        occured.
 * @param userData      User data passed to DSMCC_ClientEventSubscribe...
 * @param name          Name of event (null terminated string)
 * @param dataLen       Length of event data
 * @param dataPtr       Pointer to event data (only valid in callback function)
 */
typedef void (*F_DSM_EVENT)( void *userData, U8BIT *name,
                             U32BIT dataLen, U8BIT *dataPtr );

/**
 * @brief Callback function to notify client of result for request of object type
 * @param userData User data passed to DSMCC_ClientObjectType
 * @param type Type of object - set to OBJECT_TYPE_UNKNOWN if an error occured
 */
typedef void (*F_DSM_TYPE)( void *userData, E_DsmObjectType type );

/*---Global Function prototypes for public use---------------------------------*/


/**
 * @brief   Perform a file load request which is executed synchronously or
 *          asynchronously. See load flags above for modifying requests.
 *
 *          Synchronous behaviour:
 *           If DSM-CC is able to determine that the file does not exist, the
 *           function returns with the FS_STATUS_ERROR.
 *
 *           If DSM-CC has the requested file in cache, it can read the file
 *           contents and pass a pointer to it in the pFileData return parameter,
 *           along with the file length in the pFileSize returned parameter. Once
 *           the contents of the file has been used, resource allocated to it
 *           must be released by calling the dsmFDataAck function. The function
 *           return value will be set to FS_STATUS_OK.
 *
 *          Asynchronous behaviour:
 *           If DSM-CC does not have the file in cache it will perform an
 *           asynchronous load. This is indicated by returning FS_STATUS_PENDING.
 *           When the load is completed, the 'cb_func' is called
 *
 *          If the path parameter points to a stream object within the DSM-CC
 *          carousel this will be loaded. The referenced multiplex will be located
 *          and a reference to this will be passed back to the MHEG-5 engine.
 *
 * @param   dsmctrl DSMCC controling Instance
 * @param   path Path of Object (directory, file or stream)
 * @param   flags Load flags containing cache priority, and other control bits
 * @param   cb_func Call-back Function
 * @param   pContent Pointer to the file content details.
 *          Note: When content data for object is no longer required, the destroy
 *          function in S_CONTENT must be called to free any allocated resource.
 * @return  E_FsStatus
 *          FS_STATUS_OK
 *          The Object has been sucessfully loaded
 */
E_FsStatus DSMCC_ClientLoadObject( H_DsmControl dsmctrl,
   U8BIT *path, U32BIT flags,
   F_DSM_CBLOAD cb_func, S_CONTENT *pContent );

/**
 * @brief   Request object type for given path. This may be executed synchronously
 *          or asynchronously - see comment for DSMCC_ClientLoadObject()
 * @param   dsmctrl DSMCC controling Instance
 * @param   path Path of Object (directory, file or stream)
 * @param   cbfunc Call-back Function
 * @param   userData User data to be passed back to client in callback function.
 * @param   pType  Pointer to object type - return value is valid when FS_STATUS_OK.
 * @return  E_FsStatus, FS_STATUS_OK - The Object has been sucessfully found
 */
E_FsStatus DSMCC_ClientObjectType( H_DsmControl dsmctrl, U8BIT *path,
   F_DSM_TYPE cbfunc, void *userData, E_DsmObjectType *pType );

/**
 * @brief   Provide a hint that the specified file may be required in the future.
 *          If possible this should be loaded into the DSM-CC cache.
 * @param   dsmctrl DSMCC Instance
 * @param   path Path of Object (directory, file or stream)
 * @return  void
 */
void DSMCC_ClientPreloadHint( H_DsmControl dsmctrl, U8BIT *path );

/**
 * @brief   Client request to subscribe to DSMCC Stream Event, specified by name.
 *          DSMCC notifies when the named stream event occurs, by calling cb_func
 *          (a callback function defined by the Client).
 *          Before calling this function the Client must request that the stream object of
 *          interest be loaded using DSMCC_ClientLoadObject() function. This action will
 *          deliver a handle to the object stream - this is passed as the streamObject
 *          argument to this function. Do not have to wait until load is completed before
 *          subscribing.
 * @param   dsmctrl DSMCC Instance
 * @param   streamObject DSMCCs file system handle
 * @param   eventName Name of stream event required by the Client.
 * @param   cb_func Callback funcrtion for client.
 * @param   userData Optional data to be passed back to client. It is stored
 *          and returned unchanged.
 * @param   pEventHandle Handle to the stream event returned by subscribe.
 *          The caller may use this handle when calling DSMCC_ClientEventUnsubscribe
 * @return  E_FsStatus
 *          FS_STATUS_OK
 *          The event has been successfully subscribed to.
 */
E_FsStatus DSMCC_ClientEventSubscribeName( H_DsmControl dsmctrl,
   FS_HANDLE streamObject, U8BIT *eventName,
   F_DSM_EVENT cb_func, void *userData,
   H_DsmEvent *pEventHandle );

/**
 * @brief   Client request to subscribe to DSMCC Stream Event, specified by event ID
 *          and association tag of stream object.
 *          DSMCC notifies when the stream event occurs, by calling cb_func
 *          (a callback function defined by the Client).
 * @param   dsmctrl DSMCC Instance
 * @param   associationTag association tag.
 * @param   evemntId Id of stream event.
 * @param   cb_func Callback funcrtion for client.
 * @param   userData Optional data to be passed back to client. It is stored
 *          and returned unchanged.
 * @param   pEventHandle Handle to the stream event returned by subscribe.
 *          The caller may use this handle when calling DSMCC_ClientEventUnsubscribe
 * @return  E_FsStatus
 *          FS_STATUS_OK
 *          The event has been successfully subscribed to.
 */
E_FsStatus DSMCC_ClientEventSubscribeId( H_DsmControl dsmctrl,
   U16BIT associationTag, U16BIT eventId,
   F_DSM_EVENT cb_func, void *userData,
   H_DsmEvent *pEventHandle );

/**
 * @brief   Client request to UN-subscribe to DSMCC Stream Event.
 * @param   dsmctrl DSMCC Instance
 * @param   event Handle to previously subscribed event
 * @return  void
 */
void DSMCC_ClientEventUnsubscribe(H_DsmControl dsmctrl, H_DsmEvent event);

/**
 * @brief   Client request to retrieve MHEG5 File system acceleration file groups
 *          The client must release the array of group data, by calling
 *          DSMCC_ClientUnloadFileGroups, when finished with the data
 * @param   dsmctrl DSMCC Instance
 * @param   hOC Handle to carousel
 * @param   total Pointer to return number of file groups in the carousel info
 * @param   pGroups Pointer to return an array of file group data.
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_ClientLoadFileGroups( H_DsmControl dsmctrl, H_ObjCarousel hOC,
   U16BIT *total, S_CarouselInfoFileGroup **pGroups );

/**
 * @brief   Client request to release file group data returned from
 *          DSMCC_ClientLoadFileGroups
 * @param   dsmctrl DSMCC Instance
 * @param   hOC Handle to carousel
 * @param   groups Array of file group data.
 * @return  void
 */
void DSMCC_ClientUnloadFileGroups( H_DsmControl dsmctrl, H_ObjCarousel hOC,
   S_CarouselInfoFileGroup *groups );




#endif /* _VGR_DSMCLIENT_H */

