/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DSM-CC API functions to control DSM-CC instance
 * @file    dsm_control.h
 * @date    23/01/2013
 * @author  Adam Sturtridge
 */

#ifndef _DSMAPI_H
#define _DSMAPI_H

#include "siqfuncs.h"
#include "dmxtypes.h"
#include "dsm_types.h"
#include "stdfuncs.h"

/*---Constant and macro definitions for public use-----------------------------*/

/* Macro for API backward compatibiliy */
#define obtainSiDirect  controlFlags

/* Following flags to be use in 'controlFlags' */
/* Set GET_PMT_DIRECT_FROM_HW_FLAG when DSMCC should get PMT data from hardware,
 * rather than notifications from DVB stack using STB_SIRegisterPmtObserver */
#define GET_PMT_DIRECT_FROM_HW_FLAG    (1 << 0)

/* Set DISABLE_AUTO_CACHING_FLAG when DSMCC should not automaticaaly download
 * and cache data blocks of DSMCC modules */
#define DISABLE_AUTO_CACHING_FLAG      (2 << 0)

/*---Enumerations for public use-----------------------------------------------*/

/*---Global type defs for public use-------------------------------------------*/


typedef void (*F_CarouselLoad)( H_ObjCarousel hOC, E_OCLoadStatus result, U32BIT carouselId );


typedef struct s_DsmccConfig
{
   /* DSMCC task priority value - passed to STB_OSCreateTask
    * The specified task priority value is used by the DSMCC task
    */
   U32BIT taskPriority;

   /* Size of internal section buffers memory pool in 4096 byte blocks. This is
    * used to provide data directly to DSM-CC instance. The task priority has a
    * direct influence over section buffer size. Lower task priority means larger
    * buffer pool requirements.
    * Minimum value: 100  (400 Kbytes)
    * Recommended:   200
    */
   U8BIT sectionBuffPoolSize;

   /* Size of section buffer manager cache in 64K byte blocks. This is separate
    * to sectionBuffPoolSize and this memory is used to store section data not
    * immediately required by DSM-CC instance. This facilitates faster DSM-CC file
    * loading - particularly when the number of hardware section filters is limited.
    * Minimum value: 0
    * Suggested value: 16
    */
   U8BIT sectionBuffCacheSize;

   /* It is manditory to set memAlloc and memFree */
   F_MemAlloc memAlloc;
   F_MemFree memFree;

   /* parsePmtDone is optional and may be set to NULL */
   F_PmtParseInit parsePmtInit;

   /* It is manditory to set at least one of the following
   * either parseDataBroadcastId or parseSsuSelectorBytes */
   F_ParseDataBcastId parseDataBroadcastId;
   F_ParseSsuSelectorBytes parseSsuSelectorBytes;

   /* parsePmtDone is optional and may be set to NULL */
   F_PmtParseDone parsePmtDone;

   /* It is manditory to set either 'carouselLoad' or the four functions in 'F_SsuConfig' */
   F_CarouselLoad carouselLoad;
   S_SsuConfig ssuFuncs;

   /* notifyAitInfo is optional and may be set to NULL */
   F_NotifyAitInfo notifyAitInfo;

   U32BIT controlFlags;
} S_DsmccConfig;


/*---Global Function prototypes for public use---------------------------------*/

/**
 * @brief   Open DSM-CC and create controling instance.
 * @param   config Configuration of control instance.
 * @return  H_DsmControl instance handle
 */
H_DsmControl DSMCC_Open( S_DsmccConfig *config );

/**
 * @brief   Close DSM-CC controling instance, and destroy all resources.
 * @param   dsmctrl DSM control instance handle
 * @return  void
 */
void DSMCC_Close( H_DsmControl instance );

/**
 * @brief   The function is depreciated. It starts DSM-CC controlling instance,
 *          and will start search for a boot carousel on service specified by
 *          locator. The search depends on 'boot_carousel_id' parameter.
 *          When boot carousel id is set to INVALID_CAROUSEL_ID, DSM-CC will
 *          attempt to start on a carousel which the client recognises in callback
 *          function 'parseDataBroadcastId'. Even when the PMT does not contain
 *          a recognised data broadcast id, DSMCC_Start() will return TRUE, and the
 *          DSM-CC will keep trying to load a boot carousel.
 *          When boot carousel id is set to a specific value or UNKNOWN_CAROUSEL_ID
 *          and a carousel of the required ID is not present, either DSMCC_Start()
 *          will return FALSE or 'carouselLoad' callback will be passed a result of
 *          OC_LOAD_ABORTED_ERROR.
 *          When boot carousel id is set to UNKNOWN_CAROUSEL_ID, DSM-CC will attempt
 *          to start on whatever carousel is signalled in PMT for that service.
 * @param   instance DSM control instance handle
 * @param   locator DVB location of service.
 * @param   boot_carousel_id carousel id - INVALID_CAROUSEL_ID,
 *          or UNKNOWN_CAROUSEL_ID, or a specific value.
 * @param   DMXREF dmxref demux reference to be used in section filter API
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_Start( H_DsmControl instance, S_DvbLocator *locator,
   U32BIT boot_carousel_id, DMXREF dmxref );

/**
 * @brief   Inform DSMCC instance of the tuned transport details. This must be
 *          called before calling DSMCC_StartFs() or DSMCC_StartSsu()
 * @param   instance DSM control instance handle
 * @param   onet_id Original Network Id.
 * @param   tran_id Transport ID for the transport stream.
 * @param   DMXREF dmxref demux reference to be used in section filter API
 * @return  BOOLEAN - TRUE if success
 */
void DSMCC_SetTunedTransportInfo( H_DsmControl instance, U16BIT onet_id,
   U16BIT tran_id, DMXREF dmxref );

/**
 * @brief   Start DSM-CC controlling instance. This must follow call to
 *          DSMCC_SetTunedTransportInfo.
 *          The function will start search for an Object Carousel on the specified
 *          service. The search depends on the value given 'carousel_id' parameter.
 *          When carousel id is set to INVALID_CAROUSEL_ID, DSM-CC will attempt
 *          to start on a carousel which the client recognises in callback
 *          function 'parseDataBroadcastId'. Even when the PMT does not contain
 *          a recognised data broadcast id, this function will return TRUE, and the
 *          DSM-CC will keep searching.
 *          When the carousel id is set to a specific value or UNKNOWN_CAROUSEL_ID
 *          and a carousel of the required ID is not present, either this function
 *          will return FALSE or 'carouselLoad' callback will be passed a result of
 *          OC_LOAD_ABORTED_ERROR.
 *          When carousel id is set to UNKNOWN_CAROUSEL_ID, DSM-CC will attempt
 *          to start on whatever carousel is signalled in PMT for that service.
 * @param   instance DSM control instance handle
 * @param   service_id Service Id.
 * @param   carousel_id Carousel Id
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_StartFs( H_DsmControl instance, U16BIT service_id, U32BIT carousel_id );

/**
 * @brief   Start DSM-CC controling instance. This must follow call to
 *          DSMCC_SetTunedTransportInfo.
 *          The function will start loading Update Carousel for SSU on specified service
 * @param   instance DSM control instance handle
 * @param   service_id Service Id.
 * @param   oui Manufacturer's OUI
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_StartSsu( H_DsmControl instance, U16BIT service_id, U32BIT oui );

/**
 * @brief   Start DSM-CC controling instance. This must follow call to
 *          DSMCC_SetTunedTransportInfo. The function will start loading Update Carousel
 *          for SSU on a single specified PID.
 * @param   instance DSM control instance handle
 * @param   pid PID on which to find all SSU data
 * @param   oui Manufacturer's OUI
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_StartSsuWithPid( H_DsmControl instance, U16BIT pid, U32BIT oui );

/**
 * @brief   Stop DSM-CC controling instance. Causes all carousels to unload
 * @param   dsmctrl DSM control instance handle
 * @param   mode Mode of operation when unloading
 * @return  void
 */
void DSMCC_Stop( H_DsmControl instance, E_DsmRstMode mode );

/**
 * @brief   Reboot DSM-CC controling instance. Causes all carousels to unload and
 *          initiate search for boot carousel.
 * @param   instance DSM control instance handle
 * @return  void
 */
void DSMCC_Reboot( H_DsmControl instance );

/**
 * @brief   Get currently loaded carousel handle
 * @param   dsmctrl DSM control instance handle
 * @return  carousel handle - NULL if none loaded
 */
H_ObjCarousel DSMCC_CurrentCarousel( H_DsmControl dsmctrl );

/**
 * @brief   Get currently loaded carousel ID
 * @param   dsmctrl DSM control instance handle
 * @return  carousel id - INVALID_CAROUSEL_ID if none loaded
 */
U32BIT DSMCC_CurrentCarouselId( H_DsmControl dsmctrl );

/**
 * @brief   Set current carousel.
 *          This does not unload any previously loaded carousels
 * @param   dsmctrl DSM control instance handle
 * @param   hOC Handle to carousel
 * @return  BOOLEAN - TRUE if success (i.e. had valid carousel handle)
 */
BOOLEAN DSMCC_SetCurrentCarousel(H_DsmControl dsmctrl, H_ObjCarousel hOC);

/**
 * @brief   Unload specified Carousel
 * @param   dsmctrl DSM control instance handle
 * @param   hOC Handle to carousel
 * @param   mode Mode of operation when unloading
 * @return  void
 */
void DSMCC_UnloadCarousel(H_DsmControl dsmctrl, H_ObjCarousel hOC, E_DsmRstMode mode);

/**
 * @brief   Clear cached PMT for particular service
 * @param   dsmctrl DSM control instance handle.
 * @param   serviceId Service whose PMT needs to be cleared
 * @return  void
 */
void DSMCC_SiqCacheClearPmt( H_DsmControl dsmctrl, U16BIT serviceId );

/**
 * @brief   Returns version major.minor.patch as string
 * @return  a version string
 */
const char* DSMCC_VersionString(void);

/**
 * @brief   Find DSMCC instance that has beed started on specified service
 * @param   serviceId Service on which DSMCC is started
 * @param   dmxref demux reference used for HW section filtering
 * @return  H_DsmControl instance handle, or NULL if not found
 */
H_DsmControl DSMCC_FindInstance( U16BIT serviceId, DMXREF dmxref  );

/**
 * @brief   Set limit of DSMCC dynamic memory usage for file system caching
 *          This function overrides the default size is 6 MB.
 *          If this size is less than the currently allocated memory size,
 *          then it will release file system data until limit is reached - but
 *          only releasing data not currently in used by the client.
 * @param   dsmctrl DSM control instance handle.
 * @param   cacheSize Size in bytes (minimum of 1 MB)
 * @return  void
 */
void DSMCC_SetFileSystemCacheLimit( H_DsmControl dsmctrl, U32BIT cacheSize );

/**
 * @brief   Unload SSU module. This is intended to be called once the client
 *          has finished saving the data for this module.
 * @param   dsmctrl DSM control instance handle.
 * @param   hCarousel Handle to the DSMCC Update Carousel
 * @param   moduleRef Reference to a module
 * @return  void
 */
void DSMCC_SsuUnloadModule( H_DsmControl dsmctrl, H_DsmCarousel hCarousel,
   U32BIT moduleRef );


#endif /*_DSMAPI_H */
