/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DSM-CC types used for client control
 * @file    dsm_types.h
 * @date    23/01/2014
 * @author  Adam Sturtridge
 */

#ifndef _DSM_TYPES_H
#define _DSM_TYPES_H

#include "dvblocator.h"

#define INVALID_CAROUSEL_ID      0xFFFFFFFF
#define UNKNOWN_CAROUSEL_ID      0xFFFFFFFE

/* For compilation backward compatability, define OC_LOAD_FILE_GROUP_CHANGE */
#define OC_LOAD_FILE_GROUP_CHANGE OC_FILE_GROUP_LIST_CHANGE

/* Enum used to specify type of behaviour in unload type operation.
 * The unload operation (normally on carousel) can cause any pending
 * object load requests to either:
 *   1. forced to abort, or
 *   2. kept pending until a new carousel is loaded.
 *  This enum is used in relevant API functions.
 */
typedef enum
{
   RST_MODE_FORCE,
   RST_MODE_PENDING,
} E_DsmRstMode;

/* Enum used to report Object Carousel load status */
typedef enum
{
   /*0*/ OC_LOAD_UNINITIATED = 0,
   /*1*/ OC_LOAD_IN_PROGRESS,
   /*2*/ OC_LOAD_BOOTED,
   /*3*/ OC_LOAD_COMPLETED,
   /*4*/ OC_LOAD_ABORTED_TIMEOUT,
   /*5*/ OC_LOAD_ABORTED_PATH_ERROR,
   /*6*/ OC_LOAD_ABORTED_ERROR,
   /*7*/ OC_LOAD_ABORTED_UNLOAD,
   /*8*/ OC_FILE_GROUP_LIST_LOADED, /* Loaded carousel has file group items - signalled just before OC_LOAD_COMPLETED */
   /*9*/ OC_FILE_GROUP_LIST_CHANGE, /* file group change includes added and/or deleted items */
   /*10*/OC_FILE_GROUP_VERS_CHANGE  /* only a version change on existing file group(s) */
} E_OCLoadStatus;

/* Enum used to report System Software Update load status */
typedef enum
{
   /*  No recogised SSU download */
   SSU_NOT_AVAILABLE,

   /*  Recogised SSU download ready - about to start */
   SSU_CRSL_READY,

   /*  Starting to download a module */
   SSU_LOAD_STARTING,

   /*  downloading a module */
   SSU_LOAD_RECEIVING,

   /*  Completed download of a module */
   SSU_LOAD_COMPLETE,

   /*  Aborted download of a module */
   SSU_LOAD_ABORTED,

   /*  Completed SSU download */
   SSU_CRSL_DONE,

   /*  Aborted SSU download */
   SSU_CRSL_ABORT,

   /*  Unloading SSU carousel */
   SSU_CRSL_UNLOAD
} E_UCLoadStatus;

typedef enum
{
   MTYP_EXECUTABLE,
   MTYP_MEMORY_MAP,
   MTYP_DATA_MODULE
} E_ModuleType;

/* Carousel SRG User Info for File Groups - available once carousel is loaded
 * - used by Australia/South Africa MHEG profile */
typedef struct
{
   U16BIT owner_id;
   U16BIT group_id;
   U8BIT group_priority;
   U8BIT use_flags;
   U8BIT receiver_profile;
   U8BIT group_version;
   U8BIT *location;
} S_CarouselInfoFileGroup;

typedef struct s_SsuModelVersion
{
   U16BIT hw_model;
   U16BIT hw_version;
   U16BIT sw_model;
   U16BIT sw_version;
} S_SsuModelVersion;

typedef U32BIT H_DsmEvent;
typedef struct s_DsmCarousel *H_ObjCarousel;
typedef struct s_DsmCarousel *H_DsmCarousel;
typedef struct s_DsmControl *H_DsmControl;
typedef void* H_UsrRef;
typedef union
{
   H_UsrRef ur;
   U32BIT id;
} U_StatusRef;

/**
 * @brief   SSU callback to inform client of status
 * @param   hCarousel Handle to the DSMCC Update Carousel
 * @param   status Current load status
 * @param   sr Meaning depends on status value:
 *             SSU_NOT_AVAILABLE - sr.id is OUI
 *             SSU_CRSL_READY    - sr.id is Group ID
 *             SSU_LOAD_STARTING - sr.ur is H_UsrRef
 *             SSU_LOAD_COMPLETE - sr.ur is H_UsrRef
 *             SSU_LOAD_ABORTED  - sr.ur is H_UsrRef
 *             SSU_CRSL_DONE     - sr.id is Group ID
 *             SSU_CRSL_ABORT    - sr.id is OUI
 *             SSU_CRSL_UNLOAD   - sr.id is OUI
 * @param   data Extra data
 * @return  void
 */
typedef void (*F_SsuStatus)( H_DsmCarousel hCarousel, E_UCLoadStatus status,
   U_StatusRef sr, U32BIT data );

/**
 * @brief   SSU callback to ask client whether the download details matches the product
 *          and is a valid/required sofware upgrade.
 * @param   grp_id Group ID (Bits: 1-15 are Download ID)
 * @param   grp_size Total size of group (i.e. size of all modules for this image data)
 * @param   swv Model and Version details for this group
 * @return  BOOLEAN - TRUE if this is required
 */
typedef BOOLEAN (*F_SsuWanted)( U32BIT grp_id, U32BIT grp_size, S_SsuModelVersion *smv );

/**
 * @brief   SSU callback to ask client whether if client wants to download data connected
 *          to this module. On returning TRUE, this is always followed by calls to
 *          F_SsuWantModuleData() and F_SsuSaveModuleData().
 *          Another call to this function will only occur after SSU_LOAD_COMPLETE event
 * @param   mtyp Type of module download
 * @param   name Name of module download
 * @return  H_UsrRef - NULL if module data is not required
 */
typedef H_UsrRef (*F_SsuStartModule)( E_ModuleType mtyp, U8BIT *name );

/**
 * @brief   SSU callback function supplied by client which DSM-CC calls to check
 *          whether client requires this module. It is expected that the client
 *          may keep previously downloaded modules in NVM, and therefore avoid
 *          repeating downloads if power cycle interrupted a previously started update.
 * @param   usrRef Reference returned by F_SsuStartModule()
 * @param   moduleRef Reference to module data
 * @param   offset Offset for the module data
 * @param   size Size of this module data
 * @param   crc CRC for of this module data
 * @return  BOOLEAN - TRUE if module is required
 */
typedef BOOLEAN (*F_SsuWantModuleData)( H_UsrRef usrRef, U32BIT moduleRef,
   U32BIT offset, U32BIT size, U32BIT crc );

/**
 * @brief   SSU callback function supplied by client - called when download of a
 *          module has completed. The data will remain in memory after returning
 *          from this funcrtion and until the client calls DSMCC_SsuUnloadModule
 * @param   usrRef Reference returned by F_SsuStartModule()
 * @param   moduleRef Reference to module data
 * @param   offset Offset for the module data
 * @param   size Size of the module data
 * @param   data Pointer to the data for this module data
 * @return  void
 */
typedef void (*F_SsuSaveModuleData)( H_UsrRef usrRef, U32BIT moduleRef,
   U32BIT offset, U32BIT size, U8BIT *data );

/* Convenient struct to hold SSU callbacks */
typedef struct s_SsuConfig
{
   F_SsuStatus status;
   F_SsuWanted wanted;
   F_SsuStartModule startModule;
   F_SsuWantModuleData wantModuleData;
   F_SsuSaveModuleData saveModuleData;
} S_SsuConfig;


#endif /*_DSM_TYPES_H*/
