/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Service Information Query: external function prototypes
 * @file    siqfuncs.h
 * @date    23/01/2013
 * @author  Adam Sturtridge
 */

#ifndef _SIQFUNCS_H
#define _SIQFUNCS_H

typedef void *H_PmtRef;

/**
 * @brief   Inform the DSM-CC client that parsing of PMT data is about to start
 *          The return value is passed to subsequent PMT parsing call back
 *          functions: i.e. F_ParseDataBcastId and F_PmtParseDone
 * @param   U16BIT     serviceId    Service id of the PMT
 * @return  H_PmtRef  - handle or user data of client
 */
typedef H_PmtRef (*F_PmtParseInit)( U16BIT serviceId );

/**
 * @brief   Parse data broadcast id descriptor for recognised application(s)
 *          SIQ wants to know whether this PMT stream can be used to launch
 *          a DSM-CC boot carousel. When there is more than one choice, SIQ
 *          chooses the PMT stream that had the highest preference value
 *          returned by this function.
 *          Note that this may be called multiple times for a PMT stream,
 *          and always called from within SIQ_ProcessPmt().
 *          Service ID and carousel ID are provided for convenience and are
 *          not required for a basic implementation of this function.
 * @param   pmtref Reference supplied by F_PmtParseInit. If F_PmtParseInit
 *          is not supplied to DSMCC_Open, this value is equal to service ID.
 * @param   carouselId Carousel associated with this descriptor
 * @param   dbcPtr Pointer to data broadcast id descriptor
 * @param   dbcLen Length of data broadcast id descriptor
 * @return  U32BIT - Preference value for this data broadcast id and
 *          associated application type. Zero means that data broadcast id or
 *          application type is not recognised, and SIQ will ignore it.
 */
typedef U32BIT (*F_ParseDataBcastId)( H_PmtRef pmtref, U32BIT carouselId,
                                      U8BIT *dbcPtr, U8BIT dbcLen );

/**
 * @brief   Inform the DSM-CC client that parsing of PMT data has completed
 * @param   pmtref Reference supplied by F_PmtParseInit. If F_PmtParseInit
 *          is not supplied to DSMCC_Open, this value is equal to service ID.
 * @return  void
 */
typedef void (*F_PmtParseDone)( H_PmtRef pmtref );

/**
 * @brief   Parse "Selector Bytes" contained in "System Software Update Info"
 *          of PMT's data broadcast Id decriptor (for SSU).
 *          This function is only called where OUI matches the value given in
 *          DSMCC_StartSsu().
 *          This function should return TRUE if the client wishes DSMCC to load
 *          the associated Update Carousel.
 * @param   selPtr Pointer to selector bytes
 * @param   selLen Length of selector bytes
 * @return  BOOLEAN - TRUE if selector bytes has the required product details
 */
typedef BOOLEAN (*F_ParseSsuSelectorBytes)( U8BIT *selPtr, U8BIT selLen );

/**
 * @brief   Callback to inform of first or changed AIT info in PMT
 * @param   serviceId Service id.
 * @param   type Type of AIT application
 * @param   pid PID containing AIT
 * @param   version Current AIT version
 * @return  void
 */
typedef void (*F_NotifyAitInfo)( U16BIT serviceId, U16BIT appType,
                                 U16BIT pid, U8BIT version );

#endif //_SIQFUNCS_H
