/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Interface to Platform Section Filtering
 * @file    stb_filter.h
 * @date    23/01/2013
 * @author  Adam Sturtridge
 */


#if !defined(_STB_FILTER_H)
#define       _STB_FILTER_H

#include "dmxtypes.h"

#if defined(__cplusplus)
extern "C" {
#endif

typedef void (*F_FILTER_CALLBACK)(DMXREF dmx, U16BIT bytes, PIDFILT pfid);

/**
 * @brief   Get a New PID Filter & Setup Associated Buffer and Callback
 *          Function Address.
 * @param   dmx Required Decode Refernce.
 * @param   pid Required PID to Demux.
 * @param   func_ptr User's Interrupt Procedure Function Address.
 * @return  New PID filter identifier or invalid id.
 */
PIDFILT STB_DMXGrabPIDFilter(DMXREF dmx, U16BIT pid, F_FILTER_CALLBACK func);

/**
 * @brief   Copies a filtered section to caller's buffer
 * @param   dmx the demux to use
 * @param   buffer the caller's buffer
 * @param   size the size of the caller's buffer
 * @param   pfid the handle of the PID filter to read from
 * @return  TRUE copied ok
 * @return  FALSE no data to copy
 */
BOOLEAN STB_DMXCopyPIDFilterSect(DMXREF dmx, U8BIT *buffer, U16BIT size, PIDFILT pfid);

/**
 * @brief   Skips (discards) a section in the PID filter buffer
 * @param   dmx the demux ref for the filter
 * @param   pfid the PID filter handle
 */
void STB_DMXSkipPIDFilterSect(DMXREF dmx, PIDFILT pfid);

/**
 * @brief   Allocated a new section filter on the specified PID filter
 * @param   dmx the demux ref to use
 * @param   pfid the PID filter to assign the section filter to
 * @return  The section filter handle
 */
SECFILT STB_DMXGrabSectFilter(DMXREF dmx, PIDFILT pfid);

/**
 * @brief   Configures a match and mask for a specified section filter
 * @param   dmx the demux ref of the section filter
 * @param   sfilt_id the handle of the section filter
 * @param   match_ptr pointer to the match bytes
 * @param   mask_ptr pointer to the mask bytes
 * @param   not_equal_byte_index the byte position for a not equal compare
 * @param   crc TRUE to use CRC checking FALSE to ignore
 */
void STB_DMXSetupSectFilter(DMXREF dmx, SECFILT sfid, U8BIT *match_ptr, U8BIT *mask_ptr,
   U8BIT not_equal_byte_index, BOOLEAN crc);

/**
 * @brief   Start Specified PID Filter Collecting Data.
 * @param   dmx Required Decode Reference.
 * @param   pfilter_id Required PID Filter Identifier.
 */
void STB_DMXStartPIDFilter(DMXREF dmx, PIDFILT pfid);

/**
 * @brief   Stop Specified PID Filter Collecting Data.
 * @param   dmx Required Decode Reference.
 * @param   pfilter_id Required PID Filter Identifier.
 */
void STB_DMXStopPIDFilter(DMXREF dmx, PIDFILT pfid);

/**
 * @brief   Releases a previously allocated PID filter
 * @param   dmx the demux ref of the filter
 * @param   pfid the handle of the filter
 */
void STB_DMXReleasePIDFilter(DMXREF dmx, PIDFILT pfid);

/**
 * @brief   Releases a previously allocated section filter
 * @param   dmx the demux ref of the filter
 * @param   sfilt_id the handle of the section filter
 */
void STB_DMXReleaseSectFilter(DMXREF dmx, SECFILT sfid);


#if defined(__cplusplus)
}
#endif

#endif /* _STB_FILTER_H */
