/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Function prototypes for operating system
 * @file    stb_os.h
 * @date    12/02/2004
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error
#ifndef _STB_OS_H
#define _STB_OS_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

// timeouts for queues, semaphores etc
#define TIMEOUT_NOW     0
#define TIMEOUT_NEVER   0xffff

//---Enumerations for public use-----------------------------------------------

//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Get Current Computer Clock Time.
 * @return  Time in Milliseconds.
 */
U32BIT STB_OSGetClockMilliseconds(void);

/**
 * @brief   Create a New Task to the calling process. Upon success, the
 *          created task runs on its own stack, as part of the calling
 *          process context.  Task termination must be achieved by calling
 *          STB_OSDestroyTask()
 * @param   function task entry point
 * @param   param user defined parameter passed when task is started
 * @param   stack stack size
 * @param   priority task priority, min 0, max 15
 * @param   name task name
 * @return  handle of task
 */
void *STB_OSCreateTask(void (*function)(void *), void *param, U32BIT stack, U8BIT priority, U8BIT * name);

/**
 * @brief   Delay Task for Specifed Time Period.
 * @param   timeout delay in milliSeconds.
 */
void STB_OSTaskDelay(U16BIT timeout);

/**
 * @brief   Delete Task must be called upon termination of each task as it
 *          frees all OS specific resources allocated for the specific task.
 */
void STB_OSDestroyTask(void *task);

/**
 * @brief   Create Queue of given number of messages and size of message
 * @param   msg_size  Queue Message Packet Size
 * @param   num_msgs  Queue Message Depth in Packets
 * @return  Queue Handle - Number for success, NULL upon failure.
 */
void* STB_OSCreateQueue(U16BIT msg_size, U16BIT msg_max);

/**
 * @brief   Read a message from a queue
 * @param   queue Queue Handle
 * @param   data User's Read Message Buffer Start Address.
 * @param   msg_size Message Packet Size in Bytes.
 * @param   timeout timeout in milliseconds
 * @return  TRUE for success, FALSE upon failure.
 */
BOOLEAN STB_OSReadQueue(void *queue, void *msg, U16BIT msg_size, U16BIT timeout);

/**
 * @brief   Write a message to the queue
 * @param   queue Queue Handle
 * @param   data message to be queued
 * @param   msg_size size of message in bytes
 * @param   timeout timeout in milliseconds
 * @return  TRUE for success, FALSE upon failure.
 */
BOOLEAN STB_OSWriteQueue(void *queue, void *msg, U16BIT msg_size, U16BIT timeout);

/**
 * @brief   Destroy Queue
 * @param   queue Unique Queue Handle Identifier Variable Address.
 * @return  TRUE for success, FALSE upon failure.
 */
BOOLEAN STB_OSDestroyQueue(void *queue);

/**
 * @brief   Create a Semaphore
 * @return  Seamphore Handle Address upon success, or NULL upon failure.
 */
void* STB_OSCreateSemaphore(void);

/**
 * @brief   Create a counting semaphore.
 * @param   value initial value for semaphore.
 * @return  Seamphore handle upon success, or NULL upon failure.
 */
void* STB_OSCreateCountSemaphore(U32BIT value);

/**
 * @brief   Wait on Semaphore Indefinity or Until Released.
 * @param   semaphore Semaphore handle.
 * @return  TRUE for success, FALSE upon failure.
 */
void STB_OSSemaphoreWait(void *semaphore);

/**
 * @brief   Signal a Semaphore to Release it by decrementing its counter.
 * @param   semaphore Semaphore handle.
 */
void STB_OSSemaphoreSignal(void *semaphore);

/**
 * @brief   Wait on Semaphore for Set Time Period in an Attempt to Acquire.
 * @param   semaphore Semaphore handle.
 * @param   timeout Time Period to Wait in milliseconds.
 * @return  TRUE for success, FALSE upon failure.
 */
BOOLEAN STB_OSSemaphoreWaitTimeout(void *semaphore, U16BIT timeout);

/**
 * @brief   Delete a Semaphore
 * @param   semaphore Semaphore handle.
 * @return  TRUE for success, FALSE upon failure.
 */
void STB_OSDeleteSemaphore(void *semaphore);

/**
 * @brief   Create a mutex.
 * @return  Newly created mutex, or NULL.
 */
void* STB_OSCreateMutex(void);

/**
 * @brief   Lock a mutex (a.k.a. 'enter', 'wait' or 'get').
 * @param   mutex_var The mutex to lock.
 */
void STB_OSMutexLock(void *mutex);

/**
 * @brief   Unlock a mutex (a.k.a. 'leave', 'signal' or 'release')
 * @param   mutex_var The mutex to unlock.
 */
void STB_OSMutexUnlock(void *mutex);

/**
 * @brief   Delete a mutex.
 * @param   mutex_var The mutex to delete.
 */
void STB_OSDeleteMutex(void *mutex);

#endif //  _PM_OS_H

