/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2000 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Main API to DSM-CC core layer (provided functions and required
 *             callbacks).
 * @file    cldsmcc.h
 * @date    01/02/2002
 * @author  R.Freeman
 */

/*
 * DVB DSM-CC Object Carousel Core Layer API
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 *
 * OVERVIEW:
 *
 * This API has been specified in standard ANSI C and is designed as a completely
 * OS and Platform (ie. HW and SW driver) independent interface to the core
 * DSM-CC layer.
 *
 * To avoid confusion over the sizes of standard C types on different platforms, it
 * assumes that a standard set of unambiguous type names have been defined to
 * platform specific types in the file clDsmPlatformAPI.h (also standard
 * definitions for TRUE/FALSE/Null). NB. This currently matches the recommended DVP
 * coding standards.
 *
 *
 * DSM-CC code conforming to this API should be readily portable to any platform
 * and OS environment. It assumes that this environment (designated 'the
 * calling environment') can provide the following:
 *
 * - Filtering of MPEG2 private sections from a single transport stream on multiple
 * simultaneous PID values and section filter values.
 *
 * - Acquisition and parsing of PMT(s) for specified program stream, query/lookup
 *       of specific information from the SI data
 *
 * - Basic block memory allocation and de-allocation (at startup and shutdown)
 *
 * - Semaphore or queuing mechanisms for synchronising API calls initiated by
 * asynchronously running tasks or threads
 *
 * - Queuing mechanisms, eg. for notifying clients of asynchronous object loads
 *
 * - Multiple timers (eg. 10mS resolution)
 *
 * - Dynamic (heap) memory management that conforms to the API specified in
 * clDsmMemMgrApi.h
 *
 * NB. A non-fragmenting linked-block based memory manager that conforms to this
 * API is supplied along with the DSM-CC core stack (clDsmMemMgrBlock.c).
 *
 *
 * It is intended that all platform (and DSM-CC profile) specific functions
 * including handling of multiple clients/apps (if required) are implemented in
 * wrapper layers that call into this API. Only these layers should need to
 * be modified when porting to a new platform (or DSM-CC profile).
 *
 *
 * BASIC DATA FLOW AND LAYERS:
 *
 *          +-----------+                     +-----load/unload-----------------+
 *          |  DEMUX &  |   DSM-CC            |      carousel                   |
 *     TS-->|  section  |---private------+    |                                 |
 *          |  filters  |   sections     |    |    +--subscribe/unsubscribe--+  |
 *          +-----------+                |    |    |    for stream event     |  |
 *                A                      |    |    |                         |  |
 *                |                      |    |    |      load/unload +---------------+
 *                +---add/delete-----+   |    |    |    +-open/close--|    CLIENT(s)  |
 *                  section filter   |   |    |    |    |   object    +---------------+
 *                                   |   |    |    |    |               A    A      A
 *                +---SI Query--+    |   |    |    |    |               |    |      |
 *                |     result  |    |   V    V    V    V               |    |      |
 *         +----------------+   |  +------------------------+           |    |      |
 *     SI->|  SI (PAT/PMT)  |   +->|                        |  object/  |    |      |
 *         |    database    |      |                        |--carousel-+    |      |
 *         +----------------+  +---|                        |  loaded        |      |
 *                A            |   |                        |                |      |
 *                |            |   |   Platform & Profile   |--stream event--+      |
 *                +--SI Query--+   | (eg.MHEG/MHP) specific |                       |
 *                                 |   'porting' layer(s)   |    +-object data/info-+
 *                    +--Trigger-->|                        |    |
 *      +----------+  |            |                        |    |
 *      |  Timers  |--+      +-----|                        |    |
 *      +----------+         |     |------synchronous-------+-asynchronous-+
 *           A               |     |==============(clDsmAPI)===============|
 *           |               |     |                                       |
 *           +---start/stop--+     |                                       |
 *                                 |                DSM-CC                 |
 *                                 |            CORE (IP) LAYER            |
 *                                 |                                       |
 *                                 |                                       |
 *                                 |                                       |
 *                                 +-------------------/\------------------+
 *                                            +--------\/--------+
 *                                            | DSMCC DATA CACHE |
 *                                            +------------------+
 *
 *
 * --------------------------------------------------------------------------------
 */
#ifndef _DSMCC_H_
#define _DSMCC_H_

/*---includes for this file--------------------------------------------------*/
#include "cldsmtypes.h"
#include "stdfuncs.h"

#ifdef __cplusplus
extern "C" {
#endif


/*---Constant and macro definitions for public use---------------------------*/

#define NUM_OPEN_DSM_OBJECTS_MAX    128

#define DSMCC_MINIMUM_CACHE_SIZE    (1024 * 1024 * 1)
#define DSMCC_DEFAULT_CACHE_SIZE    (1024 * 1024 * 6)

/* -- caching priority rules */
#define CACHE_RULES_DEFAULT         0x3F
#define CACHE_RULES_FROM_STREAM     0x00
#define CACHE_RULES_STATIC          0x80

/* -- Progress codes */
#define CLDSM_PROG_CURR_SERVICE_CACHE_FULL  0
/* + ? */


/*---Enumerations for public use---------------------------------------------*/

/*---Global type defs for public use-----------------------------------------*/

/* E_CacheRules is like, and could be, an enum, with 8-bit values defined above */
typedef U8BIT E_CacheRules;

/*
-- **** TIMER EVENT TYPES ***
*/

/* -- Timer event status -- */
typedef enum
{
   TIMER_TRIGGERED = 0,
   TIMER_ABORTED
} E_TimerStatus;


/*
-- **** OBJECT DATA TYPES ***
*/

typedef struct s_DsmObject *H_DsmObject;

/* -- Object kind -- */
typedef enum
{
   ANON_OBJ = 0,
   FILE_OBJ,
   DIRECTORY_OBJ,
   SERVICE_GATEWAY_OBJ,
   STREAM_OBJ,
   STREAM_OBJ_WITH_EVENTS,
   LINK_TO_ALTERNATE_OC_OBJ
} E_DsmObjectKind;


/*
-- Status type for object load
*/
typedef enum
{
   /*0*/ OBJ_LOAD_UNINITIATED = 0,
   /*1*/ OBJ_LOAD_IN_PROGRESS,
   /*2*/ OBJ_LOAD_COMPLETED,
   /*3*/ OBJ_LOAD_ABORTED_TIMEOUT,
   /*4*/ OBJ_LOAD_ABORTED_PATH_ERROR,
   /*5*/ OBJ_LOAD_ABORTED_ERROR,
#ifdef DSM_MHP_PROFILE
   /*6*/ OBJ_LOAD_ABORTED_UNLOAD,
   /* -- NOT CURRENTLY IMPLEMENTED */
   /*7*/ OBJ_LOAD_PARENT_DIR_LOADED
#else
   /*6*/ OBJ_LOAD_ABORTED_UNLOAD
#endif
} E_ObjLoadStatus;


typedef struct s_ObjUserData *H_ObjUserData;

/*
 This is the current status of the notification of a Stream Event.
*/
typedef enum
{
   SEN_INITIAL,
   SEN_NOTIFIED,
   SEN_ABORTED_OBJ_LOAD_FAILED,
   SEN_ABORTED_SUBSCRIBE_FAILED,
   SEN_TRIGGERED,
   SEN_ACKNOWLEDGE_UNSUBSCRIBE
} E_SENotifyStatus;


/* --------------- REQUIRED API - CALLBACK TYPE DEFINITIONS ----------------- */

/*
-- NOTES:
--
-- The calling environment must provide functions that match these callback
-- type definitions (except F_Progress which is optional). All
-- required callbacks should be non-blocking. The callback addresses are
-- supplied to the DSM-CC instance at instance create/setup.
--
-- The callbacks are not expected to fail except under exception conditions.
--
-- For callbacks with 'int' return values, this should be used to indicate:
--      zero:     Successful execution/registration of call with calling env.
--      non-zero: Failure to execute/register call (eg. due to memory failure)
--
-- The DSMCC library will attempt to take some logical actions in response to a
-- failure of a start/add callback that may enable it to continue.(eg. abandon
-- request and notify client of a load failure due to system error).
--
-- The stop/delete callbacks are not provided with return values since there is
-- no meaningful action the DSM-CC library can take in response to these
-- failing. If it is possible that these callbacks fail to execute the
-- requested action then the internal state of the DSM-CC library will become
-- inconsistent with the calling environment. The calling environment must
-- handle this situatation - eg. by raising an exception and processing it when
-- the DSM-CC API call that triggered it completes. The calling environment
-- should deal with the problem before making any further DSM-CC API calls
-- (except clDsmSysReset or clDsmSysDestroy). It should either ensure that the
-- action requested by the callback is successfully executed or clear the problem
-- and reset/restart the DSM-CC library.
--
*/


/* -- ERROR/PROGRESS CALLBACKS -- */

/*******************************************************************************
* Informs calling environment of errors triggered during stream/section
* processing
*******************************************************************************/
typedef void (*F_Error) ( /*I*/ E_DscError err, void *args );


/*******************************************************************************
* Informs calling environment of state changes triggered during
* stream/section processing
*******************************************************************************/
typedef void (*F_Progress) ( /*I*/ U32BIT prog, void *args );



/* -- TIMER CALLBACKS -- */

/*******************************************************************************
* Starts timer with specified period
*
* Timer trigger/timeout notified via CDSM_SysProcessTimerEvent().
*
* When notifying a timeout then the value of clDsmTmrUserData
* supplied to this call MUST also be input to clDsmProcessTimerEvent.
*
* Returning a timerHandle value is optional but it should be set to Null if
* not used/available. This value is input to the stopTimerFunc callback.
*
* If a timerHandle is not available synchronously then it can be also be
* supplied when clDsmSysProcessTimerEvent is called.
*
*******************************************************************************/
typedef E_DscError (*F_StartTimer) ( H_DsmControl dsmControl,
                                             /*I*/ U32BIT timePeriod,
                                             void *clDsmTmrUserData,
                                             /*O*/ void **pTimerHandle );


/*******************************************************************************
* Stops timer
*
* Timers will always be explicity stopped via this call, even after they have
* triggered. Both application and component handles/references are supplied for
* flexibility.
*
* NB. If a timer is stopped (aborted) before it has triggered (ie. before a
* timeout has been notified via clDsmSysProcessTimerEvent), timer abort must be
* notified via CDSM_SysProcessTimerEvent() instead. The call to
* clDsmSysProcessTimerEvent can be made from within this callback.
*
*******************************************************************************/
typedef void (*F_StopTimer) ( H_DsmControl dsmControl,
                                      /*I*/ void *timerHandle );



/* -- SI QUERY CALLBACKS -- */

/*******************************************************************************
* Makes (starts) SI query (eg. database lookup)
*
* Meaning of return values in this context:
*
* CLDSM_OK
* - Query has (synchronously) returned successful result.
*
* CLDSM_PENDING
* - Query is being executed asynchronously - results/events
*                   will be returned via CDSM_SysProcessSIQueryEvent()
*
* CLDSM_ERR_SI_QUERY_FAILED
* - Query has (synchronously) failed to determine requested
*                   information (due to error in SI lookup).
*
* If a query result is to be returned asynchronously then the value of
* clDsmSIQueryRef and clDsmSIUserData supplied to this call MUST also be input
* to clDsmProcessSIQueryEvent along with the result.
*
* Returning a queryHandle value in pResult is optional. If not used/available,
* it should be set to Null. This value is input to the stopSIQueryFunc callback.
*
* If a queryHandle is not available synchronously then it can be also be
* supplied when clDsmSysProcessSIQueryEvent is called.
*
* NB. In this case, if the query is stopped for any reason before this event
* then the calling environment will have to identify the correct query to stop
* from the DSM-CC reference (clDsmSIQueryRef).
*
*******************************************************************************/
typedef E_DscError (*F_StartSIQuery) ( H_SiqInstance siqInstance,
   P_SIQueryRequest pQuery, H_SIQueryRef clDsmSIQueryRef,
   void* clDsmSIUserData,
   P_SIQueryResult pResult );


/*******************************************************************************
* Stops an SI query
*
* Any asynchronous SI queries (ie. those that returned a status SIQUERY_PENDING
* to the StartSIQueryFunc call) will be explicity stopped via this call, even
* after they have completed.
*
* Both application and component handles/references are supplied for
* flexibility.
*
* NB. If an SI query is stopped (aborted) before it has completed (ie. before
* a result has been supplied via clDsmSysProcessSIQueryEvent), SI query abort
* must be notified via CDSM_SysProcessSIQueryEvent() instead. The call to
* clDsmSysProcessSIQueryEvent can be made from within this callback.
*
*******************************************************************************/
typedef void (*F_StopSIQuery) ( H_SiqInstance siqInstance,
   /*I*/ void *queryHandle, H_SIQueryRef clDsmSIQueryRef );



/* -- SI CHANGE REGISTRATION CALLBACKS -- */

/*******************************************************************************
* Register to receive notifications of changes to the SI for the specified
* service (on the current transport stream/multiplex).
*
* When a duplicate subscribe is made (ie. for a service that is already
* subscribed) it should increment the count of subscriptions for that service.
* Matching unsubscribe will be made for each subscribe.
*
* Changes are notified via the clDsmSysProcessSIChangeEvent function in the
* following situations:
*
*   - The subscribed service cannot be located
*   - The service is deleted from the multiplex (eg. PAT updated)
*   - The service info is updated (ie. PMT updated).
*
* If it can be determined immediately (ie. synchronously) that the requested
* service does not exist in the current multiplex (eg. it is not listed in the
* most recent stored copy of the PAT) then the output parameter serviceNotFound
* can be set to TRUE (otherwise set it to FALSE).
*
*******************************************************************************/
typedef E_DscError (*F_SubscribeSIChanged) ( H_SiqInstance siqInstance,
                                                 /*I*/ U16BIT service_id );


/*******************************************************************************
* Decrement number of subscriptions to the specified service. Only when this
* number reaches zero should this function de-register for notifications of
* changes to the SI for the service.
*
* This will be called to de-register for a previously subscribed service,
* even if the DSM-CC library has been notified that the subscribed service
* cannot be located.
*
*******************************************************************************/
typedef void (*F_UnsubscribeSIChanged) ( H_SiqInstance siqInstance,
   /*I*/ U16BIT service_id );



/* -- SECTION FILTER CALLBACKS -- */

/*******************************************************************************
* Requests a section filter to be started.
*
* It is assumed that the number of section filters that may be simultaneously
* requested (in use) is unlimited. Typically it is expected that the worst case
* requirement (eg. with full carousel caching and multiple stream events) is
* less than that specified in the setup.
*
* Filtered private sections are returned via CDSM_SysProcessPrivateSection().
* The clDsmFilterRef value MUST be stored and also
* returned with any sections that matched the requested filter.
*
* Exact duplicate filters will never be requested simultaneously but filters
* with 'wildcard' (ie. tableIdExtMask) bits set may overlap with other
* requested filters with less or no mask bits set.
*
* If a section aligns with two or more such filters then it should only be
* matched to the 'narrowest' (ie. most exact) filter and that clDsmFilterRef
* value passed with the section data to clDsmSysProcessPrivateSection.
*
* Filters requested with high priority will always be narrower (ie. less mask
* bits set) than any overlapping low priority filters and this rule ensures
* that sections will always be matched to high priority filters in preference
* to low.
*
* NB. The worst case number of exact filters (ie. with no mask bits set)
* requested simultaneously may be large (eg. 100s) but the worst case number of
* simultaneous filters with (one or more) mask bits set is expected to be much
* smaller (eg. ~10).
*
* The filter should always remain in effect with continuous supply of section data,
* until delSectionFilterFunc is called.
*
* If platform requires priority rating in this callback, it can use
* CDSM_SectionFilterPriority().
*
* Parameters:
*
* PID              MPEG2 PID
*
* tableId          DSM-CC tableId filter value for sections (byte 0).
*                  NB. Only the tableId values of 0x3B, 0x3C, 0x3D will
*                  be used for DSM-CC section filters.
*
* tableIdExt       DSM-CC tableIdExtension filter value (bytes 3&4)
*
* tableIdExtMask   DSM-CC active filter bits.
*                  NB.'Wildcard' bitmasks will only be set on contiguous
*                  bits starting at LSBit.
*
* Return: void*
* This is the platform filter handle value. It is given as input to the
* delSectionFilterFunc callback. NULL value is considered failure, and any other
* value is considered a successful operation.
*
*******************************************************************************/
typedef void * (*F_AddSectionFilter) ( H_SfmInstance sfm,
   /*I*/ P_SecFilter pFilter, H_DscSFRef dsmFilterRef );


/*******************************************************************************
* Requests a section filter to be stopped
*
* To give flexibility in identifying the filter to delete, both application
* and component handles/references are supplied. Also a repeat of the section
* filter details are supplied (which are unique for any active filter) .
*
* NB. To avoid problems due to delays in deleting filters or caused by queuing
* sections the clDsmSysProcessPrivateSection function can detect sections being
* supplied from 'stale' (ie. deleted) filters. This will waste CPU time so the
* calling environment should ensure as soon as possible (after this callback
* is made) that sections matching the deleted filter are stopped.
*
*******************************************************************************/
typedef void (*F_DelSectionFilter) ( H_SfmInstance sfm,
   /*I*/ void *filterHandle, H_DscSFRef clDsmFilterRef );


/*******************************************************************************
* Notify a section filter priority changed
*
*******************************************************************************/
typedef void (*F_SFPriorityChanged) ( H_SfmInstance sfm,
   /*I*/ void *filterHandle, H_DscSFRef clDsmFilterRef,
   E_SFPriority priority );


/* -- LOAD & STREAM EVENT CALLBACKS -- */

/*******************************************************************************
* Notify of a load carousel event.
*
* Called when carousel booted, load completed (ie. SRG module loaded) or load
* aborted before completion (due to errors, timeout or unload). Component
* handle and userData values supplied for flexibility.
*
* If CDSM_UnloadCarousel() is called before the carousel load is completed
* (ie. before this callback has been called with status OC_LOAD_COMPLETED)
* then the callback will actually be made from within the clDsmUnloadCarousel
* call with status OC_LOAD_ABORTED_UNLOAD.
*
* Any of the Synchronous or Asynchronous Client API calls (ie. API calls that
* do NOT start with clDsmSys...) can be made from within this callback. When
* doing this, care must be taken to avoid blocking or re-entrancy problems in
* the calling environment. Re-entrancy may occur when unloading objects or
* carousels since this can cause a (re-entrant) call to this callback.
*
*
*******************************************************************************/
typedef void (*F_NotifyCarouselLoadEvent) (
   /*I*/ H_DsmCarousel clDsmCarouselHandle, E_OCLoadStatus status,
   U32BIT carouselId );


/*******************************************************************************
* Notify of a load object event.
*
* Called when object load completed or load aborted before completion (due to
* errors, timeout or unload of object or carousel). Component handle and
* userData values supplied for flexibility.
*
* If CDSM_UnloadObject() (or clDsmUnloadCarousel) is called before the object
* load is completed (ie. before this callback has been called with status
* OBJ_LOAD_COMPLETED) then the callback will actually be made from within the
* CDSM_UnloadObject(or clDsmUnloadCarousel) call with status
* OBJ_LOAD_ABORTED_UNLOAD.
*
* Any of the Synchronous or Asynchronous Client API calls (ie. API calls that
* do NOT start with clDsmSys...) can be made from within this callback. When
* doing this, care must be taken to avoid blocking or re-entrancy problems in
* the calling environment. Re-entrancy may occur when unloading objects or
* carousels since this can cause a (re-entrant) call to this callback.
*
*
*******************************************************************************/
typedef void (*F_NotifyObjectLoadEvent) (
   /*I*/ H_DsmObject clDsmObjectHandle, E_ObjLoadStatus status,
   H_ObjUserData pCopyOfObjLoadUserData );


/**
 * @brief   The Client defines this function to enable the DSMCC to communicate to the
 *          Client when any subscribed to stream event has occurred.
 *          DSMCC prototypes this function; the Client defines this function.
 *          i.e. This function must be implemented in the Client not in the DSMCC.
 *          There will be a single callback function that will be called when an event
 *          occurs. This callback function was specified when the Client created the
 *          DSMCC instance. The callback may be called a number of times if the event
 *          is re-defined.
 *          The CDSM_StreamEventSubscribe() function must have been successfully called
 *          before the Client callback function will be called.
 *          The callback can be called when an error occurs.
 *          The callback is also called when an unsubscribe has completed.
 *          The 'status' argument to the callback differentiates between unsubscribe,
 *          errors and event triggered.
 *          The two streamEventUserData values allow the Client to differentiate between
 *          the many stream events that are notified through the single callback function.
 * @param   eventHandle This is the handle obtained when the stream
 *          event was subscribed to.
 *          The client will use this handle when this
 *          stream event is referred to in the future.
 * @param   status Status of the subscription.
 * @param   userData1 Optional data to be used by client.
 *          This data was supplied by the Client when a
 *          subscription was taken out on a stream event.
 * @param   userData2 Optional data to be used by client.
 *          This data was supplied by the Client when a
 *          subscription was taken out on a stream event.
 * @param   namePtr Event Name pointer
 * @param   dataPtr Event Data pointer
 * @param   namelen Event Name length
 * @param   dataLen Event Data length
 * @return
 *          There is no return value.
 */
typedef void (*F_NotifyStreamEvent)(
   /*I*/ H_DsmEvent eventHandle, E_SENotifyStatus status,
   void *userData1, void *userData2,
   U8BIT *namePtr, U8BIT *dataPtr, U8BIT namelen, U8BIT dataLen );



/**
 * @brief   Called when request for deferred service of stream object has completed after
 *          clDsmStreamGetDeferredService returned CLDSM_PENDING
 *          userData values supplied for flexibility.
 * @param   userData1 Optional data to be used by client.
 *          This data was supplied by the Client when a
 *          subscription was taken out on a stream event.
 * @param   userData2 Optional data to be used by client.
 *          This data was supplied by the Client when a
 *          subscription was taken out on a stream event.
 * @param   deferredService Deferred Service for Stream object .
 * @return
 *          There is no return value.
 */
typedef void (*F_NotifyDeferredService) (
   /*I*/ void *userData1, void *userData2,
   S_DvbLocator *pDeferredService );


/*
-- Type definitions for setup variables
--
*/

/*
-- Setup structure for use at initialisation
*/
typedef struct
{
   /*
   -- General callback pointers
   --
   -- NOTES:
   -- All callbacks MUST be provided except where indicated
   --
   */
   F_MemAlloc allocFunc;
   F_MemFree freeFunc;

   /* -- errorFunc - optional (Null if not used) */
   F_Error errorFunc;
   /* -- progressFunc - optional (Null if not used) */
   F_Progress progressFunc;

   F_AddSectionFilter addSectionFilterFunc;
   F_DelSectionFilter delSectionFilterFunc;
   /* -- sfPriorityChangeFunc - optional (Null if not used) */
   F_SFPriorityChanged sfPriorityChangeFunc;

   F_StartTimer startTimerFunc;
   F_StopTimer stopTimerFunc;

   F_StartSIQuery startSIQueryFunc;
   F_StopSIQuery stopSIQueryFunc;

   F_SubscribeSIChanged subscribeSIChangeFunc;
   F_UnsubscribeSIChanged unsubscribeSIChangeFunc;

   /* -- notifyCarouselLoadEventFunc - optional (Null if not used) */
   F_NotifyCarouselLoadEvent notifyCarouselLoadEventFunc;
   /* -- notifyObjectLoadEventFunc - optional (Null if not used) */
   F_NotifyObjectLoadEvent notifyObjectLoadEventFunc;

   /* notifyStreamEventFunc can be Null when Stream Events are not required */
   F_NotifyStreamEvent notifyStreamEventFunc;

   /* notifyDeferredServiceFunc can be Null when deferred service for Stream Objects are not required */
   F_NotifyDeferredService notifyDeferredServiceFunc;

   S_SsuConfig ssuFuncs;

   /* -- Control flags */

   H_DsmControl dsmControl;
   /*
   -- Instance handle for external Service Information Query module
   */
   H_SiqInstance siqInstance;             /* -- Suggested default: Null */

   /*
   -- Instance handle for external Section Filter Manager module
   */
   H_SfmInstance sfmInstance;             /* -- Suggested default: Null */

   /*
   -- Pointer (optional) that can be used by calling environment for
   -- passing setup info to the memory manager (via clDsmMemMgr API) at
   -- memStart time.
   */
   void *memMgrSetup;               /* -- Suggested default: Null */


   /*
   -- Specifies resolution of time units used on API in _milliseconds_
   -- (can be set to resolution of external timers)
   --
   -- NB. Time values are represented on the API and internally as U32BIT so
   -- timeUnitResolution should be chosen so that overflow problems will not
   -- occur during any likely period of continuous use for a receiver (eg. if
   -- timeUnitResolution = 10mS, wrap-round time = 1.36 years!).
   */
   U32BIT timeUnitResolution;       /* -- Suggested default: 10 */


   /*
   -- Sets the size of dynamic memory available for DSM-CC to use.
   -- Apart from 100KB approx. the remainder of this memory will be requested
   -- and accessed via the dynamic memory manager API (clDsmMemMgrAPI.h).
   --
   -- NB. In debug builds DSM-CC may allocate slightly (<5%) more than this
   --     specified amount.
   --
   -- If clDsmMemMgrAPI is mapped to a system memory manager then it
   -- has an additional option (when it is opened) to decrease the amount
   -- of (the remainder of this) memory it makes available.
   */
   U32BIT maxMemorySize;        /* -- Absolute minimum: DSMCC_MINIMUM_CACHE_SIZE */
                                /* -- Recommended min: DSMCC_DEFAULT_CACHE_SIZE  */

   /*
   -- Specifies the maximum number of section filters that are available
   -- in the calling environment (to this instance) for simultaneously
   -- acquiring DSM-CC private sections.
   --
   -- If client requests (or other events) cause the core stack to
   -- simultaneously require more section filters than the number available
   -- it will rotate its requests through the available filters. This may
   -- cause delays in responding to client requests so the larger this
   -- number the better.
   --
   -- The minimum value is NUM_SECTION_FILTERS_MINIMUM.
   -- Setting to zero will be interpreted as no restriction on the
   -- number of section filters available in the calling environment, and
   -- NUM_SECTION_FILTERS_MAXIMUM will be used.
   -- Setting to a value less than this the minimum will be ignored, and
   -- NUM_SECTION_FILTERS_MINIMUM will be used.
   -- Suggested Default: NUM_SECTION_FILTERS_DEFAULT
   */
   U16BIT maxAvailableSectionFilters;

   /*
   -- Sets whether DSM-CC component maintains an internal record
   -- of SI query results so that it does not need to repeatedly
   -- make the same SI queries (to the calling env). Typically this
   -- will be used for storing association_tag to PID mappings.
   */
   BOOLEAN storeSIQueryResults;        /* -- Default: FALSE */


   /*
   -- Sets whether DSM-CC component pre-fetches modules into its
   -- internal cache.
   --
   -- NB. This requires a large available memory size and a large number of
   --     available section filters.
   --
   -- Enabling turboCaching also activates full transparent caching (ie.
   -- automatic re-loading of cached modules that have version updates).
   -- This is because turboCaching will not work effectively without this
   -- and also because only when turboCaching is enabled are there
   -- likely to be sufficient section filters available to support
   -- automatic module re-loading.
   */
   BOOLEAN turboCaching;               /* -- Default: FALSE */

   /*
    */
   BOOLEAN multipleSsuDownloads;
} S_DsmSetup, *P_DsmSetup;



/*---Global variable declarations for public use-----------------------------*/


/******************************************
* EXPORTED (PROVIDED) FUNCTION PROTOTYPES *
*******************************************/

/* ---------------------- SYNCHRONOUS SYSTEM API CALLS ---------------------- */

/*
-- NOTES:
--
-- The following functions must be called synchronously with each other and
-- with the Synchronous Client API calls since they use and modify common
-- instance data. If they need to be called/triggered from different threads
-- the calling environment must ensure synchronous behaviour - eg. by using
-- a mutex semaphore round the call (or queueing and calling from the same
-- thread).
--
-- In some circumstances it is possible to make one Synchronous API call from
-- within another one:
--
-- eg. clDsmSysProcessTimerEvent can be called within the stopTimerFunc
-- callback which itself may have been called from within.
-- clDsmSysProcessPrivateSection
--
-- eg. All Synchronous Client API calls can be called from within the
-- notifyCarouselLoadEvent or notifyObjectLoadEvent callbacks which may
-- themselves have been called within clDsmSysProcessPrivateSection.
--
-- These cases are clearly specified in the comments. The calling environment
-- must avoid possible permanent blocks in these circumstances, eg. from trying
-- to get a mutex on a thread that has it already.
--
*/


/*******************************************************************************
* Create DSM-CC Core Layer instance
*
* Only call at startup of an instance (before ANY other DSM-CC Core Layer API
* calls are made for that instance). Allocates all required 'static' memory
* for the instance (via allocFunc). Also stores (copies) data supplied in
* setup structure.
*
* NB. Dynamic (heap) memory used by the instance is allocated/deallocated and
* accessed via the dynamic memory manager API (clDsmMemMgrApi.h).
*
* Returns instance value which is used to reference the unique data used by
* this DSM Core instance. It MUST be stored by the calling environment and
* passed to any clDsm API calls for the instance.
*
* Also returns memContext parameter (obtained when memory manager started).
* If implemented/required, this gives the calling environment independent
* access to the memory manager context used by this DSM Core instance.
*
* CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
*   allocFunc
*   freeFunc - only on error
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_ALLOC_FAILED,
* CLDSM_ERR_MEM_HEAP_FULL
* CLDSM_ERR_ILLEGAL_SETUP
*
*******************************************************************************/
E_DscError CDSM_SysCreate(
   /*I*/ P_DsmSetup pSetup,
   /*O*/ H_DsmCoreInst *pInstance, void **pMemContext );



/**
 * @brief   Destroy DSM-CC Core Layer instance.
 *          De-allocates all 'static' memory used by the instance (via freeFunc). Only
 *          call at shutdown of DSM-CC instance.
 *          NB. clDsmDestroy will not 'fail' to execute even if it detects error
 *          conditions - ie. it will always de-allocate the instance memory. API
 *          calls MUST NOT be made with the old (stale) instance value after
 *          clDsmDestroy has been called since they will cause illegal memory accesses.
 *          To avoid problems, the instance should always be reset (via clDsmReset)
 *          before calling clDsmDestroy. This will detect if there are any active client
 *          or system transactions - eg. loaded objects or pending SI Query events.
 *          This would indicate it may be dangerous to call clDsmDestroy until they are
 *          completed.
 *          If clDsmDestroy detects that the instance is not in a reset state - ie.
 *          immediately after clDsmSysCreate or after clDsmReset has been successfully
 *          called (with force switches if necessary), it will return an error.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          freeFunc
 * @param   instance DSMCC instance handle
 * @return
 *          CLDSM_OK (0)
 *          Destroy was executed with no problems.
 *          CLDSM_ERR_INSTANCE_NOT_RESET
 *          Destroy was executed but instance was not in a reset state so there may
 *          still be client and system transactions outstanding.
 *          CLDSM_ERR_MEMSTOP_PROBLEM
 *          Destroy was executed but memory manager detected a problem while stopping.
 *          NB. Should be regarded as an internal error unless clDsmMemMgrAPI is mapped
 *          to a system (external) memory manager (DSM_SYSTEM_MEMMGR defined).
 */
E_DscError CDSM_SysDestroy( H_DsmCoreInst instance,
   H_SiqInstance *pSiqInst, H_SfmInstance *pSfmInst );


/**
 * @brief   Get control handle from DSM instance .
 * @param   instance
 * @return  NULL if an invalid instance handle, otherwise returns control handle
 */
H_DsmControl CDSM_SysGetControl( H_DsmCoreInst instance );


/**
 * @brief   Reset DSM-CC Core Layer instance.
 *          Resets conditions to the same state as after CDSM_SysCreate(ie. keeps
 *          setup parameters).
 *          In normal operation the client MUST first unsubscribe all events, close and
 *          and unload all objects and unload all carousels before calling this function.
 *          This will have also stopped/deleted all associated system resources (ie. SI
 *          queries, timers, SI change monitors, section filters).
 *          If any client transactions or system transactions are still active when
 *          clDsmReset is called the reset will not be executed and an error returned.
 *          NB. The calling environment (system side) must acknowledge stop
 *          commands for SI Queries and timers (via processSIQueryEvent,
 *          processTimerEvent) and that these may occur asynchronously (ie. after a
 *          delay). This means that there may still be some active system transactions
 *          for some time after all objects and carousels are unloaded.
 *          FORCING RESET IN ERROR CONDITIONS:
 *          If all client and system transactions are apparently completed but still
 *          returns an error (eg. because handles are unknown/lost or system
 *          resource stop requests are not acknowledged) then clDsmReset can be called
 *          with appropriate force switches to clear the error condition.
 *          Any client transaction errors must first be cleared before system transaction
 *          errors can be cleared and both force switches cannot be set at the same time.
 *          NB. Only use force switches in known error conditions. It is potentially
 *          fatal (may cause illegal memory access) to call the functions:
 *          streamEventUnsubscribe, unloadObject, unloadCarousel, processSIQueryEvent,
 *          processTimerEvent with old/stale handles after clDsmReset has been forced.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          deleteSectionFilter
 *          stopTimer
 *          stopSIQuery
 *          unsubscribeSIChange
 * @param   instance DSMCC instance handle
 * @param   mode RST_MODE_FORCE or RST_MODE_PENDING
 * @return
 *          CLDSM_OK (0)
 *          The reset was executed with no problems.
 */
E_DscError CDSM_SysReset( H_DsmCoreInst instance, E_DsmRstMode mode );

/**
 * @brief   Set maximum memory usage.
 * @param   instance
 * @return  n/a
 */
void CDSM_SysSetMemoryMax( H_DsmCoreInst instance, U32BIT maxMemory );

/**
 * @brief   Set/notify the current service (initially and when changing it).
 *          This MUST be called initially (ie. after clDsmCreate or clDsmReset), before
 *          the boot carousel (or any carousels) can be loaded.
 *          To change the current service (ie. once one is already set) then the client
 *          MUST first unsubscribe all stream events and successfully unload all objects
 *          and carousels (from the current service). If any errors occur then
 *          clDsmReset may need to be called (with force switches - see Additional
 *          information below) before a new (or even the same) service can be set.
 *          When this function is re-called. If the service details match those currently
 *          set then it will take no action. If they are different then the new service
 *          details are stored (unless objects and/or carousels are still loaded, in
 *          which case they are ignored and an error generated).
 *        To forcefully destroy loaded objects and/or carousels (eg. where the handles
 *        may be unknown/lost) then clDsmReset can be used with force switches.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          none
 * @param   instance DSMCC instance handle
 * @param   original_network_id DVB Network Identifier
 * @param   transport_stream_id DVB Transport stream/Multiplex Identifier
 * @param   service_id DVB Service Identifier
 * @return
 *          CLDSM_OK (0)
 *          The new current service details were successfully set.
 *          CLDSM_ERR_CAROUSELS_STILL_LOADED
 *          The new current service details could not be set because one or more
 *          carousels from this service were still loaded
 *          CLDSM_ERR_OBJECTS_STILL_LOADED
 *          The new current service details could not be set because one or more objects
 *          from this service were still loaded
 */
E_DscError CDSM_SysSetCurrService( H_DsmCoreInst instance,
   /*I*/ U16BIT original_network_id, U16BIT transport_stream_id,
   U16BIT service_id );

U16BIT CDSM_SysCurrServiceId( H_DsmCoreInst instance );


/**
 *          Notifies that the specified timer has either timed-out (triggered) or
 *          been stopped (aborted) before triggering.
 *          Input parameter clDsmTmrUserData must be set to the value
 *          supplied when the timer was started.
 *          If the timerHandle supplied here is non-Null it will supercede any value
 *          returned from the startTimerFunc callback (when input to stopTimerFunc
 *          callback).
 *          NB. Although this function should be called synchronously with the other
 *          (synchronous) API functions, it can also be called from within the
 *          stopTimerFunc (required API) callback if required.
 *          CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          case (status)
 *          TIMER_TRIGGERED:
 *          notifyObjectLoadEvent
 *          notifyCarouselLoadEvent
 *          deleteSectionFilter
 *          stopSIQuery
 *          stopTimer
 *          TIMER_ABORTED
 *          none
 *          endcase
 *          RETURNS:
 *          CLDSM_OK (0)
 */
E_DscError CDSM_SysProcessTimerEvent( H_DsmCoreInst instance,
   /*I*/ void *clDsmTmrUserData, E_TimerStatus status, void *timerHandle );



/**
 * @brief   Notifies the result of the specified SI query (ie. a callback to
 *          startSIQueryFunc that returned SIQUERY_PENDING) or notifies that the
 *          query was stopped (aborted) before completion.
 *          Meaning of status values in this context:
 *          SIQUERY_SUCCESS - Query has returned successful result.
 *          SIQUERY_FAILURE - Query has failed to determine requested information (for
 *          any reason).
 *          SIQUERY_PENDING - *** INVALID VALUE FOR THIS CALL ***
 *          SIQUERY_ABORTED - Query has been stopped before completion.
 *          If the queryHandle supplied here is non-Null it will supercede any value
 *          returned from the startSIQueryFunc callback (when input to the
 *          stopSIQueryFunc callback).
 *          NB. Although this function should be called synchronously with the other
 *          (synchronous) API functions, it can also be called from within the
 *          stopSIQueryFunc (required API) callback if required.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          addSectionFilter
 *          stopSIQuery
 *          notifyObjectLoadEvent
 *          notifyCarouselLoadEvent
 *          deleteSectionFilter
 *          stopTimer
 * @param   instance DSMCC instance handle
 * @param   clDsmSIQueryRef Refererence for this query (value supplied at
 *          start time).
 * @param   clDsmSIUserData User data for this query (value supplied at
 *          start time).
 * @param   status Status of query results.
 * @param   pResult Pointer to results of query (if successful).
 * @return
 *          CLDSM_OK (0)
 *          The process SI query event was executed with no problems.
 *          CLDSM_ERR_INVALID_SIQUERY_REF
 *          Supplied clDsmSIQueryRef does not reference a valid query.
 *          CLDSM_ERR_INVALID_SIQUERY_STATUS
 *          Supplied status value is invalid here.
 *          CLDSM_ERR_SYSTEM_ADD_SECTION_FILTER,
 *          Problem with add section filter callback.
 */
E_DscError CDSM_SysProcessSIQueryEvent( H_DsmCoreInst instance,
   /*I*/ H_SIQueryRef clDsmSIQueryRef, void *clDsmSIUserData,
   P_SIQueryResult pResult );



/**
 * @brief   Notify that the SI for the indicated service has changed
 *          Changes should be notified in the following situations:
 *          - The service cannot be located
 *          - The service is deleted from the multiplex (eg. PAT updated)
 *          - The service info is updated (ie. PMT updated).
 *          CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          errorFunc
 *          startSIQueryFunc
 *          delSectionFilterFunc
 *          addSectionFilterFunc
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          addSectionFilter
 *          stopSIQuery
 *          notifyObjectLoadEvent
 *          notifyCarouselLoadEvent
 *          deleteSectionFilter
 *          stopTimer
 * @param   instance DSMCC instance handle
 * @param   clDsmSIQueryRef Refererence for this query (value supplied at
 *          start time).
 * @param   clDsmSIUserData User data for this query (value supplied at
 *          start time).
 * @param   status Status of query results.
 * @param   pResult Pointer to results of query (if successful).
 * @return
 *          CLDSM_OK (0)
 *          The process SI query event was executed with no problems.
 *          CLDSM_ERR_INVALID_SIQUERY_REF
 *          Supplied clDsmSIQueryRef does not reference a valid query.
 *          CLDSM_ERR_INVALID_SIQUERY_STATUS
 *          Supplied status value is invalid here.
 *          CLDSM_ERR_SYSTEM_ADD_SECTION_FILTER,
 *          Problem with add section filter callback.
 */
E_DscError CDSM_SysProcessSIChangeEvent( H_DsmCoreInst instance,
   E_SIChangeEvent evnt, U16BIT service_id, U32BIT carousel_id );



/*******************************************************************************
* Process DSM-CC private section data
*
* NOTES:
*
* Call each time the Demux provides a private section from one of the
* requested PIDs (main TS data input of DSM-CC Core API).
*
* It is assumed that the calling environment has already CRC checked the
* section (if relevant) and only sections that pass this check will be
* supplied to clDsmSysProcessPrivateSection.
*
* The clDsmFilterRef and clDsmSfUserData inputs MUST be set to the values
* supplied when the filter for this section was requested
* (via addSectionFilterFunc).
*
* If sections have to be queued (or discarded) in the calling environment
* then those matching filters with a high priority MUST always be supplied
* to this function in preference to those matching low priority filters.
*
* NB. Although sections matching low priority filters (or even high priority
* filters) can be discarded if necessary. The order in which they were
* broadcast should always be maintained when they are supplied to this
* function.
*
* CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
*   errorFunc
*   progressFunc
*   addSectionFilterFunc
*   delSectionFilterFunc
*   stopTimerFunc
*   startSIQueryFunc
*   notifyCarouselLoadEventFunc
*   notifyObjectLoadEventFunc
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_INSTANCE
* CLDSM_ERR_ILLEGAL_PARAMETER
* CLDSM_ERR_ABORTED           When dsmFilterRef is not recognised (maybe stale)
*
*******************************************************************************/
E_DscError CDSM_SysProcessPrivateSection( H_DsmCoreInst instance,
   /*I*/ U8BIT *pSection, H_DscSFRef dsmFilterRef );


/*******************************************************************************
* Request priority for DSM-CC private section data
*
*******************************************************************************/
E_DscError CDSM_SectionFilterPriority( H_DsmCoreInst instance,
   H_DscSFRef dsmFilterRef,
   E_SFPriority *pPriority );


/* ---------------------- SYNCHRONOUS CLIENT API CALLS ---------------------- */

/*
-- NOTES:
--
-- The following functions must be called synchronously with each other and
-- with the above System API calls. If they need to be called/triggered
-- from different threads the calling environment must ensure synchronous
-- behaviour - eg. by using a mutex semaphore round the call (or queueing and
-- calling from the same thread).
--
-- Any of the Synchronous (or Asynchronous) Client API calls (ie. API
-- calls that do NOT start with clDsmSys...) can be made from within the
-- notifyCarouselLoadEvent or notifyObjectLoadEvent callbacks. When doing
-- this, care must be taken to avoid blocking or re-entrancy problems in
-- the calling environment. Re-entrancy may occur when unloading objects or
-- carousels since this can cause a (re-entrant) call to these callbacks.
--
*/


/* -- CAROUSEL LOAD FUNCS -- */

/*******************************************************************************
*
* Loads DSM-CC object carousel specified by carousel_id from indicated service
* (on the current multiplex/transport stream)
*
* If carousel_id is set to INVALID_CAROUSEL_ID, then this function loads the
* initiate boot carousel for the service
*
* NOTES:
*
* The returned clDsmCarouselHandle value must always be supplied by the
* calling environment whenever accessing objects in this carousel.
*
* Initially a section filter will be set up to acquire the DSI message for the
* carousel. Once the DSI message is acquired the calling environment is
* notified via the notifyCarouselLoadEventFunc callback (if supplied at
* setup/create) that the carousel is BOOTED. Open object requests can then be
* made for this carousel.
*
* When the service gateway (SRG) is loaded for this carousel the calling
* environment is notified again via the callback that the carousel is now
* LOADED. Objects in the carousel cannot be accessed (opened) until the
* service gateway for the carousel is loaded.
*
* If timeout occurs before the SRG is loaded a timeout error is returned. If
* an error is encountered while loading the carousel then an aborted error is
* returned.
*
* Using the ocLoadUserData1 and ocLoadUserData2 values is optional. These
* values are returned unchecked into the load event callback. They can be used
* for any purpose (eg. to aid in matching callbacks with corresponding load
* requests and/or clients).
*
* If timeout value is set to zero it means no (ie. infinite) timeout.
*
* If a duplicate request is made to load a boot carousel then an error will be
* generated but the clDsmCarouselHandle will be set to the value of the already
* loaded carousel.
*
* CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
*   startTimerFunc
*   startSIQueryFunc
*   addSectionFilterFunc
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_DUPLICATE_REQUEST
* CLDSM_ERR_MEM_HEAP_FULL
* CLDSM_ERR_NO_CURRENT_SERVICE_SET
* ?
*
*******************************************************************************/
E_DscError CDSM_LoadCarousel( H_DsmCoreInst instance,
   U16BIT service_id, U32BIT carousel_id, E_SIQueryKind kind,
   H_DsmCarousel *pclDsmCarouselHandle );




/**
 * @brief   Unload (or cancel the requested load of) a DSM-CC Object Carousel (service
 *          domain).
 *          The client MUST first unload all objects before calling this function
 *          (which implies it must also have previously closed and/or unsubscribed any
 *          events on any loaded objects).
 *          If there are any active object loads in the carousel when
 *          clDsmUnloadCarousel is called then the unload will not be executed and
 *          an error returned.
 *          If the carousel is in the process of loading and/or booting the load
 *          operation will be aborted (and the notifyCarouselLoadEventFunc callback
 *          made with status OC_LOAD_ABORTED_UNLOAD).
 *          Any carousel load request that returned a valid CDSM_CarouselHandle(ie.
 *          did not generate an error at clDsmLoadCarousel call time), must be explicitly
 *          unloaded via clDsmUnloadCarousel. This is necessary, whether the carousel was
 *          successfully loaded or the load was aborted due to timeout or error.
 *          Once unload is called for a carousel, the clDsmCarouselHandle is no longer
 *          valid and must not be used by the client/calling env. If it needs to access
 *          the carousel subsequently it must re-load it and get a new
 *          clDsmCarouselHandle value.
 *        To forcefully destroy loaded objects and/or carousels (eg. where the handles
 *        may be unknown/lost) then clDsmReset can be used with force switches. Note
 *        that this physically deletes them from the cache whereas unloading (objects
 *        or carousels) does not.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          delSectionFilterFunc
 *          stopTimerFunc
 *          stopSIQueryFunc
 *          unsubscribeSIChange
 *          notifyCarouselLoadEventFunc
 * @param   instance DSMCC instance handle
 * @param   clDsmCarouselHandle Handle to carousel to be unloaded.
 * @return
 *          CLDSM_OK (0)
 *          Unload was executed with no problems.
 *          CLDSM_ERR_INVALID_CAROUSEL_HANDLE
 *          Unload was not executed because an invalid carousel handle was supplied.
 *          CLDSM_ERR_OBJECTS_STILL_LOADED
 *          Unload was not executed because there were still active object loads.
 */
E_DscError CDSM_UnloadCarousel( H_DsmCoreInst instance,
   /*I*/ H_DsmCarousel clDsmCarouselHandle, E_DsmRstMode mode );

/**
 * @brief   Unload SSU module. To be called once the client has finished saving
 *          this module data.
 * @param   idp DSMCC instance handle.
 * @param   hCarousel Handle to the DSMCC Update Carousel
 * @param   moduleRef Reference to a module
 * @return  error type
 */
E_DscError CDSM_UnloadModule( H_DsmCoreInst idp, H_DsmCarousel hCarousel,
   U32BIT moduleRef );

 /**
 * @brief   Retrieve Carousel Id for the loaded DSM-CC Object Carousel
 * @param   instance DSMCC instance handle
 * @param   clDsmCarouselHandle Handle to carousel to be unloaded.
 * @param   pCarouselId Carousel ID
 * @return
 *          CLDSM_OK
 *          CLDSM_ERR_INVALID_CAROUSEL_HANDLE
 */
E_DscError CDSM_GetCarouselId( H_DsmCoreInst instance,
   /*I*/ H_DsmCarousel carouselHandle,
   /*O*/ U32BIT *pCarouselId );

/**
 * @brief   Retrieve handle to the current carousel for DSM-CC Instance
 * @param   instance DSMCC instance handle
 * @return  H_DsmCarousel - handle to current carousel
 */
H_DsmCarousel CDSM_CurrentCarousel( H_DsmCoreInst instance );

/**
 * @brief   Sets current carousel for DSM-CC Instance
 * @param   instance DSMCC instance handle
 * @param   carouselHandle Handle to carousel
 * @return
 *          CLDSM_OK
 *          CLDSM_ERR_INVALID_CAROUSEL_HANDLE
 */
E_DscError CDSM_SetCurrentCarousel( H_DsmCoreInst instance,
   /*I*/ H_DsmCarousel carouselHandle );

/**
 * @brief   Retrieve from Object Carousel's User Info, the "File Group Info" used by
 *          Australia and South Africa MHEG profiles
 * @param   instance DSMCC instance handle
 * @param   clDsmCarouselHandle Handle to carousel.
 * @param   total total number of file groups
 * @param   pGroups array of File Group Info
 * @return
 *          CLDSM_OK
 *          CLDSM_ERR_INVALID_CAROUSEL_HANDLE
 */
E_DscError CDSM_CarouselLoadFileGroups( H_DsmCoreInst instance,
   /*I*/ H_DsmCarousel clDsmCarouselHandle,
   /*O*/ U16BIT *total, S_CarouselInfoFileGroup **pGroups );

/**
 * @brief   Release "File Group Info" data allocated by used by
 *          CDSM_CarouselLoadFileGroups()
 * @param   instance DSMCC instance handle
 * @param   clDsmCarouselHandle Handle to carousel to be unloaded.
 * @param   groups array of File Group Info
 * @return
 *          CLDSM_OK (0)
 *          Unload was executed with no problems.
 *          CLDSM_ERR_INVALID_CAROUSEL_HANDLE
 *          Unload was not executed because an invalid carousel handle was supplied.
 */
E_DscError CDSM_CarouselUnloadFileGroups( H_DsmCoreInst instance,
   /*I*/ H_DsmCarousel clDsmCarouselHandle,
   /*I*/ S_CarouselInfoFileGroup *groups );

/* -- OBJECT LOAD FUNCS -- */

/*******************************************************************************
*
* Load an object with the requested pathname into cache.
*
* Unless the carousel is specifed in the path, this will load an object from
* the 'Current' object carousel. The client may specify another carousel by
* setting the 'Current' object carousel with use of CDSM_SetCurrentCarousel().
* If the path does specify a new carousel (as defined by "ETSI TS 102 851"),
* then the function will attempt to load this carousel and make it the new
* 'Current' object carousel.
*
* NOTES:
*
* Currently, an object can only be loaded once the carousel is booted (DSI
* message acquired).
*
* Requested pathname must be a full (absolute) object pathname starting with
* an object in the root (ie. service gateway) directory of the specified
* carousel, eg. dir1/dir2/file3 is treated as DSM:/dir1/dir1/file3. Any
* characters defined in PATH_SEPERATOR_LIST_STRING will be treated as path
* token seperators all other chars are treated as part of a path token. An
* empty pathname U8BIT* (or U8BIT* containing only path token seperators)
* will open the service gateway directory itself.
*
* **** NB. SRG (& directory) ACCESS NOT YET IMPLEMENTED ****
*
* Returns an clDsmObjectHandle which the client uses to access the requested
* object. If the object is already cached then the status parameter will be
* set to OBJ_LOAD_COMPLETED. In this case the objectRef value can be used to
* access (eg. open for reading) the object/file data immediately.
*
* If the object is not in the cache then a fetch from the input stream is
* initiated and the status parameter set to OBJ_LOAD_IN_PROGRESS. When the
* object is loaded into the cache the calling environment is notified via the
* notifyObjectlLoadEventFunc callback (if supplied at setup/create).
*
* Using the objLoadUserData1 and objLoadUserData2 values is optional. These
* values are returned unchecked into the load event callback. They can be used
* for any purpose (eg. to aid in matching callbacks with corresponding load
* requests and/or clients).
*
* If multiple load requests are made for the same object before it is
* available each request will initiate a new (parallel) fetch from the input
* stream and a callback will be generated for each request when the object is
* loaded.
*
* While an object is LOADED it's contents are guaranteed to remain available
* and constant in cache memory (ie. it will not be updated from the input
* stream or deleted to create space in the cache).
*
* Any load request for an object (that is not already loaded) will only get
* completed by a call to clDsmysProcessPrivateSection. This will either have
* delivered the last part of the module containing the object or delivered a
* module containing a previously missing higher level directory in the object
* path that results in the requested object now being accessible in the cache.
*
* If a load proceeds part way down a path and then encounters an error, the
* load is aborted and the appropriate status returned via the callback. The
* clDsmObjectHandle cannot be used to access object/file DATA until/unless
* it is successfully loaded.
*
* **** NB. ONLY CACHING RULE 'fromStream' CURRENTLY IMPLEMENTED ****
* If caching rules are specified they are applied as below. If pCachingRules
* is set to Null the object will be given the default caching rules.
*
* If the object is not in the cache then the caching rules only have an
* effect on the subsequent caching of the object/module once the object load
* is completed and the object is actually unloaded (via clDsmUnloadObject),
* ie. when it is made available for caching.
*
* If the same object is requested multiple times in parallel, then once the
* object load is completed the highest priority caching rules are selected
* (fromStream overrides any other settings). These are applied only when
* the last of the object load instances is unloaded.
*
* If the requested object is already in the cache and the caching rules have
* fromStream set, then the object/module is immediately deleted from the cache
* and a new fetch from the input stream is initiated.
*
* If the same object is loaded multiple times in parallel then the caching
* rules specified at each load call will only replace the currently set
* caching rules for this object if they specify a higher caching priority
* (fromStream overrides any priority setting). If fromStream is set on any
* load then it will cause a re-fetch of the object for that load instance.
* The data for any previous load instances will only get physically deleted
* when they are unloaded.
*
* NB. For module based caching (as currently implemented) the whole module
* inherits the caching rules MOST RECENTLY SPECIFIED for any object in that
* module.
*
* The rules for when multiple objects from the same module are requested (or
* loaded) in parallel are applied to a module on the same basis as described
* for multiple instances of the same object.
*
* **** NOT YET IMPLEMENTED ****
* If timeout value is set to zero it means no (ie. infinite) timeout. If a
* timeout is specified and the object has not been loaded in the specified
* time (for any reason) then the load request is aborted and the client
* notified. Timeout is expressed in periods of timerResolution (specified at
* setup/create).
*
*
* CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
*   addSectionFilterFunc
*   startTimerFunc
*   startSIQueryFunc
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_MEM_HEAP_FULL
* CLDSM_ERR_INVALID_CAROUSEL_HANDLE
* CLDSM_ERR_INVALID_CACHING_RULES
* CLDSM_ERR_CAROUSEL_NOT_BOOTED
*
* CLDSM_ERR_INVALID_PATHNAME    - Null pointer or path too long
*
* CLDSM_ERR_LOAD_FAILURE        - Failed to load requested file (because
*                                 some part of path could not be located)
* +?
*
*******************************************************************************/
E_DscError CDSM_LoadObject( H_DsmCoreInst instance,
   /*I*/ U8BIT *pathname, U32BIT timeout, E_CacheRules cachingRules,
   H_ObjUserData pUserData, U32BIT sizeOfUserData,
   /*O*/ E_ObjLoadStatus *pStatus, H_DsmObject *pclDsmObjectHandle );



/**
 * @brief   Unload (or cancel the requested load of) a DSM-CC object
 *          The client MUST first close and/or unsubscribe any events on the object
 *          before calling this function.
 *          If the object is open or there are any active subscribed events on it when
 *          clDsmUnloadObject is called then the unload will not be executed and
 *          an error returned.
 *          If the object load is still in progress the load operation will be aborted
 *          (and the notifyObjectLoadEventFunc callback made with status
 *          OBJ_LOAD_ABORTED_UNLOAD).
 *          If the object load is completed then unloading the object does not
 *          (necessarily) mean it (or it's parent module) is deleted from the cache.
 *          It does mean the cached object (or parent module) can now be updated from
 *          the input stream or deleted from the cache when creating space for new items.
 *          All object load requests that return a valid CDSM_ObjectHandle(ie. did not
 *          generate an error at clDsmLoadObject call time) must be explicitly unloaded
 *          via clDsmUnloadObject. This is mandatory, whether the object was
 *          successfully loaded or the load was aborted due to timeout or error.
 *          Once unload is called for an object, the clDsmObjectHandle is no longer
 *          valid and must not be used by the client/calling env. If it needs to access
 *          the object subsequently it must re-load it and get a new clDsmObjectHandle
 *          value.
 *          **** NB. ONLY CACHING RULE 'fromStream' CURRENTLY IMPLEMENTED ****
 *          Unloading an object causes the object's current caching rules to be
 *          applied and these affect whether or not it is subsequently kept in
 *          the cache. If fromStream is set, the object (or parent module) is deleted
 *          from the cache.
 *          If the same object is loaded multiple times in parallel, then the caching
 *          rules are applied only when the last of the object load instances is unloaded.
 *          NB. For module based caching (as currently implemented) the whole module
 *          inherits the caching rules MOST RECENTLY SPECIFIED for any object in that
 *          module.
 *          The rules for when multiple objects from the same module are requested (or
 *          loaded) in parallel are applied to a module on the same basis as described
 *          for multiple instances of the same object.
 *        To forcefully destroy loaded objects and/or carousels (eg. where the handles
 *        may be unknown/lost) then clDsmReset can be used with force switches. Note
 *        that this physically deletes them from the cache whereas unloading (objects
 *        or carousels) does not.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          delSectionFilterFunc
 *          stopTimerFunc
 *          stopSIQueryFunc
 *          notifyObjectLoadEventFunc
 * @param   instance DSMCC instance handle
 * @param   clDsmObjectHandle Handle to object to be unloaded.
 *          mode                  IN
 * @return
 *          CLDSM_OK (0)
 *          Unload was executed with no problems.
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          Unload was not executed because an invalid object handle was supplied.
 *          CLDSM_ERR_OBJECT_OPEN
 *          Unload was not executed because the object was still open.
 *          CLDSM_ERR_STREAM_EVENTS_SUBSCRIBED
 *          Unload was not executed because there were still subscribed events on this
 *          object.
 */
E_DscError CDSM_UnloadObject( H_DsmCoreInst instance,
   /*I*/ H_DsmObject clDsmObjectHandle, E_DsmRstMode mode );


/**
 * @brief   Open object data for subsequent access via API functions for given object kind.
 *          The object must have already been loaded.
 *          Data that is specific to object kind is read and possibly stored here.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          none
 * @param   instance Instance of DSMCC containing object
 * @param   clDsmObjectHandle Handle to the Object
 * @param   pKind Populated with the object kind
 * @return
 *          CLDSM_OK
 *          The object was successfully opened
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The handle is not a valid object handle
 *          CLDSM_ERR_OBJECT_NOT_LOADED
 *          The object has not been loaded
 *          CLDSM_ERR_OPEN_OBJECT_LIMIT
 *          The maximum number of open objects allowed has been reached
 *          CLDSM_ERR_OBJECT_OPEN
 *          The object is already open
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The object is not a 'Stream Object', or a 'Stream Object with Events'
 *          or a 'File Object'.
 *          CLDSM_ERR_END_OF_DATA
 *          The object has zero length.
 */
E_DscError CDSM_OpenObject(
   /*I*/ H_DsmCoreInst instance,
   /*I*/ H_DsmObject clDsmObjectHandle,
   /*O*/ E_DsmObjectKind *pKind );




/*******************************************************************************
* Close a loaded object
*
* NOTES:
*
* Closes the loaded object data. If an object is open this must be called
* before it can be unloaded (via clDsmUnloadObject).
*
*
* CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
*   none
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* ?
*
*******************************************************************************/
E_DscError CDSM_CloseObject( H_DsmCoreInst instance,
   /*I*/ H_DsmObject clDsmObjectHandle );


/**
 * @brief   Get DSM instance to which the object belongs.
 * @param   clDsmObjectHandle DSM object handle
 * @return  NULL if an invalid object handle, otherwise returns instance handle
 */
H_DsmCoreInst CDSM_ObjectGetInstance(H_DsmObject clDsmObjectHandle);


/*******************************************************************************
* **** NB. ONLY CACHING RULE 'fromStream' CURRENTLY IMPLEMENTED ****
*
* Set caching rules for any (cached) object.
*
* Unless the carousel is specifed in the path, this will load an object from
* the 'Current' object carousel. The client may specify another carousel by
* setting the 'Current' object carousel with use of CDSM_SetCurrentCarousel().
* If the path does specify a new carousel (as defined by "ETSI TS 102 851"),
* then the function will attempt to load this carousel and make it the new
* 'Current' object carousel.
*
* NOTES:
*
* If the object (OR ANY DIRECTORY ON THE PATH TO THE OBJECT) is not found in
* the cache this command is ignored. Object load failure problems detected at
* call time are not indicated.
*
* If the object is located in the cache the new caching rules are applied
* immediately. If fromStream is set the object is deleted from the cache.
*
* If the object is found in the cache but it is currently loaded the current
* caching rules for this object are set to the specified values (whatever they
* were previously). If fromStream is set, the current object load instance is
* marked for deletion, ie. it is no longer available to subsequent loads
* (although the data is only physically deleted when the object is unloaded
* via clDsmUnloadObject).
*
* Other caching rules are only applied when the object is unloaded.
* If the same object is loaded multiple times in parallel, then these caching
* rules are applied when the last of the object load instances is unloaded.
* Any caching rules set by this call may get overriden if a subsequent parallel
* load of the object that specifies higher priority caching rules is made.
*
* NB. For module based caching (as currently implemented) the whole module
* inherits the caching rules MOST RECENTLY SPECIFIED for any object in that
* module.
*
* The rules for when multiple objects from the same module are requested (or
* loaded) in parallel are applied to a module on the same basis as described
* for multiple instances of the same object.
*
*
* CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
*   delSectionFilterFunc
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_PATHNAME
* CLDSM_ERR_INVALID_CACHING_RULES
* CLDSM_ERR_INVALID_CAROUSEL_HANDLE
* CLDSM_ERR_CAROUSEL_NOT_BOOTED
* CLDSM_ERR_CAROUSEL_LOAD_FAILED
* ?
*
*******************************************************************************/
E_DscError CDSM_SetObjectCachingRules( H_DsmCoreInst instance,
   /*I*/ U8BIT *pathname, E_CacheRules cachingRules );



/*******************************************************************************
*
* Request pre-fetch into cache of the specified object.
*
* Unless the carousel is specifed in the path, this will load an object from
* the 'Current' object carousel. The client may specify another carousel by
* setting the 'Current' object carousel with use of CDSM_SetCurrentCarousel().
* If the path does specify a new carousel (as defined by "ETSI TS 102 851"),
* then the function will attempt to load this carousel and make it the new
* 'Current' object carousel.
*
* NOTES:
*
* The pre-fetch is always 'silent', ie. no notification is given when (or
* whether) the object has been pre-fetched into the cache (ie. object load
* failure errors detected at start time are also not indicated). If the object
* is already in the cache then the pre-fetch request is ignored.
*
* Pre-fetches work like load requests (ie. they are always performed) but
* if necessary, load requests (clDsmLoadObject) have priority on section filter
* resources.
*
* If the object has to be fetched into the cache then it is given the default
* caching rules (nb. fromStream is not set!).
*
* **** NOT YET IMPLEMENTED ****
* If timeout value is set to zero it means no (ie. infinite) timeout. If a
* timeout is specified and the object has not been pre-fetched in the specified
* time (for any reason) then the pre-fetch request is abandoned (silently).
* Timeout is expressed in periods of timerResolution (specified at
* setup/create).
*
*
* CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
*   addSectionFilterFunc
*   startTimerFunc
*   startSIQueryFunc
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_MEM_HEAP_FULL
* CLDSM_ERR_INVALID_PATHNAME
* CLDSM_ERR_INVALID_CAROUSEL_HANDLE
* CLDSM_ERR_CAROUSEL_NOT_BOOTED
* CLDSM_ERR_CAROUSEL_LOAD_FAILED
* ?
*
*******************************************************************************/
E_DscError CDSM_PrefetchObject( H_DsmCoreInst instance,
   /*I*/ U8BIT *pathname, U32BIT timeout );



/**
 * @brief   The Client uses this function to request that the DSMCC notifies it when a
 *          named stream event occurs. The notification is implemented by means of a
 *          callback function defined by the Client.
 *          Refer to the description of F_NotifyStreamEvent for information about
 *          the callback function.
 *          Before calling this function the Client must request that the stream object of
 *          interest be loaded using the CDSM_ObjectLoad() function. This action will
 *          deliver a handle to the object stream - this is passed as the streamObject
 *          argument to this function.
 *          Do not have to wait until load is completed before subscribing.
 *          Do not have to open an object before subscribing (since synchronous function).
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          none
 * @param   dsmccInstance Allows support >= 1 instance of a DSMCC.
 * @param   streamObject This references a BIOP::StreamEventMessage as well
 *          as indicating the carousel, module and the kind of
 *          the object.
 * @param   eventName Name of stream event required by the Client.
 * @param   userData1 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   userData2 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   pEventHandle The is the handle to the stream event returned by
 *          subscribe. The caller will use this handle when the
 *          stream event being subscribed to is referred
 *          to in the future.
 * @return
 *          CLDSM_ERR_INVALID_INSTANCE
 *          The Instance value is invalid
 *          CLDSM_ERR_INVALID_STREAM_EVENT_NAME
 *          The supplied event name is invalid
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The supplied object handle is invalid
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The message referred to by streamObject is not a BIOP::StreamEventMessage
 *          CLDSM_ERR_NO_STREAM_EVENT_NOTIFY_CALLBACK
 *          If Stream Events are to be used then a callback must be registered.
 *          CLDSM_ERR_MEM_HEAP_FULL
 *          Out of memory.
 *          CLDSM_OK
 *          The event has been successfully subscribed to.
 */
E_DscError CDSM_StreamEventSubscribe(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmObject streamObject,
   /*I*/ U8BIT *eventName,
   /*I*/ void *userData1,
   /*I*/ void *userData2,
   /*O*/ H_DsmEvent *pEventHandle );


/**
 * @brief   This function is called by the Client to let DSMCC know when a previously
 *          subscribed stream event is no longer needed.
 *          Callback made to acknowledge unsubscribe.
 *          All data relating to this stream event will no longer be valid, including
 *          the eventHandle.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          none
 * @param   dsmccInstance Allows support of >= 1 instance of a DSMCC.
 * @param   eventHandle This is a handle to an entry which is to be
 *          deleted from a list of subscribed to stream events
 * @return
 *          CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE
 *          CLDSM_OK
 */
E_DscError CDSM_StreamEventUnsubscribe(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmEvent eventHandle );


/**
 * @brief   Like clDsmStreamEventSubscribe, except that no stream object is required.
 *          This function subscribes to events specified by event ID, and association
 *          tag for the current service.
 * @param   dsmccInstance Allows support >= 1 instance of a DSMCC.
 * @param   associationTag Association tag used to retrieve PID that contains
 *          required stream events
 * @param   eventId Stream event ID required by the Client.
 * @param   userData1 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   userData2 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   pEventHandle The is the handle to the stream event returned by
 *          subscribe. The caller will use this handle when the
 *          stream event being subscribed to is referred
 *          to in the future.
 * @return
 *          CLDSM_ERR_INVALID_INSTANCE
 *          The Instance value is invalid
 *          CLDSM_ERR_INVALID_STREAM_EVENT_NAME
 *          The supplied event name is invalid
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The supplied object handle is invalid
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The message referred to by streamObject is not a BIOP::StreamEventMessage
 *          CLDSM_ERR_NO_STREAM_EVENT_NOTIFY_CALLBACK
 *          If Stream Events are to be used then a callback must be registered.
 *          CLDSM_ERR_MEM_HEAP_FULL
 *          Out of memory.
 *          CLDSM_ERR_NO_CURRENT_SERVICE_SET
 *          clDsmSysSetCurrService has not been called prior to calling this function
 *          CLDSM_OK
 *          The event has been successfully subscribed to.
 */
E_DscError CDSM_SpecialEventSubscribe(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ U16BIT associationTag,
   /*I*/ U16BIT eventId,
   /*I*/ void *userData1,
   /*I*/ void *userData2,
   /*O*/ H_DsmEvent *pEventHandle );


/**
 * @brief   Same as clDsmStreamEventUnsubscribe - used in conjuction with
 *          clDsmSpecialEventSubscribe.
 * @param   dsmccInstance Allows support of >= 1 instance of a DSMCC.
 * @param   eventHandle This is a handle to an entry which is to be
 *          deleted from a list of subscribed to stream events
 * @return
 *          CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE
 *          The eventHandle is not valid.
 *          CLDSM_OK
 *          The stream event was successfully unsubscribed
 */
E_DscError CDSM_SpecialEventUnsubscribe(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmEvent eventHandle );



/* ---------------------- ASYNCHRONOUS CLIENT API CALLS --------------------- */

/*
-- NOTES:
--
-- These functions can always be called asynchronously (ie. from a different
-- task/thread) since they can only access an individual object/file data area
-- which must have first been opened (& locked) by a previous synchronous API
-- call.
--
*/


/*******************************************************************************
* Get specified object kind (also returned by clDsmOpenObject)
*
* The object (clDsmObjectHandle) must be LOADED.
*
* NB. This can also be called to poll for when an object is loaded.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_OBJECT_NOT_LOADED
*
*******************************************************************************/
E_DscError CDSM_ObjectGetKind(
   /*I*/ H_DsmObject clDsmObjectHandle,
   /*O*/ E_DsmObjectKind *pKind );



/*******************************************************************************
* Get specified object carousel handle.
*
* The object (clDsmObjectHandle) must be LOADED.
*
* NB. This can also be called to poll for when an object is loaded.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
*
*******************************************************************************/
E_DscError CDSM_ObjectGetCarouselHandle(
   /*I*/ H_DsmObject clDsmObjectHandle,
   /*O*/ void **hCarouselHandle );



/* -- SERVICE GATEWAY ACCESS FUNCS -- */



/**
 * @brief   This function gets service context of DSM object (if available)
 *          Need by MHEG5 for specified service gateway (SRG) object.
 *          The Dsm object (clDsmObjectHandle) must be LOADED and OPEN.
 *          When serviceId cannot be found then
 *          matched then a pathname of '\0' is returned.
 *          On return from this function the current location within the SRG object
 *          message will remain the same as before the function was called. This means that
 *          this function can be called more than once in succession without reopening the
 *          memory sequence.
 * @param   clDsmSRGObjectHandle SRG object handle supplied when object was
 *          loaded (via clDsmLoadObject).
 * @param   serviceId ServiceID in a ServiceContextList in a
 *          ServiceGatewayMessage. For MHEG this is made
 *          from data_broadcast_id (hi-order bytes) and
 *          application_type_code (lo-order bytes).
 * @param   pSrvCtxtData Returned pointer of service context data
 * @param   pSrvCtxtLen Returned length of service context data
 * @return
 *          CLDSM_OK (0)
 *          The SRG object was successfully parsed for an MHEG5 initial object path
 *          (nb. this does not imply an initial object path was actually found).
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          clDsmSRGObjectHandle is not a valid object reference.
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The object is not an SRG.
 *          CLDSM_ERR_OBJECT_NOT_LOADED
 *          The object is not yet loaded.
 *          CLDSM_ERR_OBJECT_NOT_OPEN
 *          The object is not open.
 *          CLDSM_ERR_END_OF_DATA
 *          No matching service context for 'serviceId'
 */
E_DscError CDSM_ObjectGetServiceContext(
   /*I*/ H_DsmObject clDsmObjectHandle,
   /*I*/ U32BIT serviceId,
   /*O*/ U8BIT **pSrvCtxtData,
   /*O*/ U32BIT *pSrvCtxtLen);


/* -- FILE OBJECT ACCESS FUNCS -- */

/*******************************************************************************
* Get length of specified file data
*
* The file object (clDsmFileObjectHandle) must be LOADED and OPEN.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* ?
*
*******************************************************************************/
E_DscError CDSM_FileGetSize(
   /*I*/ H_DsmObject clDsmFileObjectHandle,
   /*O*/ U32BIT *pSize );



/*******************************************************************************
* Read specified number of bytes of data from (opened) file into
* destination address, starting at the current 'cursor' ('file' ptr) position.
*
* The file object (clDsmFileObjectHandle) must be LOADED and OPEN. The
* destination memory area MUST be large enough to hold the numBytes requested.
*
* If call attempts to read past end of the file data it will read as many
* bytes as possible and returns error code: CLDSM_ERR_END_OF_DATA
*
* Number of bytes read is stored in pNumBytesActual and 'cursor' position is
* incremented by numBytesActual.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* CLDSM_ERR_END_OF_DATA
* ?
*
*******************************************************************************/
E_DscError CDSM_FileRead(
   /*I*/ H_DsmObject clDsmFileObjectHandle, U32BIT numBytes,
   /*O*/ U8BIT *pDest, U32BIT *pNumBytesActual );

/*******************************************************************************
* Returns pointer to file data and number of bytes in the file.
*
* The file object (clDsmFileObjectHandle) must be LOADED and OPEN. The
* destination memory area MUST be large enough to hold the numBytes requested.
*
* Number of bytes read is stored in pNumBytesActual.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_OBJECT_NOT_LOADED
*
*******************************************************************************/
E_DscError CDSM_FileDirect(
   /*I*/ H_DsmObject clDsmFileObjectHandle,
   /*O*/ U8BIT **ppDest, U32BIT *pNumBytesActual );


/*******************************************************************************
* Read single byte of data from (opened) file at current 'cursor' position
* into destination and increment 'cursor' position.
*
* The file object (clDsmFileObjectHandle) must be LOADED and OPEN.
*
* If call attempts to read past end of the object data, returns error code:
* CLDSM_ERR_END_OF_DATA
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* CLDSM_ERR_END_OF_DATA
* ?
*
*******************************************************************************/
E_DscError CDSM_FileReadByte(
   /*I*/ H_DsmObject clDsmFileObjectHandle,
   /*O*/ U8BIT *pDest );



/*******************************************************************************
* Set absolute file data 'cursor' position
*
* The file object (clDsmFileObjectHandle) must be LOADED and OPEN.
*
* Cursor can point to any byte in file data AND to notional 'position'
* immediately after last byte of data but setting to anywhere else will return
* error code: CLDSM_ERR_END_OF_DATA
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* CLDSM_ERR_END_OF_DATA
* ?
*
*******************************************************************************/
E_DscError CDSM_FileSetPosAbs(
   /*I*/ H_DsmObject clDsmFileObjectHandle, U32BIT absPosition );



/*******************************************************************************
* Set file data 'cursor' position relative to current position
*
* The file object (clDsmFileObjectHandle) must be LOADED and OPEN.
*
* Cursor can point to any byte in file data AND to notional 'position'
* immediately after last byte of data but setting to anywhere else will return
* error code: CLDSM_ERR_END_OF_DATA
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* CLDSM_ERR_END_OF_DATA
* ?
*
*******************************************************************************/
E_DscError CDSM_FileSetPosRel(
   /*I*/ H_DsmObject clDsmFileObjectHandle, S32BIT relPosition );



/*******************************************************************************
* Read (absolute) file data 'cursor' position
*
* The file object (clDsmFileObjectHandle) must be LOADED and OPEN.
*
* Value returned in 'position' is the offset (in bytes) from the start (zero
* position) of the file data to the current cursor position.
*
* If the current cursor position points to past the end of the file data, then
* the position returned is equal to the file data size. Note that it is not
* legal to _read_ data from this cursor position/offset.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* ?
*
*******************************************************************************/
E_DscError CDSM_FileGetPos(
   /*I*/ H_DsmObject clDsmFileObjectHandle,
   /*O*/ U32BIT *pPosition );


/* -- DIRECTORY ACCESS FUNCS -- */

/*******************************************************************************
*
* The directory object (dirObj) must be LOADED and OPEN.
*
* Retrieves the total number of directory entries and the total length of all
* entry names.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
*
*******************************************************************************/
E_DscError CDSM_DirEntrySizes(
   /*I*/ H_DsmObject dirObj,
   /*O*/ U16BIT *nameCount,
   /*O*/ U16BIT *totalNameLength );

/*******************************************************************************
*
* The directory object must be LOADED and OPEN.
*
* Retrieve the first entry handle
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* CLDSM_ERR_END_OF_DATA
*
*******************************************************************************/
E_DscError CDSM_DirEntryFirst(
   /*I*/ H_DsmObject dirObj,
   /*O*/ void **pFirstEntry );

/*******************************************************************************
*
* Directory entry handle (entryHandle) is return by dsmDirEntryFirst or
* dsmDirEntryNext.
*
* Retrieve the next entry handle
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INTERNAL
*
*******************************************************************************/
E_DscError CDSM_DirEntryNext(
   /*I*/ void *entryHandle,
   /*O*/ void **pNextEntry );

/*******************************************************************************
*
* The directory entry handle is return by dsmDirEntryFirst or
* dsmDirEntryNext.
*
* Retrieve name length
*
* RETURNS:
* Length of name copied
*
*******************************************************************************/
U16BIT CDSM_DirEntryNameLength(
   /*I*/ void *entryHandle );

/*******************************************************************************
*
* The directory entry handle is return by dsmDirEntryFirst or
* dsmDirEntryNext.
*
* Copy name for entry to supplied memory location (pName)
* Caller should ensure there is enough space in location
*
* RETURNS:
* Length of name copied
*
*******************************************************************************/
U16BIT CDSM_DirEntryNameCopy(
   /*I*/ void *entryHandle,
   /*O*/ U8BIT *pName );

/*******************************************************************************
*
* Get the kind of entry this is
*
* RETURNS:
* E_DsmObjectKind
*
*******************************************************************************/
E_DsmObjectKind CDSM_DirEntryKind( void *entryHandle );


/* -- STREAM OBJECT ACCESS FUNCS -- */

/**
 * @brief   The Client calls this function to obtain the Association Tag which can be
 *          used to determine the PID on which stream data can be found.
 *          Before calling this function the Client must have completed loading and opened
 *          the stream object.
 * @param   dsmccInstance Allows support >= 1 instance of a DSMCC.
 * @param   streamObject This handle indicates the stream object
 * @param   userData1 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   userData2 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   ppDeferredService return a pointer to deferred service structure
 * @return
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The supplied object handle was not valid.
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The streamObject does not refer to data containing the association tag.
 *          CLDSM_ERR_OBJECT_NOT_LOADED
 *          The streamObject does not refer to a loaded object.
 *          CLDSM_ERR_OBJECT_NOT_OPEN
 *          The streamObject does not refer to an open object.
 *          CLDSM_ERR_UNABLE_TO_GET_PROGRAM_ASSOC_TAG
 *          No BIOP_PROGRAM_USE tap could be found.
 *          CLDSM_PENDING
 *          Deferred service request is pending, and will be given in callback
 *          CLDSM_OK
 *          Deferred service successfully obtained.
 */
E_DscError CDSM_StreamGetDeferredService(  H_DsmCoreInst instance,
   /*I*/ H_DsmObject streamObject, void *userData1, void *userData2,
   /*O*/ S_DvbLocator **ppDeferredService );

/* -- STREAM EVENT ACCESS FUNCS -- */

/**
 * @brief   The Client uses this function to request list of event names for stream object
 *          Before calling this function the Client must request that the stream object of
 *          interest be loaded using the CDSM_ObjectLoad() function. This action will
 *          deliver a handle to the object stream - this is passed as the streamObject
 *          argument to this function.
 *          Do not have to wait until load is completed
 *          Do not have to open an object
 * @param   dsmccInstance Allows support >= 1 instance of a DSMCC.
 * @param   streamObject This references a BIOP::StreamEventMessage as well
 *          as indicating the carousel, module and the kind of
 *          the object.
 * @param   pNamesPtr Pointer to comma separated list of Event names
 * @param   pNamesLen Length of comma separated list of Event names
 * @return
 *          CLDSM_ERR_INVALID_INSTANCE
 *          The Instance value is invalid
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The supplied object handle is invalid
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The message referred to by streamObject is not a BIOP::StreamEventMessage
 *          CLDSM_ERR_MEM_HEAP_FULL
 *          Out of memory.
 *          CLDSM_OK
 *          Successfully retrieved list of names for this stream object.
 */
E_DscError CDSM_StreamEventNameList(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmObject streamObject,
   /*O*/ U8BIT **pNamesPtr, U16BIT *pNamesLen );

/**
 * @brief   The Client uses this function to request data in XML format for stream object and
 *          associated events. This XML format is defined by 'TS 102 809' v 1.2.1, section 8.2
 *          Before calling this function the Client must request that the stream object of
 *          interest be loaded using the CDSM_ObjectLoad() function.
 * @param   dsmccInstance instance of DSMCC.
 * @param   streamObject handle to stream object
 * @param   pXmlPtr Pointer to XML data for Stream Object
 * @param   pXmlLen Length of XML data for Stream Object
 * @return  CLDSM_ERR_INVALID_INSTANCE - The Instance value is invalid
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE - The supplied object handle is invalid
 *          CLDSM_ERR_INVALID_OBJECT_TYPE - streamObject is not a BIOP::StreamEventMessage
 *          CLDSM_ERR_MEM_HEAP_FULL - Out of memory.
 *          CLDSM_OK - Successfully retrieved XML data for stream object.
 */
E_DscError CDSM_StreamEventXmlData(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmObject streamObject,
   /*O*/ U8BIT **ppXmlData, U16BIT *pXmlLen);

U32BIT CDSM_UtilCalculateCRC(U8BIT *msg, U32BIT len);

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif
#endif /* _DSMCC_H_ */

