/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Dynamic cache memory management funcs
 * @file    cacheMgr.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "cacheMgr.h"

#include "linkList.h"
#include "loadMgr.h"
#include "module.h"


/*------------------------------- Local Macros -------------------------------*/



/*------------------------------  Exported Data  -----------------------------*/



/*--------------------------------Local Types --------------------------------*/



/*------------------------------- Local Statics ------------------------------*/

/*------------------- local prototypes/forward declarations ------------------*/


/*---------------------------- Exported Functions ----------------------------*/

BOOLEAN DSC_CmMemPurgeCache( P_DsmCoreInst idp, U32BIT maxMemory )
{
   P_Module pModule;

   do
   {
      /* -- Find lowest priority cached module to delete and free up some
         -- heap space.
         -- NB. Can only delete CACHED modules in priority list to avoid
         --     error condition of this memGet trying to create space for
         --     a load or update of the module being deleted (can only
         --     occur when turboCaching enabled) */

      pModule = (P_Module)LLTail( idp->llcModulePriority );

      while (pModule != NULL)
      {
         if (pModule->status == MS_CACHED && pModule->currentUseCount == 0)
         {
            break;
         }
         pModule = LLPrev( pModule, MODULE_PRIORITY_LIST );
      }
      if (pModule == NULL)
      {
         dsmDP1(("ERROR: memMgr heap full\n"));
         return FALSE;
      }

      /* -- NB. Module should not be loaded */
      dsmAssert((pModule->loadedCount == 0));

      /* -- NB. Module should not have any loadRequests */
      dsmAssert((pModule->llcLoadRequests == NULL));

      dsmDP2(("INFO: ModuleId %u deleted to free cache space\n", pModule->moduleInfo.moduleId));

      /* -- Notify the first time the cache fills up (on any
         -- given service) */
      if (idp->cacheFull == FALSE)
      {
         dsmDP3(("INFO: *** Cache full on current service ***\n"));
         idp->cacheFull = TRUE;
         if (idp->setup.progressFunc)
         {
            idp->setup.progressFunc( CLDSM_PROG_CURR_SERVICE_CACHE_FULL, /*args*/ NULL );
         }
      }
      DSC_ModuleDeleteDcTidyUp( idp, pModule );
   }
   while (idp->cacheMemoryUsage > maxMemory);
   return TRUE;
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_CmMemGet
//
// Tries to allocate size bytes of memory from memMgr. If memMgr heap is full
// it will delete 'old' modules from the cache until is successful (or there
// are no more modules it can delete).
//
// If memory cannot be allocated, it sets returned memory handle to NULL and
// returns error.
//
///////////////////////////////////////////////////////////////////////////// */
MemHandle DSC_CmMemAlloc( P_DsmCoreInst idp, U32BIT size, int line )
{
   MemHandle memptr;

   dsmDP4(("DSC_CmMemGet()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((size != 0));

   if (size > idp->setup.maxMemorySize)
   {
      ERRPRINT("ERROR: memMgr size too large %d", size)
      memptr = NULL;
   }
   else if (idp->cacheMemoryUsage + size > idp->setup.maxMemorySize &&
            !DSC_CmMemPurgeCache( idp, idp->setup.maxMemorySize - size ))
   {
      ERRPRINT("ERROR: unable to purge, to alloc size %d", size)
      memptr = NULL;
   }
   else
   {
      do
      {
         memptr = memAlloc( MEM_CONTEXT, &idp->setup, size, line );
         if (memptr)
         {
            idp->cacheMemoryUsage += size;
            if (idp->cacheMaximumUsage < idp->cacheMemoryUsage)
            {
               idp->cacheMaximumUsage = idp->cacheMemoryUsage;
               dsmDP3(("INFO: Maximum Memory Usage is %u\n", idp->cacheMaximumUsage));
            }
            break;
         }
      }
      while (DSC_CmMemPurgeCache( idp, idp->setup.maxMemorySize ));
   }
   return memptr;
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_CmMemRelease
//
///////////////////////////////////////////////////////////////////////////// */
void DSC_CmMemRelease( P_DsmCoreInst idp, MemHandle hMemArea )
{
   dsmDP4(("DSC_CmMemRelease()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((hMemArea != NULL));

   idp->cacheMemoryUsage -= memRelease( MEM_CONTEXT, &idp->setup, hMemArea );

   dsmDP4(("exit DSC_CmMemRelease\n"));
}

/*------------------------------ Local Functions -----------------------------*/


/*----------------------------------------------------------------------------*/
