/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2002 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Synchronous Client API functions and general purpose asynchronous
 *             client API functions (for DSM object access)
 * @file    clDsmClientMain.c
 * @date    7/2/2002
 * @author  N Kirkland
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>
#include "clDsmSystem.h"

#include "dsmObject.h"
#include "module.h"
#include "dataCarousel.h"
#include "objectCarousel.h"
#include "updateCarousel.h"
#include "loadMgr.h"
#include "sectionFilter.h"
#include "siQuery.h"
#include "clDsmUtils.h"
#include "moduleData.h"

/*------------------------------- Local Macros -------------------------------*/

#define PATH_SEPARATOR_LIST_STRING  "\\/:"  /* -- Recognised path seperators */

/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/

typedef struct
{
   U16BIT original_network_id;
   U16BIT transport_stream_id;
   U16BIT service_id;
   U16BIT component_tag;
   U32BIT carousel_id;
} S_CarouselLocator;


/*------------------------------- Local Statics ------------------------------*/


/*------------------- local prototypes/forward declarations ------------------*/

static E_DscError RetrieveCarouselPath( P_DsmCoreInst idp,
   U8BIT *ipPath, U8BIT *opPath,
   P_ObjectCarousel *ppOC );


/*---------------------------- Exported Functions ----------------------------*/

/**
 * @brief   Request load of a DSM-CC Object Carousel (service domain).
 */
E_DscError CDSM_LoadCarousel( H_DsmCoreInst instance,
   U16BIT serviceId, U32BIT couId, E_SIQueryKind kind,
   H_DsmCarousel *pclDsmCarouselHandle )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   E_DscError err;
   S_SIQueryRequest siQueryData;
   S_SIQueryResult siQueryResult;
   P_RootCarousel pRC;

   dsmDP2(("CDSM_LoadCarousel( %p,srvID: %u, carouselID: %u,  %u, %p )\n",
           instance, serviceId, couId, pclDsmCarouselHandle));

   if (!instance || !pclDsmCarouselHandle)
   {
      dsmDP1(("ERROR: Illegal parameter\n"));
      err = CLDSM_ERR_INVALID_INSTANCE;
   }
   else if (!idp->currentServiceSet)
   {
      err = CLDSM_ERR_NO_CURRENT_SERVICE_SET;
   }
   else if ((couId == INVALID_CAROUSEL_ID || couId == UNKNOWN_CAROUSEL_ID) &&
            idp->pBootingCarousel != NULL)
   {
      *pclDsmCarouselHandle = (H_DsmCarousel)idp->pBootingCarousel;
      err = CLDSM_DUPLICATE_REQUEST;
   }
   else
   {
      /* to see if carousel is already loaded (for multi-carousel).
      NB. Only one 'anonymous' carousel can be booted per program.
      */
      pRC = DSC_RootCrslListFindById( idp->llcRootCarousels, serviceId, couId );
      if (pRC != NULL)
      {
         *pclDsmCarouselHandle = (H_DsmCarousel)pRC;

         err = CLDSM_DUPLICATE_REQUEST;
      }
      else
      {
         /* -- Load a new carousel -- */
         *pclDsmCarouselHandle = NULL;
         switch (kind)
         {
            case SIQUERY_SSU_PID:
            {
               err = DSC_UpdCrslCreate( idp, 0, couId, (P_UpdateCarousel*)&pRC );
               break;
            }
            case SIQUERY_SSU_CAROUSEL:
            {
               err = DSC_UpdCrslCreate( idp, serviceId, couId, (P_UpdateCarousel*)&pRC );
               break;
            }
            default:
            {
               err = DSC_ObjCrslCreate( idp, serviceId, couId, (P_ObjectCarousel*)&pRC );
               break;
            }
         }
         if (!err)
         {
            dsmDP4(("carousel load created\n"));
            siQueryData.dataId = couId;
            siQueryData.serviceId = serviceId;
            siQueryData.kind = kind;

            pRC->status = RCS_PENDING_BOOTINFO;
            err = siQueryStart( idp, &siQueryData, (void *)pRC, &siQueryResult );
            switch (err)
            {
               case CLDSM_PENDING:
                  DBGLOG((DD_OC|DD_QUERY),"Query Handle=%p\n", siQueryResult.data.queryHandle)
                  dsmAssert((siQueryResult.kind == SIRESULT_PENDING));
                  /* -- Store query ref in carousel */
                  pRC->pPendingSiQueryRef = siQueryResult.data.queryHandle;
                  err = CLDSM_OK;
                  break;

               case CLDSM_OK:
                  DBGLOG((DD_OC|DD_QUERY), "Crsl PID=%d\n", siQueryResult.data.carouselInfo.pid)
                  err = DSC_RootCrslBootCarousel( idp, pRC, &(siQueryResult.data.carouselInfo));
                  if (!err)
                  {
                     break;
                  } /* else fall through to default error case*/

               default: /* ERROR */
                  if (kind == SIQUERY_SSU_CAROUSEL || kind == SIQUERY_SSU_PID)
                  {
                     DSC_UpdCrslDestroy( idp, (P_UpdateCarousel)pRC );
                  }
                  else
                  {
                     DSC_ObjCrslDestroy( idp, (P_ObjectCarousel*)&pRC );
                  }
            }

            if (!err)
            {
               *pclDsmCarouselHandle = (H_DsmCarousel)pRC;

               /* -- Insert Carousel in list */
               LLInsertTail( idp->llcRootCarousels, pRC );

               if (idp->pBootingCarousel == NULL)
               {
                  /* -- Store booting OC handle in instance data */
                  idp->pBootingCarousel = pRC;
               }
               if (idp->pCurrentCarousel == NULL)
               {
                  idp->pCurrentCarousel = pRC;
               }
            }
         }
      }
   }
   switch (err)
   {
      case CLDSM_OK:
         DBGLOG(DD_OC, "ptr=%p", *pclDsmCarouselHandle);
         break;
      case CLDSM_ERR_SI_QUERY_FAILED:
         DBG3(DD_OC, "SI query fail loading carousel")
         break;
      default:
         ERRPRINT("ERROR: clDsmLoadCarousel: %u", err);
   }
   return err;
}

/*******************************************************************************
*
* Unload (or cancel the requested load of) a DSM-CC Object Carousel (service
* domain).
*
*******************************************************************************/
E_DscError CDSM_GetCarouselId( H_DsmCoreInst instance, H_DsmCarousel hCarousel, U32BIT *pCarouselId )
{
   P_ObjectCarousel pOC;
   E_DscError err;

   dsmDP2(("CDSM_GetCarouselId( %p, %p )\n", instance, hCarousel));
   if (instance == NULL)
   {
      err = CLDSM_ERR_INVALID_INSTANCE;
   }
   else if (!memValidate((MemHandle)hCarousel))
   {
      err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
   }
   else
   {
      pOC = (P_ObjectCarousel)hCarousel;
      if (pOC->root.magic != OC_MAGIC)
      {
         ERRPRINT("API ERROR: Invalid Carousel handle (magic)")
         err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
      }
      else
      {
         dsmDP2(("CDSM_GetCarouselId() is %u\n", pOC->root.rcid));
         *pCarouselId = pOC->root.rcid;
         err = CLDSM_OK;
      }
   }
   return err;
}

H_DsmCarousel CDSM_CurrentCarousel( H_DsmCoreInst instance )
{
   return (H_DsmCarousel)instance->pCurrentCarousel;
}

E_DscError CDSM_SetCurrentCarousel( H_DsmCoreInst idp,
   /*I*/ H_DsmCarousel hCarousel )
{
   E_DscError err;

   dsmDP2(("CDSM_GetCarouselId( %p, %p )\n", idp, hCarousel));
   if (idp == NULL)
   {
      err = CLDSM_ERR_INVALID_INSTANCE;
   }
   else if (!memValidate((MemHandle)hCarousel))
   {
      err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
   }
   else
   {
      if (((P_ObjectCarousel)hCarousel)->root.magic != OC_MAGIC)
      {
         dsmDP1(("API ERROR: Invalid Carousel handle (magic)\n"));
         err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
      }
      else
      {
         idp->pCurrentCarousel = (P_RootCarousel)hCarousel;
         err = CLDSM_OK;
      }
   }
   return err;
}

E_DscError CDSM_CarouselLoadFileGroups( H_DsmCoreInst instance,
   H_DsmCarousel hCarousel, U16BIT *total,
   S_CarouselInfoFileGroup **pGroups )
{
   P_ObjectCarousel pOC;
   E_DscError err;

   dsmDP2(("CDSM_CarouselLoadFileGroups( %p, %p )\n", instance, hCarousel));
   if (instance == NULL)
   {
      err = CLDSM_ERR_INVALID_INSTANCE;
   }
   else if (!memValidate((MemHandle)hCarousel))
   {
      err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
   }
   else
   {
      pOC = (P_ObjectCarousel)hCarousel;
      if (pOC->root.magic != OC_MAGIC)
      {
         dsmDP1(("API ERROR: Invalid Carousel handle (magic)\n"));
         err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
      }
      else
      {
         switch (pOC->root.status)
         {
            case RCS_BOOTED:
            case RCS_LOADED:
               err = DSC_ObjCrslRetrieveFileGroups( pOC, total, pGroups );
               break;

            default:
               err = CLDSM_ERR_CAROUSEL_NOT_BOOTED;
         }
      }
   }
   return err;
}

/*******************************************************************************
*
* Unload DSM-CC Object Carousel File Group List
*
*******************************************************************************/
E_DscError CDSM_CarouselUnloadFileGroups( H_DsmCoreInst instance,
   /*I*/ H_DsmCarousel hCarousel,
   /*I*/ S_CarouselInfoFileGroup *groups )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   P_ObjectCarousel pOC;
   E_DscError err;

   dsmDP2(("CDSM_CarouselUnloadFileGroups( %p, %p, %p )\n", instance, hCarousel, groups));
   if (instance == NULL)
   {
      err = CLDSM_ERR_INVALID_INSTANCE;
   }
   else
   {
      err = CLDSM_OK;
      if (!memValidate((MemHandle)hCarousel))
      {
         DSC_ObjCrslReleaseFileGroups( idp, NULL, groups );
      }
      else
      {
         pOC = (P_ObjectCarousel)hCarousel;
         if (pOC->root.magic != OC_MAGIC)
         {
            dsmDP1(("API ERROR: Invalid Carousel handle (magic)\n"));
            DSC_ObjCrslReleaseFileGroups( idp, NULL, groups );
         }
         else
         {
            DSC_ObjCrslReleaseFileGroups( idp, pOC, groups );
         }
      }
   }
   return err;
}

/*******************************************************************************
*
* Unload (or cancel the requested load of) a DSM-CC Object Carousel (service
* domain).
*
*******************************************************************************/
E_DscError CDSM_UnloadCarousel( H_DsmCoreInst instance,
   /*I*/ H_DsmCarousel clDsmCarouselHandle, E_DsmRstMode mode )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   P_RootCarousel pRC = (P_RootCarousel)clDsmCarouselHandle;
   P_LoadRequest pLoadRequest;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_UnloadCarousel( %p, %p, %d )\n", instance, clDsmCarouselHandle, mode));

   dsmAssert((instance != NULL));

   if (!RC_VALID( pRC ))
   {
      dsmDP1(("API ERROR: Invalid Carousel handle (handle)\n"));
      err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
   }
   else if ( (pRC->magic == OC_MAGIC) &&
             DSC_ObjCrslUnloadObjects(idp,(P_ObjectCarousel)pRC,mode) != CLDSM_OK )
   {
      dsmDP1(("ERROR: Objects still loaded in Carousel\n"));
      err = CLDSM_ERR_OBJECTS_STILL_LOADED;
   }
   else
   {
      /* -- NULL booting carousel reference in instance data (in case
       -- it is still set) */
      idp->pBootingCarousel = NULL;
      if (idp->pCurrentCarousel == pRC)
      {
         idp->pCurrentCarousel = NULL;
      }

      switch (pRC->status)
      {
         case RCS_PENDING_BOOTINFO:
            /*
            -- In this state, the carousel is waiting for an
            -- pending SI Query to complete so abort the query.
             */
            dsmAssert((pRC->pPendingSiQueryRef != NULL));
            siQueryAbortPending( idp, pRC->pPendingSiQueryRef );

            /* -- Clear Query ref in carousel (finished with) */
            pRC->pPendingSiQueryRef = NULL;

         /* fall through to RCS_BOOTING */

         case RCS_BOOTING:
            /* -- OC loadRequest must exist in this state */
            /* -- No module load will be started in this state so stop
               -- the carousel load request */
            pLoadRequest = (P_LoadRequest)pRC->pLoadRqst;
            pLoadRequest->rlr.status = LRS_ABORTED_BY_REQUESTER;
            /* -- Ack carousel unload/abort */
            DSC_LoadRsqtFinalise( idp, (P_RootLoadRqst)pLoadRequest );
            break;

         case RCS_BOOTED:
            if (pRC->magic == OC_MAGIC)
            {
               /* -- In this state, SRG module load will be started.
                  -- Set load request status and abort module load. */
               /* -- OC loadRequest must exist */
               pLoadRequest = (P_LoadRequest)pRC->pLoadRqst;
               dsmAssert((pLoadRequest != NULL));
               pLoadRequest->rlr.status = LRS_ABORTED_BY_REQUESTER;
               lmStopModuleLoadRequest( idp, pLoadRequest );
            }
            break;

         case RCS_LOADED:
         case RCS_LOADING:
         case RCS_LOAD_FAILED:
            /* -- In these states, carousel load is already finalised so
               -- nothing specific to do */
            if (pRC->magic == OC_MAGIC)
            {
               if (idp->setup.notifyCarouselLoadEventFunc)
               {
                  idp->setup.notifyCarouselLoadEventFunc((H_DsmCarousel)pRC, OC_LOAD_ABORTED_UNLOAD, pRC->rcid);
               }
            }
            else
            {
               dsmAssert((pRC->magic == UC_MAGIC));
               if (idp->setup.ssuFuncs.status)
               {
                  U_StatusRef sr;
                  sr.id = pRC->rcid;
                  idp->setup.ssuFuncs.status((H_DsmCarousel)pRC, SSU_CRSL_UNLOAD, sr, 0 );
               }
            }
            break;

         default:
            /* -- Illegal status */
            dsmDP1(("ERROR: Illegal Carousel status = %u\n", pRC->status));
            dsmAssert((0));

            err = CLDSM_ERR_INTERNAL;
            break;
      }

      /*
      -- In this state, the DSI section filter will have been
      -- requested, so stop it (also Nulls reference in OC).
      */
      if (pRC->pDsiSf)
      {
         DSC_SectionFilterStop( idp, &pRC->pDsiSf );
      }

      if (!err)
      {
         DSC_RootCrslUnload( idp, pRC );
         DSC_RootCrslDestroy( idp, pRC );
      }

      if (idp->pCurrentCarousel == NULL)
      {
         /* Just unloaded current carousel -
          * find next loaded carousel and set it as current carousel */
         idp->pCurrentCarousel = LLHead( idp->llcRootCarousels );
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: clDsmUnloadCarousel: %u\n", err)));
   dsmDP2(("exit clDsmUnloadCarousel -> rtn: %u \n", err));
   return err;
}

/**
 * @brief   Unload SSU module. To be called once the client has finished saving
 *          this module data.
 * @param   idp DSMCC instance handle.
 * @param   hCarousel Handle to the DSMCC Update Carousel
 * @param   downloadId Download ID
 * @param   moduleId Module ID
 * @return  error type
 */
E_DscError CDSM_UnloadModule( H_DsmCoreInst idp, H_DsmCarousel hCarousel,
   U32BIT moduleRef )
{
   E_DscError err;
   P_RootCarousel pRC = (P_RootCarousel)hCarousel;
   if (!memValidate(pRC) || pRC->magic != UC_MAGIC)
   {
      ERRPRINT("not Update Carousel")
      err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
   }
   else
   {
      DSC_RootCrslUnloadModule( idp, pRC, moduleRef );
      err = CLDSM_OK;
   }
   DBGERRCHK(err)
   return err;
}



/*
-- Load a DSM-CC object
--
*/
E_DscError CDSM_LoadObject( H_DsmCoreInst idp,
   U8BIT *pathname, U32BIT timeout, E_CacheRules cachingRules,
   H_ObjUserData pUserData, U32BIT sizeOfUserData,
   E_ObjLoadStatus *pStatus, H_DsmObject *pclDsmObjectHandle )
{
   U8BIT path[MAX_PATH_NAME_SIZE];
   P_LoadRequest pLoadRequest;
   P_DsmObject pDsmObject;
   E_DscError err;
   P_ObjectCarousel pOC;

   dsmDP2(("CDSM_LoadObject( %p, %s, %u, %u, %p, %u, %p, %p )\n",
           idp, pathname, timeout, cachingRules,
           pUserData, sizeOfUserData, pStatus, pclDsmObjectHandle));

   API_PARAM_CHK( idp, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );
   API_PARAM_CHK( pStatus, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );
   API_PARAM_CHK( pclDsmObjectHandle, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );
   API_PARAM_CHK( pathname, dsmDP1(("ERROR: Invalid pathname\n")),
      err = CLDSM_ERR_INVALID_PATHNAME; goto _return );

   err = RetrieveCarouselPath( idp, pathname, path, &pOC );
   if (!err)
   {
      *pclDsmObjectHandle = NULL;
      *pStatus = OBJ_LOAD_UNINITIATED;

      err = objectCreate( idp, &pDsmObject );
      if (!err)
      {
         /* -- NB. Unless an error is encountered in this function, this object
            -- is held open until it is unloaded and the real memory pointer
            -- is passed to the client */

         pDsmObject->pObjCarousel = pOC;

         /* -- NB. dsmObject is created with default caching rules in-case
           -- cache prioritising is disabled or none are supplied */
        #ifndef DSM_DISABLE_CACHE_PRIORITISING
         pDsmObject->cachingRules = cachingRules;
        #endif
         err = lmRequestObjectLoad( idp, pOC, path, timeout, pUserData, sizeOfUserData,
               pDsmObject, &pLoadRequest );

         if (!err)
         {
            switch (pLoadRequest->rlr.status)
            {
               case LRS_LOADED:
                  DBGLOG(DD_MOD,"%s LOADED",pathname)
                  pDsmObject->pObjCarousel = pLoadRequest->pObjCarousel;
                  pDsmObject->pModule = pLoadRequest->pModule;
                  pDsmObject->objectDataOffset =
                     pLoadRequest->targetObjectOffset;
                  pDsmObject->objectInfo = pLoadRequest->targetObjectInfo;
                  pDsmObject->kind =
                     convertObjectKindStr(
                        pLoadRequest->targetObjectInfo.objectKind );

                  lmSetObjectModuleLoaded( idp, pDsmObject->pModule );

                  pDsmObject->status = OBJ_LOAD_COMPLETED;

                  *pStatus = OBJ_LOAD_COMPLETED;
                  *pclDsmObjectHandle = (void *) pDsmObject;
                  break;

               case LRS_STALLED_MODULE:
               case LRS_STALLED_SRG_MODULE:
                  pDsmObject->r.pLoadRequest = pLoadRequest;
                  pDsmObject->status = OBJ_LOAD_IN_PROGRESS;

                  *pStatus = OBJ_LOAD_IN_PROGRESS;
                  *pclDsmObjectHandle = (void *) pDsmObject;
                  break;

               case LRS_LITE_OPTIONS_LOADED:
                  dsmDP2(("LRS_LITE_OPTIONS_LOADED pOC=%p\n", pLoadRequest->pObjCarousel));
                  pDsmObject->pObjCarousel = pLoadRequest->pObjCarousel;
                  pDsmObject->pModule = pLoadRequest->pModule;
                  pDsmObject->objectDataOffset =
                     pLoadRequest->targetObjectOffset;
                  pDsmObject->objectInfo = pLoadRequest->targetObjectInfo;
                  pDsmObject->kind =
                     convertObjectKindStr(
                        pLoadRequest->targetObjectInfo.objectKind );

                  lmSetObjectModuleLoaded( idp, pDsmObject->pModule );

                  pDsmObject->status = OBJ_LOAD_COMPLETED;
                  *pStatus = OBJ_LOAD_COMPLETED;
                  *pclDsmObjectHandle = (void *) pDsmObject;
                  break;

               case LRS_LITE_OPTIONS_PENDING:
                  dsmDP2(("LRS_LITE_OPTIONS_PENDING pOC=%p\n", pLoadRequest->pObjCarousel));
                  pDsmObject->r.pLoadRequest = pLoadRequest;
                  pDsmObject->pObjCarousel = pLoadRequest->pObjCarousel;
                  pDsmObject->objectDataOffset =
                     pLoadRequest->targetObjectOffset;
                  pDsmObject->objectInfo = pLoadRequest->targetObjectInfo;
                  pDsmObject->kind =
                     convertObjectKindStr(
                        pLoadRequest->targetObjectInfo.objectKind );

                  pDsmObject->status = OBJ_LOAD_IN_PROGRESS;
                  *pStatus = OBJ_LOAD_IN_PROGRESS;
                  *pclDsmObjectHandle = (void *) pDsmObject;
                  break;

               /*
               *** PLACEHOLDER - NOT CURRENTLY VALID ***
               case LRS_STALLED_DSI:
                 TODO: Allow object loads before DSI acquired
               break;
               */

               case LRS_ABORTED_PATH_ERROR:
                  *pStatus = OBJ_LOAD_ABORTED_PATH_ERROR;
                  ERRPRINT("%p", pDsmObject)
                  err = CLDSM_ERR_LOAD_FAILED;
                  break;

               case LRS_ABORTED_LOAD_ERROR:
                  *pStatus = OBJ_LOAD_ABORTED_ERROR;
                  ERRPRINT("%p", pDsmObject)
                  err = CLDSM_ERR_LOAD_FAILED;
                  break;

               case LRS_ABORTED_TIMEOUT:
                  *pStatus = OBJ_LOAD_ABORTED_TIMEOUT;
                  /* was OC_LOAD_ABORTED_TIMEOUT which was equivalent to OBJ_LOAD_ABORTED_PATH_ERROR !
                  * As it happens, this was not important to any calling func */
                  ERRPRINT("%p", pDsmObject)
                  err = CLDSM_ERR_LOAD_FAILED;
                  break;

               default:
                  /* -- Illegal status */
                  dsmDP1(("ERROR: Illegal loadRequest status = %u\n",
                          pLoadRequest->rlr.status));
                  *pStatus = OBJ_LOAD_ABORTED_ERROR;
                  ERRPRINT("%p", pDsmObject)
                  dsmAssert((0));
                  err = CLDSM_ERR_INTERNAL;
                  break;
            }
         }

         if (!err)
         {
            /* -- Increment loaded object count on this carousel */
            if (pLoadRequest->rlr.status == LRS_LITE_OPTIONS_LOADED ||
                pLoadRequest->rlr.status == LRS_LITE_OPTIONS_PENDING)
            {
               P_ObjectCarousel pLiteOC;

               idp->pCurrentCarousel = (P_RootCarousel)pLoadRequest->pObjCarousel;

               /* Count the object with the NEW carousel */
               pLiteOC = pDsmObject->pObjCarousel;
               pLiteOC->loadedObjectCount++;
            }
            else
            {
               pOC->loadedObjectCount++;
            }

            /* -- Add to list of all loaded objects */
            LLInsertHead( pOC->llcCurrObjects, pDsmObject );
         }
      }
      if (err)
      {
         /* -- Error so destroy/free any allocated memory objects */
         if (pDsmObject)
         {
            if (pLoadRequest)
            {
               lmLoadDestroy( idp, pLoadRequest );
            }
            objectDestroy( idp, pDsmObject );
         }
      }
      else
      {
         if (*pStatus == OBJ_LOAD_COMPLETED)
         {
            /* -- Object loaded so destroy loadRequest */
            lmLoadDestroy( idp, pLoadRequest );
         }
      }
   }
_return:
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("clDsmLoadObject:%u Path=%s not found\n", err, path)));
   dsmDP2(("clDsmLoadObject -> rtn: %u, o/p: %u, Hl=%p pLoadRequest = 0x%p path=%s\n",
           err, *pStatus, *pclDsmObjectHandle, pLoadRequest, path));
   return err;
}

/*
-- Unload (or cancel the requested load for) a DSM-CC object
*/
E_DscError CDSM_UnloadObject( H_DsmCoreInst instance,
   /*I*/ H_DsmObject clDsmObjectHandle,
   E_DsmRstMode mode )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   P_DsmObject pObject = (P_DsmObject) clDsmObjectHandle;
   P_LoadRequest pLoadRequest;
   E_DscError err = CLDSM_OK;
   P_ObjectCarousel pOC;

   dsmDP2(("CDSM_UnloadObject( %p, Hl=%p )\n", instance, clDsmObjectHandle));

   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );

   if (!memValidate( pObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (!DSM_OBJECT_CORRECT( pObject, idp ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pObject->status == OBJ_LOAD_ABORTED_UNLOAD)
   {
      dsmDP1(("API ERROR: Invalid object handle (already unloaded)\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pObject->objectDataSeq != NULL)
   {
      dsmDP1(("API ERROR: Object still open\n"));
      err = CLDSM_ERR_OBJECT_OPEN;
   }
   else
   {
      /*
      -- Remove record of object from carousel and remove object from list
      -- of loaded objects.
      -- NB. This ensures that object is no longer 'visible' (ie. appears
      -- unloaded) to any relevant API functions called from within object
      -- event notify callback.
      */

      pOC = pObject->pObjCarousel;
      if (pOC != NULL)
      {
         /* -- Decrement loaded object count on relevant carousel */
         pOC->loadedObjectCount--;
      }
      else
      {
         /* this is case when mode was PENDING, followed by FORCE */
         DBGLOG(DD_GEN, "Object's Carousel is NULL")
      }

      /* -- Remove from list of all loaded objects */
      LLRemove( pObject, CURR_OBJECT_LIST );

      switch (pObject->status)
      {
         case OBJ_LOAD_IN_PROGRESS:
            pLoadRequest = pObject->r.pLoadRequest;
            ASSERT( pLoadRequest != NULL )
            if (mode == RST_MODE_PENDING)
            {
               if (pLoadRequest->rlr.status != LRS_ABORT_PENDING_RELOAD)
               {
                  DBGLOG(DD_GEN, "Object remains pending a re-load")
                  pLoadRequest->rlr.status = LRS_ABORT_PENDING_RELOAD;
                  lmStopModuleLoadRequest( idp, pLoadRequest );
                  /* Special circumstance where Object remains, while Carousel is unloaded
                   * and later the object is attached to new carousel on another service */
                  pObject->pObjCarousel = NULL;
               }
            }
            else
            {
               if (pLoadRequest->rlr.status == LRS_ABORT_PENDING_RELOAD)
               {
                  pLoadRequest->rlr.status = LRS_ABORTED_BY_REQUESTER;
                  DSC_LoadRsqtFinalise( idp, (P_RootLoadRqst)pLoadRequest );
               }
               else
               {
                  /* -- Set loadRequest status and abort module load */
                  pLoadRequest->rlr.status = LRS_ABORTED_BY_REQUESTER;
                  lmStopModuleLoadRequest( idp, pLoadRequest );
               }
               objectDestroy( idp, pObject );
            }
            break;

         case OBJ_LOAD_COMPLETED:
            lmSetObjectModuleUnloaded( idp, pObject->pModule );

            /* -- NULL module reference in object (finished with) */
            pObject->pModule = NULL;

         case OBJ_LOAD_ABORTED_TIMEOUT:
         case OBJ_LOAD_ABORTED_PATH_ERROR:
         case OBJ_LOAD_ABORTED_ERROR:
            /* -- Nothing to do */
            objectDestroy( idp, pObject );
            break;

         default:
            /* -- Illegal status */
            objectDestroy( idp, pObject );
            dsmDP1(("ERROR: Illegal DSM object status = %u\n",
                    pObject->status));
            dsmAssert((0));
            err = CLDSM_ERR_INTERNAL;
            break;
      }
   }
_return:
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: clDsmUnloadObject: %u\n", err)));
   dsmDP2(("exit clDsmUnloadObject -> rtn: %u \n", err));
   return err;
}

/**
 * @brief   Open object data for subsequent access via API functions for given object kind.
 *          The object must have already been loaded.
 *          Data that is specific to object kind is read and possibly stored here.
 *          The object MUST be one of the following types:
 *          . Stream Object
 *          . Stream Object with Events
 *          . File Object
 *          . Service Gateway Object
 * @param   instance Instance of DSMCC containing object
 * @param   clDsmObjectHandle Handle to the Object
 * @param   pKind Populated with the object kind
 * @return
 *          CLDSM_OK
 *          The object was successfully opened
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The handle is not a valid object handle
 *          CLDSM_ERR_OBJECT_NOT_LOADED
 *          The object has not been loaded
 *          CLDSM_ERR_OPEN_OBJECT_LIMIT
 *          The maximum number of open objects allowed has been reached
 *          CLDSM_ERR_OBJECT_OPEN
 *          The object is already open
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The object is not a supported object type (see above list of valid types).
 *          CLDSM_ERR_END_OF_DATA
 *          The object has zero length.
 */
E_DscError CDSM_OpenObject(
   /*I*/ H_DsmCoreInst instance,
   /*I*/ H_DsmObject clDsmObjectHandle,
   /*O*/ E_DsmObjectKind *pKind)
{
   E_DscError err = CLDSM_OK;
   P_DsmCoreInst idp = (P_DsmCoreInst)instance;
   P_DsmObject pDsmObject = (P_DsmObject)clDsmObjectHandle;
   P_Module pModule;
   U32BIT dataOffset = 0;             /* Initialise prevents compiler warnings */
   U32BIT dataLength = 0;             /* Initialise prevents compiler warnings */

   dsmDP2(("CDSM_OpenObject( %p, %p, %p )\n",
           instance, clDsmObjectHandle, pKind));

   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );
   API_PARAM_CHK( clDsmObjectHandle, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );
   API_PARAM_CHK( pKind, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );

   if (!DSM_OBJECT_CORRECT( pDsmObject, idp ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->pModule == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->objectDataSeq != NULL)
   {
      err = CLDSM_ERR_OBJECT_OPEN;
   }
   else if (idp->dsmObjectsOpen >= NUM_OPEN_DSM_OBJECTS_MAX)
   {
      err = CLDSM_ERR_OPEN_OBJECT_LIMIT;
   }
   else
   {
      pModule = pDsmObject->pModule;
      dsmAssert((pModule->hModuleData != NULL));
      dsmAssert((pModule->moduleInfo.crslMagic == OC_MAGIC));

      /* Determine offset of object information based on object kind.
         Obtain and store any other general purpose information */
      switch (pDsmObject->kind)
      {
         case STREAM_OBJ:
         /* Same code as STREAM_OBJ_WITH_EVENTS, so, no break */
         case STREAM_OBJ_WITH_EVENTS:
            /* Offset will point to taps_count field */
            dataOffset = pDsmObject->objectInfo.messageBodyOffset;
            dataLength = pDsmObject->objectInfo.messageBodyLength;
            break;
         case FILE_OBJ:
            /* Skip over 32-bit length field - hence the '4'.
               Offset will point to start of the object body.
               The length will indicate the number of bytes in body.
               For example a FILE_OBJ offset would be to start of file
               content and length would be number of bytes in file. */
            dataOffset = pDsmObject->objectInfo.messageBodyOffset + 4;
            dataLength = pDsmObject->objectInfo.messageBodyLength - 4;
            break;
         case DIRECTORY_OBJ:
         case SERVICE_GATEWAY_OBJ:
            /* SRG objectInfoOffset initially points to the field
               immediately after objectInfo_length field */

            /* Calc. dataOffset to point to serviceContextList_count field */
            dataOffset = pDsmObject->objectInfo.objectInfoOffset +
               pDsmObject->objectInfo.objectInfoLength;

            /* Calc. dataLength to cover the serviceContextList_count field
               to the end of the object */
            dataLength = pDsmObject->objectInfo.objectLen -
               pDsmObject->objectInfo.objectInfoOffset -
               pDsmObject->objectInfo.objectInfoLength;


            break;
         default:
            dsmDP1(("API ERROR: Invalid object kind\n"));
            err = CLDSM_ERR_INVALID_OBJECT_TYPE;
            break;
      }

      /* Check for zero-length object */
      if (dataLength == 0)
      {
         *pKind = pDsmObject->kind;
         err = CLDSM_ERR_END_OF_DATA;
      }

      if (err == CLDSM_OK)
      {
         DBGLOG(DD_OBJ,"Module %p, status %u",pModule,pModule->status)
         pDsmObject->hModuleData = pModule->hModuleData;

         /* Open the object to enable future access */
         CHECK_MEM_ERR(memSeqOpen(MEM_CONTEXT, (MemHandle)moduleDataOpen(pDsmObject->hModuleData),
               (pDsmObject->objectDataOffset + dataOffset), dataLength,
               TRUE /*async*/, &(pDsmObject->objectDataSeq)));

         /*
         -- NB. Should never generate MEM_SEQ_OPEN_LIMIT error since core stack
         -- configures memMgr (via memStart) to support sufficient MemSeqRefs to
         -- cover it's worst case internal use plus NUM_OPEN_DSM_OBJECTS_MAX
         */

         *pKind = pDsmObject->kind;

         idp->dsmObjectsOpen++;
      }
   }

   goto _return;  /* Prevents compiler warnings */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: clDsmOpenObject: %u\n", err)));
   dsmDP2(("exit clDsmOpenObject -> rtn: %u, o/p: %u \n", err, *pKind));

   return err;
}

/*
-- Close a loaded object
--
*/
E_DscError CDSM_CloseObject( H_DsmCoreInst instance,
   /*I*/ H_DsmObject clDsmObjectHandle )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   P_DsmObject pDsmObject = (P_DsmObject) clDsmObjectHandle;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_CloseObject( %p, %p )\n",
           instance, clDsmObjectHandle));

   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );
   API_PARAM_CHK( clDsmObjectHandle, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );


   if (!DSM_OBJECT_CORRECT( pDsmObject, idp ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->pModule == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      moduleDataClose(idp, pDsmObject->hModuleData);
      pDsmObject->hModuleData = NULL;
      memSeqClose( pDsmObject->objectDataSeq );
      pDsmObject->objectDataSeq = NULL;

      idp->dsmObjectsOpen--;
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: clDsmCloseObject: %u\n", err)));
   dsmDP2(("exit clDsmCloseObject -> rtn: %u\n", err));
   return err;
}

/**
 * @brief   Get DSM instance to which the object belongs.
 * @param   clDsmObjectHandle DSM object handle
 * @return  NULL if an invalid object handle, otherwise returns instance handle
 */
H_DsmCoreInst CDSM_ObjectGetInstance(H_DsmObject clDsmObjectHandle)
{
   P_DsmObject pDsmObject = (P_DsmObject) clDsmObjectHandle;

   if (!memValidate( pDsmObject ))
   {
      return NULL;
   }
   return pDsmObject->myDsmInst;
}

/*
-- Set caching rules for a cached object (path).
*/
E_DscError CDSM_SetObjectCachingRules( H_DsmCoreInst instance,
   U8BIT *pathname, E_CacheRules cachingRules )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   U8BIT path[MAX_PATH_NAME_SIZE];
   P_Module pModule;
   E_DscError err;
   P_ObjectCarousel pOC;

   dsmDP2(("CDSM_SetObjectCachingRules( %p, %s, %u )\n", instance, pathname, cachingRules));

   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );
   API_PARAM_CHK( pathname, dsmDP1(("ERROR: Invalid pathname\n")),
      err = CLDSM_ERR_INVALID_PATHNAME; goto _return );

   err = RetrieveCarouselPath( idp, pathname, path, &pOC );
   if (!err)
   {
      #ifdef DSM_DISABLE_CACHE_PRIORITISING
      /* -- Setup default caching rules */
      cachingRules = CACHE_RULES_DEFAULT;
      #endif

      err = lmGetObjectLoadState( idp, pOC, (U8BIT *)path, &pModule );
      if (!err)
      {
         if (pModule != NULL)
         {
            /* -- Module must be cached with valid moduleData */
            dsmAssert((pModule->status == MS_CACHED));
            dsmAssert((pModule->hModuleData != NULL));

            /* -- TODO: Handle prioritisation of all caching rule settings */
            /* -- If new caching rules have fromStream set they override any previous
               -- caching rules for module */
            if (cachingRules == CACHE_RULES_FROM_STREAM)
            {
               DSC_ModuleDataRefresh( idp, pModule );
            }
         }
      }

      if (err == CLDSM_ERR_LOAD_FAILED)
      {
         /* -- Do not return this error since client is not interested */
         err = CLDSM_OK;
      }
   }
_return:
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: clDsmSetObjectCachingRules: %u\n", err)));
   dsmDP2(("exit clDsmSetObjectCachingRules -> rtn: %u\n", err));
   return err;
}

/*
-- Pre-fetch (into cache) a DSM-CC object
--
*/
E_DscError CDSM_PrefetchObject( H_DsmCoreInst instance,
   U8BIT *pathname, U32BIT timeout )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   U8BIT path[MAX_PATH_NAME_SIZE];
   P_LoadRequest pLoadRequest;
   E_DscError err;
   P_ObjectCarousel pOC;
   E_LRStatus lrStatus;

   dsmDP2(("CDSM_PrefetchObject( %p, %s, %u )\n", instance, pathname, timeout));

   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );
   API_PARAM_CHK( pathname, dsmDP1(("ERROR: Invalid pathname\n")),
      err = CLDSM_ERR_INVALID_PATHNAME; goto _return );

   err = RetrieveCarouselPath( idp, pathname, path, &pOC );
   if (!err)
   {
      err = lmPrefetchObjectLoad( idp, pOC, path, timeout, &pLoadRequest );

      if (!err)
      {
         lrStatus = pLoadRequest->rlr.status;

         switch (lrStatus)
         {
            case LRS_LOADED:
            case LRS_ABORTED_LOAD_ERROR:
            case LRS_ABORTED_PATH_ERROR:
               /* -- NB. Do not return load failure errors since client
                  -- is not interested */
               lmLoadDestroy( idp, pLoadRequest );
               break;

            /*
            *** PLACEHOLDER - NOT CURRENTLY VALID ***
            case LRS_STALLED_DSI:
                TODO: Allow object pre-fetches before DSI acquired
            */
            case LRS_STALLED_MODULE:
            case LRS_STALLED_SRG_MODULE:
               /* -- Do nothing */
               break;

            default:
               /* -- Illegal status */
               dsmDP1(("ERROR: Illegal loadRequest status = %u\n",
                       lrStatus));
               dsmAssert((0));
               err = CLDSM_ERR_INTERNAL;
               break;
         }
      }

      if (err)
      {
         /* -- Error so destroy/free any allocated memory objects */
         if (pLoadRequest)
         {
            lmLoadDestroy( idp, pLoadRequest );
         }
      }
   }
_return:
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: clDsmPrefetchObject: %u\n", err)));
   dsmDP2(("exit clDsmPrefetchObject -> rtn: %u\n", err));
   return err;
}

/***************************************/
/* -- Asynchronous Client API Calls -- */
/***************************************/

/*
-- Get specified object kind
*/
E_DscError CDSM_ObjectGetKind(
   /*I*/ H_DsmObject object,
   /*O*/ E_DsmObjectKind *pKind )
{
   P_DsmObject pDsmObject = (P_DsmObject) object;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_ObjectGetKind( %p, %p )\n", object, pKind));

   API_PARAM_CHK( object, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );
   API_PARAM_CHK( pKind, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else
   {
      *pKind = pDsmObject->kind;
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: dsmObjectGetKind: %u\n", err)));
   dsmDP2(("exit dsmObjectGetKind -> rtn: %u, o/p: %u \n", err, *pKind));
   return err;
}

/*******************************************************************************
* Get specified object carousel handle.
*
* The object (clDsmObjectHandle) must be LOADED.
*
* NB. This can also be called to poll for when an object is loaded.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
*
*******************************************************************************/
E_DscError CDSM_ObjectGetCarouselHandle(
   /*I*/ H_DsmObject object,
   /*O*/ void **hCarouselHandle )
{
   P_DsmObject pDsmObject = (P_DsmObject) object;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_ObjectGetCarouselHandle( %p, %p )\n", object, hCarouselHandle));

   API_PARAM_CHK( object, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );
   API_PARAM_CHK( hCarouselHandle, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else
   {
      *hCarouselHandle = pDsmObject->pObjCarousel;
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: dsmObjectGetCarouselHandle: %u\n", err)));
   dsmDP2(("exit dsmObjectGetCarouselHandle -> rtn: %u, o/p: %p \n", err, *hCarouselHandle));
   return err;
}

/**
 * @brief   This function gets service context of DSM object (if available)
 *          Need by MHEG5 for specified service gateway (SRG) object.
 *          The Dsm object (clDsmObjectHandle) must be LOADED and OPEN.
 *          When serviceId cannot be found then
 *          matched then a pathname of '\0' is returned.
 *          On return from this function the current location within the SRG object
 *          message will remain the same as before the function was called. This means that
 *          this function can be called more than once in succession without reopening the
 *          memory sequence.
 * @param   clDsmSRGObjectHandle SRG object handle supplied when object was
 *          loaded (via clDsmLoadObject).
 * @param   serviceId ServiceID in a ServiceContextList in a
 *          ServiceGatewayMessage. For MHEG this is made
 *          from data_broadcast_id (hi-order bytes) and
 *          application_type_code (lo-order bytes).
 * @param   pSrvCtxtData Returned pointer of service context data
 * @param   pSrvCtxtLen Returned length of service context data
 * @return
 *          CLDSM_OK (0)
 *          The SRG object was successfully parsed for an MHEG5 initial object path
 *          (nb. this does not imply an initial object path was actually found).
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          clDsmSRGObjectHandle is not a valid object reference.
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The object is not an SRG.
 *          CLDSM_ERR_OBJECT_NOT_LOADED
 *          The object is not yet loaded.
 *          CLDSM_ERR_OBJECT_NOT_OPEN
 *          The object is not open.
 *          CLDSM_ERR_END_OF_DATA
 *          No matching service context for 'serviceId'
 */
E_DscError CDSM_ObjectGetServiceContext(
   /*I*/ H_DsmObject clDsmObjectHandle,
   /*I*/ U32BIT serviceId,
   /*O*/ U8BIT **pSrvCtxtData,
   /*O*/ U32BIT *pSrvCtxtLen)
{
   E_DscError err;
   P_DsmObject pObject = (P_DsmObject)clDsmObjectHandle;
   U32BIT initialOffset;
   MemPtr addr;
   U32BIT srvId;
   U16BIT application_specific_data_length;
   U8BIT serviceIdNum;
   U8BIT serviceContextList_count;

   dsmDP3(("CDSM_ObjectGetServiceContext(clDsmObjectHandle=%p, serviceId=0x%x)\n",
           clDsmObjectHandle, serviceId));

   if (!DSM_OBJECT_VALIDATE(pObject))
   {
      dsmDP1(("API ERROR: Invalid SRG Object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pObject->status != OBJ_LOAD_COMPLETED)
   {
      dsmDP1(("API ERROR: SRG Object not loaded\n"));
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pObject->objectDataSeq == NULL)
   {
      dsmDP1(("API ERROR Object not open\n"));
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      err = CLDSM_ERR_END_OF_DATA;

      /* Remember the offset, so we can restore position before return */
      memSeqReadPos((MemSeqRef)pObject->objectDataSeq, &initialOffset);

      /* Get current location within the SRG object message */
      addr = pObject->objectDataSeq;

      /* Read serviceContextList_count */
      memSeqReadByte( addr, &serviceContextList_count );

      for (serviceIdNum = 0; serviceIdNum != serviceContextList_count; serviceIdNum++)
      {
         /* Read ServiceID (4 bytes) - This is a combination (concatanation) of
            data_broadcast_id (2 bytes) and application_type_code (2 bytes) */
         readUInt32Seq(addr, &srvId);

         /* Read application_specific_data_length */
         readUInt16Seq(addr, &application_specific_data_length);

         /* Have we found the required ServiceID ? */
         if (srvId == serviceId)
         {
            *pSrvCtxtLen = application_specific_data_length;
            memSeqAccessCurrent( addr, (U8BIT **)pSrvCtxtData, pSrvCtxtLen );
            err = CLDSM_OK;
            break;
         }
         /* Skip application_specific_data_byte loop */
         memSeqSetPosRel( addr, application_specific_data_length);
      }
      /* Reset offset to original location in SRG object message */
      memSeqSetPosAbs(addr, initialOffset);
   }
   DEBUG_CHK(err == CLDSM_OK, dsmDP1(("ERROR: clDsmSRGObjectHandle=%u\n", err)));
   return err;
}

/*------------------------------ Local Functions -----------------------------*/

static U8BIT* GetHexValue16( U8BIT *url, U16BIT *pValue )
{
   int i, val;
   val = 0;
   for (i = 0; i != 4; i++, url++)
   {
      if (*url >= '0' && *url <= '9')
      {
         val = (val * 16) + *url - '0';
      }
      else if (*url >= 'a' && *url <= 'f')
      {
         val = (val * 16) + 0xa + *url - 'a';
      }
      else if (*url >= 'A' && *url <= 'F')
      {
         val = (val * 16) + 0xA + *url - 'A';
      }
      else
      {
         break;
      }
   }
   if (i != 0)
   {
      *pValue = val;
   }
   return url;
}

static U8BIT* GetHexValue32( U8BIT *url, U32BIT *pValue )
{
   int i, val;
   val = 0;
   for (i = 0; i != 8; i++, url++)
   {
      if (*url >= '0' && *url <= '9')
      {
         val = (val * 16) + *url - '0';
      }
      else if (*url >= 'a' && *url <= 'f')
      {
         val = (val * 16) + 0xa + *url - 'a';
      }
      else if (*url >= 'A' && *url <= 'F')
      {
         val = (val * 16) + 0xA + *url - 'A';
      }
      else
      {
         break;
      }
   }
   *pValue = val;
   return url;
}

static U8BIT* ParseDvbLocator( U8BIT *url, S_CarouselLocator *pLoc )
{
   url = GetHexValue16( url, &pLoc->original_network_id );
   if (*url == '.')
   {
      url++;
      url = GetHexValue16( url, &pLoc->transport_stream_id );
      if (*url == '.')
      {
         url++;
         url = GetHexValue16( url, &pLoc->service_id );
         if (*url == '.')
         {
            url++;
            url = GetHexValue16( url, &pLoc->component_tag );
            if (*url == '$')
            {
               url++;
               url = GetHexValue32( url, &pLoc->carousel_id  );
            }
         }
      }
   }
   return url;
}

/* /////////////////////////////////////////////////////////////////////////////
// SanitisePath
// -- Create 'sanitised' output path from supplied input path
// -- opPath must reference an array at least MAX_PATH_NAME_SIZE in length
///////////////////////////////////////////////////////////////////////////// */
static E_DscError SanitisePath( U8BIT *ipPath, U8BIT *opPath, S_CarouselLocator *pLoc  )
{
   E_DscError err = CLDSM_OK;
   size_t pathLen;
   size_t pathElementLen;

   dsmDP3(("SanitisePath()\n"));
   dsmAssert((ipPath != NULL));
   dsmAssert((opPath != NULL));

   opPath[0] = '\0';
   if (strlen((char *)ipPath ) != 0)
   {
      pathLen = 0;
      if (memcmp(ipPath, "DSM:", 4) == 0)
      {
         ipPath += 4;
      }
      else if (memcmp(ipPath, "dvb:", 4) == 0)
      {
         /* according to "ETSI TS 102 851", url has format:
          * "dvb:/d/s/2.png" (for same carousel) or,
          * "dvb://233a..1044$1/g/h/2.png" (for possibly a different carousel)
          */
         ipPath += 4;
         if (ipPath[0] == '/' && ipPath[1] == '/')
         {
            /* Section 6.1 table 1 (BNF), this is 'dvb_net_path' */
            ipPath += 2;
            if (*ipPath != '/')
            {
               ipPath = ParseDvbLocator( ipPath, pLoc );
            }
         }
      }

      /* -- Discard any leading path separator chars */
      while (*ipPath == '/' || *ipPath == '\\')
         ipPath++;

      /* -- Find length of next path element in input */
      pathElementLen = strcspn((char *)ipPath, PATH_SEPARATOR_LIST_STRING );

      /* -- Check if there is room for this path element in the output array */
      pathLen = pathElementLen;
      while (pathLen < (MAX_PATH_NAME_SIZE - 1) && pathElementLen != 0)
      {
         /* -- Copy path element from input to output */
         memcpy( opPath, ipPath, pathElementLen );

         /* -- Step input & output pointers */
         ipPath += pathElementLen;
         opPath += pathElementLen;

         if (*ipPath == '\0')
         {
            break;
         }
         /* -- Add a single seperator char to output */
         *opPath++ = PATH_SEPERATOR_CHAR;
         pathLen++;

         /* -- Step over/discard any/next seperator chars in input */
         ipPath += strspn((char *)ipPath, PATH_SEPARATOR_LIST_STRING );

         /* -- Find length of next path element in input */
         pathElementLen = strcspn((char *)ipPath, PATH_SEPARATOR_LIST_STRING );
         pathLen += pathElementLen;
      }
      if (pathLen >= (MAX_PATH_NAME_SIZE - 1))
      {
         dsmDP1(("API ERROR: Invalid pathname length: %u\n", pathLen));
         err = CLDSM_ERR_INVALID_PATHNAME;
      }
      else
      {
         *opPath = '\0';
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: SanitisePath: %u\n", err)));
   dsmDP3(("exit SanitisePath -> rtn: %u\n", err));
   return err;
}

static E_DscError RetrieveCarouselPath( P_DsmCoreInst idp,
   U8BIT *ipPath, U8BIT *opPath,
   P_ObjectCarousel *ppOC )
{
   P_ObjectCarousel pOC;
   E_DscError err;
   S_CarouselLocator locator;

   /* set default values */
   locator.original_network_id = idp->dvbLocator.original_network_id;
   locator.transport_stream_id = idp->dvbLocator.transport_stream_id;
   locator.service_id = 0xFFFF;
   locator.component_tag = 0xFFFF;
   locator.carousel_id = INVALID_CAROUSEL_ID;
   err = SanitisePath( ipPath, opPath, &locator );
   if (!err)
   {
      if (locator.original_network_id != idp->dvbLocator.original_network_id ||
          (locator.transport_stream_id != 0 &&
           locator.transport_stream_id != idp->dvbLocator.transport_stream_id &&
           idp->dvbLocator.transport_stream_id != 0))
      {
         ERRPRINT("different transport: onet=0x%x trans=0x%x",
            locator.original_network_id, locator.transport_stream_id )
         err = CLDSM_ERR_CAROUSEL_UNAVAILABLE;
      }
      else
      {
         pOC = (P_ObjectCarousel)idp->pCurrentCarousel;
         if (locator.service_id == 0xFFFF)
         {
            if (pOC == NULL || !memValidate( pOC ) || pOC->root.magic != OC_MAGIC)
            {
               ERRPRINT("ERROR: Invalid Current Carousel %p", pOC)
               err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
            }
            else
            {
               locator.service_id = pOC->root.serviceId;
               locator.carousel_id = pOC->root.rcid;
               DBGLOG(DD_OC, "cid=%d", locator.carousel_id)
            }
         }
         else if (locator.component_tag == 0xFFFF)
         {
            if (pOC == NULL || !memValidate( pOC ) || pOC->root.magic != OC_MAGIC)
            {
               ERRPRINT("ERROR: Current Carousel is invalid")
               err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
            }
            else if (locator.service_id != pOC->root.serviceId)
            {
               ERRPRINT("current carousel on different service: sid=0x%x cc_sid=0x%x",
                  locator.service_id, pOC->root.serviceId)
               err = CLDSM_ERR_CAROUSEL_UNAVAILABLE;
            }
            else
            {
               locator.carousel_id = pOC->root.rcid;
               DBGLOG(DD_OC, "cid=%d", locator.carousel_id)
            }
         }
         else if (locator.carousel_id == INVALID_CAROUSEL_ID)
         {
            S_SIQueryRequest siQueryData;
            S_SIQueryResult siQueryResult;
            DBGLOG(DD_OC, "ctag=%d", locator.component_tag)
            siQueryData.kind = SIQUERY_CAROUSEL_INFO;
            siQueryData.dataId = (U8BIT)locator.component_tag;
            siQueryData.serviceId = locator.service_id;
            err = siQueryStart( idp, &siQueryData, NULL, &siQueryResult );
            switch (err)
            {
               case CLDSM_PENDING:
                  /*ignore result*/
                  err = CLDSM_OK;
                  break;
               case CLDSM_OK:
                  locator.carousel_id = siQueryResult.data.carouselInfo.carouselId;
                  DBGLOG(DD_OC, "cid=%d", locator.carousel_id)
                  break;
               default:
                  ERRPRINT("err=%d", err);
            }
         }
         if (!err)
         {
            if (locator.carousel_id != INVALID_CAROUSEL_ID)
            {
               pOC = (P_ObjectCarousel)DSC_RootCrslListFindById( idp->llcRootCarousels, locator.service_id, locator.carousel_id );
               if (pOC != NULL)
               {
                  DBGLOG(DD_OC, "pOC=%p", pOC) *
                  ppOC = pOC;
               }
               else
               {
                  DBGLOG(DD_OC, "cid=%d", locator.carousel_id)
                  err = CDSM_LoadCarousel( idp, locator.service_id, locator.carousel_id,
                        SIQUERY_CAROUSEL, (H_DsmCarousel *)ppOC );
               }
            }
            else
            {
               err = CDSM_LoadCarousel( idp, locator.service_id, (U8BIT)locator.component_tag,
                        SIQUERY_CAROUSEL_INFO, (H_DsmCarousel *)ppOC );
            }
         }
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: %s: %u\n", __FUNCTION__, err)));
   return err;
}

/*----------------------------------------------------------------------------*/
