/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Directory related functions
 * @file    clDsmDir.c
 * @date    01/12/2004
 * @author  Ocean Blue
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "dsmObject.h"
#include "module.h"
#include "moduleData.h"


/*------------------------------- Local Macros -------------------------------*/


/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/


/*------------------- Local prototypes/forward declarations ------------------*/


/*---------------------------- Exported Functions ----------------------------*/


/*******************************************************************************
*
* The directory object (dirObj) must be LOADED and OPEN.
*
* Retrieves the total number of directory entries and the total length of all
* entry names.
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
*
*******************************************************************************/
E_DscError CDSM_DirEntrySizes(
   /*I*/ H_DsmObject dir,
   /*O*/ U16BIT *nameCount,
   /*O*/ U16BIT *totalNameLength )
{
   P_DsmObject pDsmObject = (P_DsmObject) dir;
   E_DscError err;
   P_Module pModule;
   MemPtr mpObjectData;
   MemPtr mpBindingData;
   U16BIT numBindings;
   U16BIT nameLens;

   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      switch (pDsmObject->kind)
      {
         default:
            err = CLDSM_ERR_INVALID_OBJECT_TYPE;
            break;

         case DIRECTORY_OBJ:
         case SERVICE_GATEWAY_OBJ:
            pModule = pDsmObject->pModule;
            if (!moduleDataFindObject(pModule->hModuleData, pModule->moduleInfo.originalSize,
                   &(pDsmObject->objectInfo.objectKey), &mpObjectData ))
            {
               err = CLDSM_ERR_OBJECT_NOT_LOADED;
            }
            else
            {
               numBindings = odDirCountAndFirstBinding( mpObjectData, &(pDsmObject->objectInfo), &mpBindingData );
               *nameCount = numBindings;
               nameLens = 0;
               while (numBindings--)
               {
                  nameLens += odDirBindingNameLength( mpBindingData );
                  if (!odDirGetNextBinding( mpBindingData, &mpBindingData ))
                  {
                     break;
                  }
               }
               *totalNameLength = nameLens;
               err = CLDSM_OK;
            }
      }
   }
   return err;
}

/*******************************************************************************
*
* The directory object (dirObj) must be LOADED and OPEN.
*
* Retrieve the first entry handle
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INVALID_OBJECT_HANDLE
* CLDSM_ERR_INVALID_OBJECT_TYPE
* CLDSM_ERR_OBJECT_NOT_LOADED
* CLDSM_ERR_OBJECT_NOT_OPEN
* CLDSM_ERR_END_OF_DATA
*
*******************************************************************************/
E_DscError CDSM_DirEntryFirst(
   /*I*/ H_DsmObject dir,
   /*O*/ void **pFirstEntry )
{
   P_DsmObject pDsmObject = (P_DsmObject) dir;
   E_DscError err;
   P_Module pModule;
   MemPtr mpObjectData;

   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      switch (pDsmObject->kind)
      {
         default:
            err = CLDSM_ERR_INVALID_OBJECT_TYPE;
            break;

         case DIRECTORY_OBJ:
         case SERVICE_GATEWAY_OBJ:
            pModule = pDsmObject->pModule;
            if (!moduleDataFindObject(pModule->hModuleData, pModule->moduleInfo.originalSize,
                   &(pDsmObject->objectInfo.objectKey), &mpObjectData ))
            {
               err = CLDSM_ERR_OBJECT_NOT_LOADED;
            }
            else
            {
               if (!odDirCountAndFirstBinding( mpObjectData, &(pDsmObject->objectInfo), (PU8BIT *)pFirstEntry ))
               {
                  *pFirstEntry = NULL;
                  err = CLDSM_ERR_END_OF_DATA;
               }
               else
               {
                  err = CLDSM_OK;
               }
            }
      }
   }
   return err;
}

/*******************************************************************************
*
* Directory entry handle (entryHandle) is return by dsmDirEntryGetFirst or
* dsmDirEntryGetNext.
*
* Retrieve the next entry handle
*
* RETURNS:
* CLDSM_OK (0)
* CLDSM_ERR_INTERNAL
*
*******************************************************************************/
E_DscError CDSM_DirEntryNext(
   /*I*/ void *entryHandle,
   /*O*/ void **pNextEntry )
{
   if (!odDirGetNextBinding((const PU8BIT)entryHandle, (PU8BIT *)pNextEntry ))
   {
      return CLDSM_ERR_INTERNAL;
   }
   return CLDSM_OK;
}

/*******************************************************************************
*
* The directory entry handle is return by dsmDirEntryGetFirst or
* dsmDirEntryGetNext.
*
* Retrieve name length
*
* RETURNS:
* Length of name copied
*
*******************************************************************************/
U16BIT CDSM_DirEntryNameLength(
   /*I*/ void *entryHandle )
{
   return odDirBindingNameLength((PU8BIT)entryHandle );
}

/*******************************************************************************
*
* The directory entry handle is return by dsmDirEntryGetFirst or
* dsmDirEntryGetNext.
*
* Copy name for entry to supplied memory location (pName)
* Caller should ensure there is enough space in location
*
* RETURNS:
* Length of name copied
*
*******************************************************************************/
U16BIT CDSM_DirEntryNameCopy(
   /*I*/ void *entryHandle,
   /*O*/ U8BIT *pName )
{
   return odDirBindingNameCopy((PU8BIT)entryHandle, pName );
}

/*******************************************************************************
*
* Get the kind of entry this is
*
* RETURNS:
* E_DsmObjectKind
*
*******************************************************************************/
E_DsmObjectKind CDSM_DirEntryKind( void *entryHandle )
{
   U32BIT kind;
   if (odDirGetBindingKind((PU8BIT)entryHandle, &kind))
   {
      return convertObjectKindStr( kind );
   }
   return ANON_OBJ;
}

/*----------------------------------------------------------------------------*/
