/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2002 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   API functions providing asynchronous access to DSM file objects
 * @file    clDsmFile.c
 * @date    7/2/2002
 * @author  N Kirkland
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "dsmObject.h"


/*------------------------------- Local Macros -------------------------------*/


/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/


/*------------------- Local prototypes/forward declarations ------------------*/


/*---------------------------- Exported Functions ----------------------------*/

/*
-- Get length of specified file data
*/
E_DscError CDSM_FileGetSize(
   /*I*/ H_DsmObject file,
   /*O*/ U32BIT *pSize )
{
   P_DsmObject pDsmObject = (P_DsmObject) file;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_FileGetSize( %p, %p )\n", file, pSize));

   API_PARAM_CHK( file, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );
   API_PARAM_CHK( pSize, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->kind != FILE_OBJ)
   {
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      memSeqSize( pDsmObject->objectDataSeq, pSize );
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: dsmFileGetSize: %u\n", err)));
   dsmDP2(("exit dsmFileGetSize -> rtn: %u, o/p: %u \n", err, *pSize));
   return err;
}

/*
-- Read specified number of bytes of data from (opened) file
*/
E_DscError CDSM_FileRead(
   /*I*/ H_DsmObject file, U32BIT numBytes,
   /*O*/ U8BIT *pDest, U32BIT *pNumBytesActual )
{
   P_DsmObject pDsmObject = (P_DsmObject) file;
   E_DsmMemErr memErr;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_FileRead( %p, %u, %p, %p )\n",
           file, numBytes, pDest, pNumBytesActual));

   API_PARAM_CHK( file, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );
   API_PARAM_CHK( pDest, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );
   API_PARAM_CHK( pNumBytesActual, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->kind != FILE_OBJ)
   {
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      memErr = memSeqRead( pDsmObject->objectDataSeq,
            pDest, numBytes, pNumBytesActual );

      if (memErr == MEM_ERR_SEQ_END_OF_DATA)
      {
         err = CLDSM_ERR_END_OF_DATA;
      }
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: dsmFileRead: %u\n", err)));
   dsmDP2(("exit dsmFileRead -> rtn: %u, o/p: %u data bytes\n",
           err, *pNumBytesActual));
   return err;
}

E_DscError CDSM_FileDirect(
   /*I*/ H_DsmObject file,
   /*O*/ U8BIT **ppDest, U32BIT *pNumBytesActual )
{
   P_DsmObject pDsmObject = (P_DsmObject) file;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_FileDirect( %p, %u, %p, %p )\n",
           file, ppDest, pNumBytesActual));

   API_PARAM_CHK( file, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );
   API_PARAM_CHK( ppDest, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );
   API_PARAM_CHK( pNumBytesActual, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->kind != FILE_OBJ)
   {
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      memSeqAccessContig( pDsmObject->objectDataSeq, ppDest, pNumBytesActual );
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: clDsmFileDirect: %u\n", err)));
   dsmDP2(("exit clDsmFileDirect -> rtn: %u, o/p: %u data bytes\n",
           err, *pNumBytesActual));
   return err;
}

/*
-- Read single byte of data from (opened) file
*/
E_DscError CDSM_FileReadByte(
   /*I*/ H_DsmObject file,
   /*O*/ U8BIT *pDest )
{
   P_DsmObject pDsmObject = (P_DsmObject) file;
   E_DsmMemErr memErr;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_FileReadByte( %p, %p )\n", file, pDest));

   API_PARAM_CHK( file, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );
   API_PARAM_CHK( pDest, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->kind != FILE_OBJ)
   {
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      memErr = memSeqReadByte( pDsmObject->objectDataSeq, pDest );

      if (memErr == MEM_ERR_SEQ_END_OF_DATA)
      {
         err = CLDSM_ERR_END_OF_DATA;
      }
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: dsmFileReadByte: %u\n", err)));
   dsmDP2(("exit dsmFileReadByte -> rtn: %u, o/p: %u \n", err, *pDest));
   return err;
}

/*
-- Set absolute file data 'cursor' position
*/
E_DscError CDSM_FileSetPosAbs(
   /*I*/ H_DsmObject file, U32BIT absPosition )
{
   P_DsmObject pDsmObject = (P_DsmObject) file;
   E_DsmMemErr memErr;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_FileSetPosAbs( %p, %u )\n", file, absPosition));

   API_PARAM_CHK( file, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->kind != FILE_OBJ)
   {
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      memErr = memSeqSetPosAbs( pDsmObject->objectDataSeq, absPosition );

      if (memErr == MEM_ERR_SEQ_END_OF_DATA)
      {
         err = CLDSM_ERR_END_OF_DATA;
      }
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: dsmFileSetPosAbs: %u\n", err)));
   dsmDP2(("exit dsmFileSetPosAbs -> rtn: %u \n", err));
   return err;
}

/*
-- Set relative file data 'cursor' position
*/
E_DscError CDSM_FileSetPosRel(
   /*I*/ H_DsmObject file, S32BIT relPosition )
{
   P_DsmObject pDsmObject = (P_DsmObject) file;
   E_DsmMemErr memErr;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_FileSetPosRel( %p, %d )\n", file, relPosition));

   API_PARAM_CHK( file, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->kind != FILE_OBJ)
   {
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      memErr = memSeqSetPosRel( pDsmObject->objectDataSeq, relPosition );

      if (memErr == MEM_ERR_SEQ_END_OF_DATA)
      {
         err = CLDSM_ERR_END_OF_DATA;
      }
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: dsmFileSetPosRel: %u\n", err)));
   dsmDP2(("exit dsmFileSetPosRel -> rtn: %u \n", err));
   return err;
}

/*
-- Read file data 'cursor' position
*/
E_DscError CDSM_FileGetPos(
   /*I*/ H_DsmObject file, U32BIT *pPosition )
{
   P_DsmObject pDsmObject = (P_DsmObject) file;
   E_DscError err = CLDSM_OK;

   dsmDP2(("CDSM_FileGetPos( %p, %p )\n", file, pPosition));

   API_PARAM_CHK( file, dsmDP1(("ERROR: Invalid object handle\n")),
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE; goto _return );
   API_PARAM_CHK( pPosition, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );


   if (!DSM_OBJECT_VALIDATE( pDsmObject ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->kind != FILE_OBJ)
   {
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else
   {
      memSeqReadPos( pDsmObject->objectDataSeq, pPosition );
   }

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: dsmFileGetPos: %u\n", err)));
   dsmDP2(("exit dsmFileGetPos -> rtn: %u, o/p: %u \n", err, *pPosition));
   return err;
}

/*----------------------------------------------------------------------------*/
