/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Main Synchronous System (broadcast) API functions
 *             References:
 *             [1] - BS ISO/IEC 13818-6:1998
 *             [2] - ETSI/DVB TR 101 202 v1.1.1 (1999-02)
 * @file    clDsmMain.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "cldsmcc.h"
#include "clDsmSystem.h"
#include "clDsmUtils.h"
#include "linkList.h"
#include "sectionFilter.h"
#include "dsmObject.h"
#include "module.h"
#include "dataCarousel.h"
#include "objectCarousel.h"
#include "updateCarousel.h"
#include "moduleDecompress.h"
#include "loadMgr.h"
#include "siQuery.h"
#include "cacheMgr.h"

#include "defMemUtilsContig.h"  /* -- Default mem type for module */
#include "pmtUpdate.h"
#include "streamEvent.h"

#if (!defined(NDEBUG) && !defined(NO_DP))
#include <stdio.h>             /* -- For debug prints etc. */
#endif

/*------------------------------- Local Macros -------------------------------*/

/* Slightly more debug info for processDSI if debug level >=3 */
/*#define DEBUG_PROCESS_DSI*/
#define DSI_MSG_ID   0x1006
#define DII_MSG_ID   0x1002
#define DDB_MSG_ID   0x1003

/*------------------------------  Exported Data  -----------------------------*/
#if (!defined(NDEBUG) && !defined(NO_DP))
F_Printf DBG_ErrorPrintfFunc = NULL;
F_Printf DBG_WarnPrintfFunc = NULL;
F_Printf DBG_DebugPrintfFunc = NULL;
F_Printf DBG_InfoPrintfFunc = NULL;
U32BIT dsmDbgState = 0;
#endif

/*--------------------------------Local Types --------------------------------*/

typedef struct _accelerate_carousel_loading_struct
{
   U16BIT storeWritePos;
   U16BIT storeReadPos;
   U16BIT storedSectionsNB;
   U16BIT SectionsReinjection;
   BOOLEAN bLargeDDBFilterSet;
   U16BIT requestedModidPos;
   P_SecFilterInfo largeDDBFilter;
} accelerateCarouselLoading_t;

accelerateCarouselLoading_t accCarouselLoadingInfos;

/*------------------------------- Local Statics ------------------------------*/

#ifdef ACCELERATE_CAROUSEL_LOADING



#define kMAX_PRESTORED_SECTIONS 100

#define kMAX_SECTION_SIZE 5000

typedef struct
{
   U8BIT sectionSize;
   U16BIT moduleID;
   U16BIT blockNum;
   U8BIT injectionDone;
   U8BIT moduleVersion;
   U8BIT section[kMAX_SECTION_SIZE];
} tPrestoredSection;

tPrestoredSection *prestoredSectionsTable = NULL;


#define kMAX_MODULE_ID 0xFFFF
#define kNOT_REQUESTED 0
#define kDO_INJECT 1
U8BIT *requestedModuleIDList = NULL;

U16BIT GBL_NewResquetedModuleID = 0;

#endif /* ACCELERATE_CAROUSEL_LOADING */

/*------------------- Local prototypes/forward declarations ------------------*/

static E_DscError processDSI( P_DsmCoreInst idp,
   /*I*/ U8BIT *pDSISection, P_SecFilterInfo pFilterInfo );

static E_DscError processDII( P_DsmCoreInst idp,
   /*I*/ U8BIT *pDIISection, P_SecFilterInfo pFilterInfo );

static E_DscError processDDB( P_DsmCoreInst idp,
   /*I*/ U8BIT *pDDBSection, P_SecFilterInfo pFilterInfo );

#ifdef ACCELERATE_CAROUSEL_LOADING

static E_DscError storeSection( P_DsmCoreInst idp,
   U8BIT *pDDBSection);
static E_DscError injectStoredSections( P_DsmCoreInst idp);

E_DscError requestedModuleIdSet(U16BIT moduleID);

E_DscError requestedModuleIdReset(U16BIT moduleID);

#endif /* ACCELERATE_CAROUSEL_LOADING */

/*---------------------------- Exported Functions ----------------------------*/

/**************************************/
/* -- Main (synchronous) API Calls -- */
/**************************************/

void CDSM_SetPrintFuncs( F_Printf errorFunc, F_Printf warnFunc,
   F_Printf debugFunc, F_Printf infoFunc )
{
#if (!defined(NDEBUG) && !defined(NO_DP))
   DBG_ErrorPrintfFunc = errorFunc;
   DBG_WarnPrintfFunc = warnFunc;
   DBG_DebugPrintfFunc = debugFunc;
   DBG_InfoPrintfFunc = infoFunc;
#endif
}

void CDSM_SetPrintState( U32BIT state )
{
#if (!defined(NDEBUG) && !defined(NO_DP))
   dsmDbgState = state;
#endif
}

/**
 * @brief   Set Debug mask for core DSM.
 * @param   dsm DSMCC core instance handle.
 * @param   dbgMask Mask to filter debug messages
 * @return
 */
void CDSM_SetDebugState( H_DsmCoreInst dsm, U32BIT dbgMask )
{
#ifndef NDEBUG
   dsm->dbgMask = (U16BIT)dbgMask;
#endif
}

/*
-- API CALLS INITIATED IN RESPONSE TO SYSTEM EVENTS
*/

/*
-- Create DSM-CC Core Layer instance
*/
E_DscError CDSM_SysCreate(
   /*I*/ P_DsmSetup pSetup,
   /*O*/ H_DsmCoreInst *pInstance, void **pMemContext )
{
   P_DsmCoreInst idp = NULL;
   E_DscError err = CLDSM_OK;
   E_DsmMemErr memErr = MEM_ERR_ALLOC_FAILED;
   U32BIT sfHeapSize;

   dsmDP1(("CDSM_SysCreate( %p, %p, %p )\n", pSetup, pInstance, pMemContext));


   /* -- memory size & usage info prints/checks */
   dsmDP3(("\n-----------------------------------------\n"));
   dsmDP3(("DEBUG CHECKS ON MEMORY SIZES (bytes):\n\n"));
   dsmDP3(("INFO: LinkList Ctrl block size = %u\n", sizeof(S_LLControl)));
   dsmAssert((sizeof(S_LLControl) <= MIN_MEM_BLK_SIZE));
   dsmDP3(("INFO: Object Carousel size     = %u\n", sizeof(S_ObjectCarousel)));
   dsmAssert((sizeof(S_ObjectCarousel) <= MIN_MEM_BLK_SIZE));
   dsmDP3(("INFO: Data Carousel size       = %u\n", sizeof(S_DataCarousel)));
   dsmAssert((sizeof(S_DataCarousel) <= MIN_MEM_BLK_SIZE));
   dsmDP3(("INFO: Module size              = %u\n", sizeof(S_Module)));
   dsmAssert((sizeof(S_Module) <= MIN_MEM_BLK_SIZE));
   dsmDP3(("INFO: Module builder size      = %u\n", sizeof(S_ModuleBuilder)));
   dsmAssert((sizeof(S_ModuleBuilder) <= MIN_MEM_BLK_SIZE));
   dsmDP3(("INFO: LoadRequest size         = %u\n", sizeof(S_LoadRequest)));
   dsmAssert((sizeof(S_LoadRequest) <= MIN_MEM_BLK_SIZE));
   dsmDP3(("INFO: DSM Object size          = %u\n", sizeof(S_DsmObject)));
   dsmAssert((sizeof(S_DsmObject) <= MIN_MEM_BLK_SIZE));
   dsmDP3(("INFO: Max pathname size        = %u\n", MAX_PATH_NAME_SIZE));
   dsmAssert((MAX_PATH_NAME_SIZE <= MIN_MEM_BLK_SIZE));

   dsmDP3(("INFO: Section filter size      = %u\n", sizeof(S_SecFilterInfo)));
   dsmDP3(("-----------------------------------------\n"));

   dsmDP3(("\nINFO: Core Instance data size (including zlib heap) = %u bytes\n",
           sizeof(clDsmInstData_t)));

   *pInstance = NULL;
   *pMemContext = NULL;

   API_PARAM_CHK( pInstance, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );
   API_PARAM_CHK( pMemContext, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );
   API_PARAM_CHK( pSetup, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );
   API_PARAM_CHK( pSetup->allocFunc, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );
   API_PARAM_CHK( pSetup->freeFunc, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );
   /* API_PARAM_CHK( pSetup->errorFunc, dsmDP1(("ERROR: Illegal setup\n")),
                               err = CLDSM_ERR_ILLEGAL_SETUP; goto _return ); */
   API_PARAM_CHK( pSetup->addSectionFilterFunc, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );
   API_PARAM_CHK( pSetup->delSectionFilterFunc, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );
   API_PARAM_CHK( pSetup->startSIQueryFunc, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );
   API_PARAM_CHK( pSetup->stopSIQueryFunc, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );
   API_PARAM_CHK( pSetup->startTimerFunc, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );
   API_PARAM_CHK( pSetup->stopTimerFunc, dsmDP1(("ERROR: Illegal setup\n")),
      err = CLDSM_ERR_ILLEGAL_SETUP; goto _return );

   /* TODO: error check other setup parameters */

   /* -- Allocate instance memory */
   idp = pSetup->allocFunc(sizeof(S_DsmCoreInst));

   if (!idp)
   {
      dsmDP1(("ERROR: Unable to allocate instance memory\n"));
      err = CLDSM_ERR_ALLOC_FAILED;
      goto _return;
   }

   /* -- Make copy of setup info */
   idp->setup = *pSetup;


   /* -- Initialise recursion depth counters */
   idp->pfdlfCallDepth = 0;
   idp->pfrdoCallDepth = 0;
   idp->pfsdndCallDepth = 0;


   /* -- Check SI query result store and turbo-caching are only active
      -- if explicitly enabled */
   if (idp->setup.storeSIQueryResults != TRUE)
   {
      idp->setup.storeSIQueryResults = FALSE;
   }

   if (idp->setup.turboCaching != TRUE)
   {
      idp->setup.turboCaching = FALSE;
   }

   /* -- Initialise instance data */

   /* -- Initialise list handles */
   idp->llcRootCarousels = NULL;
   idp->llcLoadedModules = NULL;
   idp->llcModulePriority = NULL;
   idp->llcExpiredModules = NULL;
   idp->llcCurrLoadRequests = NULL;
   idp->llcCurrSiQueries = NULL;

   idp->hSubscribedEventList = NULL;

   /* -- Initialise current service info */
   idp->dvbLocator.service_id = 0;
   idp->currentServiceSet = FALSE;
   idp->cacheFull = FALSE;


   /* Initialise the stale generation counter for siQueries and Sections */
   idp->generation.ptr = NULL;

   /* Initialise cache memory usage */
   idp->cacheMemoryUsage = 0;
   idp->cacheMaximumUsage = 0;

   /* -- NULL booting carousel reference */
   idp->pBootingCarousel = NULL;
   idp->pCurrentCarousel = NULL;

   /* -- Initialise SI query result store */
   idp->lastPIDResultStored = FALSE;
   idp->lastServiceId = 0;
   idp->lastAssociationTag = 0;
   idp->lastPID = 0;

   /* -- Set number of open objects to 0 */
   idp->dsmObjectsOpen = 0;

   if (pSetup->maxAvailableSectionFilters == 0 ||
       pSetup->maxAvailableSectionFilters > NUM_SECTION_FILTERS_MAXIMUM)
   {
      sfHeapSize = NUM_SECTION_FILTERS_MAXIMUM;
   }
   else if (pSetup->maxAvailableSectionFilters < NUM_SECTION_FILTERS_MINIMUM)
   {
      sfHeapSize = NUM_SECTION_FILTERS_MINIMUM;
   }
   else
   {
      sfHeapSize = pSetup->maxAvailableSectionFilters;
   }

   if (pSetup->maxMemorySize < DSMCC_MINIMUM_CACHE_SIZE)
   {
      dsmDP2(("WARN: setting cache size to minimum %d\n", DSMCC_MINIMUM_CACHE_SIZE));
      pSetup->maxMemorySize = DSMCC_MINIMUM_CACHE_SIZE;
   }

   /* -- Create and initialise section filter heap memory */
   idp->sectionFilterHeap = NULL;
   err = DSC_SsectionFilterHeapCreate( idp, &idp->sectionFilterHeap, &sfHeapSize );

   if (err)
   {
      dsmDP1(("ERROR: Unable to create section filter heap\n"));
      idp->sectionFilterHeap = NULL;
      goto _return;
   }

   /* -- Initialise zlib heap (module decompress) memory */
   moduleDecompressInit( idp );

   memErr =
      memStart( MIN_MEM_BLK_SIZE, pSetup->maxMemorySize - (sizeof(S_DsmCoreInst) + sfHeapSize),
         (NUM_INTERNAL_MEM_SEQ_MAX + NUM_OPEN_DSM_OBJECTS_MAX),
         idp->setup.memMgrSetup,
         &idp->memContext );

   if (memErr != MEM_NO_ERR)
   {
      dsmDP1(("ERROR: Unable to start memory manager\n"));
      err = CLDSM_ERR_MEMMGR_START_PROBLEM;
      idp->memContext = NULL;
      goto _return;
   }

#ifdef ACCELERATE_CAROUSEL_LOADING
   prestoredSectionsTable =
      (tPrestoredSection *)pSetup->allocFunc( sizeof(tPrestoredSection) * kMAX_PRESTORED_SECTIONS );

   accCarouselLoadingInfos.bLargeDDBFilterSet = FALSE;
   accCarouselLoadingInfos.largeDDBFilter = NULL;

   requestedModuleIDList =
      (U8BIT *)pSetup->allocFunc(kMAX_MODULE_ID + 1);
#endif

   /* -- Create default lists etc. */
   err = LLCreate( idp, NULL, ROOT_CAROUSEL_LIST,
         &idp->llcRootCarousels );
   if (err)
      goto _return;                                                 /* EXIT */

   err = LLCreate( idp, NULL, MODULE_LOADED_LIST,
         &idp->llcLoadedModules );
   if (err)
      goto _return;                                                 /* EXIT */

   err = LLCreate( idp, NULL, MODULE_PRIORITY_LIST,
         &idp->llcModulePriority );
   if (err)
      goto _return;                                                 /* EXIT */

   err = LLCreate( idp, NULL, MODULE_DELETE_LIST,
         &idp->llcExpiredModules );
   if (err)
      goto _return;                                                 /* EXIT */

   err = LLCreate( idp, NULL, CURR_LOAD_REQUEST_LIST,
         &idp->llcCurrLoadRequests );
   if (err)
      goto _return;                                                 /* EXIT */

   err = LLCreate( idp, NULL, CURR_SI_QUERY_LIST,
         &idp->llcCurrSiQueries );
   if (err)
      goto _return;                                                 /* EXIT */

   *pMemContext = idp->memContext;
   *pInstance = idp;

_return:
   if (idp)
   {
      if (err)
      {
         /* -- Destroy any allocated memory */
         if (idp->llcRootCarousels)
         {
            LLDestroy( idp, &idp->llcRootCarousels );
         }
         if (idp->llcLoadedModules)
         {
            LLDestroy( idp, &idp->llcLoadedModules );
         }
         if (idp->llcModulePriority)
         {
            LLDestroy( idp, &idp->llcModulePriority );
         }
         if (idp->llcExpiredModules)
         {
            LLDestroy( idp, &idp->llcExpiredModules );
         }
         if (idp->llcCurrLoadRequests)
         {
            LLDestroy( idp, &idp->llcCurrLoadRequests);
         }
         if (idp->llcCurrSiQueries)
         {
            LLDestroy( idp, &idp->llcCurrSiQueries );
         }

         if (memErr == MEM_NO_ERR)
         {
            memStop( MEM_CONTEXT );
         }
         if (idp->sectionFilterHeap)
         {
            idp->setup.freeFunc( idp->sectionFilterHeap );
         }
         idp->setup.freeFunc( idp );
      }
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: clDsmSysCreate: %u\n", err)));
   dsmDP1(("exit clDsmSysCreate -> rtn: %u, o/p: %p, %p \n",
           err, *pInstance, *pMemContext));
   return err;
}

/*
-- Destroy DSM-CC Core Layer instance
*/
E_DscError CDSM_SysDestroy( H_DsmCoreInst instance,
   H_SiqInstance *pSiqInst, H_SfmInstance *pSfmInst )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   E_DscError err = CLDSM_OK;
   E_DsmMemErr memErr = MEM_NO_ERR;

   dsmDP2(("CDSM_SysDestroy( %p )\n", instance));

   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );

   /* -- Current service setting is only reset after clDsmSysCreate or after a
      -- successful clDsmSysReset call */
   if (idp->currentServiceSet)
   {
      err = CLDSM_ERR_INSTANCE_NOT_RESET;
   }
   else
   {
      /* -- Only safe to do these if instance is reset */
      LLDestroy( idp, &idp->llcRootCarousels );
      LLDestroy( idp, &idp->llcLoadedModules );
      LLDestroy( idp, &idp->llcModulePriority );
      LLDestroy( idp, &idp->llcExpiredModules );
      LLDestroy( idp, &idp->llcCurrLoadRequests );
      LLDestroy( idp, &idp->llcCurrSiQueries );
   }

   /* -- Free section filter heap memory */
   dsmAssert((idp->sectionFilterHeap != NULL));
   idp->setup.freeFunc( idp->sectionFilterHeap );

#ifdef ACCELERATE_CAROUSEL_LOADING
   idp->setup.freeFunc((tPrestoredSection *)prestoredSectionsTable );
   idp->setup.freeFunc((U8BIT *)requestedModuleIDList );
#endif

   /* -- Always stop memory manager regardless of any error conditions
      -- NB. memStop will return an error if there are any allocated
      -- memory objects but it still frees it's memory heap */
   memErr = memStop( MEM_CONTEXT );

   if ((!err) && memErr)
   {
      err = CLDSM_ERR_MEMMGR_STOP_PROBLEM;
   }
   *pSiqInst = idp->setup.siqInstance;
   *pSfmInst = idp->setup.sfmInstance;

   idp->setup.freeFunc( idp );

   goto _return;    /* -- Prevents compiler warnings when no API checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: clDsmSysDestroy: %u\n", err)));
   dsmDP2(("exit clDsmSysDestroy -> rtn: %u \n", err));
   return err;
}

/**
 * @brief   Get control handle from DSM instance .
 * @param   instance
 * @return  NULL if an invalid instance handle, otherwise returns control handle
 */
H_DsmControl CDSM_SysGetControl( H_DsmCoreInst instance )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   return idp->setup.dsmControl;
}

E_DscError CDSM_UnloadAllCarousels( P_DsmCoreInst idp, U8BIT mode )
{
   H_DsmCarousel hOC;
   E_DscError intErr = CLDSM_OK;

   /* -- Any active carousel loads will still be in the carousel list
    -- so check if there is anything in it */
   hOC = LLHead( idp->llcRootCarousels );

   /* -- Forcibly unload/destroy any active carousel loads */

   while (hOC && !intErr)
   {
      intErr = CDSM_UnloadCarousel( idp, hOC, mode );

      /* -- Any error is internal since we are internally calling
         -- an API function) */
      dsmAssert((intErr == CLDSM_OK));

      if (!intErr)
      {
         /* -- Check if there anything else left in the carousel
            -- list */
         hOC = LLHead( idp->llcRootCarousels );
      }
   }

   return intErr;
}

static void CDSM_UnloadCarouselsOnService( P_DsmCoreInst idp, U16BIT serviceId )
{
   E_DscError err;
   P_RootCarousel pRC, pNext;
   ListId_t listId;

   listId = LListId( idp->llcRootCarousels );
   pRC = LLHead( idp->llcRootCarousels );
   while (pRC != NULL)
   {
      pNext = LLNext( pRC, listId );
      if (pRC->serviceId == serviceId)
      {
         err = CDSM_UnloadCarousel( idp, (H_DsmCarousel)pRC, RST_MODE_FORCE );
         if (err)
         {
            ERRPRINT("Unload carousel err=%d", err)
         }
      }
      pRC = pNext;
   }
}

static void ClearAllQueries( P_DsmCoreInst idp )
{
   P_SiQuery pSiQueryRef;
   E_SiQueryState queryState;

   /* -- Any active SI queries will still be in the current SI query
    -- list so check if there is anything in it */
   pSiQueryRef = LLHead( idp->llcCurrSiQueries );
   while (pSiQueryRef)
   {
      /* -- Copy internal state since query ref will be destroyed */
      queryState = pSiQueryRef->state;

      /* -- If SI Query is not in an ABORTED or EXPIRED state there has
         -- been an internal error */
      if ((queryState == SIQS_ABORTED) || (queryState == SIQS_EXPIRED))
      {
         dsmDP2(("ClearAllQueries: Stopping SI Query=%p state=%d\n", pSiQueryRef, queryState));
         /* -- Force stop/destroy the SI Query (ie. without waiting for
            -- any external abort/ack.)
            -- NB. This also removes it from the current query list */
         siQueryStop( idp, pSiQueryRef );
      }
      else
      {
         /* -- NB. Since there is probably some internal memory
            -- corruption it is safest not to do anything else here.
            -- DSM-CC should be shut down (clDsmDestroy) in response to
            -- to this internal error otherwise this may cause a memory
            -- leak in the MemMgr */
         ERRPRINT("Invalid query state")
         /* should it do:
          * LLRemove( pSiQueryRef, CURR_SI_QUERY_LIST );
          */
         break;
      }
      pSiQueryRef = LLHead( idp->llcCurrSiQueries );
   }
}

static void UnloadLoadRequests( P_DsmCoreInst idp )
{
   P_LoadRequest pNextRequest;
   P_LoadRequest pLoadRequest;
   ListId_t listId;

   listId = LListId( idp->llcCurrLoadRequests );
   pLoadRequest = LLHead( idp->llcCurrLoadRequests );

   do
   {
      pNextRequest = LLNext( pLoadRequest, listId );
      ERRPRINT("Load request status=%d", pLoadRequest->rlr.status)
      DSC_LoadRsqtDestroy( idp, (P_RootLoadRqst)pLoadRequest );
      pLoadRequest = pNextRequest;
   }
   while (pLoadRequest);
}

static void UnloadModules( P_DsmCoreInst idp, P_LLControl llc, ListId_t listId )
{
   P_Module pModule;
   P_Module pNextMod;
   pModule = LLHead( llc );
   do
   {
      pNextMod = LLNext( pModule, listId );
      dsmAssert((pModule->loadedCount == 0));
      dsmAssert((pModule->llcLoadRequests == NULL));
      DSC_ModuleDeleteDcTidyUp( idp, pModule );
      pModule = pNextMod;
   }
   while (pModule);
}

/*
-- Reset DSM-CC Core Layer instance
*/
E_DscError CDSM_SysReset( H_DsmCoreInst instance, E_DsmRstMode mode )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   E_DscError err = CLDSM_OK;
   int listCount;

   dsmDP2(("CDSM_SysReset( %p, %u )\n", instance, mode));

   if (!idp || !idp->currentServiceSet)
   {
      ERRPRINT("ERROR: invalid instance? %p", idp);
   }
   else
   {
      err = CDSM_UnloadAllCarousels( idp, mode );
      if (err)
      {
         ERRPRINT("Failed to unload all carousels err=%d", err)
      }
      else
      {
         if (mode != RST_MODE_PENDING)
         {
            listCount = LLCount( idp->llcCurrLoadRequests );
            if (listCount)
            {
               ERRPRINT("Load requests still present, count=%d", listCount)
               UnloadLoadRequests( idp );
            }
         }
         listCount = LLCount( idp->llcModulePriority );
         if (listCount)
         {
            ERRPRINT("Priority Modules still present, count=%d", listCount)
            UnloadModules( idp, idp->llcModulePriority, MODULE_PRIORITY_LIST );
         }
         listCount = LLCount( idp->llcLoadedModules );
         if (listCount)
         {
            ERRPRINT("Loaded Modules still present, count=%d", listCount)
            UnloadModules( idp, idp->llcLoadedModules, MODULE_LOADED_LIST );
         }
         listCount = LLCount( idp->llcExpiredModules );
         if (listCount)
         {
            ERRPRINT("Expired Modules still present count=%d", listCount)
            UnloadModules( idp, idp->llcExpiredModules, MODULE_DELETE_LIST );
         }

         ClearAllQueries( idp );

         DSC_StrmEventListReset( idp );

         /* -- Re-initialise instance data (may not all be necessary) */
         dsmDP3(("lastPID=%d lpStored=%d, lastAssocTag=%d",
                 idp->lastPID, idp->lastPIDResultStored, idp->lastAssociationTag));

         if (idp->dvbLocator.service_id != 0)
         {
            if (idp->setup.unsubscribeSIChangeFunc != NULL)
            {
               /* unregister for PMT changes for previous service */
               idp->setup.unsubscribeSIChangeFunc( idp->setup.siqInstance, idp->dvbLocator.service_id );
            }
         }

         /* -- Initialise current service info */
         idp->dvbLocator.service_id = 0;
         idp->currentServiceSet = FALSE;
         idp->cacheFull = FALSE;

         /* -- Initialise SI query result store */
         idp->lastPIDResultStored = FALSE;
         idp->lastAssociationTag = 0;
         idp->lastPID = 0;

         /* -- NULL booting carousel reference */
         idp->pBootingCarousel = NULL;

         /* -- Set number of open objects to 0
          -- nb. should not be any open objects at this point */
         dsmAssert((idp->dsmObjectsOpen == 0));
         idp->dsmObjectsOpen = 0;

         if (mode != RST_MODE_PENDING)
         {
            /* -- Reset section filter heap memory */
            DSC_SsectionFilterHeapReset( idp );
         }
         /* -- Initialise zlib heap (module decompress) memory */
         moduleDecompressInit( idp );
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: clDsmSysReset: %u\n", err)));
   dsmDP2(("exit clDsmSysReset -> rtn: %u \n", err));
   return err;
}

/**
 * @brief   Set maximum memory usage.
 * @param   instance
 * @return  n/a
 */
void CDSM_SysSetMemoryMax( H_DsmCoreInst instance, U32BIT maxMemory )
{
   P_DsmCoreInst idp;
   if (maxMemory > DSMCC_MINIMUM_CACHE_SIZE)
   {
      idp = (P_DsmCoreInst) instance;
      if (idp->setup.maxMemorySize > maxMemory)
      {
         DSC_CmMemPurgeCache( idp, maxMemory );
      }
      idp->setup.maxMemorySize = maxMemory;
   }
}

/**
 * @brief   Set/notify the current service (initially and when changing it).
 *          This MUST be called initially (ie. after clDsmCreate or clDsmReset), before
 *          the boot carousel (or any carousels) can be loaded.
 *          To change the current service (ie. once one is already set) then the client
 *          MUST first unsubscribe all stream events and successfully unload all objects
 *          and carousels (from the current service). If any errors occur then
 *          clDsmReset may need to be called (with force switches - see Additional
 *          information below) before a new (or even the same) service can be set.
 *          **** NB. THIS FUNCTIONALITY IS NOT CURRENTLY IMPLEMENTED ****
 *          When this function is re-called. If the service details match those currently
 *          set then it will take no action. If they are different then the new service
 *          details are stored (unless objects and/or carousels are still loaded, in
 *          which case they are ignored and an error generated).
 *          **** NB. THIS FUNCTIONALITY IS NOT CURRENTLY IMPLEMENTED ****
 *        To forcefully destroy loaded objects and/or carousels (eg. where the handles
 *        may be unknown/lost) then clDsmReset can be used with force switches.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          none
 * @param   instance DSMCC instance handle
 * @param   original_network_id DVB Network Identifier
 * @param   transport_stream_id DVB Transport stream/Multiplex Identifier
 * @param   service_id DVB Service Identifier
 * @return
 *          CLDSM_OK (0)
 *          The new current service details were successfully set.
 *          CLDSM_ERR_CAROUSELS_STILL_LOADED
 *          The new current service details could not be set because one or more
 *          carousels from this service were still loaded
 *          CLDSM_ERR_OBJECTS_STILL_LOADED
 *          The new current service details could not be set because one or more objects
 *          from this service were still loaded
 */
E_DscError CDSM_SysSetCurrService( H_DsmCoreInst instance,
   /*I*/ U16BIT original_network_id, U16BIT transport_stream_id,
   U16BIT service_id )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   E_DscError err = CLDSM_OK;
   MemHandle hOC;

   dsmDP2(("CDSM_SysSetCurrService( %p, %u, %u, %u )\n", instance,
           original_network_id, transport_stream_id, service_id));

   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );

   /* -- Any active carousel loads will still be in the carousel list
      -- so check if there is anything in it */
   hOC = LLHead( idp->llcRootCarousels );

   if (hOC)
   {
      err = CLDSM_ERR_CAROUSELS_STILL_LOADED;
      goto _return;
   }

   dsmDP4(("lastPID=%d lpStored=%d, lastAssocTag=%d",
           idp->lastPID, idp->lastPIDResultStored, idp->lastAssociationTag));

   if (idp->dvbLocator.service_id == service_id &&
       idp->dvbLocator.original_network_id == original_network_id)
   {
      idp->dvbLocator.transport_stream_id = transport_stream_id;
      idp->currentServiceSet = TRUE;
   }
   else
   {
      if (idp->dvbLocator.service_id != 0)
      {
         if (NULL != idp->setup.unsubscribeSIChangeFunc)
         {
            /* unregister for PMT changes for previous service */
            idp->setup.unsubscribeSIChangeFunc( idp->setup.siqInstance, idp->dvbLocator.service_id );
         }
      }
      idp->dvbLocator.original_network_id = original_network_id;
      idp->dvbLocator.transport_stream_id = transport_stream_id;
      idp->dvbLocator.service_id = service_id;
      idp->currentServiceSet = TRUE;
      if (service_id != 0)
      {
         if (NULL != idp->setup.subscribeSIChangeFunc)
         {
            idp->setup.subscribeSIChangeFunc( idp->setup.siqInstance, service_id );
         }
      }
   }

   /* Increment the generation counter */
   idp->generation.u32++;

   /* -- Clear SI query result store for new service */
   idp->lastPIDResultStored = FALSE;
   idp->lastAssociationTag = 0;
   idp->lastPID = 0;

#ifdef ACCELERATE_CAROUSEL_LOADING
   {
      memset(&requestedModuleIDList[0], 0, kMAX_MODULE_ID);

      memset(&prestoredSectionsTable[0], 0, sizeof(tPrestoredSection) * kMAX_PRESTORED_SECTIONS);


      accCarouselLoadingInfos.storeWritePos = 0;
      accCarouselLoadingInfos.storeReadPos = 0;
      accCarouselLoadingInfos.storedSectionsNB = 0;
      accCarouselLoadingInfos.bLargeDDBFilterSet = FALSE;
   }
#endif

_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: clDsmSysSetCurrService: %u\n", err)));
   dsmDP2(("exit clDsmSysSetCurrService -> rtn: %u \n", err));
   return err;
}

U16BIT CDSM_SysCurrServiceId( H_DsmCoreInst instance )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   return idp->dvbLocator.service_id;
}

/*******************************************************************************
*
* Notifies that the specified timer has either timed-out (triggered) or
* been stopped (aborted) before triggering.
*
*******************************************************************************/
E_DscError CDSM_SysProcessTimerEvent( H_DsmCoreInst instance,
   void *clDsmTmrUserData, E_TimerStatus status, void *timerHandle )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   P_SecFilterInfo pFilterInfo = (P_SecFilterInfo)clDsmTmrUserData;

   dsmDP3(("CDSM_SysProcessTimerEvent( %p, %u, %p, %p )\n", instance,
           status, clDsmTmrUserData, timerHandle));

   if (pFilterInfo->tms.mainTimerHandle == timerHandle)
   {
      pFilterInfo->tms.mainTimerHandle = NULL;
   }
   else if (pFilterInfo->tms.nextTimerHandle == timerHandle)
   {
      pFilterInfo->tms.nextTimerHandle = NULL;
   }
   switch (status)
   {
      case TIMER_TRIGGERED:
         switch (pFilterInfo->target.kind)
         {
            case SFK_DSI:
               dsmDP2(("TIMER_TRIGGERED: SFK_DSI: pFilterInfo = 0x%p\n", pFilterInfo));
               if (DSC_RootCrslMagic(pFilterInfo->target.u.pRC) == OC_MAGIC)
               {
                  lmObjCarouselTimeout(idp, pFilterInfo->target.u.pOC);
               }
               DSC_RootCrslLoadRequestFail(idp, pFilterInfo->target.u.pRC);
               break;

            case SFK_DII:
               dsmDP2(("TIMER_TRIGGERED: SFK_DII: pFilterInfo = 0x%p\n", pFilterInfo));
               DSC_DataCrslDelete(idp, pFilterInfo->target.u.pDC);
               break;

            case SFK_DDB:
               dsmDP2(("TIMER_TRIGGERED: SFK_DDB: pFilterInfo = 0x%p\n", pFilterInfo));
               lmAbortLoadRequestsOnModuleTimeout(idp, pFilterInfo->target.u.pModule);
               break;

            default:
               break;
         }
         break;

      case TIMER_ABORTED:
         switch (pFilterInfo->target.kind)
         {
            case SFK_DSI:
               dsmDP3(("TIMER_ABORTED: SFK_DSI: pFilterInfo = 0x%p\n", pFilterInfo));
               break;

            case SFK_DII:
               dsmDP3(("TIMER_ABORTED: SFK_DII: pFilterInfo = 0x%p\n", pFilterInfo));
               break;

            case SFK_DDB:
               dsmDP3(("TIMER_ABORTED: SFK_DDB: pFilterInfo = 0x%p\n", pFilterInfo));
               break;

            default:
               break;
         }
         break;

      default:
         dsmDP2(("TIMER_STAUTS_???: pFilterInfo = 0x%p\n", pFilterInfo));
         break;
   }

   return CLDSM_OK;
}

/**
 * @brief   Notifies the result of the specified SI query (ie. a callback to
 *          startSIQueryFunc that returned SIQUERY_PENDING) or notifies that the
 *          query was stopped (aborted) before completion.
 *          Meaning of status values in this context:
 *          SIQUERY_SUCCESS - Query has returned successful result.
 *          SIQUERY_FAILURE - Query has failed to determine requested information (for
 *          any reason).
 *          SIQUERY_PENDING - *** INVALID VALUE FOR THIS CALL ***
 *          SIQUERY_ABORTED - Query has been stopped before completion.
 *          If the queryHandle supplied here is non-NULL it will supercede any value
 *          returned from the startSIQueryFunc callback (when input to the
 *          stopSIQueryFunc callback).
 *          NB. Although this function should be called synchronously with the other
 *          (synchronous) API functions, it can also be called from within the
 *          stopSIQueryFunc (required API) callback if required.
 *          Discard siQuery when there is no current service or siQueries generation
 *          count is different to the DSMCC instances generation count.
 *        CALLBACKS THAT MAY BE INITIATED DURING THIS CALL:
 *          addSectionFilter
 *          stopSIQuery
 *          notifyObjectLoadEvent
 *          notifyCarouselLoadEvent
 *          deleteSectionFilter
 *          stopTimer
 * @param   instance DSMCC instance handle
 * @param   clDsmSIQueryRef Refererence for this query (value supplied at
 *          start time).
 * @param   clDsmSIUserData User data for this query (value supplied at
 *          start time).
 * @param   status Status of query results.
 * @param   pResult Pointer to results of query (if successful).
 * @param   queryHandle Calling env. handle to query (optional - NULL
 *          if not used).
 * @return
 *          CLDSM_OK (0)
 *          The process SI query event was executed with no problems.
 *          CLDSM_ERR_INVALID_SIQUERY_REF
 *          Supplied clDsmSIQueryRef does not reference a valid query.
 *          CLDSM_ERR_INVALID_SIQUERY_STATUS
 *          Supplied status value is invalid here.
 *          CLDSM_ERR_SYSTEM_ADD_SECTION_FILTER,
 *          Problem with add section filter callback.
 */
E_DscError CDSM_SysProcessSIQueryEvent( H_DsmCoreInst idp,
   H_SIQueryRef clDsmSIQueryRef, void *clDsmSIUserData, P_SIQueryResult pResult )
{
   E_DscError err = CLDSM_OK;
   P_SiQuery pSiQueryRef = (P_SiQuery) clDsmSIQueryRef;
   P_SiQuery pDuplSiQueryRef;
   E_DscError newErr = CLDSM_OK;

   dsmDP2(("CDSM_SysProcessSIQueryEvent( %p, %p, %p )\n", idp, clDsmSIQueryRef, pResult));

   if (idp == NULL)
   {
      dsmDP1(("ERROR: Invalid instance\n"));
     err = CLDSM_ERR_INVALID_INSTANCE;
   }
   else if ((idp->currentServiceSet == FALSE) ||
       (idp->generation.ptr != clDsmSIUserData))
   {
     /* Discard siQuery when there is no current service or siQueries
      generation count is different to generation count in DSMCC instance */
      dsmDP3(("INFO: Stale siQuery discarded (generation)\n"));
   }
   else if (!memValidate(pSiQueryRef))
   {
      dsmDP1(("API ERROR: Invalid SI Query ref (handle)\n"));
      err = CLDSM_ERR_INVALID_SI_QUERY_REF;
   }
   else
   {
      if (pSiQueryRef->magic != SI_QUERY_MAGIC)
      {
         dsmDP1(("API ERROR: Invalid SI Query ref (magic)\n"));
         err = CLDSM_ERR_INVALID_SI_QUERY_REF;
      }
      else if (pSiQueryRef->state == SIQS_COMPLETED)
      {
         dsmDP1(("API ERROR: Invalid SI Query ref (already completed)\n"));
         err = CLDSM_ERR_INVALID_SI_QUERY_REF;
      }
      else
      {
         /* -- Should only be called for original queries */
         dsmAssert((pSiQueryRef->original == TRUE));

         if (pSiQueryRef->llcDuplSiQuerys != NULL)
         {
            /* -- Process the duplicates first */
            pDuplSiQueryRef = LLRemoveHead( pSiQueryRef->llcDuplSiQuerys );

            while (pDuplSiQueryRef != NULL)
            {
               newErr = siQueryProcessResult( idp, pDuplSiQueryRef, pResult );
               if (newErr)
               {
                  err = handleInLoopError( idp, err, newErr );
               }
               pDuplSiQueryRef = LLRemoveHead( pSiQueryRef->llcDuplSiQuerys );
            }
            /* -- Now destroy the duplicate list */
            LLDestroy( idp, &pSiQueryRef->llcDuplSiQuerys );
         }
         /* -- Now process the original query */
         newErr = siQueryProcessResult( idp, pSiQueryRef, pResult );
         if (newErr)
         {
            err = handleInLoopError( idp, err, newErr );
         }
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: clDsmSysProcessSIQueryEvent: %u\n", err)));
   dsmDP2(("exit clDsmSysProcessSIQueryEvent -> rtn: %u \n", err));
   return err;
}

/*******************************************************************************
*
* Notify that the SI for the indicated service has changed
*
*******************************************************************************/
E_DscError CDSM_SysProcessSIChangeEvent( H_DsmCoreInst instance,
   E_SIChangeEvent event, U16BIT service_id, U32BIT carousel_id )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   E_DscError err = CLDSM_OK;
   H_DsmCarousel pOC = NULL;

   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );

   dsmDP2(("CDSM_SysProcessSIChangeEvent( %p, %u, %u )\n", instance,
           service_id, event));

   dsmDP4(("lastPID=%d lpStored=%d, lastAssocTag=%d",
           idp->lastPID, idp->lastPIDResultStored, idp->lastAssociationTag));


   switch (event)
   {
      case SICHANGE_SERVICE_UPDATED:
         if (idp->dvbLocator.service_id == service_id)
         {
            idp->lastPIDResultStored = FALSE;
         }
         /* refresh filters */
         err = pmtUpdtAllSectionFiltersReset(instance, service_id);
         break;

      case SICHANGE_CAROUSEL_DELETED:
         pOC = (H_DsmCarousel)DSC_RootCrslListFindById( idp->llcRootCarousels, service_id, carousel_id );
         if (pOC != NULL)
         {
            err = CDSM_UnloadCarousel( instance, pOC, RST_MODE_FORCE );
         }
         break;

      case SICHANGE_SERVICE_DELETED:
         CDSM_UnloadCarouselsOnService( instance, service_id );
         break;

      default:;
   }

_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: clDsmSysProcessSIChangeEvent: %u\n", err)));
   dsmDP2(("exit clDsmSysProcessSIChangeEvent -> rtn: %u \n", err));
   return err;
}

/*
-- API CALLS INITIATED IN RESPONSE TO TRANSPORT STREAM EVENTS:
*/

E_DscError CDSM_SectionPriority( H_DsmCoreInst instance,
   H_DscSFRef dsmFilterRef, E_SFPriority *priority )
{
   E_DscError err;
   P_DsmCoreInst idp;
   P_SecFilterInfo pFilterInfo;
   if (instance == NULL)
   {
      err = CLDSM_ERR_INVALID_INSTANCE;
      ERRPRINT("ERROR: Invalid instance")
   }
   else
   {
      idp = (P_DsmCoreInst) instance;
      pFilterInfo = DSC_SectionFilterRetrieve( idp, dsmFilterRef );
      if (pFilterInfo == NULL)
      {
         err = CLDSM_ERR_ABORTED;
         DBGLOG(DD_SF, "Stale section discarded")
      }
      else
      {
         err = CLDSM_OK;
         *priority = pFilterInfo->filter.priority;
      }
   }
   DBGLOG(DD_SF, "rtn: %u", err)
   return err;
}

/*
-- Process DSM-CC private section data
--
-- On entry - pSection points at first byte of section (ie. tableId)
--
*/
E_DscError CDSM_SysProcessPrivateSection( H_DsmCoreInst idp,
   /*I*/ U8BIT *pSection, H_DscSFRef dsmFilterRef )
{
   E_DscError err;
   P_SecFilterInfo pFilterInfo;
   #ifdef ACCELERATE_CAROUSEL_LOADING
   U8BIT *pTempSection = pSection;
   #endif /* ACCELERATE_CAROUSEL_LOADING */

   DBG3(DD_SF, "CDSM_SysProcessPrivateSection( %p, %p, %p )", idp, pSection, dsmFilterRef)

   if (idp == NULL)
   {
      err = CLDSM_ERR_INVALID_INSTANCE;
      ERRPRINT("ERROR: Invalid instance")
   }
   else if (pSection == NULL)
   {
      err = CLDSM_ERR_ILLEGAL_PARAMETER;
      ERRPRINT("ERROR: Invalid section")
   }
   else
   {
      dsmDP3(("%s sz=0x%x tid=0x%x tide=0x%x, %d of %d\n", __FUNCTION__,
              (((int)(pSection[1] & 0x0f) << 8) | pSection[2]) + 3, *pSection,
              ((pSection[3] << 8) | pSection[4]), pSection[6] + 1, pSection[7] + 1));

      pFilterInfo = DSC_SectionFilterRetrieve( idp, dsmFilterRef );
      if (pFilterInfo == NULL)
      {
         err = CLDSM_ERR_ABORTED;
         DBGLOG(DD_SF, "Stale section discarded ref=0x%x", dsmFilterRef)
      }
      else
      {
         switch (pFilterInfo->target.kind)
         {
            case SFK_DSI:
               err = processDSI( idp, pSection, pFilterInfo );
               break;

            case SFK_DII:
            #ifdef ACCELERATE_CAROUSEL_LOADING
               GBL_NewResquetedModuleID = 0;
            #endif
               err = processDII( idp, pSection, pFilterInfo );

            #ifdef ACCELERATE_CAROUSEL_LOADING
               if (accCarouselLoadingInfos.storedSectionsNB > 0)
               {
                  err = injectStoredSections(idp);
               }
            #endif
               break;

            case SFK_DDB:
            #ifdef ACCELERATE_CAROUSEL_LOADING
               /* processDDB moves pSection pointer !!*/
               pTempSection = pSection;
            #endif

               err = processDDB( idp, pSection, pFilterInfo );

            #ifdef ACCELERATE_CAROUSEL_LOADING
               if (CLDSM_OK != err)
               {
                  /* store current section */
                  storeSection(idp, pTempSection);
               }
            #endif
               break;

            case SFK_STREAM_DESCR:
               err = DSC_StrmEventUpdate( idp, pSection, pFilterInfo->target.u.pEvent, pFilterInfo->target.id );
               break;

            default:
               /* -- Discard sections from old filters */
               DBGLOG(DD_SF, "WARNING: Stale/invalid section filter handle (targetKind)")
               err = CLDSM_ERR_ABORTED;
               break;
         }
         /* -- Filter and only return system or internal (ie. fatal) errors here */
         switch (err)
         {
            case CLDSM_ERR_INTERNAL:
            case CLDSM_ERR_SYSTEM_ADD_SECTION_FILTER:
            case CLDSM_ERR_SECTION_FILTER_HEAP_FULL:
            case CLDSM_ERR_MEM_HEAP_FULL:
               DBGLOG(DD_SF, "rtn: %u", err)
               break;

            /* -- Do not notify or return any other errors */
            default:
               err = CLDSM_OK;
               break;
         }
      }
   }
   return err;
}

#ifdef ACCELERATE_CAROUSEL_LOADING
E_DscError internalSysProcessPrivateSection( H_DsmCoreInst instance,
   /*I*/ U8BIT *pSection, P_SecFilterInfo pFilterInfo )
{
   U8BIT tableId;
   U16BIT tableIdExtension;
   E_DscError err = CLDSM_OK;
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;

   /* -- Set DP level to 3 to prevent excessive debug prints at default
      -- level (L2) caused by DII monitoring */
   /* TODO: Set back to DP2 when 'next DII' monitoring implemented? */
   dsmDP3(("internalSysProcessPrivateSection( %p, %p, %x )\n",
           instance, pSection, clDsmFilterRef));
   API_PARAM_CHK( instance, dsmDP1(("ERROR: Invalid instance\n")),
      err = CLDSM_ERR_INVALID_INSTANCE; goto _return );
   API_PARAM_CHK( pSection, dsmDP1(("ERROR: Illegal parameter\n")),
      err = CLDSM_ERR_ILLEGAL_PARAMETER; goto _return );

   /* Discard Section when there is no current service or Section
      generation count is different to generation count in DSMCC instance */
   if (idp->currentServiceSet == FALSE)
   {
      dsmDP3(("INFO: Stale section discarded (generation)\n"));
      goto _return;                                                 /* EXIT */
   }

   if (pFilterInfo)
   {
      switch (pFilterInfo->target.kind)
      {
         case SFK_DDB:
            err = processDDB( idp, pSection, pFilterInfo );
            break;

         default:
            /* -- Discard sections from old filters */
            dsmDP2(("WARNING: Stale/invalid section filter handle (targetKind)\n"));
            break;
      }
      /* -- Filter and only return system or internal (ie. fatal) errors here */
      switch (err)
      {
         case CLDSM_ERR_INTERNAL:
         case CLDSM_ERR_SYSTEM_ADD_SECTION_FILTER:
         case CLDSM_ERR_SYSTEM_START_TIMER:
         case CLDSM_ERR_SECTION_FILTER_HEAP_FULL:
         case CLDSM_ERR_MEM_HEAP_FULL:
            break;
         /* -- Do not notify or return any other errors */
         default:
            err = CLDSM_OK;
            break;
      }
   }
   else
   {
      /* -- Cannot determine target Data Carousel */
      err = CLDSM_ERR_ILLEGAL_PARAMETER;
      dsmDP1(("API ERROR: No section filter reference supplied\n"));
   }

_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: clDsmSysProcessPrivateSection: %u\n", err)));

   /* -- Set DP level to 3 to prevent excessive debug prints at default
      -- level (L2) caused by DII monitoring */
   /* TODO: Set back to DP2 when 'next DII' monitoring implemented */
   dsmDP3(("exit clDsmSysProcessPrivateSection -> rtn: %u \n", err));
   return err;
}

#endif /* ACCELERATE_CAROUSEL_LOADING */


/*------------------------------ Local Functions -----------------------------*/

/* -- SECTION PROCESSING FUNCTIONS */

static U8BIT* ValidateDSI( U8BIT *pData, U16BIT *pPrivateDataLength, U32BIT *pTransactionId )
{
   U16BIT ui16, msglen;
   U8BIT ui8, adaptationHdrLen;

   /* -- tableId = 0x3B */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x3B)
   {
      DBG1(DD_OC, "DATA ERROR: DSI tableId = %x", ui8)
      return NULL;
   }
   /* -- dsmcc_section_length */
   /* DSMCC specifies:
   --  - Maximum DSMCC message length is from last_section_number field to
   --    start of CRC_32/checksum field (ie. dsmcc_section_length - 9)
   --      - Ref[1] - section 9.2.2.1, pg 287
   */
   READ_UINT16( pData, ui16 );
   if (((ui16 & LSB12_MSK) > MAX_SECTION_LEN) ||
       (((ui16 & LSB12_MSK) - 9) < MIN_DSI_MSG_LEN))
   {
      DBG1(DD_OC, "DATA ERROR: DSI section len = %x", ui16)
      return NULL;
   }

   /* -- tableIdExtension = 0x0000 or 0x0001 */
   READ_UINT16( pData, ui16 );
   if (ui16 > 1)
   {
      DBG1(DD_OC, "DATA ERROR: DSI tableIdExtension = %x", ui16)
      return NULL;
   }

   /* -- Skip versionNumber, sectionNumber, lastSectionNumber - NOT USED */
   SET_POS_REL( pData, 3 );

   /* -- pDSISection -> dsmccMessageHeader */
   /* -- protocolDescriminator */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x11)
   {
      DBG1(DD_OC, "DATA ERROR: protocolDescriminator = %x", ui8)
      return NULL;
   }

   /* -- dsmccType = U-N Download Message */
   /* -- L1 check because this should be correct */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x03)
   {
      DBG1(DD_OC, "DATA ERROR: DSI dsmccType = %x", ui8)
      return NULL;
   }

   /* -- messageID = DownloadServerInitiate */
   /* -- L1 check because this should be correct */
   READ_UINT16( pData, ui16 );
   if (ui16 != DSI_MSG_ID)
   {
      DBG1(DD_OC, "DATA ERROR: DSI messageID = %x", ui16)
      return NULL;
   }

   /* -- transactionId: originator bits = 10B (DSM-CC [1]) */
   /* -- NK 16/10/01 - L2 check since some transmissions are known to set these wrong */
   READ_UINT32( pData, *pTransactionId );
   if ((*pTransactionId & TRANSACTION_ID_ORIG_MASK) != 0x80000000)
   {
      DBG1(DD_OC, "DATA ERROR: DSI transactionId (originator (bits 31-30) != 10B) = %x", *pTransactionId)
      return NULL;
   }
   /* -- transactionId: ID bits = 0x0000 (DVB [2]) */
   if ((*pTransactionId & TRANSACTION_ID_IDENT_MASK) != 0x00000000)
   {
      DBG1(DD_OC, "DATA ERROR: DSI transactionId (ID (bits 15-1) != 0x0000) = %x", *pTransactionId)
      return NULL;
   }

   /* -- Skip reserved data - NOT USED */
   SET_POS_REL( pData, 1 );

   /* -- Read adaptationLength */
   READ_UINT8( pData, adaptationHdrLen );

   /* -- Check messageLength */
   READ_UINT16( pData, msglen );
   if (msglen < (DSI_HDR_LEN+adaptationHdrLen))
   {
      DBG1(DD_OC, "DATA ERROR: DSI messageLength (< DSI_HDR_LEN) = %u", ui16)
      return NULL;
   }
   if (adaptationHdrLen)
   {
      /* -- Skip adaptationHeader - NOT USED */
      SET_POS_REL( pData, adaptationHdrLen );
      msglen -= adaptationHdrLen;
   }

   /* -- pDSISection -> DSIMessageBody */

   /* -- Skip serverId field - NOT USED (DVB) */
   SET_POS_REL( pData, 20 );

   /* --  compatibilityDescriptorLength == 0 - NOT USED (DVB) */
   READ_UINT16( pData, ui16 );
   if (ui16 != 0x0000)
   {
      DBG3(DD_OC, "DSI compatibilityDescriptorLength (!= 0) = %u", ui16)
      /* this is error according to MHEG or MHP/HBBTV or SSU spec's and should do:
       * return NULL;
       * But some streams may have it non-zero (e.g. with length of 'decriptorCount' only - i.e. 2)
       * We can just skip over this descriptor, anyway.
       */
      SET_POS_REL( pData, ui16 );
      msglen -= ui16;
   }
   msglen -= DSI_HDR_LEN;

   /* -- Check privateDataLength (ServiceGatewayInfo or GroupInfoIndication) */
   READ_UINT16( pData, *pPrivateDataLength );
   if (msglen != *pPrivateDataLength)
   {
      DBG1(DD_OC, "DATA ERROR: DSI private data len, %u, not matching msgLen = %u", *pPrivateDataLength, msglen)
      return NULL;
   }

   /* Return pointer to private data */
   return pData;
}

/*
// *** NB. Currently only ONE objectCarousel supported ***
// *** TODO: Do we need to handle DSI updates ? ***
//
//  If a carousel boot is in progress
//
//      Get SRG info from dsiMessage
//
//      Identify target objectCarousel for this DSI section
//
//      If target objectCarousel identified
//          Delete DSI section filter
//          Start load of OC service gateway [lmLoadServiceGateway]
//      Else
//          Discard DSI section
//      Endif
//
//  Else
//      Discard DSI section
//  Endif
//
*/
static E_DscError processDSI( P_DsmCoreInst idp,
   /*I*/ U8BIT *pDSISection, P_SecFilterInfo pFilterInfo )
{
   E_DscError err = CLDSM_OK;
   P_RootCarousel pRC;
   U32BIT transactionId;
   U16BIT dsiPrivateLen;

   dsmAssert((idp != NULL));
   dsmAssert((pDSISection != NULL));

#if defined(DEBUG_PROCESS_DSI) && (DSM_DP_LEVEL >= 3)
   /* Dump filter information */
   dsmDP3(("processDSI(pFilterInfo = %#p boot %#p)\n", pFilterInfo, idp->pBootingCarousel));
   dsmDP3(("filter.pid = %x\n", pFilterInfo->filter.pid));
   dsmDP3(("filter.tableId = %x\n", pFilterInfo->filter.tableId));
   dsmDP3(("filter.tableIdExt = %x\n", pFilterInfo->filter.tableIdExt));

   /* Dump first 32 bytes of section */
   U8BIT *ptr = pDSISection;
   U16BIT i;
   for (i = 0; i < 32; i++)
   {
      dsmDP3(("%02x ", *ptr++));
      if (((i + 1) % 8 == 0) && i != 31)
         dsmDP3(("\n"));
   }
   dsmDP3(("\n"));
#endif

   /* -- Only process DSIs if currently booting a carousel */
   dsmAssert((pFilterInfo != NULL));
   pRC = pFilterInfo->target.u.pRC;
   if (FALSE == LLIsObjectInList(idp->llcRootCarousels, pRC))
   {
      ERRLOG(DD_OC, "pRC=%p NOT FOUND", pRC)
   }
   else if (pFilterInfo != pRC->pDsiSf || pRC->pDsiSf->status != SFA_COMMITTED)
   {
      ERRLOG(DD_OC, " mis-match filter %p %p", pFilterInfo, pRC->pDsiSf)
   }
   else if (pRC->rcid != pFilterInfo->target.id)
   {
      ERRLOG(DD_OC, "DATA ERROR: Mismatch Ids rcid=%d tid=%d",
         pRC->rcid, pFilterInfo->target.id )
   }
   else
   {
      DBG4(DD_OC,"pRC=%p", pRC)

      pDSISection = ValidateDSI( pDSISection, &dsiPrivateLen, &transactionId );
      if (pDSISection == NULL || pRC->dsiTransactionId == transactionId)
      {
         DBG4(DD_OC,"DSI discarded p=%p t=%x",pDSISection,transactionId)
      }
      else
      {
         DBGLOG(DD_OC,"dsiTransactionId change: %x, %x", pRC->dsiTransactionId, transactionId)
         pRC->dsiTransactionId = transactionId;
         switch (pRC->magic)
         {
            case OC_MAGIC:
            {
               err = DSC_ObjCrslParseSrgInfo( idp, (P_ObjectCarousel)pRC, pDSISection, dsiPrivateLen );
               break;
            }

            case UC_MAGIC:
            {
               err = DSC_UpdCrslParseGroupInfo( idp, (P_UpdateCarousel)pRC, pDSISection, dsiPrivateLen );
               break;
            }

            default:
               err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
         }
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: processDSI: %u\n", err)));
   return err;
}

static U8BIT* ValidateDII( U8BIT *pData, U16BIT *pMsgLength, U32BIT *pTransactionId )
{
   U16BIT ui16, msglen;
   U8BIT ui8, adaptationHdrLen;

   /* -- tableId = 0x3B */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x3B)
   {
      DBG1(DD_OC, "DATA ERROR: DII tableId = %x", ui8)
      return NULL;
   }
   /* -- dsmcc_section_length */
   /* DSMCC specifies:
   --  - Maximum DSMCC message length is from last_section_number field to
   --    start of CRC_32/checksum field (ie. dsmcc_section_length - 9)
   --      - Ref[1] - section 9.2.2.1, pg 287
   */
   READ_UINT16( pData, ui16 );
   if (((ui16 & LSB12_MSK) > MAX_SECTION_LEN) ||
       (((ui16 & LSB12_MSK) - 9) < MIN_DII_MSG_LEN))
   {
      DBG1(DD_OC, "DATA ERROR: DII section len = %x", ui16)
      return NULL;
   }

   /* -- tableIdExtension = 0x0002-0xFFFF */
   READ_UINT16( pData, ui16 );
   if (ui16 < 2)
   {
      DBG1(DD_OC, "DATA ERROR: DII tableIdExtension = %x", ui16)
      return NULL;
   }

   /* -- Skip versionNumber, sectionNumber, lastSectionNumber - NOT USED */
   SET_POS_REL( pData, 3 );

   /* -- dsmccMessageHeader */
   /* -- protocolDescriminator */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x11)
   {
      DBG1(DD_OC, "DATA ERROR: protocolDescriminator = %x", ui8)
      return NULL;
   }

   /* -- dsmccType = U-N Download Message */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x03)
   {
      DBG1(DD_OC, "DATA ERROR: dsmccType = %x", ui8)
      return NULL;
   }

   /* -- messageID = DownloadInfoIndication */
   READ_UINT16( pData, ui16 );
   if (ui16 != DII_MSG_ID)
   {
      DBG1(DD_OC, "DATA ERROR: messageID = %x", ui16)
      return NULL;
   }

   /* -- transactionId: originator bits = 10B (DSM-CC [1]) */
   READ_UINT32( pData, *pTransactionId );
   if ((*pTransactionId & TRANSACTION_ID_ORIG_MASK) != 0x80000000)
   {
      DBG1(DD_OC, "DATA ERROR: DSI transactionId (originator (bits 31-30) != 10B) = %x", *pTransactionId)
      return NULL;
   }

   /* -- Skip reserved data - NOT USED */
   SET_POS_REL( pData, 1 );

   /* -- Read adaptationLength */
   READ_UINT8( pData, adaptationHdrLen );

   /* -- Check messageLength */
   READ_UINT16( pData, msglen );
   if (msglen < MIN_DII_MSG_BODY_LEN)
   {
      DBG1(DD_OC, "DATA ERROR: DSI messageLength (< DSI_HDR_LEN) = %u", ui16)
      return NULL;
   }

   /* -- Skip adaptationHeader - NOT USED */
   SET_POS_REL( pData, adaptationHdrLen );

   *pMsgLength = msglen - adaptationHdrLen;

   /* Return pointer to data */
   return pData;
}

/*
// *** NB. Currently only ONE objectCarousel supported ***
//
//  Get carouselId from diiMessage
//  Get transactionId from diiMessage
//
//  Identify relevant OC and target dataCarousel for this DII section
//
//  If target dataCarousel identified
//      Update dataCarousel with diiMessage [lmUpdateDataCarousel]
//  Else
//      Discard DII section
//  Endif
//
*/
static E_DscError processDII( P_DsmCoreInst idp, U8BIT *pDIISection, P_SecFilterInfo pFilterInfo )
{
   E_DscError err = CLDSM_OK;
   U16BIT diiMsgDataLen;
   P_DataCarousel pDataCarousel = NULL;
   U32BIT transactionId;

   dsmDP3(("processDII() pFilterInfo = %p\n", pFilterInfo));
   dsmAssert((idp != NULL));
   dsmAssert((pDIISection != NULL));

   if (pFilterInfo == NULL)
   {
      err = CLDSM_ERR_ILLEGAL_PARAMETER;
      dsmDP1(("API ERROR: No section filter reference supplied\n"));
   }
   else
   {
      dsmAssert((pFilterInfo->target.u.pDC != NULL));
      pDataCarousel = pFilterInfo->target.u.pDC;

      pDIISection = ValidateDII( pDIISection, &diiMsgDataLen, &transactionId );
      if (pDIISection == NULL)
      {
         ERRPRINT("")
      }
      else if (pFilterInfo->target.id != (transactionId & TRANSACTION_ID_IDENT_MASK))
      {
         /* -- Target ID and section ID do not match */
         dsmDP3(("INFO: DII filter target ID != transactionId: %u, %u\n",
                 pFilterInfo->target.id, (transactionId & TRANSACTION_ID_IDENT_MASK)));
      }
      else if (pDataCarousel != NULL)
      {
      #ifndef NDEBUG
         /* -- Double-check DC matches filter */
         dsmAssert((pFilterInfo == pDataCarousel->pDiiSf));
         dsmAssert((pDataCarousel->pDiiSf->status == SFA_COMMITTED));
      #endif
         /* -- NB. DC may be deleted if emptied */
         err = DSC_DataCrslUpdate( idp, pDataCarousel, transactionId, pDIISection, diiMsgDataLen );
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: processDII: %u\n", err)));
   return err;
}


static U8BIT* ValidateDDB( U8BIT *pData, U16BIT *pMsgLength, U32BIT *pTransactionId )
{
   U16BIT ui16, msglen;
   U8BIT ui8, adaptationHdrLen;

   /* -- tableId = 0x3C */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x3C)
   {
      DBG1(DD_OC, "DATA ERROR: TableId = %x", ui8)
      return NULL;
   }
   /* -- dsmcc_section_length */
   /* DSMCC specifies:
   --  - Maximum DSMCC message length is from last_section_number field to
   --    start of CRC_32/checksum field (ie. dsmcc_section_length - 9)
   --      - Ref[1] - section 9.2.2.1, pg 287
   */
   READ_UINT16( pData, ui16 );
   if (((ui16 & LSB12_MSK) > MAX_SECTION_LEN) ||
       (((ui16 & LSB12_MSK) - 9) < MIN_DDB_MSG_LEN))
   {
      DBG1(DD_OC, "DATA ERROR: section len = %x", ui16)
      return NULL;
   }


   /* -- Skip tableIdExtension, versionNumber, sectionNumber, lastSectionNumber - NOT USED */
   SET_POS_REL( pData, 5 );

   /* -- dsmccMessageHeader */
   /* -- protocolDescriminator */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x11)
   {
      DBG1(DD_OC, "DATA ERROR: protocolDescriminator = %x", ui8)
      return NULL;
   }

   /* -- dsmccType = U-N Download Message */
   READ_UINT8( pData, ui8 );
   if (ui8 != 0x03)
   {
      DBG1(DD_OC, "DATA ERROR: dsmccType = %x", ui8)
      return NULL;
   }

   /* -- messageID = DownloadDataBlock */
   READ_UINT16( pData, ui16 );
   if (ui16 != DDB_MSG_ID)
   {
      DBG1(DD_OC, "DATA ERROR: messageID = %x", ui16)
      return NULL;
   }

   /* -- transactionId: originator bits = 10B (DSM-CC [1]) */
   READ_UINT32( pData, *pTransactionId );
  /* -- NB. According to TDNMHEG group - downloadId/carouselId equivalence
      --     is not/cannot be guaranteed by broadcasters for DDB*/

   /* -- Skip reserved data - NOT USED */
   SET_POS_REL( pData, 1 );

   /* -- Read adaptationLength */
   READ_UINT8( pData, adaptationHdrLen );

   /* -- Check messageLength */
   READ_UINT16( pData, msglen );
   if (msglen < MIN_DDB_MSG_BODY_LEN)
   {
      DBG1(DD_OC, "DATA ERROR: DSI messageLength (< DSI_HDR_LEN) = %u", ui16)
      return NULL;
   }

   /* -- Skip adaptationHeader - NOT USED */
   SET_POS_REL( pData, adaptationHdrLen );

   *pMsgLength = msglen - adaptationHdrLen;

   /* Return pointer to data */
   return pData;
}

/*
// **** NB. Currently only ONE objectCarousel supported ****
//
//  Get carouselId from ddbMessage
//  Get moduleId from ddbMessage
//
//  Identify relevant OC and target module for this DDB section
//
//  If target module identified
//      Build ddbMessage into moduleData [DSC_ModuleUpdate]
//      If moduleData completed
//          Update loadRequest(s)/monitor(s) for module [lmUpdateModule]
//      Endif
//  Else
//      Discard DDB section
//  Endif
//
*/
static E_DscError processDDB( P_DsmCoreInst idp, U8BIT *pDDBSection, P_SecFilterInfo pFilterInfo )
{
   E_DscError err = CLDSM_OK;
   U32BIT transactionId;
   U16BIT ddbMsgDataLen;
   U16BIT moduleId;

   dsmAssert((idp != NULL));
   dsmAssert((pDDBSection != NULL));

   if (pFilterInfo == NULL)
   {
      err = CLDSM_ERR_ILLEGAL_PARAMETER;
      dsmDP1(("API ERROR: No section filter reference supplied\n"));
   }
   else
   {
      dsmAssert((pFilterInfo->target.u.pModule != NULL));

      pDDBSection = ValidateDDB( pDDBSection, &ddbMsgDataLen, &transactionId );
      if (pDDBSection != NULL)
      {
         /* -- pDDBSection -> DDBMessageBody/Payload */
         /* -- Get moduleId (NB. DO NOT advance 'cursor') */
         GET_UINT16( pDDBSection, moduleId );
         if (pFilterInfo->target.id != moduleId)
         {
            /* -- Target ID and section ID do not match */
            dsmDP3(("INFO: DDB filter target ID != moduleId: %u, %u\n", pFilterInfo->target.id, moduleId));
         }
         else
         {
            P_Module pModule;
            pModule = pFilterInfo->target.u.pModule;
            if (pModule != NULL && pFilterInfo == pModule->pDdbSf)
            {
               /* -- Double-check Module matches filter */
               dsmAssert((pFilterInfo->status == SFA_COMMITTED));
               err = DSC_ModuleUpdate( idp, pModule, pDDBSection, ddbMsgDataLen );
            }
            else
            {
               /* -- This was not a DDB we were looking for so discard */
               dsmDP3(("INFO: DDB discarded - moduleId: %u\n", moduleId));

         #ifdef ACCELERATE_CAROUSEL_LOADING
               err = CLDSM_ERR_MODULE_ACQUISITION_FAILURE;
         #endif
            }
         }
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: processDDB: %u\n", err)));
   return err;
}

#ifdef ACCELERATE_CAROUSEL_LOADING

static E_DscError storeSection( P_DsmCoreInst idp,
   U8BIT *pDDBSection)
{
   U8BIT *pDDBSectionIn = pDDBSection;
   E_DscError err = CLDSM_OK;
   U8BIT adaptationHdrLen;
   U16BIT ddbMsgLen;
   U16BIT ddbMsgDataLen;
   U16BIT moduleId, blockNum;
   U8BIT ui8 = 0;
   U16BIT ui16 = 0;
   U8BIT moduleVersion;
   int i;
   BOOLEAN bStoreSection = TRUE;
   U16BIT msgLength = 0;
   U8BIT iMiddlePos = 0;


   ui8 += 0; ui16 += 0;   /* -- stop compiler warnings when no checking */


   dsmDP3(("processDDB()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pDDBSection != NULL));

   /* TODO: report soft errors in L0, L1 (and L2?) checks? */

   /* -- pDDBSection -> section header (tableId) */

   /* -- NB. Only have L2 checks on section header values since
      -- section filter should have already checked these! */

   /* -- tableId = 0x3C */
   ADV_UINT8_L2CHK( pDDBSection, ui8, ui8 == 0x3C,
      dsmDP2(("DATA ERROR: DDB tableId = %x\n", ui8)),
      goto _return                                                  /* EXIT */
      );

   /* -- dsmcc_section_length */
   /*
   -- DSMCC specifies:
   --  - Maximum DSMCC message length is from last_section_number field to
   --    start of CRC_32/checksum field (ie. dsmcc_section_length - 9)
   --      - Ref[1] - section 9.2.2.1, pg 287
   */
   /*   ADV_UINT16_L2CHK( pDDBSection, ui16, */
   /*          ( ((ui16 & LSB12_MSK) <= MAX_SECTION_LEN) && */
   /*           (((ui16 & LSB12_MSK) - 9) >= MIN_DDB_MSG_LEN) ), */
   /*         dsmDP2(("DATA ERROR: DDB section len = %u\n", ui16 )), */
   /*         goto _return                                                 */
   /*     ); */

   READ_UINT16( pDDBSection, msgLength );
   msgLength = msgLength & LSB12_MSK;

   /* -- Skip tableIdExtension, versionNumber, sectionNumber,
      -- lastSectionNumber - NOT USED HERE */
   SET_POS_REL( pDDBSection, 5 );


   /* -- pDDBSection -> dsmccDownloadDataHeader */

   /* -- protocolDescriminator */
   /* -- L1 check because this should be correct */
   ADV_UINT8_L1CHK( pDDBSection, ui8, ui8 == 0x11,
      dsmDP2(("DATA ERROR: DDB protocolDescriminator = %x\n", ui8)),
      goto _return );

   /* -- dsmccType = U-N Download Message */
   /* -- L1 check because this should be correct */
   ADV_UINT8_L1CHK( pDDBSection, ui8, ui8 == 0x03,
      dsmDP2(("DATA ERROR: DDB dsmccType = %x\n", ui8)),
      goto _return );

   /* -- messageID = DownloadDataBlock */
   /* -- L1 check because this should be correct */
   ADV_UINT16_L1CHK( pDDBSection, ui16, ui16 == 0x1003,
      dsmDP2(("DATA ERROR: DDB messageID = %x\n", ui16)),
      goto _return );

   /* -- Skip downloadId - NOT USED */
   /* -- NB. According to TDNMHEG group - downloadId/carouselId equivalence
      --     is not/cannot be guaranteed by broadcasters */
   SET_POS_REL( pDDBSection, 4 );

   /* -- Skip reserved data - NOT USED */
   SET_POS_REL( pDDBSection, 1 );

   /* -- Read adaptationLength */
   READ_UINT8( pDDBSection, adaptationHdrLen );

   /* -- Read messageLength */
   READ_UINT16( pDDBSection, ddbMsgLen );

   /* -- Calculate DDB message payload/data length */
   ddbMsgDataLen = (U16BIT)(ddbMsgLen - adaptationHdrLen);

   /* -- Check message data size */
   L2_DATA_CHK( ddbMsgDataLen >= MIN_DDB_MSG_BODY_LEN,
      dsmDP2(("DATA ERROR: DDB messageLength (< MIN_DDB_MSG_BODY_LEN) = %x\n",
              ddbMsgDataLen)),
      goto _return );

   /* -- Skip adaptationHeader - NOT USED */
   SET_POS_REL( pDDBSection, adaptationHdrLen );


   /* -- pDDBSection -> DDBMessageBody/Payload */

   /* -- Get moduleId (NB. DO NOT advance 'cursor') */
   GET_UINT16( pDDBSection, moduleId );

   /* -- Skip moduleId */
   SET_POS_REL( pDDBSection, 2 );

   /* -- Read moduleVersion */
   READ_UINT8(pDDBSection, moduleVersion );

   /* -- Skip reserved data */
   SET_POS_REL(pDDBSection, 1 );

   /* -- Read blockNumber */
   READ_UINT16( pDDBSection, blockNum );


   iMiddlePos = accCarouselLoadingInfos.storedSectionsNB / 2;

   if (iMiddlePos > 0)
   {
      for (i = 0; i < iMiddlePos; i++)
      {
         if ((prestoredSectionsTable[i].moduleID == moduleId) &&
             (prestoredSectionsTable[i].blockNum == blockNum))
         {
            bStoreSection = FALSE;
            break;
         }

         if ((prestoredSectionsTable[iMiddlePos + i].moduleID == moduleId) &&
             (prestoredSectionsTable[iMiddlePos + i].blockNum == blockNum))
         {
            bStoreSection = FALSE;
            break;
         }
      }

      if ((prestoredSectionsTable[accCarouselLoadingInfos.storedSectionsNB - 1].moduleID == moduleId) &&
          (prestoredSectionsTable[accCarouselLoadingInfos.storedSectionsNB - 1].blockNum == blockNum))
      {
         bStoreSection = FALSE;
      }
   }
   else
   {
      for (i = 0; i < accCarouselLoadingInfos.storedSectionsNB; i++)
      {
         if ((prestoredSectionsTable[i].moduleID == moduleId) &&
             (prestoredSectionsTable[i].blockNum == blockNum))
         {
            bStoreSection = FALSE;
            break;
         }
      }
   }

   if (bStoreSection == TRUE)
   {
      if (accCarouselLoadingInfos.storeWritePos >= kMAX_PRESTORED_SECTIONS)
      {
         accCarouselLoadingInfos.storeWritePos = 0;
      }

      prestoredSectionsTable[accCarouselLoadingInfos.storeWritePos].moduleID = moduleId;
      prestoredSectionsTable[accCarouselLoadingInfos.storeWritePos].blockNum = blockNum;


      prestoredSectionsTable[accCarouselLoadingInfos.storeWritePos].sectionSize = msgLength;
      memcpy(&prestoredSectionsTable[accCarouselLoadingInfos.storeWritePos].section[0], pDDBSectionIn, msgLength);

      accCarouselLoadingInfos.storeWritePos++;

      if (accCarouselLoadingInfos.storedSectionsNB < kMAX_PRESTORED_SECTIONS)
      {
         accCarouselLoadingInfos.storedSectionsNB++;
      }
   }

   goto _return;    /* -- stop compiler warnings when no checking */
_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: storeSection: %u\n", err)));
   dsmDP3(("exit storeSection -> rtn: %u\n", err));
   return err;
} /* storeSection */

static E_DscError injectStoredSections( P_DsmCoreInst idp)
{
   U8BIT *pDDBSection;
   E_DscError err = CLDSM_OK;
   H_DscSFRef clDsmFilterRef;
   S_SecFilterInfo filterInfo;
   P_ObjectCarousel pOC;
   pModule pModule = NULL;
   U16BIT moduleId;
   U16BIT moduleCounter = 0;

   memset(&filterInfo, 0, sizeof(S_SecFilterInfo));

   accCarouselLoadingInfos.storeReadPos = 0;
   accCarouselLoadingInfos.SectionsReinjection++;

   while (moduleCounter < accCarouselLoadingInfos.storedSectionsNB)
   {
      moduleCounter++;

      if (accCarouselLoadingInfos.storeReadPos >= kMAX_PRESTORED_SECTIONS)
      {
         accCarouselLoadingInfos.storeReadPos = 0;
      }

      moduleId = prestoredSectionsTable[accCarouselLoadingInfos.storeReadPos].moduleID;

      if ((prestoredSectionsTable[accCarouselLoadingInfos.storeReadPos].injectionDone != 1) &&
          (requestedModuleIDList[moduleId] == kDO_INJECT))
      {
         pDDBSection = &prestoredSectionsTable[accCarouselLoadingInfos.storeReadPos].section[0];

         pOC = LLHead( idp->llcRootCarousels );

         if (pOC != NULL)
         {
            pModule = DSC_ModuleListFindById( pOC->llcOcModuleAcquires, moduleId );
            if (pModule != NULL)
            {
               filterInfo.hCarousel = (MemHandle)pOC;
               filterInfo.target.u.pModule = pModule;
               filterInfo.target.id = prestoredSectionsTable[accCarouselLoadingInfos.storeReadPos].moduleID;
               filterInfo.target.kind = SFK_DDB;
               err = internalSysProcessPrivateSection( idp, pDDBSection, (void *)&filterInfo);
               prestoredSectionsTable[accCarouselLoadingInfos.storeReadPos].injectionDone = 1;
            }
         }
      }
      accCarouselLoadingInfos.storeReadPos++;
   }
}

E_DscError requestedModuleIdSet(U16BIT moduleID)
{
   if (moduleID < kMAX_MODULE_ID)
   {
      if (requestedModuleIDList[moduleID] != kDO_INJECT)
      {
         requestedModuleIDList[moduleID] = kDO_INJECT;
         GBL_NewResquetedModuleID++;
      }
   }
}

E_DscError requestedModuleIdReset(U16BIT moduleID)
{
   if (moduleID < kMAX_MODULE_ID)
   {
      requestedModuleIDList[moduleID] = kNOT_REQUESTED;
   }
}

#endif /* ACCELERATE_CAROUSEL_LOADING */


