/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DSM-CC dynamic memory manager API
 * @file    clDsmMemMgrAPI.h
 * @date    30/08/2004
 * @author  Ocean Blue
 */
#ifndef _CLDSMMEMMGRAPI_H_
#define _CLDSMMEMMGRAPI_H_


/*-------------------------------- Includes ----------------------------------*/

#include "techtype.h"

/*-------------------------------- Defines -----------------------------------*/

/*
-- ERROR CODES
--
-- To reduce unneccessary overheads in the final code, most of the memory
-- manager API functions do not return error codes. It is expected that the
-- caller (if necessary) will check parameters for legality before making the
-- the memory manager API call. It is sensible to do error checking
-- (via asserts) inside the memory manager functions during DEBUG.
--
-- Error codes are only returned by functions that can generate valid run-time
-- errors (ie. not due to coding errors). These errors codes are returned via
-- the S32BIT return value of the API funcs.
*/

typedef enum
{
   MEM_NO_ERR,
   MEM_ERR_CACHE_FULL,
   MEM_ERR_ALLOC_FAILED,
   MEM_ERR_SEQ_OPEN_LIMIT,
   MEM_ERR_SEQ_END_OF_DATA,
} E_DsmMemErr;


/*
-- DEFAULT SIZE LIMITS
*/

#define MEM_BLOCK_SIZE_DFLT (256)
#define MEM_HEAP_SIZE_DFLT  (512 * 1024)
#define MEM_SEQ_OPEN_DFLT   (256)   /* -- concurrently used MemSeqRef values */


#if defined(VGR_TRK_MEM) && !defined(USE_DSM_MEMCHK) && !defined(DBG_STORE_LINE)
#define DBG_STORE_LINE
#endif


/*-------------------------------- Types -------------------------------------*/

/*
-- Generic MemHandle definition
--
-- If clDsm is supplied as a pre-built library for a particular platform it
-- will store a MemHandle as this type (ie. a standard pointer). This could
-- be cast to a different type in the memory manager if required.
--
-- If clDsm is being compiled from source then MemHandle can be (re)defined to
-- anything appropriate.
--
-- NOTE:
--  1)  Zero (Null) MUST NOT be a legal value for a MemHandle.
--  2)  It is rather convenient to use void*. In general, a compiler (eg 'gcc')
--      detects differing levels of indirection. So, does not matter is some don't.
*/
typedef void* MemHandle;


/*
-- Generic MemSeqRef definition
--
-- If clDsm is supplied as a pre-built library for a particular platform it
-- will store a MemSeqRef as this type (ie. a pointer to a structure). This can
-- be cast to the appropriate (structure) type in the memory manager.
--
-- If clDsm is being compiled from source then MemSeqRef can be (re)defined to
-- anything appropriate.
--
*/
/*typedef struct _MemSeqRef_struct *MemSeqRef; */

typedef U8BIT *MemSeqRef;

/*----------------------------- Exported Data --------------------------------*/

/*------------------------------ Prototypes ----------------------------------*/

#ifdef DSM_NO_MEM_CONTEXT
 #define memStop( x )                    memStop()
 #ifdef DBG_STORE_LINE
  #define memAlloc( x, i, a, l )         memAlloc( i, a, l )
 #else
  #define memAlloc( x, i, a, l )         memAlloc( i, a )
 #endif
 #define memRelease( x, i, a )           memRelease( i, a )
 #define memSeqOpen( x, a, b, c, d, e )  memSeqOpen( a, b, c, d, e )
 #define memNumOpen( x )                 memNumOpen()
 #define memSeqNumOpen( x )              memSeqNumOpen()
#else
 #ifndef DBG_STORE_LINE
  #define memAlloc( x, i, a, l )         memAlloc( x, i, a )
 #endif
#endif

/*
-- Start memory manager [SYNCHRONOUS]
--
-- Specifies minimum contiguous memory block size in bytes (mandatory).
-- Specifies minimum size of heap memory in bytes (mandatory).
-- Specifies minimum number of MemSeq areas that can be opened
--      simultaneously (mandatory).
-- Requests total size of heap memory in bytes (optional - set to memHeapSizeMin
--      value if not used).
-- Supplies reference for memSetup parameters (optional).
--
-- Returns pointer to this memory manager context (if implemented, otherewise
-- should be returned as Null), this will be supplied to any subsequent mem
-- function calls that require it.
--
-- If any problems encountered (eg. in block/heap sizes) abort start and return
-- error code: MEM_ERR_START_PROBLEM
--
*/
E_DsmMemErr memStart(
   /*I*/ U16BIT memBlockSizeMin, U32BIT memHeapSizeMin, U16BIT memSeqOpenMin,
   void *memSetup,
   /*O*/ void **memContext );


/*
-- Stop (forcibly) the memory manager [SYNCHRONOUS]
--
-- If any problems noticed (eg. memory objects still open) still complete
-- stop (eg. free memory) but return error code: MEM_ERR_STOP_PROBLEM
--
*/
E_DsmMemErr memStop( void *context );


/*
-- Allocate an area of managed (heap) memory [SYNCHRONOUS]
*/
MemHandle memAlloc( void *context, P_DsmSetup setup, U32BIT size, int line );

/*
-- Release a previously allocated area of managed (heap) memory [SYNCHRONOUS]
*/
U32BIT memRelease( void *context, P_DsmSetup setup, MemHandle memArea );


/*
-- Check if a MemHandle is valid [SYNCHRONOUS]
--
-- NB. This should be 100% reliable at determining if a MemHandle reperesents
-- a currently allocated (ie. not free) memory area in the memory heap.
--
*/
BOOLEAN memValidate( void *memArea );


/*
-- ************************************************************************
-- NB. FOR DEBUG ONLY [SYNCHRONOUS]:
-- Return a count of number of currently open (via memOpen) memory objects.
-- Can be used to help detect objects left open erroneously.
-- ************************************************************************
*/
#ifndef NDEBUG
U32BIT memNumOpen( void *context );
#endif


/*
-- SEQUENTIAL ('FILE TYPE') MEMORY ACCESS FUNCTIONS:
--
-- NOTES:
--
-- Only memSeqOpen requires a context to be passed (if supported). Memory
-- managers that do support a context must store the context value via the
-- MemSeqRef pointer which is passed to all other memSeq*** functions (nb. this
-- is because asynchronous clients using these calls will not have access to
-- the context value)
--
-- It is legal for a memory sequence cursor to incremented or set to the
-- position _immediately_ after the last byte of the sequential memory
-- area (ie. an error should not be generated in this case). However, it
-- is illegal (should generate an error) if an attempt is made to read or
-- write at this cursor position.
--
*/


/* -- SYNCHRONOUS ONLY FUNCTIONS */

/*
-- Open all or part of a memory area for sequential ('file type') access
-- [SYNCHRONOUS]
--
-- Offset and length specify which part of the memory area is being
-- accessed sequentially. They must always be set appropriately even if access
-- to the whole area is required (note - this should make end of data (EOF)
-- detection easier to implement in the memory manager).
--
-- Different parts of a single memory area can be concurrently opened if
-- required (note - if they overlap, then behaviour if writing asynchronously
-- will be unpredictable !).
--
-- The asyncAccess flag indicates whether subsequent read/write access to this
-- area (via the memSeq* functions) will be asynchronous or synchronous. This
-- holds true until the subsequent memSeqClose for this memory area (or part).
--
-- If the number of memory areas (or parts) opened concurrently for sequential
-- access reaches the memory manager limit (minimum value memSeqOpenMin passed
-- in at memStart), return error code: MEM_ERR_SEQ_OPEN_LIMIT
--
*/
E_DsmMemErr memSeqOpen( void *context,
   MemHandle memArea, U32BIT offset, U32BIT length, BOOLEAN asyncAccess,
   MemSeqRef *memAreaRef );


/*
-- Clone/copy an open sequential memory area reference (including current cursor
-- position) [SYNCHRONOUS]
--
-- Allocates, opens and returns another sequential memory area (MemSeqRef value)
--
-- NB. Context can be determined from memAreaRefOrig (if needed).
--
-- If the number of memory areas (or parts) opened concurrently for sequential
-- access reaches the memory manager limit (minimum value memSeqOpenMin passed
-- in at memStart), return error code: MEM_ERR_SEQ_OPEN_LIMIT
--
*/
E_DsmMemErr memSeqOpenClone(
   MemSeqRef memAreaRefOrig, MemSeqRef *memAreaRefClone );


/*
-- Close a memory area for sequential ('file type') access [SYNCHRONOUS]
--
-- NB. Context can be determined from memAreaRef (if needed).
--
*/
void memSeqClose( MemSeqRef memAreaRef );


/*
-- Get a _real_ memory pointer for the current 'cursor' position and indicate
-- the number of contiguous bytes of memory directly accessible from this
-- pointer (within this memSeq area). Advance 'cursor' by numBytesContig
-- [SYNCHRONOUS]
--
-- Once called, the cursor should either end up pointing to the start of the
-- next block (in block based memory managers) or the end of the memSeq area
-- (in block based or contiguous memory managers). In either case the
-- function can then be called again to access the next block or determine
-- if the end of the memSeq area has been reached. If the 'cursor' is at the
-- end of the memSeq area when this function is called then memPtr is returned
-- as Null and numContigBytes set to 0.
--
-- NB. This must be used VERY CAREFULLY since directly accessing beyond the
-- supplied numContigBytes is likely to cause a fatal error (especially with
-- block based memory managers).
--
*/
void memSeqAccessContig(
   MemSeqRef memAreaRef, U8BIT **memPtr, U32BIT *numContigBytes );

void memSeqAccessCurrent(
   MemSeqRef memAreaRef, U8BIT **memPtr, U32BIT *numContigBytes );

/*
-- ************************************************************************
-- NB. FOR DEBUG ONLY [SYNCHRONOUS]:
-- Return a count of number of currently open (via memSeqOpen or
-- meSeqOpenClone) memory sequences
-- Can be used to help detect MemSeqRef areas left open erroneously.
-- ************************************************************************
*/
#ifndef NDEBUG
U32BIT memSeqNumOpen( void *context );
#endif



/* -- SYNCHRONOUS/ASYNCHRONOUS FUNCTIONS */

/*
-- Check if a MemSeqRef is valid [SYNCHRONOUS/ASYNCHRONOUS]
*/
void memSeqValidate( MemSeqRef memAreaRef, BOOLEAN *valid );


/*
-- Get size of a sequential memory area [SYNCHRONOUS/ASYNCHRONOUS]
--
-- NB. The size returned should be the exact number of bytes supplied as the
--     length parameter at memSeqOpen.
--
*/
void memSeqSize( MemSeqRef memAreaRef, U32BIT *size );


/*
-- Read single byte from current 'cursor' position in managed memory,
-- increment cursor [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to read past end of the sequential memory area,
-- return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqReadByte( MemSeqRef src, U8BIT *byte );


/*
-- Put single byte to current 'cursor' position in managed memory,
-- increment cursor [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to write past end of the sequential memory area,
-- return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqWriteByte( U8BIT byte, MemSeqRef dest );


/*
-- Read numBytes into contiguous memory (starting at current 'cursor' position
-- in managed memory), increment 'cursor' position by numBytes
-- [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to read past end of the sequential memory area,
-- read as many bytes as possible, store the number actually read in
-- numBytesActual and return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqRead(
   MemSeqRef src, U8BIT *dest, U32BIT numBytes, U32BIT *numBytesActual );


/*
-- Write numBytes from contiguous memory into managed memory (starting at
-- current 'cursor' position in managed memory), increment 'cursor' position by
-- numBytes [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to write past end of the sequential memory area,
-- write as many bytes as possible, store the number actually written in
-- numBytesActual and return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqWrite(
   U8BIT *src, MemSeqRef dest, U32BIT numBytes, U32BIT *numBytesActual );


/*
-- Copy numBytes from one managed memory area into another managed memory area
-- (starting at current 'cursor' positions in each), increment 'cursor'
-- positions in each by numBytes [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to write/read past the end of either of the sequential
-- memory areas, copy as many bytes as possible, store the number actually
-- copied in numBytesActual and return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqCopy(
   MemSeqRef src, MemSeqRef dest, U32BIT numBytes, U32BIT *numBytesActual );


/*
-- Compare sequence of bytes (numBytes long) in contiguous memory with bytes
-- in managed memory (starting at current 'cursor' position), do not change
-- 'cursor' position [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to compare bytes past the end of the sequential
-- memory area, return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqCompContig(
   U8BIT *contig, MemSeqRef memAreaRef, U32BIT numBytes, BOOLEAN *equal );


/*
-- Compare sequence of bytes (numBytes long) in one managed memory area with
-- bytes in another managed memory area (starting at current 'cursor' positions
-- in each), do not change 'cursor' positions for either memory area
-- [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to compare bytes past the end of either of the sequential
-- memory areas, return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqCompMgd(
   MemSeqRef memAreaRef1, MemSeqRef memAreaRef2, U32BIT numBytes,
   BOOLEAN *equal );


/*
-- Set sequential access 'cursor' position relative to current position
-- [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to set cursor past the end of the sequential memory area,
-- return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqSetPosRel( MemSeqRef memAreaRef, S32BIT position );


/*
-- Set absolute sequential access 'cursor' position [SYNCHRONOUS/ASYNCHRONOUS]
--
-- If call attempts to set cursor past the end of the sequential memory area,
-- return error code: MEM_ERR_SEQ_END_OF_DATA
--
*/
E_DsmMemErr memSeqSetPosAbs( MemSeqRef memAreaRef, U32BIT position );


/*
-- Read sequential access 'cursor' position [SYNCHRONOUS/ASYNCHRONOUS]
--
-- Value returned in 'position' is the offset from the start of the memory area
-- to the next byte to be read. If the current position has passed the end of
-- the memory area, then the position returned is equal to the memory area size.
-- The value returned therefore ranges from 0 to (memory area size).
*/
void memSeqReadPos( MemSeqRef memAreaRef, U32BIT *position );


/*----------------------------------------------------------------------------*/
#endif /* _CLDSMMEMMGRAPI_H_ */

