/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2000 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DSM-CC basic memory manager
 * @file    clDsmMemMgrBasic.c
 * @date    2000
 * @author  Bassett

 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

#include "cldsmcc.h"
#include "clDsmMemMgrAPI.h"
#include "dsmDbg.h"


/*------------------------------ Local macros --------------------------------*/

#define MEM_NUM_SEQS        (MEM_SEQ_OPEN_DFLT)
/* Check seq refs in debug only */
#define memCheckRef(x)      dsmAssert(((x->magic == MEM_MAGIC) && (x->pStartPos != NULL)))
#ifdef DBG_STORE_LINE
#define MEM_BLOCK_HDR_SIZE  12
#else
#define MEM_BLOCK_HDR_SIZE  8
#endif
#define MEM_BLOCK_START     0x4d656d3e /* = string "Mem>" */
#define MEM_BLOCK_END       0x3c4d656d /* = string "<Mem" */
#define MEM_MAGIC           0x5365713e /* = string "Seq>" */


/*----------------------------- Exported data --------------------------------*/


/*------------------------------ Local types----------------------------------*/

typedef struct
{
   U32BIT magic;            /* Magic number for debug */
   MemHandle memArea;       /* MemHandle of memArea */
   U8BIT *pStartPos;        /* Pointer to start of sequence */
   U8BIT *pCurrentPos;      /* Pointer to current position in sequence */
   U32BIT seqLength;        /* Length of sequence */
   U32BIT currentOffset;    /* Offset (in bytes) from start of sequence to
                               current position */
   U32BIT savedOffset;      /* Offset (in bytes) from start of sequence to
                               saved position */
} localSeqRef, *pLocalSeqRef;

typedef struct
{
   U32BIT numOpen;
   U32BIT numAllocd;
   S32BIT numSeqsOpen;
   localSeqRef memSeqs[MEM_NUM_SEQS];
} MemContext_t, *pMemContext_t;


/*-------------------------------local statics--------------------------------*/

static MemContext_t localMemContext;


/*---------------------local prototypes/forward declarations------------------*/


/*-----------------------------exported functions-----------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// memStart
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memStart(
   /*I*/ U16BIT memBlockSizeMin, U32BIT memHeapSizeMin, U16BIT memSeqOpenMin,
   void *memSetup,
   /*O*/ void **memContext )
{
   S32BIT loop;
   pLocalSeqRef pMemSeq = NULL;

   dsmDP4(("memStart()\n"));

   dsmAssert((memContext != NULL));

   *memContext = NULL;
   localMemContext.numOpen = 0;
   localMemContext.numAllocd = 0;
   localMemContext.numSeqsOpen = 0;

   for (loop = MEM_NUM_SEQS - 1; loop; loop--)
   {
      pMemSeq = &localMemContext.memSeqs[loop];

      pMemSeq->magic = MEM_MAGIC;
      pMemSeq->memArea = NULL;
      pMemSeq->pStartPos = NULL;
      pMemSeq->pCurrentPos = NULL;
      pMemSeq->seqLength = 0;
      pMemSeq->currentOffset = 0;
      pMemSeq->savedOffset = 0;
   }

   return MEM_NO_ERR;
}

/* /////////////////////////////////////////////////////////////////////////////
// memStop
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memStop( void *context )
{
   dsmDP4(("memStop()\n"));

   dsmAssert((localMemContext.numOpen == 0));
   dsmAssert((localMemContext.numAllocd == 0));
   dsmAssert((localMemContext.numSeqsOpen == 0));

   return MEM_NO_ERR;
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_MmAlloc
//
///////////////////////////////////////////////////////////////////////////// */
MemHandle memAlloc( void *context, P_DsmSetup setup, U32BIT size, int line )
{
   MemHandle memArea;
   U32BIT sz = (size + 3) >> 2;
   U32BIT *pMemBlock;

   /* Allocate 2 x 4 byte blocks before data so that magic number and block
      size may be stored */
   pMemBlock = setup->allocFunc((sz << 2) + MEM_BLOCK_HDR_SIZE + 4 );
   if (pMemBlock)
   {
   #ifndef NDEBUG
      BOOLEAN valid;
   #endif

      *pMemBlock++ = MEM_BLOCK_START;
      *pMemBlock++ = size;
#ifdef DBG_STORE_LINE
      *pMemBlock++ = line;
#endif
      memArea = (MemHandle)pMemBlock;
      *(pMemBlock + sz) = MEM_BLOCK_END;

      DBG4(DD_GEN, "memArea=%p size=%d cnt=%d", memArea, size, ++localMemContext.numAllocd)

   #ifndef NDEBUG
      valid = memValidate( memArea );
      dsmAssert((valid));
   #endif
   }
   else
   {
      memArea = NULL;
   }
   return memArea;
}

/* /////////////////////////////////////////////////////////////////////////////
// memRelease
//
///////////////////////////////////////////////////////////////////////////// */
U32BIT memRelease( void *context, /*I*/ P_DsmSetup setup, MemHandle memArea )
{
   U32BIT size, *pMem;

   dsmAssert((memArea != NULL));
#ifndef NDEBUG
   {
      BOOLEAN valid;
      S32BIT i;

      /* Check memArea is valid */
      valid = memValidate( memArea );
      dsmAssert((valid));

      /* Check for mem sequence that is still open on the MemArea */
      for (i = 0; i < MEM_NUM_SEQS; i++)
      {
         dsmAssert((localMemContext.memSeqs[i].memArea != memArea));
      }
   }
#endif
   pMem = (U32BIT *)memArea;
#ifdef DBG_STORE_LINE
   --pMem;
#endif
   --pMem;
   size = *pMem;
   --pMem;
   *pMem = 0; /* remove magic block start */
   DBG4(DD_GEN, "memArea=%p size=%d cnt=%d", memArea, size, --localMemContext.numAllocd)

   setup->freeFunc((void *)pMem );

   return size;
}

/* /////////////////////////////////////////////////////////////////////////////
// memValidate
//
///////////////////////////////////////////////////////////////////////////// */
BOOLEAN memValidate( void *memArea )
{
   if (memArea != NULL)
   {
      U32BIT *pMemStart = (U32BIT *)memArea - (MEM_BLOCK_HDR_SIZE / 4);
      U32BIT size = *(pMemStart + 1);
      U32BIT *pMemEnd = (U32BIT *)memArea + ((size + 3) >> 2);

      /* Check for MEM_BLOCK_START before memArea */
      if ((*pMemStart == MEM_BLOCK_START) && (*pMemEnd == MEM_BLOCK_END))
      {
         DBG4(DD_GEN, "memArea=%p size=%d", memArea, size)
         return TRUE;
      }

      ERRPRINT("memArea=%p size=%d", memArea, size)
   }
   else
   {
      DBG3(DD_GEN, "memArea is NULL")
   }
   return FALSE;
}

/* /////////////////////////////////////////////////////////////////////////////
// memOpen
//
///////////////////////////////////////////////////////////////////////////// */
void memOpen( void *context, /*I*/ MemHandle memArea, /*O*/ void **memPtr )
{
   dsmDP4(("\tmemOpen(): %d, %x\n", ++localMemContext.numOpen, memArea));

#ifndef NDEBUG
   if (memArea != NULL)
   {
      /* Check memArea is valid */
      BOOLEAN valid = memValidate( memArea );
      if (!valid)
      {
         dsmDP1(("memArea=%p", memArea));
      }
   }
#endif

   *memPtr = (void *) memArea;
}

#ifndef NDEBUG
/* /////////////////////////////////////////////////////////////////////////////
// memNumOpen
//
///////////////////////////////////////////////////////////////////////////// */
U32BIT memNumOpen( void *context )
{
   dsmDP4(("memNumOpen()\n"));

   return localMemContext.numOpen;
}

#endif  /* NDEBUG */


/* /////////////////////////////////////////////////////////////////////////////
// memSeqOpen
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqOpen( void *context,
   /*I*/ MemHandle memArea, U32BIT offset, U32BIT length, BOOLEAN asyncAccess,
   /*O*/ MemSeqRef *memAreaRef )
{
   pLocalSeqRef pMemSeq = NULL;
   S32BIT loop;

   dsmDP4(("memSeqOpen()\n"));

   dsmAssert((memArea != NULL));
   dsmAssert((memAreaRef != NULL));
   if (!length)
   {
      DBG2(DD_GEN,"zero length")
   }

   /* Have we reached the limit for open seqs? */
   if (localMemContext.numSeqsOpen == MEM_NUM_SEQS)
   {
      dsmDP4(("INF: Memory sequence open limit reached!\n"));
      return MEM_ERR_SEQ_OPEN_LIMIT;
   }

   /* Put this ref's details in a blank element */
   for (loop = MEM_NUM_SEQS - 1; loop; loop--)
   {
      pMemSeq = &localMemContext.memSeqs[loop];
      if (pMemSeq->pStartPos == NULL)
      {
         pMemSeq->memArea = memArea;
         pMemSeq->pStartPos = (U8BIT *)memArea + offset;
         pMemSeq->pCurrentPos =
            localMemContext.memSeqs[loop].pStartPos;
         pMemSeq->seqLength = length;
         pMemSeq->currentOffset = 0;
         pMemSeq->savedOffset = 0;
         break;
      }
   }

   localMemContext.numSeqsOpen++;
   *memAreaRef = (MemSeqRef) pMemSeq;

   return MEM_NO_ERR;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqOpenClone
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqOpenClone(
   /*I*/ MemSeqRef memAreaRefOrig,
   /*O*/ MemSeqRef *memAreaRefClone )
{
   pLocalSeqRef pMemSeqOrig = (pLocalSeqRef) memAreaRefOrig;
   pLocalSeqRef pMemSeqClone = NULL;
   S32BIT loop;

   dsmDP4(("memSeqClone()\n"));

   dsmAssert((memAreaRefClone != NULL));

   /* Have we reached the limit for open seqs? */
   if (localMemContext.numSeqsOpen == MEM_NUM_SEQS)
   {
      dsmDP4(("INF: Memory sequence open limit reached!\n"));
      return MEM_ERR_SEQ_OPEN_LIMIT;
   }

   /* Put this ref's details in a blank element */
   for (loop = MEM_NUM_SEQS - 1; loop; loop--)
   {
      pMemSeqClone = &localMemContext.memSeqs[loop];
      if (pMemSeqClone->pStartPos == NULL)
      {
         pMemSeqClone->memArea = pMemSeqOrig->memArea;
         pMemSeqClone->pStartPos = pMemSeqOrig->pStartPos;
         pMemSeqClone->pCurrentPos = pMemSeqOrig->pCurrentPos;
         pMemSeqClone->seqLength = pMemSeqOrig->seqLength;
         pMemSeqClone->currentOffset = pMemSeqOrig->currentOffset;
         pMemSeqClone->savedOffset = pMemSeqOrig->savedOffset;
         break;
      }
   }

   localMemContext.numSeqsOpen++;
   *memAreaRefClone = (MemSeqRef) pMemSeqClone;

   return MEM_NO_ERR;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqClose
//
///////////////////////////////////////////////////////////////////////////// */
void memSeqClose( /*I*/ MemSeqRef memAreaRef )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef)memAreaRef;

   dsmDP4(("memSeqClose()\n"));

   dsmAssert((memAreaRef != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   pMemSeq->memArea = NULL;
   pMemSeq->pStartPos = NULL;

   localMemContext.numSeqsOpen--;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqAccessContig
//
///////////////////////////////////////////////////////////////////////////// */
void memSeqAccessContig(
   MemSeqRef memAreaRef, U8BIT **memPtr, U32BIT *numContigBytes )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef)memAreaRef;

   dsmDP4(("memSeqAccessContig()\n"));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((memPtr != NULL));
   dsmAssert((numContigBytes != NULL));


   /* Check for valid ref */
   memCheckRef(pMemSeq);

   *memPtr = pMemSeq->pStartPos;
   *numContigBytes = pMemSeq->seqLength;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqAccessCurrent
//
///////////////////////////////////////////////////////////////////////////// */
void memSeqAccessCurrent(
   MemSeqRef memAreaRef, U8BIT **memPtr, U32BIT *numContigBytes )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef)memAreaRef;
   U32BIT bytesRemaining;

   dsmDP4(("memSeqAccessContig()\n"));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((memPtr != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   bytesRemaining = pMemSeq->seqLength - pMemSeq->currentOffset;

   *memPtr = pMemSeq->pCurrentPos;
   if (*numContigBytes > bytesRemaining)
   {
      *numContigBytes = bytesRemaining;
   }
   pMemSeq->pCurrentPos += *numContigBytes;
}

#ifndef NDEBUG
/* /////////////////////////////////////////////////////////////////////////////
// memSeqNumOpen
//
///////////////////////////////////////////////////////////////////////////// */
U32BIT memSeqNumOpen( void *context )
{
   dsmDP4(("memSeqNumOpen()\n"));

   return localMemContext.numSeqsOpen;
}

#endif  /* NDEBUG */


/* /////////////////////////////////////////////////////////////////////////////
// memSeqSize
//
///////////////////////////////////////////////////////////////////////////// */
void memSeqSize( /*I*/ MemSeqRef memAreaRef, /*O*/ U32BIT *size )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef)memAreaRef;

   dsmDP4(("memSeqSize()\n"));

   dsmAssert((size != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   *size = pMemSeq->seqLength;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqValidate
//
///////////////////////////////////////////////////////////////////////////// */
void memSeqValidate( /*I*/ MemSeqRef memAreaRef, /*O*/ BOOLEAN *pValid )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef)memAreaRef;

   dsmDP4(("memSeqValidate()\n"));

   dsmAssert((memAreaRef != NULL));

   /* Check for valid ref */
   if ((pMemSeq < &localMemContext.memSeqs[0]) ||
       (pMemSeq > &localMemContext.memSeqs[MEM_NUM_SEQS - 1]) ||
       (pMemSeq->pStartPos == NULL) ||
       (pMemSeq->magic != MEM_MAGIC))
   {
      dsmDP4(("INF: Memory sequence reference is invalid!\n"));
      *pValid = FALSE;
   }
   else
   {
      *pValid = TRUE;
   }
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqReadByte
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqReadByte( /*I*/ MemSeqRef src, /*O*/ U8BIT *byte )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef) src;

   dsmDP4(("memSeqReadU8BIT()\n"));

   dsmAssert((src != NULL));
   dsmAssert((byte != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   /* Check not at end of sequence */
   if (pMemSeq->currentOffset == pMemSeq->seqLength)
   {
      dsmDP4(("INF: Reached end of sequence\n"));
      return MEM_ERR_SEQ_END_OF_DATA;
   }

   *byte = *pMemSeq->pCurrentPos;
   pMemSeq->pCurrentPos++;
   pMemSeq->currentOffset++;

   return MEM_NO_ERR;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqWriteByte
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqWriteByte( /*I*/ U8BIT byte, MemSeqRef dest )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef) dest;

   dsmDP4(("memSeqWriteU8BIT()\n"));

   dsmAssert((dest != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   /* Check that current position is ok */
   if (pMemSeq->currentOffset == pMemSeq->seqLength)
   {
      dsmDP4(("INF: Reached end of sequence\n"));
      return MEM_ERR_SEQ_END_OF_DATA;
   }

   *pMemSeq->pCurrentPos = byte;
   pMemSeq->pCurrentPos++;
   pMemSeq->currentOffset++;

   return MEM_NO_ERR;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqRead
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqRead(
   MemSeqRef src, U8BIT *dest, U32BIT numBytes, U32BIT *numBytesActual )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef) src;
   U32BIT bytesToRead, bytesRemaining;
   E_DsmMemErr retVal = MEM_NO_ERR;

   dsmDP4(("memSeqRead()\n"));

   dsmAssert((src != NULL));
   dsmAssert((numBytes > 0));
   dsmAssert((dest != NULL));
   dsmAssert((numBytesActual != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   bytesRemaining = pMemSeq->seqLength - pMemSeq->currentOffset;

   if (numBytes > bytesRemaining)
   {
      bytesToRead = bytesRemaining;
      retVal = MEM_ERR_SEQ_END_OF_DATA;
   }
   else
   {
      bytesToRead = numBytes;
   }

   memcpy( dest, pMemSeq->pCurrentPos, bytesToRead );

   pMemSeq->pCurrentPos += bytesToRead;
   pMemSeq->currentOffset += bytesToRead;

   *numBytesActual = bytesToRead;

   return retVal;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqWrite
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqWrite(
   U8BIT *src, MemSeqRef dest, U32BIT numBytes, U32BIT *numBytesActual )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef) dest;
   U32BIT bytesToWrite, bytesRemaining;
   E_DsmMemErr retVal = MEM_NO_ERR;

   dsmDP4(("memSeqWrite()\n"));

   dsmAssert((src != NULL));
   dsmAssert((numBytes > 0));
   dsmAssert((dest != NULL));
   dsmAssert((numBytesActual != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   bytesRemaining = pMemSeq->seqLength - pMemSeq->currentOffset;

   if (numBytes > bytesRemaining)
   {
      bytesToWrite = bytesRemaining;
      retVal = MEM_ERR_SEQ_END_OF_DATA;
   }
   else
   {
      bytesToWrite = numBytes;
   }

   memcpy( pMemSeq->pCurrentPos, src, bytesToWrite );

   pMemSeq->pCurrentPos += bytesToWrite;
   pMemSeq->currentOffset += bytesToWrite;

   *numBytesActual = bytesToWrite;

   return retVal;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqCopy
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqCopy(
   MemSeqRef src, MemSeqRef dest, U32BIT numBytes, U32BIT *numBytesActual )
{
   pLocalSeqRef pMemSeqSrc = (pLocalSeqRef) src;
   pLocalSeqRef pMemSeqDest = (pLocalSeqRef) dest;
   U32BIT bytesToCopy, bytesRemainingSrc, bytesRemainingDest;
   E_DsmMemErr retVal = MEM_NO_ERR;

   dsmDP4(("memSeqCopy()\n"));

   dsmAssert((src != NULL));
   dsmAssert((dest != NULL));
   dsmAssert((numBytes > 0));
   dsmAssert((numBytesActual != NULL));

   /* Check for valid refs */
   memCheckRef(pMemSeqSrc);
   memCheckRef(pMemSeqDest);

   bytesRemainingSrc = pMemSeqSrc->seqLength - pMemSeqSrc->currentOffset;
   bytesRemainingDest = pMemSeqDest->seqLength - pMemSeqDest->currentOffset;

   if (numBytes > bytesRemainingDest)
   {
      bytesToCopy = bytesRemainingDest;
      retVal = MEM_ERR_SEQ_END_OF_DATA;
   }
   else
   {
      bytesToCopy = numBytes;
   }

   if (bytesToCopy > bytesRemainingSrc)
   {
      bytesToCopy = bytesRemainingSrc;
      retVal = MEM_ERR_SEQ_END_OF_DATA;
   }

   memcpy( pMemSeqDest->pCurrentPos, pMemSeqSrc->pCurrentPos, bytesToCopy );

   pMemSeqSrc->pCurrentPos += bytesToCopy;
   pMemSeqSrc->currentOffset += bytesToCopy;
   pMemSeqDest->pCurrentPos += bytesToCopy;
   pMemSeqDest->currentOffset += bytesToCopy;

   return retVal;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqCompContig
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqCompContig(
   /*I*/ U8BIT *contig, MemSeqRef memAreaRef, U32BIT numBytes,
   /*O*/ BOOLEAN *equal )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef) memAreaRef;

   dsmDP4(("memSeqCompContig()\n"));

   dsmAssert((contig != NULL));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((numBytes > 0));
   dsmAssert((equal != NULL));

   memCheckRef(pMemSeq);

   /* Check that the entire compare area fits into the sequence */
   if (pMemSeq->seqLength < pMemSeq->currentOffset + numBytes)
   {
      return MEM_ERR_SEQ_END_OF_DATA;
   }

   /* Set equal to TRUE/FALSE if strings are equal/unequal */
   *equal = ((memcmp( contig, pMemSeq->pCurrentPos, numBytes ) != 0) ? FALSE : TRUE);

   return MEM_NO_ERR;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqCompMgd
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqCompMgd(
   /*I*/ MemSeqRef memAreaRef1, MemSeqRef memAreaRef2, U32BIT numBytes,
   /*O*/ BOOLEAN *equal )
{
   pLocalSeqRef pMemSeq1 = (pLocalSeqRef) memAreaRef1;
   pLocalSeqRef pMemSeq2 = (pLocalSeqRef) memAreaRef2;

   dsmDP4(("memSeqCompMgd()\n"));

   dsmAssert((memAreaRef1 != NULL));
   dsmAssert((memAreaRef2 != NULL));
   dsmAssert((numBytes > 0));
   dsmAssert((equal != NULL));

   memCheckRef(pMemSeq1);
   memCheckRef(pMemSeq2);

   /* Check that portions to compare fit into the sequences */
   if (pMemSeq1->seqLength < pMemSeq1->currentOffset + numBytes)
   {
      return MEM_ERR_SEQ_END_OF_DATA;
   }
   if (pMemSeq2->seqLength < pMemSeq2->currentOffset + numBytes)
   {
      return MEM_ERR_SEQ_END_OF_DATA;
   }

   /* Set equal to TRUE/FALSE if strings are equal/unequal */
   *equal = ((memcmp(
                 pMemSeq1->pCurrentPos, pMemSeq2->pCurrentPos, numBytes ) != 0)
             ? FALSE : TRUE);

   return MEM_NO_ERR;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqSetPosRel
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqSetPosRel( /*I*/ MemSeqRef memAreaRef, S32BIT position )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef)memAreaRef;
   E_DsmMemErr retVal = MEM_NO_ERR;
   S32BIT actualOffset;

   dsmDP4(("memSeqSetPosRel()\n"));

   dsmAssert((memAreaRef != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   actualOffset = pMemSeq->currentOffset + position;

   if (actualOffset < 0)
   {
      /* Overrun the start of the sequence */
      pMemSeq->pCurrentPos = pMemSeq->pStartPos;
      pMemSeq->currentOffset = 0;
      retVal = MEM_ERR_SEQ_END_OF_DATA;
   }
   else if ((U32BIT)actualOffset > pMemSeq->seqLength)
   {
      /* Overrun the end of the sequence */
      pMemSeq->pCurrentPos = pMemSeq->pStartPos + pMemSeq->seqLength;
      pMemSeq->currentOffset = pMemSeq->seqLength;
      retVal = MEM_ERR_SEQ_END_OF_DATA;
   }
   else
   {
      /* Set the current cursor position */
      pMemSeq->pCurrentPos = pMemSeq->pStartPos + actualOffset;
      pMemSeq->currentOffset = actualOffset;
   }

   return retVal;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqSetPosAbs
//
///////////////////////////////////////////////////////////////////////////// */
E_DsmMemErr memSeqSetPosAbs( /*I*/ MemSeqRef memAreaRef, U32BIT position )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef)memAreaRef;
   E_DsmMemErr retVal = MEM_NO_ERR;

   dsmDP4(("memSeqSetPosAbs()\n"));

   dsmAssert((memAreaRef != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   if (position > pMemSeq->seqLength)
   {
      /* Overrun the end of the sequence */
      pMemSeq->pCurrentPos = pMemSeq->pStartPos + pMemSeq->seqLength;
      pMemSeq->currentOffset = pMemSeq->seqLength;
      retVal = MEM_ERR_SEQ_END_OF_DATA;
   }
   else
   {
      /* Set the current cursor position */
      pMemSeq->pCurrentPos = pMemSeq->pStartPos + position;
      pMemSeq->currentOffset = position;
   }

   return retVal;
}

/* /////////////////////////////////////////////////////////////////////////////
// memSeqReadPos
//
///////////////////////////////////////////////////////////////////////////// */
extern void memSeqReadPos( MemSeqRef memAreaRef, U32BIT *position )
{
   pLocalSeqRef pMemSeq = (pLocalSeqRef)memAreaRef;

   dsmDP4(("memSeqReadPos()\n"));

   dsmAssert((memAreaRef != NULL));

   /* Check for valid ref */
   memCheckRef(pMemSeq);

   *position = pMemSeq->currentOffset;
}

/*----------------------------------------------------------------------------*/
