/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Implementation of the clDsmUtils module.
 *             Contains generally useful functions.
 * @file    clDsmUtils.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "clDsmUtils.h"

#include "cacheMgr.h"

/* Include function to calculate CRC */
#include "calculateCRC.h"

/*------------------------------- Local Macros -------------------------------*/

#define SYSTEM_HARDWARE_DESC     0x01
#define SYSTEM_SOFTWARE_DESC     0x02

/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/


/*------------------- local prototypes/forward declarations ------------------*/


/*---------------------------- Exported Functions ----------------------------*/

/* -- GENERAL UTILITIES */

/*
-- Convert DSM-CC object kind to internal object kind
*/
E_DsmObjectKind convertObjectKindStr( U32BIT objectKindStr )
{
   E_DsmObjectKind kind;

   switch (objectKindStr)
   {
      case SRG_STR:
         kind = SERVICE_GATEWAY_OBJ;
         break;
      case DIR_STR:
         kind = DIRECTORY_OBJ;
         break;
      case FIL_STR:
         kind = FILE_OBJ;
         break;
      case STR_STR:
         kind = STREAM_OBJ;
         break;
      case STE_STR:
         kind = STREAM_OBJ_WITH_EVENTS;
         break;
      default:
         kind = ANON_OBJ;
         break;
   }
   return(kind);
}

/* /////////////////////////////////////////////////////////////////////////////
// handleInLoopError
//
// Notify any system or internal (ie. fatal) errors via errorFunc callback.
//
// Set error return value for an internal error and set it for a system error
// if one is not currently set.
//
// Don't notify CLDSM_ERR_MEM_HEAP_FULL or CLDSM_ERR_SECTION_FILTER_HEAP_FULL
// in since it may get cleared later in loop?!
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError handleInLoopError( P_DsmCoreInst idp,
   E_DscError currErr, E_DscError newErr )
{
   E_DscError err = CLDSM_OK;

   switch (newErr)
   {
      /* -- NB. Fall-through is intentional */
      case CLDSM_ERR_INTERNAL:
         err = newErr;
      case CLDSM_ERR_SYSTEM_ADD_SECTION_FILTER:
      case CLDSM_ERR_SI_QUERY_FAILED:
         if (idp->setup.errorFunc)
         {
            idp->setup.errorFunc( newErr, NULL );
         }
         err = currErr ? currErr : newErr;
         break;

      default:
         /* -- Do not notify or return any other errors */
         break;
   }
   return err;
}

/* -- MANAGED MEMORY ACCESS FUNCTIONS */

#include "defMemUtilsMgd.h"

/* /////////////////////////////////////////////////////////////////////////////
// readUInt16Seq
// Reads a 16 bit value from the current memory area (advances cursor by 2)
///////////////////////////////////////////////////////////////////////////// */
void readUInt16Seq( MemSeqRef memAreaRef, U16BIT *pUi16 )
{
   U8BIT data[2];
   U32BIT dummy;
   U16BIT result;

   dsmDP4(("readUInt16Seq()\n"));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((pUi16 != NULL));

   CHECK_MEM_ERR( memSeqRead( memAreaRef, data, 2, &dummy ));

   result = (U16BIT)data[0];
   *pUi16 = (U16BIT)((result << 8) | data[1]);

   dsmDP4(("exit readUInt16Seq: %x\n", *pUi16));
}

/* /////////////////////////////////////////////////////////////////////////////
// readUInt32Seq
// Reads a 32 bit value from the current memory area (advances cursor by 4)
///////////////////////////////////////////////////////////////////////////// */
void readUInt32Seq( MemSeqRef memAreaRef, U32BIT *pUi32 )
{
   U8BIT data[4];
   U32BIT dummy;
   U32BIT result;

   dsmDP4(("readUInt32Seq()\n"));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((pUi32 != NULL));

   CHECK_MEM_ERR( memSeqRead( memAreaRef, data, 4, &dummy ));

   result = (U32BIT)data[0];
   result = (U32BIT)((result << 8) | data[1]);
   result = (U32BIT)((result << 8) | data[2]);
   *pUi32 = (U32BIT)((result << 8) | data[3]);

   dsmDP4(("exit readUInt32Seq: %x\n", *pUi32));
}

/* /////////////////////////////////////////////////////////////////////////////
// getUInt8Seq
// Gets an 8 bit value from the current memory area (restores cursor to start)
///////////////////////////////////////////////////////////////////////////// */
void getUInt8Seq( MemSeqRef memAreaRef, U8BIT *pUi8 )
{
   dsmDP4(("getUInt8Seq()\n"));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((pUi8 != NULL));

   CHECK_MEM_ERR( memSeqReadByte( memAreaRef, pUi8 ));
   CHECK_MEM_ERR( memSeqSetPosRel( memAreaRef, -1 ));

   dsmDP4(("exit getUInt8Seq: %x\n", *pUi8));
}

/* /////////////////////////////////////////////////////////////////////////////
// getUInt16Seq
// Gets a 16 bit value from the current memory area (restores cursor to start)
///////////////////////////////////////////////////////////////////////////// */
void getUInt16Seq( MemSeqRef memAreaRef, U16BIT *pUi16 )
{
   U8BIT data[2];
   U32BIT dummy;
   U16BIT result;

   dsmDP4(("getUInt16Seq()\n"));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((pUi16 != NULL));

   CHECK_MEM_ERR( memSeqRead( memAreaRef, data, 2, &dummy ));
   CHECK_MEM_ERR( memSeqSetPosRel( memAreaRef, -2 ));

   result = (U16BIT)data[0];
   *pUi16 = (U16BIT)((result << 8) | data[1]);

   dsmDP4(("exit getUInt16Seq: %x\n", *pUi16));
}

/* /////////////////////////////////////////////////////////////////////////////
// getUInt32Seq
// Gets a 32 bit value from the current memory area (restores cursor to start)
///////////////////////////////////////////////////////////////////////////// */
void getUInt32Seq( MemSeqRef memAreaRef, U32BIT *pUi32 )
{
   U8BIT data[4];
   U32BIT dummy;
   U32BIT result;

   dsmDP4(("getUInt32Seq()\n"));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((pUi32 != NULL));

   CHECK_MEM_ERR( memSeqRead( memAreaRef, data, 4, &dummy ));
   CHECK_MEM_ERR( memSeqSetPosRel( memAreaRef, -4 ));

   result = (U32BIT)data[0];
   result = (U32BIT)((result << 8) | data[1]);
   result = (U32BIT)((result << 8) | data[2]);
   *pUi32 = (U32BIT)((result << 8) | data[3]);

   dsmDP4(("exit getUInt32Seq: %x\n", *pUi32));
}

/* /////////////////////////////////////////////////////////////////////////////
// readObjectKeySeq
//
// memAreaRef must be an already opened MemSeq area with the cursor pointing at
//    the objectKeyLength field of required DVB (ie. 1-4 byte) objectKey data area
//
// pObjectKey returns the objectKey value.
//
// pValid indicates if objectKey successfully recovered
//
// On exit - If valid:  cursor advanced by objectKeyLength + 1
//           Else:      cursor set back to initial position
///////////////////////////////////////////////////////////////////////////// */
void readObjectKeySeq( MemSeqRef memAreaRef,
   P_ObjectKey pObjectKey, BOOLEAN *pValid )
{
   U8BIT objKeyLen;
   U32BIT numBytesActual;
   E_DsmMemErr memErr;

   dsmDP4(("readObjectKeySeq()\n"));
   dsmAssert((memAreaRef != NULL));
   dsmAssert((pObjectKey != NULL));
   dsmAssert((pValid != NULL));


   *pValid = FALSE;
   pObjectKey->length = 0;

   /* -- Read objectKey size */
   CHECK_MEM_ERR( memSeqReadByte( memAreaRef, &objKeyLen ));

   if ((objKeyLen < 1) || (objKeyLen > 4))
   {
      dsmDP2(("DATA ERROR: ObjectKey length = %d\n", objKeyLen));
      memSeqSetPosRel( memAreaRef, -1 );
   }
   else
   {
      memErr = memSeqRead(
            memAreaRef, pObjectKey->data, objKeyLen, &numBytesActual );

      if (memErr != MEM_NO_ERR)
      {
         memSeqSetPosRel( memAreaRef, -(S32BIT)(numBytesActual + 1));
      }
      else
      {
         pObjectKey->length = objKeyLen;
         *pValid = TRUE;
      }
   }

   dsmDP4(("exit readObjectKeySeq - length: %u, data: %02x%02x%02x%02x\n",
           pObjectKey->length, pObjectKey->data[0], pObjectKey->data[1],
           pObjectKey->data[2], pObjectKey->data[3]));
}

#ifndef MEM_CONTIGUOUS

/* /////////////////////////////////////////////////////////////////////////////
// getIorInfoSeq
//      get location and tap info from IOR data held in managed (MemSeq) memory
//      mpIorData (MemPtr) is normally an opened MemSeqRef with the cursor at the
//          first byte of IOR (nb. if MEM_CONTIGUOUS MemPtr defined to U8BIT*)
//      pValid indicates whether info was successfully recovered
//
//      NB. Uses GET_IOR_INFO Common Code with getIorInfoContig
//          mpIorData (MemSeqRef) cursor is left unaltered
//
///////////////////////////////////////////////////////////////////////////// */
U16BIT getIorInfoSeq( const MemPtr mpIorData, U32BIT *pIorTypeId,
   P_ObjectLocation pLocation, P_DeliveryParaTap pTap )
{
   U32BIT length_used;

   dsmDP3(("getIorInfoSeq()\n"));

#include "getIopIor_include_src.h"

   dsmDP3(("exit getIorInfoSeq -> rtn: %u\n", length_used));
   return (U16BIT)length_used;
}

#endif /*!MEM_CONTIGUOUS*/

/* -- CONTIGUOUS MEMORY ACCESS FUNCTIONS */

#include "defMemUtilsContig.h"

/* /////////////////////////////////////////////////////////////////////////////
// getIorInfoContig
//      get location and tap info from IOR data held in contiguous memory
//      mpIorData points to first byte of IOR
//      pValid indicates whether info was successfully recovered
//
//      NB. Uses GET_IOR_INFO Common Code with getIorInfoMemSeq
///////////////////////////////////////////////////////////////////////////// */
U16BIT getIorInfoContig( const MemPtr mpIorData, U32BIT *pIorTypeId,
   P_ObjectLocation pLocation, P_DeliveryParaTap pTap )
{
   U32BIT length_used;

   dsmDP3(("getIorInfoContig()\n"));

#include "getIopIor_include_src.h"

   dsmDP3(("exit getIorInfoContig -> rtn: %u\n", length_used));
   return (U16BIT)length_used;
}

P_Compatibility DSC_UtilParseCompatibilityDesc( P_DsmCoreInst idp,
   MemPtr pData, U16BIT length, U32BIT specifier )
{
   P_Compatibility pCompat;
   S_modelVersion *pMV;
   U32BIT oui;
   U16BIT count, cnt;
   U8BIT dlen, type;
   MemPtr mp;
   READ_UINT16( pData, count );
   pCompat = (P_Compatibility)DSC_CmMemGet( idp, sizeof(S_Compatibility) + (count*sizeof(S_modelVersion)) );
   if (pCompat != NULL)
   {
      pMV = (S_modelVersion *)(pCompat + 1);
      pCompat->hwlist = pMV;
      pCompat->hw_num = 0;
      cnt = count;
      mp = pData;
      while (cnt--)
      {
         READ_UINT8( mp, type );
         READ_UINT8( mp, dlen );
         READ_UINT32( mp, oui );
         if (type == SYSTEM_HARDWARE_DESC && oui == specifier)
         {
            pCompat->hw_num++;
            READ_UINT16( mp, pMV->model );
            READ_UINT16( mp, pMV->version );
            pMV++;
         }
         else
         {
            SET_POS_REL( mp, 4 );
         }
         if (dlen > 8)
         {
            SET_POS_REL( mp, dlen - 8 );
         }
      }

      pCompat->swlist = pMV;
      pCompat->sw_num = 0;
      cnt = count;
      mp = pData;
      while (cnt--)
      {
         READ_UINT8( mp, type );
         READ_UINT8( mp, dlen );
         READ_UINT32( mp, oui );
         if (type == SYSTEM_SOFTWARE_DESC && oui == specifier)
         {
            pCompat->sw_num++;
            READ_UINT16( mp, pMV->model );
            READ_UINT16( mp, pMV->version );
            pMV++;
         }
         else
         {
            SET_POS_REL( mp, 4 );
         }
         if (dlen > 8)
         {
            /* Skip subDescriptor */
            SET_POS_REL( mp, dlen - 8 );
         }
      }
   }
   return pCompat;
}


BOOLEAN DSC_UtilCheckCompatibility( U8BIT *pData, U16BIT len,
   U32BIT specifier, S_SsuModelVersion smv )
{
   U32BIT oui;
   U16BIT count, match;
   U16BIT model, version;
   U8BIT dlen, type;
   READ_UINT16( pData, count );
   match = 0;
   while (count--)
   {
      READ_UINT8( pData, type );
      READ_UINT8( pData, dlen );
      READ_UINT32( pData, oui );
      READ_UINT16( pData, model );
      READ_UINT16( pData, version );
      if (oui == specifier)
      {
         if (type == SYSTEM_HARDWARE_DESC &&
             smv.hw_model == model &&
             smv.hw_version == version)
         {
            match |= 1;
         }
         else if (type == SYSTEM_SOFTWARE_DESC &&
             smv.sw_model == model &&
             smv.sw_version == version)
         {
            match |= 2;
         }
         if (match == 3)
         {
            break;
         }
      }
      if (dlen > 8)
      {
         SET_POS_REL( pData, dlen - 8 );
      }
   }
   return (match == 3)? TRUE : FALSE;
}

/*------------------------------ Local Functions -----------------------------*/


/*----------------------------------------------------------------------------*/
