/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   eader to the clDsmUtils module.
 * @file    clDsmUtils.h
 * @date    28/9/2001
 * @author  R Taylor
 */
#ifndef _CLDSMUTILS_H_
#define _CLDSMUTILS_H_


/*--------------------------------  Includes  --------------------------------*/

#include "clDsmSystem.h"


#ifdef __cplusplus
extern "C" {
#endif


/*-----------------------------  Exported Macros  ----------------------------*/

/*
-- These macro definitions provide three 'priority' levels (0, 1, 2) of error
-- checking on 8/16/32bit values stored in contiguous memory OR managed memory.
-- The highest priority (0) is always performed and cannot be disabled, the
-- lower two priorities can be optionally disabled for retail mode (at compile
-- time).
--
-- The lower priority levels (2 or 1 & 2) can be enabled/disabled for retail
-- builds (via DSM_DATA_CHECK_LEVEL).
--
-- Both lower levels (1 & 2) of checking are always enabled in debug mode
--
-- Syntax is:
-- <action>_UINT<sz>_L<lvl>CHK( pData, var<sz>, condn, dbgErrActn, relErrActn )
--
-- <action> = READ - Always reads value + advances read ptr
-- <action> = GET  - Always reads value only
-- <action> = ADV  - If checking level enabled (or debug mode)
--                      reads value + advances read ptr
--                   else
--                      advances read ptr only
-- <sz>     = 8, 16, 32
-- <lvl>    = 1 - Level 1 priority checking only
-- <lvl>    = 2 - Level 1 and level 2 checking
--
-- var<sz>      = 'temporary' variable to receive UINT<sz>
-- *** NB. Must be of the correct type                                  ***
-- ***     Is not used/assigned if <action> = ADV_ & checking disabled ***
--
-- pData        = pointer/reference to MSByte of UINT<size> in memory
--
-- condn        = valid condition to be checked for error (eg. var8 >= 4 )
--
-- dbgErrActn   = command(s) to execute on error condition in debug builds
-- relErrActn   = command(s) to execute on error condition in all builds
--                *** NB. dbgErrAction always executed before relErrAction ***
--
*/
#if !defined(DSM_DATA_CHECK_LEVEL)
    #define DSM_DATA_CHECK_LEVEL    1
#endif


/* -- Level 0 data checking always performed */
/* -- No advance only macros at level 0 since data always read and checked */

#if (defined(NDEBUG) || defined(DSM_PROFILING_BUILD))

    #define READ_UINT8_L0CHK( pData, var8, condn, dbgErrActn, relErrActn )  \
   READ_UINT8( pData, var8 );                                          \
   if (!(condn)) { relErrActn; }

    #define READ_UINT16_L0CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   READ_UINT16( pData, var16 );                                        \
   if (!(condn)) { relErrActn; }

    #define READ_UINT32_L0CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   READ_UINT32( pData, var32 );                                        \
   if (!(condn)) { relErrActn; }


    #define GET_UINT8_L0CHK( pData, var8, condn, dbgErrActn, relErrActn )   \
   GET_UINT8( pData, var8 );                                           \
   if (!(condn)) { relErrActn; }

    #define GET_UINT16_L0CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   GET_UINT16( pData, var16 );                                         \
   if (!(condn)) { relErrActn; }

    #define GET_UINT32_L0CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   GET_UINT32( pData, var32 );                                         \
   if (!(condn)) { relErrActn; }

#else
/* -- Debug build */

    #define READ_UINT8_L0CHK( pData, var8, condn, dbgErrActn, relErrActn )  \
   READ_UINT8( pData, var8 );                                          \
   if (!(condn)) { dbgErrActn; relErrActn; }

    #define READ_UINT16_L0CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   READ_UINT16( pData, var16 );                                        \
   if (!(condn)) { dbgErrActn; relErrActn; }

    #define READ_UINT32_L0CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   READ_UINT32( pData, var32 );                                        \
   if (!(condn)) { dbgErrActn; relErrActn; }


    #define GET_UINT8_L0CHK( pData, var8, condn, dbgErrActn, relErrActn )   \
   GET_UINT8( pData, var8 );                                           \
   if (!(condn)) { dbgErrActn; relErrActn; }

    #define GET_UINT16_L0CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   GET_UINT16( pData, var16 );                                         \
   if (!(condn)) { dbgErrActn; relErrActn; }

    #define GET_UINT32_L0CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   GET_UINT32( pData, var32 );                                         \
   if (!(condn)) { dbgErrActn; relErrActn; }

#endif


/* -- Level 1 data checking is optional */

#if DSM_DATA_CHECK_LEVEL >= 1

/* -- Level 1 data checking in all builds (default) */

    #if (defined(NDEBUG) || defined(DSM_PROFILING_BUILD))

        #define READ_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn )  \
   READ_UINT8( pData, var8 );                                          \
   if (!(condn)) { relErrActn; }

        #define READ_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   READ_UINT16( pData, var16 );                                        \
   if (!(condn)) { relErrActn; }

        #define READ_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   READ_UINT32( pData, var32 );                                        \
   if (!(condn)) { relErrActn; }

        #define ADV_UINT8_L1CHK     READ_UINT8_L1CHK
        #define ADV_UINT16_L1CHK    READ_UINT16_L1CHK
        #define ADV_UINT32_L1CHK    READ_UINT32_L1CHK

        #define GET_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn )   \
   GET_UINT8( pData, var8 );                                           \
   if (!(condn)) { relErrActn; }

        #define GET_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   GET_UINT16( pData, var16 );                                         \
   if (!(condn)) { relErrActn; }

        #define GET_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   GET_UINT32( pData, var32 );                                         \
   if (!(condn)) { relErrActn; }

    #else
/* -- Debug build */

        #define READ_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn )  \
   READ_UINT8( pData, var8 );                                          \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define READ_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   READ_UINT16( pData, var16 );                                        \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define READ_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   READ_UINT32( pData, var32 );                                        \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define ADV_UINT8_L1CHK     READ_UINT8_L1CHK
        #define ADV_UINT16_L1CHK    READ_UINT16_L1CHK
        #define ADV_UINT32_L1CHK    READ_UINT32_L1CHK

        #define GET_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn )   \
   GET_UINT8( pData, var8 );                                           \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define GET_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   GET_UINT16( pData, var16 );                                         \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define GET_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   GET_UINT32( pData, var32 );                                         \
   if (!(condn)) { dbgErrActn; relErrActn; }

    #endif

#else

/* -- Level 1 data checking in debug builds only */

    #if (defined(NDEBUG) || defined(DSM_PROFILING_BUILD))

        #define READ_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   READ_UINT8( pData, var8 )

        #define READ_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn) \
   READ_UINT16( pData, var16 )

        #define READ_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn) \
   READ_UINT32( pData, var32 )

        #define ADV_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   SET_POS_REL( pData, 1 )

        #define ADV_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   SET_POS_REL( pData, 2 )

        #define ADV_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   SET_POS_REL( pData, 4 )

        #define GET_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   GET_UINT8( pData, var8 )

        #define GET_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn) \
   GET_UINT16( pData, var16 )

        #define GET_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn) \
   GET_UINT32( pData, var32 )

    #else
/* -- Debug Build */

        #define READ_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   READ_UINT8( pData, var8 );                                      \
   if (!(condn)) { dbgErrActn; }

        #define READ_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn) \
   READ_UINT16( pData, var16 );                                    \
   if (!(condn)) { dbgErrActn; }

        #define READ_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn) \
   READ_UINT32( pData, var32 );                                    \
   if (!(condn)) { dbgErrActn; }

        #define ADV_UINT8_L1CHK     READ_UINT8_L1CHK
        #define ADV_UINT16_L1CHK    READ_UINT16_L1CHK
        #define ADV_UINT32_L1CHK    READ_UINT32_L1CHK

        #define GET_UINT8_L1CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   GET_UINT8( pData, var8 );                                       \
   if (!(condn)) { dbgErrActn; }

        #define GET_UINT16_L1CHK( pData, var16, condn, dbgErrActn, relErrActn) \
   GET_UINT16( pData, var16 );                                     \
   if (!(condn)) { dbgErrActn; }

        #define GET_UINT32_L1CHK( pData, var32, condn, dbgErrActn, relErrActn) \
   GET_UINT32( pData, var32 );                                     \
   if (!(condn)) { dbgErrActn; }

    #endif
#endif


/* -- Level 2 data checking is optional */

#if DSM_DATA_CHECK_LEVEL >= 2

/* -- Level 2 data checking in all builds */

    #if (defined(NDEBUG) || defined(DSM_PROFILING_BUILD))

        #define READ_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn )  \
   READ_UINT8( pData, var8 );                                          \
   if (!(condn)) { relErrActn; }

        #define READ_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   READ_UINT16( pData, var16 );                                        \
   if (!(condn)) { relErrActn; }

        #define READ_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   READ_UINT32( pData, var32 );                                        \
   if (!(condn)) { relErrActn; }

        #define ADV_UINT8_L2CHK     READ_UINT8_L2CHK
        #define ADV_UINT16_L2CHK    READ_UINT16_L2CHK
        #define ADV_UINT32_L2CHK    READ_UINT32_L2CHK

        #define GET_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn )   \
   GET_UINT8( pData, var8 );                                           \
   if (!(condn)) { relErrActn; }

        #define GET_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   GET_UINT16( pData, var16 );                                         \
   if (!(condn)) { relErrActn; }

        #define GET_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   GET_UINT32( pData, var32 );                                         \
   if (!(condn)) { relErrActn; }

    #else
/* -- Debug build */

        #define READ_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn )  \
   READ_UINT8( pData, var8 );                                          \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define READ_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   READ_UINT16( pData, var16 );                                        \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define READ_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   READ_UINT32( pData, var32 );                                        \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define ADV_UINT8_L2CHK     READ_UINT8_L2CHK
        #define ADV_UINT16_L2CHK    READ_UINT16_L2CHK
        #define ADV_UINT32_L2CHK    READ_UINT32_L2CHK

        #define GET_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn )   \
   GET_UINT8( pData, var8 );                                           \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define GET_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   GET_UINT16( pData, var16 );                                         \
   if (!(condn)) { dbgErrActn; relErrActn; }

        #define GET_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   GET_UINT32( pData, var32 );                                         \
   if (!(condn)) { dbgErrActn; relErrActn; }

    #endif

#else

/* -- Level 2 data checking in debug builds only (default) */

    #if (defined(NDEBUG) || defined(DSM_PROFILING_BUILD))

        #define READ_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   READ_UINT8( pData, var8 )

        #define READ_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn) \
   READ_UINT16( pData, var16 )

        #define READ_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn) \
   READ_UINT32( pData, var32 )

        #define ADV_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   SET_POS_REL( pData, 1 )

        #define ADV_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn ) \
   SET_POS_REL( pData, 2 )

        #define ADV_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn ) \
   SET_POS_REL( pData, 4 )

        #define GET_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   GET_UINT8( pData, var8 )

        #define GET_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn) \
   GET_UINT16( pData, var16 )

        #define GET_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn) \
   GET_UINT32( pData, var32 )

    #else

        #define READ_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   READ_UINT8( pData, var8 );                                      \
   if (!(condn)) { dbgErrActn; }

        #define READ_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn) \
   READ_UINT16( pData, var16 );                                    \
   if (!(condn)) { dbgErrActn; }

        #define READ_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn) \
   READ_UINT32( pData, var32 );                                    \
   if (!(condn)) { dbgErrActn; }

        #define ADV_UINT8_L2CHK     READ_UINT8_L2CHK
        #define ADV_UINT16_L2CHK    READ_UINT16_L2CHK
        #define ADV_UINT32_L2CHK    READ_UINT32_L2CHK

        #define GET_UINT8_L2CHK( pData, var8, condn, dbgErrActn, relErrActn ) \
   GET_UINT8( pData, var8 );                                       \
   if (!(condn)) { dbgErrActn; }

        #define GET_UINT16_L2CHK( pData, var16, condn, dbgErrActn, relErrActn) \
   GET_UINT16( pData, var16 );                                     \
   if (!(condn)) { dbgErrActn; }

        #define GET_UINT32_L2CHK( pData, var32, condn, dbgErrActn, relErrActn) \
   GET_UINT32( pData, var32 );                                     \
   if (!(condn)) { dbgErrActn; }

    #endif
#endif




/*
-- These macros provide standardised reading of 8/16/32bit values from
-- contiguous memory (assuming Big endian data)
--
-- 'GET' macros 'return' the value and do not change the data pointer
--
-- 'READ' macros write the value into a parameter and increment the data
-- pointer by the data length
*/

/*
-- pData type is U8BIT*, opVar type is U8BIT
-- On entry - pData->MSByte
-- On exit  - pData->MSByte
*/
#define GET_UINT8_C( pData, opVar )    opVar = *(pData)


/*
-- pData type is U8BIT*, opVar type is U16BIT
-- On entry - pData->MSByte
-- On exit  - pData->MSByte
*/
#define GET_UINT16_C( pData, opVar )                        \
   {                                                       \
      U16BIT result;                                      \
                                                            \
      result = (U16BIT)*(pData);                          \
      opVar = (U16BIT)((result << 8) | *((pData) + 1));  \
   }


/*
-- pData type = U8BIT*, opVar type = U16BIT
-- On entry - pData->MSByte
-- On exit  - pData->MSByte
*/
#define GET_UINT32_C( pData, opVar )                        \
   {                                                       \
      U32BIT result;                                      \
                                                            \
      result = (U32BIT)*(pData);                          \
      result = (U32BIT)((result << 8) | *((pData) + 1));  \
      result = (U32BIT)((result << 8) | *((pData) + 2));  \
      opVar = (U32BIT)((result << 8) | *((pData) + 3));  \
   }


/*
-- pData type is U8BIT*, opVar type is U8BIT
-- On entry - pData->MSByte
-- On exit  - pData += 1
*/
#define READ_UINT8_C( pData, opVar )    (opVar = *(pData)++)


/*
-- pData type is U8BIT*, opVar type is U16BIT
-- On entry - pData->MSByte
-- On exit  - pData += 2
*/
#define READ_UINT16_C( pData, opVar )                   \
   {                                                   \
      U16BIT result;                                  \
                                                        \
      result = (U16BIT)*(pData)++;                    \
      opVar = (U16BIT)((result << 8) | *(pData)++);  \
   }


/*
-- pData type = U8BIT*, opVar type = U32BIT
-- On entry - pData->MSByte
-- On exit  - pData += 3
*/
#define READ_UINT24_C( pData, opVar )                   \
   {                                                   \
      U32BIT result;                                  \
                                                        \
      result = (U32BIT)*(pData)++;                    \
      result = (U32BIT)((result << 8) | *(pData)++);  \
      opVar = (U32BIT)((result << 8) | *(pData)++);  \
   }

/*
-- pData type = U8BIT*, opVar type = U16BIT
-- On entry - pData->MSByte
-- On exit  - pData += 4
*/
#define READ_UINT32_C( pData, opVar )                   \
   {                                                   \
      U32BIT result;                                  \
                                                        \
      result = (U32BIT)*(pData)++;                    \
      result = (U32BIT)((result << 8) | *(pData)++);  \
      result = (U32BIT)((result << 8) | *(pData)++);  \
      opVar = (U32BIT)((result << 8) | *(pData)++);  \
   }


/*
-- This macro provides standardised reading of object key values from
-- contiguous memory (assuming Big endian data).
--
-- Uses switch to make it fast (& avoid loop or function call overhead
-- of memcpy). Currently only used in two places so code overhead
-- not a problem.
--
-- pData type = U8BIT*, objKey type = S_ObjectKey, valid type = BOOLEAN
-- On entry - pData -> objectKeyLength field
-- On exit  - If valid: pData += objectKeyLength + 1
--            Else:     pData set back to initial position
*/
#define READ_OBJECT_KEY_C( pData, objKey, valid )               \
   {                                                           \
      U8BIT objKeyLen;                                      \
      U8BIT *pResultData = objKey.data;                      \
                                                                \
      objKeyLen = *(pData)++;                                 \
                                                                \
      valid = TRUE;                                           \
      objKey.length = objKeyLen;                              \
                                                                \
      switch (objKeyLen) {                                    \
                                                                \
         case 1:                                             \
            *pResultData = *(pData)++;                      \
            break;                                              \
                                                                \
         case 2:                                             \
            *(pResultData)++ = *(pData)++;                  \
            *pResultData = *(pData)++;                      \
            break;                                              \
                                                                \
         case 3:                                             \
            *(pResultData)++ = *(pData)++;                  \
            *(pResultData)++ = *(pData)++;                  \
            *pResultData = *(pData)++;                      \
            break;                                              \
                                                                \
         case 4:                                             \
            *(pResultData)++ = *(pData)++;                  \
            *(pResultData)++ = *(pData)++;                  \
            *(pResultData)++ = *(pData)++;                  \
            *pResultData = *(pData)++;                      \
            break;                                              \
                                                                \
         default:                                            \
            objKey.length = 0;                              \
            pData -= 1;                                     \
            valid = FALSE;                                  \
            break;                                              \
      }                                                       \
   }

#define OUI_SPECIFIER_TYPE       0x01000000
#define OUI_SPECIFIER_MASK       0xFF000000
#define OUI_DATA_MASK            0x00FFFFFF

/*------------------------------  Exported Types  ----------------------------*/

typedef struct s_modelVersion
{
   U16BIT model;
   U16BIT version;
} S_modelVersion;

typedef struct s_compatibility
{
   U16BIT hw_num;
   U16BIT sw_num;
   S_modelVersion *hwlist;
   S_modelVersion *swlist;
} S_Compatibility, *P_Compatibility;

/*------------------------------  Exported Data  -----------------------------*/


/*---------------------------  Exported Prototypes  --------------------------*/


E_DsmObjectKind convertObjectKindStr( U32BIT objectKindStr );

E_DscError handleInLoopError( P_DsmCoreInst idp,
   E_DscError currErr, E_DscError newErr );

void readUInt16Seq( MemSeqRef memAreaRef, U16BIT *pUi16 );

void readUInt32Seq( MemSeqRef memAreaRef, U32BIT *pUi32 );

void getUInt8Seq( MemSeqRef memAreaRef, U8BIT *pUi8 );

void getUInt16Seq( MemSeqRef memAreaRef, U16BIT *pUi16 );

void getUInt32Seq( MemSeqRef memAreaRef, U32BIT *pUi32 );

void readObjectKeySeq( MemSeqRef memAreaRef, P_ObjectKey pObjectKey,
   BOOLEAN *pValid );


/* -- CONTIGUOUS MEMORY DATA ACCESS FUNCTIONS */

#include "defMemUtilsContig.h"  /* -- Default mem type for functions */

U16BIT getIorInfoContig( const MemPtr mpIorData, U32BIT *pIorTypeId,
   P_ObjectLocation pLocation, P_DeliveryParaTap pTap );

P_Compatibility DSC_UtilParseCompatibilityDesc( P_DsmCoreInst idp,
   MemPtr pData, U16BIT length, U32BIT specifier );

BOOLEAN DSC_UtilCheckCompatibility( U8BIT *pData, U16BIT len,
   U32BIT specifier, S_SsuModelVersion smv );

#ifdef MEM_CONTIGUOUS
 #define getIorInfoSeq getIorInfoContig
#else

/* -- MANAGED MEMORY DATA ACCESS FUNCTIONS */

#include "defMemUtilsMgd.h"  /* -- Default mem type for functions */

U16BIT getIorInfoSeq( const MemPtr mpIorData, U32BIT *pIorTypeId,
   P_ObjectLocation pLocation, P_DeliveryParaTap pTap );

#endif /*!MEM_CONTIGUOUS*/

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif
#endif /* _CLDSMUTILS_H_ */
