/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions/methods for creating/destroying and managing attributes of
 *             DSM-CC data carousels.
 * @file    dataCarousel.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "dataCarousel.h"

#include "module.h"
#include "cacheMgr.h"
#include "clDsmUtils.h"
#include "sectionFilter.h"
#include "sectionTimer.h"
#include "rootCarousel.h"

/*------------------------------- Local Macros -------------------------------*/



/*------------------------------  Exported Data  -----------------------------*/



/*--------------------------------Local Types --------------------------------*/



/*------------------------------- Local Statics ------------------------------*/



/*------------------- local prototypes/forward declarations ------------------*/

static E_DscError UpdateDiiModules( P_DsmCoreInst idp, P_DataCarousel pDC );
static E_DscError FetchDiiModules( P_DsmCoreInst idp, P_DataCarousel pDC );
static E_DscError FetchSsuModules( P_DsmCoreInst idp, P_DataCarousel pDC );
static E_DscError StartNextSsuDownload( P_DsmCoreInst idp, P_DataCarousel pDC );
static void AbortSsuDownload( P_DsmCoreInst idp, P_DataCarousel pDC );
static void LoadFinalise( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst );

/* -- CONTIGUOUS MEMORY DATA ACCESS FUNCTIONS */
static BOOLEAN GetDiiMsgInfoContig( P_DataCarousel pDC, const U8BIT *pDiiMsg );
static BOOLEAN dcFindModuleInDiiMsgContig( const MemPtr pDiiMsg, U32BIT dataLength,
    U16BIT moduleId, MemPtr *pModuleInfoDescStart );
static BOOLEAN dcGetModuleInfoContig( const MemPtr pModuleInfoDescStart,
   S_ModuleInfo *pModuleInfo );
static void dcGetFirstModuleDescContig( const MemPtr mpDiiMsg, MemPtr *mpModuleInfoDescStart );
static void dcGetNextModuleDescContig( const MemPtr mpCurrModuleDescStart, MemPtr *mpNextModuleDescStart );

#ifdef MEM_CONTIGUOUS

#define dcFindModuleInDiiMsgSeq dcFindModuleInDiiMsgContig
#define dcGetModuleInfoSeq dcGetModuleInfoContig

#else

/* -- MANAGED MEMORY DATA ACCESS FUNCTIONS */
static BOOLEAN dcFindModuleInDiiMsgSeq( const MemPtr mpDiiMsg, U32BIT dataLength,
   U16BIT moduleId, MemPtr *mpModuleInfoDescStart );
static BOOLEAN dcGetModuleInfoSeq( const MemPtr mpModuleInfoDescStart, S_ModuleInfo *pModuleInfo );

#endif

/*---------------------------- Exported Functions ----------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// DSC_DataCrslCreate
// Creates an instance of the DataCarousel struct and initialises it.
///////////////////////////////////////////////////////////////////////////// */
E_DscError DSC_DataCrslCreate( P_DsmCoreInst idp, P_RootCarousel pRC,
   P_DeliveryParaTap pTap, P_DataCarousel *ppDataCarousel )
{
   P_DataCarousel pDC = NULL;
   E_DscError err;

   dsmDP3(("DSC_DataCrslCreate()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pTap != NULL));
   dsmAssert((ppDataCarousel != NULL));

   pDC = (P_DataCarousel)DSC_CmMemGet( idp, sizeof(S_DataCarousel) );
   if (!pDC)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      err = CLDSM_OK;
      llLinkInit( pDC->llData, NUM_LISTS_DATA_CAROUSEL );

      pDC->dataCarouselId =
         (U16BIT)(pTap->transactionId & TRANSACTION_ID_IDENT_MASK);

      pDC->transIdver = INVALID_TRANSIDVER;
      pDC->diiCrc = 0;
      pDC->tap = *pTap;
      pDC->pDiiSf = NULL;
      pDC->diiRequestCount = 0;
      pDC->diiMonitorCount = 0;

      pDC->diiInfo.blockSize = 0;
      pDC->diiInfo.numberOfModules = 0;

      pDC->diiMsgDataLen = 0;
      pDC->hDiiMsgData = NULL;
      pDC->llcLoadRequests = NULL;

      /* -- Create the module list header block */
      err = LLCreate( idp, pDC, DC_MODULE_LIST, &pDC->llcDcModules );
      dsmDP4(("%x pDiiSf=%x\n", err, pDC->pDiiSf));

      if (err)
      {
         DSC_CmMemRelease( idp, pDC );
         *ppDataCarousel = NULL;
      }
      else
      {
         err = LLCreate( idp, pDC, MODULE_ACQUIRE_LIST, &pDC->llcModuleAcquires );
         if (err)
         {
            LLDestroy( idp, &(pDC->llcDcModules));
            DSC_CmMemRelease( idp, pDC );
            *ppDataCarousel = NULL;
         }
         else
         {
            DSC_RootCrslAddDataCarousel( pRC, pDC );
            *ppDataCarousel = pDC;
         }
      }
   }

   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: DSC_DataCrslCreate: %u\n", err)));
   dsmDP3(("exit DSC_DataCrslCreate -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_DataCrslDestroy
// Destroys an instance of the DataCarousel struct, and all associated data.
///////////////////////////////////////////////////////////////////////////// */
void DSC_DataCrslDestroy( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   dsmAssert((idp != NULL));
   dsmAssert((pDC != NULL));

   /* -- Should not be in acquire list */
   dsmAssert((pDC->llData[OC_DII_ACQUIRE_LIST].pLLCtrl == NULL));

   /* -- Remove from object carousel DC list */
   LLRemove( pDC, OC_DATA_CAROUSEL_LIST );

   /* -- Should not have a load request */
   dsmAssert((pDC->llcLoadRequests == NULL));
   /* -- Should not have a section filter assigned */
   dsmAssert((pDC->pDiiSf == NULL));
   /* -- Should still have a module list ctrl block */
   dsmAssert((pDC->llcDcModules != NULL));
   /* -- Should still have a module list ctrl block */
   dsmAssert((pDC->llcModuleAcquires != NULL));

    #ifndef NDEBUG
   {
      int listCount;
      /* -- module list should be empty */
      listCount = LLCount( pDC->llcDcModules );
      dsmAssert((listCount == 0));
   }
    #endif

   LLDestroy( idp, &(pDC->llcDcModules));
   LLDestroy( idp, &(pDC->llcModuleAcquires));

   if (pDC->hDiiMsgData)
   {
      DSC_CmMemRelease( idp, pDC->hDiiMsgData );
      pDC->hDiiMsgData = NULL;
   }

   DSC_CmMemRelease( idp, pDC );

   dsmDP3(("exit DSC_DataCrslDestroy\n"));
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_DataCrslDelete
//
///////////////////////////////////////////////////////////////////////////// */
void DSC_DataCrslDelete( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   P_Module pModule;
   P_Module pPrev;
   P_RootLoadRqst pLR;

   dsmDP3(("DSC_DataCrslDelete()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pDC != NULL));

   DBGLOG(DD_DC, "aqc %d,%d", pDC->diiRequestCount, pDC->diiMonitorCount)

   while (pDC->llcLoadRequests != NULL)
   {
      pLR = LLRemoveTail( pDC->llcLoadRequests );
      if (pLR == NULL)
      {
         LLDestroy( idp, &pDC->llcLoadRequests );
         break;
      }
      dsmAssert((pLR->status != LRS_ABORT_PENDING_RELOAD));
      DSC_LoadRsqtAbort( idp, pLR );
      /* When finalise function is LoadFinalise, DSC_LoadRsqtAbort() ends up
       * calling DSC_DataCrslCheckSsuStatus(), and if it is the only remaining
       * load request for the data carousel, then pDC->llcLoadRequests will
       * have been destroyed and set to NULL
       */
   }

   /* -- Delete the attached module(s) */
   pModule = LLTail( pDC->llcDcModules );
   /* -- Should be at-least one module */
   //    dsmAssert(( pModule != NULL ));
   while (pModule != NULL)
   {
      pPrev = LLPrev( pModule, DC_MODULE_LIST );
      DSC_ModuleDelete( idp, pModule );
      pModule = pPrev;
   }

   /* -- Clear any existing DII acquisition requests */
   if ((pDC->diiRequestCount != 0) || (pDC->diiMonitorCount != 0))
   {
      LLRemove( pDC, OC_DII_ACQUIRE_LIST );
      if (pDC->pDiiSf != NULL)
      {
         DBGLOG(DD_SF, "pDiiSf=%x", pDC->pDiiSf)
         DSC_SectionFilterStop( idp, &pDC->pDiiSf );
      }
   }

   DSC_DataCrslDestroy( idp, pDC );

   dsmDP3(("exit DSC_DataCrslDelete\n"));
}

E_DscError DSC_DataCrslAcquireStart( P_DsmCoreInst idp, P_DataCarousel pDC, E_SFPriority sfPriority )
{
   S_SfTarget target;
   E_DscError err;
   P_RootCarousel pRC;
   pRC = (P_RootCarousel)LLParent(pDC, OC_DATA_CAROUSEL_LIST);
   if (LLInsertHead( pRC->llcDiiAcquires, pDC ))
   {
      target.kind = SFK_DII;
      target.id = pDC->dataCarouselId;
      target.associationTag = pDC->tap.associationTag;
      target.serviceId = (target.associationTag == INVALID_ASSOCIATION_TAG) ?
         DSC_RootCrslGetPid(pRC) : DSC_RootCrslGetServiceId(pRC);
      target.u.pDC = pDC;
      dsmDP3(("INFO: Load Data Carousel (DII) Id: %u\n", (pDC->dataCarouselId >> 1)));
      /* -- DC/DII not already being acquired, so start section filter */
      err = DSC_SectionFilterStart( idp, &target, sfPriority, &pDC->pDiiSf );
      if (err == CLDSM_ERR_SI_QUERY_FAILED)
      {
         DBGLOG((DD_SF|DD_DC), " Treating query failure as OK")
         err = CLDSM_OK;
      }
   }
   else
   {
      dsmAssert((pDC->pDiiSf));
      /* -- DC/DII already being acquired so update filter priority */
      err = DSC_SsectionFilterUpdatePriority( idp, pDC->pDiiSf, sfPriority, /*latchHighest*/ FALSE );
   }
   if (err)
   {
      ERRPRINT(" err=%d", err)
      LLRemove( pDC, OC_DII_ACQUIRE_LIST );
   }
   else if (sfPriority == SF_PRIORITY_HIGH)
   {
      pDC->diiRequestCount++;
      DBGLOG((DD_SF|DD_DC), "HIGH priority cnt %d sf %p",pDC->diiRequestCount,pDC->pDiiSf)
   }
   else
   {
      pDC->diiMonitorCount++;
      DBGLOG((DD_SF|DD_DC), "LOW priority cnt %d sf %p",pDC->diiMonitorCount,pDC->pDiiSf)
   }
   return err;
}

E_DscError DSC_DataCrslAcquireRestart( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   S_SfTarget target;
   E_DscError err;
   E_SFPriority sfPriority;

   if (pDC->pDiiSf != NULL)
   {
      target = pDC->pDiiSf->target;
      sfPriority = pDC->pDiiSf->filter.priority;
      DSC_SectionFilterStop( idp, &pDC->pDiiSf );
   }
   else
   {
      P_RootCarousel pRC = (P_RootCarousel)LLParent(pDC, OC_DATA_CAROUSEL_LIST);
      target.kind = SFK_DII;
      target.id = pDC->dataCarouselId;
      target.associationTag = pDC->tap.associationTag;
      target.serviceId = (target.associationTag == INVALID_ASSOCIATION_TAG) ?
         DSC_RootCrslGetPid(pRC) : DSC_RootCrslGetServiceId(pRC);
      target.u.pDC = pDC;
      sfPriority = (pDC->diiRequestCount)? SF_PRIORITY_HIGH : SF_PRIORITY_LOW;
   }
   err = DSC_SectionFilterStart( idp, &target, sfPriority, &pDC->pDiiSf );
   DBGLOG((DD_SF|DD_DC), " err=%d pDC->pDiiSf=%x", err, pDC->pDiiSf)
   return err;
}

void DSC_DataCrslAcquireStop( P_DsmCoreInst idp, P_DataCarousel pDC, E_SFPriority sfPriority )
{
   if (sfPriority == SF_PRIORITY_HIGH)
   {
      if (pDC->diiRequestCount > 0)
         pDC->diiRequestCount--;
   }
   else
   {
      if (pDC->diiMonitorCount > 0)
         pDC->diiMonitorCount--;
   }
   if ((pDC->diiRequestCount == 0) && (pDC->diiMonitorCount == 0))
   {
      LLRemove( pDC, OC_DII_ACQUIRE_LIST );
      if (pDC->pDiiSf != NULL)
      {
         DBGLOG(DD_SF, "pDiiSf=%x", pDC->pDiiSf)
         DSC_SectionFilterStop( idp, &pDC->pDiiSf );
      }
   }
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_DataCrslListFindById
// Find a dataCarousel in a dataCarouselList by it's transactionId (ID part)
// Returns NULL in phDataCarousel if not found.
///////////////////////////////////////////////////////////////////////////// */
P_DataCarousel DSC_DataCrslListFindById( P_LLControl pDcList, U32BIT transactionId )
{
   U16BIT searchDcId = (U16BIT)(transactionId & TRANSACTION_ID_IDENT_MASK);
   P_DataCarousel pDcFromList = NULL;
   ListId_t listId;

   dsmDP3(("DSC_DataCrslListFindById()\n"));

   /* Get listId and first DC in list from Control block */
   listId = LListId( pDcList );
   pDcFromList = (P_DataCarousel)LLHead( pDcList );

   /* Go through each DC in list until the one with a matching transactionId
    is found */
   while (pDcFromList)
   {
      /* Is this the DC we want? */
      if (pDcFromList->dataCarouselId == searchDcId)
      {
         /* Found the DC */
         break;
      }
      pDcFromList = LLNext( pDcFromList, listId );
   }
   dsmDP3(("exit DSC_DataCrslListFindById: %p\n", pDcFromList));
   return pDcFromList;
}

U16BIT DSC_DataCrslGetServiceId( P_DataCarousel pDC )
{
   return DSC_RootCrslGetServiceId( LLParent(pDC, OC_DATA_CAROUSEL_LIST));
}

U16BIT DSC_DataCrslGetPid( P_DataCarousel pDC )
{
   return DSC_RootCrslGetPid( LLParent(pDC, OC_DATA_CAROUSEL_LIST));
}

P_Module DSC_DataCrslFirstModule( P_DataCarousel pDC )
{
   return LLHead( pDC->llcDcModules );
}

void DSC_DataCrslUnloadModule( P_DsmCoreInst idp, P_DataCarousel pDC,
   U16BIT moduleId )
{
   P_Module pModule;
   pModule = LLHead( pDC->llcDcModules );
   while ( pModule != NULL )
   {
      if (pModule->moduleInfo.moduleId == moduleId)
      {
         dsmAssert((pModule->status == MS_CACHED));
         dsmAssert((pModule->hModuleData != NULL));
         /* finished with module - delete it */
         DSC_ModuleDelete( idp, pModule );
         break;
      }
      pModule = LLNext( pModule, DC_MODULE_LIST );
   }
}


E_DscError DSC_DataCrslNewModuleLoad( P_DsmCoreInst idp, P_DataCarousel pDC, P_Module pModule )
{
   E_DscError err;
   MemPtr mpDiiMsg;
   MemPtr mpStart;

   dsmAssert((pModule->status == MS_INITIAL));
   if (pDC->hDiiMsgData != NULL)
   {
      MEMPTR_SEQ_OPEN( MEM_CONTEXT, pDC->hDiiMsgData, 0, pDC->diiMsgDataLen, FALSE, mpDiiMsg );
      MEMPTR_OPEN( mpDiiMsg, mpStart );

      pModule->moduleInfo.crslMagic = DSC_RootCrslMagic(LLParent(pDC, OC_DATA_CAROUSEL_LIST));

      if ( dcFindModuleInDiiMsgSeq( mpDiiMsg, pDC->diiMsgDataLen, pModule->moduleInfo.moduleId, &mpStart ) &&
           dcGetModuleInfoSeq( mpStart, &pModule->moduleInfo ) )
      {
         /* -- Add monitor on DII in-case version change occurs */
         err = DSC_DataCrslAcquireStart( idp, pDC, SF_PRIORITY_LOW );
         if (!err)
         {
            pModule->moduleInfo.blockSize = pDC->diiInfo.blockSize;
            pModule->status = MS_PENDING_DDB;
         }
         else
         {
            /* -- First request for new module so abort module on failure */
            DSC_ModuleDeleteDcTidyUp( idp, pModule );
         }
         MEMPTR_CLOSE( mpStart );
         MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pDC->hDiiMsgData, mpDiiMsg );
      }
      else
      {
         DBGLOG(DD_DC," ModuleId %u not found in STORED DII", pModule->moduleInfo.moduleId)
         /* This may be an erroneous load (broadcast error) or
         -- it may be a request for a new module for which
         -- the updated DII has not yet arrived.
         -- In-case it is the latter, force DII to be
         -- re-acquired by destroying 'erroneous' DII message
         -- data store.
         -- Module stays at status MS_INITIAL so DII is
         -- re-fetched (see below).
         -- NB. DII is only re-fetched once. If module still
         -- cannot be found when new DII arrives the load will
         -- be aborted
         */
         MEMPTR_CLOSE( mpStart );
         MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pDC->hDiiMsgData, mpDiiMsg );
         DSC_CmMemRelease( idp, pDC->hDiiMsgData );
         pDC->hDiiMsgData = NULL;
         pDC->diiMsgDataLen = 0;
         pDC->transIdver = INVALID_TRANSIDVER;
         pDC->diiCrc = 0;
         err = DSC_DataCrslAcquireStart( idp, pDC, SF_PRIORITY_HIGH );
         if (!err)
         {
            pModule->status = MS_PENDING_INFO;
         }
         else
         {
            DSC_ModuleDeleteDcTidyUp( idp, pModule );
         }
      }
   }
   else /* if (pDC->hDiiMsgData == NULL) So no DII message data store */
   {
      /* First request for new module so request DII */
      err = DSC_DataCrslAcquireStart( idp, pDC, SF_PRIORITY_HIGH );
      if (!err)
      {
         pModule->status = MS_PENDING_INFO;
      }
      else
      {
         DSC_ModuleDeleteDcTidyUp( idp, pModule );
      }
   }
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
//
// Called when a new DII message arrives to supply info for new module
// builds and check current modules for version updates.
//
// If turboCaching enabled, also creates and requests (loads) any modules
// not already in the cache.
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError DSC_DataCrslUpdate( P_DsmCoreInst idp, P_DataCarousel pDC, U32BIT transactionId,
   U8BIT *pDiiMsg, U16BIT diiMsgDataLen )
{
   E_DscError err = CLDSM_OK;
   U32BIT diiCrc;
   MemPtr mpDiiMsgData;
   U16BIT transIdver;

   dsmAssert((idp != NULL));
   dsmAssert((pDC != NULL));
   dsmAssert((pDiiMsg != NULL));

   sectionTimerRemove(idp, pDC->pDiiSf);

   diiCrc = CDSM_UtilCalculateCRC(pDiiMsg, diiMsgDataLen);
   transIdver = (transactionId & TRANSACTION_ID_VERSION_MASK) >> 16;

   if (pDC->diiRequestCount ||
       (pDC->diiMonitorCount && (transIdver != pDC->transIdver || pDC->diiCrc != diiCrc)))
   {
      if (pDC->hDiiMsgData != NULL)
      {
         DSC_CmMemRelease( idp, pDC->hDiiMsgData );
         pDC->hDiiMsgData = NULL;
      }
      if (GetDiiMsgInfoContig( pDC, pDiiMsg ))
      {
         if (pDC->diiInfo.numberOfModules == 0)
         {
            DSC_DataCrslDelete( idp, pDC );
         }
         else
         {
            pDC->hDiiMsgData = DSC_CmMemGet( idp, diiMsgDataLen );
            if (pDC->hDiiMsgData == NULL)
            {
               err = CLDSM_ERR_MEM_HEAP_FULL;
            }
            else
            {
               /* -- Store new DII message data and info store */
               MEMPTR_SEQ_OPEN( MEM_CONTEXT, pDC->hDiiMsgData, 0,
                  diiMsgDataLen, FALSE, mpDiiMsgData );
               MEMPTR_WRITE( pDiiMsg, mpDiiMsgData, diiMsgDataLen );
               MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pDC->hDiiMsgData, mpDiiMsgData );
               pDC->diiMsgDataLen = diiMsgDataLen;

               if (DSC_RootCrslMagic(LLParent(pDC, OC_DATA_CAROUSEL_LIST)) == UC_MAGIC)
               {
                  if (pDC->diiCrc != diiCrc || transIdver != pDC->transIdver)
                  {
                     if (pDC->diiCrc != 0)
                     {
                        /* This should never happen with SSU
                         - a DII change while downloading modules */
                        AbortSsuDownload( idp, pDC );
                     }
                     err = FetchSsuModules( idp, pDC );
                  }
               }
               else
               {
                  err = UpdateDiiModules( idp, pDC );
                  if (!err && idp->setup.turboCaching && !idp->cacheFull)
                  {
                     err = FetchDiiModules( idp, pDC );
                  }
               }
               pDC->transIdver = transIdver;
               pDC->diiCrc = diiCrc;
            }
         }
      }
   }
   DBGERRCHK(err)
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// updateCachedModuleVersion
//
///////////////////////////////////////////////////////////////////////////// */
static E_DscError UpdateCachedModuleVersion( P_DsmCoreInst idp,
   P_DataCarousel pDC, P_Module pModule, S_ModuleInfo *pModuleInfo )
{
   P_Module pNewModule;
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));
   dsmAssert((pModuleInfo != NULL));
   dsmAssert(((pModule->status == MS_CACHED) ||
              (pModule->status == MS_ACQ_ABORTED)));

   if (pModule->loadedCount > 0)
   {
      if (DSC_DataCrslFirstModule(pDC) == pModule)
      {
         DSC_RootCrslSrgObjectReset(LLParent(pDC, OC_DATA_CAROUSEL_LIST));
      }

      /* -- Remove old module from loaded list */
      LLRemove( pModule, MODULE_LOADED_LIST );

      /* -- Create new module version */
      err = DSC_ModuleCreate( idp, pModule->moduleInfo.moduleId, &pNewModule );
      if (!err)
      {
         /* -- Set new module version info */
         pNewModule->moduleInfo = *pModuleInfo;
         pNewModule->status = MS_PENDING_DDB;

         /* -- Insert new module in place of old module in all lists
            -- of which it is a member */
         LLReplaceAll( pModule, pNewModule, NUM_LISTS_MODULE );

         err = DSC_ModuleAcquireStart( idp, pNewModule, SF_PRIORITY_DIRECT );
         if (err)
         {
            DSC_ModuleDelete( idp, pNewModule );
         }
      }
      /* -- Add old module to delete list */
      pModule->status = MS_EXPIRED;
      LLInsertHead( idp->llcExpiredModules, pModule );
   }
   else     /* -- Module not loaded (ie. not in use) */
   {    /* -- Reset module state and update with new version info */
      DSC_ModuleResetState( idp, pModule, MS_PENDING_INFO );
      pModule->moduleInfo = *pModuleInfo;
      pModule->status = MS_PENDING_DDB;

      if (DSC_DataCrslFirstModule(pDC) == pModule &&
          DSC_RootCrslSrgObjectReset(LLParent(pDC, OC_DATA_CAROUSEL_LIST)) == CLDSM_OK)
      {
         err = DSC_ModuleAcquireStart( idp, pModule, SF_PRIORITY_DIRECT );
      }
      else
      {
         err = DSC_ModuleAcquireStart( idp, pModule, SF_PRIORITY_CLEAR );
      }
      if (err)
      {
         DSC_ModuleDelete( idp, pModule );
      }
   }
   return err;
}

/*
-- Update a Data Carousel module with new module info
*/
static E_DscError UpdateDiiModule( P_DsmCoreInst idp,
   P_DataCarousel pDC, P_ModuleInfo pModuleInfo, P_Module pModule )
{
   E_DscError err = CLDSM_OK;

   dsmAssert((idp != NULL));
   dsmAssert((pDC != NULL));
   dsmAssert((pModuleInfo != NULL));
   dsmAssert((pModule != NULL));

   if (pModule->status == MS_PENDING_INFO)
   {
      pModule->moduleInfo = *pModuleInfo;
      pModule->status = MS_PENDING_DDB;

      /* -- Add monitor on DII in-case version change occurs (and remove
         -- request for DII) */
      /* -- NB. Call in this order to prevent unneccessary delete/add of
         -- sectionFilter (nb. add call cannot fail since DII section filter
         -- must already be requested so can ignore returns) */
      err = DSC_DataCrslAcquireStart( idp, pDC, SF_PRIORITY_LOW );
      dsmAssert((err == CLDSM_OK));
      DSC_DataCrslAcquireStop( idp, pDC, SF_PRIORITY_HIGH );

      /* -- Now ready to acquire module data. */
      /* -- Set high SF priority if this module has client load requests
         -- otherwise set low SF priority (ie. it is an internal module
         -- pre-fetch) */
      err = DSC_ModuleAcquireStart( idp, pModule, DSC_ModulePriority(pModule));
      if (err)
      {
         DSC_ModuleDelete( idp, pModule );
      }
   }
   else /* -- Module status != MS_PENDING_INFO */
   {
      dsmAssert((pModule->status != MS_INITIAL));
      dsmAssert((pModule->status != MS_EXPIRED));

      if (pModule->moduleInfo.version != pModuleInfo->version ||
          pModule->moduleInfo.moduleSize != pModuleInfo->moduleSize)
      {
         dsmDP4(("Module Id: %u version %u,%u status=%x",
                 pModule->moduleInfo.moduleId, pModule->moduleInfo.version, pModuleInfo->version, pModule->status));

         switch (pModule->status)
         {
            case MS_PENDING_DDB:
            case MS_BUILDING:
            {
               dsmDP4(("(acquiring)\n"));
               /* -- Module acquisition is in progress so reset and
                  -- continue with new module info */
               DSC_ModuleResetState( idp, pModule, MS_PENDING_INFO );
               pModule->moduleInfo = *pModuleInfo;
               pModule->status = MS_PENDING_DDB;
               break;
            }
            case MS_ACQ_ABORTED:
            {
               dsmDP4(("(acquiring)\n"));
               /*
               -- This module was previously aborted due to decompression
               -- failures. Module is now updated so abort or re-acquire
               -- (depending on whether turbo-caching enabled).
               */
               /*fall thro*/
            }
            case MS_CACHED:
            {
               if (idp->setup.turboCaching == TRUE)
               {
                  /* -- Update out-of-date module */
                  /* -- NB. ppModule value may get changed */
                  err = UpdateCachedModuleVersion( idp, pDC, pModule, pModuleInfo );
               }
               else
               {
                  DSC_ModuleDelete( idp, pModule );
               }
               break;
            }
            default:
            {
               /* -- Illegal status */
               dsmDP1(("ERROR: Illegal module status = %u\n", pModule->status));
               dsmAssert((0));
               /* -- Set flag to force module abort */
               err = CLDSM_ERR_INTERNAL;
               DSC_ModuleDelete( idp, pModule );
               break;
            }
         }
      }
   }
   DBGERRCHK(err)
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
//
// Called when a new DII message arrives to supply info for new module
// builds and check current modules for version updates.
//
// If turboCaching enabled, also creates and requests (loads) any modules
// not already in the cache.
//
///////////////////////////////////////////////////////////////////////////// */
static E_DscError UpdateDiiModules( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   P_Module pModule;
   P_Module pNextModule;
   U8BIT *pModuleDescStart;
   S_ModuleInfo moduleInfo;
   E_DscError err = CLDSM_OK;
   E_DscError newErr;

   dsmAssert((idp != NULL));
   dsmAssert((pDC != NULL));

   /* -- Module list should always be present */
   dsmAssert((pDC->llcDcModules != NULL));

   moduleInfo.crslMagic = DSC_RootCrslMagic(LLParent(pDC, OC_DATA_CAROUSEL_LIST));

   /* -- Get first module in dataCarousel */
   pModule = LLHead( pDC->llcDcModules );

   /* -- Update all relevant modules in dataCarousel */
   while (pModule != NULL)
   {
      /* -- Determine next module in list before handling module
         -- update since a version update may change the pModule
         -- value (actually should be OK because 'new'
         -- module should be in same position in all lists) */
      pNextModule = LLNext( pModule, DC_MODULE_LIST );

      dsmAssert((pModule->status != MS_INITIAL));
      dsmAssert((pModule->status != MS_EXPIRED));

      moduleInfo.blockSize = pDC->diiInfo.blockSize;
      moduleInfo.associationTag = pDC->tap.associationTag;

      /* -- NB. Use contiguous memory copy of DII here for speed */
      if (dcFindModuleInDiiMsgContig( pDC->hDiiMsgData, pDC->diiMsgDataLen,
             pModule->moduleInfo.moduleId, &pModuleDescStart ) &&
          dcGetModuleInfoContig( pModuleDescStart, &moduleInfo ))
      {
         /* -- NB. pModule value may get changed */
         newErr = UpdateDiiModule( idp, pDC, &moduleInfo, pModule );
         if (newErr)
         {
            err = handleInLoopError( idp, err, newErr );
         }
      }
      else
      {
         dsmDP3(("INFO: Module Id: %u not found or not valid in RECEIVED DII\n", pModule->moduleInfo.moduleId));
         DSC_ModuleDelete( idp, pModule );
      }
      pModule = pNextModule;
   }
   DBGERRCHK(err)
   return err;
}

/*
-- Internally prefetch a module into a Data Carousel using DII info
--
-- NB. This does nothing if the cache is full
--
*/
static E_DscError FetchModule( P_DsmCoreInst idp,
   P_DataCarousel pDC, P_ModuleInfo pModuleInfo )
{
   P_Module pModule;
   E_DscError err = CLDSM_OK;

   DBGLOG(DD_DC, " Id: %u", pModuleInfo->moduleId)
   dsmAssert((idp != NULL));
   dsmAssert((pDC != NULL));
   dsmAssert((pModuleInfo != NULL));

   dsmAssert((idp->cacheFull == FALSE));

   /* -- Create and request module */
   err = DSC_ModuleCreate( idp, pModuleInfo->moduleId, &pModule );
   if (!err)
   {
      LLInsertTail( pDC->llcDcModules, pModule );
      pModule->moduleInfo = *pModuleInfo;
      pModule->status = MS_PENDING_DDB;

      /* -- Add monitor on DII in-case version change occurs */
      err = DSC_DataCrslAcquireStart( idp, pDC, SF_PRIORITY_LOW );
      if (!err)
      {
         /* -- Now ready to acquire module data */
         err = DSC_ModuleAcquireStart( idp, pModule, SF_PRIORITY_LOW );
      }
      if (err)
      {
         DSC_ModuleDelete( idp, pModule );
      }
   }
   DBGERRCHK(err)
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
//
// Creates and requests (loads) any modules not already in the cache.
//
///////////////////////////////////////////////////////////////////////////// */
static E_DscError FetchDiiModules( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   U8BIT *pModuleDescStart;
   S_ModuleInfo moduleInfo;
   E_DscError err = CLDSM_OK;
   E_DscError newErr;
   U16BIT numModules;

   dsmDP3(("lmUpdateDataCarousel()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pDC != NULL));

   /* -- Module list should always be present */
   dsmAssert((pDC->llcDcModules != NULL));
   dsmAssert((pDC->diiInfo.numberOfModules != 0));

   /* -- Loop round all modules in DII message to see if there
      -- are new ones that need creating/requesting */
   /* -- NB. Use contiguous memory copy of DII here for speed */
   numModules = pDC->diiInfo.numberOfModules;
   /* -- Get first module descriptor in DII */
   dcGetFirstModuleDescContig( pDC->hDiiMsgData, &pModuleDescStart );

   memset( &moduleInfo, 0, sizeof(S_ModuleInfo) );
   moduleInfo.blockSize = pDC->diiInfo.blockSize;
   moduleInfo.crslMagic = OC_MAGIC;

   while (numModules--)
   {
      if (dcGetModuleInfoContig( pModuleDescStart, &moduleInfo ))
      {
         /* -- Check if this module already exists */
         if (DSC_ModuleListFindById( pDC->llcDcModules, moduleInfo.moduleId ) == NULL)
         {
            newErr = FetchModule( idp, pDC, &moduleInfo );
            if (newErr)
            {
              err = handleInLoopError( idp, err, newErr );
            }
         }
      }
      /* -- Get next module descriptor in DII */
      dcGetNextModuleDescContig( pModuleDescStart, &pModuleDescStart );
   }
   DBGERRCHK(err)
   return err;
}

static void DeleteModulesOnLoadRqst( P_DsmCoreInst idp, P_DataCarousel pDC, P_RootLoadRqst pLoadRqst )
{
   P_Module pModule, pPrev;
   pModule = LLTail( pDC->llcDcModules );
   while (pModule != NULL)
   {
      pPrev = LLPrev( pModule, DC_MODULE_LIST );
      if (pModule->pLoadRqst == pLoadRqst)
      {
         DSC_ModuleDelete( idp, pModule );
      }
      pModule = pPrev;
   }
}

static void LoadFinalise( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst )
{
   P_DataCarousel pDC;
   E_UCLoadStatus status;
   switch (pLoadRqst->status)
   {
      case LRS_INITIAL:
      {
         pDC = LLParent((P_Module)pLoadRqst->target, DC_MODULE_LIST);
         status = SSU_LOAD_ABORTED;
         break;
      }
      case LRS_STALLED_MODULE:
      {
         pDC = (P_DataCarousel)pLoadRqst->target;
         status = SSU_LOAD_STARTING;
         break;
      }
      case LRS_LOADING:
      {
         pDC = (P_DataCarousel)pLoadRqst->target;
         status = SSU_LOAD_RECEIVING;
         break;
      }  
      case LRS_LOADED:
      {
         pDC = (P_DataCarousel)pLoadRqst->target;
         status = SSU_LOAD_COMPLETE;
         dsmAssert((pLoadRqst->remaining == 0));
         if (!idp->setup.multipleSsuDownloads)
         {
            StartNextSsuDownload( idp, pDC );
         }
         pLoadRqst->target = NULL;
         break;
      }
      default:
      case LRS_ABORTED_LOAD_ERROR:
      {
         pDC = (P_DataCarousel)pLoadRqst->target;
         status = SSU_LOAD_ABORTED;
         pLoadRqst->target = NULL;
         DeleteModulesOnLoadRqst( idp, pDC, pLoadRqst );
      }
   }
   if (idp->setup.ssuFuncs.status)
   {
      U_StatusRef sr;
      sr.ur = pLoadRqst->usrRef;
      idp->setup.ssuFuncs.status( (H_DsmCarousel)LLParent(pDC, OC_DATA_CAROUSEL_LIST), status,
         sr, (SSU_LOAD_RECEIVING == status) ? pLoadRqst->percent : pLoadRqst->remaining );
   }
   DSC_DataCrslCheckSsuStatus( idp, pDC );
}

/* /////////////////////////////////////////////////////////////////////////////
//
// Creates and requests (loads) all ssu modules
//
///////////////////////////////////////////////////////////////////////////// */
static E_DscError FetchSsuModules( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   P_Module pModule;
   U8BIT *pModuleDescStart;
   S_ModuleInfo moduleInfo;
   S_LLControl llcModules;
   P_RootLoadRqst pLoadRqst;
   E_DscError err;
   U16BIT numModules;

   dsmAssert((idp != NULL));
   dsmAssert((pDC != NULL));

   /* -- Module list should always be present */
   dsmAssert((pDC->llcDcModules != NULL));
   dsmAssert((pDC->diiInfo.numberOfModules != 0));
   dsmAssert((DSC_RootCrslMagic(LLParent(pDC, OC_DATA_CAROUSEL_LIST)) == UC_MAGIC));
   dsmAssert((LLHead(pDC->llcDcModules) == NULL));
   /* First load request is for Group */
   dsmAssert((LLHead(pDC->llcLoadRequests) != NULL));

   LLCtrlBlockInit( &llcModules, DC_MODULE_LIST, pDC );
   pLoadRqst = LLHead(pDC->llcLoadRequests);
   if (pLoadRqst == NULL)
   {
      err = CLDSM_ERR_INTERNAL;
   }
   else
   {
      err = CLDSM_OK;
      /* -- Loop round all modules in DII message to see if there
         -- are new ones that need creating/requesting */
      /* -- NB. Use contiguous memory copy of DII here for speed */
      numModules = pDC->diiInfo.numberOfModules;

      /* -- Get first module descriptor in DII */
      dcGetFirstModuleDescContig( pDC->hDiiMsgData, &pModuleDescStart );

      memset( &moduleInfo, 0, sizeof(S_ModuleInfo) );
      moduleInfo.blockSize = pDC->diiInfo.blockSize;
      moduleInfo.crslMagic = UC_MAGIC;
      moduleInfo.associationTag = pDC->tap.associationTag;

      while (numModules--)
      {
         if (dcGetModuleInfoContig( pModuleDescStart, &moduleInfo ))
         {
            /* this module should not already exist */
            dsmAssert((DSC_ModuleListFindById(pDC->llcDcModules, moduleInfo.moduleId) == NULL));
            err = DSC_ModuleCreate( idp, moduleInfo.moduleId, &pModule );
            if (err)
            {
               pModule = LLRemoveTail(&llcModules);
               while (pModule != NULL)
               {
                  DSC_ModuleDestroy( idp, pModule );
                  pModule = LLRemoveTail(&llcModules);
               }
               break;
            }
            LLInsertTail( &llcModules, pModule );
            pModule->moduleInfo = moduleInfo;
            moduleInfo.u.ssup.offset += moduleInfo.originalSize;
         }
         dcGetNextModuleDescContig( pModuleDescStart, &pModuleDescStart );
      }
      pLoadRqst->remaining = moduleInfo.u.ssup.offset;
   }
   if (!err)
   {
      pModule = LLHead(&llcModules);
      while (pModule != NULL && !err)
      {
         switch ( pModule->moduleInfo.u.ssup.positionType )
         {
            case PTYP_INIT:
            {
               LLRemove( pModule, DC_MODULE_LIST );
               LLInsertTail( pDC->llcDcModules, pModule );
               err = DSC_LoadRsqtCreate( idp, sizeof(S_RootLoadRqst), TT_SSU_MODULE, pModule,
                     LoadFinalise, &pLoadRqst );
               if (!err)
               {
                  LLInsertTail( pDC->llcLoadRequests, pLoadRqst );
                  pModule->moduleInfo.u.ssup.offset = 0;
                  pModule->pLoadRqst = pLoadRqst;
                  pLoadRqst->remaining = pModule->moduleInfo.originalSize;
               }
               pModule = LLHead(&llcModules);
               break;
            }
            case PTYP_FIRST:
            {
               LLRemove( pModule, DC_MODULE_LIST );
               LLInsertTail( pDC->llcDcModules, pModule );
               err = DSC_LoadRsqtCreate( idp, sizeof(S_RootLoadRqst), TT_SSU_MODULE, pModule,
                     LoadFinalise, &pLoadRqst );
               if (!err)
               {
                  LLInsertTail( pDC->llcLoadRequests, pLoadRqst );
                  pModule->moduleInfo.u.ssup.offset = 0;
                  pModule->pLoadRqst = pLoadRqst;
                  moduleInfo.u.ssup.offset = pModule->moduleInfo.originalSize;
                  pModule = DSC_ModuleListFindById(&llcModules, pModule->moduleInfo.u.ssup.nextModuleId);
                  while (pModule != NULL && (pModule->moduleInfo.u.ssup.positionType & PTYP_INTER))
                  {
                     LLRemove( pModule, DC_MODULE_LIST );
                     LLInsertTail( pDC->llcDcModules, pModule );
                     pModule->pLoadRqst = pLoadRqst;
                     pModule->moduleInfo.u.ssup.offset = moduleInfo.u.ssup.offset;
                     moduleInfo.u.ssup.offset += pModule->moduleInfo.originalSize;
                     if (pModule->moduleInfo.u.ssup.positionType == PTYP_LAST)
                     {
                        break;
                     }
                     pModule = DSC_ModuleListFindById(&llcModules, pModule->moduleInfo.u.ssup.nextModuleId);
                  }
                  if (pModule == NULL || pModule->moduleInfo.u.ssup.positionType != PTYP_LAST)
                  {
                     ERRLOG(DD_DC,"Invalid end of module link %p",pModule)
                     err = CLDSM_ERR_END_OF_DATA;
                  }
                  else
                  {
                     pLoadRqst->remaining = moduleInfo.u.ssup.offset;
                  }
               }
               pModule = LLHead(&llcModules);
               break;
            }
            default:
            {
               ERRLOG(DD_DC,"Unknown position type %d",pModule->moduleInfo.u.ssup.positionType)
               err = CLDSM_ERR_END_OF_DATA;
               break;
            }
            case PTYP_LAST:
            case PTYP_INTER:
            {
               pModule = LLNext(pModule,DC_MODULE_LIST);
            }
         }
      }
      if (LLCount(&llcModules))
      {
         ERRLOG(DD_DC,"unassigned modules %d",LLCount(&llcModules))
         if (!err)
         {
            err = CLDSM_ERR_END_OF_DATA;
         }
      }
      if (err)
      {
         pModule = LLRemoveTail(&llcModules);
         while (pModule != NULL)
         {
            DSC_ModuleDestroy( idp, pModule );
            pModule = LLRemoveTail(&llcModules);
         }
         pModule = LLRemoveTail(pDC->llcDcModules);
         while (pModule != NULL)
         {
            DSC_ModuleDestroy( idp, pModule );
            pModule = LLRemoveTail(pDC->llcDcModules);
         }
         while (LLCount(pDC->llcLoadRequests) > 1)
         {
            DSC_LoadRsqtDestroy(idp, LLRemoveTail(pDC->llcLoadRequests));
         }
      }
      else
      {
         DSC_LoadRsqtNotify( idp, LLHead(pDC->llcLoadRequests), LRS_STALLED_MODULE );
         DSC_DataCrslAcquireStop( idp, pDC, SF_PRIORITY_HIGH );
         err = StartNextSsuDownload( idp, pDC );
         DSC_DataCrslCheckSsuStatus( idp, pDC );
      }
   }
   DBGERRCHK(err)
   return err;
}

static H_UsrRef SsuStartModule( P_DsmCoreInst idp, P_Module pModule )
{
   H_UsrRef usrRef;
   U8BIT *chptr,chsav;
   chptr = pModule->moduleInfo.u.ssup.mpName;
   if (chptr)
   {
      chptr += pModule->moduleInfo.u.ssup.nameLen;
      chsav = *chptr;
      *chptr = 0;
   }
   usrRef = idp->setup.ssuFuncs.startModule(pModule->moduleInfo.u.ssup.moduleType,
      pModule->moduleInfo.u.ssup.mpName);
   if (chptr)
   {
      *chptr = chsav;
   }
   return usrRef;
}

static E_DscError StartNextSsuDownload( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   E_DscError err = CLDSM_OK;
   P_RootLoadRqst pLoadRqst;
   P_Module pModule;
   pLoadRqst = LLHead(pDC->llcLoadRequests);
   if (pLoadRqst != NULL)
   {
      dsmAssert((pLoadRqst->targetKind == TT_SSU_GROUP));
      pLoadRqst = LLNext(pLoadRqst,MODULE_LOAD_REQUEST_LIST);
   }
   while (pLoadRqst != NULL)
   {
      dsmAssert((pLoadRqst->targetKind == TT_SSU_MODULE));
      if (pLoadRqst->target != NULL && pLoadRqst->usrRef == NULL)
      {
         pModule = (P_Module)pLoadRqst->target;
         pLoadRqst->usrRef = SsuStartModule( idp, pModule );
         if (pLoadRqst->usrRef == NULL)
         {
            P_RootLoadRqst pLR = pLoadRqst;
            pLoadRqst = LLPrev(pLR,MODULE_LOAD_REQUEST_LIST);
            LLRemove( pLR, MODULE_LOAD_REQUEST_LIST );
            pLR->target = NULL;
            pLR->remaining = 0;
            pLR->status = LRS_ABORTED_PATH_ERROR;
            DeleteModulesOnLoadRqst( idp, pDC, pLR );
            DSC_LoadRsqtDestroy( idp, pLR );
         }
         else
         {
            pLoadRqst->target = pDC;
            DSC_LoadRsqtNotify( idp, pLoadRqst, LRS_STALLED_MODULE );
            if (!idp->setup.ssuFuncs.wantModuleData)
            {
               while (pModule->pLoadRqst == pLoadRqst)
               {
                  dsmAssert((pModule->status == MS_INITIAL));
                  DSC_DataCrslAcquireStart( idp, pDC, SF_PRIORITY_LOW );
                  if (DSC_ModuleAcquireStart( idp, pModule, SF_PRIORITY_LOW ) == CLDSM_OK)
                  {
                     pModule->status = MS_PENDING_DDB;
                  }
                  pModule = LLNext(pModule,DC_MODULE_LIST);
               }
               if (!idp->setup.multipleSsuDownloads)
               {
                  break;
               }
            }
            else
            {
               while (pModule != NULL && pModule->pLoadRqst == pLoadRqst)
               {
                  dsmAssert((pModule->status == MS_INITIAL));
                  if (idp->setup.ssuFuncs.wantModuleData(pLoadRqst->usrRef, MAKE_MODULE_REF(pDC->dataCarouselId, pModule->moduleInfo.moduleId),
                          pModule->moduleInfo.u.ssup.offset, pModule->moduleInfo.originalSize, pModule->moduleInfo.u.ssup.moduleCrc))
                  {
                     DSC_DataCrslAcquireStart( idp, pDC, SF_PRIORITY_LOW );
                     if (DSC_ModuleAcquireStart( idp, pModule, SF_PRIORITY_LOW ) == CLDSM_OK)
                     {
                        pModule->status = MS_PENDING_DDB;
                     }
                  }
                  else
                  {
                     pModule->pLoadRqst = NULL;
                     pLoadRqst->remaining -= pModule->moduleInfo.originalSize;
                  }
                  pModule = LLNext(pModule,DC_MODULE_LIST);
               }
               if (!pLoadRqst->remaining)
               {
                  P_RootLoadRqst pLR = pLoadRqst;
                  pLoadRqst = LLPrev(pLR,MODULE_LOAD_REQUEST_LIST);
                  DSC_LoadRsqtFinalNotify( idp, pLR, LRS_LOADED );
               }
               else if (!idp->setup.multipleSsuDownloads)
               {
                  break;
               }
            }
         }
      }
      pLoadRqst = LLNext(pLoadRqst,MODULE_LOAD_REQUEST_LIST);
   }
   DBGERRCHK(err)
   return err;
}

static void AbortSsuDownload( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   P_RootLoadRqst pLoadRqst, pLR;
   pLoadRqst = LLHead(pDC->llcLoadRequests);
   ASSERT(pLoadRqst != NULL)
   if (pLoadRqst != NULL && pLoadRqst->status == LRS_STALLED_MODULE)
   {
      while (LLCount(pDC->llcLoadRequests) > 1)
      {
         pLR = LLTail(pDC->llcLoadRequests);
         DSC_LoadRsqtAbort( idp, pLR );
      }
   }
}

void DSC_DataCrslCheckSsuStatus( P_DsmCoreInst idp, P_DataCarousel pDC )
{
   P_RootLoadRqst pLoadRqst;
   pLoadRqst = LLHead(pDC->llcLoadRequests);
   if (pLoadRqst != NULL && pLoadRqst->status != LRS_LOADED)
   {
      dsmAssert((pLoadRqst->targetKind == TT_SSU_GROUP));
      pLoadRqst = LLNext(pLoadRqst,MODULE_LOAD_REQUEST_LIST);
      while (pLoadRqst != NULL && pLoadRqst->target == NULL)
      {
         dsmAssert((pLoadRqst->targetKind == TT_SSU_MODULE));
         pLoadRqst = LLNext(pLoadRqst,MODULE_LOAD_REQUEST_LIST);
      }
      if (pLoadRqst == NULL)
      {
         pLoadRqst = LLHead(pDC->llcLoadRequests);
         dsmAssert((pLoadRqst->finalise != LoadFinalise));
         DSC_LoadRsqtFinalNotify( idp, pLoadRqst, LRS_LOADED );
         if (!LLCount(pDC->llcLoadRequests))
         {
            LLDestroy( idp, &pDC->llcLoadRequests );
         }
      }
   }
}

/* -- CONTIGUOUS MEMORY DATA ACCESS FUNCTIONS */

#include "defMemUtilsContig.h"  /* -- Default mem type for functions */

static BOOLEAN GetDiiMsgInfoContig( P_DataCarousel pDC, const U8BIT *pDiiMsg )
{
   U8BIT *pDiiMsgInfo = (U8BIT *) pDiiMsg;
   BOOLEAN valid;
   U16BIT compatLen;

   dsmAssert((pDiiMsg != NULL));
   dsmAssert((pDC != NULL));

   /* -- pDiiMsg -> DIIMessageBody */

   /* -- Skip downloadId - not used here */
   SET_POS_REL( pDiiMsgInfo, 4 );

   /* -- blockSize <= MAX_DDB_BLOCKSIZE */
   /* -- L0 check because if this is too large we cannot process the module */
   /* -- NB. This is VERY unlikely since it with the default (UK MHEG)
      -- settings it would imply a broadcast module containing one object >3Mb
      -- in size or a broadcast module containing multiple objects with a DDB
      -- blockSize of only 80 bytes approx. */
   READ_UINT16( pDiiMsgInfo, pDC->diiInfo.blockSize );
   if ( pDC->diiInfo.blockSize > MAX_DDB_BLOCKSIZE )
   {
      dsmDP2(("DATA ERROR: DII Info DDB Blocksize (> %u) = %u\n", MAX_DDB_BLOCKSIZE, pDC->diiInfo.blockSize));
      valid = FALSE;
   }
   else
   {
      /* -- Skip windowSize, ackPeriod, tCDownloadWindow, tCDownloadScenario - unused in DVB OC */
      SET_POS_REL( pDiiMsgInfo, 10 );

      /* -- get compatibilityDescriptorLength */
      READ_UINT16( pDiiMsgInfo, compatLen );
      if (compatLen == 0)
      {
         valid = TRUE;
      }
      else
      {
         dsmDP3(("DATA: DII Info compatibilityDescriptorLength (!= 0) = %u\n", compatLen));
         valid = DSC_RootCrslCheckCompatibility(
               (P_RootCarousel)LLParent(pDC, OC_DATA_CAROUSEL_LIST), pDiiMsgInfo, compatLen );
         SET_POS_REL( pDiiMsgInfo, compatLen );
      }
      READ_UINT16( pDiiMsgInfo, pDC->diiInfo.numberOfModules );
      DBG2(DD_DC, "No of modules = %u", pDC->diiInfo.numberOfModules)
   }
   return valid;
}

static BOOLEAN dcFindModuleInDiiMsgContig(
   /*I*/ const MemPtr mpDiiMsg, U32BIT dataLength, U16BIT moduleId,
   /*O*/ MemPtr *mpModuleInfoDescStart )
{
   BOOLEAN found;

   dsmDP3(("dcFindModuleInDiiMsgContig()\n"));

    #include "findModuleInDii_include_src.h"

   dsmDP3(("exit dcFindModuleInDiiMsgContig -> rtn: %u\n", found));
   return found;
}

static BOOLEAN dcGetModuleInfoContig(
   const MemPtr mpModuleInfoDescStart, S_ModuleInfo *pModuleInfo )
{
   BOOLEAN valid;

   dsmDP3(("dcGetModuleInfoContig()\n"));

    #include "getModuleInfo_include_src.h"

   dsmDP3(("exit dcGetModuleInfoContig -> rtn: %u\n", valid));
   return valid;
}

static void dcGetFirstModuleDescContig(
   /*I*/ const MemPtr mpDiiMsg,
   /*O*/ MemPtr *mpModuleInfoDescStart )
{
   MemPtr mpModuleInfoDesc;
   MemPos currPos = 0;
   U16BIT compatLen;

   dsmDP3(("dcGetFirstModuleDescContig()\n"));

   dsmAssert((mpDiiMsg != NULL));
   dsmAssert((mpModuleInfoDescStart != NULL));


   /* -- Open MemPtr for accessing module info in DII message */
   MEMPTR_OPEN( mpDiiMsg, mpModuleInfoDesc );

   /* -- mpModuleInfoDesc -> DIIMessageBody */

   /* -- Skip downloadId, blockSize - not used here */
   SET_POS_REL( mpModuleInfoDesc, 6 );

   /* -- Skip windowSize, ackPeriod, tCDownloadWindow, tCDownloadScenario - unused in DVB */
   SET_POS_REL( mpModuleInfoDesc, 10 );

   /* --  get compatibilityDescriptorLength, and skip */
   READ_UINT16( mpModuleInfoDesc, compatLen );
   if (compatLen != 0)
   {
      SET_POS_REL( mpModuleInfoDesc, compatLen );
   }

   /* -- Skip numberOfModules - not used here */
   SET_POS_REL( mpModuleInfoDesc, 2 );

   /* -- mpModuleInfoDesc -> start of first module descriptor */

   GET_POS( mpModuleInfoDesc, currPos );
   SET_POS_ABS( *mpModuleInfoDescStart, currPos );

   MEMPTR_CLOSE( mpModuleInfoDesc );

   dsmDP3(("exit dcGetFirstModuleDescContig\n"));
}

static void dcGetNextModuleDescContig(
   /*I*/ const MemPtr mpCurrModuleDescStart,
   /*O*/ MemPtr *mpNextModuleDescStart )
{
   MemPtr mpModuleInfoDesc;
   MemPos currPos = 0;
   U8BIT moduleInfoLength;

   dsmDP3(("dcGetNextModuleDescContig()\n"));

   dsmAssert((mpCurrModuleDescStart != NULL));
   dsmAssert((mpNextModuleDescStart != NULL));

   /* -- Open MemPtr for accessing module info in DII message */
   MEMPTR_OPEN( mpCurrModuleDescStart, mpModuleInfoDesc );

   /* -- mpModuleInfoDesc -> start of current module descriptor */

   /* -- Skip moduleId - not used here */
   SET_POS_REL( mpModuleInfoDesc, 2 );

   /* -- Skip moduleSize, moduleVersion */
   SET_POS_REL( mpModuleInfoDesc, 5 );

   /* -- Read moduleInfoLength */
   READ_UINT8( mpModuleInfoDesc, moduleInfoLength );

   /* -- Skip objectInfo */
   SET_POS_REL( mpModuleInfoDesc, (S32BIT)moduleInfoLength );

   /* -- mpModuleInfoDesc -> start of next module descriptor */

   GET_POS( mpModuleInfoDesc, currPos );
   SET_POS_ABS( *mpNextModuleDescStart, currPos );

   MEMPTR_CLOSE( mpModuleInfoDesc );

   dsmDP3(("exit dcGetNextModuleDescContig\n"));
}

#ifndef MEM_CONTIGUOUS

/* -- MANAGED MEMORY DATA ACCESS FUNCTIONS */

#include "defMemUtilsMgd.h"  /* -- Default mem type for functions */

static BOOLEAN dcFindModuleInDiiMsgSeq(
   /*I*/ const MemPtr mpDiiMsg, U32BIT dataLength, U16BIT moduleId,
   /*O*/ MemPtr *mpModuleInfoDescStart )
{
   BOOLEAN found;

   dsmDP3(("dcFindModuleInDiiMsgSeq()\n"));

    #include "findModuleInDii_include_src.h"

   dsmDP3(("exit dcFindModuleInDiiMsgSeq -> rtn: %u\n", found));
   return found;
}

static BOOLEAN dcGetModuleInfoSeq(
   const MemPtr mpModuleInfoDescStart, S_ModuleInfo *pModuleInfo )
{
   BOOLEAN valid;

   dsmDP3(("dcGetModuleInfoSeq()\n"));

    #include "getModuleInfo_include_src.h"

   dsmDP3(("exit dcGetModuleInfoSeq -> rtn: %u\n", valid));
   return valid;
}

#endif /*!MEM_CONTIGUOUS*/

/*------------------------------ Local Functions -----------------------------*/


/*----------------------------------------------------------------------------*/
