/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Internal Platform dependent definitions, for debug
 * @file    dsmDbg.h
 * @date    01/12/2004
 * @author  Ocean Blue
 */
#ifndef _DSMDBG_H_
#define _DSMDBG_H_

/*-------------------------------- Includes ----------------------------------*/
#include "cldsmdbg.h"

/*-------------------------------- Defines -----------------------------------*/
/* -- DEBUG PRINT DEFINITIONS */

/* -- All debug output goes to stdout (printf) only
***** Because currently we don't know how to use memory based DPs in DVP/MIPS!
*/

#if (defined(NDEBUG) || defined(NO_DP)) /* -- ie. retail and/or assert build */
  #define dsmAssert( assertion )
  #define ASSERT(condition)
  #ifdef DSM_DP_LEVEL
    #undef DSM_DP_LEVEL
  #endif
  #define DSM_DP_LEVEL 0
#else
 #ifdef DEBUG_ASSERT
  #include "stb_debug.h"
  #define dsmAssert( assertion )    if (!assertion) STB_SPDebugAssertFail(__FILE__, __LINE__, # assertion);
  #define ASSERT(condition)         if (!(condition)) STB_SPDebugAssertFail(__FILE__, __LINE__, # condition);
 #else
  #include <assert.h>
  #define dsmAssert( assertion )    assert assertion
  #define ASSERT(condition)         assert(condition);
 #endif

  #ifndef DSM_DP_LEVEL
    #define DSM_DP_LEVEL  (1)   /* -- default */
  #endif
#endif

/* -- Level 1 Debug Print
*
* For fatal errors and conditions that should not occur frequently.
*
* eg.
* - Internal and API errors (optional since also assert)
* - Runtime errors
* - Fatal data errors
* - Critical debug info
*
* NB. Directed to a both debug print ports for high visibility.
*
*/
#if DSM_DP_LEVEL >= 1
  #define dsmDP1( x )     if (DBG_ErrorPrintfFunc) DBG_ErrorPrintfFunc x
  #define ERRPRINT(x, ...) \
   if (DBG_ErrorPrintfFunc) DBG_ErrorPrintfFunc("%s:%d ERROR: " x "\n", __FUNCTION__, __LINE__, ##__VA_ARGS__);
  #define DBG1(f, x, ...) \
   if (DBG_ErrorPrintfFunc && (f & dsmDbgState)) DBG_ErrorPrintfFunc("%s:%d " x "\n", __FUNCTION__, __LINE__, ##__VA_ARGS__);
  #define ERRLOG(f, x, ...) \
   if (DBG_ErrorPrintfFunc) DBG_ErrorPrintfFunc("%s:%d " x "\n", __FUNCTION__, __LINE__, ##__VA_ARGS__);
  #define DBGLOG(f, x, ...) \
   if (DBG_ErrorPrintfFunc && (f & idp->dbgMask)) DBG_ErrorPrintfFunc("%s:%d " x "\n", __FUNCTION__, __LINE__, ##__VA_ARGS__);
  #define DBGERRCHK(err)  if (err) { ERRPRINT("err=%d",err) }
 #else
  #define dsmDP1( x )
  #define ERRPRINT(x, ...)
  #define DBG1(f, x, ...)
  #define ERRLOG(f, x, ...)
  #define DBGLOG(f, x, ...)
  #define DBGERRCHK(err)
#endif

/* -- Level 2 Debug Print
*
* For warnings, conditions etc. that may occur more frequently.
*
* eg.
* - Non-fatal data errors (ie. warnings)
* - API Function entry/exit print
*
*/
#if DSM_DP_LEVEL >= 2
  #define dsmDP2( x )     if (DBG_WarnPrintfFunc) DBG_WarnPrintfFunc x
  #define DBG2(f, x, ...) \
   if (DBG_WarnPrintfFunc && (f & dsmDbgState)) DBG_WarnPrintfFunc("%s:%d " x "\n", __FUNCTION__, __LINE__, ##__VA_ARGS__);
#else
  #define dsmDP2( x )
  #define DBG2(f, x, ...)
#endif


/* -- Level 3 Debug Print
*
* For general debug info and conditions that occur fairly frequently
* but only print limited output.
*
* eg. - Lower level function entry/exit points
*
*/
#if DSM_DP_LEVEL >= 3
  #define dsmDP3( x )   if (DBG_DebugPrintfFunc) DBG_DebugPrintfFunc x
  #define DBG3(f, x, ...) \
   if (DBG_DebugPrintfFunc && (f & dsmDbgState)) DBG_DebugPrintfFunc("%s:%d " x "\n", __FUNCTION__, __LINE__, ##__VA_ARGS__);
#else
  #define dsmDP3( x )
  #define DBG3(f, x, ...)
#endif


/* -- Level 4 Debug Print
*
* For detailed debug info and conditions that occur VERY frequently
* and/or print large amounts of output.
*
* eg. - Very low level function entry/exit points
*     - Data array values printed from loop
*
*/
#if DSM_DP_LEVEL >= 4
  #define dsmDP4( x )   if (DBG_InfoPrintfFunc) DBG_InfoPrintfFunc x
  #define DBG4(f, x, ...) \
   if (DBG_InfoPrintfFunc && (f & dsmDbgState)) DBG_InfoPrintfFunc("%s:%d " x "\n", __FUNCTION__, __LINE__, ##__VA_ARGS__);

#else
  #define dsmDP4( x )
  #define DBG4(f, x, ...)
#endif


void STB_SPDumpWrite (U8BIT *data, U32BIT size );

/*-------------------------------- Types -------------------------------------*/


/*------------------------------  Exported Data  -----------------------------*/

#if (!defined(NDEBUG) && !defined(NO_DP))
extern F_Printf DBG_ErrorPrintfFunc;
extern F_Printf DBG_WarnPrintfFunc;
extern F_Printf DBG_DebugPrintfFunc;
extern F_Printf DBG_InfoPrintffunc;
extern U32BIT dsmDbgState;
#endif

/*------------------------------ Prototypes ----------------------------------*/


/*----------------------------------------------------------------------------*/

#endif /* _DSMDBG_H_ */

