/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2002 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions for managing DSM object heap
 * @file    dsmObject.c
 * @date    1/2/2002
 * @author  N Kirkland
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "dsmObject.h"

#include "linkList.h"
#include "cacheMgr.h"


/*------------------------------- Local Macros -------------------------------*/


/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/


/*------------------- Local prototypes/forward declarations ------------------*/


/*---------------------------- Exported Functions ----------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// objectCreate
// Creates an instance of the Object struct, and initialises it.
// NB. objectKey not supplied since it may not be known when object
//     is first created.
///////////////////////////////////////////////////////////////////////////// */
E_DscError objectCreate( P_DsmCoreInst idp, P_DsmObject *ppObject )
{
   P_DsmObject pObject = NULL;
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((ppObject != NULL));

   pObject = (P_DsmObject)DSC_CmMemGet( idp, sizeof(S_DsmObject) );
   if (!pObject)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      err = CLDSM_OK;
      memset( pObject, 0, sizeof(S_DsmObject));

      llLinkInit( pObject->llData, NUM_LISTS_OBJECT );

      pObject->magic = DSM_OBJECT_MAGIC;
      pObject->myDsmInst = idp;
      pObject->cachingRules = CACHE_RULES_DEFAULT;
   }
   *ppObject = pObject;

   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: objectCreate: %u\n", err)));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// objectDestroy
// Destroys an instance of the Object struct, and all associated data.
///////////////////////////////////////////////////////////////////////////// */
void objectDestroy( P_DsmCoreInst idp, P_DsmObject pObject )
{
   dsmAssert((idp != NULL));
   dsmAssert((pObject != NULL));

   dsmAssert((pObject->llData[CURR_OBJECT_LIST].pLLCtrl == NULL));
   dsmAssert((pObject->llData[MODULE_OBJECT_LIST].pLLCtrl == NULL));
   dsmAssert((pObject->magic == DSM_OBJECT_MAGIC));

   /* -- Should not have loadRequest assigned */
   dsmAssert((pObject->r.pLoadRequest == NULL));

   /* -- Should not have hModule assigned */
   dsmAssert((pObject->pModule == NULL));

   /* -- TODO: Should we do this here or just have debug assert ? */
   if (pObject->objectDataSeq)
   {
      dsmDP3(("INFO: objectDestroy - object still open (forced closed)\n"));
      memSeqClose( pObject->objectDataSeq );
      idp->dsmObjectsOpen--;
   }

   /* -- NULL dsmObject magic and myDsmInst.
      -- NB. Should allow error to be detected if destroyed object is
     -- erroneously re-used (not guaranteed since may depend on MemMgr
     -- implementation) */
   pObject->magic = 0;
   pObject->myDsmInst = NULL;

   /* -- Use original handle here so it gets Nulled */
   DSC_CmMemRelease( idp, pObject );

   dsmDP3(("exit objectDestroy\n"));
}

/*----------------------------------------------------------------------------*/
