/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Common source code for parsing IOP::IOR data, to be included by
 *             functions processing data in contiguous or managed (MemSeq) memory.
 * @file    getIopIor_include_src.h
 * @date    28/9/2001
 * @author  R Taylor
 */

/*
-- getIorInfo - Common Code
--
-- Get typeId, location and tap info from IOR data (compatible with
-- contiguous or managed/MemSeq memory)
--
-- Uses/assumes the following vars:
--      length_used, mpIorData, pIorTypeId, pLocation, pTap
--
-- mpIorData references ('points' to) first byte of IOP::IOR
-- length_used (U16BIT) indicates whether info was successfully recovered
--
-- mpIorData    = pUInt8 OR MemSeqRef,
-- pIorTypeId   = U32BIT *
-- pLocation    = P_ObjectLocation
-- pTap         = P_DeliveryParaTap
-- length_used  = U16BIT
-- On entry - mpIorData -> first byte of IOR
-- On exit  - mpIorData -> UNCHANGED
*/
{
   U32BIT profileIdTag;
   MemPtr mpIor;
   MemPtr mpIorStart;
   MemPtr mpIopProfile;
   BOOLEAN keyValid;
   U8BIT ui8 = 0;
   U16BIT ui16 = 0;
   U32BIT ui32 = 0;
   U32BIT profileIndex;
   U32BIT taggedProfilesCount;
   U32BIT profileDataLength;
   /*U8BIT *pDump  ;*/
   /* -- stop compiler warnings when no checking */
   ui8 += 0;

   dsmAssert((mpIorData != NULL));
   dsmAssert((pIorTypeId != NULL));
   dsmAssert((pLocation != NULL));
   dsmAssert((pTap != NULL));


   MEMPTR_OPEN( mpIorData, mpIor );

   mpIorStart = mpIor;

   /*pDump  = mpIor;*/

   /* -- mpIor -> start of IOP::IOR */

   /* -- typeIdLength = 4 (DVB/UK DTT) */
   ADV_UINT32_L2CHK( mpIor, ui32, ui32 == 4,
      dsmDP2(("DATA ERROR: IOR typeIdLength (!= 4) = %u\n", ui32)),
      goto _return_False );

   /* -- Read typeId */
   READ_UINT32( mpIor, *pIorTypeId );

   /* -- taggedProfilesCount >= 1 (DVB/UK DTT) */
   READ_UINT32_L2CHK( mpIor, taggedProfilesCount, taggedProfilesCount >= 1,
      dsmDP2(("DATA ERROR: IOR taggedProfilesCount (< 1) = %u\n",
              taggedProfilesCount)),
      goto _return_False );

   /* -- Store location of first IOP::taggedProfile */
   mpIopProfile = mpIor;

   /* -- Read profileIdTag */
   READ_UINT32( mpIor, profileIdTag );

   pLocation->uiBindingTag = profileIdTag;

   /* -- mpIor -> start of (first) BIOPProfileBody */

   if (profileIdTag == TAG_BIOP)
   {
      /* -- profileDataLength */
      ADV_UINT32_L2CHK( mpIor, ui32, ui32 >=
         (2 + MIN_OBJ_LOCATION_LEN + MIN_CONN_BINDER_LEN),
         dsmDP2(("DATA ERROR: IOR TAG_BIOP profileDataLength = %u\n", ui32)),
         goto _return_False );

      /* -- byteOrder = big_endian (DVB) */
      /* -- L1 check because if this is wrong, subsequent data will be garbaged */
      ADV_UINT8_L1CHK( mpIor, ui8, ui8 == 0,
         dsmDP2(("DATA ERROR: IOR TAG_BIOP byteOrder = %u\n", ui8)),
         goto _return_False );

      /* -- liteComponentsCount >= 1 (UK DTT) */
      /* -- L1 check because if this is wrong, subsequent data will be garbaged */
      ADV_UINT8_L1CHK( mpIor, ui8, ui8 >= 1,
         dsmDP2(("DATA ERROR: IOR TAG_BIOP liteComponentsCount (< 1) = %u\n", ui8)),
         goto _return_False );

      /* -- mpIor -> start of BIOP::ObjectLocation */

      /* -- componentIdTag = BIOP::ObjectLocation (DVB/UK DTT) */
      /* -- L0 check because this must be correct */
      READ_UINT32_L0CHK( mpIor, ui32, ui32 == TAG_OBJECT_LOCATION,
         dsmDP2(("DATA ERROR: IOR TAG_BIOP 1st componentIdTag (!= ObjectLocation) = %u\n", ui32)),
         goto _return_False );

      /* -- componentDataLength */
      ADV_UINT8_L2CHK( mpIor, ui8, ui8 >= MIN_OBJLOC_COMPONENT_DATA_LEN,
         dsmDP2(("DATA ERROR: IOR ObjectLocation componentDataLength = %u\n", ui8)),
         goto _return_False );

      /* -- Read carouselId */
      READ_UINT32( mpIor, pLocation->carouselId );

      /* -- Read moduleId */
      READ_UINT16( mpIor, pLocation->moduleId );

      /* -- BIOP protocol version - major = 0x01, minor = 0x00 (DVB) */
      ADV_UINT16_L2CHK( mpIor, ui16, ui16 == 0x0100,
         dsmDP2(("DATA ERROR: IOR ObjectLocation protocol version (!=0x0100) = %u\n", ui16)),
         goto _return_False );

      /* -- mpIor -> start of objectKey info */
      READ_OBJECT_KEY( mpIor, pLocation->objectKey, keyValid );
      if (!keyValid)
      {
         dsmDP2(("DATA ERROR: IOR ObjectLocation objectKey invalid\n"));
         goto _return_False;
      }

      /* -- mpIor -> start of DSM::ConnBinder */

      /* -- componentIdTag = DSM::ConnBinder (DVB/UK DTT) */
      /* -- L0 check because this must be correct */
      READ_UINT32_L0CHK( mpIor, ui32, ui32 == TAG_CONN_BINDER,
         dsmDP2(("DATA ERROR: IOR TAG_BIOP 2nd componentIdTag (!= ConnBinder) = %u\n", ui32)),
         goto _return_False );

      /* -- componentDataLength */
      ADV_UINT8_L2CHK( mpIor, ui8, ui8 >= MIN_CONBIND_COMPONENT_DATA_LEN,
         dsmDP2(("DATA ERROR: IOR ConnBinder componentDataLength = %u\n", ui8)),
         goto _return_False );

      /* -- tapsCount >= 1 (DVB) */
      /* -- L1 check because if this is wrong, subsequent data will be garbaged */
      ADV_UINT8_L1CHK( mpIor, ui8, ui8 >= 1,
         dsmDP2(("DATA ERROR: IOR ConnBinder tapsCount (< 1) = %u\n", ui8)),
         goto _return_False );


      /* -- mpIor -> start of first BIOP::Tap */

      /* -- Read tapId - TODO: Not required ? */
      READ_UINT16( mpIor, pTap->id );

      /* -- 1st tapUse = BIOP_DELIVERY_PARA_USE (in IORs - DVB UK DTT) */
      /* -- L0 check because this must be correct */
      READ_UINT16_L0CHK( mpIor, ui16, ui16 == BIOP_DELIVERY_PARA_USE,
         dsmDP2(("DATA ERROR: IOR ConnBinder 1st tapUse (!= BIOP_DELIVERY_PARA_USE) = %u\n", ui16)),
         goto _return_False );

      /* -- Read associationTag */
      READ_UINT16( mpIor, pTap->associationTag );

      /* -- selectorLength = 10 (DVB/UK DTT) */
      ADV_UINT8_L2CHK( mpIor, ui8, ui8 == 0x0A,
         dsmDP2(("DATA ERROR: IOR ConnBinder tap selectorLength (!= 10) = %u\n", ui8)),
         goto _return_False );

      /* -- selectorType = 1 (DVB/UK DTT) */
      ADV_UINT16_L2CHK( mpIor, ui16, ui16 == 0x0001,
         dsmDP2(("DATA ERROR: IOR ConnBinder tap selectorType (!= 1) = %u\n", ui16)),
         goto _return_False );

      /* -- Read transactionId */
      READ_UINT32( mpIor, pTap->transactionId );

      /* -- Read timeout */
      READ_UINT32( mpIor, pTap->timeout );
   }
   else if (profileIdTag == TAG_LITE_OPTIONS)
   {
      U8BIT *pWriteNSAPaddr = NULL;
      U32BIT i = 0;

      /* -- profileDataLength */
      SET_POS_REL( mpIor, 4 );    /* changed from thing below as the check did nothing
        ADV_UINT32_L2CHK( mpIor, ui32, ui32 >= 0,
            dsmDP2(("DATA ERROR: IOR TAG_BIOP profileDataLength = %u\n", ui32)),
            goto _return_False );*/

      /* -- byteOrder = big_endian (DVB) */
      /* -- L1 check because if this is wrong, subsequent data will be garbaged */
      ADV_UINT8_L1CHK( mpIor, ui8, ui8 == 0,
         dsmDP2(("DATA ERROR: IOR TAG_BIOP byteOrder = %u\n", ui8)),
         goto _return_False );

      /* -- liteComponentsCount >= 1 (UK DTT) */
      /* -- L1 check because if this is wrong, subsequent data will be garbaged */
      ADV_UINT8_L1CHK( mpIor, ui8, ui8 >= 1,
         dsmDP2(("DATA ERROR: IOR TAG_BIOP liteComponentsCount (< 1) = %u\n", ui8)),
         goto _return_False );

      /* -- mpIor -> start of BIOP::ObjectLocation */

      /* -- componentIdTag = BIOP::ObjectLocation (DVB/UK DTT) */
      /* -- L0 check because this must be correct */
      READ_UINT32_L0CHK( mpIor, ui32, ui32 == TAG_SERVICE_LOCATION,
         dsmDP2(("DATA ERROR: IOR TAG_LITE_OPTIONS 1st componentIdTag (!= ServiceLocation) = %u\n", ui32)),
         goto _return_False );

      /* -- componentDataLength */
      READ_UINT8( mpIor, ui8 );

      /* -- Read service domain lg */
      ADV_UINT8_L1CHK( mpIor, ui8, ui8 == 0x14,
         dsmDP2(("DATA ERROR: IOR TAG_LITE_OPTIONS bad NSAP address length (!=  0x14) = %u\n", ui8)),
         goto _return_False );

      pWriteNSAPaddr = &pLocation->dvbCarouselNSAPaddress[0];

      i = 0;
      while (i++ < 0x14)
      {
         READ_UINT8( mpIor, *(pWriteNSAPaddr++));
      }

   #ifdef READ_COSNAMING
      {
         U32BIT uiComponentCount;
         S8BIT *pPathPtr;
         /* read name component count */
         READ_UINT32( mpIor, uiComponentCount);
         pPathPtr = &pLocation->pathName[0];
         for (i = 0; i < uiComponentCount; i++)
         {
            U32BIT uiNameId_lg = 0;
            U32BIT uiKing_lg = 0;
            U32BIT j = 0;

            /* read name_id lg */
            READ_UINT32( mpIor, uiNameId_lg);
            if (uiNameId_lg > MAX_PATH_NAME_SIZE)
            {
               uiNameId_lg = MAX_PATH_NAME_SIZE;
            }
            /* get path */
            for (j = 0; j < uiNameId_lg; j++)
            {
               READ_UINT8( mpIor, ui8);
               *pPathPtr++ = ui8;
            }
            *(pPathPtr++) = '/';
            /* read path kind lg */
            READ_UINT32( mpIor, uiKing_lg);
            /* SKIP path kind data */
            SET_POS_REL( mpIor, uiKing_lg);
         }
      }
   #endif
   }
   else
   {
      dsmDP2(("DATA ERROR: IOR - Invalid profileIdTag for DVB object carousel: %x\n",
              profileIdTag));
      goto _return_False;
   }

   /* -- SKIP ALL TAGGED PROFILES IN IOP::IOR */

   length_used = (U16BIT)(mpIopProfile - mpIorStart);
   mpIor = mpIopProfile;
   for (profileIndex = 0; profileIndex < taggedProfilesCount; profileIndex++)
   {
      /* -- Read profileIdTag */
      READ_UINT32( mpIor, profileIdTag );

      /* -- profileDataLength */
      READ_UINT32_L2CHK( mpIor, profileDataLength, profileDataLength >=
         (2 + MIN_OBJ_LOCATION_LEN + MIN_CONN_BINDER_LEN),
         dsmDP2(("DATA ERROR: IOR TAG_BIOP profileDataLength = %u\n", ui32)),
         goto _return_False );

      /* -- Skip profile */
      SET_POS_REL( mpIor, profileDataLength );

      /* -- profileIdTag, profileDataLength and the actual data */
      length_used += 8 + profileDataLength;
   }

   goto _return;

_return_False:
   length_used = 0;

_return:
   /*  printf("DUMP -----\n"); */

/*     while(pDump < mpIor) */
/*       printf(" 0x%2x",*(pDump++)); */

/*     printf("\nEND DUMP -----\n"); */

   MEMPTR_CLOSE( mpIor );
   DEBUG_CHK( length_used != 0,
      dsmDP1(("ERROR: getIorInfo failure (invalid)\n")));
}


/*----------------------------------------------------------------------------*/



