/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Common source code to extract module info from module info descriptor
 *             in DII.
 *             To be included by functions processing data in contiguous or managed
 *             (MemSeq) memory.
 * @file    getModuleInfo_include_src.h
 * @date    28/9/2001
 * @author  R Taylor
 */

/*
-- getpModuleInfo_source - Common Code
--
-- Get module info from module info descriptor in DII message (compatible with
-- contiguous or managed/MemSeq memory)
--
-- Uses/assumes the following vars:
--      valid, mpModuleInfoDescStart, *pModuleInfo
--
-- mpModuleInfoDescStart references ('points' to) first byte of module info
-- descriptor, valid (BOOLEAN) indicates whether module info was valid
--
-- mpModuleInfoDescStart    = const pUInt8/MemSeqRef,
-- pModuleInfo              = P_ModuleInfo
-- valid                    = BOOLEAN
--
-- NB. Only DVB (UK Profile 1) spec. currently implemented
*/

{
   MemPtr mpModuleInfoDesc;
   U8BIT tapsCount;
   U8BIT infoLength;
   MemPos currPos, endPos;
   U8BIT descriptorTag;
   U8BIT descriptorLength;
   U8BIT ui8 = 0;
   U16BIT ui16 = 0;

   dsmAssert((mpModuleInfoDescStart != NULL));
   dsmAssert((pModuleInfo != NULL));

   /* -- Open MemPtr for accessing module descriptor info */
   MEMPTR_OPEN( mpModuleInfoDescStart, mpModuleInfoDesc );

   /* -- Read moduleId */
   READ_UINT16( mpModuleInfoDesc, pModuleInfo->moduleId );

   /* -- Read moduleSize */
   READ_UINT32( mpModuleInfoDesc, pModuleInfo->moduleSize );
   if ( pModuleInfo->moduleSize == 0 )
   {
      /* 0 length modules are not useful! */
      dsmDP2(("DATA ERROR: Zero module size\n"));
      valid = FALSE;
   }
   else
   {
     #ifndef NDEBUG
      /* Initialise debug monitor flag */
      pModuleInfo->blkRcvd = 0;
     #endif

      /* -- Read moduleVersion */
      READ_UINT8( mpModuleInfoDesc, pModuleInfo->version );

      /* -- Initialise module compression info */
      pModuleInfo->compressed = FALSE;
      pModuleInfo->originalSize = pModuleInfo->moduleSize;

      if ( pModuleInfo->crslMagic == OC_MAGIC )
      {
         /* -- Skip pModuleInfoLength */
         SET_POS_REL( mpModuleInfoDesc, 1 );

         /* -- mpModuleInfoDesc -> BIOP::pModuleInfo */
         /* -- Read moduleTimeOut */
         READ_UINT32( mpModuleInfoDesc, pModuleInfo->u.mhgp.moduleTimeout );

         /* -- Read blockTimeOut */
         READ_UINT32( mpModuleInfoDesc, pModuleInfo->u.mhgp.blockTimeout );

         /* -- Read minBlockTime */
         READ_UINT32( mpModuleInfoDesc, pModuleInfo->u.mhgp.minBlockTime );

         /* -- Read/check tapsCount >= 1 (DVB) */
         /* -- L1 check because this should be correct */
         READ_UINT8( mpModuleInfoDesc, tapsCount );
         if ( !tapsCount )
         {
            dsmDP2(("DATA ERROR: Module info tapsCount zero\n"));
            valid = FALSE;
         }
         else
         {
            /* -- mpModuleInfoDesc -> start of first BIOP::Tap */
            /* -- Read info from first Tap */
            /* -- Read tapId - TODO: Not required ? */
            READ_UINT16( mpModuleInfoDesc, pModuleInfo->u.mhgp.tapBiopId );

            /* -- tapUse = BIOP_OBJECT_USE (in BIOP::pModuleInfoMessage - DVB/UK DTT) */
            /* -- L0 check because this must be correct */
            READ_UINT16( mpModuleInfoDesc, ui16 );
            if (ui16 != BIOP_OBJECT_USE)
            {
               dsmDP2(("DATA ERROR: Module info 1st tapUse (!= BIOP_OBJECT_USE) = %u\n", ui16));
               valid = FALSE;
            }
            else
            {
               /* -- Read associationTag */
               READ_UINT16( mpModuleInfoDesc, pModuleInfo->associationTag );

               /* -- selectorLength = 0 (DVB/UK DTT) */
               READ_UINT8( mpModuleInfoDesc, ui8 );
               if (ui8 != 0x00)
               {
                  dsmDP2(("DATA ERROR: Module info tap selectorLength (!= 0) = %u\n", ui8));
                  valid = FALSE;
               }
               else
               {
                  valid = TRUE;

                  /* -- Skip any remaining Taps */
                  while (--tapsCount > 0)
                  {
                     /* -- Skip tapId, tapUse, associationTag */
                     SET_POS_REL( mpModuleInfoDesc, 6 );
                     /* -- Read selectorLength */
                     READ_UINT8( mpModuleInfoDesc, ui8 );
                     /* -- Skip selector field */
                     SET_POS_REL( mpModuleInfoDesc, (S32BIT)ui8 );
                  }

                  /* -- mpModuleInfoDesc -> infoLength */
                  /* -- Read infoLength */
                  READ_UINT8( mpModuleInfoDesc, infoLength );

                  if (infoLength != 0)
                  {
                     /* -- Search module descriptor loop for compressed module descriptor */
                     /* -- Determine end position of search data */
                     GET_POS( mpModuleInfoDesc, currPos );
                     endPos = currPos + infoLength;
                     while (currPos < endPos)
                     {
                        /* -- mpModuleInfoDesc -> current module descriptor */

                        /* -- Read descriptorTag */
                        READ_UINT8( mpModuleInfoDesc, descriptorTag );

                        /* -- Read descriptorLength */
                        READ_UINT8( mpModuleInfoDesc, descriptorLength );

                        switch (descriptorTag)
                        {
                           case DESCRIPTOR_COMP_MODULE_TAG:
                              /* -- compressionMethod LSNibble == 0x8 (DVB/RFC1950)*/
                              /* -- L1 check because this should be correct (subsequently
                                 -- zlib inflate should also check compression method of
                                 -- module data/stream before attempting to decompress) */
                              READ_UINT8( mpModuleInfoDesc, ui8 );
                              if ( (ui8 & 0x0f) != 0x08 )
                              {
                                 dsmDP2(("DATA ERROR: Module info compressionMethod (!= 0x08) = %u\n", ui8));
                                 SET_POS_REL( mpModuleInfoDesc, 4 );
                              }
                              else
                              {
                                 /* -- Indicate module is compressed */
                                 pModuleInfo->compressed = TRUE;
                                 /* -- Read originalSize */
                                 READ_UINT32( mpModuleInfoDesc, pModuleInfo->originalSize );
                              }
                              /* -- Found the required info so quit the loop */
                              break;

                           case DESCRIPTOR_MHP_CACHING_PRIORITY:
                              /* TS 101 812, section B.2.2.4.2 Caching priority descriptor */
                              READ_UINT8( mpModuleInfoDesc, pModuleInfo->u.mhgp.priority );
                              READ_UINT8( mpModuleInfoDesc, pModuleInfo->u.mhgp.transparency );
                              break;

                           default:
                              dsmDP2(("INFO: Module - Unknown descriptor tag: %x\n", descriptorTag));
                              /* -- Skip this module descriptor */
                              SET_POS_REL( mpModuleInfoDesc, (S32BIT)descriptorLength );
                        }
                        /* -- Find next (potential) descriptor start position */
                        GET_POS( mpModuleInfoDesc, currPos );
                     }
                  }
               }
            }
         }
      }

      else if ( pModuleInfo->crslMagic == UC_MAGIC )
      {
         valid = TRUE;
         READ_UINT8( mpModuleInfoDesc, infoLength );
         if (infoLength != 0)
         {
            GET_POS( mpModuleInfoDesc, currPos );
            endPos = currPos + infoLength;
            while (currPos < endPos)
            {
               READ_UINT8( mpModuleInfoDesc, descriptorTag );
               READ_UINT8( mpModuleInfoDesc, descriptorLength );
               switch (descriptorTag)
               {
                  case DESCRIPTOR_TYPE_TAG:
                  {
                     pModuleInfo->u.ssup.typeLen = descriptorLength;
                     pModuleInfo->u.ssup.mpType = mpModuleInfoDesc;
                     break;
                  }
                  case DESCRIPTOR_NAME_TAG:
                  {
                     pModuleInfo->u.ssup.nameLen = descriptorLength;
                     pModuleInfo->u.ssup.mpName = mpModuleInfoDesc;
                     break;
                  }
                  case DESCRIPTOR_INFO_TAG:
                  {
                     if (descriptorLength > 3)
                     {
                        READ_UINT24( mpModuleInfoDesc, pModuleInfo->u.ssup.infoLang );
                        pModuleInfo->u.ssup.infoLen = descriptorLength - 3;
                        pModuleInfo->u.ssup.mpInfo = mpModuleInfoDesc;
                     }
                     break;
                  }
                  case DESCRIPTOR_MODULE_LINK_TAG:
                  {
                     READ_UINT8( mpModuleInfoDesc, ui8 );
                     READ_UINT16( mpModuleInfoDesc, pModuleInfo->u.ssup.nextModuleId );
                     pModuleInfo->u.ssup.positionType = ui8 + 1; /* increment so know received desc */
                     DBG3(DD_DC,"mod=%x module_link pos=%d, nxt=%x",
                        pModuleInfo->moduleId, ui8, pModuleInfo->u.ssup.nextModuleId )
                     break;
                  }
                  case DESCRIPTOR_CRC32_TAG:
                  {
                     READ_UINT32( mpModuleInfoDesc, pModuleInfo->u.ssup.moduleCrc );
                     break;
                  }
                  case DESCRIPTOR_LOCATION_TAG:
                  {
                     READ_UINT8( mpModuleInfoDesc, ui8 );
                     pModuleInfo->associationTag = ui8;
                     break;
                  }
                  case DESCRIPTOR_EST_DOWNLOAD_TIME_TAG:
                  {
                     break;
                  }
                  case DESCRIPTOR_COMP_MODULE_TAG:
                  {
                     READ_UINT8( mpModuleInfoDesc, ui8 );
                     if ( (ui8 & 0x0f) != 0x08 )
                     {
                        dsmDP2(("DATA ERROR: Module info compressionMethod (!= 0x08) = %u\n", ui8));
                     }
                     else
                     {
                        /* -- Indicate module is compressed */
                        pModuleInfo->compressed = TRUE;
                        /* -- Read originalSize */
                        READ_UINT32( mpModuleInfoDesc, pModuleInfo->originalSize );
                     }
                     break;
                  }
                  case DESCRIPTOR_SSU_MODULE_TYPE_TAG:
                  {
                     READ_UINT8( mpModuleInfoDesc, pModuleInfo->u.ssup.moduleType );
                     break;
                  }
                  default:
                  {
                     DBG2(DD_DC,"Module - Unknown descriptor tag: %x", descriptorTag);
                     break;
                  }
               }
               SET_POS_ABS( mpModuleInfoDesc, currPos + 2 + descriptorLength );
               GET_POS( mpModuleInfoDesc, currPos );
            }
         }
      }

      else
      {
         DBG2(DD_DC,"Module Info magic unknown =%x",pModuleInfo->crslMagic)
         valid = FALSE;
      }
   }
   MEMPTR_CLOSE( mpModuleInfoDesc );
   DEBUG_CHK( valid == TRUE,
      dsmDP1(("ERROR: getpModuleInfo failure (invalid)\n")));
}


/*----------------------------------------------------------------------------*/



