/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions for managing load requests and loading of object carousels,
 *             data carousels, modules and objects.
 * @file    loadMgr.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>
#include "clDsmSystem.h"
#include "loadMgr.h"

#include "sectionFilter.h"
#include "cacheMgr.h"
#include "module.h"
#include "moduleData.h"
#include "moduleDecompress.h"
#include "object.h"
#include "dataCarousel.h"
#include "dsmObject.h"
#include "clDsmUtils.h"
#include "cldsmcc.h"
#include "defMemUtilsMgd.h"  /* -- Default mem type for module */
#include "sectionTimer.h"


/*---constant definitions for this file--------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

typedef struct LiteOptionObject_tag LiteOptionObject_t, *pLiteOptionObject_t;

struct LiteOptionObject_tag
{
   S_LLObject llData[NUM_LITE_OBJECT_LOAD_LIST];
   U8BIT name[257];
   P_LoadRequest pLoadRequest;
};

/*---local (static) variable declarations for this file----------------------*/

/****************************
* LOCAL FUNCTION PROTOTYPES *
*****************************/

static E_DscError ProgressLoadRequest( P_DsmCoreInst idp, P_LoadRequest pLoadRequest );

static E_DscError LoadStalledModule( P_DsmCoreInst idp, P_LoadRequest pLoadRequest );

static E_DscError UpdateModuleLoadRequests( P_DsmCoreInst idp, P_Module pModule );

static E_DscError PreFetchRelativeDirObj( P_DsmCoreInst idp,
   P_LoadRequest pCurrLR, P_ObjectLocation pLocation,
   P_DeliveryParaTap pTap, U32BIT timeout );

static E_DscError PreFetchRelativeNonDirObj( P_DsmCoreInst idp,
   P_LoadRequest pCurrLR, P_ObjectLocation pLocation,
   P_DeliveryParaTap pTap, U32BIT timeout );

static E_DscError PreFetchNewDirsAndDIIsFromLoadedDir( P_DsmCoreInst idp,
   P_LoadRequest pLoadRequest );

static void UpdateModuleCachingNoFlush( P_DsmCoreInst idp, P_LoadRequest pLoadRequest );

static void ObjectLoadFinalise( P_DsmCoreInst idp, P_LoadRequest pLoadRequest );

static void PreFetchDirLoadFinalise( P_DsmCoreInst idp, P_LoadRequest pLoadRequest );

static E_DscError lmLiteOptionsObjectOnModuleUpdate(P_DsmCoreInst idp, U8BIT *name,
   P_ObjectCarousel pCurrOC, P_Module pLoadedModule, P_LoadRequest pLoadRequest );

/*---------------------------- Exported Functions ----------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// lmObjectLoadCreate
//
//  If objectCarousel SRG is loaded (check OC info)
//      Create loadRequest
//      Initialise loadRequest
//      Store loadRequest target type and set finalise func
//      Store path and timeout in loadRequest
//      Store SRG info from OC in loadRequest
//      Advance loadRequest as far as poss. with currently cached data
//          [loadRequestAdvance]
//      If loadRequest stalled on a module
//          create module load [LoadStalledModule]
//      Assign o/p loadRequest handle
//      Return OK to caller
//  Else
//      Set o/p loadRequest handle to NULL
//      Return SRG_NOT_LOADED_ERROR to caller
//  Endif
//
///////////////////////////////////////////////////////////////////////////// */
static E_DscError ObjectLoadCreate( P_DsmCoreInst idp, P_ObjectCarousel pOC,
   U8BIT *path, U32BIT timeout, H_ObjUserData pUserData, U32BIT sizeOfUserData,
   P_DsmObject pDsmObject, P_LoadRequest *ppLoadRequest )
{
   E_DscError err;
   U8BIT *pRemaining;
   U32BIT pathlen;
   P_LoadRequest pLoadRequest;

   pathlen = strlen((char *)path);

   err = DSC_LoadRsqtCreate( idp, sizeof(S_LoadRequest) + sizeOfUserData + pathlen + 1,
         TT_GEN_OBJECT, pDsmObject, (F_LoadFinalise)ObjectLoadFinalise, (P_RootLoadRqst *)&pLoadRequest );
   if (!err)
   {
      pLoadRequest->pObjCarousel = pOC;
      pLoadRequest->rlr.cachingRules = pDsmObject->cachingRules;

      pRemaining = (U8BIT *)(pLoadRequest + 1);
      pLoadRequest->rlr.usrRef = pRemaining;
      memcpy(pRemaining, pUserData, sizeOfUserData);
      pRemaining += sizeOfUserData;
      pLoadRequest->pRemainingPath = pRemaining;
      memcpy(pRemaining, path, pathlen);
   }
   *ppLoadRequest = pLoadRequest;
   return err;
}

static E_DscError LoadRequestOnSrg( P_DsmCoreInst idp, P_ObjectCarousel pOC,
   P_LoadRequest pLoadRequest )
{
   E_DscError err;
   P_Module pModule;

   /* -- NB. SRG DC & module would normally exist but may
    --     (temporarily) have been deleted due to update
    --     or cache priority fromStream (CCP0) being set
    --     on an object in the same module as the SRG */
   pModule = DSC_ObjCrslSrgModule(pOC);
   pLoadRequest->pDataCarousel = DSC_RootCrslFirstDataCarousel(&pOC->root);
   if (pModule == NULL)
   {
      DBG1(DD_LM,"No Module on carousel %p",pOC)
      err = CLDSM_OK;
      pLoadRequest->rlr.status = LRS_STALLED_SRG_MODULE;
   }
   else
   {
      pLoadRequest->pModule = pModule;
      if (pOC->srgObjectInfo.objectKind != SRG_STR)
      {
         pLoadRequest->rlr.status = LRS_STALLED_SRG_MODULE;
      }
      /* SRG module is in the process of being (or already is) updated/re-loaded
       * ie. transparent caching (when turbo-caching enabled). SRG data carousel must also be assigned
       */
      /* If SRG module is now cached (loaded) Setup and advance load */
      switch (pModule->status)
      {
         case MS_INITIAL:
         case MS_PENDING_INFO:
         case MS_PENDING_DDB:
         case MS_BUILDING:
            DBG1(DD_LM,"Module %p, status %u",pModule,pModule->status)
            pLoadRequest->rlr.status = LRS_STALLED_SRG_MODULE;
            err = CLDSM_OK;
            break;
         case MS_CACHED:
            pLoadRequest->pDataCarousel = LLParent(pModule, DC_MODULE_LIST);
            /* -- NB. pSrgModule must be closed since it can get deleted/unloaded */
            err = ProgressLoadRequest( idp, pLoadRequest );
            break;
         default:
            DBG1(DD_LM,"Module %p, status %u",pModule,pModule->status)
            err = CLDSM_OK;
            break;
      }
   }
   return err;
}

static void LoadRequestHandleCaching( P_DsmCoreInst idp, P_ObjectCarousel pOC,
   P_LoadRequest pLoadRequest )
{
   P_Module pCurrModule;
   BOOLEAN loadRequestsOnModule;

   dsmAssert((pLoadRequest->pModule != NULL));

   /* If this gets called during the notify callback of another loadRequest on the same module
    * there may still be outstanding loads on this module that have not yet been processed.
    * Only update the cache status here when/if there are no other outstanding load requests on this module.
    */
   pCurrModule = pLoadRequest->pModule;
   dsmAssert((pCurrModule->status == MS_CACHED));
   dsmAssert((pCurrModule->hModuleData != NULL));
   /* -- TODO: Handle prioritisation of all caching rule settings */
   /* -- If new caching rules have fromStream set they override any previous
      -- caching rules for module */
   if (pLoadRequest->rlr.cachingRules < pCurrModule->cachingRules)
   {
      pCurrModule->cachingRules = pLoadRequest->rlr.cachingRules;
   }

   dsmAssert((pCurrModule->status == MS_CACHED));
   loadRequestsOnModule = (pCurrModule->llcLoadRequests == NULL) ? FALSE : TRUE;

   if (loadRequestsOnModule == FALSE)
   {
      /* Update cache status of module according to module's current caching rules.
       * Don't flush parent DC since we will immediately reload module if it is destroyed. */
      UpdateModuleCachingNoFlush( idp, pLoadRequest );
   } /* else - there are further requests on the module so don't update module cache status now */
}

/* /////////////////////////////////////////////////////////////////////////////
// lmRequestObjectLoad
//
//  If objectCarousel SRG is loaded (check OC info)
//      Create loadRequest
//      Initialise loadRequest
//      Store loadRequest target type and set finalise func
//      Store path and timeout in loadRequest
//      Store SRG info from OC in loadRequest
//      Advance loadRequest as far as poss. with currently cached data
//          [loadRequestAdvance]
//      If loadRequest stalled on a module
//          create module load [LoadStalledModule]
//      Assign o/p loadRequest handle
//      Return OK to caller
//  Else
//      Set o/p loadRequest handle to NULL
//      Return SRG_NOT_LOADED_ERROR to caller
//  Endif
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError lmRequestObjectLoad( P_DsmCoreInst idp, P_ObjectCarousel pOC,
   U8BIT *path, U32BIT timeout,
   H_ObjUserData pUserData, U32BIT sizeOfUserData,
   P_DsmObject pDsmObject,
   P_LoadRequest *ppLoadRequest )
{
   P_LoadRequest pLoadRequest;
   E_DscError err = CLDSM_OK;

   dsmDP3(("lmRequestObjectLoad()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pOC != NULL));
   dsmAssert((path != NULL));
   dsmAssert((ppLoadRequest != NULL));

   switch (pOC->root.status)
   {
      case RCS_PENDING_BOOTINFO:
      case RCS_BOOTING:
         err = ObjectLoadCreate( idp, pOC, path, timeout, pUserData, sizeOfUserData,
               pDsmObject, ppLoadRequest );
         if (err == CLDSM_OK)
         {
            pLoadRequest = *ppLoadRequest;
            pLoadRequest->rlr.status = LRS_STALLED_SRG_MODULE;
         }
         break;

      case RCS_BOOTED:
      case RCS_LOADED:
         err = ObjectLoadCreate( idp, pOC, path, timeout, pUserData, sizeOfUserData,
               pDsmObject, ppLoadRequest );
         if (err == CLDSM_OK)
         {
            pLoadRequest = *ppLoadRequest;
            pLoadRequest->objectLocation = pOC->srgObjLoc;
            pLoadRequest->tap = pOC->srgTap;
            err = LoadRequestOnSrg( idp, pOC, pLoadRequest );
            if (err == CLDSM_OK)
            {
               if (pLoadRequest->rlr.status == LRS_LOADED)
               {
                  LoadRequestHandleCaching( idp, pOC, pLoadRequest );
               }
               if ((pLoadRequest->rlr.status == LRS_STALLED_SRG_MODULE) ||
                   (pLoadRequest->rlr.status == LRS_STALLED_MODULE))
               {
                  err = LoadStalledModule( idp, pLoadRequest );
               }
            }
            if (err)
            {
               /* -- Error so destroy/free allocated memory objects */
               DSC_LoadRsqtDestroy( idp, (P_RootLoadRqst)pLoadRequest );
               *ppLoadRequest = NULL;
            }
         }
         break;

      case RCS_LOAD_FAILED:
         err = CLDSM_ERR_CAROUSEL_LOAD_FAILED;
         break;

      default:
         dsmDP1(("ERROR: Illegal carousel status = %u\n", pOC->root.status));
         dsmAssert((0));
         err = CLDSM_ERR_INTERNAL;
         break;
   }

   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: lmRequestObjectLoad: %u\n", err)));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// lmPrefetchObjectLoad
//
//  If objectCarousel SRG is loaded (check OC info)
//      Create loadRequest
//      Initialise loadRequest
//      Store loadRequest target type and set finalise func
//      Store path and timeout in loadRequest
//      Store SRG info from OC in loadRequest
//      Advance loadRequest as far as poss. with currently cached data
//          [loadRequestAdvance]
//      If loadRequest stalled on a module
//          create module load [LoadStalledModule]
//      Assign o/p loadRequest handle
//      Return OK to caller
//  Else
//      Set o/p loadRequest handle to NULL
//      Return SRG_NOT_LOADED_ERROR to caller
//  Endif
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError lmPrefetchObjectLoad( P_DsmCoreInst idp,
   P_ObjectCarousel pOC, U8BIT *path, U32BIT timeout,
   P_LoadRequest *ppLoadRequest )
{
   E_DscError err;
   U8BIT *pRemaining;
   U32BIT pathlen;
   P_LoadRequest pLoadRequest;

   dsmDP3(("lmPrefetchObjectLoad()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pOC != NULL));
   dsmAssert((path != NULL));
   dsmAssert((ppLoadRequest != NULL));

   switch (pOC->root.status)
   {
      case RCS_PENDING_BOOTINFO:
      case RCS_BOOTING:
         err = CLDSM_ERR_CAROUSEL_NOT_BOOTED;
         break;

      case RCS_BOOTED:
      case RCS_LOADED:
         pathlen = strlen((char *)path);
         err = DSC_LoadRsqtCreate( idp, sizeof(S_LoadRequest), TT_GEN_OBJECT, pOC, NULL, (P_RootLoadRqst *)&pLoadRequest );
         if (!err)
         {
            pLoadRequest->pObjCarousel = pOC;

            pRemaining = (U8BIT *)(pLoadRequest + 1);
            pLoadRequest->pRemainingPath = pRemaining;
            memcpy(pRemaining, path, pathlen);

            pLoadRequest->objectLocation = pOC->srgObjLoc;
            pLoadRequest->tap = pOC->srgTap;

            err = LoadRequestOnSrg( idp, pOC, pLoadRequest );
            if (err == CLDSM_OK)
            {
               if ((pLoadRequest->rlr.status == LRS_STALLED_SRG_MODULE) ||
                   (pLoadRequest->rlr.status == LRS_STALLED_MODULE))
               {
                  err = LoadStalledModule( idp, pLoadRequest );
               }
            }
            if (err)
            {
               DSC_LoadRsqtDestroy( idp, (P_RootLoadRqst)pLoadRequest );
               pLoadRequest = NULL;
            }
         }
         break;

      case RCS_LOAD_FAILED:
         err = CLDSM_ERR_CAROUSEL_LOAD_FAILED;
         break;

      default:
         dsmDP1(("ERROR: Illegal carousel status = %u\n", pOC->root.status));
         dsmAssert((0));
         err = CLDSM_ERR_INTERNAL;
         break;
   }

   *ppLoadRequest = pLoadRequest;

   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: lmPrefetchObjectLoad: %u\n", err)));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// lmLoadDestroy
//
///////////////////////////////////////////////////////////////////////////// */
void lmLoadDestroy( P_DsmCoreInst idp, P_LoadRequest pLoadRequest )
{
   DSC_LoadRsqtDestroy( idp, (P_RootLoadRqst)pLoadRequest );
}

/* /////////////////////////////////////////////////////////////////////////////
// lmGetObjectLoadState
//
// Determine if an object (ie. object's module) is currently 'loaded' (ie. in
// the cache)
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError lmGetObjectLoadState( P_DsmCoreInst idp,
   P_ObjectCarousel pOC, U8BIT *path, P_Module *ppModule )
{
   E_DscError err = CLDSM_OK;
   U8BIT *pRemaining;
   U32BIT pathlen;
   P_LoadRequest pLoadRequest;
   P_Module pModule;

   dsmDP3(("lmGetObjectLoadState()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pOC != NULL));
   dsmAssert((path != NULL));
   dsmAssert((ppModule != NULL));

   *ppModule = NULL;

   switch (pOC->root.status)
   {
      case RCS_PENDING_BOOTINFO:
      case RCS_BOOTING:
         err = CLDSM_ERR_CAROUSEL_NOT_BOOTED;
         break;

      case RCS_BOOTED:
         /* -- Object cannot be loaded */
         break;

      case RCS_LOADED:
         /* -- Check SRG object is currently loaded (ie. not in process of
            -- being updated) */
         if (DSC_ObjCrslSrgObjectLoaded(pOC))
         {
            pathlen = strlen((char *)path);
            err = DSC_LoadRsqtCreate( idp, sizeof(S_LoadRequest) + pathlen + 1, TT_TMP_OBJECT, pOC, NULL, (P_RootLoadRqst *)&pLoadRequest );
            if (!err)
            {
               pLoadRequest->pObjCarousel = pOC;

               pRemaining = (U8BIT *)(pLoadRequest + 1);
               pLoadRequest->pRemainingPath = pRemaining;
               memcpy(pRemaining, path, pathlen);

               pLoadRequest->objectLocation = pOC->srgObjLoc;
               pLoadRequest->tap = pOC->srgTap;

               /* -- SRG module and DC must both be assigned */
               dsmAssert((DSC_ObjCrslSrgModule(pOC) != NULL));

               pLoadRequest->pDataCarousel = DSC_RootCrslFirstDataCarousel(&pOC->root);
               pLoadRequest->pModule = DSC_ObjCrslSrgModule(pOC);

               err = ProgressLoadRequest( idp, pLoadRequest );
               if (!err)
               {
                  pModule = pLoadRequest->pModule;
                  switch (pLoadRequest->rlr.status)
                  {
                     case LRS_LOADED:
                        *ppModule = pModule;
                        break;
                     case LRS_STALLED_MODULE:
                     {
                        if (pModule != NULL)
                        {
                           if (pModule->status == MS_ACQ_FAILED)
                           {
                              /*
                              -- This (transient) state only
                              -- occurs when turbo-caching is
                              -- enabled and decompress has failed
                              -- on a module that was cached
                              -- compressed (ie. not in use)
                              --
                              -- In this case, attempt to
                              -- re-acquire module (low priority
                              -- since not in use).
                              */
                              dsmAssert((idp->setup.turboCaching == TRUE));
                              pModule->status = MS_PENDING_DDB;
                              err = DSC_ModuleAcquireStart( idp, pModule, SF_PRIORITY_LOW );
                           }
                        }
                        break;
                     }
                     default:
                        /* -- Any other states - do nothing */
                        break;
                  }
               }
               /* -- Free the allocated memory (finished with) */
               DSC_LoadRsqtDestroy( idp, (P_RootLoadRqst)pLoadRequest );
            }
         }
         break;

      case RCS_LOAD_FAILED:
         err = CLDSM_ERR_CAROUSEL_LOAD_FAILED;
         break;

      default:
         /* -- Illegal status */
         dsmDP1(("ERROR: Illegal carousel status = %u\n", pOC->root.status));
         dsmAssert((0));
         err = CLDSM_ERR_INTERNAL;
         break;
   }

   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: lmGetObjectLoadState: %u\n", err)));
   dsmDP3(("exit lmGetObjectLoadState -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// lmUpdateCarouselSRGInfo
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError lmUpdateCarouselSRGInfo( P_DsmCoreInst idp,
   P_ObjectCarousel pOC, P_DeliveryParaTap pSrgTap, P_ObjectLocation pSrgLocn )
{
   P_LoadRequest pLoadRequest;
   E_DscError err = CLDSM_OK;

   dsmDP3(("lmUpdateCarouselSRGInfo()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pOC != NULL));
   dsmAssert((pSrgTap != NULL));
   dsmAssert((pSrgLocn != NULL));

   /*
   -- DSI section filter is finished with so stop it (also Nulls ref in OC).
   */

   /* AMCarousel MONITORING => Do not STOP section filtering */
   dsmAssert((pOC->root.pDsiSf));

#ifdef DSI_TIMEOUT_SUPPORT
   /* stop DSI timer */
   sectionTimerRemove(idp, pOC->root.pDsiSf);
#endif /* DSI_TIMEOUT_SUPPORT */

   if (pOC->root.status == RCS_BOOTING)
   {
      DBGLOG(DD_OC, "DSI acquired carouselId = %u\n", pOC->root.rcid);

      pOC->srgObjLoc = *pSrgLocn;
      pOC->srgTap = *pSrgTap;
      pOC->root.status = RCS_BOOTED;

      dsmAssert((pOC->root.pLoadRqst != NULL));
      pLoadRequest = (P_LoadRequest)pOC->root.pLoadRqst;

      /* -- These should not be assigned since they cannot exist
         -- when this is executed */
      dsmAssert((pLoadRequest->pDataCarousel == NULL));
      dsmAssert((pLoadRequest->pModule == NULL));

      pLoadRequest->objectLocation = *pSrgLocn;
      pLoadRequest->tap = *pSrgTap;
      pLoadRequest->rlr.status = LRS_STALLED_SRG_MODULE;

      err = LoadStalledModule( idp, pLoadRequest );
      if (err)
      {
         if (err == CLDSM_ERR_LOAD_FAILED)
         {
            /* -- Do not want to return this error from this function since
               -- client is notified of failure via Load Finalise function */
            err = CLDSM_OK;
         }

         pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;

         /* -- NB. hOC must be closed when making this call
            -- since it may get unloaded from notify callback */
         DSC_LoadRsqtFinalise( idp, (P_RootLoadRqst)pLoadRequest );
      }
      else
      {
         /* -- NB. hOC must be closed when making this call
            -- since it may get unloaded from notify callback */
         if (pLoadRequest->rlr.finalise)
         {
            pLoadRequest->rlr.finalise( idp, (P_RootLoadRqst)pLoadRequest );
         }
         /* -- NB. SRG load request not completed yet so don't destroy here */
      }
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: lmUpdateCarouselSRGInfo: %u\n", err)));
   dsmDP3(("exit lmUpdateCarouselSRGInfo -> rtn: %u\n", err));
   return err;
}

static void lmLiteOptionsModuleUpdate( P_DsmCoreInst idp, P_Module pModule )
{
   P_DataCarousel pDC;
   P_ObjectCarousel pOC;
   pLiteOptionObject_t pLiteObjFromList, pLiteObjFromListToDelete;
   ListId_t listId;

   pDC = (P_DataCarousel)LLParent( pModule, MODULE_ACQUIRE_LIST );
   if (memValidate(pDC))
   {
      pOC = LLParent(pDC, OC_DATA_CAROUSEL_LIST);
   }
   else
   {
      pOC = NULL;
   }
   if (memValidate(pOC) && TRUE == pOC->bOCLiteOptionsObjectProcessing)
   {
      pLiteObjFromListToDelete = NULL;
      /* Get listId and first OC in list from Control block */
      listId = LListId( pOC->llcOcPostponedLiteObjectLoads );
      pLiteObjFromList = LLHead( pOC->llcOcPostponedLiteObjectLoads );
      dsmDP2(("LITE list %u => handle = %#p\n", listId, pLiteObjFromList));

      while (pLiteObjFromList != NULL)
      {
         dsmDP2(("CALL liteOptionsObjectHandle: name = %s pRequest = %#p\n",
                 pLiteObjFromList->name, pLiteObjFromList->pLoadRequest));

         lmLiteOptionsObjectOnModuleUpdate( idp, pLiteObjFromList->name,
            pOC, pModule, pLiteObjFromList->pLoadRequest );

         if (LRS_LITE_OPTIONS_LOADED == pLiteObjFromList->pLoadRequest->rlr.status)
         {
            dsmDP2(("LITE_OPTIONS: Finalise object loading\n"));
            ObjectLoadFinalise(idp, pLiteObjFromList->pLoadRequest);
            pLiteObjFromListToDelete = pLiteObjFromList;
         }
         pLiteObjFromList = LLNext( pLiteObjFromList, listId );
         /* remove current from list */
         if (NULL != pLiteObjFromListToDelete)
         {
            LLRemove( pLiteObjFromListToDelete, OC_LITE_OBJECT_LOAD_LIST );
            pLiteObjFromListToDelete = NULL;
         }
      }
   }
}

/* /////////////////////////////////////////////////////////////////////////////
// lmUpdateModule
//
// Called when a new module build is completed to update any waiting
// load requests
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError lmUpdateModule( P_DsmCoreInst idp, P_Module pModule )
{
   E_DscError err;

   dsmDP3(("lmUpdateModule()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));
   dsmAssert((pModule->status == MS_CACHED));
   dsmAssert((pModule->loadedCount == 0));

   dsmDP2(("lmUpdateModule: loaded ModuleId=%u, version=%d\n", pModule->moduleInfo.moduleId, pModule->moduleInfo.version));

   lmLiteOptionsModuleUpdate( idp, pModule );

   if (pModule->llcLoadRequests)
   {
      /* -- By default, add new module to head of module priority list (this
         -- may subsequently be changed within UpdateModuleLoadRequests) */
      LLInsertHead( idp->llcModulePriority, pModule );
      err = UpdateModuleLoadRequests( idp, pModule );
   }
   else
   {
      /* -- This is either a module version update or new module
         -- pre-fetch */
      /* -- If module is not already in priority list (eg. when a new
         -- module pre-fetch), add to priority list tail */
      if (!LLCheckInListCtrl( idp->llcModulePriority, pModule ))
      {
         /* -- Add new module to tail of priority list */
         LLInsertTail( idp->llcModulePriority, pModule );
      }
      err = CLDSM_OK;
   }
   DBGERRCHK(err)
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// lmStopModuleLoadRequest
//
//  If the loadRequest is attached to a module
//      Abort and destroy loadRequest
//
//      If there are no more outstanding loadRequests on this module
//          If turboCaching enabled
//              continue acquiring module
//          Else
//              destroy module
//
///////////////////////////////////////////////////////////////////////////// */
void lmStopModuleLoadRequest( P_DsmCoreInst idp, P_LoadRequest pLoadRequest )
{
   P_ObjectCarousel pOC;
   P_Module pModule;
   BOOLEAN inList;
   BOOLEAN listEmpty;

   dsmDP3(("lmStopModuleLoadRequest()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pLoadRequest != NULL));

   inList = LLCheckInListId( MODULE_LOAD_REQUEST_LIST, pLoadRequest );
   if (!inList)
   {
      pLiteOptionObject_t pLiteObjFromList;
      ListId_t listId;

      pOC = pLoadRequest->pObjCarousel;

      if (pOC->bOCLiteOptionsObjectProcessing)
      {
         /* Get listId and first OC in list from Control block */
         listId = LListId( pOC->llcOcPostponedLiteObjectLoads );
         pLiteObjFromList = LLHead( pOC->llcOcPostponedLiteObjectLoads );
         dsmDP2(("LITE list %u => handle = %#p\n", listId, pLiteObjFromList));

         while (pLiteObjFromList)
         {
            if (pLiteObjFromList->pLoadRequest == pLoadRequest)
            {
               DSC_LoadRsqtFinalise( idp, (P_RootLoadRqst)pLoadRequest );
               break;
            }
            pLiteObjFromList = LLNext( pLiteObjFromList, listId );
         }
      }
      else
      {
         /* - Load request not attached to a module (error - DO NOT ignore) */
         DSC_LoadRsqtAbort( idp, &pLoadRequest->rlr );
      }
   }
   else
   {
      /* -- Remove load request from module load request list */
      dsmAssert((LLParent(pLoadRequest, MODULE_LOAD_REQUEST_LIST) == pLoadRequest->pModule));
      listEmpty = DSC_ModuleRemoveLoadRequest(idp, pLoadRequest);

      /* -- Save the module location handles and target kind */
      pModule = (P_Module)pLoadRequest->pModule;

      /* -- NB. Handles should never be NULL when loadRequest on a module */
      dsmAssert((pLoadRequest->pObjCarousel));
      dsmAssert((pLoadRequest->pDataCarousel));
      dsmAssert((pModule));

      /* -- Abort/destroy the loadRequest */
      DSC_LoadRsqtAbort( idp, &pLoadRequest->rlr );

      /*
      --  If there are no other loadRequests waiting for module
      --
      --      If this is a carousel load (or turboCaching disabled)
      --          Destroy the module (and any parent DC if empty)
      --      else
      --          Continue acquiring module (as pre-fetch)
      */
      if (listEmpty)
      {
         /* -- There are no other loadRequests on module */

         /* -- Module load should always be 'in progress' */
         dsmAssert(((pModule->status == MS_PENDING_INFO) ||
                    (pModule->status == MS_PENDING_DDB) ||
                    (pModule->status == MS_BUILDING)));

         if ((pLoadRequest->rlr.targetKind == TT_CAROUSEL) ||
             (idp->setup.turboCaching == FALSE) ||
             (idp->cacheFull == TRUE))
         {
            /* Destroy the module and destroy parent DC if now empty */
            DSC_ModuleDeleteDcTidyUp( idp, pModule );
         }
         /* else - Continue acquiring (pre-fetching) module
                   NB. Cannot re-set priority of DDB filters to low
                       so no need to update filter priority */
      }
      /* else - There are still active load requests on module */
   }

   dsmDP3(("exit lmStopModuleLoadRequest\n"));
}

/* /////////////////////////////////////////////////////////////////////////////
// lmSetObjectModuleLoaded
//
///////////////////////////////////////////////////////////////////////////// */
void lmSetObjectModuleLoaded( P_DsmCoreInst idp, P_Module pModule )
{
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));
   dsmAssert((pModule->status == MS_CACHED));
   dsmAssert((pModule->hModuleData != NULL));

   pModule->loadedCount++;

   if (pModule->loadedCount == 1)
   {
      /* -- This is first object loaded in this module */

      /* -- Remove module from priority list */
      LLRemove( pModule, MODULE_PRIORITY_LIST );
      /* -- Add module to loaded list */
      LLInsertHead( idp->llcLoadedModules, pModule );
   }
}

/* /////////////////////////////////////////////////////////////////////////////
// lmSetObjectModuleUnloaded
//
///////////////////////////////////////////////////////////////////////////// */
void lmSetObjectModuleUnloaded( P_DsmCoreInst idp, P_Module pModule )
{
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

   dsmAssert((pModule->loadedCount > 0));
   pModule->loadedCount--;
   if (pModule->loadedCount == 0)
   {
      if (pModule->status == MS_EXPIRED)
      {
#ifndef NDEBUG
         {
            BOOLEAN inList;

            inList = LLCheckInListCtrl( idp->llcExpiredModules, pModule );
            dsmAssert((inList == TRUE));
         }
#endif

         /* TODO: use llRemoveCtrl here (when implemented) */
         LLRemove( pModule, MODULE_DELETE_LIST );
         DSC_ModuleDestroy( idp, pModule );
      }
      else if (pModule->status == MS_CACHED)
      {
         /* -- Remove module from loaded list */
         LLRemove( pModule, MODULE_LOADED_LIST );
         /* -- Add/promote module to head of priority list */
         LLInsertHead( idp->llcModulePriority, pModule );
      }
      /* else if (pModule->status == MS_PENDING_DDB) then reloading for CCP0
       * so do not move from loaded list to priority list - avoids being purged by cache manager */
   }
}

void lmObjCarouselTimeout( P_DsmCoreInst idp, P_ObjectCarousel pOC )
{
   if (pOC->root.pLoadRqst)
   {
      P_LoadRequest pLoadRequest = (P_LoadRequest)pOC->root.pLoadRqst;
      if (LLCheckInListId(MODULE_LOAD_REQUEST_LIST, pLoadRequest))
      {
         dsmAssert((LLParent(pLoadRequest, MODULE_LOAD_REQUEST_LIST) == pLoadRequest->pModule));
         DSC_ModuleRemoveLoadRequest(idp, pLoadRequest);
      }
   }
}

/* /////////////////////////////////////////////////////////////////////////////
// abortLoadRequestsOnModuleTimeout
//
///////////////////////////////////////////////////////////////////////////// */
void lmAbortLoadRequestsOnModuleTimeout( P_DsmCoreInst idp, P_Module pModule)
{
   P_LoadRequest pLoadRequest;
   BOOLEAN listEmpty;

   dsmDP2(("lmAbortLoadRequestsOnModuleTimeout()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));
   if (pModule->llcLoadRequests != NULL)
   {
      pLoadRequest = LLHead( pModule->llcLoadRequests );
      while (pLoadRequest != NULL)
      {
         dsmAssert((LLParent(pLoadRequest, MODULE_LOAD_REQUEST_LIST) == pLoadRequest->pModule));
         listEmpty = DSC_ModuleRemoveLoadRequest(idp, pLoadRequest);

         pLoadRequest->rlr.status = LRS_ABORTED_TIMEOUT;
         DSC_LoadRsqtAbort( idp, &pLoadRequest->rlr );

         if (listEmpty)
            break;

         pLoadRequest = LLHead( pModule->llcLoadRequests );
      }
   }
}

/*------------------------------ Local Functions -----------------------------*/


/* /////////////////////////////////////////////////////////////////////////////
// LoadStalledModule
//
// Load the module which a load request has stalled on.
//
///////////////////////////////////////////////////////////////////////////// */
static E_DscError LoadStalledModule( P_DsmCoreInst idp, P_LoadRequest pLoadRequest )
{
   P_ObjectCarousel pOC;
   P_DataCarousel pDC;
   P_Module pModule;
   E_DscError err = CLDSM_OK;
   E_SFPriority sfPriority;

   dsmDP3(("LoadStalledModule()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pLoadRequest != NULL));

   dsmAssert(((pLoadRequest->rlr.status == LRS_STALLED_MODULE) ||
              (pLoadRequest->rlr.status == LRS_STALLED_SRG_MODULE)));

   pOC = pLoadRequest->pObjCarousel;
   dsmAssert((pOC != NULL));
   pDC = pLoadRequest->pDataCarousel;
   pModule = pLoadRequest->pModule;

   if (pDC != NULL)
   {
      dsmAssert(((pLoadRequest->tap.transactionId & TRANSACTION_ID_IDENT_MASK) == pDC->dataCarouselId));
      if (pModule == NULL)
      {
         err = DSC_ModuleCreate( idp, pLoadRequest->objectLocation.moduleId, &pModule );
         if (!err)
         {
            LLInsertTail( pDC->llcDcModules, pModule );
            err = DSC_DataCrslNewModuleLoad( idp, pDC, pModule );
         }
         else  /* error in DSC_ModuleCreate */
         {
            if (!LLCount(pDC->llcDcModules))
            {
               DSC_DataCrslDestroy( idp, pDC );
            }
         }
      }
      /* else - module already exists */
   }
   else
   {
      /* -- Data carousel does not exist */
      err = DSC_DataCrslCreate( idp, &pOC->root, &(pLoadRequest->tap), &pDC );
      if (!err)
      {
         err = DSC_ModuleCreate( idp, pLoadRequest->objectLocation.moduleId, &pModule );
         if (!err)
         {
            LLInsertTail( pDC->llcDcModules, pModule );
            err = DSC_DataCrslNewModuleLoad( idp, pDC, pModule );
         }
         else
         {
            /* -- Error creating module so destroy new data carousel also */
            DSC_DataCrslDestroy( idp, pDC );
         }
      }
      /* else - error in dataCarousel create */
   }

   if (!err)
   {
      pLoadRequest->pDataCarousel = pDC;
      pLoadRequest->pModule = pModule;

      /* -- Module should not be cached, expired or loaded */
      /*        dsmAssert((pModule->status != MS_CACHED)); */
      dsmAssert((pModule->status != MS_INITIAL));
      dsmAssert((pModule->status != MS_EXPIRED));

      err = DSC_ModuleAddLoadRequest( idp, pModule, pLoadRequest );
      if (err)
      {
         DSC_ModuleDeleteDcTidyUp( idp, pModule );
      }
      else
      {
         switch (pModule->status)
         {
            case MS_ACQ_ABORTED:
               /* Module was previously aborted due multiple decompression
               -- failures. This is new load request so try to acquire again. */
               pModule->decompressFailureCount = 0;
            /*fall thro*/
            case MS_ACQ_FAILED:
               /* -- Module had a decompression failure. Acquire again */
               pModule->status = MS_PENDING_DDB;
            /*fall thro*/
            case MS_PENDING_DDB:
            case MS_BUILDING:
            {
               /* -- We are ready to acquire (or are already acquiring) this module */
               /* -- Determine section filter priority */
               sfPriority = DSC_LoadRsqtPriority((P_RootLoadRqst)pLoadRequest);
               /* NB. If we are already acquiring module, DSC_ModuleAcquireStart
                     will only update/check section filter priority. */
               err = DSC_ModuleAcquireStart( idp, pModule, sfPriority );
               if (err)
               {
                  /* -- Failure to start (or update priority of) section filter so abort module */
                  DSC_ModuleDeleteDcTidyUp( idp, pModule );
               }
               break;
            }
            default: /*MS_CACHED or MS_EXPIRED*/
               break;
         }
      }
   }
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: LoadStalledModule: %u\n", err)));
   dsmDP3(("exit LoadStalledModule -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// loadRequestStepPath
//
///////////////////////////////////////////////////////////////////////////// */
static void loadRequestStepPath( P_LoadRequest pLoadRequest, U8BIT *nextPathElement )
{
   U8BIT *pRemainingPath;
   U8BIT *currPath;
   U8BIT *pFirstTokenEnd;
   U32BIT remainingLength;

   dsmDP3(("loadRequestStepPath()\n"));
   dsmAssert((pLoadRequest != NULL));
   dsmAssert((nextPathElement != NULL));

   pRemainingPath = pLoadRequest->pRemainingPath;

   currPath = pRemainingPath + pLoadRequest->remainingPathOffset;
   remainingLength = strlen((char *)currPath );
   if (remainingLength > 0)
   {
      pFirstTokenEnd = (U8BIT *)strchr((char *)currPath, PATH_SEPERATOR_CHAR );

      if (pFirstTokenEnd == NULL)
      {
         strcpy((char *)nextPathElement, (char *)currPath );
         pLoadRequest->remainingPathOffset += remainingLength;
      }
      else
      {
         remainingLength = pFirstTokenEnd - currPath;
         strncpy((char *)nextPathElement, (char *)currPath, remainingLength );
         nextPathElement[remainingLength] = '\0';

         /* Skip to the position *after* the separator */
         pLoadRequest->remainingPathOffset += remainingLength + 1;
      }
   }
   else
   {
      *nextPathElement = '\0';
   }

   dsmDP3(("exit loadRequestStepPath\n"));
}

E_DscError lmLiteOptionsObjectHandle( P_DsmCoreInst idp, U8BIT *name, P_LoadRequest pLoadRequest )
{
   P_ObjectCarousel pCurrOC = NULL;
   P_Module pCurrModule = NULL;
   P_DataCarousel pCurrDC = NULL;
   MemPtr mpModuleData, mpObjectData;
   E_DscError err = CLDSM_OK;
   U16BIT service_id, temp16;
   U32BIT moduleDataSize, carousel_id, temp32, objectLength, i;
   P_Module pModuleFromList = NULL;
   ListId_t moduleListId;
   BOOLEAN found;
   S32BIT objOffset;
   ObjectDataInfo_t objInfo = { {0, {0}}, 0, 0, 0, 0, 0, 0};
   /*MemPos              currPos,endPos;*/
   BOOLEAN valid;
   MemPtr mpBindingData;
   S_ObjectLocation objectLocation;
   S_DeliveryParaTap objectTap;
   BOOLEAN bDoLoadObject = TRUE;
   pLiteOptionObject_t pLiteObject;

   dsmDP4(("LITE_OPTIONS: lmLiteOptionsObjectHandle\n"));

   dsmDP4(("LITE_OPTIONS: Process LITE OPTION for object = %#p\n",
           pLoadRequest->rlr.target));

   carousel_id = (U32BIT)pLoadRequest->objectLocation.dvbCarouselNSAPaddress[2];
   carousel_id <<= 24;
   carousel_id &= 0xFF000000;
   temp32 = (U32BIT)pLoadRequest->objectLocation.dvbCarouselNSAPaddress[3];
   carousel_id |= ((temp32 << 16) & 0x00FF0000);
   temp32 = (U32BIT)pLoadRequest->objectLocation.dvbCarouselNSAPaddress[4];
   carousel_id |= ((temp32 << 8) & 0x0000FF00);
   temp32 = (U32BIT)pLoadRequest->objectLocation.dvbCarouselNSAPaddress[5];
   carousel_id |= (temp32 & 0x000000FF);

   service_id = pLoadRequest->objectLocation.dvbCarouselNSAPaddress[14];
   service_id <<= 8;
   service_id &= 0xFF00;
   temp16 = pLoadRequest->objectLocation.dvbCarouselNSAPaddress[15];
   service_id |= (temp16 & 0x00FF);

   pCurrOC = (P_ObjectCarousel)DSC_RootCrslListFindById( idp->llcRootCarousels, service_id, carousel_id );

   if (NULL == pCurrOC)
   {
      dsmDP1(("LITE_OPTIONS: Carousel %u not loaded\n", carousel_id));

      err = CDSM_LoadCarousel(idp, service_id, carousel_id,
            SIQUERY_CAROUSEL, (H_DsmCarousel *)&pCurrOC);

      pLoadRequest->rlr.status = LRS_LITE_OPTIONS_PENDING;
      pLoadRequest->pObjCarousel = pCurrOC;

      if (!err && pCurrOC && pLoadRequest->rlr.targetKind != TT_TMP_OBJECT)
      {
         pLiteObject = (pLiteOptionObject_t)DSC_CmMemGet( idp, sizeof(LiteOptionObject_t) );
         if (!pLiteObject)
         {
            err = CLDSM_ERR_MEM_HEAP_FULL;
         }
         else
         {
            err = CLDSM_OK;
            dsmDP2(("LITE_OPTIONS:  STORE POSTPONED LITE LOAD: pLiteObject = %#p\n", pLiteObject));
            llLinkInit( pLiteObject->llData, NUM_LITE_OBJECT_LOAD_LIST );
            pLiteObject->pLoadRequest = pLoadRequest;
            strncpy((char *)pLiteObject->name, (char *)name, 257);

            /* Store request to be managed when carousel Load will be completed */
            LLInsertHead( pCurrOC->llcOcPostponedLiteObjectLoads, pLiteObject );

            /* Tag carousel as loaded for lite Otpions object */
            pCurrOC->bOCLiteOptionsObjectProcessing = TRUE;
         }
      }
   }
   else
   {
      dsmDP3(("LITE_OPTIONS: Carousel %u load on going, status:%u\n",
              pCurrOC->carouselId,
              pCurrOC->root.status));

      if (RCS_LOADED != pCurrOC->root.status)
      {
         if (pLoadRequest->rlr.targetKind != TT_TMP_OBJECT)
         {
            pLiteObject = (pLiteOptionObject_t)DSC_CmMemGet( idp, sizeof(LiteOptionObject_t) );
            if (!pLiteObject)
            {
               err = CLDSM_ERR_MEM_HEAP_FULL;
            }
            else
            {
               err = CLDSM_OK;
               memset(pLiteObject, 0, sizeof(LiteOptionObject_t));
               llLinkInit( pLiteObject->llData, NUM_LITE_OBJECT_LOAD_LIST);

               pLiteObject->pLoadRequest = pLoadRequest;
               strncpy((char *)pLiteObject->name, (char *)name, 257);

               LLInsertHead( pCurrOC->llcOcPostponedLiteObjectLoads, pLiteObject );
            }
         }

         /* carousel not completely loaded => so object loading is set to PENDING */
         pLoadRequest->rlr.status = LRS_LITE_OPTIONS_PENDING;
         pLoadRequest->pObjCarousel = pCurrOC;
         bDoLoadObject = FALSE;
      }

      if (TRUE == bDoLoadObject)
      {
         objInfo = pCurrOC->srgObjectInfo;

         /* RETRIEVE object in SRG INFO to get moduleId */
         pCurrModule = DSC_ObjCrslSrgModule(pCurrOC);

         moduleDataSize = pCurrModule->moduleInfo.originalSize;

         MEMPTR_SEQ_OPEN( MEM_CONTEXT, moduleDataPtr(pCurrModule->hModuleData), 0,
            moduleDataSize, FALSE, mpModuleData );

         if (moduleDataFindObject( pCurrModule->hModuleData, moduleDataSize,
                &(objInfo.objectKey), &mpObjectData ))
         {
            valid = objectDataGetInfo( mpObjectData, &objInfo );

            pCurrDC = LLParent(pCurrModule, DC_MODULE_LIST);

            found = odDirFindBinding( mpObjectData, &objInfo, name, &mpBindingData );

            MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pCurrModule->hModuleData, mpModuleData );

            if (found)
            {
               valid = odDirGetBindingInfo( mpBindingData, &objectLocation, &objectTap );
               if (valid)
               {
                  /* go through all MODULES */

                  /* Get listId and first module in list from Control block */
                  moduleListId = LListId( pCurrDC->llcDcModules );
                  pModuleFromList = LLHead( pCurrDC->llcDcModules );

                  /* Go through each module in list until the one with a matching moduleId
                     is found */
                  while (pModuleFromList)
                  {
                     moduleDataSize = pModuleFromList->moduleInfo.originalSize;

                     if (pModuleFromList->moduleInfo.moduleId == objectLocation.moduleId)
                     {
                        MEMPTR_SEQ_OPEN( MEM_CONTEXT, moduleDataPtr(pCurrModule->hModuleData), 0,
                           moduleDataSize, FALSE, mpModuleData );

                        MEMPTR_OPEN( mpModuleData, mpObjectData );

                        /*GET_POS( mpObjectData, currPos );
                         *endPos = currPos + moduleDataSize;*/

                        /* COPY objectkey retrieved in SRG */
                        objInfo.objectKey.length = objectLocation.objectKey.length;
                        for (i = 0; i < objectLocation.objectKey.length; i++)
                        {
                           objInfo.objectKey.data[i] = objectLocation.objectKey.data[i];
                        }

                        if (moduleDataFindObject( pCurrModule->hModuleData, moduleDataSize,
                               &(objInfo.objectKey), &mpObjectData ))
                        {
                           dsmDP3(("LITE_OPTIONS: object path:%s found\n", name));

                           objectDataGetInfo(mpObjectData, &objInfo);

                           MEMPTR_GET_DIFF( mpModuleData, mpObjectData, objOffset);

                           objectDataGetKeyAndLen( mpObjectData,
                              &objInfo.objectKey, &objectLength );

                           /* COPY object info to output */
                           pLoadRequest->targetObjectOffset = (U32BIT) objOffset;
                           pLoadRequest->targetObjectInfo = objInfo;
                           pLoadRequest->pModule = pModuleFromList;
                           pLoadRequest->pObjCarousel = pCurrOC;
                           pLoadRequest->rlr.status = LRS_LITE_OPTIONS_LOADED;
                        }
                     }

                     pModuleFromList = LLNext( pModuleFromList, moduleListId );
                  }

                  MEMPTR_CLOSE( mpObjectData );
                  MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pCurrModule->hModuleData, mpModuleData );
               }
            }
         }
      }
   }

   dsmDP4(("exit lmLiteOptionsObjectHandle err:%u reqStatus:%u\n", err, pLoadRequest->rlr.status));
   return err;
}

E_DscError lmLiteOptionsObjectOnModuleUpdate(P_DsmCoreInst idp, U8BIT *name,
   P_ObjectCarousel pCurrOC, P_Module pLoadedModule, P_LoadRequest pLoadRequest )
{
   P_Module pSrgModule = NULL;
   MemPtr mpModuleData, mpObjectData;
   E_DscError err = CLDSM_OK;
   U32BIT moduleDataSize, objectLength, i;
   BOOLEAN found;
   S32BIT objOffset;
   ObjectDataInfo_t objInfo = { {0, {0}}, 0, 0, 0, 0, 0, 0};
   /*MemPos              currPos,endPos;*/
   BOOLEAN valid;
   MemPtr mpBindingData;
   S_ObjectLocation objectLocation;
   S_DeliveryParaTap objectTap;

   dsmDP4(("LITE_OPTIONS: lmLiteOptionsObjectOnModuleUpdate\n"));

   dsmDP4(("LITE_OPTIONS: Process LITE OPTION for object = %#p path:%s\n",
           pLoadRequest->rlr.target, name));

   if ((NULL != pCurrOC) && (NULL != pLoadedModule))
   {
      dsmDP3(("LITE_OPTIONS: Carousel %u load on going, status:%u\n",
              pCurrOC->carouselId,
              pCurrOC->root.status));

      if (DSC_ObjCrslSrgObjectLoaded(pCurrOC))
      {
         if ((RCS_LOADED == pCurrOC->root.status) || (RCS_BOOTED == pCurrOC->root.status))
         {
            /* look for object in SRG info to get moduleId and objectKey */
            objInfo = pCurrOC->srgObjectInfo;
            pSrgModule = DSC_ObjCrslSrgModule(pCurrOC);

            moduleDataSize = pSrgModule->moduleInfo.originalSize;

            MEMPTR_SEQ_OPEN( MEM_CONTEXT, moduleDataPtr(pSrgModule->hModuleData), 0,
               moduleDataSize, FALSE, mpModuleData );

            if (moduleDataFindObject( pSrgModule->hModuleData, moduleDataSize,
                   &(objInfo.objectKey), &mpObjectData))
            {
               valid = objectDataGetInfo( mpObjectData, &objInfo );

               found = odDirFindBinding( mpObjectData, &objInfo,
                     name, &mpBindingData );


               MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pSrgModule->hModuleData, mpModuleData );

               if (found)
               {
                  valid = odDirGetBindingInfo( mpBindingData,
                        &objectLocation,
                        &objectTap );

                  dsmDP4(("LITE_OPTIONS: name %s FOUND in SRG modid = %u\n",
                          name, objectLocation.moduleId));

                  if ((objectLocation.moduleId == pLoadedModule->moduleInfo.moduleId) && (valid))
                  {
                     moduleDataSize = pLoadedModule->moduleInfo.originalSize;

                     MEMPTR_SEQ_OPEN( MEM_CONTEXT, moduleDataPtr(pLoadedModule->hModuleData), 0,
                        moduleDataSize, FALSE, mpModuleData );

                     MEMPTR_OPEN( mpModuleData, mpObjectData );

                     /*GET_POS( mpObjectData, currPos );
                      *endPos = currPos + moduleDataSize;*/

                     objInfo.objectKey.length = objectLocation.objectKey.length;

                     for (i = 0; i < objectLocation.objectKey.length; i++)
                     {
                        objInfo.objectKey.data[i] = objectLocation.objectKey.data[i];
                     }

                     if (moduleDataFindObject( pLoadedModule->hModuleData, moduleDataSize,
                            &(objInfo.objectKey), &mpObjectData ))
                     {
                        objectDataGetInfo(mpObjectData, &objInfo);

                        MEMPTR_GET_DIFF( mpModuleData, mpObjectData, objOffset);

                        objectDataGetKeyAndLen( mpObjectData,
                           &objInfo.objectKey, &objectLength );

                        pLoadRequest->targetObjectOffset = (U32BIT) objOffset;
                        pLoadRequest->targetObjectInfo = objInfo;
                        pLoadRequest->pModule = pLoadedModule;
                        pLoadRequest->pObjCarousel = pCurrOC;
                        pLoadRequest->rlr.status = LRS_LITE_OPTIONS_LOADED;
                     }

                     MEMPTR_CLOSE( mpObjectData );
                     MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pLoadedModule->hModuleData, mpModuleData );
                  }
               }
            }
         }
      }
      else
      {
         dsmDP2(("LITE_OPTIONS : abort temporarly object load SRG info not loaded !!\n"));
      }
   }

   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// ProgressLoadRequest
//
///////////////////////////////////////////////////////////////////////////// */
static E_DscError ProgressLoadRequest( P_DsmCoreInst idp, P_LoadRequest pLoadRequest )
{
//    MemHandle           hCurrOC         = NULL;
   P_ObjectCarousel pCurrOC = NULL;
   P_Module pCurrModule = NULL;
   P_DataCarousel pCurrDC = NULL;
   P_DataCarousel pNextDC = NULL;
   P_Module pNextModule = NULL;
   BOOLEAN progressLoad = FALSE;
   BOOLEAN processCurrObj = FALSE;
   BOOLEAN found = FALSE;
   BOOLEAN valid = FALSE;
   ObjectDataInfo_t objInf = { {0, {0}}, 0, 0, 0, 0, 0, 0};
   U32BIT moduleDataSize;
   S32BIT objOffset;
   U8BIT nextPathElement[MAX_OBJ_NAME_SIZE];
   MemPtr mpModuleData;
   MemPtr mpObjectData;
   MemPtr mpBindingData;
   E_DscError err = CLDSM_OK;

   dsmDP3(("ProgressLoadRequest()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pLoadRequest != NULL));
   dsmAssert(((pLoadRequest->rlr.status == LRS_INITIAL) ||
              (pLoadRequest->rlr.status == LRS_STALLED_MODULE) ||
              (pLoadRequest->rlr.status == LRS_STALLED_SRG_MODULE)));

   do
   {
      progressLoad = FALSE;
      processCurrObj = FALSE;

      if (pCurrModule != pLoadRequest->pModule)
      {
         /* -- This is a 'new' (cached) module so ensure it is decompressed */
         pNextModule = pLoadRequest->pModule;

         /* -- Must be cached ! */
         dsmAssert((pNextModule->status == MS_CACHED));

         /* -- If module is not yet decompressed then decompress it.
            -- NB. Module must be closed since it can get deleted/unloaded. */
         if (pNextModule->hModuleData == NULL)
         {
            err = DSC_ModuleDecompress( idp, pNextModule );
            if (err == CLDSM_ERR_END_OF_DATA)
            {
               /* Retry */
               err = CLDSM_OK;
               if ((pLoadRequest->rlr.status == LRS_INITIAL) &&
                   (pLoadRequest->rlr.targetKind & TT_GEN_OBJECT))
               {
                  /*
                  -- Load is now stalled on SRG module.
                  -- NB. For internal (relative) pre-fetches the
                  -- initial module always starts decompressed so
                  -- this code should only get executed on client
                  -- requested loads.
                  */
                  pLoadRequest->rlr.status = LRS_STALLED_SRG_MODULE;
               }
               /* else - don't change current load request status */
               break;                                    /* -- EXIT LOOP */
            }
            else if (err)
            {
               /* -- Error decompressing module so abort load request */
               pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
               break;                                       /* -- EXIT LOOP */
            }
            /* else - decompress successful so continue */
         }
         /* else - module not compressed so continue */
      }
      /* else - we have already used this module (ie. it must be decompressed)
                so continue */

      pCurrModule = pLoadRequest->pModule;
      pCurrDC = pLoadRequest->pDataCarousel;
      pCurrOC = pLoadRequest->pObjCarousel;

      moduleDataSize = pCurrModule->moduleInfo.originalSize;

      MEMPTR_SEQ_OPEN( MEM_CONTEXT, moduleDataPtr(pCurrModule->hModuleData), 0,
         moduleDataSize, FALSE, mpModuleData );
      MEMPTR_OPEN( mpModuleData, mpObjectData );

      if (pLoadRequest->rlr.status == LRS_INITIAL)
      {
         switch (pLoadRequest->rlr.targetKind)
         {
            case TT_TMP_OBJECT:
            case TT_GEN_OBJECT:
               /*
               -- This is initial SRG access of object load so we can
               -- read current object details from the SRG object
               -- info store in OC
               */

               /* -- SRG module and DC must be assigned and
                  -- SRG object loaded */
               dsmAssert((DSC_ObjCrslSrgModule(pCurrOC) != NULL));
               dsmAssert((pCurrOC->srgObjectInfo.objectKind == SRG_STR));

               objOffset = pCurrOC->srgObjectOffset;
               SET_POS_REL( mpObjectData, objOffset );
               objInf = pCurrOC->srgObjectInfo;

               pLoadRequest->rlr.status = LRS_STALLED_MODULE;
               processCurrObj = TRUE;
               break;

            case TT_PREFETCH_DIR_OBJ:
            case TT_PREFETCH_NON_DIR_OBJ:
               /* -- This is the start of an internal (relative) prefetch
                  -- so we do not need to access current object */
               pLoadRequest->rlr.status = LRS_STALLED_MODULE;
               progressLoad = TRUE;
               break;

            default:
               dsmDP1(("ERROR: Load request - invalid initial target kind: %u\n",
                       pLoadRequest->rlr.targetKind));
               dsmAssert((0));

               pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
               err = CLDSM_ERR_INTERNAL;
               break;
         }
      }
      else
      {
         /* -- Load request is stalled on current (SRG or other) module */

         /* -- Get object details from current module data */

         if (moduleDataFindObject( pCurrModule->hModuleData, moduleDataSize,
                &(pLoadRequest->objectLocation.objectKey), &mpObjectData ))
         {
            valid = objectDataGetInfo( mpObjectData, &objInf );
            if (valid)
            {
               /*
               -- NB. Don't change status here since later we need to test
               --     whether load is/was stalled on SRG or other module
               */
               processCurrObj = TRUE;
            }
            else
            {
               dsmDP1(("DATA ERROR: Load request aborted - object not found\n"));
               pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
            }
         }
         else
         {
            dsmDP1(("DATA ERROR: Load request aborted - object invalid\n"));
            pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
         }
      }


      if (processCurrObj)
      {
         if (pLoadRequest->rlr.status == LRS_STALLED_SRG_MODULE)
         {
            /*
            -- NB. This can be executed on a carousel/SRG load and
            -- also on an object load that was waiting for the
            -- SRG module (either initially after carousel boot or if
            -- the SRG module was (temporarily) deleted due to update).
            */

            /* -- Check if SRG object info is already assigned (ie. from
               -- previously processed load request) */

            if (!pCurrOC->srgObjectInfo.objectKind)
            {
               MEMPTR_GET_DIFF( mpModuleData, mpObjectData, objOffset );
               dsmAssert((objOffset >= 0));

               /* -- SRG module and DC must be assigned */
               dsmAssert((DSC_ObjCrslSrgModule(pCurrOC) != NULL));
               dsmAssert((objInf.objectKind == SRG_STR));

               pCurrOC->srgObjectOffset = (U32BIT) objOffset;
               pCurrOC->srgObjectInfo = objInf;
            }
         }

         if (pLoadRequest->pRemainingPath == NULL ||
             pLoadRequest->pRemainingPath[pLoadRequest->remainingPathOffset] == 0)
         {
            /* -- Load request has no path so it must now be complete */

            pLoadRequest->rlr.status = LRS_LOADED;
         }
         else
         {
            /* -- This is an object load with a path */

            if ((objInf.objectKind == DIR_STR) ||
                (objInf.objectKind == SRG_STR))
            {
               /* -- Current object is a 'directory' so determine
                  -- next item in path (if any) */

               loadRequestStepPath( pLoadRequest, nextPathElement );

               if (*nextPathElement != '\0')
               {
                  MEMPTR_OPEN( mpObjectData, mpBindingData );

                  found = odDirFindBinding( mpObjectData, &objInf,
                        nextPathElement, &mpBindingData );


                  if (found)
                  {
                     valid = odDirGetBindingInfo( mpBindingData,
                           &pLoadRequest->objectLocation,
                           &pLoadRequest->tap );

                     if (valid)
                     {
                        if (pLoadRequest->objectLocation.uiBindingTag == TAG_LITE_OPTIONS)
                        {
                           err = lmLiteOptionsObjectHandle(idp, nextPathElement, pLoadRequest);
                           return err;
                        }


                        /* -- Load request should be progressed */
                        progressLoad = TRUE;

                        /* -- Check Carousel ID of objectLocation
                           -- matches current Carousel ID */
                        /*
                        -- NB. L2 check since this should be correct
                        -- by definition (ie. referring to objects in
                        -- other carousels is only done via
                        -- LiteOptionsProfile). If wrong, it is
                        -- probably a broadcast error and can/should
                        -- be ignored.
                        */
                        L2_DATA_CHK(
                           pLoadRequest->objectLocation.carouselId == pCurrOC->root.rcid,
                           dsmDP1(("DATA ERROR: objectLocation carouselId (%u) does not match current ID (%u)\n",
                                   pLoadRequest->objectLocation.carouselId, pCurrOC->root.rcid)),
                           pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
                           progressLoad = FALSE
                           );
                     }
                     else
                     {
                        dsmDP1(("DATA ERROR: Invalid binding\n"));
                        pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
                     }
                  }
                  else
                  {
                     dsmDP2(("INFO: Binding not found: %s\n",
                             nextPathElement));
                     pLoadRequest->rlr.status = LRS_ABORTED_PATH_ERROR;
                  }

                  MEMPTR_CLOSE( mpBindingData );
               }
               else
               {
                  /* -- This is a target directory object so load is
                     -- completed */

                  switch (pLoadRequest->rlr.targetKind)
                  {
                     case TT_PREFETCH_DIR_OBJ:    /* internal pre-fetch of directory */
                        pLoadRequest->rlr.status = LRS_LOADED;
                        break;

                     case TT_TMP_OBJECT:
                     case TT_GEN_OBJECT:
                        dsmAssert((objInf.objectKind == SRG_STR ||
                                   objInf.objectKind == DIR_STR));
                        /* -- This is client load of SRG or DIR object */
                        pLoadRequest->rlr.status = LRS_LOADED;
                        break;

                     default:
                        dsmDP1(("ERROR: Load request - invalid"));
                        dsmDP1((" directory target kind: %u\n",
                                pLoadRequest->rlr.targetKind));
                        dsmAssert((0));

                        pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
                        err = CLDSM_ERR_INTERNAL;
                        break;
                  }
               }
            }
            else
            {
               /* -- This is a 'leaf' object so load is completed */

               dsmAssert((objInf.objectKind == FIL_STR ||
                          objInf.objectKind == STR_STR ||
                          objInf.objectKind == STE_STR));

               pLoadRequest->rlr.status = LRS_LOADED;
            }
         }


         /* -- If load is completed, extract/store the object info */

         if (pLoadRequest->rlr.status == LRS_LOADED)
         {
            MEMPTR_GET_DIFF( mpModuleData, mpObjectData, objOffset );
            dsmAssert((objOffset >= 0));

            pLoadRequest->targetObjectOffset = (U32BIT) objOffset;
            pLoadRequest->targetObjectInfo = objInf;
         }


         /*
         -- If the current module has been (successfully) used to advance
         -- or complete a client request (and it is not 'loaded' - ie. in
         -- use by a client) then promote it in the module priority list
         */
         if ((pLoadRequest->rlr.status == LRS_LOADED) || (progressLoad))
         {
            if (DSC_LoadRsqtPriority(&pLoadRequest->rlr) != SF_PRIORITY_LOW)
            {
               if (pCurrModule->loadedCount == 0)
               {
                  /* -- Promote current module to head of priority list */
                  LLInsertHead( idp->llcModulePriority, pCurrModule );
               }
               /* else - module is loaded */
            }
            /* else - this is not a client request */
         }
         /* else - load is not completed or progressing */
      }
      /* else - don't process current object */

      MEMPTR_CLOSE( mpObjectData );
      MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pCurrModule->hModuleData, mpModuleData );


      if (progressLoad)
      {
         /*
         -- Load is not finished (ie. completed or aborted). Determine if
         -- module containing 'next' object is in cache so we can progress
         -- the load request further.
         */

         /* -- Initial state */
         progressLoad = FALSE;

         /* -- Is the next module the same as the current? */
         if (pLoadRequest->objectLocation.moduleId ==
             pCurrModule->moduleInfo.moduleId)
         {
            /* -- Load request can be progressed further */
            progressLoad = TRUE;
         }
         else
         {
            /* -- Next module is not the same as current module */

            /* -- Current module no longer required */
            pLoadRequest->pModule = NULL;

            /* -- Is the next DC (not) the same as the current DC? */
            if ((pLoadRequest->tap.transactionId &
                 TRANSACTION_ID_IDENT_MASK) !=
                pCurrDC->dataCarouselId)
            {
               /* -- Current DC no longer required */
               /* -- Search DC list to see if next one is available */
               pLoadRequest->pDataCarousel =
                  DSC_DataCrslListFindById( pCurrOC->root.llcDataCarousels,
                     pLoadRequest->tap.transactionId );
            }
            /* else - next DC is the same as current */

            /* -- Is the next DC available */
            if (pLoadRequest->pDataCarousel != NULL)
            {
               /* -- Next DC is available */
               pNextDC = pLoadRequest->pDataCarousel;

               /* -- Find if next module is available */
               pLoadRequest->pModule =
                  DSC_ModuleListFindById( pNextDC->llcDcModules,
                     pLoadRequest->objectLocation.moduleId );

               if (pLoadRequest->pModule != NULL)
               {
                  /* -- Next module is available */
                  pNextModule = pLoadRequest->pModule;

                  /* -- Is next module cached? */
                  if (pNextModule->status == MS_CACHED)
                  {
                     /* -- Load request can be progressed further */
                     progressLoad = TRUE;
                  }
                  /* else - next module is not cached so load cannot be
                            progressed */
               }
               /* else - next module is not available so load cannot be
                         progressed */
            }
            /* else - next DC is not available so load cannot be
                      progressed */
         }

         if (!progressLoad)
         {
            /* -- This load request is stalled */
            pLoadRequest->rlr.status = LRS_STALLED_MODULE;
         }
      }
   }
   while (progressLoad);

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: ProgressLoadRequest: %u\n", err)));
   dsmDP3(("exit ProgressLoadRequest -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// UpdateModuleLoadRequests
//
///////////////////////////////////////////////////////////////////////////// */
static E_DscError UpdateModuleLoadRequests( P_DsmCoreInst idp, P_Module pCurrModule )
{
   P_LoadRequest pLoadRequest;
   E_DscError newErr = CLDSM_OK;
   E_DscError err = CLDSM_OK;
   BOOLEAN currModuleContainsTargetObject = FALSE;
   BOOLEAN finaliseLoad = FALSE;

   dsmDP3(("UpdateModuleLoadRequests()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pCurrModule != NULL));

   /* -- Current module must be cached */
   dsmAssert((pCurrModule->status == MS_CACHED));

   /* -- Current module must be decompressed */
   dsmAssert((pCurrModule->hModuleData != NULL));

   /* -- Current module must have load request list */
   dsmAssert((pCurrModule->llcLoadRequests != NULL));

   /* -- Update any/each loadRequest waiting on this module (oldest first) */
   /* -- There should always be at-least one loadRequest in the list */
   pLoadRequest = LLTail( pCurrModule->llcLoadRequests );
   while (pLoadRequest != NULL)
   {
      dsmAssert((LLParent(pLoadRequest, MODULE_LOAD_REQUEST_LIST) == pCurrModule));
      DSC_ModuleRemoveLoadRequest(idp, pLoadRequest);

      /* -- These must be the same */
      dsmAssert((pLoadRequest->pModule == pCurrModule));

      DSC_ModuleUse( idp, pCurrModule );

      /* -- Load request must be in one of these states */
      dsmAssert(((pLoadRequest->rlr.status == LRS_STALLED_MODULE) ||
                 (pLoadRequest->rlr.status == LRS_STALLED_SRG_MODULE)));

      /*
      -- NB. Current module is always decompressed so we do not have to
      --     handle condition where this module is re-acquired or unloaded
      --     in ProgressLoadRequest (due to decompress failure).
      */
      err = ProgressLoadRequest( idp, pLoadRequest );
      if (!err)
      {
         finaliseLoad = FALSE;

         switch (pLoadRequest->rlr.status)
         {
            case LRS_LOADED:
               if ((pLoadRequest->rlr.targetKind == TT_GEN_OBJECT) &&
                   (pLoadRequest->rlr.target != NULL))
               {
                  /* -- This is a client object load so handle the
                     -- caching rules */
                  dsmAssert((pLoadRequest->pModule != NULL));
                  dsmAssert((pLoadRequest->pModule->status == MS_CACHED));
                  dsmAssert((pLoadRequest->pModule->hModuleData != NULL));
                  if (pCurrModule == pLoadRequest->pModule)
                  {
                     /*
                     -- The current (just cached) module contains the
                     -- target object so complete the load and
                     -- update the module cache status when all load
                     -- requests have been processed (see below current
                     -- loop)
                     */
                     currModuleContainsTargetObject = TRUE;
                     finaliseLoad = TRUE;
                  }
                  else
                  {
                     /*
                     -- The current module only supplied a missing part
                     -- of the path to the target object which is actually
                     -- contained in another module that was already
                     -- in the cache . In this case, update this (target)
                     -- module's cache status before finalising the load
                     -- (may delete/destroy this (target) module).
                     */
                     /* -- Update cache status of module according to
                        -- module's current caching rules. Don't flush
                        -- parent DC since we will immediately reload
                        -- module if it is destroyed. */
                     UpdateModuleCachingNoFlush( idp, pLoadRequest );

                     if (pLoadRequest->pModule == NULL)
                     {
                        newErr = LoadStalledModule( idp, pLoadRequest );
                        if (newErr)
                        {
                           err = handleInLoopError( idp, err, newErr );

                           /* -- Abort load request */
                           pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
                           finaliseLoad = TRUE;
                        }
                     }
                     else
                     {
                        /* -- Object's module is still cached so load
                           -- is complete */
                        finaliseLoad = TRUE;
                     }
                  }
               }
               else
               {
                  /* -- This is either a client/internal pre-fetch or it
                     -- is not an object load (eg. a carousel load) - so
                     -- load is complete */
                  finaliseLoad = TRUE;
               }
               break;


            case LRS_ABORTED_PATH_ERROR:
            case LRS_ABORTED_LOAD_ERROR:
               finaliseLoad = TRUE;
               break;


            case LRS_STALLED_MODULE:
            case LRS_STALLED_SRG_MODULE:

               newErr = LoadStalledModule( idp, pLoadRequest );

               if (newErr)
               {
                  err = handleInLoopError( idp, err, newErr );

                  /* -- Abort load request & notify client */
                  pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
                  finaliseLoad = TRUE;
               }
               break;

            /*
            *** PLACEHOLDER - NOT CURRENTLY VALID ***
            case LRS_STALLED_DSI:
                TODO: Allow object loads before DSI acquired
            break;
            */

            default:
               /* -- Illegal status */
               dsmDP1(("ERROR: Illegal loadRequest status = %u\n",
                       pLoadRequest->rlr.status));
               dsmAssert((0));

               err = handleInLoopError( idp, err, CLDSM_ERR_INTERNAL );
               finaliseLoad = TRUE;
               break;
         }
      }
      else
      {
         err = handleInLoopError( idp, err, CLDSM_ERR_INTERNAL );

         /* -- Abort load request & notify client */
         pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
         finaliseLoad = TRUE;
      }

      DSC_ModuleUnUse( idp, pCurrModule );

      if (finaliseLoad)
      {
         /* -- NB. hOC must be closed when making this call
            -- since it may get unloaded from notify callback */
         DSC_LoadRsqtFinalise( idp, (P_RootLoadRqst)pLoadRequest );
      }
      pLoadRequest = (pCurrModule->llcLoadRequests) ? LLTail( pCurrModule->llcLoadRequests ) : NULL;
   }

   if (currModuleContainsTargetObject &&
       pCurrModule->cachingRules == CACHE_RULES_FROM_STREAM)
   {
      /* -- The current module contains the target of at least one
          -- object load so update cache status of module according to
          -- its current caching rules (may delete/destroy module) */
      /* -- Module must be cached with valid moduleData */
      dsmAssert((pCurrModule->status == MS_CACHED));
      dsmAssert((pCurrModule->hModuleData != NULL));

      /* -- Module must not have an active section filter */
      dsmAssert((pCurrModule->pDdbSf == NULL));

      /* -- Must not be any outstanding load requests on module */
      dsmAssert((pCurrModule->llcLoadRequests == NULL));

      /* -- NB. Without outstanding Load Requests, the data on
         -- this module can be refreshed now */
      DSC_ModuleDataRefresh( idp, pCurrModule );
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: UpdateModuleLoadRequests: %u\n", err)));
   dsmDP3(("exit UpdateModuleLoadRequests -> rtn: %u\n", err));
   return err;
}

/*
-- Internally prefetch a directory object
--
-- NB. This can be called re-entrantly (max depth = MAX_DIRECTORY_DEPTH).
-- NB. This does nothing if the cache is full
--
*/
static E_DscError PreFetchRelativeDirObj( P_DsmCoreInst idp,
   P_LoadRequest pCurrLR, P_ObjectLocation pLocation,
   P_DeliveryParaTap pTap, U32BIT timeout )
{
   P_ObjectCarousel pOC;
   P_LoadRequest pLoadRequest;
   E_DscError err = CLDSM_OK;
   BOOLEAN loadFinished = FALSE;

   dsmDP3(("PreFetchRelativeDirObj()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pCurrLR != NULL));

   /* -- Increment recursion depth counter */
   idp->pfrdoCallDepth++;

   /* -- Detect possible 'infinite loops' caused by erroneous circular
      -- references in directory hierarchy */
   if (idp->pfrdoCallDepth > MAX_DIRECTORY_DEPTH)
   {
      dsmDP1(("ERROR: PreFetchRelativeDirObj recursion limit reached: %u\n", idp->pfrdoCallDepth));
      err = CLDSM_ERR_RECURSION_LIMIT_REACHED;
      dsmAssert((0));
   }
   else if (idp->cacheFull == FALSE)
   {
      pOC = pCurrLR->pObjCarousel;

      /* -- Mark current module used since this function allocates cache
         -- memory and we don't want the module to be deleted! */
      DSC_ModuleUse( idp, pCurrLR->pModule );

      dsmAssert((pOC->root.status == RCS_LOADED));

      err = DSC_LoadRsqtCreate( idp, sizeof(S_LoadRequest), TT_PREFETCH_DIR_OBJ, NULL,
            (F_LoadFinalise)PreFetchDirLoadFinalise, (P_RootLoadRqst *)&pLoadRequest );
      if (!err)
      {
         pLoadRequest->pObjCarousel = pOC;

         pLoadRequest->pDataCarousel = pCurrLR->pDataCarousel;
         pLoadRequest->pModule = pCurrLR->pModule;
         pLoadRequest->objectLocation = *pLocation;
         pLoadRequest->tap = *pTap;

         err = ProgressLoadRequest( idp, pLoadRequest );
         if (!err)
         {
            switch (pLoadRequest->rlr.status)
            {
               case LRS_LOADED:
                  /* -- NB. Possible re-entrant call */
                  PreFetchDirLoadFinalise( idp, pLoadRequest );
                  loadFinished = TRUE;
                  break;

               case LRS_STALLED_MODULE:
                  err = LoadStalledModule( idp, pLoadRequest );
                  break;

               case LRS_ABORTED_LOAD_ERROR:
                  loadFinished = TRUE;
                  break;

               default:
                  err = CLDSM_ERR_INTERNAL;
                  break;
            }
         }
         if (err || loadFinished)
         {
            /* -- Error so destroy/free any allocated memory objects */
            DSC_LoadRsqtDestroy( idp, (P_RootLoadRqst)pLoadRequest );
         }
      }

      /* -- Finished using current module */
      DSC_ModuleUnUse( idp, pCurrLR->pModule );
   }

   /* -- Decrement recursion depth counter */
   idp->pfrdoCallDepth--;

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: PreFetchRelativeDirObj: %u\n", err)));
   dsmDP3(("exit PreFetchRelativeDirObj -> rtn: %u\n", err));
   return err;
}

/*
-- Internally prefetch a non-directory object
--
-- NB. This does nothing if the cache is full
--
*/
static E_DscError PreFetchRelativeNonDirObj( P_DsmCoreInst idp,
   P_LoadRequest pCurrLR, P_ObjectLocation pLocation,
   P_DeliveryParaTap pTap, U32BIT timeout )
{
   P_ObjectCarousel pOC;
   P_LoadRequest pLoadRequest;
   E_DscError err = CLDSM_OK;
   BOOLEAN loadFinished = FALSE;

   dsmDP3(("PreFetchRelativeNonDirObj()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pCurrLR != NULL));

   if (idp->cacheFull == FALSE)
   {
      pOC = pCurrLR->pObjCarousel;

      /* -- Mark current module used since this function allocates cache
         -- memory and we don't want the module to be deleted! */
      DSC_ModuleUse( idp, pCurrLR->pModule );

      dsmAssert((pOC->root.status == RCS_LOADED));

      err = DSC_LoadRsqtCreate( idp, sizeof(S_LoadRequest), TT_PREFETCH_NON_DIR_OBJ, pOC,
            NULL, (P_RootLoadRqst *)&pLoadRequest );
      if (!err)
      {
         pLoadRequest->pObjCarousel = pOC;

         pLoadRequest->pDataCarousel = pCurrLR->pDataCarousel;
         pLoadRequest->pModule = pCurrLR->pModule;
         pLoadRequest->objectLocation = *pLocation;
         pLoadRequest->tap = *pTap;

         err = ProgressLoadRequest( idp, pLoadRequest );
         if (!err)
         {
            switch (pLoadRequest->rlr.status)
            {
               case LRS_LOADED:
                  loadFinished = TRUE;
                  break;

               case LRS_STALLED_MODULE:
                  err = LoadStalledModule( idp, pLoadRequest );
                  break;

               case LRS_ABORTED_LOAD_ERROR:
                  loadFinished = TRUE;
                  break;

               default:
                  err = CLDSM_ERR_INTERNAL;
                  break;
            }
         }
         if (err || loadFinished)
         {
            /* -- Error so destroy/free any allocated memory objects */
            DSC_LoadRsqtDestroy( idp, (P_RootLoadRqst)pLoadRequest );
         }
      }
      /* -- Finished using current module */
      DSC_ModuleUnUse( idp, pCurrLR->pModule );
   }
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: PreFetchRelativeNonDirObj: %u\n", err)));
   dsmDP3(("exit PreFetchRelativeNonDirObj -> rtn: %u\n", err));
   return err;
}

/*
-- Parse all DIR bindings and pre-fetch any sub-dirs.
--
-- For all non-dir bindings also pre-fetch any objs that are found in
-- new (ie. as yet, not cached) data carousels/DIIs.
--
-- Stop parsing bindings if any error encountered.
--
-- NB. This can be called re-entrantly
--
*/
static E_DscError PreFetchNewDirsAndDIIsFromLoadedDir( P_DsmCoreInst idp,
   P_LoadRequest pLoadRequest )
{
   P_Module pModule = NULL;
   U32BIT bindingKind;
   U16BIT numBindings;
   MemPtr mpObjectData;
   MemPtr mpBindingData;
   E_DscError err = CLDSM_OK;
   P_DataCarousel pDC = NULL;
   P_ObjectCarousel pOC;
   S_ObjectLocation objLocation;
   S_DeliveryParaTap objTap;

   dsmDP3(("PreFetchNewDirsAndDIIsFromLoadedDir()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pLoadRequest != NULL));


   /* -- Increment recursion depth counter */
   idp->pfsdndCallDepth++;

   /* -- Detect possible 'infinite loops' caused by erroneous circular
      -- references in directory hierarchy */
   if (idp->pfsdndCallDepth > MAX_DIRECTORY_DEPTH)
   {
      dsmDP1(("ERROR: PreFetchNewDirsAndDIIsFromLoadedDir recursion limit reached: %u\n",
              idp->pfrdoCallDepth));
      err = CLDSM_ERR_RECURSION_LIMIT_REACHED;
      dsmAssert((0));
      goto _return;
   }

   pModule = pLoadRequest->pModule;

   /* -- Module should be cached and already decompressed (when
      -- SRG object info was extracted */
   dsmAssert((pModule->status == MS_CACHED));
   dsmAssert((pModule->hModuleData != NULL));

   MEMPTR_SEQ_OPEN( MEM_CONTEXT,
      moduleDataPtr(pModule->hModuleData), pLoadRequest->targetObjectOffset,
      pLoadRequest->targetObjectInfo.objectLen,
      /*threadsafe*/ FALSE, mpObjectData );

   MEMPTR_OPEN( mpObjectData, mpBindingData );

   numBindings = odDirCountAndFirstBinding( mpObjectData, &(pLoadRequest->targetObjectInfo), &mpBindingData );

   while (numBindings--)
   {
      if (odDirGetBindingKind( mpBindingData, &bindingKind ))
      {
         if (bindingKind == DIR_STR)
         {
            /* -- Directory binding so pre-fetch it */
            if (odDirGetBindingInfo( mpBindingData, &objLocation, &objTap ))
            {
               err = PreFetchRelativeDirObj( idp, pLoadRequest, &objLocation, &objTap,
                     /*timeout*/ 0 );

               if (err)
               {
                  break;
               }
            }
         }
         else
         {
            /* -- Non-directory binding */
            /* -- If binding location DII does not already exist
               --   pre-fetch binding object*/

            /*
            -- TODO: Improve efficiency since current
            --       implementation is VERY processor intensive.
            --
            -- NB. This is due to calling DSC_DataCrslListFindById (which
            -- is itself very inefficient/processor intensive)
            -- and getting all binding info on every (non-dir) binding
            -- (when only need transactionId on most of them).
            */
            if (odDirGetBindingInfo( mpBindingData, &objLocation, &objTap ))
            {
               if (objLocation.uiBindingTag == TAG_LITE_OPTIONS)
               {
                  U16BIT service_id = 0;
                  U32BIT carousel_id = 0;

                  H_DsmCarousel carouselHandle;
                  U32BIT temp32 = 0;
                  U16BIT temp16 = 0;
                  P_ObjectCarousel pNewOC;

                  carousel_id = (U32BIT)objLocation.dvbCarouselNSAPaddress[2];
                  carousel_id <<= 24;
                  carousel_id &= 0xFF000000;
                  temp32 = (U32BIT)objLocation.dvbCarouselNSAPaddress[3];
                  carousel_id |= ((temp32 << 16) & 0x00FF0000);
                  temp32 = (U32BIT)objLocation.dvbCarouselNSAPaddress[4];
                  carousel_id |= ((temp32 << 8) & 0x0000FF00);
                  temp32 = (U32BIT)objLocation.dvbCarouselNSAPaddress[5];
                  carousel_id |= (temp32 & 0x000000FF);

                  service_id = objLocation.dvbCarouselNSAPaddress[14];
                  service_id <<= 8;
                  service_id &= 0xFF00;
                  temp16 = objLocation.dvbCarouselNSAPaddress[15];
                  service_id |= (temp16 & 0x00FF);

                  dsmDP2(("CALL CDSM_LoadCarousel(carID:%u)\n", carousel_id));

                  err = CDSM_LoadCarousel(idp, service_id, carousel_id,
                        SIQUERY_CAROUSEL, &carouselHandle);

                  /* Abort */
                  if (err != CLDSM_OK)
                  {
                     break;
                  }

                  pNewOC = (P_ObjectCarousel)carouselHandle;

                  pNewOC->bOCLiteOptionsObjectProcessing = TRUE;
               }
               else
               {
                  pOC = pLoadRequest->pObjCarousel;
                  dsmAssert((pOC != NULL));

                  pDC = DSC_DataCrslListFindById( pOC->root.llcDataCarousels, objTap.transactionId );

                  if (pDC == NULL)
                  {
                     err = PreFetchRelativeNonDirObj( idp, pLoadRequest, &objLocation, &objTap, /*timeout*/ 0 );
                     if (err)
                     {
                        break;
                     }
                  }
               }
            }
         }
      }

      if (!odDirGetNextBinding( mpBindingData, &mpBindingData ))
      {
         break;
      }
   }

   MEMPTR_CLOSE( mpBindingData );
   MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pModule->hModuleData, mpObjectData );


_return:
   /* -- Decrement recursion depth counter */
   idp->pfsdndCallDepth--;

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: PreFetchNewDirsAndDIIsFromLoadedDir: %u\n", err)));
   dsmDP3(("exit PreFetchNewDirsAndDIIsFromLoadedDir -> rtn: %u\n", err));

   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// UpdateModuleCachingNoFlush
//
// Special version that does not flush the parent data carousel (ie. even if
// it is now empty).
// NB. Only used for special cases of cache priority 0 (fromStream) handling.
//
///////////////////////////////////////////////////////////////////////////// */
static void UpdateModuleCachingNoFlush( P_DsmCoreInst idp, P_LoadRequest pLoadRequest )
{
   P_Module pModule;
   P_ObjectCarousel pOC;
   P_DataCarousel pDC;

   dsmAssert((idp != NULL));
   dsmAssert((pLoadRequest != NULL));

   pModule = pLoadRequest->pModule;
   dsmAssert((pModule != NULL));

   /* -- Module must be cached with valid moduleData */
   dsmAssert((pModule->status == MS_CACHED));
   dsmAssert((pModule->hModuleData != NULL));

   /* -- Module must not have an active section filter */
   dsmAssert((pModule->pDdbSf == NULL));

   /* -- Must not be any outstanding load requests on module */
   dsmAssert((pModule->llcLoadRequests == NULL));

   if (pModule->cachingRules == CACHE_RULES_FROM_STREAM)
   {
      /* -- Find parent DC and OC for module */
      pDC = LLParent( pModule, DC_MODULE_LIST );
      pOC = LLParent( pDC, OC_DATA_CAROUSEL_LIST );
      /* -- NB. Destroy's module or moves it to delete list
         -- depending on whether it has any objects loaded */
      dsmAssert((pOC != NULL));
      dsmAssert((pDC != NULL));

      if (DSC_ModuleDataRefresh( idp, pModule ) == CLDSM_OK)
      {
         /* Caching rules have caused the object's module to be destroyed/deleted, so
          * reset the state of the load to be stalled on the now 'missing' final module. */
         if (DSC_ObjCrslSrgModule(pOC) == pModule)
         {
            /* -- SRG module was destroyed */
            pLoadRequest->rlr.status = LRS_STALLED_SRG_MODULE;
         }
         else
         {
            /* -- non-SRG module was destroyed */
            pLoadRequest->rlr.status = LRS_STALLED_MODULE;
         }
      }
   }
   dsmDP3(("exit UpdateModuleCachingNoFlush\n"));
}

static void ProcessLiteOptionsObjects( P_DsmCoreInst idp,
   P_ObjectCarousel pCarousel )
{
   pLiteOptionObject_t pLiteObjFromListToDelete = NULL;
   pLiteOptionObject_t pLiteObjFromList = NULL;
   ListId_t listId;

   /* Get listId and first OC in list from Control block */
   listId = LListId( pCarousel->llcOcPostponedLiteObjectLoads );
   pLiteObjFromList = LLHead( pCarousel->llcOcPostponedLiteObjectLoads );
   dsmDP2(("LITE list %u => handle = %#p\n", listId, pLiteObjFromList));

   while (pLiteObjFromList)
   {
      pLiteObjFromListToDelete = NULL;

      dsmDP2(("CALL liteOptionsObjectHandle: name = %s hRequest = %#p\n",
              pLiteObjFromList->name, pLiteObjFromList->pLoadRequest));

      lmLiteOptionsObjectHandle(idp, pLiteObjFromList->name, pLiteObjFromList->pLoadRequest);

      if (LRS_LITE_OPTIONS_LOADED == pLiteObjFromList->pLoadRequest->rlr.status)
      {
         dsmDP2(("LITE_OPTIONS: Finalise object loading\n"));
         ObjectLoadFinalise(idp, pLiteObjFromList->pLoadRequest);

         pLiteObjFromListToDelete = pLiteObjFromList;
      }


      /* remove current from list */
      if (NULL != pLiteObjFromListToDelete)
      {
         LLRemove( pLiteObjFromListToDelete, OC_LITE_OBJECT_LOAD_LIST );
      }
      pLiteObjFromList = LLNext( pLiteObjFromList, listId );
   }
}

static void ProcessLoadRequestsOnCarousel( P_DsmCoreInst idp, P_ObjectCarousel pOC )
{
   P_LoadRequest pNextRequest;
   P_LoadRequest pLoadRequest;
   ListId_t listId;

   listId = LListId( idp->llcCurrLoadRequests );
   pLoadRequest = LLHead( idp->llcCurrLoadRequests );

   do
   {
      pNextRequest = LLNext( pLoadRequest, listId );
      if (pLoadRequest->rlr.targetKind & TT_GEN_OBJECT)
      {
         if (pLoadRequest->rlr.status != LRS_ABORT_PENDING_RELOAD)
         {
            //lmStopModuleLoadRequest( idp, &pLoadRequest );
            if (pLoadRequest->rlr.status == LRS_STALLED_MODULE || pLoadRequest->rlr.status == LRS_STALLED_SRG_MODULE)
            {
               dsmAssert((LLParent(pLoadRequest, MODULE_LOAD_REQUEST_LIST) == pLoadRequest->pModule));
               DSC_ModuleRemoveLoadRequest(idp, pLoadRequest);
            }
         }
         else
         {
            pLoadRequest->pDataCarousel = NULL;
            pLoadRequest->pModule = NULL;
            if (pLoadRequest->rlr.target)
            {
               P_DsmObject pDsmObject = (P_DsmObject)pLoadRequest->rlr.target;
               pDsmObject->pObjCarousel = pOC;
               pOC->loadedObjectCount++;
            }
         }
         pLoadRequest->pObjCarousel = pOC;
         pLoadRequest->rlr.status = LRS_INITIAL;
         pLoadRequest->objectLocation = pOC->srgObjLoc;
         pLoadRequest->tap = pOC->srgTap;
         pLoadRequest->remainingPathOffset = 0;
         if (LoadRequestOnSrg( idp, pOC, pLoadRequest ) == CLDSM_OK &&
             pLoadRequest->rlr.status == LRS_STALLED_MODULE)
         {
            if (LoadStalledModule( idp, pLoadRequest ) == CLDSM_OK)
            {
            }
            else
            {
               pLoadRequest->rlr.status = LRS_ABORTED_LOAD_ERROR;
               DSC_LoadRsqtFinalise( idp, (P_RootLoadRqst)pLoadRequest );
            }
         }
         else
         {
            dsmAssert((pLoadRequest->rlr.status != LRS_STALLED_SRG_MODULE));
            DSC_LoadRsqtFinalise( idp, (P_RootLoadRqst)pLoadRequest );
         }
      }
      pLoadRequest = pNextRequest;
   }
   while (pLoadRequest);
}

/* -- LOAD FINALISE FUNCTIONS */

void lmCarouselLoadFinalise( P_DsmCoreInst idp,
   P_LoadRequest pLoadRequest )
{
   P_ObjectCarousel pCarousel;
   E_OCLoadStatus status = OC_LOAD_ABORTED_ERROR;
   E_DscError err = CLDSM_OK;

   dsmDP3(("carouselLoadFinalise()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pLoadRequest != NULL));
   dsmAssert((pLoadRequest->rlr.targetKind == TT_CAROUSEL));

   pCarousel = pLoadRequest->pObjCarousel;

   dsmDP2(("carouselLoadFinalise() pCarousel->carouselId = %u\n", pCarousel->root.rcid));

   dsmDP2(("carouselLoadFinalise() status = %u\n", pLoadRequest->rlr.status));

   switch (pLoadRequest->rlr.status)
   {
      case LRS_STALLED_SRG_MODULE:
         dsmAssert((pCarousel->root.status == RCS_BOOTED));
         status = OC_LOAD_BOOTED;
         break;

      case LRS_LOADED:
         pCarousel->root.status = RCS_LOADED;
         status = OC_LOAD_COMPLETED;

         DSC_ObjCrslUpdateSuiLoaded(idp,pCarousel);

         if (!pCarousel->bOCLiteOptionsObjectProcessing &&
             idp->setup.turboCaching == TRUE)
         {
            /* -- Initiate hierarchical cache pre-fill */
            err = PreFetchNewDirsAndDIIsFromLoadedDir( idp, pLoadRequest );

            if (err)
            {
               dsmDP1(("ERROR: carouselLoadFinalise: %u\n", err));
               /* -- TODO: Provide error returns from finalise funcs? */
               /* -- No error return from this function
                  -- (currently) so handle as a loop error */
               err = handleInLoopError( idp, CLDSM_OK, err );
            }
         }
         break;

      case LRS_ABORTED_TIMEOUT:
         pCarousel->root.status = RCS_LOAD_FAILED;
         status = OC_LOAD_ABORTED_TIMEOUT;
         break;

      case LRS_ABORTED_PATH_ERROR:
         pCarousel->root.status = RCS_LOAD_FAILED;
         status = OC_LOAD_ABORTED_PATH_ERROR;
         break;

      case LRS_ABORTED_LOAD_ERROR:
         pCarousel->root.status = RCS_LOAD_FAILED;
         status = OC_LOAD_ABORTED_ERROR;
         break;

      case LRS_ABORTED_BY_REQUESTER:
         pCarousel->root.status = RCS_LOAD_FAILED;
         status = OC_LOAD_ABORTED_UNLOAD;
         break;

      default: /*** INTERNAL ERROR ***/
         dsmAssert((0));
         pCarousel->root.status = RCS_LOAD_FAILED;
         status = OC_LOAD_ABORTED_ERROR;
         break;
   }
   if (pCarousel->root.status != RCS_BOOTED)
   {
      /* -- Load request is completed so NULL ref */
      pCarousel->root.pLoadRqst = NULL;
   }

   if (idp->setup.notifyCarouselLoadEventFunc)
   {
      idp->setup.notifyCarouselLoadEventFunc((H_DsmCarousel)pCarousel, status, pCarousel->root.rcid  );
   }
   if (status == OC_LOAD_COMPLETED)
   {
      if (TRUE == pCarousel->bOCLiteOptionsObjectProcessing)
      {
         ProcessLiteOptionsObjects(idp, pCarousel);
      }
      else
      {
         int listCount;
         listCount = LLCount( idp->llcCurrLoadRequests );
         if (listCount > 1)
         {
            ProcessLoadRequestsOnCarousel(idp, pCarousel);
         }
      }
   }

   dsmDP3(("exit carouselLoadFinalise\n"));
}

static void ObjectLoadFinalise( P_DsmCoreInst idp,
   P_LoadRequest pLoadRequest )
{
   P_DsmObject pDsmObject = pLoadRequest->rlr.target;

   dsmAssert((idp != NULL));
   dsmAssert((pLoadRequest != NULL));
   dsmAssert((pDsmObject != NULL));
   dsmAssert((pLoadRequest->rlr.targetKind == TT_GEN_OBJECT));
   dsmAssert((pDsmObject->magic == DSM_OBJECT_MAGIC));

   dsmDP2(("########### ObjectLoadFinalise(status=%d) ###############\n", pLoadRequest->rlr.status));

   /* -- NULL pLoadRequest value in DsmObject since loadRequest is finished
      -- with and will subsequently be destroyed */
   pDsmObject->r.pLoadRequest = NULL;

   switch (pLoadRequest->rlr.status)
   {
      case LRS_LOADED:
      case LRS_LITE_OPTIONS_LOADED:
         dsmDP2(("objectLoaded ModuleId=%d\n",pLoadRequest->pModule->moduleInfo.moduleId));
         pDsmObject->pObjCarousel = pLoadRequest->pObjCarousel;
         pDsmObject->pModule = pLoadRequest->pModule;
         pDsmObject->objectDataOffset =
            pLoadRequest->targetObjectOffset;
         pDsmObject->objectInfo = pLoadRequest->targetObjectInfo;
         pDsmObject->kind =
            convertObjectKindStr(
               pLoadRequest->targetObjectInfo.objectKind );

         lmSetObjectModuleLoaded( idp, pDsmObject->pModule );

         pDsmObject->status = OBJ_LOAD_COMPLETED;
         break;

      case LRS_ABORTED_TIMEOUT:
         ERRPRINT("%p", pDsmObject)
         pDsmObject->status = OBJ_LOAD_ABORTED_TIMEOUT;
         break;

      case LRS_ABORTED_PATH_ERROR:
         ERRPRINT("%p", pDsmObject)
         pDsmObject->status = OBJ_LOAD_ABORTED_PATH_ERROR;
         break;

      case LRS_ABORTED_LOAD_ERROR:
         ERRPRINT("%p", pDsmObject)
         pDsmObject->status = OBJ_LOAD_ABORTED_ERROR;
         break;

      case LRS_ABORTED_BY_REQUESTER:
         ERRPRINT("%p", pDsmObject)
         pDsmObject->status = OBJ_LOAD_ABORTED_UNLOAD;
         break;

      default:  /*** INTERNAL ERROR ***/
         ERRPRINT("%p", pDsmObject)
         pDsmObject->status = OBJ_LOAD_ABORTED_ERROR;
         dsmAssert((0));
         break;
   }

   if (idp->setup.notifyObjectLoadEventFunc)
   {
      idp->setup.notifyObjectLoadEventFunc( pDsmObject, pDsmObject->status,
         pLoadRequest->rlr.usrRef );
   }
   dsmDP3(("exit ObjectLoadFinalise\n"));
}

/*
-- Parse directory bindings and create new prefetch dir load for all dirs
--
-- NB. This can be called re-entrantly (max depth = MAX_DIRECTORY_DEPTH).
--
*/
static void PreFetchDirLoadFinalise( P_DsmCoreInst idp,
   P_LoadRequest pLoadRequest )
{
   E_DscError err = CLDSM_OK;

   dsmDP3(("PreFetchDirLoadFinalise()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pLoadRequest != NULL));
   dsmAssert((pLoadRequest->rlr.targetKind == TT_PREFETCH_DIR_OBJ));


   /* -- Increment recursion depth counter */
   idp->pfdlfCallDepth++;

   /* -- Detect possible 'infinite loops' caused by erroneous circular
      -- references in directory hierarchy */
   if (idp->pfdlfCallDepth > MAX_DIRECTORY_DEPTH)
   {
      dsmDP1(("ERROR: PreFetchDirLoadFinalise recursion limit reached: %u\n",
              idp->pfrdoCallDepth));
      err = CLDSM_ERR_RECURSION_LIMIT_REACHED;
      dsmAssert((0));
      goto _return;
   }

   switch (pLoadRequest->rlr.status)
   {
      case LRS_LOADED:

         err = PreFetchNewDirsAndDIIsFromLoadedDir( idp, pLoadRequest );

         if (err)
         {
            dsmDP1(("ERROR: PreFetchDirLoadFinalise: %u\n", err));
            /* -- TODO: Provide error returns from finalise funcs? */
            /* -- No error return from this function
               -- (currently) so handle as a loop error */
            err = handleInLoopError( idp, CLDSM_OK, err );
         }
         break;

      case LRS_ABORTED_TIMEOUT:
      case LRS_ABORTED_LOAD_ERROR:
      case LRS_ABORTED_BY_REQUESTER:
         /* -- Do nothing */
         break;

      default:  /*** INTERNAL ERROR ***/
         dsmAssert((0));
         break;
   }

_return:
   /* -- Decrement recursion depth counter */
   idp->pfdlfCallDepth--;

   dsmDP3(("exit PreFetchDirLoadFinalise\n"));
}

/* -- CRC CALCULATION */


/*----------------------------------------------------------------------------*/
