/*******************************************************************************
 * Copyright  2015 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   RootCarousel module - common code betweeen Update Carousel
 *          and Object Carousel.
 * @file    rootCarousel.c
 * @date    18/02/2015
 * @author  Adam Sturtridge
 */

/*---includes for this file--------------------------------------------------*/
#include <string.h>
#include "loadRqst.h"
#include "cacheMgr.h"

/*------------------------------- Local Macros -------------------------------*/


/*------------------------------  Exported Data  -----------------------------*/



/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/

/*------------------- local prototypes/forward declarations ------------------*/


/*------------------------------ Local Functions -----------------------------*/


/*---------------------------- Exported Functions ----------------------------*/

E_DscError DSC_LoadRsqtCreate( P_DsmCoreInst idp, U32BIT size,
   E_TargetType targetKind, H_Object target, F_LoadFinalise finalise,
   P_RootLoadRqst *ppLoadRqst )
{
   P_RootLoadRqst pLoadRqst;
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((size >= sizeof(S_RootLoadRqst)));

   pLoadRqst = (P_RootLoadRqst)DSC_CmMemGet( idp, size );
   if (!pLoadRqst)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      llLinkInit( pLoadRqst->llData, NUM_LISTS_LOAD_REQUEST );
      if (!LLInsertHead( idp->llcCurrLoadRequests, pLoadRqst ))
      {
         /* should never get here */
         ASSERT(0)
         ERRPRINT("insert list fail - already in list?");
         err = CLDSM_ERR_INTERNAL;
         DSC_CmMemRelease( idp, pLoadRqst );
         pLoadRqst = NULL;
      }
      else
      {
         err = CLDSM_OK;
         pLoadRqst->magic = LOAD_REQUEST_MAGIC;
         pLoadRqst->status = LRS_INITIAL;
         pLoadRqst->targetKind = targetKind;
         pLoadRqst->target = target;
         pLoadRqst->finalise = finalise;
         pLoadRqst->remaining = 0;
         pLoadRqst->cachingRules = CACHE_RULES_DEFAULT;
         pLoadRqst->usrRef = 0;
         if (size > sizeof(S_RootLoadRqst))
         {
            memset(pLoadRqst + 1, 0, size - sizeof(S_RootLoadRqst));
         }
      }
   }
   *ppLoadRqst = pLoadRqst;
   return err;
}

void DSC_LoadRsqtDestroy( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst )
{
   dsmAssert((idp != NULL));
   dsmAssert((pLoadRqst != NULL));

   dsmAssert((pLoadRqst->magic == LOAD_REQUEST_MAGIC));

   /* -- Should not be in any lists */
   dsmAssert((LLCheckInListId( MODULE_LOAD_REQUEST_LIST, pLoadRqst ) == FALSE));

   /* -- NULL carousel contents. */
   pLoadRqst->magic = 0;

   LLRemove( pLoadRqst, CURR_LOAD_REQUEST_LIST );

   DSC_CmMemRelease( idp, pLoadRqst );
}

void DSC_LoadRsqtFinalise( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst )
{
   if (pLoadRqst->finalise != NULL)
   {
      pLoadRqst->finalise( idp, pLoadRqst );
   }
   DSC_LoadRsqtDestroy( idp, pLoadRqst );
}

void DSC_LoadRsqtNotify( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst, E_LRStatus status )
{
   pLoadRqst->status = status;
   if (pLoadRqst->finalise != NULL)
   {
      pLoadRqst->finalise( idp, pLoadRqst );
   }
}

void DSC_LoadRsqtFinalNotify( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst, E_LRStatus status )
{
   pLoadRqst->status = status;
   if (pLoadRqst->finalise != NULL)
   {
      pLoadRqst->finalise( idp, pLoadRqst );
   }
   LLRemove( pLoadRqst, MODULE_LOAD_REQUEST_LIST );
   DSC_LoadRsqtDestroy( idp, pLoadRqst );
}

E_SFPriority DSC_LoadRsqtPriority( P_RootLoadRqst pLoadRqst )
{
   dsmAssert((pLoadRqst != NULL));
   dsmAssert((pLoadRqst->magic == LOAD_REQUEST_MAGIC));
   return (pLoadRqst->cachingRules == CACHE_RULES_FROM_STREAM)? SF_PRIORITY_DIRECT :
          (pLoadRqst->finalise != NULL)? SF_PRIORITY_HIGH : SF_PRIORITY_LOW;
}

/**
 * @brief   Force abort (on error) any type of load request
 *          Finalise the load request (notify client) and destroy it.
 */
void DSC_LoadRsqtAbort( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst )
{
   dsmAssert((idp != NULL));
   dsmAssert((pLoadRqst != NULL));

   switch (pLoadRqst->status)
   {
      default:
         pLoadRqst->status = LRS_ABORTED_LOAD_ERROR;
         /*follow thro*/
      case LRS_ABORTED_BY_REQUESTER:
      case LRS_ABORTED_TIMEOUT:
         DSC_LoadRsqtFinalise( idp, pLoadRqst );
         break;
      case LRS_ABORT_PENDING_RELOAD:
         /* Ignore abort */
         break;
   }
}


