/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions/methods for creating/destroying and managing attributes of
 *             DSM-CC modules.
 * @file    module.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "module.h"

#include "clDsmUtils.h"
#include "cacheMgr.h"
#include "moduleData.h"
#include "moduleDecompress.h"
#include "loadRqst.h"
#include "dataCarousel.h"
#include "sectionTimer.h"
#include "loadMgr.h"
#include "stb_os.h"

/*------------------------------- Local Macros -------------------------------*/

#define CACHE0_TIMEOUT_MS 30

/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/


/*------------------- local prototypes/forward declarations ------------------*/

#ifdef DSM_NO_MEM_CONTEXT
#define builderConstruct( x, a, b, c, d, e )  \
   builderConstruct( a, b, c, d, e )
#endif

static E_DscError builderCreate( P_DsmCoreInst idp,
   U32BIT moduleSize, U16BIT blockSize,
   P_ModuleBuilder *ppModuleBuilder );

static void builderDestroy( P_DsmCoreInst idp, P_ModuleBuilder *ppModuleBuilder );

static void builderConstruct( P_DsmCoreInst idp, P_ModuleBuilder pModuleBuilder,
   U16BIT blockNum, U16BIT ddbDataSize, U8BIT *pDDBData, P_ModuleData *phModuleData );


/*---------------------------- Exported Functions ----------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// DSC_ModuleCreate
// Creates an instance of the Module struct, and initialises it.
///////////////////////////////////////////////////////////////////////////// */
E_DscError DSC_ModuleCreate( P_DsmCoreInst idp,
   U16BIT moduleId, P_Module *ppModule )
{
   P_Module pModule = NULL;
   E_DscError err = CLDSM_OK;

   dsmDP3(("DSC_ModuleCreate()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((ppModule != NULL));

   pModule = (P_Module)DSC_CmMemGet( idp, sizeof(S_Module) );
   if (!pModule)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      memset( pModule, 0, sizeof(S_Module) );

      llLinkInit( pModule->llData, NUM_LISTS_MODULE );

      pModule->moduleInfo.moduleId = moduleId;

      pModule->moduleInfo.associationTag = INVALID_ASSOCIATION_TAG;
      pModule->cachingRules = CACHE_RULES_DEFAULT;

      err = CLDSM_OK;
   }
   *ppModule = pModule;

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: DSC_ModuleCreate: %u\n", err)));
   dsmDP3(("exit DSC_ModuleCreate -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_ModuleDestroy
// Destroys an instance of the Module struct, and all associated data.
///////////////////////////////////////////////////////////////////////////// */
void DSC_ModuleDestroy( P_DsmCoreInst idp, P_Module pModule )
{
   dsmDP3(("DSC_ModuleDestroy()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

   /* -- Check that the module is no longer linked into any lists */
   dsmAssert((pModule->llData[DC_MODULE_LIST     ].pLLCtrl == NULL));
   dsmAssert((pModule->llData[MODULE_ACQUIRE_LIST].pLLCtrl == NULL));
   dsmAssert((pModule->llData[MODULE_DELETE_LIST ].pLLCtrl == NULL));

   /* -- Check that module has no loadRequestList attached */
   dsmAssert((pModule->llcLoadRequests == NULL));

   /* -- Check that the module has no section filter associated */
   dsmAssert((pModule->pDdbSf == 0));

   /* -- Check that the module is not in use */
   dsmAssert((pModule->currentUseCount == 0));

   /* -- Check that the module has no loaded objects associated */
   dsmAssert((pModule->loadedCount == 0));

   /* -- Check that the module has no client requests */
   dsmAssert((pModule->highPriorityCount == 0));

   if (pModule->pModuleBuilder)
   {
      builderDestroy( idp, &pModule->pModuleBuilder );
   }
   if (pModule->hCompModuleData)
   {
      dsmAssert((pModule->hCompModuleData != pModule->hModuleData));
      moduleDataDestroy( idp, &pModule->hCompModuleData );
   }
   if (pModule->hModuleData)
   {
      moduleDataDestroy( idp, &pModule->hModuleData );
   }

   DSC_CmMemRelease( idp, pModule );

   dsmDP3(("exit DSC_ModuleDestroy\n"));
}

/*
-- Increments module currentUseCount
--
-- NB. Regardless of it's priority, if module is in use (ie. currentUseCount
--     is > 0), it cannot be deleted to make space in the cache.
--
*/
void DSC_ModuleUse( P_DsmCoreInst idp, P_Module pModule )
{
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

#define MAX_CURRENT_USE_COUNT   \
   (~(0xFFFFFFFF << ((sizeof(pModule->currentUseCount) * 8) - 1)))

   /* -- NB. Should never reach max value */
   dsmAssert((pModule->currentUseCount <= MAX_CURRENT_USE_COUNT));

   /* -- Detect/prevent overflow errors in retail */
   if (pModule->currentUseCount < MAX_CURRENT_USE_COUNT)
   {
      pModule->currentUseCount++;
   }
}

/*
-- Decrements module currentUseCount
*/
void DSC_ModuleUnUse( P_DsmCoreInst idp, P_Module pModule )
{
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

   /* -- Should never underrun */
   dsmAssert((pModule->currentUseCount > 0));

   /* -- Detect/prevent underrun errors in retail */
   if (pModule->currentUseCount > 0)
   {
      pModule->currentUseCount--;
   }
}

/**
 * @brief   Start aquisition of module
 */
E_DscError DSC_ModuleAcquireStart( P_DsmCoreInst idp,
   P_Module pModule, E_SFPriority sfPriority )
{
   S_SfTarget target;
   P_DataCarousel pDC;
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));
   pDC = LLParent(pModule, DC_MODULE_LIST);
   dsmAssert((pDC != NULL));
   if (LLInsertHead( pDC->llcModuleAcquires, pModule ))
   {
      target.kind = SFK_DDB;
      target.id = pModule->moduleInfo.moduleId;
      target.associationTag = pModule->moduleInfo.associationTag;
      target.serviceId = (target.associationTag == INVALID_ASSOCIATION_TAG) ?
         DSC_DataCrslGetPid(pDC) : DSC_DataCrslGetServiceId(pDC);
      target.u.pModule = pModule;
      dsmDP3(("INFO: Load Module Id: %u pModule = 0x%p [timeout = %u]\n",
              pModule->moduleInfo.moduleId, pModule,
              pModule->moduleInfo.u.mhgp.moduleTimeout));
      /* -- Module/DDB not already being acquired, so start section filter */
      err = DSC_SectionFilterStart( idp, &target, sfPriority, &pModule->pDdbSf );
      if (err == CLDSM_ERR_SI_QUERY_FAILED)
      {
         DBGLOG((DD_SF|DD_MOD), " Treating query failure as OK")
         err = CLDSM_OK;
      }
   }
   else
   {
      /* -- Module/DDB already being acquired so update filter priority */
      err = DSC_SsectionFilterUpdatePriority( idp, pModule->pDdbSf, sfPriority, /*latchHighest*/ TRUE );
   }
   if (err)
   {
      LLRemove( pModule, MODULE_ACQUIRE_LIST );
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: ocModuleAcquireStart %u\n", err)));
   return err;
}

/**
 * @brief   Restart aquisition of module
 */
E_DscError DSC_ModuleAcquireRestart( P_DsmCoreInst idp, P_Module pModule )
{
   S_SfTarget target;
   E_SFPriority sfPriority;
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

   if (pModule->pDdbSf != NULL)
   {
      target = pModule->pDdbSf->target;
      sfPriority = pModule->pDdbSf->filter.priority;
      DSC_SectionFilterStop( idp, &pModule->pDdbSf );
   }
   else
   {
      P_DataCarousel  pDC = LLParent(pModule, DC_MODULE_LIST);
      dsmAssert((pDC != NULL));
      target.kind = SFK_DDB;
      target.id = pModule->moduleInfo.moduleId;
      target.associationTag = pModule->moduleInfo.associationTag;
      target.serviceId = (target.associationTag == INVALID_ASSOCIATION_TAG) ?
         DSC_DataCrslGetPid(pDC) : DSC_DataCrslGetServiceId(pDC);
      target.u.pModule = pModule;
      sfPriority = DSC_ModulePriority(pModule);
   }
   err = DSC_SectionFilterStart( idp, &target, sfPriority, &pModule->pDdbSf );
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: DSC_ModuleAcquireRestart %u\n", err)));
   return err;
}

/**
 * @brief   Stop aquisition of module
 */
void DSC_ModuleAcquireStop( P_DsmCoreInst idp, P_Module pModule )
{
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

   LLRemove( pModule, MODULE_ACQUIRE_LIST );

   /* -- Stop DDB section filter (also Nulls reference in Module) */
   dsmAssert((pModule->pDdbSf));
   DSC_SectionFilterStop( idp, &pModule->pDdbSf );
}

/* /////////////////////////////////////////////////////////////////////////////
//
///////////////////////////////////////////////////////////////////////////// */
void DSC_ModuleDelete( P_DsmCoreInst idp, P_Module pModule )
{
   P_DataCarousel pDC;
   BOOLEAN inList;

   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

   pDC = LLParent( pModule, DC_MODULE_LIST );
   dsmAssert((pDC != NULL));

   DBGLOG(DD_MOD, "pModule->status=%d", pModule->status)
   /* -- Determine which related section filters etc. to stop */
   switch (pModule->status)
   {
      case MS_INITIAL:
         /* -- Nothing to do */
         break;

      case MS_PENDING_INFO:
         /* -- DII request must be made */
         DSC_DataCrslAcquireStop( idp, pDC, SF_PRIORITY_HIGH );
         break;

      case MS_PENDING_DDB:
         /* -- DII monitor must be active */
         DSC_DataCrslAcquireStop( idp, pDC, SF_PRIORITY_LOW );

         /*
         -- If module is in acquire list then stop acquisition.
         -- NB. Module is only put in acquire list when section filter
         --     start (or resulting SI query start) is successful.
         */
         inList = LLCheckInListId( MODULE_ACQUIRE_LIST, pModule );
         if (inList)
         {
            DSC_ModuleAcquireStop( idp, pModule );
         }
         break;

      case MS_BUILDING:
         /* -- DII monitor must be active and section filter
            -- started so stop them */
         DSC_DataCrslAcquireStop( idp, pDC, SF_PRIORITY_LOW );
         DSC_ModuleAcquireStop( idp, pModule );
         break;

      case MS_CACHED:
      case MS_ACQ_ABORTED:
         /* TODO: -- Only remove DII monitor here if module is
                     transparently cached ? */
         DSC_DataCrslAcquireStop( idp, pDC, SF_PRIORITY_LOW );
         break;

      case MS_ACQ_FAILED: /* -> Transient state */
      case MS_EXPIRED:  /* -> Module already deleted! */
      default:
         /* -- This should not be called if module is in any other state */
         dsmAssert((0));
         break;
   }

   /* -- Unlink module from any remaining lists */
   LLRemoveFromAll( pModule, NUM_LISTS_MODULE );

   /* -- Abort any loadRequests waiting on this module */
   /* -- NB. Must do this here because hOC must be finished with in case
      --     it is unloaded/destroyed from notify callback */
   DSC_ModuleAbortLoadRequests( idp, pModule );

   /* -- Delete/destroy module */
   if (pModule->loadedCount > 0)
   {
      /* -- When this function is called, any loaded modules must be in
         -- cached state */
      dsmAssert((pModule->status == MS_CACHED));
      pModule->status = MS_EXPIRED;
      LLInsertHead( idp->llcExpiredModules, pModule );
   }
   else
   {
      DSC_ModuleDestroy( idp, pModule );
   }
}

void DSC_ModuleDeleteDcTidyUp( P_DsmCoreInst idp, P_Module pModule )
{
   P_RootLoadRqst pLoadRqst;
   P_DataCarousel pDC;
   pLoadRqst = pModule->pLoadRqst;
   pDC = LLParent( pModule, DC_MODULE_LIST );
   DSC_ModuleDelete( idp, pModule );
   if (pLoadRqst)
   {
      DSC_LoadRsqtFinalNotify( idp, pLoadRqst, LRS_ABORTED_LOAD_ERROR );
   }
   else if (!LLCount(pDC->llcDcModules))
   {
      DSC_DataCrslDestroy( idp, pDC );
   }
}

/*
 *
 */
E_DscError DSC_ModuleDataRefresh( P_DsmCoreInst idp, P_Module pModule )
{
   E_DscError err;
   P_DataCarousel pDC;

   /* -- Update cache status of object's module according to
      -- module's currently set caching rules */
   /* -- Module must not have an active section filter */
   dsmAssert((pModule->pDdbSf == NULL));

   /* -- Is there any outstanding load requests on module, or recently loaded request */
   if (pModule->llcLoadRequests != NULL ||
      (pModule->cachedTime + CACHE0_TIMEOUT_MS) > STB_OSGetClockMilliseconds())
   {
      /* special case where related files have been just loaded on same module with CCP0 */
      pModule->cachingRules = CACHE_RULES_DEFAULT;

      DBG3(DD_MOD,"Up-to-date module %p, status %u, upd %u, now %u",
         pModule, pModule->status, pModule->cachedTime, STB_OSGetClockMilliseconds())

      err = CLDSM_DUPLICATE_REQUEST;
   }
   else
   {
      DBG2(DD_MOD,"REFRESH module %p, status %u, lc %u, upd %u, now %u",
         pModule, pModule->status, pModule->loadedCount, pModule->cachedTime, STB_OSGetClockMilliseconds())

      pDC = (P_DataCarousel)LLParent( pModule, DC_MODULE_LIST );

      dsmAssert((pModule->moduleInfo.blockSize == pDC->diiInfo.blockSize));
      dsmAssert((pModule->moduleInfo.associationTag == pDC->tap.associationTag));

      if (pModule->pModuleBuilder)
      {
         builderDestroy( idp, &pModule->pModuleBuilder );
      }
      if (pModule->hCompModuleData)
      {
         dsmAssert((pModule->hCompModuleData != pModule->hModuleData));
         moduleDataDestroy( idp, &pModule->hCompModuleData );
      }
      if (pModule->hModuleData)
      {
         moduleDataDestroy( idp, &pModule->hModuleData );
      }
      pModule->decompressFailureCount = 0;
      pModule->cachingRules = CACHE_RULES_DEFAULT;
      pModule->status = MS_PENDING_DDB;

      if (DSC_DataCrslFirstModule(pDC) == pModule &&
          DSC_RootCrslSrgObjectReset(LLParent(pDC, OC_DATA_CAROUSEL_LIST)) == CLDSM_OK)
      {
         err = DSC_ModuleAcquireStart( idp, pModule, SF_PRIORITY_DIRECT );
      }
      else
      {
         err = DSC_ModuleAcquireStart( idp, pModule, SF_PRIORITY_CLEAR );
      }
   }
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_ModuleResetState
// NB. Does not alter any attached load requests, client request count
//     or value of module currentUseCount.
///////////////////////////////////////////////////////////////////////////// */
void DSC_ModuleResetState( P_DsmCoreInst idp,
   P_Module pModule, E_ModuleStatus reqdState )
{
   P_DataCarousel pDC;
   dsmDP3(("DSC_ModuleResetState()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));


   /* -- Check that the module has no loaded objects associated */
   dsmAssert((pModule->loadedCount == 0));

   switch (reqdState)
   {
      case MS_PENDING_INFO:

         /* -- Reset acquisitionFailureCount because module parameters will
            -- be re-acquired */
         pModule->decompressFailureCount = 0;
         memset( &pModule->moduleInfo, 0, sizeof(S_ModuleInfo) );

         pDC = (P_DataCarousel)LLParent( pModule, DC_MODULE_LIST );
         dsmAssert((pDC != NULL));
         pModule->moduleInfo.blockSize = pDC->diiInfo.blockSize;
         pModule->moduleInfo.associationTag = pDC->tap.associationTag;
         pModule->cachingRules = CACHE_RULES_DEFAULT;

         if (pModule->pModuleBuilder)
         {
            builderDestroy( idp, &pModule->pModuleBuilder );
         }
         if (pModule->hCompModuleData)
         {
            dsmAssert((pModule->hCompModuleData != pModule->hModuleData));
            moduleDataDestroy( idp, &pModule->hCompModuleData );
         }
         if (pModule->hModuleData)
         {
            moduleDataDestroy( idp, &pModule->hModuleData );
         }

         pModule->status = MS_PENDING_INFO;
         break;


      case MS_ACQ_FAILED:

         /* -- NB. Don't reset decompressFailureCount since module
            -- parameters are not being changed */

         pModule->cachingRules = CACHE_RULES_DEFAULT;

         if (pModule->pModuleBuilder)
         {
            builderDestroy( idp, &pModule->pModuleBuilder );
         }
         if (pModule->hCompModuleData)
         {
            dsmAssert((pModule->hCompModuleData != pModule->hModuleData));
            moduleDataDestroy( idp, &pModule->hCompModuleData );
         }
         if (pModule->hModuleData)
         {
            moduleDataDestroy( idp, &pModule->hModuleData );
         }

         pModule->status = MS_ACQ_FAILED;
         break;


      case MS_ACQ_ABORTED:

         /* -- NB. Don't reset decompressFailureCount since module
            -- parameters are not being changed */

         pModule->cachingRules = CACHE_RULES_DEFAULT;

         if (pModule->pModuleBuilder)
         {
            builderDestroy( idp, &pModule->pModuleBuilder );
         }
         if (pModule->hCompModuleData)
         {
            dsmAssert((pModule->hCompModuleData != pModule->hModuleData));
            moduleDataDestroy( idp, &pModule->hCompModuleData );
         }
         if (pModule->hModuleData)
         {
            moduleDataDestroy( idp, &pModule->hModuleData );
         }
         pModule->status = MS_ACQ_ABORTED;
         break;

      default:
         /* -- Illegal status */
         dsmDP1(("ERROR: Illegal requested module state = %u\n", reqdState));
         dsmAssert((0));
         break;
   }

   dsmDP3(("exit DSC_ModuleResetState\n"));
}

static void RemoveAnyDuplicate( P_DsmCoreInst idp, P_Module pModule )
{
   P_Module pModuleFromList, pNextModule;
   ListId_t moduleListId;
   P_DataCarousel pDC;
   pDC = (P_DataCarousel)LLParent( pModule, MODULE_ACQUIRE_LIST );
   if (pDC != NULL)
   {
      /* Is this the DC we want? */
      /* go through all MODULES */
      /* Get listId and first module in list from Control block */
      moduleListId = LListId( pDC->llcDcModules );
      pModuleFromList = (P_Module)LLHead( pDC->llcDcModules );

      /* Go through each module in list until the one with a matching moduleId
         is found */
      while (pModuleFromList)
      {
         pNextModule = LLNext( pModuleFromList, moduleListId );
         if (pModuleFromList != pModule &&
             pModuleFromList->moduleInfo.moduleId == pModule->moduleInfo.moduleId &&
             pModuleFromList->moduleInfo.version == pModule->moduleInfo.version)
         {
            DSC_ModuleDelete( idp, pModuleFromList );
         }
         pModuleFromList = pNextModule;
      }
   }
}

/*
-- Decompress a cached module
--
-- If decompress fails then request module to be re-acquired (until failure
-- limit reached when error generated).
--
*/
E_DscError DSC_ModuleDecompress( P_DsmCoreInst idp, P_Module pModule )
{
   E_DscError err = CLDSM_OK;
   P_ModuleData hDecompModuleData = NULL;

   dsmDP3(("decompressCachedModule()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

   /* -- Mark current module used since this function allocates cache
      -- memory and we don't want the module to be deleted! */
   DSC_ModuleUse( idp, pModule );

   dsmAssert((pModule->hCompModuleData != NULL));
   dsmAssert((pModule->hModuleData == NULL));

   err = moduleDecompress( idp,
         pModule->moduleInfo.moduleSize,
         pModule->moduleInfo.originalSize,
         pModule->hCompModuleData, &hDecompModuleData );

   if (!err)
   {
      if (hDecompModuleData != NULL)
      {
         /* -- Decompress successful */
         dsmDP3(("INFO: Module Id: %u decompressed\n",
                 pModule->moduleInfo.moduleId));
         moduleDataDestroy( idp, &pModule->hCompModuleData );
         pModule->hModuleData = hDecompModuleData;
         pModule->decompressFailureCount = 0;
      }
      else
      {
         dsmDP1(("ERROR: Module Id: %u Rcvd: 0x%x Size: %d decompression failure - ",
                 pModule->moduleInfo.moduleId, pModule->moduleInfo.blkRcvd, pModule->moduleInfo.moduleSize));
         /*
         -- This may be a 'transmission' error (v.unlikely) or
         -- a broadcast error (most likely - but will probably
         -- be present in every module/DDB set).
         -- Module is re-acquired one or more times (up to
         -- DSM_MAX_MODULE_ACQUISITION_FAILURES). If it was
         -- still not decompressed successfully an error
         -- is generated which causes acquisition to be aborted.
         */
         pModule->decompressFailureCount++;
         if (pModule->decompressFailureCount >=
             MAX_MODULE_DECOMPRESS_FAILURES)
         {
            /* -- Abort module */
            dsmDP1(("abort\n"));
            /* -- NB. This destroys erroneous compressed module data
               --     but does not alter decompressFailureCount */
            DSC_ModuleResetState( idp, pModule, MS_ACQ_ABORTED);
            err = CLDSM_ERR_MODULE_ACQUISITION_FAILURE;
         }
         else
         {
            dsmDP1(("continue\n"));
            /* -- NB. This destroys erroneous compressed module data
               --     but does not alter decompressFailureCount */
            DSC_ModuleResetState( idp, pModule, MS_ACQ_FAILED);
            /* signal need to re-try */
            err = CLDSM_ERR_END_OF_DATA;
         }
      }
   }
   /* -- Finished using current module */
   DSC_ModuleUnUse( idp, pModule );
   DBGERRCHK(err)
   return err;
}

static E_DscError ModuleUpdateSsu( P_DsmCoreInst idp, P_Module pModule )
{
   E_DscError err = CLDSM_OK;
   P_DataCarousel pDC;
   P_RootLoadRqst pLoadRqst;
   pDC = LLParent(pModule, DC_MODULE_LIST);
   pLoadRqst = pModule->pLoadRqst;
   dsmAssert((pModule->status == MS_CACHED));
   if (pModule->hModuleData == NULL)
   {
      err = DSC_ModuleDecompress( idp, pModule );
      if (err)
      {
         ERRLOG(DD_MOD,"Decompress failed modId=%x err=%d", pModule->moduleInfo.moduleId, err)
      }
   }
   if (pModule->hModuleData != NULL && pLoadRqst != NULL)
   {
      // TODO - could double check that pLoadRqst is in DC's list.
      dsmAssert((idp->setup.ssuFuncs.saveModuleData != NULL));
      dsmAssert((pLoadRqst->remaining >= pModule->moduleInfo.originalSize));

      idp->setup.ssuFuncs.saveModuleData( pLoadRqst->usrRef, MAKE_MODULE_REF(pDC->dataCarouselId, pModule->moduleInfo.moduleId),
         pModule->moduleInfo.u.ssup.offset, pModule->moduleInfo.originalSize, moduleDataPtr(pModule->hModuleData) );

      pModule->pLoadRqst = NULL;

      if (pLoadRqst->remaining < pModule->moduleInfo.originalSize)
      {
         DSC_LoadRsqtAbort( idp, pLoadRqst );
      }
      else
      {
         pLoadRqst->remaining -= pModule->moduleInfo.originalSize;
         if (pLoadRqst->remaining == 0)
         {
            DSC_LoadRsqtFinalNotify( idp, pLoadRqst, LRS_LOADED );
         }
      }
   }
   DBGERRCHK(err)
   return err;
}

#include "defMemUtilsContig.h"  /* -- Default mem type for function */

/* /////////////////////////////////////////////////////////////////////////////
// DSC_ModuleUpdate
// Builds modules given DDB data section. pCompleted returns true if module
// completed with this DDB data section.
///////////////////////////////////////////////////////////////////////////// */
E_DscError DSC_ModuleUpdate( P_DsmCoreInst idp,
   P_Module pModule, U8BIT *pDDBMsgPayload, U16BIT ddbMsgPayloadLen )
{
   U8BIT moduleVersion;
   U16BIT ddbBlockSize;
   U16BIT blockNum;
   E_DscError err = CLDSM_OK;
   P_ModuleInfo pModInfo;
   P_ModuleData hBuiltModuleData = NULL;

   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));
   dsmAssert((pDDBMsgPayload != NULL));
   dsmAssert((ddbMsgPayloadLen >= DDB_MSG_PAYLOAD_HDR_LEN));

   pModInfo = &pModule->moduleInfo;

   /* -- Parse DDB data section to extract module version, blockNumber */

   /* -- pDDBMsgPayload -> moduleId */

   /* -- Skip moduleId */
   SET_POS_REL( pDDBMsgPayload, 2 );

   /* -- Read moduleVersion */
   READ_UINT8( pDDBMsgPayload, moduleVersion );

   /* -- Skip reserved data */
   SET_POS_REL( pDDBMsgPayload, 1 );

   /* -- Calculate blockSize */
   ddbBlockSize = (U16BIT)(ddbMsgPayloadLen - DDB_MSG_PAYLOAD_HDR_LEN);

   if (moduleVersion != pModInfo->version)
   {
      /*
      -- Different module version received before updated DII acquired,
      -- Discard unexpected DDB version since we cannot use it until
      -- updated DII info arrives (current state of module build data
      -- will then be destroyed and module acquistion restarted for
      -- new module version).
      */
      dsmDP3(("INFO: Module Id: %u version change at DDB (ver expd: %u, rcvd: %u) - DDB discarded\n",
              pModInfo->moduleId, pModInfo->version, moduleVersion));
   }
   else if (ddbBlockSize > pModInfo->blockSize) /* check blockSize is legal */
   {
      dsmDP1(("DATA ERROR: DDB Block size != DII Block size: %u, %u\n",
              ddbBlockSize, pModInfo->blockSize));
   }
   else
   {
      /* -- Create module builder if necessary */
      if (pModule->pModuleBuilder == NULL)
      {
         err = builderCreate( idp, pModInfo->moduleSize,
               pModInfo->blockSize, &pModule->pModuleBuilder );
         pModule->status = MS_BUILDING;
      }
      if (err)
      {
         DSC_ModuleDeleteDcTidyUp( idp, pModule );
      }
      else
      {
         /* -- Read blockNumber */
         READ_UINT16( pDDBMsgPayload, blockNum );

         dsmDP3(("Mod %d: Blk=%d sz=%d rcvd=%x\n", pModInfo->moduleId, blockNum, ddbBlockSize, pModInfo->blkRcvd));
      #ifndef NDEBUG
         pModInfo->blkRcvd |= 1 << blockNum;
      #endif
         /* -- pDDBMsgPayload -> first blockDataByte */

         /* -- Send DDB data to module builder */
         builderConstruct( idp, pModule->pModuleBuilder, blockNum,
            ddbBlockSize, pDDBMsgPayload, &hBuiltModuleData );

         P_ModuleBuilder pModuleBuilder = pModule->pModuleBuilder;
         if ((NULL != pModule->pLoadRqst) && (0 != pModuleBuilder->numBlocks))
         {
            U8BIT percent = ((pModuleBuilder->numBlocks - pModuleBuilder->blocksRqd) * 100)/ pModuleBuilder->numBlocks;
            if (percent > 100)
               percent = 100;
            
            pModule->pLoadRqst->percent = percent;
            DSC_LoadRsqtNotify (idp, pModule->pLoadRqst, LRS_LOADING);
         }

         if (hBuiltModuleData)
         {
            /* -- Module completed */
            builderDestroy( idp, &pModule->pModuleBuilder );
            if (pModInfo->crslMagic == UC_MAGIC &&
                pModInfo->u.ssup.moduleCrc != 0)
            {
               U32BIT crc = CDSM_UtilCalculateCRC(moduleDataPtr(hBuiltModuleData),pModInfo->moduleSize);
               if (crc != pModInfo->u.ssup.moduleCrc)
               {
                  ERRLOG(DD_MOD,"DATA ERROR: crc MIS-MATCH: %u, %u", crc, pModInfo->u.ssup.moduleCrc)
                  err = CLDSM_ERR_END_OF_DATA;
               }
            }
            if (!err && pModInfo->compressed)
            {
               dsmAssert((pModule->hModuleData == NULL));
               dsmAssert((pModule->hCompModuleData == NULL));
               pModule->hCompModuleData = hBuiltModuleData;
               if (pModule->llcLoadRequests)
               {
                  err = DSC_ModuleDecompress( idp, pModule );
               }
            }
            else
            {
               dsmAssert((pModule->hModuleData == NULL));
               pModule->hModuleData = hBuiltModuleData;
            }
            if (err == CLDSM_ERR_END_OF_DATA)
            {
               /* Decompression failure so re-acquire module.
                * NB. Module acquisition (section filter) is still active so no
                * need to call DSC_ModuleAcquireStart() */
               pModule->status = MS_PENDING_DDB;
               err = CLDSM_OK;
            }
            else if (err)
            {
               /* -- Error decompressing module. Stop acquiring module (remove section filter ) */
               DSC_ModuleAcquireStop( idp, pModule );
               DSC_ModuleAbortLoadRequests( idp, pModule );
            }
            else
            {
               pModule->cachedTime = STB_OSGetClockMilliseconds();
               pModule->status = MS_CACHED;
               DSC_ModuleAcquireStop( idp, pModule );
               RemoveAnyDuplicate( idp, pModule );
               if (pModInfo->crslMagic == UC_MAGIC)
               {
                  ModuleUpdateSsu( idp, pModule );
               }
               else if (pModInfo->crslMagic == OC_MAGIC)
               {
                  /* Object Carousel data */
                  err = lmUpdateModule( idp, pModule );
               }
            }
         }
         else
         {
            sectionTimerUpdate(idp, pModule->pDdbSf, FALSE, FALSE);
         }
      }
   }
   DBGERRCHK(err)
   return err;
}

/* Module List Functions */

/* /////////////////////////////////////////////////////////////////////////////
// DSC_ModuleListFindById
// Returns NULL in phModule if moduleId not found
///////////////////////////////////////////////////////////////////////////// */
P_Module DSC_ModuleListFindById( P_LLControl pModuleList, U16BIT moduleId )
{
   P_Module pModuleFromList;
   ListId_t listId;

   /* Get listId and first module in list from Control block */
   listId = LListId( pModuleList );
   pModuleFromList = (P_Module)LLHead( pModuleList );

   /* Go through each module in list until the one with a matching moduleId
      is found */
   while (pModuleFromList)
   {
      /* Is this the module we want? */
      if (pModuleFromList->moduleInfo.moduleId == moduleId)
      {
         /* Found the module */
         break;
      }
      pModuleFromList = LLNext( pModuleFromList, listId );
   }
   dsmDP3(("exit DSC_ModuleListFindById: %p\n", pModuleFromList));
   return pModuleFromList;
}

/*------------------------------ Local Functions -----------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// builderCreate
// Creates an instance of the ModuleBuilder struct, and initialises it.
///////////////////////////////////////////////////////////////////////////// */
static E_DscError builderCreate( P_DsmCoreInst idp,
   U32BIT moduleSize, U16BIT blockSize,
   P_ModuleBuilder *ppModuleBuilder )
{
   P_ModuleBuilder pModuleBuilder = NULL;
   E_DscError err;
   P_ModuleData hModuleData = NULL;
   U16BIT loop;

   dsmDP3(("builderCreate( %u, %u, %p )\n",
           moduleSize, blockSize, ppModuleBuilder));
   dsmAssert((idp != NULL));
   dsmAssert((moduleSize > 0));
   dsmAssert((blockSize > 0));
   dsmAssert((ppModuleBuilder != NULL));

   /* -- Allocate store for moduleBuilder records */
   pModuleBuilder = (P_ModuleBuilder)DSC_CmMemGet( idp, sizeof(S_ModuleBuilder) );
   if (!pModuleBuilder)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      /* -- Allocate store for broadcast module data */
      err = moduleDataCreate( idp, moduleSize, &hModuleData );
      if (err)
      {
         DSC_CmMemRelease( idp, pModuleBuilder );
         pModuleBuilder = NULL;
      }
      else
      {
         pModuleBuilder->blockSize = blockSize;
         pModuleBuilder->downloadSize = moduleSize;

         /* -- Calculate number of data blocks (DDBs) in the module */
         pModuleBuilder->numBlocks =
            (U16BIT)((moduleSize + blockSize - 1) / blockSize);

         pModuleBuilder->blocksRqd = pModuleBuilder->numBlocks;

         if (pModuleBuilder->numBlocks > MAX_BLOCKS_IN_MODULE)
         {
            /* -- FATAL ERROR */
            err = CLDSM_ERR_MODULE_TOO_LARGE;
            moduleDataDestroy( idp, &hModuleData );
            DSC_CmMemRelease( idp, pModuleBuilder );
            pModuleBuilder = NULL;
         }
         else
         {
            /* -- Attach moduleData store to moduleBuilder */
            pModuleBuilder->hModuleData = hModuleData;

            /* -- Clear register of which data blocks have been received */
            loop = (pModuleBuilder->numBlocks + 7) >> 3;
            while (loop--)
            {
               pModuleBuilder->blockRcvdRegister[loop] = 0x00;
            }
            err = CLDSM_OK;
         }
      }
   }

   *ppModuleBuilder = pModuleBuilder;

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: builderCreate: %u\n", err)));
   dsmDP3(("exit builderCreate -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// builderDestroy
// Destroys an instance of the ModuleBuilder struct, and associated data.
///////////////////////////////////////////////////////////////////////////// */
static void builderDestroy( P_DsmCoreInst idp, P_ModuleBuilder *ppModuleBuilder )
{
   P_ModuleBuilder pModuleBuilder = NULL;

   dsmDP3(("builderDestroy( %p )\n", ppModuleBuilder));
   dsmAssert((idp != NULL));
   dsmAssert((ppModuleBuilder != NULL));
   dsmAssert((*ppModuleBuilder != NULL));

   pModuleBuilder = *ppModuleBuilder;
   if (pModuleBuilder->hModuleData)
      moduleDataDestroy( idp, &pModuleBuilder->hModuleData );
   DSC_CmMemRelease( idp, *ppModuleBuilder );
   *ppModuleBuilder = NULL;

   dsmDP3(("exit builderDestroy\n"));
}

#include "defMemUtilsMgd.h"  /* -- Default mem type for function */

/* /////////////////////////////////////////////////////////////////////////////
// builderConstruct
// Assembles a Module. pDDBData must point to the start of the DDB Block Data.
// ddbDataSize is the size in bytes of the Block Data. blockNum starts from 0.
///////////////////////////////////////////////////////////////////////////// */
static void builderConstruct( P_DsmCoreInst idp, P_ModuleBuilder pModuleBuilder,
   U16BIT blockNum, U16BIT ddbDataSize, U8BIT *pDDBData, P_ModuleData *phModuleData )
{
   MemPtr mpModuleData;
   U32BIT offset;
   U16BIT regByteNum;
   U8BIT regBitPos;

   dsmDP3(("builderConstruct( %p, %u, %u, %p, %p )\n",
           pModuleBuilder, blockNum, ddbDataSize, pDDBData, phModuleData));
   dsmAssert((pModuleBuilder != NULL));
   dsmAssert((pDDBData != NULL));
   dsmAssert((phModuleData != NULL));

   offset = blockNum * pModuleBuilder->blockSize;
   if ((blockNum < pModuleBuilder->numBlocks) &&
       (offset + ddbDataSize <= pModuleBuilder->downloadSize))
   {
      regByteNum = (U16BIT)(blockNum >> 3);
      regBitPos = (U8BIT)(0x01 << (blockNum & 0x0007));

      if (!(pModuleBuilder->blockRcvdRegister[regByteNum] & regBitPos))
      {
         /* -- Have not got this block yet - so build into moduleData */
         MEMPTR_SEQ_OPEN( MEM_CONTEXT, moduleDataPtr(pModuleBuilder->hModuleData), offset,
            pModuleBuilder->downloadSize - offset, FALSE, mpModuleData );

         MEMPTR_WRITE( pDDBData, mpModuleData, ddbDataSize );

         MEMPTR_SEQ_CLOSE( MEM_CONTEXT, pModuleBuilder->hModuleData,
            mpModuleData );

         /* -- Mark block as received */
         pModuleBuilder->blockRcvdRegister[regByteNum] |= regBitPos;
         pModuleBuilder->blocksRqd--;

         if (pModuleBuilder->blocksRqd == 0)
         {
            /* -- Module is now completed */

            /* -- Pass moduleData handle back to calling routine */
            *phModuleData = pModuleBuilder->hModuleData;

            /* -- NULL builder copy of moduleData handle */
            pModuleBuilder->hModuleData = NULL;
         }
      }
   }
   else
   {
      /* -- Discard erroneous module block */
      dsmDP1(("DATA ERROR: builderConstruct (Current blockNum > Total numBlocks): %u, %u\n",
              blockNum, pModuleBuilder->numBlocks));
      dsmDP1(("OR: (offset + ddbDataSize > pModuleBuilder->downloadSize): %u, %u, %u\n",
              offset, ddbDataSize, pModuleBuilder->downloadSize));
   }
   dsmDP3(("exit builderConstruct\n"));
}

E_DscError DSC_ModuleAddLoadRequest( P_DsmCoreInst idp, P_Module pModule, H_Object hLoadRqst )
{
   E_DscError err;
   dsmAssert((pModule != NULL));

   if (pModule->llcLoadRequests != NULL)
   {
      err = CLDSM_OK;
   }
   else
   {
      err = LLCreate( idp, pModule, MODULE_LOAD_REQUEST_LIST, &pModule->llcLoadRequests );
   }
   if (!err)
   {
      LLInsertHead( pModule->llcLoadRequests, hLoadRqst );
      switch (DSC_LoadRsqtPriority((P_RootLoadRqst)hLoadRqst))
      {
         case SF_PRIORITY_HIGH:
         {
            pModule->highPriorityCount++;
            break;
         }
         case SF_PRIORITY_DIRECT:
         {
            pModule->directPriorityCount++;
            break;
         }
         default:;
      }
   }
   return err;
}

BOOLEAN DSC_ModuleRemoveLoadRequest( P_DsmCoreInst idp, H_Object hLoadRqst )
{
   P_Module pModule;
   BOOLEAN isEmpty;

   pModule = (P_Module)LLParent( hLoadRqst, MODULE_LOAD_REQUEST_LIST );
   isEmpty = LLRemove( hLoadRqst, MODULE_LOAD_REQUEST_LIST );
   if (memValidate(pModule))
   {
      switch (DSC_LoadRsqtPriority((P_RootLoadRqst)hLoadRqst))
      {
         case SF_PRIORITY_HIGH:
         {
            dsmAssert((pModule->highPriorityCount > 0));
            pModule->highPriorityCount--;
            break;
         }
         case SF_PRIORITY_DIRECT:
         {
            dsmAssert((pModule->directPriorityCount > 0));
            pModule->directPriorityCount--;
            break;
         }
         default:;
      }
   }
   if (isEmpty)
   {
      /* -- Module loadRequest list is now empty so destroy it */
      LLDestroy( idp, &pModule->llcLoadRequests );
   }
   return isEmpty;
}

void DSC_ModuleAbortLoadRequests( P_DsmCoreInst idp, P_Module pModule )
{
   P_RootLoadRqst pLoadRqst;

   dsmAssert((idp != NULL));
   dsmAssert((pModule != NULL));

   if (pModule->llcLoadRequests)
   {
      pLoadRqst = LLRemoveHead( pModule->llcLoadRequests );
      while (pLoadRqst)
      {
         switch (DSC_LoadRsqtPriority(pLoadRqst))
         {
            case SF_PRIORITY_HIGH:
            {
               dsmAssert((pModule->highPriorityCount > 0));
               pModule->highPriorityCount--;
               break;
            }
            case SF_PRIORITY_DIRECT:
            {
               dsmAssert((pModule->directPriorityCount > 0));
               pModule->directPriorityCount--;
               break;
            }
            default:;
         }

         DSC_LoadRsqtAbort( idp, pLoadRqst );

         /* -- Get next loadRequest */
         pLoadRqst = LLRemoveHead( pModule->llcLoadRequests );
      }

      /* -- module loadRequest list is now empty so destroy */
      LLDestroy( idp, &pModule->llcLoadRequests );
   }
}

E_SFPriority DSC_ModulePriority( P_Module pModule )
{
   return (pModule->directPriorityCount) ? SF_PRIORITY_DIRECT :
          (pModule->highPriorityCount) ? SF_PRIORITY_HIGH : SF_PRIORITY_LOW;
}

/*----------------------------------------------------------------------------*/
