/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header to the 'module' module - Functions/methods for
 *             creating/destroying and managing attributes of DSM-CC modules.
 * @file    module.h
 * @date    28/9/2001
 * @author  R Taylor
 */
#ifndef _MODULE_H_
#define _MODULE_H_


/*--------------------------------  Includes  --------------------------------*/
#include "clDsmSystem.h"

#include "linkList.h"
#include "sectionFilter.h"
#include "defMemUtilsMgd.h"  /* -- Default mem type for module */

#ifdef __cplusplus
extern "C" {
#endif


/*-----------------------------  Exported Macros  ----------------------------*/


/*------------------------------  Exported Types  ----------------------------*/

/* -- NB. This is only here to allow debug checking of size from clDsmMain */
typedef struct
{
   U16BIT numBlocks;
   U16BIT blocksRqd;
   U32BIT downloadSize;
   U16BIT blockSize;
   U8BIT blockRcvdRegister[(MAX_BLOCKS_IN_MODULE + 7) / 8];    /* -- round up */
   P_ModuleData hModuleData;
} S_ModuleBuilder, *P_ModuleBuilder;


typedef enum
{
   MS_INITIAL,          /* -- transient state */
   MS_PENDING_INFO,     /* -- waiting for DII info */
   MS_PENDING_DDB,      /* -- DII info available, waiting for first DDB */
   MS_BUILDING,         /* -- first DDB received */
   MS_CACHED,           /* -- module complete (compressed or decompressed) */
   MS_ACQ_FAILED,       /* -- acquisition failed due to decompress failure
                           -- (transient state)*/
   MS_ACQ_ABORTED,      /* -- acquisition aborted due to multiple decompress
                           -- failures */
   MS_EXPIRED           /* -- module stale (out of date) but still in use (ie.
                           -- has loaded object(s)) */
} E_ModuleStatus;


/* -- MHEG, MHP and HBBTV profile */
typedef struct
{
   U32BIT moduleTimeout;
   U32BIT blockTimeout;
   U32BIT minBlockTime;
   U16BIT tapBiopId;
   U8BIT priority;
   U8BIT transparency;
} S_MHGProfile;

/* -- SSU profile */
typedef struct
{
   U32BIT moduleCrc;
   U32BIT offset;
   U32BIT infoLang;
   U16BIT nextModuleId;
   U8BIT positionType; /* for values see E_PostionType */
   U8BIT moduleType;   /* for values see E_ModuleType */
   U8BIT typeLen;
   U8BIT nameLen;
   U8BIT infoLen;
   MemPtr mpType;
   MemPtr mpName;
   MemPtr mpInfo;
} S_SSUProfile;

typedef struct
{
   U16BIT moduleId;
   U16BIT blockSize;
   U16BIT crslMagic;
   U16BIT associationTag;
   U8BIT version;
   U8BIT reserved;
   BOOLEAN compressed;
#ifndef NDEBUG
   U32BIT blkRcvd;
#endif
   U32BIT moduleSize;
   U32BIT originalSize;
   union {
      S_MHGProfile mhgp;
      S_SSUProfile ssup;
   } u;
} S_ModuleInfo, *P_ModuleInfo;


typedef struct s_Module
{
   S_LLObject llData[NUM_LISTS_MODULE];
   S_ModuleInfo moduleInfo;
   E_ModuleStatus status;

   /* -- DDB filter info */
   P_SecFilterInfo pDdbSf;

   U32BIT cachedTime;

   U16BIT currentUseCount;                          /* -- Limited by design */
   U16BIT decompressFailureCount;                   /* -- Limited by MAX_MODULE_DECOMPRESS_FAILURES */

   U16BIT loadedCount;                              /* -- Limited only by memory */
   U16BIT highPriorityCount;                        /* -- Limited only by memory */
   U16BIT directPriorityCount;                      /* -- Limited only by memory */

   /* -- NB. Module inherits 'highest priority' caching rules of
      -- any object loaded from the module */
   E_CacheRules cachingRules;

   P_ModuleBuilder pModuleBuilder;
   P_ModuleData hCompModuleData;
   P_ModuleData hModuleData;

   P_RootLoadRqst pLoadRqst;
   P_LLControl llcLoadRequests;
} S_Module;


/*------------------------------  Exported Data  -----------------------------*/


/*---------------------------  Exported Prototypes  --------------------------*/

E_DscError DSC_ModuleCreate( P_DsmCoreInst idp,
   U16BIT moduleId, P_Module *ppModule );

void DSC_ModuleDestroy( P_DsmCoreInst idp, P_Module pModule );

void DSC_ModuleUse( P_DsmCoreInst idp, P_Module pModule );

void DSC_ModuleUnUse( P_DsmCoreInst idp, P_Module pModule );

E_DscError DSC_ModuleAcquireStart( P_DsmCoreInst idp,
   P_Module pModule, E_SFPriority sfPriority );

void DSC_ModuleAcquireStop( P_DsmCoreInst idp, P_Module pModule );

/**
 * @brief   Destroy the module and destroy parent DC if now empty
 */
void DSC_ModuleDelete( P_DsmCoreInst idp, P_Module pModule );

/**
 * @brief   Delete module and destroy parent DC if now empty
 */
void DSC_ModuleDeleteDcTidyUp( P_DsmCoreInst idp, P_Module pModule );

void DSC_ModuleResetState( P_DsmCoreInst idp,
   P_Module pModule, E_ModuleStatus reqdState );

E_DscError DSC_ModuleDecompress( P_DsmCoreInst idp, P_Module pModule );

E_DscError DSC_ModuleDataRefresh( P_DsmCoreInst idp, P_Module pModule );

E_DscError DSC_ModuleUpdate( P_DsmCoreInst idp,
   P_Module pModule, U8BIT *pDDBMsgPayload, U16BIT ddbMsgPayloadLen );

/* Generic module list functions */
P_Module DSC_ModuleListFindById( P_LLControl pModuleList, U16BIT moduleId );

E_SFPriority DSC_ModulePriority( P_Module pModule );

E_DscError DSC_ModuleAddLoadRequest( P_DsmCoreInst idp, P_Module pModule, H_Object hLoadRqst );

BOOLEAN DSC_ModuleRemoveLoadRequest( P_DsmCoreInst idp, H_Object hLoadRqst );

void DSC_ModuleAbortLoadRequests( P_DsmCoreInst idp, P_Module pModule );

E_DscError DSC_ModuleAcquireRestart( P_DsmCoreInst idp, P_Module pModule );

/*----------------------------------------------------------------------------*/
#ifdef __cplusplus
}
#endif
#endif /* _MODULE_H_ */
