/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions/methods for accessing DSM-CC module data..
 * @file    moduleData.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "moduleData.h"

#include "cacheMgr.h"
#include "object.h"
#include "defMemUtilsMgd.h"  /* -- Default mem type for module */


/*------------------------------- Local Macros -------------------------------*/


/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/

typedef struct s_ModuleData
{
   U32BIT size;
   U16BIT open;
   U16BIT idel;
} S_ModuleData;

/*------------------------------- Local Statics ------------------------------*/



/*------------------- local prototypes/forward declarations ------------------*/


/*---------------------------- Exported Functions ----------------------------*/


/* /////////////////////////////////////////////////////////////////////////////
// moduleDataCreate
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError moduleDataCreate( P_DsmCoreInst idp,
   U32BIT dataSize, P_ModuleData *phModuleData )
{
   E_DscError err;

   dsmDP3(("moduleDataCreate()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((dataSize > 0));
   dsmAssert((phModuleData != NULL));

   *phModuleData = (P_ModuleData)DSC_CmMemGet( idp, sizeof(S_ModuleData) + dataSize );
   if (*phModuleData == NULL)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      err = CLDSM_OK;
      (*phModuleData)->size = dataSize;
      (*phModuleData)->open = 0;
      (*phModuleData)->idel = 0;
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: moduleDataCreate %u\n", err)));
   dsmDP3(("exit moduleDataCreate -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// moduleDataDestroy
//
///////////////////////////////////////////////////////////////////////////// */
void moduleDataDestroy( P_DsmCoreInst idp, P_ModuleData *phModuleData )
{
   dsmDP3(("moduleDataDestroy()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((phModuleData != NULL));
   dsmAssert((*phModuleData != NULL));

   if ((*phModuleData)->open)
   {
      (*phModuleData)->idel = 0xFFFF;
   }
   else
   {
      DSC_CmMemRelease( idp, *phModuleData );
   }
   *phModuleData = NULL;

   dsmDP3(("exit moduleDataDestroy\n"));
}

MemPtr moduleDataPtr(P_ModuleData pModuleData)
{
   dsmAssert((pModuleData != NULL));
   return ((MemPtr)pModuleData) + sizeof(S_ModuleData);
}

MemPtr moduleDataOpen(P_ModuleData pModuleData)
{
   dsmAssert((pModuleData != NULL));
   pModuleData->open++;
   return moduleDataPtr(pModuleData);
}

void moduleDataClose(P_DsmCoreInst idp, P_ModuleData pModuleData)
{
   dsmAssert((pModuleData != NULL));
   if (pModuleData->open)
   {
      pModuleData->open--;
   }
   if (pModuleData->open == 0 && pModuleData->idel == 0xFFFF)
   {
      DSC_CmMemRelease( idp, pModuleData );
   }
}


/* /////////////////////////////////////////////////////////////////////////////
// moduleDataFindObject
//
///////////////////////////////////////////////////////////////////////////// */
U32BIT moduleDataFindObject(
   /*I*/ P_ModuleData pModuleData, U32BIT dataLength, P_ObjectKey pObjectKey,
   /*O*/ MemPtr *mpObject )
{
   MemPtr mpObjectData;
   MemPos currPos;
   MemPos endPos;
   BOOLEAN valid = FALSE;
   U32BIT objectLength = 0;
   S_ObjectKey currObjectKey;
   U32BIT currObjectLength;
   BOOLEAN objKeysMatch;
   U8BIT *pObjKeyData;
   U8BIT *pCurrKeyData;

   dsmDP3(("moduleDataFindObject()\n"));
   dsmAssert((pModuleData != NULL));

   dsmAssert((pObjectKey != NULL));
   dsmAssert(((pObjectKey->length > 0) && (pObjectKey->length <= 4)));
   dsmAssert((mpObject != NULL));

   if (!dataLength)
   {
      DBG2(DD_GEN,"zero length")
   }

   /* -- Open MemPtr for accessing objects in module data */
   MEMPTR_OPEN( moduleDataPtr(pModuleData), mpObjectData );

   /* -- Determine end position of search data */
   GET_POS( mpObjectData, currPos );
   endPos = currPos + dataLength;

   /* -- mpObjectData -> first BIOP::<object>Message */

   while (currPos < endPos)
   {
      /* -- mpObjectData -> current BIOP::<object>Message */
      valid = objectDataGetKeyAndLen( mpObjectData, &currObjectKey,
            &currObjectLength );
      if (valid)
      {
         /*TODO: check sufficient data in module for object (length)?*/

         /*
         -- Check if objectKeysMatch
         --
         -- Uses switch to make it fast (& avoid function call overhead
         -- of memcmp). By far the most common case used is case 1.
         */

         /* -- Currently only used here - should put this functionality in
            -- a macro if required elsewhere */

         dsmAssert(((currObjectKey.length > 0) &&
                    (currObjectKey.length <= 4)));

         objKeysMatch = FALSE;

         if (currObjectKey.length == pObjectKey->length)
         {
            pObjKeyData = pObjectKey->data;
            pCurrKeyData = currObjectKey.data;

            switch (pObjectKey->length)
            {
               case 1:
                  if (*pCurrKeyData == *pObjKeyData)
                  {
                     objKeysMatch = TRUE;
                  }
                  break;

               case 2:
                  if (*(pCurrKeyData)++ == *(pObjKeyData)++)
                  {
                     if (*pCurrKeyData == *pObjKeyData)
                     {
                        objKeysMatch = TRUE;
                     }
                  }
                  break;

               case 3:
                  if (*(pCurrKeyData)++ == *(pObjKeyData)++)
                  {
                     if (*(pCurrKeyData)++ == *(pObjKeyData)++)
                     {
                        if (*pCurrKeyData == *pObjKeyData)
                        {
                           objKeysMatch = TRUE;
                        }
                     }
                  }
                  break;

               case 4:
                  if (*(pCurrKeyData)++ == *(pObjKeyData)++)
                  {
                     if (*(pCurrKeyData)++ == *(pObjKeyData)++)
                     {
                        if (*(pCurrKeyData)++ == *(pObjKeyData)++)
                        {
                           if (*pCurrKeyData == *pObjKeyData)
                           {
                              objKeysMatch = TRUE;
                           }
                        }
                     }
                  }
                  break;

               default:
                  break;
            }
         }

         if (objKeysMatch == TRUE)
         {
            SET_POS_ABS( *mpObject, currPos );
            objectLength = currObjectLength;
            break;
         }
         else
         {
            /* -- Skip current object data */
            SET_POS_REL( mpObjectData, currObjectLength );

            /* -- Find next (potential) object start position */
            GET_POS( mpObjectData, currPos );
         }
      }
      else
      {
         break;
      }
   }
   MEMPTR_CLOSE( mpObjectData );

   dsmDP3(("exit moduleDataFindObject -> rtn: %u\n", objectLength));
   return objectLength;
}

/*------------------------------ Local Functions -----------------------------*/


/*----------------------------------------------------------------------------*/
