/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Decompresses DSMCC module data using zlib library (also provides 'stack
 *             like' dynamic memory management for zlib).
 * @file    moduleDecompress.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h> /* for memset() */
#include "clDsmSystem.h"
#include "moduleDecompress.h"

#include "moduleData.h"
#include "zlib.h"

/*------------------------------- Local Macros -------------------------------*/

#define ZLIB_HEAP_NUM_BYTES (((U32BIT)1 << 15 /*MAX_WBITS*/) + (1024 * 32))
#define ZLIB_HEAP_NUM_PTRS(sz)  ((sz) + sizeof(voidpf) - 1) / sizeof(voidpf)

/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/

typedef voidpf Data;
typedef Data *P_Data;

typedef struct s_ZlibHeap
{
   P_Data ptr;
   P_Data top;
} S_ZlibHeap;

/*------------------------------- Local Statics ------------------------------*/


/*------------------- local prototypes/forward declarations ------------------*/

static voidpf dsmZcalloc( voidpf opaque, uInt items, uInt size );
static void dsmZcfree( voidpf opaque, voidpf address );


/*---------------------------- Exported Functions ----------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// moduleDecompressInit
//
///////////////////////////////////////////////////////////////////////////// */
void moduleDecompressInit( P_DsmCoreInst idp )
{
   P_ZlibHeap pZlibHeap;
   U32BIT size;
   dsmDP3(("moduleDecompressInit()\n"));
   dsmAssert((idp != NULL));

   size = ZLIB_HEAP_NUM_PTRS( ZLIB_HEAP_NUM_BYTES );

   pZlibHeap = (P_ZlibHeap)idp->setup.allocFunc( sizeof(S_ZlibHeap) + (size * sizeof(voidpf)) );
   idp->pZlibHeap = pZlibHeap;

   pZlibHeap->ptr = (voidpf)(pZlibHeap + 1);
   pZlibHeap->top = pZlibHeap->ptr + size;

   /* -- Output some useful info about the zlib heap */
   dsmDP3(("\nINFO: ZLIB HEAP = %p-%p, size %u bytes\n", pZlibHeap->ptr, pZlibHeap->top, ZLIB_HEAP_NUM_BYTES));
}

/* /////////////////////////////////////////////////////////////////////////////
// moduleDecompress
//
///////////////////////////////////////////////////////////////////////////// */
E_DscError moduleDecompress( P_DsmCoreInst idp,
   /*I*/ U32BIT compressedSize, U32BIT decompressedSize,
   /*IO*/P_ModuleData hCompModuleData,
   /*O*/ P_ModuleData *ppModuleData )
{
   E_DscError err;
   P_ModuleData pModuleData;
   MemSeqRef msCompModuleData, msDecompModuleData;
   U8BIT *pCompModuleData;
   U8BIT *pDecompModuleData;
   U32BIT compNumBytesContig, decompNumBytesContig;
   BOOLEAN decompressOk = TRUE;
   int zlibErr;
   z_stream zStream;

   dsmDP3(("moduleDecompress( %u, %u, %p )\n",
           compressedSize, decompressedSize, hCompModuleData));
   dsmAssert((idp != NULL));
   dsmAssert((hCompModuleData != NULL));
   dsmAssert((ppModuleData != NULL));
   dsmAssert((compressedSize > 0));


   *ppModuleData = NULL;

   if (compressedSize == 0)
   {
      /* -- Should never get passed a 0 compressed size */

      err = CLDSM_ERR_INTERNAL;
      goto _return;
   }

   if (decompressedSize == 0)
   {
      /*
      -- Decompressed module size of 0 may be possible (eg. for modules
      -- listed in a DII that are not actually used). Since we cannot
      -- have a 0 length data area, create one with a nominal length
      -- (of 1) instead.
      */
      err = moduleDataCreate( idp, /*nominal length*/ 1, &pModuleData );
   }
   else
   {
      err = moduleDataCreate( idp, decompressedSize, &pModuleData );
   }

   if (!err)
   {
      /* -- Open compressed and decompressed moduleData areas */
      CHECK_MEM_ERR(
         memSeqOpen( MEM_CONTEXT, (MemHandle)moduleDataPtr(hCompModuleData), 0, compressedSize,
            FALSE, &msCompModuleData )
         );
      CHECK_MEM_ERR(
         memSeqOpen( MEM_CONTEXT, (MemHandle)moduleDataPtr(pModuleData), 0, decompressedSize,
            FALSE, &msDecompModuleData )
         );

      zStream.opaque = (voidpf)idp;

      zStream.zalloc = (alloc_func)dsmZcalloc;
      zStream.zfree = (free_func)dsmZcfree;

      zStream.next_in = Z_NULL;
      zStream.avail_in = 0;
      zStream.next_out = Z_NULL;
      zStream.avail_out = 0;

      zlibErr = inflateInit(&zStream);
      DEBUG_CHK( zlibErr == Z_OK,
         dsmDP1(("ERROR: zlib inflateInit: %u\n", err)));

      #ifdef MEM_CONTIGUOUS
      /* Get input data area */
      memSeqAccessContig( msCompModuleData, &pCompModuleData, &compNumBytesContig );
      zStream.next_in = (Bytef *)pCompModuleData;
      zStream.avail_in = (uInt)compNumBytesContig;
      /* Get output data area */
      memSeqAccessContig( msDecompModuleData, &pDecompModuleData, &decompNumBytesContig );
      zStream.next_out = (Bytef *)pDecompModuleData;
      zStream.avail_out = (uInt)decompNumBytesContig;
      dsmDP3(("Pre-inflate stream -> n_i: %p, a_i: %u, n_o: %p, a_o: %u\n",
              zStream.next_in, zStream.avail_in, zStream.next_out, zStream.avail_out));

      /* ZLIB - inflate in one go! */
      zlibErr = inflate( &zStream, Z_FINISH );

      #else
      while (zlibErr == Z_OK)
      {
         if (zStream.avail_in == 0)
         {
            /* -- Get next contig 'block' of compressed data */
            memSeqAccessContig( msCompModuleData, &pCompModuleData,
               &compNumBytesContig );
            zStream.next_in = (Bytef *)pCompModuleData;
            zStream.avail_in = (uInt)compNumBytesContig;
         }
         if (zStream.avail_out == 0)
         {
            /* -- Get next contig 'block' in decompressed memory area */
            memSeqAccessContig( msDecompModuleData, &pDecompModuleData,
               &decompNumBytesContig );
            zStream.next_out = (Bytef *)pDecompModuleData;
            zStream.avail_out = (uInt)decompNumBytesContig;
         }
         dsmDP4(("Pre-inflate stream -> n_i: %p, a_i: %u, n_o: %p, a_o: %u\n",
                 zStream.next_in, zStream.avail_in, zStream.next_out, zStream.avail_out));
         zlibErr = inflate( &zStream, Z_SYNC_FLUSH );
         dsmDP4(("Post-inflate stream -> n_i: %p, a_i: %u, n_o: %p, a_o: %u\n",
                 zStream.next_in, zStream.avail_in, zStream.next_out, zStream.avail_out));
      }
      #endif

      if (zlibErr != Z_STREAM_END)
      {
         decompressOk = FALSE;
         dsmDP1(("DATA ERROR: ZLIB uncompress error: %d\n", zlibErr));
      }

      zlibErr = inflateEnd( &zStream );
      DEBUG_CHK( zlibErr == Z_OK,
         dsmDP1(("ERROR: zlib inflateEnd: %u\n", err)));

      /* -- Close moduleData areas */
      memSeqClose( msCompModuleData );
      memSeqClose( msDecompModuleData );

      if (decompressOk)
      {
         L2_DATA_CHK( zStream.total_out == decompressedSize,

            dsmDP1(("DATA ERROR: DII decompress size = %u, ",
                    decompressedSize));
            dsmDP1(("zlib decompress size = %u\n", zStream.total_out)),

            moduleDataDestroy( idp, &pModuleData );
            goto _return );

         /* -- Return decompressed moduleData area handle */
         *ppModuleData = pModuleData;
      }
      else
      {
         /* -- Destroy decompressed data area */
         moduleDataDestroy( idp, &pModuleData );
      }
   }

_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: moduleDecompress: %u\n", err)));
   dsmDP3(("exit moduleDecompress -> rtn: %u\n", err));
   return err;
}

/*------------------------------ Local Functions -----------------------------*/

/*
-- ZLIB HEAP OPERATION:
--
-- The zlib heap is treated as a simple stack. It works most efficiently if
-- items are freed in the reverse order from which they were allocated (which
-- appears to be valid for the way zlib allocates/frees memory).
--
-- Each allocated item is put on the current top of the heap. When an item
-- is freed, if it is not at the top of the heap it is marked as empty (by
-- setting it's blockSize to 0), if it is at the top of the heap it is removed
-- along with any items below it also marked as empty.
--
-- Memory block sizes are rounded up to an integer number of words.
-- Each memory block has the blockSize value stored at the start (bottom) and
-- memStartPtr stored at the end (top) with the allocated memory between these.
-- Actual heap amount used for each memory block is blockSize + 2.
--
-- Note that memStartPtr points to the start of the usable memory in the block
-- (ie. the address above where the blocksize is stored).
-- TODO: NK 16/10/01 - This is rather nasty and non-obvious - revisit.
*/
static voidpf dsmZcalloc( voidpf opaque, uInt items, uInt size )
{
   P_DsmCoreInst idp = (P_DsmCoreInst)opaque;
   P_ZlibHeap pZlibHeap = idp->pZlibHeap;
   U_PARAM blksize;
   voidpf retval;

   dsmDP4(("dsmZcalloc( %u, %u )\n", items, size));

#   ifndef NDEBUG
   /*** DEBUG TEST FOR WORST CASE ZLIB HEAP REQS
    *** zlib does not specify its worst case memory reqs exactly. Limited 'dumb'
    *** testing (ie. without attempting to understand zlib) on compressed DMC-CC
    *** modules has shown that the memory reqs vary according to the data being
    *** compressed. There is only one alloc which varies significantly in the
    *** amount of memory allocated. So far it has never been observed to allocate
    *** more than the size here! */
   if ((size == 0x4) && (items > 0x13c))
   {
      dsmDP1(("\n************************************\n"));
      dsmDP1(("************************************\n"));
      dsmDP1(("* ZLIB HEAP ITEMS > 0x13c: %x *\n", items));
      dsmDP1(("************************************\n"));
      dsmDP1(("************************************\n\n"));
   }

   if (size == 0)
   {
      dsmDP1(("\n************************************\n"));
      dsmDP1(("************************************\n"));
      dsmDP1(("* ZLIB HEAP, size == 0\n"));
      dsmDP1(("************************************\n"));
      dsmDP1(("************************************\n\n"));
   }

   if (items == 0)
   {
      dsmDP1(("\n************************************\n"));
      dsmDP1(("************************************\n"));
      dsmDP1(("* ZLIB HEAP, items == 0\n"));
      dsmDP1(("************************************\n"));
      dsmDP1(("************************************\n\n"));
   }
#   endif

   if (pZlibHeap == NULL)
   {
      /* -- Error - re-initialise zlib heap */
      dsmDP1(("ERROR: Local zlib heap error\n"));
      moduleDecompressInit( idp );
   }

   blksize.ptr = NULL;
   blksize.u32 = ZLIB_HEAP_NUM_PTRS(items * size);

   /* Make sure we are allocating a non-zero size block */
   if (blksize.u32 == 0)
   {
      dsmDP1(("ERROR: Local zlib attempting to allocate zero length block.\n"));
      retval = NULL;
   }
   else if ((pZlibHeap->ptr + blksize.u32 + 2) >= pZlibHeap->top)
   {
      /* -- Error - heap full */
      dsmDP1(("ERROR: Local zlib heap overflow\n"));
      retval = NULL;
   }
   else
   {
      *pZlibHeap->ptr++ = blksize.ptr;
      retval = (voidpf)pZlibHeap->ptr;

      pZlibHeap->ptr += blksize.u32;

      /* -- Clear memory block to zero */
      memset(retval, '\0', sizeof(Data) * blksize.u32);

      /* -- Store retval value at top of block */
      *pZlibHeap->ptr = retval;

      /* -- Point to free space ready for next block */
      pZlibHeap->ptr++;
   }

   dsmDP4(("exit dsmZcalloc -> rtn: %p\n", retval));
   return(retval);
}

static void dsmZcfree( voidpf opaque, voidpf address )
{
   P_DsmCoreInst idp = (P_DsmCoreInst)opaque;
   P_ZlibHeap pZlibHeap = idp->pZlibHeap;
   P_Data blockStart = ((P_Data)address) - 1;
   P_Data topBlockStart;

   dsmDP4(("dsmZcfree( %p )\n", address));
   dsmAssert((address != NULL));

   if (pZlibHeap->ptr <= (P_Data)(pZlibHeap+1))
   {
      /* -- Error - re-initialise zlib heap */
      dsmDP1(("ERROR: Local zlib heap underflow\n"));
      moduleDecompressInit( idp );
   }
   else if (blockStart < (P_Data)(pZlibHeap+1) || blockStart > (pZlibHeap->ptr - 2))
   {
      /* blockStart is not within used heap (nb. Zero length blocks allowed) */
      dsmDP1(("ERROR: Attempt to free illegal zlib heap address\n"));
   }
   else
   {
      /* -- Zero blockSize for this block to indicate it is empty/free */
      *blockStart = 0;

      /* -- Find start of current top stack block */
      topBlockStart = ((P_Data)(*(pZlibHeap->ptr - 1))) - 1;

      if (blockStart == topBlockStart)
      {
         /* -- This block is at top of heap so remove this and any empty/free blocks below */
         while (*topBlockStart == 0)
         {
            /* -- Remove top block */
            pZlibHeap->ptr = topBlockStart;

            if (pZlibHeap->ptr <= (P_Data)(pZlibHeap+1))
            {
               break;
            }
            /* -- Find start of new top block */
            topBlockStart = ((P_Data)(*(topBlockStart - 1))) - 1;
         }
      }
   }
}

/*----------------------------------------------------------------------------*/
