/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions/methods for creating/destroying and managing attributes of
 *             DSM-CC object carousel objects.
 * @file    object.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include <string.h>

#include "object.h"

#include "linkList.h"
#include "cacheMgr.h"
#include "clDsmUtils.h"
#include "defMemUtilsMgd.h"  /* -- Default mem type for module */


/*---constant definitions for this file--------------------------------------*/


/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/

/*---local function definitions----------------------------------------------*/

/*---global function definitions---------------------------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// objectDataGetKeyAndLen
// Gets the objectKey value and object length for an object.
// Returns TRUE if successful.
///////////////////////////////////////////////////////////////////////////// */
BOOLEAN objectDataGetKeyAndLen( const MemPtr mpObjectData,
   P_ObjectKey pObjectKey, U32BIT *pObjectLength )
{
   MemPtr mpObjectHdr;
   U32BIT messageSize;
   BOOLEAN keyValid;
   BOOLEAN valid = TRUE;
   U8BIT ui8 = 0;
   U16BIT ui16 = 0;
   U32BIT ui32 = 0;

   S_ObjectKey LocalObjectKey;

   /* -- stop compiler warnings when no checking */
   ui8 += 0; ui16 += 0; ui32 += 0;

   /* -- DP level 4 since called in a loop whenever a module is searched
      -- for an object */
   dsmDP4(("objectDataGetKeyAndLen()\n"));
   dsmAssert((mpObjectData != NULL));
   dsmAssert((pObjectKey != NULL));
   dsmAssert((pObjectLength != NULL));


   /* -- Open MemPtr for accessing object data */
   MEMPTR_OPEN( mpObjectData, mpObjectHdr );

   /* -- Skip magic, BIOP protocol version, byteOrder, messageType */
   SET_POS_REL( mpObjectHdr, 8 );

   /* -- Read messageSize */
   READ_UINT32( mpObjectHdr, messageSize );

   /* -- Calculate objectLength */
   *pObjectLength = messageSize + OBJECT_NUMBYTES_TO_MESSAGE_START;

   /* -- mpObjectHdr -> start of objectKey info */
   READ_OBJECT_KEY( mpObjectHdr, LocalObjectKey, keyValid );

   memcpy(pObjectKey, &LocalObjectKey, sizeof(S_ObjectKey));

   if (!keyValid)
   {
      dsmDP2(("DATA ERROR: Object data objectKey invalid\n"));
      valid = FALSE;
   }

   MEMPTR_CLOSE( mpObjectHdr );
   DEBUG_CHK( valid == TRUE,
      dsmDP1(("ERROR: objectDataGetKeyAndLen failure (invalid)\n")));
   dsmDP4(("exit objectDataGetKeyAndLen -> rtn: %u\n", valid));
   return valid;
}

/**
 * @brief   Parse the supplied object data.
 *          Verify selected fields.
 *          Extract generic information from selected fields.
 *          Store extracted information into structure supplied by calling function.
 *          Kind specific info is not extracted and stored at this point.
 * @param   mpObjectData The object data to be parsed.
 * @param   pObjInf The structure to be populated.
 * @return
 *          TRUE on success.
 *          FALSE on failure.
 */
BOOLEAN objectDataGetInfo(
   /*I*/ const MemPtr mpObjectData,
   /*O*/ ObjectDataInfo_t *pObjInf)
{
   MemPtr mpObjectHdr;
   U32BIT messageSize;
   BOOLEAN valid = FALSE;      /* Set true when all data read and stored OK */
   U8BIT ui8 = 0;
   U16BIT ui16 = 0;
   U32BIT ui32 = 0;
   U8BIT serviceContextList_count;
   U16BIT context_data_length;
   BOOLEAN keyValid;       /* Temporary variable stating when object key valid */
   U8BIT i;             /* General purpose 8-bit counter */

   /* -- stop compiler warnings when no checking */
   ui8 += 0; ui16 += 0; ui32 += 0;

   dsmDP3(("objectDataGetInfo()\n"));
   dsmAssert((mpObjectData != NULL));
   dsmAssert((pObjInf != NULL));

   /* -- Open MemPtr for accessing object data */
   MEMPTR_OPEN( mpObjectData, mpObjectHdr );

   /* -- magic = BIOP (DSM-CC) */
   ADV_UINT32_L2CHK( mpObjectHdr, ui32, ui32 == BIOP_MAGIC,
      dsmDP2(("DATA ERROR: Object data BIOP magic = %u\n", ui32)),
      goto _return );

   /* -- BIOP protocol version - major = 0x01, minor = 0x00 (DVB) */
   ADV_UINT16_L2CHK( mpObjectHdr, ui16, ui16 == 0x0100,
      dsmDP2(("DATA ERROR: Object data protocol version (!=0x0100) = %u\n", ui16)),
      goto _return );

   /* -- byteOrder = big endian (0x00) (DVB) */
   /* -- L1 check because if this is wrong, subsequent data will be garbaged */
   ADV_UINT8_L1CHK( mpObjectHdr, ui8, ui8 == 0x00,
      dsmDP2(("DATA ERROR: Object data byteOrder = %u\n", ui8)),
      goto _return );

   /* -- Skip messageType */
   SET_POS_REL( mpObjectHdr, 1 );

   /* -- Read messageSize */
   READ_UINT32( mpObjectHdr, messageSize );

   /* -- Calculate total objectLength */
   pObjInf->objectLen = messageSize + OBJECT_NUMBYTES_TO_MESSAGE_START;

   /* mpObjectHdr -> start of objectKey info */
   READ_OBJECT_KEY(mpObjectHdr, pObjInf->objectKey, keyValid);
   if (!keyValid)
   {
      dsmDP2(("DATA ERROR: Object data objectKey invalid\n"));
      goto _return;
   }

   /* -- objectKindLength = 4 (DVB/UK DTT) */
   /* -- L1 check because if this is wrong, subsequent data will be garbaged */
   ADV_UINT32_L1CHK( mpObjectHdr, ui32, ui32 == 4,
      dsmDP2(("DATA ERROR: Object data objectKindLength (!= 4) = %u\n", ui32)),
      goto _return );

   /* -- Read objectKindData */
   READ_UINT32( mpObjectHdr, pObjInf->objectKind );

   /* Read how many bytes of object information is in BIOP message */
   READ_UINT16(mpObjectHdr, pObjInf->objectInfoLength);

   /* Calculate offset of object information from start of message.
      The data pointed to is dependant on the object kind */
   MEMPTR_GET_DIFF(mpObjectData, mpObjectHdr, pObjInf->objectInfoOffset);

   /* Skip the object information because it is specific to object type.
      This will move us to the serviceContextList_count */
   SET_POS_REL(mpObjectHdr, (U32BIT)pObjInf->objectInfoLength);

   /* Read how many service contexts are in the message */
   READ_UINT8(mpObjectHdr, serviceContextList_count);

   /* Skip the service contexts */
   for (i = 0; i < serviceContextList_count; i++)
   {
      /* Skip context_id */
      SET_POS_REL(mpObjectHdr, 4);

      /* Read context_data_length */
      READ_UINT16(mpObjectHdr, context_data_length);

      /* Skip context (8-bit) data */
      SET_POS_REL(mpObjectHdr, context_data_length);
   }

   /* Read and store messageBodyLength */
   READ_UINT32(mpObjectHdr, pObjInf->messageBodyLength);

   /* Calculate and store messageBodyOffset */
   MEMPTR_GET_DIFF(mpObjectData, mpObjectHdr, pObjInf->messageBodyOffset);

   /* -- Check lengths consistent */
   L2_DATA_CHK( messageSize + OBJECT_NUMBYTES_TO_MESSAGE_START ==
      pObjInf->messageBodyOffset + pObjInf->messageBodyLength,
      dsmDP2(("DATA ERROR: Object data message lengths inconsistent\n")),
      goto _return );

   /* -- If we get here, assume we have recovered the required data */
   valid = TRUE;

_return:
   MEMPTR_CLOSE( mpObjectHdr );
   DEBUG_CHK( valid == TRUE,
      dsmDP1(("ERROR: objectDataGetInfo failure (invalid)\n")));
   dsmDP3(("exit objectDataGetInfo -> rtn: %u\n", valid));
   return valid;
}

/* /////////////////////////////////////////////////////////////////////////////
// odDirFindBinding
// If found:    pBinding set to required binding start & BindingLength valid
// Else:        pBinding set to end of data (or binding causing error) &
//                  BindingLength invalid
//
///////////////////////////////////////////////////////////////////////////// */
BOOLEAN odDirFindBinding(
   /*I*/ const MemPtr mpObject, pObjectDataInfo_t pObjInf, U8BIT *name,
   /*O*/ MemPtr *mpBinding )
{
   MemPtr mpBindingData;
   MemPos currStartPos;
   BOOLEAN found = FALSE;
   U16BIT bindingsCount;
   U8BIT idLength;
   U8BIT nameLength;
   U32BIT profilesCount;
   U8BIT ui8 = 0;
   U16BIT ui16 = 0;
   U32BIT ui32 = 0;
   /* -- stop compiler warnings when no checking */
   ui8 += 0; ui16 += 0; ui32 += 0;

   dsmDP3(("odDirFindBinding( %s )\n", name));
   dsmAssert((mpObject != NULL));
   dsmAssert((pObjInf != NULL));
   dsmAssert((name != NULL));
   dsmAssert(((pObjInf->objectKind == DIR_STR) ||
              (pObjInf->objectKind == SRG_STR)));
   dsmAssert((mpBinding != NULL));


   nameLength = (U8BIT)(strlen((char *)name ) + 1);  /* -- Add 1 for null terminator */
   dsmAssert((nameLength <= MAX_OBJ_NAME_SIZE));

   /* -- Open MemPtr for accessing binding data in object */
   MEMPTR_OPEN( mpObject, mpBindingData );

   /* -- mpBindingData -> BIOP::DirectoryMessage */

   /* -- Skip header info and go to first message body */
   SET_POS_REL( mpBindingData, (S32BIT)pObjInf->messageBodyOffset );

   /* -- Read bindingsCount */
   READ_UINT16( mpBindingData, bindingsCount );

   /* -- Loop through bindings until we find the right one */
   /* TODO: also check for hitting end of data? */
   while (bindingsCount--)
   {
      /* -- mpBindingData -> BIOP::Name */

      /* -- Remember start position of this binding */
      GET_POS( mpBindingData, currStartPos );

      /* -- nameComponentsCount = 1 (DVB/UK DTT) */
      ADV_UINT8_L2CHK( mpBindingData, ui8, ui8 == 1,
         dsmDP2(("DATA ERROR: DIR Binding nameComponentsCount (!= 1) = %u\n", ui8)),
         break );

      /* -- Read id (name) */
      /* TODO: L2 check idLength is legal? */
      READ_UINT8( mpBindingData, idLength );

      if (nameLength == idLength)
      {
         MEMPTR_DATA_COMPARE(
            (U8BIT *)name, mpBindingData, nameLength, found );

         if (found)
         {
            SET_POS_ABS( *mpBinding, currStartPos );
            break;
         }
      }

      /* -- Skip id (name) */
      SET_POS_REL( mpBindingData, (S32BIT)idLength );

      /* -- kindLength = 4 (DVB/UK DTT) */
      /* -- L1 check because this should be correct */
      ADV_UINT8_L1CHK( mpBindingData, ui8, ui8 == 4,
         dsmDP2(("DATA ERROR: DIR Binding kindLength (!= 4) = %u\n", ui8)),
         break );

      /* -- Skip kind */
      SET_POS_REL( mpBindingData, 4 );

      /* -- Skip binding type */
      SET_POS_REL( mpBindingData, 1 );


      /* -- mpBindingData -> IOP::IOR */

      /* -- typeIdLength = 4 (DVB/UK DTT) */
      /* -- L1 check because this should be correct */
      ADV_UINT32_L1CHK( mpBindingData, ui32, ui32 == 4,
         dsmDP2(("DATA ERROR: DIR Binding typeIdLength (!= 4) = %u\n", ui32)),
         break );

      /* -- Skip typeId */
      SET_POS_REL( mpBindingData, 4 );

      /* -- Read taggedProfilesCount >= 1 (DVB/UK DTT) */
      READ_UINT32_L2CHK( mpBindingData, profilesCount, profilesCount >= 1,
         dsmDP2(("DATA ERROR: DIR Binding taggedProfilesCount (< 1) = %u\n", ui32)),
         break );

      while (profilesCount--)
      {
         /* -- Skip profileIdTag */
         SET_POS_REL( mpBindingData, 4 );

         /* -- Read profileDataLength */
         READ_UINT32( mpBindingData, ui32 );

         /* -- Skip profile data */
         SET_POS_REL( mpBindingData, (S32BIT)ui32 );
      }

      /* -- Read objectInfoLength */
      READ_UINT16( mpBindingData, ui16 );

      /* -- Skip objectInfo */
      SET_POS_REL( mpBindingData, (S32BIT)ui16 );
   }

   MEMPTR_CLOSE( mpBindingData );

   dsmDP3(("exit odDirFindBinding -> rtn: %u\n", found));
   return found;
}

/* /////////////////////////////////////////////////////////////////////////////
// odDirGetBindingInfo
// Returns TRUE if successful.
///////////////////////////////////////////////////////////////////////////// */
BOOLEAN odDirGetBindingInfo(
   /*I*/ const MemPtr mpBinding,
   /*O*/ P_ObjectLocation pLocation, P_DeliveryParaTap pTap )
{
   MemPtr mpBindingData;
   U8BIT ui8 = 0;
   U32BIT iorTypeId;
   U16BIT length = 0;

   dsmDP3(("odDirGetBindingInfo()\n"));
   dsmAssert((mpBinding != NULL));
   dsmAssert((pLocation != NULL));
   dsmAssert((pTap != NULL));


   /* -- Open MemPtr for accessing binding data */
   MEMPTR_OPEN( mpBinding, mpBindingData );

   /* -- mpBindingData -> BIOP::Name */

   /* -- nameComponentsCount = 1 (DVB/UK DTT) */
   ADV_UINT8_L2CHK( mpBindingData, ui8, ui8 == 1,
      dsmDP2(("DATA ERROR: DIR Binding nameComponentsCount (!= 1) = %u\n", ui8)),
      goto _return );

   /* -- Skip id (name) */
   /* TODO: L2 check idLength is legal? */
   READ_UINT8( mpBindingData, ui8 );
   SET_POS_REL( mpBindingData, (S32BIT)ui8 );

   /* -- kindLength = 4 (DVB/UK DTT) */
   /* -- L1 check because this should be correct */
   ADV_UINT8_L1CHK( mpBindingData, ui8, ui8 == 4,
      dsmDP2(("DATA ERROR: DIR Binding kindLength (!= 4) = %u\n", ui8)),
      goto _return );

   /* -- Skip kind */
   SET_POS_REL( mpBindingData, 4 );

   /* -- bindingType = 0x01 or 0x02 (DVB) */
   /* -- L1 check because this should be correct */
   ADV_UINT8_L1CHK( mpBindingData, ui8, ((ui8 == 0x01) || (ui8 == 0x02)),
      dsmDP2(("DATA ERROR: DIR Binding bindingType = %u\n", ui8)),
      goto _return );

   /* -- pBindingData -> start of IOP::IOR */
   length = getIorInfoSeq( mpBindingData, &iorTypeId, pLocation, pTap );
   /*TODO: check returned IOR type against binding and/or object kind? */
   /*TODO: handle LiteOptionsProfile */

   goto _return;    /* -- stop compiler warnings when no checking */
_return:
   MEMPTR_CLOSE( mpBindingData );
   DEBUG_CHK( length != 0,
      dsmDP1(("ERROR: odDirGetBindingInfo failure (invalid)\n")));
   dsmDP3(("exit odDirGetBindingInfo -> rtn: %u\n", length));
   return (length != 0) ? TRUE : FALSE;
}

/* /////////////////////////////////////////////////////////////////////////////
// odDirBindingNameLength
// Returns name length
///////////////////////////////////////////////////////////////////////////// */
extern U8BIT odDirBindingNameLength(
   /*I*/ const MemPtr mpBinding )
{
   MemPtr mpBindingData;
   U8BIT ui8 = 0;

   dsmDP3(("odDirGetBindingName()\n"));
   dsmAssert((mpBinding != NULL));

   /* -- Open MemPtr for accessing binding data */
   MEMPTR_OPEN( mpBinding, mpBindingData );

   /* -- mpBindingData -> BIOP::Name */

   /* -- nameComponentsCount = 1 (DVB/UK DTT) */
   ADV_UINT8_L2CHK( mpBindingData, ui8, ui8 == 1,
      dsmDP2(("DATA ERROR: DIR Binding nameComponentsCount (!= 1) = %u\n", ui8)), );

   /* -- Get name (id) length */
   READ_UINT8( mpBindingData, ui8 );

   MEMPTR_CLOSE( mpBindingData );
   return ui8;
}

/* /////////////////////////////////////////////////////////////////////////////
// odDirBindingNameCopy
// copies name data to 'name'
// Returns name length
///////////////////////////////////////////////////////////////////////////// */
extern U8BIT odDirBindingNameCopy(
   /*I*/ const MemPtr mpBinding,
   /*O*/ U8BIT *name )
{
   MemPtr mpBindingData;
   U8BIT ui8 = 0;

   dsmDP3(("odDirGetBindingName()\n"));
   dsmAssert((mpBinding != NULL));

   /* -- Open MemPtr for accessing binding data */
   MEMPTR_OPEN( mpBinding, mpBindingData );

   /* -- mpBindingData -> BIOP::Name */

   /* -- nameComponentsCount = 1 (DVB/UK DTT) */
   ADV_UINT8_L2CHK( mpBindingData, ui8, ui8 == 1,
      dsmDP2(("DATA ERROR: DIR Binding nameComponentsCount (!= 1) = %u\n", ui8)), );

   /* -- Get name (id) length */
   READ_UINT8( mpBindingData, ui8 );

   /* -- Read name */
   MEMPTR_READ( mpBindingData, name, ui8 );

   MEMPTR_CLOSE( mpBindingData );
   return ui8;
}

/* /////////////////////////////////////////////////////////////////////////////
// odDirGetBindingKind
// Returns TRUE if valid.
///////////////////////////////////////////////////////////////////////////// */
BOOLEAN odDirGetBindingKind(
   /*I*/ const MemPtr mpBinding,
   /*O*/ U32BIT *pKind )
{
   MemPtr mpBindingData;
   U8BIT ui8 = 0;
   BOOLEAN valid = FALSE;

   dsmDP3(("odDirGetBindingKind()\n"));
   dsmAssert((mpBinding != NULL));
   dsmAssert((pKind != NULL));


   /* -- Open MemPtr for accessing binding data */
   MEMPTR_OPEN( mpBinding, mpBindingData );

   /* -- mpBindingData -> BIOP::Name */

   /* -- nameComponentsCount = 1 (DVB/UK DTT) */
   ADV_UINT8_L2CHK( mpBindingData, ui8, ui8 == 1,
      dsmDP2(("DATA ERROR: DIR Binding nameComponentsCount (!= 1) = %u\n", ui8)),
      goto _return );

   /* -- Skip id (name) */
   /* TODO: L2 check idLength is legal? */
   READ_UINT8( mpBindingData, ui8 );
   SET_POS_REL( mpBindingData, (S32BIT)ui8 );

   /* -- kindLength = 4 (DVB/UK DTT) */
   /* -- L1 check because this should be correct */
   ADV_UINT8_L1CHK( mpBindingData, ui8, ui8 == 4,
      dsmDP2(("DATA ERROR: DIR Binding kindLength (!= 4) = %u\n", ui8)),
      goto _return );

   /* -- Read kind */
   READ_UINT32( mpBindingData, *pKind );

   /* -- bindingType = 0x01 or 0x02 (DVB) */
   /* -- L1 check because this should be correct */
   ADV_UINT8_L1CHK( mpBindingData, ui8, ((ui8 == 0x01) || (ui8 == 0x02)),
      dsmDP2(("DATA ERROR: DIR Binding bindingType = %u\n", ui8)),
      goto _return );

   /* -- If we get here, assume we have recovered the required data */
   valid = TRUE;

   goto _return;    /* -- stop compiler warnings when no checking */
_return:
   MEMPTR_CLOSE( mpBindingData );
   DEBUG_CHK( valid == TRUE,
      dsmDP1(("ERROR: odDirGetBindingKind failure (invalid)\n")));
   dsmDP3(("exit odDirGetBindingKind -> kind: %08x, rtn: %u\n", *pKind, valid));
   return valid;
}

void odDirGetBindingsCount(
   /*I*/ const MemPtr mpObject, pObjectDataInfo_t pObjInf,
   /*O*/ U16BIT *pBindingsCount )
{
   MemPtr mpDirObjData;

   dsmDP3(("odDirGetBindingsCount()\n"));
   dsmAssert((mpObject != NULL));
   dsmAssert((pObjInf != NULL));
   dsmAssert(((pObjInf->objectKind == DIR_STR) ||
              (pObjInf->objectKind == SRG_STR)));


   /* -- Open MemPtr for accessing binding data in object */
   MEMPTR_OPEN( mpObject, mpDirObjData );

   /* -- mpDirObjData -> BIOP::DirectoryMessage */

   /* -- Skip header info and go to first message body */
   SET_POS_REL( mpDirObjData, (S32BIT)pObjInf->messageBodyOffset );

   /* -- Read bindingsCount */
   READ_UINT16( mpDirObjData, *pBindingsCount );

   MEMPTR_CLOSE( mpDirObjData );

   dsmDP3(("exit odDirGetBindingsCount\n"));
}

U16BIT odDirCountAndFirstBinding(
   /*I*/ const MemPtr mpObject, pObjectDataInfo_t pObjInf,
   /*O*/ MemPtr *mpFirstBinding )
{
   MemPtr mpBindingData;
   MemPos bindingPos;
   U16BIT bindingsCount;

   dsmDP3(("odDirGetFirstBinding()\n"));
   dsmAssert((mpObject != NULL));
   dsmAssert((pObjInf != NULL));
   dsmAssert(((pObjInf->objectKind == DIR_STR) ||
              (pObjInf->objectKind == SRG_STR)));
   dsmAssert((mpFirstBinding != NULL));


   /* -- Open MemPtr for accessing binding data in object */
   MEMPTR_OPEN( mpObject, mpBindingData );

   /* -- mpBindingData -> BIOP::DirectoryMessage */

   /* -- Skip header info and go to first message body */
   SET_POS_REL( mpBindingData, (S32BIT)pObjInf->messageBodyOffset );

   /* -- Read bindingsCount */
   READ_UINT16( mpBindingData, bindingsCount );

   if (bindingsCount > 0)
   {
      /* -- mpBindingData -> BIOP::Name */

      /* -- Read current MemPtr position and assign to output MemPtr */
      GET_POS( mpBindingData, bindingPos );
      SET_POS_ABS( *mpFirstBinding, bindingPos );
   }

   MEMPTR_CLOSE( mpBindingData );

   dsmDP3(("exit odDirGetFirstBinding -> rtn: %u\n", bindingsCount));
   return bindingsCount;
}

/*
-- NB. This function does not determine if the next binding is actually a
--     valid binding (eg. it may be the end of the dir bindings list).
*/
BOOLEAN odDirGetNextBinding(
   /*I*/ const MemPtr mpCurrBinding,
   /*O*/ MemPtr *mpNextBinding )
{
   MemPtr mpBindingData;
   MemPos bindingPos;
   BOOLEAN valid = FALSE;
   U8BIT idLength;
   U32BIT profilesCount;
   U8BIT ui8 = 0;
   U16BIT ui16 = 0;
   U32BIT ui32 = 0;
   /* -- stop compiler warnings when no checking */
   ui8 += 0; ui16 += 0; ui32 += 0;

   dsmDP3(("odDirGetNextBinding()\n"));
   dsmAssert((mpCurrBinding != NULL));
   dsmAssert((mpNextBinding != NULL));


   /* -- Open MemPtr for accessing binding data in object */
   MEMPTR_OPEN( mpCurrBinding, mpBindingData );

   /* -- mpBindingData -> BIOP::Name */

   /* -- nameComponentsCount = 1 (DVB/UK DTT) */
   ADV_UINT8_L2CHK( mpBindingData, ui8, ui8 == 1,
      dsmDP2(("DATA ERROR: DIR Binding nameComponentsCount (!= 1) = %u\n", ui8)),
      goto _return );

   /* -- Read id (name) */
   /* TODO: L2 check idLength is legal? */
   READ_UINT8( mpBindingData, idLength );

   /* -- Skip id (name) */
   SET_POS_REL( mpBindingData, (S32BIT)idLength );

   /* -- kindLength = 4 (DVB/UK DTT) */
   /* -- L1 check because this should be correct */
   ADV_UINT8_L1CHK( mpBindingData, ui8, ui8 == 4,
      dsmDP2(("DATA ERROR: DIR Binding kindLength (!= 4) = %u\n", ui8)),
      goto _return );

   /* -- Skip kind */
   SET_POS_REL( mpBindingData, 4 );

   /* -- Skip binding type */
   SET_POS_REL( mpBindingData, 1 );


   /* -- mpBindingData -> IOP::IOR */

   /* -- typeIdLength = 4 (DVB/UK DTT) */
   /* -- L1 check because this should be correct */
   ADV_UINT32_L1CHK( mpBindingData, ui32, ui32 == 4,
      dsmDP2(("DATA ERROR: DIR Binding typeIdLength (!= 4) = %u\n", ui32)),
      goto _return );

   /* -- Skip typeId */
   SET_POS_REL( mpBindingData, 4 );

   /* -- Read taggedProfilesCount >= 1 (DVB/UK DTT) */
   READ_UINT32_L2CHK( mpBindingData, profilesCount, profilesCount >= 1,
      dsmDP2(("DATA ERROR: DIR Binding taggedProfilesCount (< 1) = %u\n", ui32)),
      goto _return );

   while (profilesCount--)
   {
      /* -- Skip profileIdTag */
      SET_POS_REL( mpBindingData, 4 );

      /* -- Read profileDataLength */
      READ_UINT32( mpBindingData, ui32 );

      /* -- Skip profile data */
      SET_POS_REL( mpBindingData, (S32BIT)ui32 );
   }

   /* -- Read objectInfoLength */
   READ_UINT16( mpBindingData, ui16 );

   /* -- Skip objectInfo */
   SET_POS_REL( mpBindingData, (S32BIT)ui16 );

   /* -- Read current MemPtr position and assign to output MemPtr */
   GET_POS( mpBindingData, bindingPos );
   SET_POS_ABS( *mpNextBinding, bindingPos );

   /* -- If we get here, assume we have recovered the required data */
   valid = TRUE;

   goto _return;    /* -- stop compiler warnings when no checking */
_return:
   MEMPTR_CLOSE( mpBindingData );
   dsmDP3(("exit odDirGetNextBinding -> rtn: %u\n", valid));
   return valid;
}

/*------------------------------ Local Functions -----------------------------*/


/*----------------------------------------------------------------------------*/
