/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Implementation of the objectCarousel module
 * @file    objectCarousel.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "objectCarousel.h"

#include "object.h"
#include "module.h"
#include "cacheMgr.h"
#include "loadMgr.h"

/*------------------------------- Local Macros -------------------------------*/

#define FG_HDR_LEN       8

#define DESCRIPTOR_STORED_GROUPS_TAG        ((U8BIT) 0x80)
#define DESCRIPTOR_GROUP_LOCATION_TAG       ((U8BIT) 0x81)

/*------------------------------  Exported Data  -----------------------------*/



/*--------------------------------Local Types --------------------------------*/


typedef enum {
   FGC_NONE,
   FGC_VERS,
   FGC_LIST
} E_FgChange;

typedef struct FileGroupList_s
{
   pFileGroupList_t next;
   U16BIT clientCount;
   U16BIT fgTotal;
   U16BIT dataSize;
} FileGroupList_t;


/*------------------------------- Local Statics ------------------------------*/

static pFileGroupList_t gOrphanFileGroups = NULL;


/*------------------- local prototypes/forward declarations ------------------*/


/*------------------------------ Local Functions -----------------------------*/


static void ClearFileGroups( P_DsmCoreInst idp, pFileGroupList_t *ppFG )
{
   pFileGroupList_t pFG;
   do
   {
      pFG = *ppFG;
      *ppFG = pFG->next;

      /* -- Client should have released any references to the file group list
       * but if not, add to the orphan list (don't delete) */
      if (pFG->clientCount != 0)
      {
         pFG->next = gOrphanFileGroups;
         gOrphanFileGroups = pFG;
      }
      else
      {
         DSC_CmMemRelease( idp, pFG );
      }
   }
   while (*ppFG);
}

static U8BIT* SkipTaps( U8BIT *pData )
{
   U8BIT count, secLen;
   READ_UINT8( pData, count);
   while (count--)
   {
      SET_POS_REL( pData, 6 ); /* tapId, tapUse, associationTag */
      READ_UINT8( pData, secLen ); /* Read selectorLength */
      SET_POS_REL( pData, (S32BIT)secLen ); /* Skip selector field */
   }
   return pData;
}

static U8BIT* SkipServiceContexts( U8BIT *pData )
{
   U8BIT count;
   U16BIT ctxtLen;
   READ_UINT8(pData, count);
   while (count--)
   {
      SET_POS_REL(pData, 4);    /* Skip service ID (context_id) */
      READ_UINT16(pData, ctxtLen); /* context data length */
      SET_POS_REL(pData, ctxtLen); /* Skip context data */
   }
   return pData;
}

/*----------------------------------------------------------------------------*/

/*---------------------------- Exported Functions ----------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// DSC_ObjCrslCreate
// Creates an instance of the ObjectCarousel struct and initialises it.
///////////////////////////////////////////////////////////////////////////// */
E_DscError DSC_ObjCrslCreate( P_DsmCoreInst idp, U16BIT serviceId,
   U32BIT carouselId, P_ObjectCarousel *ppObjectCarousel )
{
   P_ObjectCarousel pOC;
   P_LoadRequest pLoadRequest;
   E_DscError err;

   dsmDP3(("DSC_ObjCrslCreate()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((ppObjectCarousel != NULL));

   pOC = (P_ObjectCarousel)DSC_CmMemGet( idp, sizeof(S_ObjectCarousel) );
   if (!pOC)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
      *ppObjectCarousel = NULL;
   }
   else
   {
      memset(pOC,0,sizeof(S_ObjectCarousel));

      err = DSC_RootCrslInitialise( idp, &pOC->root, OC_MAGIC, serviceId, carouselId );
      if (err)
      {
         DSC_CmMemRelease( idp, pOC );
         *ppObjectCarousel = NULL;
      }
      else
      {
         err = LLCreate( idp, pOC, CURR_OBJECT_LIST, &pOC->llcCurrObjects );
         if (err)
         {
            DSC_RootCrslFinalise( idp, &pOC->root );
            DSC_CmMemRelease( idp, pOC );
            *ppObjectCarousel = NULL;
         }
         else
         {
            err = LLCreate( idp, pOC, OC_LITE_OBJECT_LOAD_LIST, &pOC->llcOcPostponedLiteObjectLoads );
            if (err)
            {
               LLDestroy( idp, &(pOC->llcCurrObjects));
               DSC_RootCrslFinalise( idp, &pOC->root );
               DSC_CmMemRelease( idp, pOC );
               *ppObjectCarousel = NULL;
            }
            else
            {
               err = DSC_LoadRsqtCreate( idp, sizeof(S_LoadRequest), TT_CAROUSEL, pOC,
                     (F_LoadFinalise)lmCarouselLoadFinalise, (P_RootLoadRqst *)&pLoadRequest );
               if (err)
               {
                  LLDestroy( idp, &(pOC->llcCurrObjects));
                  LLDestroy( idp, &(pOC->llcOcPostponedLiteObjectLoads));
                  DSC_RootCrslFinalise( idp, &pOC->root );
                  DSC_CmMemRelease( idp, pOC );
                  *ppObjectCarousel = NULL;
               }
               else
               {
                  pOC->root.pLoadRqst = (P_RootLoadRqst)pLoadRequest;
                  pLoadRequest->pObjCarousel = pOC;
                  *ppObjectCarousel = pOC;
                  DBGLOG(DD_OC,"pOC=%p",pOC)
               }
            }
         }
      }
   }
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: DSC_ObjCrslCreate %u\n", err)));
   dsmDP3(("exit DSC_ObjCrslCreate -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// DSC_ObjCrslDestroy
//
///////////////////////////////////////////////////////////////////////////// */
void DSC_ObjCrslDestroy( P_DsmCoreInst idp, P_ObjectCarousel *ppObjectCarousel )
{
   P_ObjectCarousel pOC;

   dsmDP3(("DSC_ObjCrslDestroy()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((ppObjectCarousel != NULL));
   dsmAssert((*ppObjectCarousel != NULL));

   pOC = *ppObjectCarousel;

   DBGLOG(DD_OC,"pOC=%p",pOC)

   /* -- Should not have any loaded objects */
   dsmAssert((pOC->loadedObjectCount == 0));

   if (pOC->root.pLoadRqst)
   {
      DSC_LoadRsqtDestroy( idp, pOC->root.pLoadRqst );
   }
   if (pOC->pFileGroupList != NULL)
   {
      ClearFileGroups( idp, &pOC->pFileGroupList );
   }

   LLDestroy( idp, &pOC->llcOcPostponedLiteObjectLoads);
   LLDestroy( idp, &pOC->llcCurrObjects );

   DSC_RootCrslFinalise( idp, &pOC->root );

   memset(pOC,0,sizeof(S_ObjectCarousel));
   DSC_CmMemRelease( idp, pOC );
   *ppObjectCarousel = NULL;

   dsmDP3(("exit DSC_ObjCrslDestroy\n"));
}

static U16BIT CountFileGroups(  U8BIT *pData, U16BIT uiLen, U16BIT *pDataSize )
{
   U16BIT count = 0;
   U8BIT descTag, descLen;
   U8BIT ui8;
   *pDataSize = 0;
   while (uiLen > FG_HDR_LEN + 2)
   {
      READ_UINT8( pData, descTag);
      READ_UINT8( pData, descLen);
      uiLen -= 2;
      if (descLen > uiLen)
      {
         dsmDP2(("DATA ERROR: DSI User Info: UI_len=%u desc_len=%u\n", uiLen, descLen));
         break;
      }
      uiLen -= descLen;
      if (descTag == DESCRIPTOR_STORED_GROUPS_TAG)
      {
         while (descLen > FG_HDR_LEN)
         {
            count++;
            /* skip:  owner_id, group_id, group_priority,use_flags,receiver_profile,group_version */
            SET_POS_REL( pData, FG_HDR_LEN );
            READ_UINT8( pData, ui8 ); // private data length
            descLen -= FG_HDR_LEN + 1;
            if (ui8 != 0)
            {
               if (ui8 > descLen)
               {
                  dsmDP2(("DATA ERROR: DSI User Info: desc_len=%u priv data len=%u\n", descLen, ui8));
                  uiLen = 0;
                  break;
               }
               /* add max possible len for "group location" */
               *pDataSize += ui8;
               /* private data - Skip it */
               SET_POS_REL( pData, ui8 );
               descLen -= ui8;
            }
            /* add 1 for null terminating char */
            *pDataSize += 1;
         }
      }
      else
      {
         SET_POS_REL( pData, descLen );
      }
   }
   return count;
}

static U8BIT* ParseGroupLocation( U8BIT *pData, U16BIT len, U8BIT *dataPtr )
{
   U8BIT descTag, descLen;
   do
   {
      READ_UINT8( pData, descTag);
      READ_UINT8( pData, descLen);
      len -= 2;
      if (descLen > len)
      {
         dsmDP2(("DATA ERROR: desc len too long: len=%u desc_len=%u\n", len, descLen));
         break;
      }
      len -= descLen;
      if (descTag == DESCRIPTOR_GROUP_LOCATION_TAG)
      {
         while (descLen--)
         {
            READ_UINT8( pData, *dataPtr );
            dataPtr++;
         }
      }
   }
   while (len > 2);
   return dataPtr;
}

static void ParseFileGroups( S_CarouselInfoFileGroup *pIFP, U8BIT *pData, U16BIT uiLen, U8BIT *dataPtr )
{
   U8BIT descTag, descLen;
   U8BIT ui8;
   do
   {
      READ_UINT8( pData, descTag);
      READ_UINT8( pData, descLen);
      uiLen -= 2;
      if (descLen > uiLen)
      {
         dsmDP2(("DATA ERROR: DSI User Info: UI_len=%u desc_len=%u\n", uiLen, descLen));
         break;
      }
      uiLen -= descLen;
      if (descTag == DESCRIPTOR_STORED_GROUPS_TAG)
      {
         while (descLen > FG_HDR_LEN)
         {
            READ_UINT16( pData, pIFP->owner_id );
            READ_UINT16( pData, pIFP->group_id );
            READ_UINT8( pData, pIFP->group_priority );
            READ_UINT8( pData, pIFP->use_flags );
            READ_UINT8( pData, pIFP->receiver_profile );
            READ_UINT8( pData, pIFP->group_version );
            READ_UINT8( pData, ui8 );
            descLen -= FG_HDR_LEN + 1;
            pIFP->location = dataPtr;
            if (ui8 != 0)
            {
               if (ui8 > descLen)
               {
                  dsmDP2(("DATA ERROR: DSI User Info: desc_len=%u priv data len=%u\n", descLen, ui8));
                  uiLen = 0;
                  break;
               }
               dataPtr = ParseGroupLocation( pData, ui8, dataPtr );
               /* now skip over private data */
               SET_POS_REL( pData, ui8 );
               descLen -= ui8;
            }
            *dataPtr = '\0';
            DBG2(DD_OC, "oid=0x%x gid=0x%x ver=%d", pIFP->owner_id, pIFP->group_id, pIFP->group_version)
            dataPtr++;
            pIFP++;
         }
      }
      else
      {
         SET_POS_REL( pData, descLen );
         uiLen -= descLen;
      }
   }
   while (uiLen > FG_HDR_LEN + 2);
}

E_DscError DSC_ObjCrslRetrieveFileGroups( P_ObjectCarousel pOC, U16BIT *total,
   S_CarouselInfoFileGroup **pGroups )
{
   E_DscError err;
   pFileGroupList_t pFG = pOC->pFileGroupList;
   if (pFG == NULL)
   {
      *total = 0;
      *pGroups = NULL;
      err = CLDSM_ERR_LOAD_FAILED;
   }
   else
   {
      pFG->clientCount++;
      *total = pFG->fgTotal;
      *pGroups = (S_CarouselInfoFileGroup *)(pFG + 1);
      err = CLDSM_OK;
   }
   return err;
}

void DSC_ObjCrslReleaseFileGroups( P_DsmCoreInst idp, P_ObjectCarousel pOC, S_CarouselInfoFileGroup *groups )
{
   pFileGroupList_t pFG;
   pFileGroupList_t *ppFG;
   BOOLEAN remove;
   if (groups != NULL)
   {
      pFG = (pFileGroupList_t)groups;
      pFG--; // move pointer to private data for the list
      remove = TRUE;
      if (pOC == NULL)
      {
         ppFG = &gOrphanFileGroups;
      }
      else
      {
         ppFG = &pOC->pFileGroupList;
         if (pFG == *ppFG)
         {
            /* Current filegroup - just dec the client count */
            if (pFG->clientCount != 0)
               pFG->clientCount--;
            remove = FALSE;
         }
      }
      if (remove)
      {
         /* Old filegroup - dec client count and remove */
         while (*ppFG)
         {
            if (pFG == *ppFG)
            {
               if (pFG->clientCount != 0)
               {
                  pFG->clientCount--;
               }
               if (!pFG->clientCount)
               {
                  *ppFG = pFG->next;
                  DSC_CmMemRelease(idp, pFG);
               }
               break;
            }
            ppFG = &((*ppFG)->next);
         }
      }
   }
}

E_FgChange FileGroupChange( S_CarouselInfoFileGroup *pOldIFP, S_CarouselInfoFileGroup *pNewIFP, U16BIT fgTotal )
{
   E_FgChange result = FGC_NONE;
   /* check to see if group list changed or if group version changed
    * (assuming they are in same order when no change) */
   while (fgTotal--)
   {
      if (pOldIFP->owner_id != pNewIFP->owner_id ||
          pOldIFP->group_id != pNewIFP->group_id )
      {
         DBG2(DD_OC,"(%x,%x) (%x,%x)",
            pOldIFP->owner_id,pOldIFP->group_id,
            pNewIFP->owner_id,pNewIFP->group_id )
         result = FGC_LIST;
         break;
      }
      else if (pOldIFP->group_version != pNewIFP->group_version )
      {
         DBG2(DD_OC, "(%x,%x) (%x,%x) oldver=%d newver=%d",
            pOldIFP->owner_id, pOldIFP->group_id,
            pNewIFP->owner_id, pNewIFP->group_id,
            pOldIFP->group_version, pNewIFP->group_version )
         result = FGC_VERS;
      }
   }
   return result;
}

static E_DscError ocUpdateSrgUserInfo( P_DsmCoreInst idp, P_ObjectCarousel pOC,
   U8BIT *pSrgData, U16BIT usrInfoLen )
{
   E_DscError err;
   pFileGroupList_t pOldFG, pNewFG;
   S_CarouselInfoFileGroup *fgListPtr;
   U16BIT fgTotal, size, dataSize;
   U8BIT *dataPtr;
   E_FgChange fgchange;

   fgTotal = CountFileGroups(pSrgData, usrInfoLen, &dataSize);
   pOldFG = pOC->pFileGroupList;
   DBGLOG(DD_OC, "fgTotal=%d", fgTotal);
   size = sizeof(FileGroupList_t) + (fgTotal * sizeof(S_CarouselInfoFileGroup)) + dataSize;
   pNewFG = (pFileGroupList_t)DSC_CmMemGet( idp, size );
   if (!pNewFG)
   {
      ERRPRINT("Mem error")
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      err = CLDSM_OK;
      memset( pNewFG, 0, size );
      pNewFG->fgTotal = fgTotal;
      pNewFG->dataSize = dataSize;
      if (fgTotal != 0)
      {
         fgListPtr = (S_CarouselInfoFileGroup *)(pNewFG + 1);
         dataPtr = (U8BIT *)(fgListPtr + fgTotal);
         ParseFileGroups( fgListPtr, pSrgData, usrInfoLen, dataPtr );
      }
      if (pOldFG == NULL)
      {
         DBGLOG(DD_OC, "pNewFG=%p", pNewFG);
         pNewFG->next = pOC->pFileGroupList;
         pOC->pFileGroupList = pNewFG;
         if (fgTotal != 0 &&
            pOC->root.status == RCS_LOADED && /* don't signal when booting - done in DSC_ObjCrslUpdateSuiLoad */
            idp->setup.notifyCarouselLoadEventFunc != NULL)
         {
            idp->setup.notifyCarouselLoadEventFunc((H_DsmCarousel)pOC, OC_FILE_GROUP_LIST_CHANGE, pOC->root.rcid);
         }
      }
      else
      {
         if (pOldFG->fgTotal != fgTotal)
         {
            fgchange = FGC_LIST;
         }
         else
         {
            fgchange = FileGroupChange((S_CarouselInfoFileGroup*)(pOldFG+1),(S_CarouselInfoFileGroup*)(pNewFG+1),fgTotal);
         }
         if (fgchange != FGC_NONE)
         {
            DBGLOG(DD_OC, "clientCount=%d", pOldFG->clientCount);
            if (pOldFG->clientCount == 0)
            {
               /* Old File Group list can be removed */
               pOC->pFileGroupList = pOldFG->next;
               DSC_CmMemRelease(idp, pOldFG);
            }
            pNewFG->next = pOC->pFileGroupList;
            pOC->pFileGroupList = pNewFG;
            if ( idp->setup.notifyCarouselLoadEventFunc )
            {
               idp->setup.notifyCarouselLoadEventFunc( (H_DsmCarousel)pOC,
                  (fgchange == FGC_LIST)? OC_FILE_GROUP_LIST_CHANGE : OC_FILE_GROUP_VERS_CHANGE,
                  pOC->root.rcid );
            }
         }
         else /*FGC_NONE*/
         {
            /* No change to File Group list or versions */
            DSC_CmMemRelease(idp, pNewFG);
         }
      }
   }
   return err;
}

void DSC_ObjCrslUpdateSuiLoaded( P_DsmCoreInst idp, P_ObjectCarousel pOC )
{
   pFileGroupList_t  pFGL;
   dsmAssert(( pOC != NULL ));
   dsmAssert(( pOC->root.status == RCS_LOADED ));
   pFGL = pOC->pFileGroupList;
   if (pFGL && pFGL->fgTotal != 0 && idp->setup.notifyCarouselLoadEventFunc != NULL)
   {
      idp->setup.notifyCarouselLoadEventFunc((H_DsmCarousel)pOC, OC_FILE_GROUP_LIST_LOADED, pOC->root.rcid);
   }

}

E_DscError DSC_ObjCrslParseSrgInfo( P_DsmCoreInst idp, P_ObjectCarousel pOC,
   U8BIT *pDsiPrivate, U16BIT dsiPrivateLen )
{
   E_DscError err = CLDSM_OK;
   U8BIT *pData;
   U32BIT iorTypeId;
   S_ObjectLocation srgLocation;
   S_DeliveryParaTap srgTap;
   U16BIT srgUsedLen;

   dsmAssert((pDsiPrivate != NULL));

   if (dsiPrivateLen < MIN_SRG_INFO_LEN)
   {
      ERRLOG(DD_OC, "DATA ERROR: SRG short len=%u", dsiPrivateLen )
   }
   else
   {
      pData = pDsiPrivate;
      srgUsedLen = getIorInfoContig( pData, &iorTypeId, &srgLocation, &srgTap );
      /* -- Check IOR type == SRG or DIR */
      /* -- NK 16/10/01 - Default to allowing SRG or DIR since some
                          transmissions are known to use DIR */
      if (srgUsedLen == 0 || (iorTypeId != SRG_STR && iorTypeId != DIR_STR))
      {
         ERRLOG(DD_OC, "DATA ERROR: Invalid IOR data for DSI, len=%u typ=%x", srgUsedLen, iorTypeId )
      }
      else if (srgLocation.carouselId != pOC->root.rcid)
      {
         ERRLOG(DD_OC, "DATA ERROR: srgLocation.carouselId=%d tid=%d", srgLocation.carouselId, pOC->root.rcid )
      }
      else
      {
         U16BIT srgUserInfoLen;

         dsmDP3(("INFO: srgLocation.carouselId=%d\n", srgLocation.carouselId));
         pData += srgUsedLen;
         pData = SkipTaps( pData );
         pData = SkipServiceContexts( pData );

         READ_UINT16( pData, srgUserInfoLen );
         if (pData + srgUserInfoLen != pDsiPrivate + dsiPrivateLen)
         {
            dsmDP2(("DATA ERROR: DSI srg Info length error: %u - %u != %u\n",
                    dsiPrivateLen, pData - pDsiPrivate, srgUserInfoLen));
         }
         else
         {
            dsmDP3(("INFO:   pOC = 0x%p  idp->pBootingCarousel = 0x%p\n", pOC, idp->pBootingCarousel));
            if (idp->pBootingCarousel == &(pOC->root))
            {
               /* -- DSI is for booting carousel */
               idp->pBootingCarousel = NULL;
            }
            ocUpdateSrgUserInfo( idp, pOC, pData, srgUserInfoLen );

            err = lmUpdateCarouselSRGInfo( idp, pOC, &srgTap, &srgLocation );
         }
      }
   }
   return err;
}


E_DscError DSC_ObjCrslUnloadObjects( P_DsmCoreInst idp, P_ObjectCarousel pOC, E_DsmRstMode mode )
{
   P_DsmObject pDsmObject;
   E_DscError err = CLDSM_OK;

   /* -- Any active object loads will still be in the current object list
    -- so check if there is anything in it */
   pDsmObject = LLHead( pOC->llcCurrObjects );
   while ( pDsmObject )
   {
      if (pDsmObject->objectDataSeq)
      {
         /* object is open - close it */
         err = CDSM_CloseObject( idp, (H_DsmObject)pDsmObject );
         if (err != CLDSM_OK) break;
      }
      err = CDSM_UnloadObject( idp, (H_DsmObject)pDsmObject, mode );
      if (err != CLDSM_OK) break;
      pDsmObject = LLHead( pOC->llcCurrObjects );
   }
   return err;
}

BOOLEAN DSC_ObjCrslSrgObjectLoaded( P_ObjectCarousel pOC )
{
   dsmAssert((pOC->srgObjectInfo.objectKind != DIR_STR));
   return (pOC->srgObjectInfo.objectKind == SRG_STR && DSC_ObjCrslSrgModule(pOC))? TRUE : FALSE;
}

P_Module DSC_ObjCrslSrgModule( P_ObjectCarousel pOC )
{
   P_DataCarousel pDC;
   pDC = DSC_RootCrslFirstDataCarousel( &pOC->root );
   return (pDC)? DSC_DataCrslFirstModule( pDC ) : NULL;
}

void DSC_ObjCrslSrgObjectReset( P_ObjectCarousel pOC )
{
   dsmAssert((pOC->root.magic == OC_MAGIC));
   pOC->srgObjectInfo.objectKind = 0;
}

