/*******************************************************************************
 * Copyright  2015 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   RootCarousel module - common code betweeen Update Carousel
 *          and Object Carousel.
 * @file    rootCarousel.c
 * @date    18/02/2015
 * @author  Adam Sturtridge
 */

/*---includes for this file--------------------------------------------------*/

#include "rootCarousel.h"
#include "module.h"
#include "objectCarousel.h"
#include "updateCarousel.h"

/*------------------------------- Local Macros -------------------------------*/

#define INVALID_DSI_TRANSACTION_ID 0xFFFFFFFF

/*------------------------------  Exported Data  -----------------------------*/



/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/

/*------------------- local prototypes/forward declarations ------------------*/


/*------------------------------ Local Functions -----------------------------*/


/*---------------------------- Exported Functions ----------------------------*/

E_DscError DSC_RootCrslInitialise( P_DsmCoreInst idp, P_RootCarousel pRC, U16BIT magic, U16BIT serviceId, U32BIT cid )
{
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((pRC != NULL));

   llLinkInit( pRC->llData, NUM_LISTS_ROOT_CAROUSEL );

   pRC->magic = magic;
   pRC->rcid = cid;
   pRC->serviceId = serviceId;
   pRC->dsiTransactionId = INVALID_DSI_TRANSACTION_ID;
   pRC->status = RCS_INITIAL;
   pRC->pCrslInfo = NULL;

   /* memset(p,0,s) should have called */
   dsmAssert((pRC->pDsiSf == NULL));
   dsmAssert((pRC->pPendingSiQueryRef == NULL));
   dsmAssert((pRC->llcDataCarousels == NULL));
   dsmAssert((pRC->llcDiiAcquires == NULL));

   /* -- Create attached lists */
   err = LLCreate( idp, pRC, OC_DII_ACQUIRE_LIST, &pRC->llcDiiAcquires );
   if (err)
   {
      pRC->llcDiiAcquires = NULL;
   }
   else
   {
      err = LLCreate( idp, pRC, OC_DATA_CAROUSEL_LIST, &pRC->llcDataCarousels );
      if (err)
      {
         LLDestroy( idp, &pRC->llcDiiAcquires );
      }
      else if (NULL != idp->setup.subscribeSIChangeFunc && serviceId != 0)
      {
         err = idp->setup.subscribeSIChangeFunc( idp->setup.siqInstance, serviceId );
         switch (err)
         {
            case CLDSM_DUPLICATE_REQUEST:
            case CLDSM_PENDING:
               err = CLDSM_OK; /* Pending and duplicate are not errors*/
            case CLDSM_OK:
               break;

            default:
            {
               LLDestroy( idp, &pRC->llcDiiAcquires );
               LLDestroy( idp, &pRC->llcDataCarousels );
            }
         }
      }
   }
   return err;
}

void DSC_RootCrslFinalise( P_DsmCoreInst idp, P_RootCarousel pRC )
{
   dsmAssert((idp != NULL));
   dsmAssert((pRC != NULL));

   dsmAssert((pRC->magic == OC_MAGIC || pRC->magic == UC_MAGIC));

   /* -- Should not be in any lists */
   dsmAssert((pRC->llData[ROOT_CAROUSEL_LIST].pLLCtrl == NULL));

   /* -- Should have a DII acquire list ctrl block */
   dsmAssert((pRC->llcDiiAcquires != NULL));

   /* -- Should have a DC list ctrl block */
   dsmAssert((pRC->llcDataCarousels != NULL));

   /* -- Should not have a section filter assigned */
   dsmAssert((pRC->pDsiSf == NULL));

   /* -- Should not have an active SI query */
   dsmAssert((pRC->pPendingSiQueryRef == NULL));

#ifndef NDEBUG
   {
      int listCount;
      /* -- DC list should be empty */
      listCount = LLCount( pRC->llcDataCarousels );
      dsmAssert((listCount == 0));
   }
#endif

   if (NULL != idp->setup.unsubscribeSIChangeFunc && pRC->serviceId != 0)
   {
      idp->setup.unsubscribeSIChangeFunc( idp->setup.siqInstance, pRC->serviceId );
   }

   /* -- NULL carousel contents. */
   pRC->magic = 0;
   pRC->rcid = INVALID_CAROUSEL_ID;
   pRC->serviceId = 0;
   pRC->dsiTransactionId = INVALID_DSI_TRANSACTION_ID;
   pRC->status = RCS_INITIAL;
   pRC->dsiAssocTag = 0;

   while (pRC->pCrslInfo != NULL)
   {
      P_CarouselInfo pCrslInfo;
      pCrslInfo = pRC->pCrslInfo->next;
      idp->setup.freeFunc(pRC->pCrslInfo);
      pRC->pCrslInfo = pCrslInfo;
   }

   LLDestroy( idp, &(pRC->llcDataCarousels));
   LLDestroy( idp, &(pRC->llcDiiAcquires));
}

P_RootCarousel DSC_RootCrslGetParentCarousel( P_DataCarousel pDC )
{
   return LLParent( pDC, OC_DATA_CAROUSEL_LIST );
}

void DSC_RootCrslAbortLoadRequest( P_DsmCoreInst idp, P_RootCarousel pRC )
{
   P_RootLoadRqst pLoadRqst;
   pLoadRqst = pRC->pLoadRqst;
   if (NULL != pLoadRqst)
   {
      pLoadRqst->status = LRS_ABORTED_LOAD_ERROR;
      DSC_LoadRsqtFinalise( idp, pLoadRqst );
      pRC->pLoadRqst = NULL;
   }
}

/**
 * @brief   Request DSI for booting carousel, since info is available
 */
E_DscError DSC_RootCrslBootCarousel( P_DsmCoreInst idp, P_RootCarousel pRC,
   P_CarouselInfo pCarouselInfo )
{
   S_SfTarget target;
   P_RootLoadRqst pLoadRqst;
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((pRC != NULL));
   dsmAssert((pCarouselInfo != NULL));
   dsmAssert((pRC->status == RCS_PENDING_BOOTINFO));

   /* TODO: Implement use of FormatID from carousel_id_descriptor
          for enhanced boot info and enhanced boot mechanism. */
   if (pRC->rcid == INVALID_CAROUSEL_ID || pRC->rcid == UNKNOWN_CAROUSEL_ID)
   {
      pRC->rcid = pCarouselInfo->carouselId;
   }

   /* -- Given component_tag references stream on which DSI is broadcast */
   pRC->dsiAssocTag = pCarouselInfo->associationTag;
   pRC->pid = pCarouselInfo->pid;
   pRC->pCrslInfo = pCarouselInfo->next;

   dsmAssert((pRC->pLoadRqst != NULL));
   pLoadRqst = (P_RootLoadRqst)pRC->pLoadRqst;
   dsmAssert((pLoadRqst->status == LRS_INITIAL));

   /* Request DSI section filter */
   target.kind = SFK_DSI;
   target.id = pRC->rcid;
   target.associationTag = pRC->dsiAssocTag;
   target.serviceId =
      (target.associationTag == INVALID_ASSOCIATION_TAG)? pRC->pid : pRC->serviceId;
   target.u.pRC = pRC;
   err = DSC_SectionFilterStart( idp, &target, SF_PRIORITY_HIGH, &pRC->pDsiSf );
   if (!err)
   {
      pLoadRqst->status = LRS_STALLED_DSI;
      /* -- Carousel is now booting */
      pRC->status = RCS_BOOTING;
   }
   else
   {
      ERRLOG(DD_OC, "ERR=%u", err)
   }
   return err;
}

/**
 * @brief   Restart section filters
 */
E_DscError DSC_RootCrslAcquireRestart( P_DsmCoreInst idp, P_RootCarousel pRC )
{
   S_SfTarget target;
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((pRC != NULL));

   if (pRC->pDsiSf != NULL)
   {
      target = pRC->pDsiSf->target;
      DSC_SectionFilterStop( idp, &pRC->pDsiSf );
   }
   else
   {
      /* Request DSI section filter */
      target.kind = SFK_DSI;
      target.id = pRC->rcid;
      target.associationTag = pRC->dsiAssocTag;
      target.serviceId = (target.associationTag == INVALID_ASSOCIATION_TAG)? pRC->pid : pRC->serviceId;
      target.u.pRC = pRC;
   }
   err = DSC_SectionFilterStart( idp, &target, SF_PRIORITY_HIGH, &pRC->pDsiSf );
   if (err)
   {
      ERRLOG(DD_OC, "ERR=%u", err)
   }
   return err;
}

/**
 * @brief   Find Carousel from ID's
 */
P_RootCarousel DSC_RootCrslListFindById( P_LLControl plcCarousels, U16BIT serviceId, U32BIT couId )
{
   P_RootCarousel pRC = NULL;
   ListId_t listId;

   dsmDP3(("ocListFindById()\n"));
   dsmAssert((plcCarousels != NULL));

   /* Get listId and first OC in list from Control block */
   listId = LListId( plcCarousels );
   pRC = (P_RootCarousel)LLHead( plcCarousels );

   /* Go through each OC in list until the one with a matching carouselId
      is found */
   while (pRC)
   {
      /* -- NB. Both service_id and carouselId must match
         -- to uniquely identify a carousel */
      if ((pRC->serviceId == serviceId) &&
          (pRC->rcid == couId))
      {
         /* Found the Carousel */
         break;
      }
      pRC = LLNext( pRC, listId );
   }
   return pRC;
}

U16BIT DSC_RootCrslGetServiceId( P_RootCarousel pRC )
{
   return (pRC) ? pRC->serviceId : 0xFFFF;
}

U16BIT DSC_RootCrslGetPid( P_RootCarousel pRC )
{
   return (pRC) ? pRC->pid : 0xFFFF;
}

P_DataCarousel DSC_RootCrslFirstDataCarousel( P_RootCarousel pRC )
{
   return LLHead( pRC->llcDataCarousels );
}

void DSC_RootCrslAddDataCarousel( P_RootCarousel pRC, P_DataCarousel pDC )
{
   dsmAssert((pRC != NULL));
   dsmAssert((pDC != NULL));
   LLInsertTail( pRC->llcDataCarousels, pDC );
}

/* /////////////////////////////////////////////////////////////////////////////
//
// Unload all data carousels (and associated modules) in carousel
//
///////////////////////////////////////////////////////////////////////////// */
void DSC_RootCrslUnload( P_DsmCoreInst idp, P_RootCarousel pRC )
{
   P_DataCarousel pDC;
   P_DataCarousel pPrev;
   dsmAssert((idp != NULL));
   dsmAssert((pRC != NULL));

   LLRemove( pRC, ROOT_CAROUSEL_LIST );

   pDC = LLTail( pRC->llcDataCarousels );
   while (pDC)
   {
      pPrev = LLPrev( pDC, OC_DATA_CAROUSEL_LIST );
      DSC_DataCrslDelete( idp, pDC );
      pDC = pPrev;
   }
}

/**
 * @brief   Unload module on Data Carousel
 * @param   idp DSMCC instance handle.
 * @param   pUC Pointer to Root Carousel
 * @param   moduleRef Reference to a module
 * @return  void
 */
void DSC_RootCrslUnloadModule( P_DsmCoreInst idp, P_RootCarousel pRC,
   U32BIT moduleRef )
{
   P_DataCarousel pDC;
   U16BIT dcId, moduleId;
   dsmAssert((idp != NULL));
   dsmAssert((pRC != NULL));

   dcId = moduleRef >> 16;
   moduleId = moduleRef & 0xFFFF;
   pDC = LLHead( pRC->llcDataCarousels );
   while (pDC)
   {
      if (pDC->dataCarouselId == dcId)
      {
         DSC_DataCrslUnloadModule( idp, pDC, moduleId );
         break;
      }
      pDC = LLNext( pDC, OC_DATA_CAROUSEL_LIST );
   }
}

void DSC_RootCrslDestroy( P_DsmCoreInst idp, P_RootCarousel pRC )
{
   switch (pRC->magic)
   {
      case OC_MAGIC:
         DSC_ObjCrslDestroy(idp, (P_ObjectCarousel *)&pRC);
         break;
      case UC_MAGIC:
         DSC_UpdCrslDestroy(idp, (P_UpdateCarousel)pRC);
         break;
      default:;
   }
}

U16BIT DSC_RootCrslMagic( P_RootCarousel pRC )
{
   return (pRC) ? pRC->magic : 0;
}

E_DscError DSC_RootCrslSrgObjectReset( P_RootCarousel pRC )
{
   E_DscError err;
   if (!memValidate(pRC) || pRC->magic != OC_MAGIC)
   {
      err = CLDSM_ERR_INVALID_CAROUSEL_HANDLE;
   }
   else
   {
      err = CLDSM_OK;
      DSC_ObjCrslSrgObjectReset((P_ObjectCarousel)pRC);
   }
   return err;
}

BOOLEAN DSC_RootCrslCheckCompatibility( P_RootCarousel pRC, U8BIT *pCompatDesc, U16BIT compatLen )
{
   if (pRC->magic != UC_MAGIC)
   {
      return FALSE;
   }
   return DSC_UpdCrslCheckCompatibility( (P_UpdateCarousel)pRC, pCompatDesc, compatLen );
}

void DSC_RootCrslLoadRequestFail( P_DsmCoreInst idp, P_RootCarousel pRC )
{
   ERRLOG(DD_OC, "RC status %u", pRC->status)
   if (pRC->pCrslInfo != NULL)
   {
      P_CarouselInfo pCrslInfo = pRC->pCrslInfo;

      /* restart load request */
      if (pRC->pDsiSf != NULL)
      {
         DSC_SectionFilterStop( idp, &pRC->pDsiSf );
      }
      pRC->status = RCS_PENDING_BOOTINFO;
      dsmAssert((pRC->pLoadRqst != NULL));
      pRC->pLoadRqst->status = LRS_INITIAL;
      DSC_RootCrslBootCarousel( idp, pRC, pCrslInfo );

      /* free up extra carousel info */
      idp->setup.freeFunc(pCrslInfo);
   }
   else
   {
      DSC_RootCrslAbortLoadRequest(idp, pRC);
   }
}
