/*******************************************************************************
 * Copyright  2015 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header defining RootCarousel module - common code betweeen Update Carousel
 *          and Object Carousel.
 * @file    rootCarousel.h
 * @date    18/02/2015
 * @author  Adam Sturtridge
 */

#ifndef _ROOTCAROUSEL_H_
#define _ROOTCAROUSEL_H_

/*--------------------------------  Includes  --------------------------------*/
#include "clDsmSystem.h"

#include "linkList.h"
#include "sectionFilter.h"
#include "dataCarousel.h"
#include "loadRqst.h"

#ifdef __cplusplus
extern "C" {
#endif

/*-----------------------------  Exported Macros  ----------------------------*/

#define OC_MAGIC                 0xC2A5
#define UC_MAGIC                 0xC550

#define RC_VALID(p)              (memValidate(p) && (p->magic == OC_MAGIC || p->magic == UC_MAGIC))

/*------------------------------  Exported Types  ----------------------------*/

typedef enum
{
   RCS_INITIAL,
   RCS_PENDING_BOOTINFO,    /* -- waiting on SI Query for boot info */
   RCS_BOOTING,             /* -- DSI requested */
   RCS_BOOTED,              /* -- DSI (SRG info) acquired, SRG requested */
   RCS_LOADING,             /* -- SSU loading modules */
   RCS_LOADED,              /* -- SRG loaded */
   RCS_LOAD_FAILED          /* -- failed to boot or load */
} E_RcStatus;


typedef struct s_RootCarousel
{
   S_LLObject llData[NUM_LISTS_ROOT_CAROUSEL];

   U16BIT magic;
   U16BIT pid;

   U32BIT dsiTransactionId;
   U32BIT rcid; /* Carousel ID, or OUI (Organization Unique Identifier) */

   U16BIT serviceId;
   U16BIT dsiAssocTag;
   P_SecFilterInfo pDsiSf;

   E_RcStatus status;
   P_SiQuery pPendingSiQueryRef;                /* -- Only used initially */
   P_CarouselInfo pCrslInfo;
   P_RootLoadRqst pLoadRqst;

   P_LLControl llcDataCarousels;
   P_LLControl llcDiiAcquires;
} S_RootCarousel;


/*------------------------------  Exported Data  -----------------------------*/


/*---------------------------  Exported Prototypes  --------------------------*/

E_DscError DSC_RootCrslInitialise( P_DsmCoreInst idp, P_RootCarousel pRC, U16BIT magic, U16BIT serviceId, U32BIT cid );

void DSC_RootCrslFinalise( P_DsmCoreInst idp, P_RootCarousel pRC );

void DSC_RootCrslDestroy( P_DsmCoreInst idp, P_RootCarousel pRC );

E_DscError DSC_RootCrslDiiAcquireStart( P_DsmCoreInst idp,
   P_RootCarousel pRC, P_DataCarousel pDC, E_SFPriority sfPriority );

void DSC_RootCrslDiiAcquireStop( P_DsmCoreInst idp, P_DataCarousel pDC );

void DSC_RootCrslAbortLoadRequest(P_DsmCoreInst idp, P_RootCarousel pRC );

/**
 * @brief   Request DSI for booting carousel, since info is available
 */
E_DscError DSC_RootCrslBootCarousel( P_DsmCoreInst idp, P_RootCarousel pRC,
   P_CarouselInfo pCarouselInfo );

/**
 * @brief   Restart section filters
 */
E_DscError DSC_RootCrslAcquireRestart( P_DsmCoreInst idp, P_RootCarousel pRC );

/**
 * @brief   Unload module on Data Carousel
 * @param   idp DSMCC instance handle.
 * @param   pUC Pointer to Root Carousel
 * @param   moduleRef Reference to a module
 * @return  void
 */
void DSC_RootCrslUnloadModule( P_DsmCoreInst idp, P_RootCarousel pRC,
   U32BIT moduleRef );

/**
 * @brief   Find Carousel from ID's
 */
P_RootCarousel DSC_RootCrslListFindById( P_LLControl plcCarousels, U16BIT serviceId, U32BIT couId );

U16BIT DSC_RootCrslGetServiceId( P_RootCarousel pRC );
U16BIT DSC_RootCrslGetPid( P_RootCarousel pRC );

P_DataCarousel DSC_RootCrslFirstDataCarousel( P_RootCarousel pRC );

void DSC_RootCrslAddDataCarousel( P_RootCarousel pRC, P_DataCarousel pDC );

void DSC_RootCrslUnload( P_DsmCoreInst idp, P_RootCarousel pRC );

U16BIT DSC_RootCrslMagic( P_RootCarousel pRC );

E_DscError DSC_RootCrslSrgObjectReset( P_RootCarousel pRC );

BOOLEAN DSC_RootCrslCheckCompatibility( P_RootCarousel pRC, U8BIT *pCompatDesc, U16BIT compatLen );

void DSC_RootCrslLoadRequestFail( P_DsmCoreInst idp, P_RootCarousel pRC );

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif
#endif /* _ROOTCAROUSEL_H_ */

