/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2001 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Module for allocating/freeing and managing internal section filters.
 * @file    sectionFilter.c
 * @date    28/9/2001
 * @author  R Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "sectionFilter.h"

#include "objectCarousel.h"
#include "dataCarousel.h"
#include "module.h"
#include "siQuery.h"
#include "loadMgr.h"
#include "sectionTimer.h"
#include "streamEvent.h"

/*------------------------------- Local Macros -------------------------------*/

#define HASH_SHIFT  3
#define HASH_NUM  (1 << HASH_SHIFT)
#define HASH_MASK (HASH_NUM - 1)
#define HNDL_MAGIC 0xABBA0000
#define HNDL_MASK  0xFFFF0000

/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/

typedef struct s_SectionFilterHeap
{
   /* -- Section filter heap control */
   P_SecFilterInfo sfFreeListHead;
   P_SecFilterInfo sfFreeListTail;
   U16BIT sectionFiltersInUse;

   P_SecFilterInfo sfActiveHeads[HASH_NUM];
   U16BIT activeCounter;

   /* -- Section filter heap memory */
   U32BIT totalFilters;
   P_SecFilterInfo sectionFilters;
} S_SecFilterHeap, *P_SecFilterHeap;


/*------------------------------- Local Statics ------------------------------*/
#ifdef ACCELERATE_CAROUSEL_LOADING
P_SecFilterInfo wideOpenFilter;

typedef struct _accelerate_carousel_loading_struct
{
   U16BIT storeWritePos;
   U16BIT storeReadPos;
   U16BIT storedSectionsNB;
   U16BIT SectionsReinjection;
   BOOLEAN bLargeDDBFilterSet;
   U16BIT requestedModidPos;
   P_SecFilterInfo largeDDBFilter;
} accelerateCarouselLoading_t;

extern accelerateCarouselLoading_t accCarouselLoadingInfos;
#endif

#ifdef SF_STATS
static U32BIT sfStatsNum = 0;
static U32BIT sfStatsMax = 0;
#endif

/*------------------- local prototypes/forward declarations ------------------*/

static void SectionFilterHeapInit( P_SecFilterHeap pSfHeap );

static P_SecFilterInfo AllocSectionFilterBlock( P_DsmCoreInst idp );

static void FreeSectionFilterBlock( P_DsmCoreInst idp,
   /*I*/ P_SecFilterInfo pSectionFilter );

#ifdef ACCELERATE_CAROUSEL_LOADING
E_DscError sectionLargeDDBFilterStart( P_DsmCoreInst idp,
   E_SecFiltKind sfKind,
   U16BIT service_id,
   E_SFPriority sfPriority,
   P_SecFilterInfo *pSectionFilter,
   U16BIT assoctag);

extern E_DscError requestedModuleIdSet(U16BIT moduleID);

extern E_DscError requestedModuleIdReset(U16BIT moduleID);

#endif

/*---------------------------- Exported Functions ----------------------------*/

/*
-- NOTES:
-- The heap of section filter memory blocks is managed so that new filters
-- are assigned from the oldest free block in the heap.
-- This helps with detecting sections arriving from filters after the filter
-- has been deleted (due to delays in the calling environment).
*/


/*
-- Create section filter heap
*/
E_DscError DSC_SsectionFilterHeapCreate( P_DsmCoreInst idp,
   /*O*/ void **pSectionFilterHeap, U32BIT *pHeapSize )
{
   E_DscError err = CLDSM_OK;
   P_SecFilterHeap pSfHeap;
   U32BIT size;

   dsmDP3(("DSC_SsectionFilterHeapCreate()\n"));
   dsmAssert((idp != NULL));

   size = (*pHeapSize) * sizeof(S_SecFilterInfo);
   size += sizeof(S_SecFilterHeap);

   *pSectionFilterHeap = NULL;

   pSfHeap = (P_SecFilterHeap)idp->setup.allocFunc( size );

   if (pSfHeap)
   {
      /* -- Output debug info about the section filter heap */
      DBGLOG(DD_SF, "Section Filter heap size = %u bytes", size)

      pSfHeap->totalFilters = *pHeapSize;
      pSfHeap->sectionFilters = (P_SecFilterInfo)(pSfHeap + 1);
      pSfHeap->activeCounter = 0x0000;

      SectionFilterHeapInit( pSfHeap );

      *pSectionFilterHeap = pSfHeap;

      *pHeapSize = size;
   }
   else
   {
      ERRPRINT("Failed to get section filter heap (%u); sz=%u",*pHeapSize,size)
      err = CLDSM_ERR_ALLOC_FAILED;
   }

   dsmDP3(("exit DSC_SsectionFilterHeapCreate -> rtn: %u\n", err));
   return err;
}

/*
-- Reset section filter heap
*/
void DSC_SsectionFilterHeapReset( P_DsmCoreInst idp )
{
   P_SecFilterHeap pSfHeap;
   P_SecFilterInfo pSF, *pSFHead;
   int ndx;

   dsmDP3(("DSC_SsectionFilterHeapReset()\n"));

   ASSERT( idp != NULL )
   pSfHeap = idp->sectionFilterHeap;
   ASSERT( pSfHeap != NULL )

   /* -- NB. should not be any active section filters at this point */
   if (pSfHeap->sectionFiltersInUse != 0)
   {
      ERRPRINT("************ Section filters still in USE! %d *************",
         pSfHeap->sectionFiltersInUse )
      for (ndx = 0; ndx != HASH_NUM; ndx++)
      {
         pSFHead = pSfHeap->sfActiveHeads + ndx;
         while (*pSFHead != NULL)
         {
            pSF = *pSFHead;

            ERRPRINT( "handle=0x%x targetKind=%d status=%d targetId=0x%x", pSF->handle, pSF->target.kind, pSF->status, pSF->target.id )

            DSC_SectionFilterStop(idp, &pSF);
         }
         if (!pSfHeap->sectionFiltersInUse)
            break;
      }
   }

   SectionFilterHeapInit( pSfHeap );

   dsmDP3(("exit DSC_SsectionFilterHeapReset\n"));
}

/*
-- Start a section filter of the specified kind.
-- Make SI query to get PID.
-- If PID returned synchronously then commit external filter (ie. add it
-- via callback).
*/
E_DscError DSC_SectionFilterStart( P_DsmCoreInst idp, P_SfTarget pTarget,
   E_SFPriority sfPriority, P_SecFilterInfo *ppSF )
{
   P_SecFilterInfo pSF;
   E_DscError err = CLDSM_OK;
   S_SIQueryRequest siQueryData;
   S_SIQueryResult siQueryResult;

   dsmDP3(("DSC_SectionFilterStart()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pTarget != NULL));
   dsmAssert((ppSF != NULL));

   *ppSF = NULL;

   pSF = AllocSectionFilterBlock( idp );
   if (!pSF)
   {
      /* TODO: HANDLE LIMITED SECTION FILTERS CORRECTLY */
      dsmDP1(("ERROR: Limited section filters not yet supported\n"));
      dsmAssert((0));
      err = CLDSM_ERR_SECTION_FILTER_HEAP_FULL;
   }
   else if (pTarget->u.pRC == NULL)
   {
      ERRPRINT("target ref is NULL")
      err = CLDSM_ERR_INVALID_SI_QUERY_REF;
      FreeSectionFilterBlock( idp, pSF );
   }
   else
   {
      switch (pTarget->kind)
      {
         case SFK_DSI:
         {
            pSF->filter.tableId = DSIDII_ID;
            pSF->filter.tableIdExt = 0x0000;
            pSF->filter.tableIdExtMask = 0xFFFE;
         #ifdef ACCELERATE_CAROUSEL_LOADING
            if (FALSE == accCarouselLoadingInfos.bLargeDDBFilterSet)
            {
               sectionLargeDDBFilterStart( idp, SFK_DDB, pTarget->serviceId,
                  SF_PRIORITY_LOW, &accCarouselLoadingInfos.largeDDBFilter,
                  pTarget->associationTag);
               accCarouselLoadingInfos.bLargeDDBFilterSet = TRUE;
            }
         #endif /* ACCELERATE_CAROUSEL_LOADING */
            break;
         }
         case SFK_DII:
         {
            pSF->filter.tableId = DSIDII_ID;
            pSF->filter.tableIdExt = (U16BIT)pTarget->id;
            pSF->filter.tableIdExtMask = 0xFFFE;
            break;
         }
   #if 0
         /* TODO: Getting next DII version only (to reduce filtering
            reqs. when monitoring for updates) ? */
         case SFK_DII_NEXT:
         {
            pSF->filter.tableId = DSIDII_ID;
            pSF->filter.tableIdExt = (U16BIT)pTarget->id;
            pSF->filter.tableIdExtMask = 0xFFFF;
            break;
         }
   #endif
         case SFK_DDB:
         {
            pSF->filter.tableId = DDB_ID;
            pSF->filter.tableIdExt = (U16BIT)pTarget->id;
            pSF->filter.tableIdExtMask = 0xFFFF;
         #ifdef ACCELERATE_CAROUSEL_LOADING
            requestedModuleIdSet(pTarget->id);
         #endif /* ACCELERATE_CAROUSEL_LODING */
            break;
         }
         case SFK_STREAM_DESCR:
         {
            /* Tid=0x3D (define a constant)*/
            /* Tidex= EventId&0x3F */
            /* (if possible target id will be eventId)*/
            pSF->filter.tableId = STREAM_DESCR_ID;
            pSF->filter.tableIdExt = (U16BIT)pTarget->id;
            pSF->filter.tableIdExtMask = 0xFFFF;
            break;
         }
         default:
         {
            ERRPRINT("section filter kind = %u", pTarget->kind)
            err = CLDSM_ERR_INTERNAL;
            break;
         }
      }
      if (err)
      {
         ERRLOG(DD_SF, "setup fail: err=%d", err)
         FreeSectionFilterBlock( idp, pSF );
      }
      else
      {
         /* clear timers */
         pSF->tms.mainTimerHandle = NULL;
         pSF->tms.nextTimerHandle = NULL;
         pSF->tms.nextDuration = 0;

         pSF->filter.priority = (U8BIT)sfPriority;
         pSF->target = *pTarget;

         if (pTarget->associationTag == INVALID_ASSOCIATION_TAG)
         {
            /* pTarget->serviceId is really the PID value */
            pSF->filter.pid = pTarget->serviceId;
            err = DSC_SectionFilterCommit( idp, pSF );
            if (!err)
            {
               *ppSF = pSF;
            }
            else
            {
               ERRLOG(DD_SF, "Commit fail: err=%d", err)
               FreeSectionFilterBlock( idp, pSF );
            }
         }
         else
         {
            /* -- Make SI Query to get PID for section filter -- */
            siQueryData.kind = SIQUERY_PID;
            siQueryData.serviceId = pTarget->serviceId;
            siQueryData.associationTag = pTarget->associationTag;
            err = siQueryStart( idp, &siQueryData, (void *) pSF, &siQueryResult );
            switch (err)
            {
               case CLDSM_PENDING:
               {
                  /* -- Store query ref with filter */
                  DBGLOG(DD_SF, "PENDING kind=%d qhdl=%p\n", siQueryResult.kind, siQueryResult.data.queryHandle);
                  pSF->u.pPendingSiQueryRef = siQueryResult.data.queryHandle;
                  pSF->status = SFA_PENDING;
                  *ppSF = pSF;
                  err = CLDSM_OK;
                  break;
               }
               case CLDSM_OK:
               {
                  /* -- Store PID in filter */
                  pSF->filter.pid = siQueryResult.data.pid;
                  if (pSF->filter.pid == 0)
                  {
                     dsmDP1(("zero PID!"));
                     err = CLDSM_ERR_INVALID_SI_INFO;
                  }
                  else
                  {
                     DBGLOG(DD_SF, "kind=%d pid=0x%x\n", siQueryResult.kind, siQueryResult.data.pid);
                     err = DSC_SectionFilterCommit( idp, pSF );
                     if (!err)
                     {
                        *ppSF = pSF;
                        break;
                     }
                     dsmDP1(("DSC_SectionFilterCommit.err = %x", err));
                  }
               }
               default:
               {
                  ERRLOG(DD_SF, "Query fail: err=%d", err)
                  FreeSectionFilterBlock( idp, pSF );
               }
            }
         }
      }
   }
   dsmDP3(("exit DSC_SectionFilterStart -> rtn: %u\n", err));
   return err;
}

/*
-- DSC_SsectionFilterUpdatePriority
--
-- If latchHighest is set
--     If new priority is HIGHER than current priority change filter priority
--      to new priority
-- Else
--     Always change filter priority to new priority
--
-- If filter is committed then delete current filter and add new filter using
-- same filter reference.
--
*/
E_DscError DSC_SsectionFilterUpdatePriority( P_DsmCoreInst idp,
   /*I*/ P_SecFilterInfo pSF, E_SFPriority newSfPriority,
   BOOLEAN latchHighest )
{
   E_DscError err = CLDSM_OK;

   dsmDP3(("DSC_SsectionFilterUpdatePriority()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pSF != NULL));

   /* update timer */
   sectionTimerUpdate(idp, pSF, FALSE, TRUE);

   /* -- NB. Make it safe in retail mode */
   if (pSF != NULL)
   {
      if ((newSfPriority != pSF->filter.priority) &&
          !((latchHighest == TRUE) &&
            (newSfPriority < pSF->filter.priority)))
      {
         /* -- Update stored filter priority */
         pSF->filter.priority = (U8BIT)newSfPriority;

         switch (pSF->status)
         {
            case SFA_PENDING:
               /*
               -- In this state, the section filter has not yet been
               -- added (committed) since it is waiting for an
               -- pending SI Query to complete, so nothing specific to do.
               */
               break;


            case SFA_COMMITTED:
               DBGLOG(DD_SF, "pSF->u.filterHandle=%p", pSF->u.filterHandle)
               /* -- In this state, the section filter is actually started
                 -- in the calling env. so delete it and restart it with
                 -- the same filter ref */
               if (idp->setup.sfPriorityChangeFunc != NULL)
               {
                  idp->setup.sfPriorityChangeFunc( idp->setup.sfmInstance,
                     pSF->u.filterHandle, (H_DscSFRef)pSF->handle, newSfPriority );
               }
               break;


            case SFA_ABORTED:
               /* -- In this state, the section filter commit has failed so
                 -- nothing specific to do */
               break;

            default:
               /* -- 'Added' section filters should not be in any other state */
               ERRPRINT( "Illegal section filter status = %u\n", pSF->status )
               err = CLDSM_ERR_INTERNAL;
               break;
         }
      }
   }
   DBG3(DD_SF, "exit DSC_SsectionFilterUpdatePriority -> rtn: %u\n", err)
   return err;
}

/*
-- Stop section filter.
-- If state is pending, stop associated SI Query.
-- If state is committed then delete external filter via callback.
*/
void DSC_SectionFilterStop( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo *ppSF )
{
   P_SecFilterInfo pSF;

   #ifdef ACCELERATE_CAROUSEL_LOADING
   P_Module pTargetModule;
   #endif /* ACCELERATE_CAROUSEL_LOADING */

   dsmDP3(("DSC_SectionFilterStop()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((ppSF != NULL));
   //dsmAssert((*ppSF != NULL));

   pSF = *ppSF;

   /* -- NB. Make it safe in retail mode */
   if (pSF != NULL)
   {
   #ifdef ACCELERATE_CAROUSEL_LOADING
      if (pSF->target.kind == SFK_DDB)
      {
         pTargetModule = pSF->target.u.pModule;
         requestedModuleIdReset(pTargetModule->moduleInfo.moduleId);
      }
   #endif /* ACCELERATE_CAROUSEL_LOADING */

      /* stop acquisition timer */
      sectionTimerRemove(idp, pSF);

      switch (pSF->status)
      {
         case SFA_PENDING:
            DBGLOG(DD_SF, "SFA_PENDING 0x%x", pSF->u.pPendingSiQueryRef)
            /*
            -- In this state, the section filter has not yet been
            -- added (committed) since it is waiting for an
            -- pending SI Query to complete, so abort the query.
            */
            dsmAssert((pSF->u.pPendingSiQueryRef != NULL));
            siQueryAbortPending( idp, pSF->u.pPendingSiQueryRef );

            /* -- Clear Query ref in filter (finished with) */
            pSF->u.pPendingSiQueryRef = NULL;
            break;


         case SFA_COMMITTED:
            DBGLOG(DD_SF, "SFA_COMMITTED 0x%x", pSF->u.filterHandle)
            /* -- In this state, the section filter is actually started
             -- in the calling env. so delete it */
            idp->setup.delSectionFilterFunc( idp->setup.sfmInstance, pSF->u.filterHandle,
            (H_DscSFRef)pSF->handle );
            break;


         case SFA_ABORTED:
            /* -- In this state, the section filter commit has failed so
             -- nothing specific to do (just allow filter to be freed) */
            DBGLOG(DD_SF, "SFA_ABORTED")
            break;

         default:
            /* -- 'Added' section filters should not be in any other state */
            dsmDP1(("ERROR: Illegal section filter status = %u\n", pSF->status));
            /* -- Notify an internal error here since can't return one */
            if (idp->setup.errorFunc)
            {
               idp->setup.errorFunc( CLDSM_ERR_INTERNAL, NULL);
            }
            break;
      }
      FreeSectionFilterBlock( idp, pSF );
      *ppSF = NULL;
   }

   dsmDP3(("exit DSC_SectionFilterStop\n"));
}

/*
-- Commit (add) an external section filter via callback (called once
-- PID is resolved)
*/
E_DscError DSC_SectionFilterCommit( P_DsmCoreInst idp,
   /*I*/ P_SecFilterInfo pSF )
{
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((pSF != NULL));
   dsmAssert((pSF->status != SFA_COMMITTED));
   dsmAssert((pSF->filter.pid != 0));

   DBGLOG(DD_SF, " PID=0x%x, tid=0x%x tidExt=0x%x msk=0x%x hdl=0x%x",
      pSF->filter.pid, pSF->filter.tableId, pSF->filter.tableIdExt,
      pSF->filter.tableIdExtMask, pSF->handle)

   /* -- Commit section filter */
   pSF->status = SFA_COMMITTED;

   /* u.filterHandle overwrites any previous value in union (ie. Query Ref) */
   pSF->u.filterHandle = idp->setup.addSectionFilterFunc( idp->setup.sfmInstance,
         &pSF->filter, pSF->handle );

   if (!pSF->u.filterHandle)
   {
      ERRLOG(DD_SF, "addSectionFilterFunc. returned NULL")
      pSF->status = SFA_ABORTED;
      err = CLDSM_ERR_SYSTEM_ADD_SECTION_FILTER;
   }
   else
   {
      /* Start acquisition timer */
      SectionTimerArm(idp, pSF);
      err = CLDSM_OK;
   }

   DBGLOG(DD_SF, " rtn: %u", err)
   return err;
}

/*
-- Abort (and notify) anything depending on filter
*/
void DSC_SectionFilterAbortDependants( P_DsmCoreInst idp,
   /*I*/ P_SecFilterInfo pSectionFilter )
{
   P_RootCarousel pRC;

   dsmDP3(("DSC_SectionFilterAbortDependants()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));
   dsmAssert(((pSectionFilter->status == SFA_PENDING) ||
              (pSectionFilter->status == SFA_ABORTED)));

   pSectionFilter->status = SFA_ABORTED;

   switch (pSectionFilter->target.kind)
   {
      case SFK_DSI:
         /*
         -- In this case, a carousel load is waiting for the DSI
         -- to arrive so notify a load error. Carousel load will then
         -- be unloaded/aborted by client.
         */
         pRC = pSectionFilter->target.u.pRC;
         dsmAssert((pRC->status == RCS_BOOTING));

         /*
         -- In this state, the DSI section filter will have been
         -- requested, so stop it (also Nulls reference in OC).
         */
         dsmAssert((pRC->pDsiSf));
         DSC_SectionFilterStop( idp, &pRC->pDsiSf );

         /* -- Finalise and destroy carousel load request */
         DSC_RootCrslAbortLoadRequest( idp, pRC );
         break;


      case SFK_DII:
         /*
         -- In this case, one or more object loads are waiting for a DII
         -- message to arrive so they can start the module loads, so
         -- unload the corresponding data carousel (which in turn deletes
         -- the relevant section filter). Unloading the data carousel also
         -- deletes all attached modules, this notifies any load requests of
         -- a load error and corresponding object loads will subsequently
         -- be unloaded/aborted by client
         */
         DSC_DataCrslDelete( idp, pSectionFilter->target.u.pDC );
         break;

      /*
      case SFK_DII_NEXT:
          TODO: Abort anything depending on next DII filter
                (maybe just marks cache content as stale?)
      break;
      */

      case SFK_DDB:
         /*
         -- In this case, one or more object loads are waiting for
         -- a module to arrive so unload the module (which in turn deletes
         -- relevant section filters). Unloading the module notifies all
         -- load requests of a load error and corresponding object loads
         -- will subsequently be unloaded/aborted by client
         */
         DSC_ModuleDeleteDcTidyUp( idp, pSectionFilter->target.u.pModule );
         break;

      case SFK_STREAM_DESCR:
         /* TODO: Abort anything depending on stream descriptor filter */
         break;

      default:
         dsmDP1(("ERROR: Illegal section filter kind = %u\n",
                 pSectionFilter->target.kind));
         dsmAssert((0));

         /* -- Notify an internal error here since can't return one */
         if (idp->setup.errorFunc)
         {
            idp->setup.errorFunc( CLDSM_ERR_INTERNAL, NULL);
         }
         break;
   }

   dsmDP3(("exit DSC_SectionFilterAbortDependants\n"));
}

/*
-- Retrieve filter from handle - called when section data arrives
*/
P_SecFilterInfo DSC_SectionFilterRetrieve( P_DsmCoreInst idp, U32BIT handle )
{
   P_SecFilterHeap pSfHeap;
   P_SecFilterInfo pSectionFilter;

   ASSERT(idp != NULL);
   pSfHeap = idp->sectionFilterHeap;

   pSectionFilter = pSfHeap->sfActiveHeads[handle & HASH_MASK];
   while (pSectionFilter != NULL)
   {
      if (pSectionFilter->handle == handle)
      {
         break;
      }
      pSectionFilter = pSectionFilter->nextBlock;
   }
   return pSectionFilter;
}

/*------------------------------ Local Functions -----------------------------*/

/*
-- Initialise section filter heap
*/
static void SectionFilterHeapInit( P_SecFilterHeap pSfHeap )
{
   S_SecFilterInfo *p_secFlt;
   U32BIT i, total;

   dsmDP3(("SectionFilterHeapInit()\n"));
   dsmAssert((pSfHeap != NULL));

   dsmDP4(("pSfHeap->sectionFiltersInUse = %d", pSfHeap->sectionFiltersInUse));

   pSfHeap->sectionFiltersInUse = 0;

   p_secFlt = pSfHeap->sectionFilters;
   pSfHeap->sfFreeListHead = p_secFlt;
   total = pSfHeap->totalFilters;

   /* -- Clear all section filter blocks & setup free list */
   for (i = 0; i != total; i++, p_secFlt++)
   {
      /* -- NULL filter contents */
      p_secFlt->filter.pid = 0;
      p_secFlt->filter.priority = (U8BIT)SF_PRIORITY_LOW;
      p_secFlt->filter.tableId = 0;
      p_secFlt->filter.tableIdExt = 0;
      p_secFlt->filter.tableIdExtMask = 0x0000;

      p_secFlt->status = SFA_INITIAL;
      p_secFlt->target.kind = SFK_NONE;
      p_secFlt->target.id = 0;
      p_secFlt->target.u.pRC = NULL;

      p_secFlt->handle = 0;

      /* -- Link blocks into free list */
      p_secFlt->nextBlock = p_secFlt + 1;
   }

   p_secFlt--;
   p_secFlt->nextBlock = NULL;
   pSfHeap->sfFreeListTail = p_secFlt;

   for (i = 0; i != HASH_NUM; i++)
   {
      pSfHeap->sfActiveHeads[i] = NULL;
   }

   dsmDP3(("exit SectionFilterHeapInit\n"));
}

/*
-- Allocate section filter memory block from heap
*/
static P_SecFilterInfo AllocSectionFilterBlock( P_DsmCoreInst idp )
{
   P_SecFilterHeap pSfHeap;
   P_SecFilterInfo pSectionFilter = NULL;
   U32BIT handle;

   dsmDP3(("AllocSectionFilterBlock()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((idp->sectionFilterHeap != NULL));

   pSfHeap = idp->sectionFilterHeap;

#ifdef SF_STATS
   sfStatsNum++;
   if (sfStatsMax < sfStatsNum)
      sfStatsMax = sfStatsNum;
   ERRPRINT("sfStatsNum=%d sfStatsMax=%d", sfStatsNum, sfStatsMax)
#endif

   dsmAssert((pSfHeap->sectionFiltersInUse <= pSfHeap->totalFilters));

   handle = pSfHeap->activeCounter++;
   handle|= HNDL_MAGIC;

   if (pSfHeap->sectionFiltersInUse < pSfHeap->totalFilters)
   {
      ASSERT(pSfHeap->sfFreeListHead != NULL)

      pSfHeap->sectionFiltersInUse++;

      /* -- Get section filter block at head of free list */
      pSectionFilter = pSfHeap->sfFreeListHead;
      pSectionFilter->handle = handle;

      pSfHeap->sfFreeListHead = pSectionFilter->nextBlock;
      if (pSfHeap->sfFreeListHead == NULL)
      {
         pSfHeap->sfFreeListTail = NULL;
      }

      handle = handle & HASH_MASK;
      pSectionFilter->nextBlock = pSfHeap->sfActiveHeads[handle];
      pSfHeap->sfActiveHeads[handle] = pSectionFilter;
      DBGLOG(DD_SF, "rtn: %p handle=0x%x", pSectionFilter, pSectionFilter->handle)
   }
   else
   {
      ERRPRINT("no free section filters")
   }
   return pSectionFilter;
}

/*
-- Return section filter memory block to free heap
*/
static void FreeSectionFilterBlock( P_DsmCoreInst idp,
   /*I*/ P_SecFilterInfo pSectionFilter )
{
   P_SecFilterHeap pSfHeap;
   P_SecFilterInfo *ppSfActiveHead;

   DBGLOG(DD_SF, " handle=0x%x", pSectionFilter->handle)
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));

   pSfHeap = idp->sectionFilterHeap;

#ifdef SF_STATS
   ERRPRINT("sfStatsNum=%d sfStatsMax=%d", sfStatsNum, sfStatsMax)
   sfStatsNum--;
#endif

   dsmAssert((
         (pSectionFilter >= &pSfHeap->sectionFilters[0]) &&
         (pSectionFilter < &pSfHeap->sectionFilters[ pSfHeap->totalFilters ])
         ));
   dsmAssert((pSfHeap->sectionFiltersInUse > 0));
   dsmAssert((pSfHeap->sectionFiltersInUse <= pSfHeap->totalFilters));
   dsmAssert(((pSectionFilter->handle&HNDL_MASK) == HNDL_MAGIC));
   if ((pSectionFilter->handle&HNDL_MASK) == HNDL_MAGIC)
   {
      ppSfActiveHead = pSfHeap->sfActiveHeads + (pSectionFilter->handle & HASH_MASK);
      while (*ppSfActiveHead != NULL)
      {
         if (*ppSfActiveHead == pSectionFilter)
         {
            *ppSfActiveHead = pSectionFilter->nextBlock;
            break;
         }
         ppSfActiveHead = &((*ppSfActiveHead)->nextBlock);
      }

      /* -- NULL contents of section filter block (helps stale filter handles
       -- to be detected) */
      pSectionFilter->filter.pid = 0;
      pSectionFilter->filter.priority = (U8BIT)SF_PRIORITY_LOW;
      pSectionFilter->filter.tableId = 0;
      pSectionFilter->filter.tableIdExt = 0;
      pSectionFilter->filter.tableIdExtMask = 0x0000;

      dsmDP4(("pSectionFilter = %p", pSectionFilter));
      pSectionFilter->status = SFA_INITIAL;
      pSectionFilter->target.kind = SFK_NONE;
      pSectionFilter->target.id = 0;
      pSectionFilter->target.u.pRC = NULL;

      pSectionFilter->handle = 0;

      pSectionFilter->nextBlock = NULL;

      /* -- Put block on tail of free list */
      if (pSfHeap->sfFreeListTail == NULL)
      {
         ASSERT( pSfHeap->sfFreeListHead == NULL )
         pSfHeap->sfFreeListHead = pSectionFilter;
      }
      else
      {
         pSfHeap->sfFreeListTail->nextBlock = pSectionFilter;
      }
      pSfHeap->sfFreeListTail = pSectionFilter;

      pSfHeap->sectionFiltersInUse--;
   }
   dsmDP3(("exit FreeSectionFilterBlock\n"));
}

#ifdef ACCELERATE_CAROUSEL_LOADING

E_DscError sectionLargeDDBFilterStart( P_DsmCoreInst idp,
   E_SecFiltKind sfKind,
   U16BIT service_id,
   E_SFPriority sfPriority,
   /*O*/ P_SecFilterInfo *pSectionFilter,
   U16BIT assocTag)
{
   P_SecFilterInfo pSF;
   E_DscError err = CLDSM_OK;
   S_SIQueryRequest siQueryData;
   S_SIQueryResult siQueryResult;

   dsmDP3(("sectionLargeDDBFilterStart()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));


   *pSectionFilter = NULL;

   pSF = AllocSectionFilterBlock( idp );

   if (!pSF)
   {
      /* TODO: HANDLE LIMITED SECTION FILTERS CORRECTLY */
      dsmDP1(("ERROR: Limited section filters not yet supported\n"));
      dsmAssert((0));
      err = CLDSM_ERR_SECTION_FILTER_HEAP_FULL;
      goto _return;                                                 /* EXIT */
   }

   pSF->target.kind = sfKind;
   pSF->target.u.pRC = 0;
   pSF->hCarousel = NULL;

   pSF->target.id = 145;


   pSF->filter.tableId = DDB_ID;
   pSF->filter.tableIdExt = 0xFFFF;

   pSF->filter.tableIdExtMask = 0x0000;

   if (!err)
   {
      pSF->filter.priority = sfPriority;

      /* -- Make SI Query to get PID for section filter -- */
      siQueryData.kind = SIQUERY_PID;
      siQueryData.service_id = service_id;
      siQueryData.data.association_tag = assocTag;

      err = siQueryStart( idp, &siQueryData, (void *) pSF, &siQueryResult );
      switch (err)
      {
         case CLDSM_PENDING:
            /* -- Store query ref with filter */
            pSF->u.pPendingSiQueryRef = siQueryResult.data.queryHandle;
            pSF->status = SFA_PENDING;
            *pSectionFilter = pSF;
            err = CLDSM_OK;
            break;

         case CLDSM_OK:
            /* -- Store PID in filter */
            pSF->filter.PID = siQueryResult.data.PID;
            err = DSC_SectionFilterCommit( idp, pSF );
            if (!err)
            {
               *pSectionFilter = pSF;
               break;
            }

         default:
            FreeSectionFilterBlock( idp, pSF );
      }
   }

   if (err)
   {
      /* -- NB. Cannot get here if add section filter successful so
         -- no requirement to stop section filter */
   }
   else
   {
   }

_return:
   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: sectionLargeDDBFilterStart %u\n", err)));

   return err;
}

void largeDDBsectionFilterStop( P_DsmCoreInst idp)
{
   P_SecFilterInfo pSF;

   P_SecFilterInfo *pSectionFilter = &accCarouselLoadingInfos.largeDDBFilter;

   dsmDP3(("largeDDBsectionFilterStop()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));
   dsmAssert((*pSectionFilter != NULL));

   pSF = *pSectionFilter;


   /* -- NB. Make it safe in retail mode */
   if (pSF != NULL)
   {
      switch (pSF->status)
      {
         case SFA_PENDING:
            /*
            -- In this state, the section filter has not yet been
            -- added (committed) since it is waiting for an
            -- pending SI Query to complete, so abort the query.
             */
            dsmAssert((pSF->u.pPendingSiQueryRef != NULL));
            siQueryAbortPending( idp, &pSF->u.pPendingSiQueryRef );

            /* -- Clear Query ref in filter (finished with) */
            pSF->u.pPendingSiQueryRef = NULL;
            break;


         case SFA_COMMITTED:
            /* -- In this state, the section filter is actually started
               -- in the calling env. so delete it */
            idp->setup.delSectionFilterFunc( pSF->u.filterHandle, (H_DscSFRef)pSF,
            (P_SecFilter)pSF );
            break;


         case SFA_ABORTED:
            /* -- In this state, the section filter commit has failed so
               -- nothing specific to do (just allow filter to be freed) */
            break;


         default:
            /* -- 'Added' section filters should not be in any other state */
            dsmDP1(("ERROR: Illegal section filter status = %u\n",
                    pSF->status));
            dsmAssert((0));

            /* -- Notify an internal error here since can't return one */
            if (idp->setup.errorFunc)
            {
               idp->setup.errorFunc( CLDSM_ERR_INTERNAL, NULL);
            }
            break;
      }
      FreeSectionFilterBlock( idp, pSF );
      *pSectionFilter = NULL;
   }

   dsmDP3(("exit largeDDBsectionFilterStop\n"));
}

#endif /* ACCELERATE_CAROUSEL_LOADING */


/*----------------------------------------------------------------------------*/
