/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2002 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Module for managing timers for Load timeouts implementation.
 * @file    sectionTimer.c
 * @date    01/02/2002
 * @author  R.Freeman
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "sectionFilter.h"
#include "sectionTimer.h"

#include "objectCarousel.h"
#include "dataCarousel.h"
#include "module.h"
#include "siQuery.h"
#include "loadMgr.h"


/*------------------------------- Local Macros -------------------------------*/
#define SECTION_TIMER_DSI_TIMEOUT 5000

/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/


/*------------------- local prototypes/forward declarations ------------------*/

#ifdef DSI_TIMEOUT_SUPPORT
static E_DscError sectionTimerDSIArm( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo pSectionFilter);
#endif  /* DSI_TIMEOUT_SUPPORT */

static E_DscError sectionTimerDIIArm( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo pSectionFilter);

static E_DscError sectionTimerDDBArm( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo pSectionFilter);


/*---------------------------- Exported Functions ----------------------------*/

/*
-- According pSectionFilter-> targetKind call local function:
-- sectionTimerDSIArm(), sectionTimerDIIArm(), sectionTimerDSIArm()
*/
E_DscError SectionTimerArm( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo pSectionFilter)
{
   E_DscError err = CLDSM_OK;

   dsmDP3(("SectionTimerArm(pSectionFilter = 0x%p)\n", pSectionFilter));
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));

   pSectionFilter->tms.mainTimerHandle = NULL;
   pSectionFilter->tms.nextTimerHandle = NULL;
   pSectionFilter->tms.nextDuration = 0;

   switch (pSectionFilter->target.kind)
   {
      case SFK_DSI:
      #ifdef DSI_TIMEOUT_SUPPORT
         sectionTimerDSIArm(idp, pSectionFilter);
      #endif  /* DSI_TIMEOUT_SUPPORT */
         break;
      case SFK_DII:
         sectionTimerDIIArm(idp, pSectionFilter);
         break;
      case SFK_DDB:
         sectionTimerDDBArm(idp, pSectionFilter);
         break;

      case SFK_STREAM_DESCR: /* no timeout => avoid assert in default case */
         break;

      default:
         dsmDP1(("ERROR: Illegal section filter kind = %u\n", pSectionFilter->target.kind));
         dsmAssert((0));
         err = CLDSM_ERR_INTERNAL;
         break;
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: sectionTimerUpdate %u\n", err)));
   dsmDP3(("exit sectionTimerUpdate -> rtn: %u\n", err));
   return err;
}

/*
-- If the timer is refreshed, then re-arm main timer for the main duration
-- If the timer is not completed, then re-arm next timer for the
   next duration.
-- If completed, destroy all timers
-- In any case update the timer handle fields, reset it to 0 if timer is not
   re-armed

*/
E_DscError sectionTimerUpdate( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo pSectionFilter,
   /*I*/ BOOLEAN completed, BOOLEAN refresh)
{
   E_DscError err = CLDSM_OK;
   int cbErr;

   dsmDP3(("sectionTimerUpdate(pSectionFilter = 0x%p)\n", pSectionFilter));
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));

   if (idp->setup.startTimerFunc != NULL &&
       idp->setup.stopTimerFunc != NULL)
   {
      if (completed)
      {
         /* Destroy all timers */
         if (pSectionFilter->tms.nextTimerHandle != NULL)
         {
            idp->setup.stopTimerFunc( idp->setup.dsmControl, pSectionFilter->tms.nextTimerHandle );
         }

         if (pSectionFilter->tms.mainTimerHandle != NULL)
         {
            /* Destroy the main timer */
            idp->setup.stopTimerFunc( idp->setup.dsmControl, pSectionFilter->tms.mainTimerHandle );
         }

         /* Reset the timer handle fields */
         pSectionFilter->tms.mainTimerHandle = NULL;
         pSectionFilter->tms.nextTimerHandle = NULL;
         pSectionFilter->tms.nextDuration = 0;
      }
      else
      {
         void **timerHandle = &pSectionFilter->tms.nextTimerHandle;
         U32BIT duration = pSectionFilter->tms.nextDuration;
         if (refresh && pSectionFilter->target.kind == SFK_DDB)
         {
            timerHandle = &pSectionFilter->tms.mainTimerHandle;
            duration = pSectionFilter->tms.mainDuration;
         }

         /* Destroy the next timer if running */
         if (*timerHandle != NULL)
         {
            idp->setup.stopTimerFunc( idp->setup.dsmControl, *timerHandle );
         }

         if (duration != 0xFFFFFFFF && duration > 0)
         {
            /* Re-arm next timer for the next duration */
            cbErr = idp->setup.startTimerFunc( idp->setup.dsmControl, duration, pSectionFilter, timerHandle );
            if (cbErr)
            {
               dsmDP1(("ERROR: sectionTimerUpdate : startTimerFunc failure"));
               //dsmAssert((0));
               err = CLDSM_ERR_INTERNAL;
            }
         }
      }   /* if (completed) */
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: sectionTimerUpdate %u\n", err)));
   dsmDP3(("exit sectionTimerUpdate -> rtn: %u\n", err));
   return err;
}

/*
-- If available, destroy the timer
*/
E_DscError sectionTimerRemove(  P_DsmCoreInst idp,
   /*I*/ P_SecFilterInfo pSectionFilter)
{
   E_DscError err = CLDSM_OK;

   dsmDP3(("sectionTimerRemove(pSectionFilter = 0x%p)\n", pSectionFilter));
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));

   if (idp->setup.stopTimerFunc != NULL)
   {
      if (NULL != pSectionFilter->tms.mainTimerHandle)
      {
         DBGLOG(DD_TIMER,"(pSF = 0x%p) thdl=%x", pSectionFilter, pSectionFilter->tms.mainTimerHandle);
         idp->setup.stopTimerFunc( idp->setup.dsmControl, pSectionFilter->tms.mainTimerHandle );
         pSectionFilter->tms.mainTimerHandle = NULL;
      }

      if (NULL != pSectionFilter->tms.nextTimerHandle)
      {
         idp->setup.stopTimerFunc( idp->setup.dsmControl, pSectionFilter->tms.nextTimerHandle );
         pSectionFilter->tms.nextTimerHandle = NULL;
      }
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: sectionTimerRemove %u\n", err)));
   dsmDP3(("exit sectionTimerRemove -> rtn: %u\n", err));
   return err;
}

/*------------------------------ Local Functions -----------------------------*/

#ifdef DSI_TIMEOUT_SUPPORT
/*
-- Arm a timer for DSI section acquisition Timeout, that is to say 5000 milliseconds
*/
static E_DscError sectionTimerDSIArm( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo pSectionFilter)
{
   E_DscError err = CLDSM_OK;
   int cbErr;

   DBGLOG(DD_TIMER,"(pSectionFilter = 0x%p)", pSectionFilter)
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));

   if (idp->setup.startTimerFunc != NULL)
   {
      cbErr = idp->setup.startTimerFunc( idp->setup.dsmControl, SECTION_TIMER_DSI_TIMEOUT,
            pSectionFilter, &(pSectionFilter->tms.mainTimerHandle));
      if (cbErr)
      {
         dsmDP1(("ERROR: sectionTimerDSIArm : startTimerFunc failure\n"));
         dsmAssert((0));
         err = CLDSM_ERR_INTERNAL;
      }
      pSectionFilter->tms.nextTimerHandle = 0;
      pSectionFilter->tms.nextDuration = 0;
   }

   DEBUG_CHK( err == CLDSM_OK,
      dsmDP1(("ERROR: sectionTimerDSIArm %u\n", err)));
   dsmDP3(("exit sectionTimerDSIArm -> rtn: %u\n", err));
   return err;
}

#endif  /* DSI_TIMEOUT_SUPPORT */

/*
-- Arm a timer for DII section acquisition Timeout
*/
static E_DscError sectionTimerDIIArm( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo pSectionFilter)
{
   E_DscError err = CLDSM_OK;
   int cbErr;
   P_DataCarousel pDC;
   U32BIT timeout;

   DBGLOG(DD_TIMER,"(pSectionFilter = 0x%p)", pSectionFilter)
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));

   if (idp->setup.startTimerFunc != NULL)
   {
      /* duration is retrieved from target handle (hDataCarousel) :: tap::timeout */
      pDC = pSectionFilter->target.u.pDC;
      timeout = pDC->tap.timeout; /*s*/

      dsmDP3(("sectionTimerDIIArm(-) timeout = %u microsecs\n", timeout));
      cbErr = idp->setup.startTimerFunc( idp->setup.dsmControl, timeout / 1000, pSectionFilter, &(pSectionFilter->tms.mainTimerHandle));
      if (cbErr)
      {
         dsmDP1(("ERROR: sectionTimerDIIArm : startTimerFunc failure\n"));
         err = CLDSM_ERR_INTERNAL;
      }
      pSectionFilter->tms.nextTimerHandle = 0;
      pSectionFilter->tms.nextDuration = 0;
   }
   return err;
}

/*
-- Arm a timer for DDB sections acquisition Timeout
*/
static E_DscError sectionTimerDDBArm( P_DsmCoreInst idp,
   /*IO*/ P_SecFilterInfo pSectionFilter)
{
   E_DscError err = CLDSM_OK;
   int cbErr;
   P_Module pModule;
   U32BIT moduleTimeout, blockTimeout;

   dsmDP3(("sectionTimerDDBArm(pSectionFilter= 0x%p)\n", pSectionFilter));
   dsmAssert((idp != NULL));
   dsmAssert((pSectionFilter != NULL));

   if ((idp->setup.startTimerFunc != NULL) && (NULL != pSectionFilter->target.u.pModule))
   {
      /* duration (s) is retrieved from target handle */
      /* (hModule) :: moduleInfo:: profileInfo:: moduleTimeout */
      pModule = pSectionFilter->target.u.pModule;
      if ( pModule->moduleInfo.crslMagic == OC_MAGIC )
      {
         moduleTimeout = pModule->moduleInfo.u.mhgp.moduleTimeout;
         if (moduleTimeout != 0)
         {
            moduleTimeout /= 1000; /*s -> ms*/

            /* Next duration (s) is retrieved from target heandle */
            /* (hModule) :: moduleInfo:: profileInfo:: blockTimeout */
            blockTimeout = pModule->moduleInfo.u.mhgp.blockTimeout / 1000; /*s -> ms*/

            dsmDP3(("sectionTimerDDBArm(-) moduleTimeout = %u blockTimeout = %u microsecs\n", moduleTimeout, blockTimeout));
            cbErr = idp->setup.startTimerFunc( idp->setup.dsmControl, moduleTimeout, pSectionFilter, &(pSectionFilter->tms.mainTimerHandle));
            if (cbErr)
            {
               dsmDP1(("ERROR: sectionTimerDDBArm : startTimerFunc failure\n"));
               err = CLDSM_ERR_INTERNAL;
            }
            pSectionFilter->tms.nextTimerHandle = 0;
            pSectionFilter->tms.mainDuration = moduleTimeout;
            pSectionFilter->tms.nextDuration = blockTimeout;
         }
      }
   }
   return err;
}

/*----------------------------------------------------------------------------*/
