/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2002 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions for managing SI queries
 * @file    siQuery.c
 * @date    07/05/2002
 * @author  N Kirkland
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "siQuery.h"

#include "linkList.h"
#include "cacheMgr.h"
#include "objectCarousel.h"
#include "sectionFilter.h"
#include "streamObject.h"
#include "loadMgr.h"

/*---constant definitions for this file--------------------------------------*/


/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/

/*---local function prototypes for this file---------------------------------*/

static E_DscError siQueryCreate( P_DsmCoreInst idp, P_SiQuery *ppSiQuery );

static void siQueryDestroy( P_DsmCoreInst idp, P_SiQuery pSiQuery );

static P_SiQuery FindPendingPIDQuery( P_DsmCoreInst idp, U16BIT service_id, U16BIT assoc_tag );


/*---global function definitions---------------------------------------------*/

/**
 * @brief   Starts an SI Query. First tests if we have already made this query and the
 *          results are stored or still pending. Otherwise tries SI query callback.
 *          If results are not synchronously returned, stores state of query ready for
 *          when they are asynchronously returned (via clDsmProcessSiQueryEvent).
 *          NB. If start returns results SYNCHRONOUSLY then siQueryStop must not be
 *          called.
 * @param   idp Pointer to instance data.
 * @param   pQueryData Pointer to structure containing query
 *          details.
 * @param   queryTarget Handle to target of query (ie. object that
 *          needs query results - eg. carousel or
 *          section filter).
 * @param   pResult Results of query if synchronously returned.
 * @return
 *          CLDSM_OK
 *          Everything hunkydory
 *          CLDSM_ERR_SI_QUERY_FAILED
 *          Synchronous SI query failed to determine requested information
 *          CLDSM_MEM_HEAP_FULL
 *          Internal memory manager heap full (treat as internal error)
 */
E_DscError siQueryStart( P_DsmCoreInst idp,
   P_SIQueryRequest pQueryData, void *queryTarget,
   P_SIQueryResult pResult )
{
   E_DscError err;
   P_SiQuery pSiQueryRef;
   P_SiQuery pOrigSiQueryRef;

   dsmDP3(("\nsiQueryStart()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pQueryData != NULL));
   dsmAssert((pResult != NULL));

   dsmDP4(("lastPID=%d lpStored=%d, lastAssocTag=%d\n",
           idp->lastPID, idp->lastPIDResultStored, idp->lastAssociationTag));
   dsmDP4(("kind=%d stored=%d, AssocTag=%d\n",
           pQueryData->kind, idp->setup.storeSIQueryResults, pQueryData->data.association_tag));

   /* -- Check if PID lookup result is already stored and we can use it */
   if ((idp->setup.storeSIQueryResults) &&
       (pQueryData->kind == SIQUERY_PID) &&
       (idp->lastPIDResultStored) &&
       (pQueryData->serviceId == idp->lastServiceId) &&
       (pQueryData->associationTag == idp->lastAssociationTag))
   {
      /* -- Get stored PID result */
      pResult->kind = SIQUERY_PID;
      pResult->data.pid = idp->lastPID;
      err = CLDSM_OK;
   }
   else if (pQueryData->kind == SIQUERY_SSU_PID)
   {
      /* Special case of ssu PID being given to DSMCC API (and held in service id field) */
      pResult->kind = SIQUERY_SSU_PID;
      pResult->data.carouselInfo.pid = pQueryData->serviceId;
      pResult->data.carouselInfo.carouselId = INVALID_CAROUSEL_ID;
      pResult->data.carouselInfo.associationTag = INVALID_ASSOCIATION_TAG;
      idp->lastServiceId = pQueryData->serviceId;
      idp->lastAssociationTag = (U16BIT)INVALID_ASSOCIATION_TAG;
      idp->lastPID = pResult->data.carouselInfo.pid;
      idp->lastPIDResultStored = TRUE;
      err = CLDSM_OK;
   }
   else
   {
      err = siQueryCreate( idp, &pSiQueryRef );
      if (!err)
      {
         dsmDP3(("pSiQueryRef = %p\n", pSiQueryRef));
         dsmDP4(("pQueryData->kind = %s\n",
                 pQueryData->kind == SIQUERY_PID ? "SIQUERY_PID" :
                 pQueryData->kind == SIQUERY_BOOT_CAROUSEL ? "SIQUERY_BOOT_CAROUSEL" :
                 pQueryData->kind == SIQUERY_FIND_CAROUSEL ? "SIQUERY_FIND_CAROUSEL" :
                 pQueryData->kind == SIQUERY_CAROUSEL_INFO ? "SIQUERY_CAROUSEL_INFO" :
                 pQueryData->kind == SIQUERY_SSU_CAROUSEL ? "SIQUERY_SSU_CAROUSEL" :
                 pQueryData->kind == SIQUERY_SSU_PID ? "SIQUERY_SSU_PID" :
                 pQueryData->kind == SIQUERY_CAROUSEL ? "SIQUERY_CAROUSEL" :
                 pQueryData->kind == SIQUERY_PIDLIST ? "SIQUERY_PIDLIST" :
                 pQueryData->kind == SIQUERY_DEFERRED_SERVICE ? "SIQUERY_DEFERRED_SERVICE" :
                 "UNKNOWN"));

         if (pQueryData->kind == SIQUERY_PID)
         {
            pOrigSiQueryRef = FindPendingPIDQuery( idp, pQueryData->serviceId, pQueryData->associationTag );
         }
         else
         {
            pOrigSiQueryRef = NULL;
         }

         if (pOrigSiQueryRef == NULL)
         {
            dsmDP3(("siQueryStart() pSiQueryRef=%p target=%p\n", pSiQueryRef, queryTarget));
            err = idp->setup.startSIQueryFunc( idp->setup.siqInstance,
                  pQueryData, (H_SIQueryRef) pSiQueryRef,
                  idp->generation.ptr, pResult );

            switch (err)
            {
               case CLDSM_PENDING:
                  DBGLOG(DD_QUERY, "kind=%d target=%p qhdl=%p\n", pResult->kind, queryTarget, pResult->data.queryHandle)
                  dsmAssert((pResult->kind == SIRESULT_PENDING));
                  /* -- Store details of query in query ref */
                  pSiQueryRef->query = *pQueryData;
                  pSiQueryRef->target = queryTarget;
                  pSiQueryRef->state = SIQS_PENDING;
                  pSiQueryRef->original = TRUE;
                  pSiQueryRef->queryHandle = pResult->data.queryHandle;
                  pResult->data.queryHandle = pSiQueryRef;

                  /* -- Insert SI Query in current active list */
                  LLInsertHead( idp->llcCurrSiQueries, pSiQueryRef );
                  break;

               case CLDSM_OK:
                  if (pQueryData->kind != pResult->kind)
                  {
                     dsmDP1(("kind=%d %d\n", pQueryData->kind, pResult->kind));
                  }
                  dsmAssert((pQueryData->kind == pResult->kind));
                  DBGLOG(DD_QUERY, "lastPID=%d lpStored=%d, lastAssocTag=%d\n",
                          idp->lastPID, idp->lastPIDResultStored, idp->lastAssociationTag)
                  switch (pQueryData->kind)
                  {
                     case SIQUERY_PID:
                        /* -- Store association_tag to PID mappings */
                        idp->lastServiceId = pQueryData->serviceId;
                        idp->lastAssociationTag = pQueryData->associationTag;
                        idp->lastPID = pResult->data.pid;
                        idp->lastPIDResultStored = TRUE;
                        dsmDP3(("data.PID = %x\n", pResult->data.pid));
                        break;

                     case SIQUERY_PIDLIST:
                        /* TODO: case SIQUERY_PIDLIST: If/when store of all PIDs
                           in PMT implemented */
                        break;

                     case SIQUERY_CAROUSEL:
                     case SIQUERY_CAROUSEL_INFO:
                     case SIQUERY_FIND_CAROUSEL:
                     case SIQUERY_BOOT_CAROUSEL:
                     case SIQUERY_SSU_CAROUSEL:
                     case SIQUERY_SSU_PID:
                        /*
                        -- Store component_tag to PID to prevent a
                        -- subsequent unnecessary SI query when determining
                        -- the PID for the DSI of this carousel
                        */
                        idp->lastServiceId = pQueryData->serviceId;
                        idp->lastAssociationTag = (U16BIT)pResult->data.carouselInfo.associationTag;
                        idp->lastPID = pResult->data.carouselInfo.pid;
                        idp->lastPIDResultStored = TRUE;

                        dsmDP3(("data.carouselInfo.PID = %x\n", pResult->data.carouselInfo.pid));
                        dsmDP3(("data.carouselInfo.component_tag = %x\n", pResult->data.carouselInfo.associationTag));
                        break;

                     case SIQUERY_DEFERRED_SERVICE:
                        break;

                     default:
                        /* -- No useful PID results in any other queries */
                        dsmAssert((0));
                        break;
                  }
                  dsmDP4(("NEW lastPID=%d lpStored=%d, lastAssocTag=%d\n",
                          idp->lastPID, idp->lastPIDResultStored, idp->lastAssociationTag));

               /* Fall though to destroy SI Query ref since no longer needed */

               default: /* ERROR cases */
                  siQueryDestroy( idp, pSiQueryRef );
            }
         }
         else
         {
            /* -- This is a duplicate PID Query */
            pSiQueryRef->query = *pQueryData;
            pSiQueryRef->target = queryTarget;
            pSiQueryRef->state = SIQS_PENDING;
            pSiQueryRef->original = FALSE;

            pResult->kind = SIRESULT_PENDING;
            pResult->data.queryHandle = pSiQueryRef;

            dsmAssert((pOrigSiQueryRef->original == TRUE));

            if (pOrigSiQueryRef->llcDuplSiQuerys == NULL)
            {
               /* -- Create list for duplicate queries */
               err = LLCreate( idp, pOrigSiQueryRef, DUPLICATE_SI_QUERY_LIST,
                     &pOrigSiQueryRef->llcDuplSiQuerys );
               if (err)
               {
                  siQueryDestroy( idp, pSiQueryRef );
               }
               else
               {
                  LLInsertTail( pOrigSiQueryRef->llcDuplSiQuerys, pSiQueryRef );
                  err = CLDSM_PENDING;
               }
            }
            else
            {
               LLInsertTail( pOrigSiQueryRef->llcDuplSiQuerys, pSiQueryRef );
               err = CLDSM_PENDING;
            }
         }
      }
   }

   DEBUG_CHK((err == CLDSM_OK || err == CLDSM_PENDING || err == CLDSM_ERR_SI_QUERY_FAILED),
      dsmDP1(("ERROR: siQueryStart: %u\n", err)));
   dsmDP3(("exit siQueryStart -> rtn: %u\n", err));
   return err;
}

/**
 * @brief   Stops the specified SI Query. Removes from any lists it is in and
 *          destroys associated memory object.
 *          NB. If siQueryStart returned results SYNCHRONOUSLY then this must not be
 *          called.
 * @param   idp Pointer to instance data.
 * @param   pSiQueryRef Pointer to handle of query to stop (nulled after call).
 * @return
 */
void siQueryStop( P_DsmCoreInst idp, P_SiQuery pSiQueryRef )
{
   dsmDP3(("siQueryStop()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pSiQueryRef != NULL));
   dsmAssert((pSiQueryRef->magic == SI_QUERY_MAGIC));

   dsmDP2(("siQueryStop(idp,pSiQueryRef=%p) state=%d target=%p\n", pSiQueryRef, pSiQueryRef->state, pSiQueryRef->target));

   if (pSiQueryRef->original == TRUE)
   {
      switch (pSiQueryRef->state)
      {
         case SIQS_EXPIRED:
         case SIQS_COMPLETED:
            /*
            -- In these states, a pending SI query is now completed. If it
            -- is an original query then stop/ack the external query.
            */

            /* -- Any duplicate queries should have already been handled */
            dsmAssert((pSiQueryRef->llcDuplSiQuerys == NULL));

            /* -- Stop/ack external query */
            idp->setup.stopSIQueryFunc( idp->setup.siqInstance, pSiQueryRef->queryHandle,
            (H_SIQueryRef) pSiQueryRef );
            break;

         case SIQS_ABORTED:
            /*
            -- In this state, an already aborted SI query is being stopped
            -- so nothing specific to do.
            */
            break;

         default:
            /* SI Query Refs that require stopping should not be in any other state.*/
            dsmDP1(("ERROR: Illegal SI query status = %u\n", pSiQueryRef->state));
            dsmAssert((0));
            /* -- Notify an internal error here since can't return one */
            if (idp->setup.errorFunc)
            {
               idp->setup.errorFunc( CLDSM_ERR_INTERNAL, NULL);
            }
            break;
      }

      /* -- Remove from active query list */
      LLRemove( pSiQueryRef, CURR_SI_QUERY_LIST );
   }
   /* else - this is a duplicate query, so nothing specific to do */


   /* -- Destroy it (use original handle so it is Nulled) */
   siQueryDestroy( idp, pSiQueryRef );

   dsmDP3(("exit siQueryStop\n"));
}

/**
 * @brief   Aborts the specified SI Query before completion. Stops/destroys duplicate
 *          queries. For original queries or duplicate queries that have expired parents
 *          (and no further duplicates) calls stop SI Query callback on relevant query.
 * @param   idp Pointer to instance data.
 * @param   phPendingSiQueryRef Pointer to handle of query to abort.
 * @return
 */
void siQueryAbortPending( P_DsmCoreInst idp, P_SiQuery pSiQueryRef )
{
   P_SiQuery pOrigSiQueryRef;
   P_SiQuery pExtSiQueryRefToStop;
   BOOLEAN duplicateQuery;
   BOOLEAN duplicateListEmpty;
   void *queryHandle;

   dsmDP3(("siQueryAbortPending()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pSiQueryRef != NULL));
   dsmAssert((pSiQueryRef->magic == SI_QUERY_MAGIC));

   pExtSiQueryRefToStop = NULL;

   switch (pSiQueryRef->state)
   {
      case SIQS_PENDING:

         /*
         -- In this state, an active SI query is being aborted. If it
         -- is an original query or the last duplicate query on an expired
         -- original then mark the original query ref as aborted and stop
         -- the external query.
         -- NB. Only stop/destroy the query here if it is a duplicate since
         -- original queries will be stopped/destroyed when the external
         -- query stop is acknowledged via clDsmProcessSIQueryEvent.
         */
         pSiQueryRef->state = SIQS_ABORTED;

         if (pSiQueryRef->original == TRUE)
         {
            duplicateQuery = FALSE;

            if (pSiQueryRef->llcDuplSiQuerys != NULL)
            {
               /*
               -- This original query has duplicates so don't stop
               -- external query yet but but set state to EXPIRED
               */
               pSiQueryRef->state = SIQS_EXPIRED;
            }
            else
            {
               /* -- No duplicates so stop the external query */
               pExtSiQueryRefToStop = pSiQueryRef;
            }
         }
         else
         {
            /*
            -- This is a duplicate query so no external query to stop for
            -- this but do need to check if there is an expired parent
            -- query that needs stopping.
            */
            duplicateQuery = TRUE;

            pOrigSiQueryRef = LLParent( pSiQueryRef, DUPLICATE_SI_QUERY_LIST );

            /* -- Should always be a parent */
            dsmAssert((pOrigSiQueryRef != NULL));

            duplicateListEmpty = LLRemove( pSiQueryRef, DUPLICATE_SI_QUERY_LIST );

            if (duplicateListEmpty == TRUE)
            {
               /* -- Destroy the parent list */
               LLDestroy( idp, &pOrigSiQueryRef->llcDuplSiQuerys );

               if (pOrigSiQueryRef->state == SIQS_EXPIRED)
               {
                  pExtSiQueryRefToStop = pOrigSiQueryRef;
               }
               /* else - parent query is still active */
            }
            /* else - still more duplicate queries */
         }

         if (duplicateQuery == TRUE)
         {
            /* -- Cannot be the same for duplicate queries */
            dsmAssert((pExtSiQueryRefToStop != pSiQueryRef));

            /* -- Use passed in handle so it gets Nulled */
            siQueryStop( idp, pSiQueryRef );
         }

         if (pExtSiQueryRefToStop != NULL)
         {
            /* -- There is an external query to stop */

            /* -- Get the correct queryHandle */
            pSiQueryRef = pExtSiQueryRefToStop;
            queryHandle = pSiQueryRef->queryHandle;

            idp->setup.stopSIQueryFunc( idp->setup.siqInstance, queryHandle,
               (H_SIQueryRef) pExtSiQueryRefToStop );
         }
         break;


      default:
         /*
         -- SI Query Refs that are aborted should not be in any other state.
         */

         dsmDP1(("ERROR: Illegal SI query status = %u\n",
                 pSiQueryRef->state));
         dsmAssert((0));

         /* -- Notify an internal error here since can't return one */
         if (idp->setup.errorFunc)
         {
            idp->setup.errorFunc( CLDSM_ERR_INTERNAL, NULL);
         }

         /* -- NB. Since there is probably some internal memory corruption
            -- it is safest not to do anything else here.
            -- DSM-CC should be shut down (clDsmDestroy) in response to
            -- this internal error otherwise this may cause a memory leak
            -- in the MemMgr */
         break;
   }

   dsmDP3(("exit siQueryAbortPending\n"));
}

#ifdef GET_PID_LIST
E_DscError siQueryGetPIDList( P_DsmCoreInst idp )
{
   E_DscError err;
   S_SIQueryRequest siQueryData;
   S_SIQueryResult siQueryResult;

   siQueryData.kind = SIQUERY_PIDLIST;
   siQueryData.service_id = idp->curr_service_id;

   err = siQueryStart( idp, &siQueryData, (void *) 0x91D, &siQueryResult );

   dsmDP1(("(%d) SI query status = %u\n", err, siQueryStatus));
   switch (err)
   {
      case CLDSM_OK:
         break;

      case CLDSM_PENDING:
         break;

      default:
         break;
   }

   return err;
}

#endif /*GET_PID_LIST*/


/**
 * @brief   Processes and stops the specified SI Query.
 * @param   idp Pointer to instance data.
 * @param   pSiQueryRef Refererence for this query (value supplied at
 *          start time).
 * @param   status Status of query results.
 * @param   pResult Pointer to results of query (if successful).
 * @return
 *          CLDSM_OK (0)
 *          The process SI query event was executed with no problems.
 *          CLDSM_ERR_INVALID_SIQUERY_STATUS
 *          Supplied status value is invalid here.
 *          CLDSM_ERR_SYSTEM_ADD_SECTION_FILTER,
 *          Problem with add section filter callback.
 */
E_DscError siQueryProcessResult( P_DsmCoreInst idp,
   P_SiQuery pSiQueryRef, P_SIQueryResult pResult )
{
   E_DscError err = CLDSM_OK;
   P_RootCarousel pRC;
   P_SecFilterInfo pSectionFilter;

   dsmAssert((idp != NULL));
   dsmAssert((pSiQueryRef != NULL));

   DBGLOG(DD_QUERY, "(idp,pSiQueryRef=%p,kind=%d) target=%p\n", pSiQueryRef, pResult->kind, pSiQueryRef->target)

   switch (pSiQueryRef->state)
   {
      case SIQS_PENDING:
         dsmAssert((pResult != NULL));
         if (pSiQueryRef->query.kind == pResult->kind)
         {
            /* -- Query is still active (ie. not previously aborted before
                  completion and result is still required) */
            dsmDP2(("lastPID=%d lpStored=%d, lastAssocTag=%d\n",
                    idp->lastPID, idp->lastPIDResultStored, idp->lastAssociationTag));

            switch (pSiQueryRef->query.kind)
            {
               case SIQUERY_PID:

                  /* -- Store result */
                  idp->lastServiceId = pSiQueryRef->query.serviceId;
                  idp->lastAssociationTag = pSiQueryRef->query.associationTag;
                  idp->lastPID = pResult->data.pid;
                  idp->lastPIDResultStored = TRUE;

                  /* -- Get section filter and store PID in it */
                  pSectionFilter = (P_SecFilterInfo) pSiQueryRef->target;
                  dsmAssert((pSectionFilter != NULL));
                  dsmAssert((pSectionFilter->status == SFA_PENDING));
                  pSectionFilter->filter.pid = pResult->data.pid;

                  err = DSC_SectionFilterCommit( idp, pSectionFilter );

                  if (err)
                  {
                     /* -- Abort (and notify) anything depending on filter
                        -- NB. This will also stop the section filter */
                     DSC_SectionFilterAbortDependants( idp, pSectionFilter );
                  }
                  break;

               case SIQUERY_CAROUSEL:
               case SIQUERY_CAROUSEL_INFO:
               case SIQUERY_FIND_CAROUSEL:
               case SIQUERY_BOOT_CAROUSEL:
               case SIQUERY_SSU_CAROUSEL:
               case SIQUERY_SSU_PID:
                  /* -- Only duplicate PID queries are handled */
                  dsmAssert((pSiQueryRef->llcDuplSiQuerys == NULL));

                  /*
                  -- Store component_tag/association_tag to PID mapping.
                  -- This can prevent a subsequent unnecessary SI query
                  -- when determining the PID for the DSI of this carousel
                  */
                  idp->lastServiceId = pSiQueryRef->query.serviceId;
                  idp->lastAssociationTag =
                     (U16BIT)pResult->data.carouselInfo.associationTag;
                  idp->lastPID = pResult->data.carouselInfo.pid;
                  idp->lastPIDResultStored = TRUE;

                  /* -- Get carousel */
                  pRC = (P_RootCarousel) pSiQueryRef->target;
                  if (pRC != NULL)
                  {
               #ifdef GET_PID_LIST
                     if (status == SIQUERY_BOOT_CAROUSEL)
                        siQueryGetPIDList( idp );
               #endif /*GET_PID_LIST*/

                     /* -- Clear Query ref in carousel (finished with) */
                     pRC->pPendingSiQueryRef = NULL;
                     err = DSC_RootCrslBootCarousel( idp, pRC, &(pResult->data.carouselInfo));
                     if (err)
                     {
                        DSC_RootCrslAbortLoadRequest( idp, pRC );
                     }
                  }
                  break;

               case SIQUERY_PIDLIST:
                  /* do nothing */
                  break;

               case SIQUERY_DEFERRED_SERVICE:

                  DSC_StrmObjectStoreDeferred( idp, pSiQueryRef->target, pResult->data.deferredService );

                  break;

               default:
                  dsmDP1(("ERROR: Illegal SI query kind = %u\n",
                          pSiQueryRef->query.kind));
                  dsmAssert((0));
                  err = CLDSM_ERR_INTERNAL;
                  break;
            }
         }
         else if (pResult->kind == SIRESULT_FAILURE)
         {
            switch (pSiQueryRef->query.kind)
            {
               case SIQUERY_PID:
                  /* -- Get section filter */
                  pSectionFilter = (P_SecFilterInfo) pSiQueryRef->target;
                  dsmAssert((pSectionFilter != NULL));

                  /* -- Clear Query ref in filter (finished with) */
                  pSectionFilter->u.pPendingSiQueryRef = NULL;

                  /* -- Abort (and notify) anything depending on filter */
                  DSC_SectionFilterAbortDependants( idp, pSectionFilter );
                  break;

               case SIQUERY_CAROUSEL_INFO:
               case SIQUERY_BOOT_CAROUSEL:
               case SIQUERY_CAROUSEL:
               case SIQUERY_SSU_CAROUSEL:
               case SIQUERY_SSU_PID:
                  /* -- Only duplicate PID queries are handled */
                  dsmAssert((pSiQueryRef->llcDuplSiQuerys == NULL));

                  /* -- Get carousel */
                  pRC = (P_RootCarousel) pSiQueryRef->target;
                  if (pRC != NULL)
                  {
                     /* -- Clear Query ref in carousel (finished with) */
                     pRC->pPendingSiQueryRef = NULL;

                     DSC_RootCrslAbortLoadRequest( idp, pRC );
                  }
                  break;

               /*
               case SIQUERY_PIDLIST:
                   TODO: Implement for turbo-caching?
               break;
               */

               case SIQUERY_DEFERRED_SERVICE:
                  DSC_StrmObjectStoreDeferred( idp, pSiQueryRef->target, idp->dvbLocator );
                  break;

               default:
                  dsmDP1(("ERROR: Illegal SI query kind = %u\n",
                          pSiQueryRef->query.kind));
                  dsmAssert((0));
                  err = CLDSM_ERR_INTERNAL;
                  break;
            }
         }
         else
         {
            dsmDP1(("ERROR: Illegal SI query status %u\n", pResult->kind));
            /*dsmAssert((0));*/
            err = CLDSM_ERR_INVALID_SIQUERY_STATUS;
         }
         dsmDP2(("NEW lastPID=%d lpStored=%d, lastAssocTag=%d\n",
                 idp->lastPID, idp->lastPIDResultStored, idp->lastAssociationTag));

         /* -- Query is now completed so change state */
         pSiQueryRef->state = SIQS_COMPLETED;
         break;

      case SIQS_ABORTED:
         /* -- Regardless of supplied status, treat this as acknowledge of
            -- previous SI Query stop request (ie. abort before completion).
            -- SI query results no longer required so nothing specific
            -- to do here. */
         /* -- Only valid for original SI Queries */
         dsmAssert((pSiQueryRef->original == TRUE));
         break;

      case SIQS_EXPIRED:
         /* -- This query is no longer required but was kept active because
             -- it had duplicates. Nothing specific to do here. */
         /* -- Only valid for original SI Queries */
         dsmAssert((pSiQueryRef->original == TRUE));
         break;

      default:
         dsmDP1(("ERROR: Illegal SI query ref state = %u\n", pSiQueryRef->state));
         dsmAssert((0));
         err = CLDSM_ERR_INTERNAL;
   }

   siQueryStop( idp, pSiQueryRef );

   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: siQueryProcessResult: %u\n", err)));
   dsmDP3(("exit siQueryProcessResult -> rtn: %u \n", err));
   return err;
}

/*---Local function definitions---------------------------------------------*/

/* /////////////////////////////////////////////////////////////////////////////
// siQueryCreate
// Creates an instance of the SI Query struct, and initialises it.
///////////////////////////////////////////////////////////////////////////// */
static E_DscError siQueryCreate( P_DsmCoreInst idp, P_SiQuery *ppSiQuery )
{
   P_SiQuery pSiQuery = NULL;
   E_DscError err = CLDSM_OK;

   dsmDP3(("siQueryCreate()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((ppSiQuery != NULL));

   pSiQuery = (P_SiQuery)DSC_CmMemGet( idp, sizeof(S_SiQuery) );
   if (!pSiQuery)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      err = CLDSM_OK;

      llLinkInit( pSiQuery->llData, NUM_LISTS_SI_QUERY );

      pSiQuery->magic = SI_QUERY_MAGIC;
      pSiQuery->state = SIQS_INITIAL;
      pSiQuery->queryHandle = NULL;
      pSiQuery->target = NULL;

      pSiQuery->query.serviceId = 0;
      pSiQuery->query.kind = SIQUERY_PID;
      pSiQuery->query.associationTag = 0;

      pSiQuery->original = FALSE;
      pSiQuery->llcDuplSiQuerys = NULL;
   }
   *ppSiQuery = pSiQuery;

   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: siQueryCreate: %u\n", err)));
   dsmDP3(("exit siQueryCreate -> rtn: %u\n", err));
   return err;
}

/* /////////////////////////////////////////////////////////////////////////////
// siQueryDestroy
// Destroys an instance of the siQuery struct, and all associated data.
///////////////////////////////////////////////////////////////////////////// */
static void siQueryDestroy( P_DsmCoreInst idp, P_SiQuery pSiQuery )
{
   dsmAssert((idp != NULL));
   dsmAssert((pSiQuery != NULL));
   dsmAssert((pSiQuery->magic == SI_QUERY_MAGIC));

   /* -- Should not be in any lists */
   dsmAssert((pSiQuery->llData[CURR_SI_QUERY_LIST].pLLCtrl == NULL));
   dsmAssert((pSiQuery->llData[DUPLICATE_SI_QUERY_LIST].pLLCtrl == NULL));

   /* -- NULL pSiQueryRef contents.
      -- NB. Should allow error to be detected if destroyed SI Query is
      -- erroneously re-used (not guaranteed since may depend on MemMgr
      -- implementation) */
   pSiQuery->magic = 0;
   pSiQuery->state = SIQS_INITIAL;
   pSiQuery->queryHandle = NULL;
   pSiQuery->target = NULL;

   pSiQuery->query.serviceId = 0;
   pSiQuery->query.kind = SIQUERY_PID;
   pSiQuery->query.associationTag = 0;

   pSiQuery->original = FALSE;
   dsmAssert((pSiQuery->llcDuplSiQuerys == NULL));


   /* -- Use original handle so it gets Nulled */
   DSC_CmMemRelease( idp, pSiQuery );

   dsmDP3(("exit siQueryDestroy\n"));
}

/* /////////////////////////////////////////////////////////////////////////////
// FindPendingPIDQuery
// Searches for a matching pending PID query.
///////////////////////////////////////////////////////////////////////////// */
static P_SiQuery FindPendingPIDQuery( P_DsmCoreInst idp, U16BIT service_id, U16BIT assoc_tag )
{
   P_SiQuery pQueryFromList;
   ListId_t listId;

   /* Get listId and first SI Query in list from Control block */
   listId = LListId( idp->llcCurrSiQueries );
   pQueryFromList = LLHead( idp->llcCurrSiQueries );

   /* Go through each SI Query in list until a pending PID query with a
      matching service_id and assoc_tag is found */
   while (pQueryFromList)
   {
      if ((pQueryFromList->query.kind == SIQUERY_PID) &&
          (pQueryFromList->query.serviceId == service_id) &&
          (pQueryFromList->query.associationTag == assoc_tag) &&
          (pQueryFromList->state == SIQS_PENDING))
      {
         /* Found a matching SI Query */
         /* -- Must always be an original query */
         dsmAssert((pQueryFromList->original == TRUE));
         break;
      }
      pQueryFromList = LLNext( pQueryFromList, listId );
   }
   return pQueryFromList;
}

/*----------------------------------------------------------------------------*/
