/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2006 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Stream event related functions
 *             References:       [1] ISO_IEC_13818-6;1998(E)
 * @file    streamEvent.c
 * @date    05/17/2006

 * @author  Ocean Blue
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>
#include <stdio.h>
#include "clDsmSystem.h"
#include "moduleData.h"
#include "sectionFilter.h"
#include "dsmObject.h"
#include "module.h"
#include "objectCarousel.h"
#include "cacheMgr.h"

/*------------------------------- Local Macros -------------------------------*/

#define MAX_EVT_LISTENERS        8
#define LIFECYCLE_STR            "Life cycle"

/* Stream descriptors. Ref[1] - table 8-1 p.279 */
#define STREAM_NPT_REF_DESC      23
#define STREAM_NPT_END_DESC      24
#define STREAM_MODE_DESC         25
#define STREAM_EVENT_DESC        26

#define GEN_E_HDL(sid, eid, lid)   (((U32BIT)sid << 24) | ((U32BIT)eid << 8) | (lid))
#define EH_TO_LID(eh)            (U8BIT)(((U32BIT)eh) & 0xFF)
#define EH_TO_SID(eh)            (U8BIT)(((U32BIT)eh) >> 24)
#define EH_TO_EID(eh)            (U16BIT)((((U32BIT)eh) >> 8) & 0xFFFF)

#ifdef DISABLE_SUBSCRIBE_REFRESH
   #define NewEventListen(i, e, ps, pe, h, u1, u2) NewEventListen(i, e, h, u1, u2)
#endif

#define XML_DATA1a "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<dsmcc:dsmcc xmlns=\"urn:dvb:mis:dsmcc:2009\" xmlns:dsmcc=\"urn:dvb:mis:dsmcc:2009\">\n" \
                   "<dsmcc:dsmcc_object dsmcc:component_tag=\""
#define XML_DATA1b "\">\n"
#define XML_DATA2a "<dsmcc:stream_event dsmcc:stream_event_id=\""
#define XML_DATA2b "\" dsmcc:stream_event_name=\""
#define XML_DATA2c "\"/>\n"
#define XML_DATA3  "</dsmcc:dsmcc_object>\n" \
                   "</dsmcc:dsmcc>\n"

/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/

typedef struct s_se_info
{
   struct s_se_info *next;
   S_ObjectKey objectKey;
   P_Module pModule;
   U16BIT eventAssocTag;
   U8BIT uniqueId;
   U8BIT lstnrCount;
   U16BIT eventCount;
   U16BIT namesLen;
   U16BIT serviceId;
   U16BIT xmlLen;
   U8BIT *namesPtr;
   U8BIT *xmlPtr;
} S_SE_INFO;

typedef struct s_listener
{
   struct s_listener *next;
   H_DsmEvent eventHandle;
   void *userData1;
   void *userData2;
   U8BIT uniqueId;
} S_LISTENER;

typedef struct s_dsmevent
{
   P_SecFilterInfo pFilter;
   U16BIT id;
   U8BIT version;
   U8BIT nlen;
   U8BIT *name;
   S_LISTENER *pListener;
#ifdef SUPPORT_SCHEDULED_EVENTS
   U64BIT eventNpt;
#endif
} S_Event;


/*------------------------------- Local Statics ------------------------------*/

static U8BIT unique_stream_ids = 0;
static U8BIT unique_listen_ids = 0;

/*------------------- local prototypes/forward declarations ------------------*/

static void ReleaseStreamInfo( P_DsmCoreInst idp, S_SE_INFO *dsi );
static E_DscError RetrieveStreamInfo( P_DsmCoreInst idp,
   /*I*/ H_DsmObject streamObject,
   /*O*/ S_SE_INFO **ppStreamInfo );

/*-------------------------- Functions ---------------------------------------*/


S_SE_INFO* FindEventInfo(P_DsmCoreInst idp, H_DsmEvent eventHandle)
{
   S_SE_INFO *pStreamInfo;
   U8BIT strmId;
   if (!eventHandle)
   {
      pStreamInfo = NULL;
   }
   else
   {
      strmId = EH_TO_SID(eventHandle);
      pStreamInfo = idp->hSubscribedEventList;
      while (pStreamInfo != NULL)
      {
         if (pStreamInfo->uniqueId == strmId)
         {
            break;
         }
         pStreamInfo = pStreamInfo->next;
      }
   }
   return pStreamInfo;
}

S_SE_INFO* FindStreamInfo(P_DsmCoreInst idp, H_DsmObject streamObject)
{
   S_SE_INFO *pStreamInfo;
   pStreamInfo = idp->hSubscribedEventList;
   while (pStreamInfo != NULL)
   {
      if (pStreamInfo->pModule == streamObject->pModule &&
          !memcmp(&pStreamInfo->objectKey, &streamObject->objectInfo.objectKey, sizeof(S_ObjectKey)))
      {
         break;
      }
      pStreamInfo = pStreamInfo->next;
   }
   return pStreamInfo;
}

#ifndef DISABLE_SUBSCRIBE_REFRESH
static
S_Event* FindMatchingEventId( S_SE_INFO *pStreamInfo, U16BIT id )
{
   S_Event *pEvent;
   U16BIT evntNdx, evntMax;
   if (pStreamInfo != NULL)
   {
      pEvent = (S_Event *)(pStreamInfo + 1);
      evntMax = pStreamInfo->eventCount;
      for (evntNdx = 0; evntNdx != evntMax; evntNdx++)
      {
         if (pEvent->id == id)
         {
            /*match*/
            return pEvent;
         }
         pEvent++;
      }
   }
   return NULL;
}
#endif

static E_DscError UnsubscribeEvent( P_DsmCoreInst idp, S_Event *pEvent, H_DsmEvent eventHandle )
{
   E_DscError err;
   S_LISTENER **ppL, *pL;
   U8BIT lstnId = EH_TO_LID(eventHandle);
   ppL = &(pEvent->pListener);
   err = CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE;
   while (*ppL != NULL)
   {
      pL = *ppL;

      if (pL->uniqueId == lstnId)
      {
         *ppL = pL->next;
         DSC_CmMemRelease( idp, pL );
         err = CLDSM_OK;
         break;
      }
      ppL = &((*ppL)->next);
   }
   if (pEvent->pListener == NULL && pEvent->pFilter != NULL)
   {
      DBGLOG((DD_SE | DD_SF), "pFilter=%p", pEvent->pFilter)
      DSC_SectionFilterStop( idp, (void *)&pEvent->pFilter );
      pEvent->version = 0xff;
   }
   return err;
}

static E_DscError NewEventListen( P_DsmCoreInst idp, S_Event *pEvent,
   S_SE_INFO *prevSEInfo, H_DsmEvent prevEventHandle, U32BIT hdl_id, void *userData1, void *userData2 )
{
   E_DscError err;
   S_LISTENER *pL;
#ifndef DISABLE_SUBSCRIBE_REFRESH
   S_Event *pPrevEvt;
   pPrevEvt = FindMatchingEventId( prevSEInfo, pEvent->id );
   if (pPrevEvt != NULL)
   {
      dsmDP2(("Previously subsribed event, id %d vers %d <- %d", pEvent->id, pEvent->version, pPrevEvt->version));
      pEvent->version = pPrevEvt->version;
   }
#endif
   pL = pEvent->pListener;
   while (pL != NULL)
   {
      DBGLOG(DD_SE, "e Lsr=%p", pL)
      if (pL->userData1 == userData1 && pL->userData2 == userData2)
      {
         err = CLDSM_ERR_STREAM_EVENTS_STILL_SUBSCRIBED;
         break;
      }
      pL = pL->next;
   }
   if (pL == NULL)
   {
      pL = (S_LISTENER *)DSC_CmMemGet( idp, sizeof(S_LISTENER) );
      if (!pL)
      {
         err = CLDSM_ERR_MEM_HEAP_FULL;
      }
      else
      {
         err = CLDSM_OK;
         DBGLOG(DD_SE, "e Lsr=%p", pL)
         pL->userData1 = userData1;
         pL->userData2 = userData2;
         pL->uniqueId = unique_listen_ids;
         pL->eventHandle = (H_DsmEvent)hdl_id; /* save handle for callback */
         pL->next = pEvent->pListener;
         pEvent->pListener = pL;
      }
   }
   return err;
}

static E_DscError StartSectionFilter( P_DsmCoreInst idp, S_SE_INFO *pStreamInfo, P_Event pEvent )
{
   S_SfTarget target;
   if (pEvent->pFilter == NULL)
   {
      DBGLOG((DD_SE | DD_SF), "serv_id=0x%x eid=%x", pStreamInfo->serviceId,pEvent->id)
      /* Request section filter */
      target.kind = SFK_STREAM_DESCR;
      target.id = pEvent->id;
      target.serviceId = pStreamInfo->serviceId;
      target.associationTag = pStreamInfo->eventAssocTag;
      target.u.pEvent = pEvent;
      return DSC_SectionFilterStart( idp, &target, SF_PRIORITY_HIGH, &pEvent->pFilter );
   }
   return CLDSM_OK;
}

/**
 * @brief   The Client uses this function to request that the DSMCC notifies it when a
 *          named stream event occurs. The notification is implemented by means of a
 *          callback function defined by the Client.
 *          Refer to the description of F_NotifyStreamEvent for information about
 *          the callback function.
 *          Before calling this function the Client must request that the stream object of
 *          interest be loaded using the CDSM_ObjectLoad() function. This action will
 *          deliver a handle to the object stream - this is passed as the streamObject
 *          argument to this function.
 *          Do not have to wait until load is completed before subscribing.
 *          Do not have to open an object before subscribing (since synchronous function).
 * @param   dsmccInstance Allows support >= 1 instance of a DSMCC.
 * @param   streamObject This references a BIOP::StreamEventMessage as well
 *          as indicating the carousel, module and the kind of
 *          the object.
 * @param   eventName Name of stream event required by the Client.
 * @param   userData1 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   userData2 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   pStatus Status of the subscription.
 * @param   pEventHandle The is the handle to the stream event returned by
 *          subscribe. The caller will use this handle when the
 *          stream event being subscribed to is referred
 *          to in the future.
 * @return
 *          CLDSM_ERR_INVALID_INSTANCE
 *          The Instance value is invalid
 *          CLDSM_ERR_INVALID_STREAM_EVENT_NAME
 *          The supplied event name is invalid
 *          CLDSM_ERR_SUBSCRIBE_FAILURE
 *          The event could not be subscribed to
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The supplied object handle is invalid
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The message referred to by streamObject is not a BIOP::StreamEventMessage
 *          CLDSM_ERR_NO_STREAM_EVENT_NOTIFY_CALLBACK
 *          If Stream Events are to be used then a callback must be registered.
 *          CLDSM_ERR_MEM_HEAP_FULL
 *          Out of memory.
 *          CLDSM_OK
 *          The event has been successfully subscribed to.
 */
E_DscError CDSM_StreamEventSubscribe(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmObject streamObject,
   /*I*/ U8BIT *eventName,
   /*I*/ void *userData1,
   /*I*/ void *userData2,
   /*O*/ H_DsmEvent *pEventHandle )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) dsmccInstance;
   E_DscError err;
   S_SE_INFO *pStreamInfo;
#ifndef DISABLE_SUBSCRIBE_REFRESH
   S_SE_INFO *prevStrmInfo;
#endif
   S_Event *pEvent;
   U32BIT ui32, usid;
   U16BIT evntNdx, evntMax;

   DBGLOG(DD_SE, "( 0x%x, %s, 0x%x, 0x%x, 0x%x )",
      streamObject, eventName, userData1, userData2, pEventHandle)

   if (idp->setup.notifyStreamEventFunc == NULL)
   {
      return CLDSM_ERR_NO_STREAM_EVENT_NOTIFY_CALLBACK;
   }

   err = RetrieveStreamInfo( idp, streamObject, &pStreamInfo );
   if (CLDSM_OK == err)
   {
      pEvent = (S_Event *)(pStreamInfo + 1);
      evntMax = pStreamInfo->eventCount;
      pStreamInfo->lstnrCount++;
      usid = pStreamInfo->uniqueId;

   #ifndef DISABLE_SUBSCRIBE_REFRESH
      /* See if pEventHandle is already referencing a valid StreamInfo */
      prevStrmInfo = FindEventInfo(idp, *pEventHandle);
   #endif
      unique_listen_ids++;

      if (eventName == NULL)
      {
         ui32 = 0;
      }
      else
      {
         ui32 = strlen((char *)eventName);
      }
      /* initialise to invalid se name, so detect if don't find any matching */
      err = CLDSM_ERR_INVALID_STREAM_EVENT_NAME;
      if (ui32 == 0)
      {
         /* generated this event handle */
         ui32 = GEN_E_HDL(usid, 0xFFFF, unique_listen_ids);

         /* all events for this stream */
         for (evntNdx = 0; evntNdx != evntMax; evntNdx++)
         {
            if (pEvent->name != NULL)
            {
               err = NewEventListen( idp, pEvent, prevStrmInfo, *pEventHandle, ui32, userData1, userData2 );
               if (err != CLDSM_OK)
               {
                  break;
               }
               else
               {
                  err = StartSectionFilter(idp, pStreamInfo, pEvent);
                  if (CLDSM_OK != err)
                  {
                     break;
                  }
               }
            }
            pEvent++;
         }
      }
      else
      {
         for (evntNdx = 0; evntNdx != evntMax; evntNdx++)
         {
            if (pEvent->name != NULL && pEvent->nlen == ui32 &&
                memcmp(pEvent->name, eventName, ui32) == 0)
            {
               /* generated this event handle */
               ui32 = GEN_E_HDL(usid, evntNdx, unique_listen_ids);

               /*match*/
               err = NewEventListen( idp, pEvent, prevStrmInfo, *pEventHandle, ui32, userData1, userData2 );
               if (err == CLDSM_OK)
               {
                  err = StartSectionFilter(idp, pStreamInfo, pEvent);
               }
               break;
            }
            pEvent++;
         }
      }
   #ifndef DISABLE_SUBSCRIBE_REFRESH
      if (prevStrmInfo)
      {
         /* unsubscribe previous */
         CDSM_StreamEventUnsubscribe( idp, *pEventHandle );
      }
   #endif
      switch (err)
      {
         case CLDSM_OK:
         case CLDSM_ERR_STREAM_EVENTS_STILL_SUBSCRIBED:
            /* return new event handle to client */
            *pEventHandle = (H_DsmEvent)ui32;
            break;

         default:
            if (EH_TO_EID(ui32) == 0xFFFF)
            {
               do
               {
                  UnsubscribeEvent( idp, pEvent, (H_DsmEvent)ui32 );
                  pEvent--;
               }
               while (evntNdx--);
            }
            else
            {
               UnsubscribeEvent( idp, pEvent, (H_DsmEvent)ui32 );
            }
         /*fall tho*/
         case CLDSM_ERR_INVALID_STREAM_EVENT_NAME: /*nothing to unsubscribe */
            ReleaseStreamInfo( idp, pStreamInfo );
            *pEventHandle = 0;
            break;
      }
   }
   return err;
}

/**
 * @brief   Like clDsmStreamEventSubscribe, except that no stream object is required.
 *          This function subscribes to events specified by event ID, and association
 *          tag for the current service.
 * @param   dsmccInstance Allows support >= 1 instance of a DSMCC.
 * @param   streamObject This references a BIOP::StreamEventMessage as well
 *          as indicating the carousel, module and the kind of
 *          the object.
 * @param   eventName Name of stream event required by the Client.
 * @param   userData1 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   userData2 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   pStatus Status of the subscription.
 * @param   pEventHandle The is the handle to the stream event returned by
 *          subscribe. The caller will use this handle when the
 *          stream event being subscribed to is referred
 *          to in the future.
 * @return
 *          CLDSM_ERR_INVALID_INSTANCE
 *          The Instance value is invalid
 *          CLDSM_ERR_INVALID_STREAM_EVENT_NAME
 *          The supplied event name is invalid
 *          CLDSM_ERR_SUBSCRIBE_FAILURE
 *          The event could not be subscribed to
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The supplied object handle is invalid
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The message referred to by streamObject is not a BIOP::StreamEventMessage
 *          CLDSM_ERR_NO_STREAM_EVENT_NOTIFY_CALLBACK
 *          If Stream Events are to be used then a callback must be registered.
 *          CLDSM_ERR_MEM_HEAP_FULL
 *          Out of memory.
 *          CLDSM_OK
 *          The event has been successfully subscribed to.
 */
E_DscError CDSM_SpecialEventSubscribe(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ U16BIT associationTag,
   /*I*/ U16BIT eventId,
   /*I*/ void *userData1,
   /*I*/ void *userData2,
   /*O*/ H_DsmEvent *pEventHandle )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) dsmccInstance;
   E_DscError err;
   S_SE_INFO *pStreamInfo;
   S_Event *pEvent;
   U32BIT hdl_id, size;

   DBGLOG(DD_SE, "( %u, %u, 0x%x, 0x%x, 0x%x )",
      associationTag, eventId, userData1, userData2, pEventHandle)

   if (idp->setup.notifyStreamEventFunc == NULL)
   {
      return CLDSM_ERR_NO_STREAM_EVENT_NOTIFY_CALLBACK;
   }
   if (!idp->currentServiceSet || idp->dvbLocator.service_id == 0)
   {
      return CLDSM_ERR_NO_CURRENT_SERVICE_SET;
   }
   pStreamInfo = idp->hSubscribedEventList;
   while (pStreamInfo != NULL)
   {
      DBGLOG(DD_SE, "pStreamInfo=%p", pStreamInfo)
      if (pStreamInfo->eventAssocTag == associationTag)
      {
         pEvent = (S_Event *)(pStreamInfo + 1);
         if (pEvent->id == eventId)
         {
            break;
         }
      }
      pStreamInfo = pStreamInfo->next;
   }

   if (pStreamInfo != NULL)
   {
      err = CLDSM_OK;
   }
   else
   {
      size = sizeof(S_SE_INFO) + sizeof(S_Event);
      pStreamInfo = (S_SE_INFO *)DSC_CmMemGet( idp, size );
      if (!pStreamInfo)
      {
         ERRPRINT("mem fail")
         err = CLDSM_ERR_MEM_HEAP_FULL;
         *pEventHandle = 0;
      }
      else
      {
         err = CLDSM_OK;
         memset( pStreamInfo, 0, size );

         pEvent = (S_Event *)(pStreamInfo + 1);
         pEvent->name = NULL;

         pEvent->id = eventId;
         pEvent->version = 0xFF;

         DBGLOG(DD_SE, "pStreamInfo=%p", pStreamInfo)
         pStreamInfo->eventAssocTag = associationTag;
         pStreamInfo->uniqueId = ++unique_stream_ids;
         pStreamInfo->eventCount = 1;
         pStreamInfo->serviceId = idp->dvbLocator.service_id;

         pStreamInfo->next = idp->hSubscribedEventList;
         idp->hSubscribedEventList = pStreamInfo;
      }
   }
   if (!err)
   {
   #ifndef DISABLE_SUBSCRIBE_REFRESH
      /* See if pEventHandle is already referencing a valid StreamInfo */
      S_SE_INFO *prevStrmInfo = FindEventInfo(idp, *pEventHandle);
   #endif
      pStreamInfo->lstnrCount++;
      hdl_id = pStreamInfo->uniqueId;
      unique_listen_ids++;
      /* generated this event handle */
      hdl_id = GEN_E_HDL(hdl_id, 0, unique_listen_ids);
      err = NewEventListen( idp, pEvent, prevStrmInfo, *pEventHandle, hdl_id, userData1, userData2 );
      if (err == CLDSM_OK)
      {
         DBGLOG(DD_SE, "e ftr=%p, hdl_id=%lx", pEvent->pFilter, hdl_id)
         err = StartSectionFilter(idp, pStreamInfo, pEvent);
         DBGLOG(DD_SF, "err=%d pEvent->pFilter=%x", err, pEvent->pFilter)
      }
   #ifndef DISABLE_SUBSCRIBE_REFRESH
      if (prevStrmInfo)
      {
         /* unsubscribe previous */
         DBGLOG(DD_SE, "unsubscr %p", *pEventHandle)
         CDSM_StreamEventUnsubscribe( idp, *pEventHandle );
      }
   #endif
      switch (err)
      {
         case CLDSM_OK:
         case CLDSM_ERR_STREAM_EVENTS_STILL_SUBSCRIBED:
            /* return new event handle to client */
            *pEventHandle = (H_DsmEvent)hdl_id;
            break;

         default:
            ERRPRINT("err=%d", err)
            UnsubscribeEvent( idp, pEvent, (H_DsmEvent)hdl_id );
            ReleaseStreamInfo( idp, pStreamInfo );
            *pEventHandle = 0;
      }
   }
   return err;
}

/**
 * @brief   This function is called by the Client to let DSMCC know when a previously
 *          subscribed stream event is no longer needed.
 *          Callback made to acknowledge unsubscribe.
 *          All data relating to this stream event will no longer be valid, including
 *          the streamEventHandle.
 * @param   dsmccInstance Allows support of >= 1 instance of a DSMCC.
 * @param   eventHandle This is a handle to an entry which is to be
 *          deleted from a list of subscribed to stream events
 * @return
 *          CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE
 *          The eventHandle is not valid.
 *          CLDSM_OK
 *          The stream event was successfully unsubscribed
 */
E_DscError CDSM_StreamEventUnsubscribe(
   /*I*/ H_DsmCoreInst idp,
   /*I*/ H_DsmEvent eventHandle)
{
   E_DscError err;
   S_SE_INFO *pStreamInfo;
   S_Event *pEvent;
   U16BIT evtId;

   DBGLOG(DD_SE, " evHl=%p", eventHandle)

   evtId = EH_TO_EID(eventHandle);

   pStreamInfo = FindEventInfo(idp, eventHandle);
   if (pStreamInfo == NULL)
   {
      err = CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE;
   }
   else
   {
      pEvent = (S_Event *)(pStreamInfo + 1);
      if (evtId == 0xFFFF)
      {
         /* listener interested in all events */
         err = CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE;
         for (evtId = 0; evtId != pStreamInfo->eventCount; evtId++, pEvent++)
         {
            if (UnsubscribeEvent( idp, pEvent, eventHandle ) == CLDSM_OK)
            {
               err = CLDSM_OK;
            }
         }
      }
      else if (evtId <= pStreamInfo->eventCount)
      {
         pEvent += evtId;
         err = UnsubscribeEvent( idp, pEvent, eventHandle );
      }
      else
      {
         err = CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE;
      }
      if (err == CLDSM_OK)
      {
         ReleaseStreamInfo( idp, pStreamInfo );
      }
   }
   return err;
}

/**
 * @brief   Same as clDsmStreamEventUnsubscribe - used in conjuction with
 *          clDsmSpecialEventSubscribe.
 * @param   dsmccInstance Allows support of >= 1 instance of a DSMCC.
 * @param   eventHandle This is a handle to an entry which is to be
 *          deleted from a list of subscribed to stream events
 * @return
 *          CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE
 *          The streamEventHandle is not valid.
 *          CLDSM_OK
 *          The stream event was successfully unsubscribed
 */
E_DscError CDSM_SpecialEventUnsubscribe(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmEvent eventHandle )
{
   return CDSM_StreamEventUnsubscribe(dsmccInstance, eventHandle);
}

#ifdef INC_STREAM_RESET
/**
 * @brief   Reset stream event versions for a stream object, when stream is stopped
 * @param   dsmccInstance The DSMCC instance
 * @param   streamRef The stream object.
 * @return  void
 */
E_DscError dsmStreamEventReset(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ void *streamObject )
{
   E_DscError err;
   S_SE_INFO *pStreamInfo;
   S_Event *pEvent;
   U16BIT evntNdx, evntMax;

   dsmDP2(("( 0x%x )", streamObject));

   pStreamInfo = FindStreamInfo(dsmccInstance, streamObject);
   if (pStreamInfo != NULL)
   {
      dsmDP1((" no stream object "));
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else
   {
      pEvent = (S_Event *)(pStreamInfo + 1);
      evntMax = pStreamInfo->eventCount;
      for (evntNdx = 0; evntNdx != evntMax; evntNdx++)
      {
         pEvent->version = 0xff;
         pEvent++;
      }
      err = CLDSM_OK;
   }
}

#endif /*INC_STREAM_RESET*/

E_DscError DSC_StrmEventUpdate( P_DsmCoreInst idp, U8BIT *pStreamDescrSection,
   P_Event pEvent, U32BIT targetId )
{
   E_DscError err = CLDSM_OK;
   S_LISTENER *pL;
#if DSM_DATA_CHECK_LEVEL >= 2
   U16BIT ui16;
#endif
   U16BIT eventID, msg_len;
   U8BIT ui8, version, dtag, dlen, edata_len, *edata_ptr;
#ifdef SUPPORT_SCHEDULED_EVENTS
   U32BIT ui32;
#endif

   dsmDP3(("DSC_StrmEventUpdate()\n"));
   dsmAssert((idp != NULL));
   dsmAssert((pStreamDescrSection != NULL));

   /* -- table_id = 0x3D */
   READ_UINT8( pStreamDescrSection, ui8);
   if (ui8 != 0x3D)
   {
      dsmDP1(("DATA ERROR: streamEvent table_id = %x", ui8));
      return CLDSM_ERR_INVALID_INSTANCE;
   }
   /* Minimum section length is 9 bytes (with message length of zero) - ie 12 byte header
    * less three bytes for table id and length fields.
    * Note: maximum DSMCC message length is 4084 bytes, following the last-section-number
       field and up to the CRC_32/checksum - Ref[1] - section 9.2.2.1, pg 287
    */
   READ_UINT16( pStreamDescrSection, msg_len );
   msg_len &= 0x0fff; /* got section length */
   if (msg_len < 9)
   {
      dsmDP1(("DATA ERROR: section length=0x%x", msg_len));
      return CLDSM_ERR_INVALID_INSTANCE;
   }
   msg_len -= 9; /* remove remaining header bytes */

   if (pEvent == NULL || pEvent->id != targetId)
   {
      dsmDP1(("no target for id=%x", targetId));
      return CLDSM_ERR_INVALID_STREAM_EVENT_HANDLE;
   }

   /*---------------------------------------------------------------------------*/
   /* -- table_id_extension = eventID !!! */

   /* description in DTG 4.0 17.2.4.4 p. 17-15 */
   /* [15] [14]     [13...8]       [7...0]   */
   /*  0     0  eventId[13..8]  eventId[7..0]  ===> section carries a SINGLE "do-it-now" EVENT */

   /* *** "do-it-now" EVENTs matches with eventID in the range 0x0001 .. 0x3FFF */
   /*---------------------------------------------------------------------------*/

   /* read eventID */
   READ_UINT16( pStreamDescrSection, eventID );

   /* EVENT IDs 0x0001-0x3FFF carries "do-it-now" event
    * EVENT IDs 0x4000-0x7FFF carries NPT ref descriptor
    * EVENT IDs 0x8000-0xBFFF carries other stream descriptors
    */
   if ((eventID != 0xFFFF) &&
       ((eventID < 0x0001) || (eventID > 0xBFFF)))
   {
      dsmDP3(("DSC_StrmEventUpdate tableId ext doesnt carry a do-it-now event = %xh\n", eventID));
      goto _return;
   }

   if (pEvent->id != eventID)
   {
      dsmDP1(("pEvent->id != eventID: %x, %x", pEvent->id, eventID));
      err = CLDSM_ERR_INVALID_STREAM_EVENT_NAME;
   }
   else
   {
      /* -- get version */
      READ_UINT8( pStreamDescrSection, ui8);

      version = (ui8 >> 1) & 0x1F;
      if (version != pEvent->version)
      {
         edata_len = 0;
         edata_ptr = NULL;
         dsmDP2(("newVer=%u oldVer=%u", version, pEvent->version));
         pEvent->version = version;

         /* get sectionNumber*/
         /* only consider section number == 0 : DTG: 4.0 paragr:17.2.4.4 p 17-15 */
         READ_UINT8_L2CHK( pStreamDescrSection, ui8, ui8 == 0,
            dsmDP2(("only consider section number = 0 here: %x\n", ui8)),
            goto _return );
         /* get lastSectionNumber*/
         READ_UINT8_L2CHK( pStreamDescrSection, ui8, ui8 == 0,
            dsmDP2(("only consider last section number = 0 here: %x\n", ui8)),
            goto _return );
         while (msg_len >= 2)
         {
            msg_len -= 2;
            /* -- get descriptor tag */
            READ_UINT8( pStreamDescrSection, dtag );
            /* -- get descriptor len */
            READ_UINT8( pStreamDescrSection, dlen );
            if (dlen > msg_len)
            {
               dsmDP1(("Invalid lengths: msg=%x, dlen=%x", msg_len, dlen));
               return CLDSM_ERR_END_OF_DATA;
            }
            msg_len -= dlen;
            switch (dtag)
            {
               case STREAM_NPT_REF_DESC:
               case STREAM_NPT_END_DESC:
               case STREAM_MODE_DESC:
               default:  /* Ignore all other tags */
                  break;

               case STREAM_EVENT_DESC:
                  if (dlen < 10)
                  {
                     dsmDP1(("Invalid desc length: %x", dlen));
                     return CLDSM_ERR_END_OF_DATA;
                  }
                  edata_len = dlen - 10;
                  if (edata_len != 0)
                  {
                     /* point at first byte of private data for the StreamEventDecriptor */
                     edata_ptr = pStreamDescrSection + 10;
                  }
            #if DSM_DATA_CHECK_LEVEL >= 2
                  READ_UINT16_L2CHK(pStreamDescrSection, ui16, ui16 == eventID,
                  dsmDP2(("DATA ERR: eventID mismatch %u,%u\n", eventID, ui16)),
                  goto _return );
            #else
                  pStreamDescrSection += 2;
            #endif
              #ifdef SUPPORT_SCHEDULED_EVENTS
                  /* read eventNPT */
                  READ_UINT32( pStreamDescrSection, ui32);
                  ULONG_Set32(pEvent->eventNpt, ui32 & 1);
                  ULONG_ShiftLeft(pEvent->eventNpt, 32);
                  READ_UINT32( pStreamDescrSection, ui32);
                  ULONG_Set32(pEvent->eventNpt, ui32); /* TODO: use eventNPT in synchronised event handling */
                  pStreamDescrSection += edata_len;
              #else
                  pStreamDescrSection += 8 + edata_len;
              #endif
            }
         }
         if (eventID == 0xFFFF || (eventID > 0 && eventID < 0x4000)) /* this is Lifecycle or "do-it-now" EVENT */
         {
            U8BIT *name = pEvent->name;
            dlen = pEvent->nlen;
            /* null terminate name */
            if (name != NULL)
            {
               ui8 = name[dlen];
               name[dlen] = 0;
            }

            dsmDP2(("\nEVENT Name:%s ID:%u version Nb:%u\n", name, eventID, version));

            dsmAssert((idp->setup.notifyStreamEventFunc != NULL));
            pL = pEvent->pListener;
            while (pL != NULL)
            {
               idp->setup.notifyStreamEventFunc( pL->eventHandle,
                  SEN_TRIGGERED, pL->userData1, pL->userData2, name, edata_ptr, dlen, edata_len );
               pL = pL->next;
            }
            /* restore original terminating char of name */
            if (name != NULL)
            {
               name[dlen] = ui8;
            }
         }
         /* else if ( eventID >= 0x8000 )
         {}
         else // if ( eventID >= 0x4000 && eventID <= 0x7FFF )
         {}
         */
      }
   }

_return:
   DEBUG_CHK( err == CLDSM_OK, dsmDP1(("ERROR: DSC_StrmEventUpdate: %u\n", err)));
   return err;
}

void DSC_StrmEventFilterReset( P_DsmCoreInst idp )
{
   S_SE_INFO *pStreamInfo;
   S_Event *pEvent;
   U16BIT evntNdx, evntMax;
   pStreamInfo = idp->hSubscribedEventList;
   while (pStreamInfo != NULL)
   {
      pEvent = (S_Event *)(pStreamInfo + 1);
      evntMax = pStreamInfo->eventCount;
      for (evntNdx = 0; evntNdx != evntMax; evntNdx++, pEvent++)
      {
         if (pEvent->pFilter != NULL)
         {
            DBGLOG((DD_SE | DD_SF), "serv_id=0x%x pFilter=%p", pStreamInfo->serviceId, pEvent->pFilter)
            DSC_SectionFilterStop( idp, &pEvent->pFilter );
         }
         StartSectionFilter(idp, pStreamInfo, pEvent);
      }
      pStreamInfo = pStreamInfo->next;
   }
}

/*----------------------------------------------------------------------------*/

/*
 * Free up stream event resources
 */
static void FreeStreamInfo( P_DsmCoreInst idp, S_SE_INFO *pStreamInfo )
{
   S_Event *pEvent;
   U16BIT eventCount;
   S_LISTENER *pL;

   if (pStreamInfo->xmlPtr)
   {
      DSC_CmMemRelease( idp, pStreamInfo->xmlPtr );
   }
   pEvent = (S_Event *)(pStreamInfo + 1);
   eventCount = pStreamInfo->eventCount;
   while (eventCount--)
   {
      while (pEvent->pListener != NULL)
      {
         pL = pEvent->pListener;
         ERRPRINT("ERROR !Listener still subscribed! hdl=%x", pL->eventHandle)
         pEvent->pListener = pL->next;
         DSC_CmMemRelease( idp, pL );
      }
      if (pEvent->pFilter != NULL)
      {
         DBGLOG((DD_SE | DD_SF), "serv_id=0x%x pFilter=%p", pStreamInfo->serviceId, pEvent->pFilter)
         DSC_SectionFilterStop( idp, &pEvent->pFilter );
      }
      pEvent++;
   }
   DSC_CmMemRelease( idp, pStreamInfo );
}

/**
 *          idp                   IN
 * @param   streamObject This references a BIOP::StreamEventMessage as
 *          well as indicating the carousel, module and
 *          the kind of the object.Handle to the object
 * @param   ppStreamInfo The is the handle to the stream event
 *          returned by subscribe.
 *          The caller will use this handle when the
 *          stream event being subscribed to is referred
 *          to in the future.
 * @return
 *          CLDSM_OK
 */
static E_DscError RetrieveStreamInfo( P_DsmCoreInst idp,
   /*I*/ H_DsmObject streamObject,
   /*O*/ S_SE_INFO **ppStreamInfo )
{
   MemPtr mpObjectData, mpObjectDataFirstPos;
   MemPos hdlPos;
   P_Module pModule;
   U8BIT ui8;
   U16BIT ui16;
   U32BIT ui32;
   U16BIT eventCount, namesLen;
   U16BIT tapUse;
   E_DscError err;
   U8BIT tapsCount;
   S_Event *pEvent;
   S_SE_INFO *pStreamInfo;
   U8BIT *pNames;

   *ppStreamInfo = NULL;

   if (!DSM_OBJECT_CORRECT( streamObject, idp ))
   {
      dsmDP1(("API ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
   }
   else if (streamObject->objectInfo.objectKind != STE_STR)
   {
      dsmDP1(("ERROR: message is not a streamEvent !! = %u\n",
              streamObject->objectInfo.objectKind));
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if ((pStreamInfo = FindStreamInfo(idp, streamObject)) != NULL)
   {
      *ppStreamInfo = pStreamInfo;
      err = CLDSM_OK;
   }
   else
   {
      pModule = streamObject->pModule;
      if (!moduleDataFindObject(pModule->hModuleData, pModule->moduleInfo.originalSize,
             &(streamObject->objectInfo.objectKey), &mpObjectData ))
      {
         dsmDP1(("failed to find object in module"));
         err = CLDSM_ERR_OBJECT_NOT_LOADED;
      }
      else
      {
         mpObjectDataFirstPos = mpObjectData;

         /* SKIP biop magic + version + byte order + msg size */
         SET_POS_REL(mpObjectData, 12);

         /* SKIP object key */
         READ_UINT8(mpObjectData, ui8 );
         SET_POS_REL(mpObjectData, ui8);

         /* get object kind */
         READ_UINT32( mpObjectData, ui32)
         if (ui32 != 4)
         {
            dsmDP2(("DATA ERROR: objectKindLg !=4 %u\n", ui32));
            err = CLDSM_ERR_INVALID_OBJECT_TYPE;
         }
         else
         {
            READ_UINT32( mpObjectData, ui32)
            if (ui32 != 0x73746500)
            {
               dsmDP2(("DATA ERROR: objectKind != ste %u\n", ui32));
               err = CLDSM_ERR_INVALID_OBJECT_TYPE;
            }
            else
            {
               /* skip objectInfo Length */
               SET_POS_REL(mpObjectData, 2);

               /* SKIP aDescriptionLg */
               READ_UINT8(mpObjectData, ui8);    /*-- N3 */
               SET_POS_REL(mpObjectData, ui8);

               /* SKIP duration seconds, microsecs, audio, video, data (4 + 4 + 1 + 1 + 1) */
               SET_POS_REL(mpObjectData, 11);

               READ_UINT16(mpObjectData, eventCount);
               GET_POS( mpObjectData, hdlPos ); /* save position */
               namesLen = 0;
               for (ui16 = eventCount; ui16--; )
               {
                  READ_UINT8( mpObjectData, ui8 );
                  SET_POS_REL(mpObjectData, ui8);
                  namesLen += ui8;
               }
               ui32 = sizeof(S_SE_INFO) + (eventCount * sizeof(S_Event)) + namesLen;
               pStreamInfo = (S_SE_INFO *)DSC_CmMemGet( idp, ui32 );
               if (!pStreamInfo)
               {
                  err = CLDSM_ERR_MEM_HEAP_FULL;
               }
               else
               {
                  err = CLDSM_OK;
                  pStreamInfo->pModule = pModule;
                  memcpy(&(pStreamInfo->objectKey), &(streamObject->objectInfo.objectKey), sizeof(S_ObjectKey));
                  pStreamInfo->uniqueId = ++unique_stream_ids;
                  if (memValidate(streamObject->pObjCarousel))
                  {
                     pStreamInfo->serviceId = streamObject->pObjCarousel->root.serviceId;
                  }
                  else
                  {
                     pStreamInfo->serviceId = idp->dvbLocator.service_id;
                  }
                  pStreamInfo->eventCount = eventCount;
                  pStreamInfo->lstnrCount = 0;
                  pEvent = (S_Event *)(pStreamInfo + 1);
                  pNames = (U8BIT *)(pEvent + eventCount);
                  pStreamInfo->namesPtr = pNames;
                  pStreamInfo->namesLen = namesLen;
                  pStreamInfo->xmlPtr = NULL;

                  /* reset module data pointer to beginning of names */
                  SET_POS_ABS( mpObjectData, hdlPos );
                  for (ui16 = eventCount; ui16--; pEvent++)
                  {
                     /* Note: DBOOK (table 17-20) says name length includes zero terminator */
                     READ_UINT8( mpObjectData, ui8 );
                     pEvent->nlen = ui8 - 1; /* keep length without the terminating 'null' char */
                     pEvent->name = pNames;
                     while (ui8--)
                     {
                        READ_UINT8( mpObjectData, *pNames );
                        pNames++;
                     }
                     /* replace last char ('null') with comma,
                      * so that 'nameList' is a comma separated list */
                     *(pNames - 1) = ',';
                     /* initialise other event properties */
                     pEvent->version = 0xFF; /* invalid value */
                     pEvent->pFilter = NULL;
                     pEvent->pListener = NULL;
                     #ifdef SUPPORT_SCHEDULED_EVENTS
                     pEvent->eventNpt = 0;
                     #endif
                     /* pEvent->id is initialised later */
                  }
                  /* revert last char back to 'null' (not comma) */
                  *(pNames - 1) = 0;

                  mpObjectData = mpObjectDataFirstPos;
                  SET_POS_REL( mpObjectData, (S32BIT)(streamObject->objectInfo.messageBodyOffset - 4));

                  /* get messageBody_length */
                  READ_UINT32(mpObjectData, ui32);
                  if (ui32 == 0)
                  {
                     dsmDP2(("RetrieveStreamInfo(-): msg len is zero\n"));
                  }

                  /* get taps_count */
                  READ_UINT8(mpObjectData, tapsCount);

                  pStreamInfo->eventAssocTag = 0;

                  while (tapsCount--)
                  {
                     READ_UINT16(mpObjectData, ui16);   /* tap id */
                     if (ui16 != 0)
                     {
                        dsmDP1(("tap Id should zero, here=%d\n", ui16));
                     }
                     READ_UINT16(mpObjectData, tapUse);
                     /* get assocTag */
                     READ_UINT16(mpObjectData, ui16);
                     dsmDP2(("TAP: use=%d assocTag=0x%x\n", tapUse, ui16));
                     /* get selector_length */
                     READ_UINT8(mpObjectData, ui8);
                     if (ui8 != 0)
                     {
                        dsmDP1(("RetrieveStreamInfo(-): selector_length should be zero here=%d\n", ui8));
                     }
                     switch (tapUse)
                     {
                        case STR_STATUS_AND_EVENT_USE:
                        case STR_EVENT_USE:
                           pStreamInfo->eventAssocTag = ui16;
                        default:;
                     }
                  }
                  /* get eventIds_count */
                  READ_UINT8(mpObjectData, ui8);
                  if (eventCount != ui8)
                  {
                     dsmDP1(("eventNamesCount=%d != eventIds_count=%d\n", eventCount, ui8));
                  }
                  pEvent = (S_Event *)(pStreamInfo + 1);
                  for (ui16 = eventCount; ui16--; pEvent++)
                  {
                     READ_UINT16(mpObjectData, pEvent->id);
                  }
                  /* add to list */
                  pStreamInfo->next = idp->hSubscribedEventList;
                  idp->hSubscribedEventList = pStreamInfo;
                  *ppStreamInfo = pStreamInfo;
                  err = CLDSM_OK;
               }
            }
         }
      }
   }
   return(err);
}

void ReleaseStreamInfo( P_DsmCoreInst idp, S_SE_INFO *sei )
{
   S_SE_INFO **ppSEI;

   sei->lstnrCount--;
   if (!sei->lstnrCount)
   {
      ppSEI = &idp->hSubscribedEventList;
      while (*ppSEI != NULL)
      {
         if (*ppSEI == sei)
         {
            *ppSEI = (*ppSEI)->next;
            FreeStreamInfo( idp, sei );
            break;
         }
         ppSEI = &((*ppSEI)->next);
      }
   }
}

void DSC_StrmEventListReset( P_DsmCoreInst idp )
{
   S_SE_INFO *sei, *nxt;
   sei = idp->hSubscribedEventList;
   while (sei != NULL)
   {
      nxt = sei->next;
      FreeStreamInfo( idp, sei );
      sei = nxt;
   }
   idp->hSubscribedEventList = NULL;
}

/**
 * @brief   The Client uses this function to request list of event names for stream object
 *          Before calling this function the Client must request that the stream object of
 *          interest be loaded using the CDSM_ObjectLoad() function. This action will
 *          deliver a handle to the object stream - this is passed as the streamObject
 *          argument to this function.
 *          Do not have to wait until load is completed
 *          Do not have to open an object
 * @param   dsmccInstance Allows support >= 1 instance of a DSMCC.
 * @param   streamObject This references a BIOP::StreamEventMessage as well
 *          as indicating the carousel, module and the kind of
 *          the object.
 * @param   pNamesPtr Pointer to comma separated list of Event names
 * @param   pNamesLen Length of comma separated list of Event names
 * @return
 *          CLDSM_ERR_INVALID_INSTANCE
 *          The Instance value is invalid
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The supplied object handle is invalid
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The message referred to by streamObject is not a BIOP::StreamEventMessage
 *          CLDSM_ERR_MEM_HEAP_FULL
 *          Out of memory.
 *          CLDSM_OK
 *          Successfully retrieved list of names for this stream object.
 */
E_DscError CDSM_StreamEventNameList(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmObject streamObject,
   /*O*/ U8BIT **pNamesPtr, U16BIT *pNamesLen )
{
   E_DscError err;
   S_SE_INFO *pStreamInfo;
   err = RetrieveStreamInfo( dsmccInstance, streamObject, &pStreamInfo );
   if (CLDSM_OK == err)
   {
      *pNamesPtr = pStreamInfo->namesPtr;
      *pNamesLen = pStreamInfo->namesLen;
   }
   return err;
}

/**
 * @brief   The Client uses this function to request data in XML format for stream object and
 *          associated events. This XML format is defined by 'TS 102 809' v 1.2.1, section 8.2
 *          Before calling this function the Client must request that the stream object of
 *          interest be loaded using the CDSM_ObjectLoad() function.
 * @param   dsmccInstance instance of DSMCC.
 * @param   streamObject handle to stream object
 * @param   pXmlPtr Pointer to XML data for Stream Object
 * @param   pXmlLen Length of XML data for Stream Object
 * @return  CLDSM_ERR_INVALID_INSTANCE - The Instance value is invalid
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE - The supplied object handle is invalid
 *          CLDSM_ERR_INVALID_OBJECT_TYPE - streamObject is not a BIOP::StreamEventMessage
 *          CLDSM_ERR_MEM_HEAP_FULL - Out of memory.
 *          CLDSM_OK - Successfully retrieved XML data for stream object.
 */
E_DscError CDSM_StreamEventXmlData(
   /*I*/ H_DsmCoreInst dsmccInstance,
   /*I*/ H_DsmObject streamObject,
   /*O*/ U8BIT **pXmlData, U16BIT *pXmlLen)
{
   E_DscError err;
   S_SE_INFO *pStreamInfo;
   S_Event *pEvent;
   err = RetrieveStreamInfo( dsmccInstance, streamObject, &pStreamInfo );
   if (CLDSM_OK == err)
   {
      if (pStreamInfo->xmlPtr == NULL)
      {
         char *data;
         U16BIT sz;
         sz = sizeof(XML_DATA2a) + sizeof(XML_DATA2b) + sizeof(XML_DATA2c) + 2; // add (5-3) to allow for max event id (65535)
         sz *= pStreamInfo->eventCount;
         sz += pStreamInfo->namesLen;
         sz += sizeof(XML_DATA1a) + sizeof(XML_DATA1b) + sizeof(XML_DATA3);
         data = (char *)DSC_CmMemGet( (P_DsmCoreInst)dsmccInstance, sz );
         if (data != NULL)
         {
            pStreamInfo->xmlPtr = (U8BIT *)data;
            strcpy(data, XML_DATA1a);
            data += sizeof(XML_DATA1a) - 1;
            data += sprintf(data, "%u", pStreamInfo->eventAssocTag);
            strcpy(data, XML_DATA1b);
            data += sizeof(XML_DATA1b) - 1;
            pEvent = (S_Event *)(pStreamInfo + 1);
            for (sz = 0; sz != pStreamInfo->eventCount; sz++, pEvent++)
            {
               strcpy(data, XML_DATA2a);
               data += sizeof(XML_DATA2a) - 1;
               data += sprintf(data, "%u", pEvent->id);
               strcpy(data, XML_DATA2b);
               data += sizeof(XML_DATA2b) - 1;
               strncpy(data, (const char *)pEvent->name, pEvent->nlen);
               data += pEvent->nlen;
               strcpy(data, XML_DATA2c);
               data += sizeof(XML_DATA2c) - 1;
            }
            strcpy(data, XML_DATA3);
            data += sizeof(XML_DATA3) - 1;
            pStreamInfo->xmlLen = (U16BIT)((U8BIT*)data - pStreamInfo->xmlPtr);
         }
      }
      *pXmlData = pStreamInfo->xmlPtr;
      *pXmlLen = pStreamInfo->xmlLen;
   }
   return err;
}

