/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 * Copyright  2002 Koninklijke Philips Electronics N.V
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions to support Stream Functionality
 * @file    streamObject.c
 * @date    06/03/2002
 * @author  R.Taylor
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "clDsmUtils.h"
#include "streamObject.h"
#include "defMemUtilsMgd.h"  /* -- Default mem type for module */
#include "siQuery.h"

/*------------------------------- Local Macros -------------------------------*/


/*------------------------------  Exported Data  -----------------------------*/


/*--------------------------------Local Types --------------------------------*/


/*------------------------------- Local Statics ------------------------------*/


/*------------------- local prototypes/forward declarations ------------------*/



/*------------------------------ Local Functions -----------------------------*/


/**
 * @brief   Search a tap list for given tap use and return the associated association_tag.
 *          The 'addr' argument must refer to the 'taps_count' field in a message
 *          containing a taps list. If not unexpected results will occur.
 *          This function is non-destructive w.r.t. the MemSeqRef address is concerned.
 * @param   addr Reference to taps_count at start of tap list
 * @param   targetTapUse Tap use to look for
 * @param   pAssocTag The required association tag
 * @return
 *          TRUE  - Tap found
 *          FALSE - Tap NOT found
 */
static BOOLEAN GetTapAssocTag( MemSeqRef addr, U16BIT targetTapUse, U16BIT *pAssocTag)
{
   BOOLEAN tapFound = FALSE;    /* Return value - assume not found */
   U32BIT initialOffset;
   U8BIT taps_count;
   U16BIT tapUse;
   U8BIT ui8;
   U8BIT i;                  /* General purpose 8-bit loop counter */

   /* Remember the offset, so we can restore position before return */
   memSeqReadPos(addr, &initialOffset);

   /* Read taps count */
   memSeqReadByte(addr, &taps_count);

   for (i = 0; (i < taps_count) && (tapFound == FALSE); i++)
   {
      /* Skip ID */
      memSeqSetPosRel(addr, 2);
      /* Read use */
      readUInt16Seq(addr, &tapUse);

      /* Have we found the tap ? */
      if (tapUse == targetTapUse)
      {
         /* Read associationTag */
         readUInt16Seq(addr, pAssocTag);
         tapFound = TRUE;
      }
      else
      {
         /* Skip use */
         memSeqSetPosRel(addr, 2);
         /* Read selectorLength */
         memSeqReadByte(addr, &ui8);
         /* Skip selector */
         memSeqSetPosRel(addr, (U32BIT)ui8);
      }
   }

   /* Reset offset to start */
   memSeqSetPosAbs(addr, initialOffset);

   return tapFound;
}

/*---------------------------- Exported Functions ----------------------------*/

/**
 * @brief   The Client calls this function to obtain the Association Tag which can be
 *          used to determine the PID on which stream data can be found.
 *          Before calling this function the Client must have completed loading and opened
 *          the stream object.
 * @param   dsmccInstance Allows support >= 1 instance of a DSMCC.
 * @param   streamObject This handle indicates the stream object
 * @param   userData1 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   userData2 Optional data to be used by client.
 *          This data is stored and returned unchanged to
 *          the Client using a callback.
 * @param   pDeferredService return a pointer to deferred service structure
 * @return
 *          CLDSM_ERR_INVALID_OBJECT_HANDLE
 *          The supplied object handle was not valid.
 *          CLDSM_ERR_INVALID_OBJECT_TYPE
 *          The streamObject does not refer to data containing the association tag.
 *          CLDSM_ERR_OBJECT_NOT_LOADED
 *          The streamObject does not refer to a loaded object.
 *          CLDSM_ERR_OBJECT_NOT_OPEN
 *          The streamObject does not refer to an open object.
 *          CLDSM_ERR_UNABLE_TO_GET_PROGRAM_ASSOC_TAG
 *          No BIOP_PROGRAM_USE tap could be found.
 *          CLDSM_ERR_SI_QUERY_FAILED
 *          SI Query Failed
 *          CLDSM_OK
 *          Association Tag was successfully obtained.
 */
E_DscError CDSM_StreamGetDeferredService(  H_DsmCoreInst instance,
   /*I*/ H_DsmObject streamObject, void *userData1, void *userData2,
   /*O*/ S_DvbLocator **ppDeferredService )
{
   P_DsmCoreInst idp = (P_DsmCoreInst) instance;
   P_DsmObject pDsmObject = (P_DsmObject)streamObject;
   E_DscError err;
   U16BIT associationTag;

   dsmDP2(("dsmStreamGetDeferredService(%u)\n", streamObject));

   if (!DSM_OBJECT_VALIDATE(pDsmObject))
   {
      dsmDP1(("ERROR: Invalid object handle\n"));
      err = CLDSM_ERR_INVALID_OBJECT_HANDLE;
      dsmAssert((0));                                            /* FATAL */
   }
   else if (pDsmObject->kind != STREAM_OBJ &&
            pDsmObject->kind != STREAM_OBJ_WITH_EVENTS)
   {
      dsmDP1(("ERROR: Incorrect kind=%d\n", pDsmObject->kind));
      err = CLDSM_ERR_INVALID_OBJECT_TYPE;
   }
   else if (pDsmObject->status != OBJ_LOAD_COMPLETED)
   {
      dsmDP1(("ERROR: Object load not completed\n"));
      err = CLDSM_ERR_OBJECT_NOT_LOADED;
   }
   else if (pDsmObject->objectDataSeq == NULL)
   {
      dsmDP1(("ERROR: Object not open\n"));
      err = CLDSM_ERR_OBJECT_NOT_OPEN;
   }
   else if (pDsmObject->data.so.retrieved == DEFFERED_SERVICE_RECEIVED)
   {
      /* Great! Deferred service has already been retrieved - just pass it back */
      *ppDeferredService = &pDsmObject->data.so.multiplex;
      err = CLDSM_OK;
   }
   else if (!GetTapAssocTag(pDsmObject->objectDataSeq, BIOP_PROGRAM_USE, &associationTag))
   {
      dsmDP1(("ERROR: Cannot find BIOP_PROGRAM_USE tap\n"));
      err = CLDSM_ERR_UNABLE_TO_GET_PROGRAM_ASSOC_TAG;
   }
   else
   {
      /* Got association tag. */
      S_SIQueryRequest siQueryData;
      S_SIQueryResult siQueryResult;

      siQueryData.kind = SIQUERY_DEFERRED_SERVICE;
      siQueryData.serviceId = idp->dvbLocator.service_id;
      siQueryData.associationTag = associationTag;

      err = siQueryStart( idp, &siQueryData, (void *)streamObject, &siQueryResult );

      switch (err)
      {
         case CLDSM_OK:
            /* Store stream object's deferred service */
            pDsmObject->data.so.multiplex = siQueryResult.data.deferredService;
            pDsmObject->data.so.retrieved = DEFFERED_SERVICE_RECEIVED;
            *ppDeferredService = &pDsmObject->data.so.multiplex;
            break;

         case CLDSM_PENDING:
            /* Store query ref in object */
            pDsmObject->r.pPendingSiQueryRef = siQueryResult.data.queryHandle;
            pDsmObject->data.ds.userData1 = userData1;
            pDsmObject->data.ds.userData2 = userData2;
            break;

         default:
            /* Store currrent service in stream object */
            pDsmObject->data.so.multiplex = idp->dvbLocator;
            pDsmObject->data.so.retrieved = DEFFERED_SERVICE_RECEIVED; /* well, we tried! */
            *ppDeferredService = &pDsmObject->data.so.multiplex;
            err = CLDSM_OK;
      }
   }
   return err;
}

/* DSC_StrmObjectStoreDeferred called when SiQuery result returned */
void DSC_StrmObjectStoreDeferred( P_DsmCoreInst idp, P_DsmObject pDsmObject,
   S_DvbLocator multiplex )
{
   void *ds_ud1;
   void *ds_ud2;

   if (DSM_OBJECT_VALIDATE(pDsmObject) &&
       (pDsmObject->status == OBJ_LOAD_COMPLETED) &&
       (pDsmObject->kind == STREAM_OBJ || pDsmObject->kind == STREAM_OBJ_WITH_EVENTS) &&
       (pDsmObject->r.pPendingSiQueryRef != NULL)
       )
   {
      ds_ud1 = pDsmObject->data.ds.userData1;
      ds_ud2 = pDsmObject->data.ds.userData2;
      pDsmObject->data.so.multiplex = multiplex;
      pDsmObject->data.so.retrieved = DEFFERED_SERVICE_RECEIVED;
      pDsmObject->r.pPendingSiQueryRef = NULL;
      if (idp->setup.notifyDeferredServiceFunc)
      {
         idp->setup.notifyDeferredServiceFunc( ds_ud1, ds_ud2, &pDsmObject->data.so.multiplex );
      }
   }
}

/*----------------------------------------------------------------------------*/
