/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Implementation of the Update Carousel module.
 * @file    updateCarousel.h
 * @date    23/02/2015
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include "clDsmSystem.h"
#include "updateCarousel.h"
#include "sectionTimer.h"

#include "cacheMgr.h"

/*------------------------------- Local Macros -------------------------------*/

#define SSU_GROUP_LIST           0
#define NUM_LISTS_SSU_GROUPS     1

/*------------------------------  Exported Data  -----------------------------*/

/*--------------------------------Local Types --------------------------------*/

typedef struct s_GroupInfo *P_GroupInfo;
typedef struct s_GroupInfoItem *P_GroupInfoItem;

typedef struct s_GroupInfo
{
   U32BIT id;
   U32BIT size;
   U32BIT oui;
   S_SsuModelVersion smv;
   U32BIT subgroup;
   U32BIT nextGroupId;
   U16BIT associationTag;
   U8BIT positionType; /* for values see E_PostionType */
} S_GroupInfo;

typedef struct s_GroupInfoItem
{
   S_LLObject llData[NUM_LISTS_SSU_GROUPS];
   S_GroupInfo group;
   P_DataCarousel pDC;
} S_GroupInfoItem;

/*------------------------------- Local Statics ------------------------------*/


/*------------------- local prototypes/forward declarations ------------------*/

static void CarouselLoadFinalise( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst );

/*------------------------------ Local Functions -----------------------------*/

/**
 * @brief   Load Group (get DII or DataCarousel)
 * @param   idp DSM core Instance
 * @param   pUC Pointer to Update Carousel
 */
static E_DscError CreateGroupRequest( P_DsmCoreInst idp, P_UpdateCarousel pUC, S_GroupInfo *pGroup )
{
   E_DscError err;
   P_DataCarousel pDC;
   S_DeliveryParaTap tap;
   P_GroupInfoItem pGII;
   P_RootLoadRqst pLoadRqst;
   pGII = (P_GroupInfoItem)DSC_CmMemGet( idp, sizeof(S_GroupInfoItem) );
   if (!pGII)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
   }
   else
   {
      llLinkInit( pGII->llData, NUM_LISTS_SSU_GROUPS );
      pGII->group = *pGroup;
      tap.id = 0;
      tap.transactionId = pGroup->id;
      tap.associationTag = pGroup->associationTag;
      tap.timeout = 0xFFFFFFFF;
      err = DSC_DataCrslCreate( idp, &pUC->root, &tap, &pDC );
      if (!err)
      {
         err = LLCreate( idp, pDC, MODULE_LOAD_REQUEST_LIST, &pDC->llcLoadRequests );
         if (!err)
         {
            err = DSC_LoadRsqtCreate( idp, sizeof(S_RootLoadRqst), TT_SSU_GROUP, pGII,
                  (F_LoadFinalise)CarouselLoadFinalise, &pLoadRqst );
            if (!err)
            {
               LLInsertHead( pDC->llcLoadRequests, pLoadRqst );
               err = DSC_DataCrslAcquireStart( idp, pDC, SF_PRIORITY_HIGH );
               if (!err)
               {
                  LLInsertTail( pUC->llcGroups, pGII );
               }
               else
               {
                  LLDestroy( idp, &pDC->llcLoadRequests );
                  DSC_LoadRsqtDestroy( idp, pLoadRqst );
               }
            }
            else
            {
               LLDestroy( idp, &pDC->llcLoadRequests );
            }
         }
         if (err)
         {
            DSC_DataCrslDestroy( idp, pDC );
         }
      }
      if (err)
      {
         DSC_CmMemRelease( idp, pGII );
      }
   }
   DBGERRCHK(err)
   return err;
}

static void UnloadGroups( P_DsmCoreInst idp, P_LLControl llcGroups )
{
   P_GroupInfoItem item;
   item = LLRemoveTail( llcGroups );
   while (item != NULL)
   {
      ERRPRINT("release=%p", item)
      DSC_CmMemRelease(idp, item);
      item = LLRemoveTail( llcGroups );
   }
}

/*----------------------------------------------------------------------------*/

/*---------------------------- Exported Functions ----------------------------*/

/**
 * @brief   Creates an instance of the UpdateCarousel and initialises it.
 * @param   idp DSM core Instance
 * @param   serviceId Service Id on which to find Update Carousel
 * @param   oui Organisation Unique Identifier of manufacturer
 * @param   ppUC Pointer to return Update Carousel
 * @return  E_DscError
 */
E_DscError DSC_UpdCrslCreate( P_DsmCoreInst idp, U16BIT serviceId,
   U32BIT oui, P_UpdateCarousel *ppUC )
{
   P_UpdateCarousel pUC;
   P_RootLoadRqst pLoadRqst;
   E_DscError err;

   dsmAssert((idp != NULL));
   dsmAssert((ppUC != NULL));

   pUC = (P_UpdateCarousel)DSC_CmMemGet( idp, sizeof(S_UpdateCarousel) );
   if (!pUC)
   {
      err = CLDSM_ERR_MEM_HEAP_FULL;
      *ppUC = NULL;
   }
   else
   {
      memset(pUC, 0, sizeof(S_UpdateCarousel));
      err = DSC_RootCrslInitialise( idp, &pUC->root, UC_MAGIC, serviceId, oui );
      if (err)
      {
         DSC_CmMemRelease(idp, pUC);
         *ppUC = NULL;
      }
      else
      {
         err = LLCreate( idp, pUC, SSU_GROUP_LIST, &pUC->llcGroups );
         if (err)
         {
            pUC->llcGroups = NULL;
            DSC_RootCrslFinalise( idp, &pUC->root );
            DSC_CmMemRelease(idp, pUC);
            *ppUC = NULL;
         }
         else
         {
            err = DSC_LoadRsqtCreate( idp, sizeof(S_RootLoadRqst), TT_CAROUSEL, pUC,
                  CarouselLoadFinalise, &pLoadRqst );
            if (err)
            {
               LLDestroy( idp, &(pUC->llcGroups));
               DSC_RootCrslFinalise( idp, &pUC->root );
               DSC_CmMemRelease(idp, pUC);
               *ppUC = NULL;
            }
            else
            {
               pUC->root.pLoadRqst = pLoadRqst;
               *ppUC = pUC;
            }
         }
      }
   }
   DEBUG_CHK( err == CLDSM_OK, ERRPRINT("Error %u", err));
   return err;
}

/**
 * @brief   Destroy UpdateCarousel
 * @param   idp DSM core Instance
 * @param   pUC Pointer to Update Carousel
 */
void DSC_UpdCrslDestroy( P_DsmCoreInst idp, P_UpdateCarousel pUC )
{
   dsmAssert((idp != NULL));
   dsmAssert((pUC != NULL));

   /* -- Free up Group list */
   if (LLCount( pUC->llcGroups ))
   {
      UnloadGroups( idp, pUC->llcGroups );
   }

   LLDestroy( idp, &(pUC->llcGroups));

   DSC_RootCrslFinalise( idp, &pUC->root );

   memset(pUC, 0, sizeof(S_UpdateCarousel));
   DSC_CmMemRelease( idp, pUC );
}

static void ParseGroupInfoDescriptors( MemPtr pData, U16BIT length, S_GroupInfo *pGroup )
{
   MemPos currPos, endPos;
   U8BIT dtag;
   U8BIT dlen;
   U8BIT ui8;
   DBG2(DD_OC,"")
   GET_POS( pData, currPos );
   endPos = currPos + length;
   while (currPos < endPos)
   {
      READ_UINT8( pData, dtag );
      READ_UINT8( pData, dlen );
      switch (dtag)
      {
         case DESCRIPTOR_TYPE_TAG:
         {
            break;
         }
         case DESCRIPTOR_NAME_TAG:
         {
            break;
         }
         case DESCRIPTOR_INFO_TAG:
         {
            break;
         }
         case DESCRIPTOR_LOCATION_TAG:
         {
            READ_UINT8( pData, ui8 );
            pGroup->associationTag = ui8;
            break;
         }
         case DESCRIPTOR_GROUP_LINK_TAG:
         {
            READ_UINT8( pData, ui8 );
            READ_UINT32( pData, pGroup->nextGroupId );
            pGroup->positionType = ui8 + 1; /* increment so know received desc */
            break;
         }
         case DESCRIPTOR_SSU_SUBGROUP_ASS_TAG:
         {
            READ_UINT32( pData, pGroup->subgroup );
            break;
         }
         default:
         {
            DBG2(DD_OC,"Unknown descriptor tag: %x", dtag);
            break;
         }
      }
      SET_POS_ABS( pData, currPos + 2 + dlen );
      GET_POS( pData, currPos );
   }
}

/**
 * @brief   Parse DSI private data from Group Info of Update Carousel
 * @param   idp DSM core Instance
 * @param   pUC Pointer to Update Carousel
 * @param   pDsiPrivate Pointer to DSI private data
 * @param   dsiPrivateLen Length of DSI private data
 * @return  E_DscError
 */
E_DscError DSC_UpdCrslParseGroupInfo( P_DsmCoreInst idp, P_UpdateCarousel pUC,
   U8BIT *pDsiPrivate, U16BIT dsiPrivateLen )
{
   E_DscError err = CLDSM_ERR_END_OF_DATA;
   S_GroupInfo group;
   U8BIT *pData;
   U16BIT glen, numGroups;
   P_Compatibility pCompat;

   dsmAssert((pDsiPrivate != NULL));

   if (dsiPrivateLen < 2)
   {
      ERRLOG(DD_OC, "DATA ERROR: Group short len=%u", dsiPrivateLen )
   }
   else
   {
      pData = pDsiPrivate;
      READ_UINT16( pData, numGroups );
      if (dsiPrivateLen < ((numGroups * 12) + 2))
      {
         ERRLOG(DD_OC, "DATA ERROR: Group short len=%u groups=%u", dsiPrivateLen, numGroups )
      }
      else
      {
         dsmAssert((pUC->root.status == RCS_BOOTING));
         pUC->root.pLoadRqst->status = LRS_STALLED_SRG_MODULE;
         pUC->root.status = RCS_BOOTED;
         DBGLOG(DD_OC, "numGroups %u", numGroups)
         while (numGroups--)
         {
            memset(&group, 0, sizeof(S_GroupInfo));
            group.oui = OUI_SPECIFIER_TYPE | (pUC->root.rcid & OUI_DATA_MASK);
            group.associationTag = pUC->root.dsiAssocTag;
            READ_UINT32( pData, group.id );
            READ_UINT32( pData, group.size );

            READ_UINT16( pData, glen ); /*Group Compatibility*/
            pCompat = DSC_UtilParseCompatibilityDesc( idp, pData, glen, group.oui );
            SET_POS_REL( pData, glen ); /* Skip */

            READ_UINT16( pData, glen ); /*group info len*/
            if (glen)
            {
               DBGLOG(DD_OC, "Group(%x) Info length=%d", group.id, glen)
               ParseGroupInfoDescriptors( pData, glen, &group );
               SET_POS_REL( pData, glen ); /* Skip */
            }
            READ_UINT16( pData, glen ); /*group private data len*/
            if (glen)
            {
               DBGLOG(DD_OC, "Group(%x) Private Data length=%d", group.id, glen)
               SET_POS_REL( pData, glen ); /* Skip */
            }
            if (pCompat == NULL)
            {
               ERRLOG(DD_OC, "fail parsing Compatibility Descriptor")
            }
            else
            {
               U16BIT h,s;
               DBGLOG(DD_OC, "pCompat->sw_num %u", pCompat->sw_num)
               for (s = 0; s < pCompat->sw_num; s++)
               {
                  group.smv.sw_model = pCompat->swlist[s].model;
                  group.smv.sw_version = pCompat->swlist[s].version;
                  DBGLOG(DD_OC, "pCompat->hw_num %u", pCompat->hw_num)
                  for (h = 0; h != pCompat->hw_num; h++)
                  {
                     group.smv.hw_model = pCompat->hwlist[h].model;
                     group.smv.hw_version = pCompat->hwlist[h].version;
                     if (idp->setup.ssuFuncs.wanted(group.id,group.size,&(group.smv)))
                     {
                        err = CreateGroupRequest( idp, pUC, &group );
                        if (err)
                        {
                           err = handleInLoopError( idp, CLDSM_OK, err );
                        }
                        /* confirmed match, so can exit both loops */
                        s = pCompat->sw_num;
                        break;
                     }
                  }
               }
               DSC_CmMemRelease( idp, pCompat );
            }
         }
      #ifdef DSI_TIMEOUT_SUPPORT
         sectionTimerRemove(idp, pUC->root.pDsiSf); /* stop DSI timer */
      #endif /* DSI_TIMEOUT_SUPPORT */
         switch (err)
         {
            case CLDSM_ERR_END_OF_DATA:
            {
               DSC_RootCrslLoadRequestFail( idp, &pUC->root );
               break;
            }
            default:
            {
               DSC_LoadRsqtFinalise( idp, pUC->root.pLoadRqst );
               pUC->root.pLoadRqst = NULL;
            }
            case CLDSM_OK:
            {
               DBGLOG(DD_OC, "Found update")
            }
         }
      }
   }
   return err;
}

BOOLEAN DSC_UpdCrslCheckCompatibility( P_UpdateCarousel pUC, U8BIT *pCompatDesc,
   U16BIT compatLen )
{
   P_GroupInfoItem item;
   item = LLHead(pUC->llcGroups);
   while (item != NULL)
   {
      if (DSC_UtilCheckCompatibility(pCompatDesc, compatLen, item->group.oui, item->group.smv))
      {
         break;
      }
      item = LLNext( item, SSU_GROUP_LIST );
   }
   return (item)? TRUE : FALSE;
}

static void CarouselLoadFinalise( P_DsmCoreInst idp, P_RootLoadRqst pLoadRqst )
{
   P_UpdateCarousel pCarousel;
   P_GroupInfoItem pGII;
   E_UCLoadStatus status;
   U_StatusRef sr;
   U32BIT data;
   data = 0;
   if (pLoadRqst->targetKind == TT_CAROUSEL)
   {
      pCarousel = pLoadRqst->target;
   }
   else
   {
      dsmAssert((pLoadRqst->targetKind == TT_SSU_GROUP));
      pGII = pLoadRqst->target;
      pCarousel = LLParent( pGII, SSU_GROUP_LIST );
      sr.id = pGII->group.id;
   }
   switch (pLoadRqst->status)
   {
      case LRS_STALLED_SRG_MODULE:
         /* No matching Update found */
         dsmAssert((pCarousel->root.status == RCS_BOOTED));
         sr.id = pCarousel->root.rcid;
         status = SSU_NOT_AVAILABLE;
         break;

      case LRS_STALLED_MODULE:
         /* received DII for Group - requesting modules */
         dsmAssert((pLoadRqst->targetKind == TT_SSU_GROUP));
         if (pCarousel->root.status == RCS_BOOTED)
         {
            pCarousel->root.status = RCS_LOADING;
            if (pGII->group.size != pLoadRqst->remaining)
            {
               ERRLOG(DD_OC, "DATA ERROR: Bad Group size=%u actual=%u", pGII->group.size, pLoadRqst->remaining )
            }
         }
         status = SSU_CRSL_READY;
         data = pLoadRqst->remaining;
         break;

      case LRS_LOADING:
         status = SSU_LOAD_RECEIVING;
         if (NULL != pLoadRqst) {
            data = pLoadRqst->percent;
         }
         break;

      case LRS_LOADED:
         /* received all modules for Group */
         pCarousel->root.status = RCS_LOADED;
         status = SSU_CRSL_DONE;
         break;

      default:
         dsmAssert((0));
      case LRS_ABORTED_TIMEOUT:
      case LRS_ABORTED_PATH_ERROR:
      case LRS_ABORTED_LOAD_ERROR:
      case LRS_ABORTED_BY_REQUESTER:
         pCarousel->root.status = RCS_LOAD_FAILED;
         sr.id = pCarousel->root.rcid;
         status = SSU_CRSL_ABORT;
         break;
   }
   if (idp->setup.ssuFuncs.status)
   {
      idp->setup.ssuFuncs.status( (H_DsmCarousel)pCarousel, status, sr, data );
   }
}

