/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Dsmcc's Section Filter Manager API
 * @file    dsfm.h
 * @date    16 October 2013
 * @author  Adam Sturtridge
 */
#ifndef _DSFM_H_
#define _DSFM_H_

#include "cldsmtypes.h"
#include "dmxtypes.h"
#include "stdfuncs.h"
#include "stb_filter.h"
 #include "dsm_debug.h"

#define SECTION_HEADER_LENGTH 8

typedef enum
{
   SFM_ERROR,
   SFM_IGNORE,
   SFM_UPDATE_CACHE,
   SFM_UPDATE_LOW,
   SFM_UPDATE_HIGH
} E_SFM_STATUS;

typedef struct s_DmxPidFilter *H_DmxPidFilter;

/*------------------- Functions supplied to SFM at setup ---------------------*/

/**
 * @brief   Report to SFM client that there is a filter update to be processed.
 *          The client must place the request on a queue that will subsequently
 *          cause SFM_FilterUpdate to be called.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   H_DmxPidFilter hPF            Handle to SFM pid filter
 * @return  void
 */
typedef void (*F_UpdateFilter)( H_SfmInstance sfm, H_DmxPidFilter hPF );

/**
 * @brief   Report to SFM client that there is an update to be processed. This
 *          is called in the normal DSM-CC thread environment, and is due to
 *          SFM finding section data in cache. The client may place the request
 *          on a queue that subsequently causes SFM_ProcessSection to be called.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   U8BIT*         pSection       Pointer to section data buffer
 * @param   void*          hBuffer        Section buffer handle
 * @param   E_SFM_STATUS   status         Status of update, either:
 *                                        SFM_UPDATE_LOW or SFM_UPDATE_HIGH
 * @return  void
 */
typedef void (*F_CacheMatch)( H_SfmInstance sfm, U8BIT *pSection,
                              void *hBuffer, E_SFM_STATUS status );

/*---------------- Setup structure for SFM initialisation -------------------*/

typedef struct
{
   /* ---------- Manditory functions ----------- */

   F_MemAlloc memAlloc;
   F_MemFree memFree;

   F_Locking mutexLock;
   F_Locking mutexUnlock;

   /* Hardware filter callback function that will handle section and lead to
    * calling SFM_RequireSection() followed by SFM_ProcessSection()
    */
   F_FILTER_CALLBACK hwFilterCallback;

   /* Callback function to inform client that filters require updating
    */
   F_UpdateFilter updateFilter;

   /* ---------- Optional functions ----------- */

   /* cacheMatch function provided so that processing of section data matched
    * in cache can happen after DSM-CC has finished processing request for
    * relevant section filter.
    * If 'cacheMatch' is NULL, then SFM_DsmccFilterAdd will directly call
    * SFM_ProcessSection.
    */
   F_CacheMatch cacheMatch;

   /* -------- Optional debug functions --------- */

   /* errPrintf and dbgPrintf functions are only used in a debug build
    * When not supplied, these must be set to NULL
    */
   F_Printf errPrintf;
   F_Printf dbgPrintf;

   /* ------------- Mutex parameters ------------ */

   /* sfmMutex is manditory, used in mutexLock / mutexUnlock functions
    * and must be created and valid before supplying to SFM_CreateInstance
    */
   void *sfmMutex;

   /* cacheMutex and bufferMutex are only required.if caching
    * - see sectionBuffCacheSize
    * It if used in mutexLock / mutexUnlock functions and must be created
    * before supplying to SFM_CreateInstance
    */
   void *cacheMutex;
   void *bufferMutex;

   /* -------- Configuration parameters --------- */

   /* Set maximum number of PID filters available to be used symultaneously
    * Minimum value: 4
    */
   U8BIT maxPidFilters;

   /* Set maximum number of section filters per PID filter used symultaneously.
    * Low values will increase requirement for SFM to do it's own
    * software section filtering.
    * Minimum value: 1
    */
   U8BIT maxSecFiltersPerPid;

   /* Size of section buffer manager cache in 64K byte blocks. This memory is
    * used to store section data not immediately required by DSM-CC instance.
    * This facilitates faster DSM-CC file loading - particularly when the number
    * of section filters is limited. See maxSecFiltersPerPid (and maxPidFilters)
    * Minimum value: 0
    * Suggested value: 16
    */
   U8BIT sectionBuffCacheSize;

   /*
    * Specifies the maximum number of section filters that are available
    * to the DSM-CC instance for simultaneously acquiring DSM-CC private sections.
    * This should be set with the same value as that given to the DSM-CC instance
    * creation - see 'maxAvailableSectionFilters' in DSM-CC setup.
    */
   U16BIT maxAvailableSectionFilters;

   /* dbgState controls type of debug output for dbgPrintf
    * For valid debug filter states see above
    */
   U32BIT dbgState;
} S_SfmSetup;


/*---------------------- Functions that configure SFM ------------------------*/

/**
 * @brief   Create Section Filter Manager instance, using setup structure.
 * @param   S_SfmSetup     setup          setup parameters
 * @return  SFM instance. NULL is failure.
 */
H_SfmInstance SFM_CreateInstance( S_SfmSetup *pSetup );

/**
 * @brief   Destroy Section Filter Manager instance, and return mutexes
 *          so that they may be destroyed by client.
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   void**         pSfmMtx     Pointer to sfmMutex provided in setup
 * @param   void**         pCchMtx     Pointer to cacheMutex provided in setup
 * @param   void**         pBufMtx     Pointer to bufferMutex provided in setup
 * @return void
 */
void SFM_DestroyInstance( H_SfmInstance sfm, void **pSfmMtx,
   void **pCchMtx, void **pBufMtx );

/**
 * @brief   This allows controlling software to reconfigure SFM debug output.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   F_SfmPrintf    errPrintf      Error print function
 * @param   F_SfmPrintf    dbgPrintf      Debug print function
 * @param   U32BIT         dbgState       State to filter debug printing
 * @return
 */
void SFM_SetDebugConfig( H_SfmInstance sfm, F_Printf errPrintf,
   F_Printf dbgPrintf, U32BIT dbgState );

/**
 * @brief   This allows controlling software to reconfigure SFM debug output.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   U32BIT         dbgState       State to filter debug printing
 * @return
 */
void SFM_SetDebugState( H_SfmInstance sfm, U32BIT dbgState );

/**
 * @brief   Set instance handle for DSM-CC that SFM is supporting.
 * @param   H_SfmInstance       sfm       SFM instance handle.
 * @param   H_DsmCoreInst       dsmcc     DSM-CC instance being supported
 * @return
 */
void SFM_SetDsmInstance( H_SfmInstance sfm, H_DsmCoreInst dsmcc );

/**
 * @brief   Set handle to be passed to F_DvpFilterSetup and F_DvpFilterRemove
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   DMXREF       demux          Demux reference handle
 * @return
 */
void SFM_SetDemuxHandle( H_SfmInstance sfm, DMXREF demux );



/*----------------- Functions that process section data  ---------------------*/

/**
 * @brief   This function performs minimal checking of section header data to
 *          find out whether SFM requires this section.
 *          Number of header bytes required is SECTION_HEADER_LENGTH.
 *          It assumes CRC has been checked and is passing valid DSM-CC sections
 *          So the first byte of data must be table id 0x3B, 0x3C or 0x3D.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_DsmccFilterAlloc and SFM_DsmccFilterDelete.
 *          It does NOT call any DSM-CC core functions.
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   PIDFILT    pfid        PID filter with incoming section
 * @param   U8BIT*         pHeader     Pointer to section header (min 8 bytes)
 * @param   void**         phBuffer    Pointer to SFM section buffer handle
 * @return  E_SFM_STATUS - status of required update
 */
E_SFM_STATUS SFM_RequireSection( H_SfmInstance sfm, PIDFILT pfid,
   U8BIT *pHeader, void **phBuffer );

/**
 * @brief   Get section data buffer pointer in SFM's cache for the handle.
 *          This should only be called after SFM_RequireSection returned with
 *          SFM_UPDATE_CACHE. This may be called from 'interrupt' environment.
 *          The returned pointer may be used by client to copy section data
 *          into SFM's cache buffer.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   void*          hBuffer        Section buffer handle returned
 *                                        by SFM_RequireSection
 * @return  U8BIT*         Pointer to buffer to store section data
 */
U8BIT* SFM_CacheBuffer( H_SfmInstance sfm, void *hBuffer );

/**
 * @brief   Release Cache buffer allocated by SFM_CacheBuffer
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   U8BIT*         pSection    Pointer to whole section data buffer
 * @param   void*          hBuffer        Section buffer handle returned
 *                                        by SFM_RequireSection
 * @return  void.
 */
void SFM_CacheReleaseBuffer( H_SfmInstance sfm, U8BIT *pSection, void *hBuffer );

/**
 * @brief   Process required section buffer.
 *          This should only be called after SFM_RequireSection has returned
 *          SFM_UPDATE_HIGH or SFM_UPDATE_LOW.
 *          This function must be called in the normal DSM-CC thread
 *          environment, because it calls DSM-CC core functions.
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   U8BIT*         pSection    Pointer to whole section data buffer
 * @param   void*          hBuffer     SFM section handle returned by
 *                                     SFM_RequireSection
 * @return  void.
 */
void SFM_ProcessSection( H_SfmInstance sfm, U8BIT *pSection, void *hBuffer );

/**
 * @brief   Update SFM PID filter to configure HW for latest requirements
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   H_DmxPidFilter hPF            Handle to SFM pid filter
 * @return  void.
 */
void SFM_FilterUpdate( H_SfmInstance sfm, H_DmxPidFilter hPF );

/*-------------------- Functions to manage SFM caching -----------------------*/

/**
 * @brief   Enables Full Sotfware Section Filtering using SFM cache.
 *          This will only be successfull if sufficient resources are available
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  BOOLEAN        TRUE on success
 */
BOOLEAN SFM_CacheEnableSSF( H_SfmInstance sfm );

/**
 * @brief   Disables Sotfware Section Filtering
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  void
 */
void SFM_CacheDisableSSF( H_SfmInstance sfm );

/**
 * @brief   Clear all SFM cached section data. Not required to be called, if
 *          setup had sectionBuffCacheSize set to ZERO. Otherwise, this MUST be
 *          called on a transport stream change and optionally called at other
 *          times - e.g on a service change.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  void.
 */
void SFM_CacheClear( H_SfmInstance sfm );



/*----------------- Functions provided to DSM-CC component -------------------*/

/**
 * @brief   Add DSM-CC section filter.
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_RequireSection.
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   P_SecFilter   pFilter        Pointer to DSM-CC filter details
 * @param   H_DscSFRef      dsmSfRef       DSM-CC SF reference handle
 * @return  void*                            SFM Filter handle
 */
void* SFM_DsmccFilterAdd( H_SfmInstance sfm, P_SecFilter pFilter,
   H_DscSFRef dsmSfRef );

/**
 * @brief   Delete DSM-CC section filter.
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_RequireSection.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   void*          filterHandle   SFM Filter handle
 * @param   H_DscSFRef   dsmSfRef   DSM-CC reference handle
 * @return  void
 */
void SFM_DsmccFilterDelete( H_SfmInstance sfm, void *filterHandle,
   H_DscSFRef dsmSfRef );


/**
 * @brief   Change DSM-CC section filter priority
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_RequireSection.
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   void*             filterHandle   SFM Filter handle
 * @param   H_DscSFRef      dsmSfRef       DSM-CC SF reference handle
 * @param   E_SFPriority priority       New filter priority
 * @return  void
 */
void SFM_DsmccFilterPriorityChange( H_SfmInstance sfm, void *filterHandle,
   H_DscSFRef dsmSfRef,
   E_SFPriority priority );

#endif /*_DSFM_H_*/
