/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Section Buffer Manager (for use by Section Filter Manager and client)
 *          Provides mechanism to allocate and release buffers for section data
 *          without cost of using standard malloc and free.
 * @file    sbm.h
 * @date    30th October 2013
 * @author  Adam Sturtridge
 */
#ifndef _DSBM_H_
#define _DSBM_H_

#include "techtype.h"
#include "stdfuncs.h"

/*------------------- Functions supplied to SBM at setup ---------------------*/


/*---------------- Setup structure for SBM initialisation -------------------*/

typedef struct
{
   // Manditory functions - used in creation, destruction of instance

   F_MemAlloc memAlloc;
   F_MemFree memFree;

   F_Locking lock;
   F_Locking unlock;

   // mutex or binary semaphore
   void *mtx_sem;

   // Size of memory in 64K byte blocks to be managed by SBM.
   // Minimum value: 0
   // Suggested value: 16
   U16BIT bufferSize;
} S_SbmSetup;

// SBM instance handle definition
typedef struct s_SbmInstance *H_SbmInstance;


/*---------------------- Functions that configure SBM ------------------------*/

/**
 * @brief   Create Section Buffer Manager instance, using setup structure.
 * @param   S_SbmSetup     setup          setup parameters
 * @return  SBM instance.  NULL is failure.
 */
H_SbmInstance SBM_CreateInstance( S_SbmSetup *pSetup );

/**
 * @brief   Destroy Section Filter Manager instance, and return mutexes
 *          so that they may be destroyed by client.
 * @param   H_SbmInstance  sbm         SBM instance handle.
 * @param   void**         pBufMtx     Pointer to mtx_sem provided in setup
 * @return void
 */
void SBM_DestroyInstance( H_SbmInstance sbm, void **pBufMtx );

/*--------------- Allocate / release section buffer functions ----------------*/

/**
 * @brief   Allocate buffer for a DVB section
 * @param   H_SbmInstance  sbm      SBM instance handle.
 * @param   U8BIT          size1    First 'size' byte in DVB section data
 *                                  That is value to be assigned to  buffer[1].
 *                                  Or section size (less 3) in 256 byte units.
 * @return  U8BIT* pointer to section buffer, NULL is failure.
 */
U8BIT* SBM_AllocateBuffer( H_SbmInstance sbm, U8BIT size1 );


/**
 * @brief   Release DVB section buffer allocated with SBM_AllocateBuffer
 * @param   H_SbmInstance  sbm      SBM instance handle.
 * @param   U8BIT*         buffer   Pointer to DVB section data buffer
 * @return  void
 */
void SBM_ReleaseBuffer( H_SbmInstance sbm, U8BIT *buffer );

#endif /*_DSBM_H_*/
