/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Section Buffer Manager (for use by Section Filter Manager and client)
 *          Provides mechanism to allocate and release buffers for section data
 *          without cost of using standard malloc and free.
 * @file    sbm.c
 * @date    30th October 2013
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

#include "stb_debug.h"

#include "sbm.h"

/*---- Constant definitions for this file-------------------------------------*/

#define INVALID_OFFSET  0xFFFFFFFF

/*---- Local typedef structs for this file------------------------------------*/

typedef struct s_SbmInstance
{
   F_MemFree memFree;
   F_Locking lock;
   F_Locking unlock;
   void *mtx_sem;
   U32BIT totalBuffs;
   U8BIT *buffers;
   U32BIT *control;
} S_SbmInstance;


/*---- Local (static) variable declarations for this file---------------------*/
/*   (internal variables declared static to make them local) */

/*---- LOCAL function definitions for this file ------------------------------*/
/*   (internal functions declared static to make them local) */


static U32BIT FindFreeBuff1( U32BIT *control, U32BIT total )
{
   U32BIT cbit, ndx, i;
   for (ndx = 0; ndx != total; ndx++, control++)
   {
      for (cbit = 0x80000000, i = 31; cbit; cbit >>= 1, i--)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   return INVALID_OFFSET;
}

static U32BIT FindFreeBuff2( U32BIT *control, U32BIT total )
{
   U32BIT cbit, ndx, i, bndl, bndh;
   bndl = (total >> 4);  // 1/16 of total
   bndh = (total >> 3);  // 1/8 of total
   control += bndl;
   for (ndx = bndl; ndx != bndh; ndx++, control++)
   {
      for (cbit = 0x03, i = 0; cbit; cbit <<= 2, i += 2)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   control -= bndh - bndl;
   while (bndl--)
   {
      control--;
      for (cbit = 0xC0000000, i = 30; cbit; cbit >>= 2, i -= 2)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (bndl << 5) + i;
         }
      }
   }
   control += bndh;
   for (ndx = bndh; ndx != total; ndx++, control++)
   {
      for (cbit = 0xC0000000, i = 30; cbit; cbit >>= 2, i -= 2)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   return INVALID_OFFSET;
}

static U32BIT FindFreeBuff3( U32BIT *control, U32BIT total )
{
   U32BIT cbit, ndx, i, bndl, bndh;
   bndl = (total >> 3);  // 1/8 of total
   bndh = (total >> 2);  // 1/4 of total
   control += bndl;
   for (ndx = bndl; ndx != bndh; ndx++, control++)
   {
      for (cbit = 0x38000000, i = 27; cbit; cbit >>= 3, i -= 3)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   control -= bndh - bndl;
   while (bndl--)
   {
      control--;
      for (cbit = 0x38000000, i = 27; cbit; cbit >>= 3, i -= 3)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (bndl << 5) + i;
         }
      }
   }
   control += bndh;
   for (ndx = bndh; ndx != total; ndx++, control++)
   {
      for (cbit = 0x38000000, i = 27; cbit; cbit >>= 3, i -= 3)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   return INVALID_OFFSET;
}

static U32BIT FindFreeBuff4( U32BIT *control, U32BIT total )
{
   U32BIT cbit, ndx, i, bndl, bndh;
   bndl = (total >> 2);   // 1/4 of total
   bndh = (total >> 1);   // 1/2 of total
   control += bndl;
   for (ndx = bndl; ndx != bndh; ndx++, control++)
   {
      for (cbit = 0xF0000000, i = 28; cbit; cbit >>= 4, i -= 4)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   control -= bndh - bndl;
   while (bndl--)
   {
      control--;
      for (cbit = 0xF0000000, i = 28; cbit; cbit >>= 4, i -= 4)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (bndl << 5) + i;
         }
      }
   }
   control += bndh;
   for (ndx = bndh; ndx != total; ndx++, control++)
   {
      for (cbit = 0xF, i = 0; cbit; cbit <<= 4, i += 4)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   return INVALID_OFFSET;
}

static U32BIT FindFreeBuff5( U32BIT *control, U32BIT total )
{
   U32BIT cbit, ndx, i, bndl, bndh;
   bndl = (total >> 2) + (total >> 3);  // 3/8 of total
   bndh = (total >> 1) + (total >> 3);  // 5/8 of total
   control += bndl;
   for (ndx = bndl; ndx != bndh; ndx++, control++)
   {
      for (cbit = 0x3E000000, i = 25; cbit; cbit >>= 5, i -= 5)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   control -= bndh - bndl;
   while (bndl--)
   {
      control--;
      for (cbit = 0x3E000000, i = 25; cbit; cbit >>= 5, i -= 5)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (bndl << 5) + i;
         }
      }
   }
   control += bndh;
   for (ndx = bndh; ndx != total; ndx++, control++)
   {
      for (cbit = 0x3E000000, i = 25; cbit; cbit >>= 5, i -= 5)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   return INVALID_OFFSET;
}

static U32BIT FindFreeBuff6( U32BIT *control, U32BIT total )
{
   U32BIT cbit, ndx, i, bndl, bndh;
   bndl = (total >> 1);               // 1/2 of total
   bndh = (total >> 1) + (total >> 2);  // 3/4 of total
   control += bndl;
   for (ndx = bndl; ndx != bndh; ndx++, control++)
   {
      for (cbit = 0x3F000000, i = 24; cbit; cbit >>= 6, i -= 6)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   control -= bndh - bndl;
   while (bndl--)
   {
      control--;
      for (cbit = 0x3F000000, i = 24; cbit; cbit >>= 6, i -= 6)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (bndl << 5) + i;
         }
      }
   }
   control += bndh;
   for (ndx = bndh; ndx != total; ndx++, control++)
   {
      for (cbit = 0xFC, i = 2; cbit; cbit <<= 6, i += 6)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   return INVALID_OFFSET;
}

static U32BIT FindFreeBuff7( U32BIT *control, U32BIT total )
{
   U32BIT cbit, ndx, i, bndl;
   bndl = (total >> 1) + (total >> 3);  // 5/8 of total
   control += bndl;
   for (ndx = bndl; ndx != total; ndx++, control++)
   {
      for (cbit = 0xFE00000, i = 21; cbit; cbit >>= 7, i -= 7)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (ndx << 5) + i;
         }
      }
   }
   control -= total - bndl;
   while (bndl--)
   {
      control--;
      for (cbit = 0x7F0, i = 4; cbit; cbit <<= 7, i += 7)
      {
         if ((cbit & *control) == 0)
         {
            *control |= cbit;
            return (bndl << 5) + i;
         }
      }
   }
   return INVALID_OFFSET;
}

static U32BIT FindFreeBuff8( U32BIT *control, U32BIT total )
{
   U32BIT cbit, i;
   control += total;
   while (total--)
   {
      control--;
      if (*control != 0xFFFFFFFF)
      {
         for (cbit = 0xFF, i = 0; cbit; cbit <<= 8, i += 8)
         {
            if ((cbit & *control) == 0)
            {
               *control |= cbit;
               return (total << 5) + i;
            }
         }
      }
   }
   return INVALID_OFFSET;
}

/*---- GLOBAL function definitions for this file -----------------------------*/


/*--------------- Functions supply / release section data --------------------*/



/**
 * @brief   Allocate buffer for a DVB section
 * @param   H_SbmInstance  sbm      SBM instance handle.
 * @param   U8BIT          size1    First 'size' byte in DVB section data
 *                                  That is 'section_data_ptr[1]' or
 *                                  section size (less 3) in 256 byte units.
 * @return  U8BIT* pointer to section buffer, NULL is failure.
 */
U8BIT* SBM_AllocateBuffer( H_SbmInstance sbm, U8BIT size1 )
{
   U8BIT *pBuff;
   U32BIT offset;
   U8BIT sz;

   sz = (size1 >> 1) & 0x7;
   sbm->lock(sbm->mtx_sem);
   switch (sz)
   {
      case 0: // < 512
         offset = FindFreeBuff1( sbm->control, sbm->totalBuffs );
         break;

      case 1: // < 1024
         offset = FindFreeBuff2( sbm->control, sbm->totalBuffs );
         break;

      case 2:
         offset = FindFreeBuff3( sbm->control, sbm->totalBuffs );
         break;

      case 3: // < 2047
         offset = FindFreeBuff4( sbm->control, sbm->totalBuffs );
         break;

      case 4:
         offset = FindFreeBuff5( sbm->control, sbm->totalBuffs );
         break;

      case 5:
         offset = FindFreeBuff6( sbm->control, sbm->totalBuffs );
         break;

      case 6:
         offset = FindFreeBuff7( sbm->control, sbm->totalBuffs );
         break;

      case 7: // < 4096
         offset = FindFreeBuff8( sbm->control, sbm->totalBuffs );
   }
   sbm->unlock(sbm->mtx_sem);
   if (offset == INVALID_OFFSET ||
       offset >= (sbm->totalBuffs << 5))   // this second condition should not happen
   {
      pBuff = NULL;
   }
   else
   {
      // scale up offset to block size of 512 bytes
      pBuff = sbm->buffers + (offset << 9);
      // place size1 byte in section length location (in case not done by client)
      pBuff[1] = size1;
   }
   return pBuff;
}

/**
 * @brief   Release DVB section buffer allocated with SBM_AllocateBuffer
 * @param   H_SbmInstance  sbm      SBM instance handle.
 * @param   U8BIT*         buffer   Pointer to DVB section data buffer
 * @return  void
 */
void SBM_ReleaseBuffer( H_SbmInstance sbm, U8BIT *buffer )
{
   U32BIT offset, mask, shft, ndx;
   U16BIT size;

   offset = (buffer - sbm->buffers) >> 9;
   size = (buffer[1] & 0x0F) >> 1;
   ndx = offset >> 5;
   shft = offset & 0x1f;

   if (ndx < sbm->totalBuffs &&
       (shft + size) < 32)
   {
      mask = (1 << (size + 1)) - 1;
      sbm->lock(sbm->mtx_sem);
      sbm->control[ndx] &= ~(mask << shft);
      sbm->unlock(sbm->mtx_sem);
   }
}

/**
 * @brief   Create Section Buffer Manager instance, using setup structure.
 * @param   S_SbmSetup     setup          setup parameters
 * @return  SBM instance.  NULL is failure.
 */
H_SbmInstance SBM_CreateInstance( S_SbmSetup *pSetup )
{
   S_SbmInstance *sbm = NULL;
   U32BIT totalBuffs, csize;
   U8BIT *buffs;

   if (pSetup != NULL)
   {
      if (pSetup->memAlloc != NULL &&
          pSetup->memFree != NULL &&
          pSetup->lock != NULL &&
          pSetup->unlock != NULL &&
          pSetup->mtx_sem != NULL &&
          pSetup->bufferSize != 0)
      {
         buffs = NULL;
         totalBuffs = pSetup->bufferSize;
         while (totalBuffs)
         {
            buffs = pSetup->memAlloc( totalBuffs << 16 );
            if (buffs != NULL)
            {
               break;
            }
            totalBuffs--;
         }
         if (buffs != NULL)
         {
            // scale totalBuffs to number of 16K blocks (4 * 4096)
            totalBuffs <<= 2;
            csize = totalBuffs * sizeof(U32BIT);
            sbm = pSetup->memAlloc( sizeof(S_SbmInstance) + csize );
            if (sbm == NULL)
            {
               pSetup->memFree( buffs );
            }
            else
            {
               sbm->memFree = pSetup->memFree;
               sbm->lock = pSetup->lock;
               sbm->unlock = pSetup->unlock;
               sbm->mtx_sem = pSetup->mtx_sem;
               sbm->totalBuffs = totalBuffs;
               sbm->buffers = buffs;
               sbm->control = (U32BIT *)(sbm + 1);
               memset(sbm->control, 0, csize);
            }
         }
      }
   }
   return sbm;
}

/**
 * @brief   Destroy Section Filter Manager instance, and return mutexes
 *          so that they may be destroyed by client.
 * @param   H_SbmInstance  sbm         SBM instance handle.
 * @param   void**         pBufMtx     Pointer to mtx_sem provided in setup
 * @return void
 */
void SBM_DestroyInstance( H_SbmInstance sbm, void **pBufMtx )
{
   if (sbm)
   {
   #ifndef NDEBUG
      U32BIT *control;
      U32BIT count;
      control = sbm->control;
      sbm->lock(sbm->mtx_sem);
      for (count = 0; count != sbm->totalBuffs; count++, control++)
      {
         if (*control != 0)
         {
            /* some data buffer(s) not freed! */
            STB_SPDebugWrite("SBM ERROR: index=%d ctrl=%08x", count, *control);
         }
      }
      sbm->unlock(sbm->mtx_sem);
   #endif
      if (pBufMtx != NULL)
      {
         *pBufMtx = sbm->mtx_sem;
      }
      sbm->memFree( sbm->buffers );
      sbm->memFree( sbm );
   }
}

