/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Section Filter Manager: caching functions
 * @file    sfm_cache.c
 * @date    19th October 2013
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

#include "sfm_main.h"
#include "sfm_cache.h"
#include "sfm_filter.h"
#include "sbm.h"

/*---- Constant definitions for this file-------------------------------------*/

#define TABLE_BITS      6
#define TABLE_SIZE      (1 << TABLE_BITS)
#define NUM_FILTERS     64
#define NUM_NODES       128
#define INVALID_VERSION 0xFF
#define STALE_VERSION   0xFE
#define STALE_BIT       0x80
#define REPORTING       0x40

#define ALLOCSIZE(s)    ((s >> 1) & 7) + 1

/*---- Local typedef structs for this file------------------------------------*/

typedef struct s_NodeBlock *H_NodeBlock;
typedef struct s_FiltBlock *H_FiltBlock;

typedef struct s_CacheNode
{
   H_CacheNode next;       // next node
   U8BIT *sbuf;            // section data pointer
   U8BIT bknum;            // block number
   U8BIT report;
} S_CacheNode;

typedef struct s_CacheFilter
{
   H_CacheNode node;       // first node
   H_CacheFilter next;     // next filter
   U16BIT tide;            // Table ID Extension
   U16BIT dsmref;          // index into H_DsmResource array (from dmxResFirst)
   U8BIT vers;             // current table version
   U8BIT ctndx;            // Index into cache table array
} S_CacheFilter;


typedef struct s_CacheTable
{
   H_CacheFilter filters[TABLE_SIZE];
   U8BIT changed[TABLE_SIZE];
} S_CacheTable;

typedef struct s_NodeBlock
{
   H_NodeBlock next;
} S_NodeBlock;

typedef struct s_FiltBlock
{
   H_FiltBlock next;
} S_FiltBlock;


typedef struct s_SfmCache
{
   H_SfmInstance sfm;
   H_SbmInstance sbm;
   H_CacheTable ctables;
   H_FiltBlock memFilters;
   H_NodeBlock memNodes;
   H_CacheFilter filtHead;
   H_CacheFilter filtTail;
   H_CacheNode nodeHead;
   H_CacheNode nodeTail;
   U32BIT counter;
#ifndef NDEBUG
   U32BIT allocated;
#endif
} S_SfmCache;


/*---- Local (static) variable declarations for this file---------------------*/
/*   (internal variables declared static to make them local) */

/*---- LOCAL function definitions for this file ------------------------------*/
/*   (internal functions declared static to make them local) */


/**
 * @brief   Return hash index from key value.
 * @param   U32BIT   key      Seed key - 32 bit value.
 * @param   U16BIT   bits     Number of bits for index
 * @return  U16BIT.
 */
static U16BIT HashIndex(U32BIT key, U16BIT bits)
{
   key = (key << 15) - key - 1;
   key = key ^ (key >> 12);
   key = key + (key << 2);
   key = key ^ (key >> 4);
   key = (key + (key << 3)) + (key << 11);
   key = key ^ (key >> 16);
   return key & ((1 << bits) - 1);
}

static BOOLEAN AllocFiltBlock( H_SfmCache cache )
{
   H_FiltBlock memFilters;
   H_CacheFilter filter;
   U32BIT size;
   BOOLEAN result;
   H_SfmInstance sfm;
   sfm = cache->sfm;
   size = sizeof(S_FiltBlock) + (NUM_FILTERS * sizeof(S_CacheFilter));
   memFilters = sfm->setup.memAlloc( size );
   if (memFilters == NULL)
   {
      result = FALSE;
   }
   else
   {
      memset(memFilters, 0, size);
      filter = (H_CacheFilter)(memFilters + 1);
      if (cache->filtHead == NULL)
      {
         cache->filtHead = filter;
      }
      else
      {
         ASSERT( cache->filtTail != NULL )
         cache->filtTail->next = filter;
      }
      for (size = NUM_FILTERS - 1; size--; filter++)
      {
         filter->next = (filter + 1);
      }
      filter->next = NULL;
      cache->filtTail = filter;
      memFilters->next = cache->memFilters;
      cache->memFilters = memFilters;
      result = TRUE;
   }
   return result;
}

static void FreeFiltBlocks( H_SfmInstance sfm, H_FiltBlock memFilters )
{
   H_FiltBlock memNext;
   while (memFilters != NULL)
   {
      memNext = memFilters->next;
      sfm->setup.memFree( memFilters );
      memFilters = memNext;
   }
}

static BOOLEAN AllocNodeBlock( H_SfmCache cache )
{
   H_NodeBlock memNodes;
   H_CacheNode node;
   U32BIT size;
   BOOLEAN result;
   H_SfmInstance sfm;
   sfm = cache->sfm;
   size = sizeof(S_NodeBlock) + (NUM_NODES * sizeof(S_CacheNode));
   memNodes = sfm->setup.memAlloc( size );
   if (memNodes == NULL)
   {
      result = FALSE;
   }
   else
   {
      memset(memNodes, 0, size);
      node = (H_CacheNode)(memNodes + 1);
      if (cache->nodeHead == NULL)
      {
         cache->nodeHead = node;
      }
      else
      {
         ASSERT( cache->nodeTail != NULL )
         cache->nodeTail->next = node;
      }
      for (size = NUM_NODES - 1; size--; node++)
      {
         node->next = (node + 1);
      }
      node->next = NULL;
      cache->nodeTail = node;
      memNodes->next = cache->memNodes;
      cache->memNodes = memNodes;
      result = TRUE;
   }
   return result;
}

static void FreeNodeBlocks( H_SfmInstance sfm, H_NodeBlock memNodes )
{
   H_NodeBlock memNext;
   while (memNodes != NULL)
   {
      memNext = memNodes->next;
      sfm->setup.memFree( memNodes );
      memNodes = memNext;
   }
}

static H_CacheNode AddCacheNode( H_SfmCache cache, H_CacheFilter cfilter, U8BIT *buff )
{
   H_CacheNode cnode;
   // lock mutex
   if (cache->nodeHead == NULL && !AllocNodeBlock( cache ))
   {
      cnode = NULL;
   }
   else
   {
      cnode = cache->nodeHead;
      cache->nodeHead = cache->nodeHead->next;
      cnode->sbuf = buff;
      cnode->next = cfilter->node;
      cfilter->node = cnode;
   }
   // unlock mutex
   return cnode;
}

static void FreeCacheFilt( H_SfmCache cache, H_CacheFilter filter )
{
   filter->vers = INVALID_VERSION;
   if (cache->filtHead == NULL)
   {
      cache->filtHead = filter;
   }
   else
   {
      ASSERT( cache->filtTail != NULL )
      cache->filtTail->next = filter;
   }
   filter->next = NULL;
   cache->filtTail = filter;
}

static void FreeCacheNode( H_SfmCache cache, H_CacheNode node )
{
#ifndef NDEBUG
   U32BIT size = ALLOCSIZE(node->sbuf[1]);
   if (cache->allocated >= size)
   {
      cache->allocated -= size;
   }
#endif
   SBM_ReleaseBuffer( cache->sbm, node->sbuf );
   if (cache->nodeHead == NULL)
   {
      cache->nodeHead = node;
   }
   else
   {
      ASSERT( cache->nodeTail != NULL )
      cache->nodeTail->next = node;
   }
   node->next = NULL;
   cache->nodeTail = node;
}

/**
 * @brief
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   void*          hCache         Handle to cache buffer given to the
 *                                        F_CacheMatch callback function
 * @return  BOOLEAN
 */
static BOOLEAN CacheValidNode( H_SfmCache cache, void *ptr )
{
   H_NodeBlock memNodes;
   PU8BIT pval, mval;
   U32BIT size;
   BOOLEAN result;

   pval = (PU8BIT)ptr;
   result = FALSE;
   size = sizeof(S_NodeBlock) + (NUM_NODES * sizeof(S_CacheNode));
   memNodes = cache->memNodes;
   while (memNodes != NULL)
   {
      mval = (PU8BIT)memNodes;
      if (pval > mval && pval < mval + size)
      {
         result = TRUE;
         break;
      }
      memNodes = memNodes->next;
   }
   return result;
}

static void CacheClearNodes( H_SfmCache cache, H_CacheFilter cfilter )
{
   H_CacheNode cnode, cnext;
   if (cfilter->node != NULL)
   {
      cnode = cfilter->node;
      do
      {
         cnext = cnode->next;
         FreeCacheNode( cache, cnode );
         cnode = cnext;
      }
      while (cnode != NULL);

      cfilter->node = NULL;
   }
}

static void CacheClearFilters( H_SfmCache cache, H_CacheTable ctable )
{
   H_CacheFilter cfilter;
   U16BIT ndx;
   // lock mutex ?
   for (ndx = 0; ndx != TABLE_SIZE; ndx++)
   {
      cfilter = ctable->filters[ndx];
      while (cfilter != NULL)
      {
         CacheClearNodes( cache, cfilter );
         cfilter = cfilter->next;
      }
   }
}

static void CtablePurgeExpired( H_SfmCache cache, H_CacheTable ctable )
{
   H_CacheFilter cfilter, *pCfilter;
   U16BIT ndx;

   // lock mutex ?
   for (ndx = TABLE_SIZE; ndx--; )
   {
      if (ctable->changed[ndx] & STALE_BIT)
      {
         ctable->changed[ndx] &= ~(STALE_BIT);
         pCfilter = &(ctable->filters[ndx]);
         while (*pCfilter != NULL)
         {
            cfilter = *pCfilter;
            if (cfilter->vers == STALE_VERSION)
            {
               CacheClearNodes( cache, cfilter );
               *pCfilter = cfilter->next;
               FreeCacheFilt( cache, cfilter );
            }
            else
            {
               pCfilter = &(cfilter->next);
            }
         }
      }
   }
}

static void CachePurgeExpired( H_SfmInstance sfm )
{
   H_SfmCache cache;
   H_CacheTable ctable;
   U16BIT ndx;

   // Purge cache of stale data
   cache = sfm->cache;
   ctable = cache->ctables;
   // lock mutex ?
   DBGLOG(DF_CACHE, "Purging stale nodes")
   for (ndx = (sfm->setup.maxPidFilters << 1); ndx--; ctable++)
   {
      CtablePurgeExpired( cache, ctable );
   }
   // TODO: Purge cache of old unused data, but NOT nodes that are being reported to dsmcc
   // (i.e. cnode->report != REPORTING)
}

/**
 * @brief   Search cache for section data
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   H_CacheTable   ctable         Cache table handle
 * @param   S_TableExt     tex            Table Extension ID and mask
 * @return  void.
 */
static void ReportSections( H_SfmInstance sfm, H_CacheFilter cfilter,
   E_SFM_STATUS status )
{
   H_CacheNode cnode;
   cnode = cfilter->node;
   while (cnode != NULL)
   {
      cnode->report = REPORTING;
      DBGLOG(DF_CACHE, "CACHE HIT: sz=0x%x t=0x%x e=0x%x, %d, %d status=%d",
         (((int)(cnode->sbuf[1] & 0x0f) << 8) | cnode->sbuf[2]) + 3, *cnode->sbuf,
         ((cnode->sbuf[3] << 8) | cnode->sbuf[4]), cnode->sbuf[6] + 1, cnode->sbuf[7] + 1, status)
      sfm->setup.cacheMatch( sfm, cnode->sbuf, cfilter, status );
      cnode = cnode->next;
   }
}

/**
 * @brief   Find Cache filter node
 * @param   H_CacheTable   ctable         Cache table handle
 * @param   U16BIT         tide           Table Extension ID
 * @return  void.
 */
static H_CacheFilter FindFilter( H_SfmInstance sfm, H_CacheTable ctable, U16BIT tide )
{
   H_CacheFilter cfilter;
   U16BIT index;
   index = HashIndex(tide, TABLE_BITS);
   // lock mutex ?
   cfilter = ctable->filters[index];
   while (cfilter != NULL)
   {
      if (cfilter->tide == tide &&
          cfilter->vers != STALE_VERSION)
      {
         break;
      }
      cfilter = cfilter->next;
   }
   return cfilter;
}

static void CacheMatch( H_SfmInstance sfm, U8BIT *pSection,
   void *hBuffer, E_SFM_STATUS status )
{
   FUNCTION_START(CacheMatch)
   SFMCacheProcessSection( sfm, pSection, hBuffer );
   FUNCTION_FINISH(CacheMatch)
}

/*---- GLOBAL function definitions for this file -----------------------------*/

H_SfmCache SFMCacheCreate( H_SfmInstance sfm )
{
   H_SfmCache cache;
   U32BIT size;
   H_SbmInstance sbm;
   S_SbmSetup setup;

   setup.memAlloc = sfm->setup.memAlloc;
   setup.memFree = sfm->setup.memFree;
   setup.lock = sfm->setup.mutexLock;
   setup.unlock = sfm->setup.mutexUnlock;
   setup.mtx_sem = sfm->setup.bufferMutex;
   setup.bufferSize = sfm->setup.sectionBuffCacheSize;
   sbm = SBM_CreateInstance( &setup );
   if (sbm == NULL)
   {
      ERRLOG("mem fail")
      sfm->setup.sectionBuffCacheSize = 0;
      cache = NULL;
   }
   else
   {
      size = sizeof(S_SfmCache) +
         (sfm->setup.maxPidFilters * sizeof(S_CacheTable) * 2);
      cache = sfm->setup.memAlloc( size );
      if (cache == NULL)
      {
         ERRLOG("mem fail")
         sfm->setup.sectionBuffCacheSize = 0;
         SBM_DestroyInstance( sbm, &setup.mtx_sem );
      }
      else
      {
         memset(cache, 0, size);
         cache->sfm = sfm;
         cache->sbm = sbm;
         if (AllocNodeBlock( cache ))
         {
            cache->ctables = (H_CacheTable)(cache + 1);
            if (sfm->setup.cacheMatch == NULL)
            {
               sfm->setup.cacheMatch = CacheMatch;
            }
            cache->counter = sfm->setup.maxPidFilters * 2;
         #ifndef NDEBUG
            cache->allocated = 0;
         #endif
         }
         else
         {
            ERRLOG("mem fail")
            sfm->setup.memFree( cache );
            SBM_DestroyInstance( sbm, &setup.mtx_sem );
            cache = NULL;
         }
      }
   }
   return cache;
}

void SFMCacheDestroy( H_SfmInstance sfm, void **pBufMtx )
{
   H_SfmCache cache;
   SFM_CacheClear( sfm );
   if (sfm->cache)
   {
      cache = sfm->cache;
      FreeNodeBlocks( sfm, cache->memNodes );
      FreeFiltBlocks( sfm, cache->memFilters );
      SBM_DestroyInstance( cache->sbm, pBufMtx );
      sfm->setup.memFree( cache );
      sfm->cache = NULL;
   }
}

/**
 * @brief   Clear all SFM cached section data.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  void.
 */
void SFM_CacheClear( H_SfmInstance sfm )
{
   H_SfmCache cache;
   H_CacheTable ctable;
   U16BIT ndx;

   cache = sfm->cache;
   if (cache != NULL)
   {
      ctable = cache->ctables;
      // lock mutex ?
      for (ndx = (sfm->setup.maxPidFilters << 1); ndx--; ctable++)
      {
         CacheClearFilters( cache, ctable );
      }
   }
}

H_CacheTable SFMCacheGetTable( H_SfmCache cache )
{
   H_CacheTable ctable;
   if (cache->counter == 0)
   {
      ctable = NULL;
   }
   else
   {
      cache->counter--;
      ctable = cache->ctables + cache->counter;
   }
   return ctable;
}

/**
 * @brief   Search cache for section data
 * @param   H_SfmCache     cache          SFM cache handle.
 * @param   H_CacheTable   ctable         Cache table handle
 * @param   U16BIT         teid           Table Extension ID
 * @return  H_CacheFilter
 */
H_CacheFilter SFMCacheRetrieveFilter( H_SfmInstance sfm, H_CacheTable ctable,
   U16BIT teid, U8BIT vers )
{
   H_SfmCache cache = sfm->cache;
   H_CacheFilter cfilter, *pFilter;
   U16BIT index;
   index = HashIndex(teid, TABLE_BITS);
   pFilter = ctable->filters + index;
   // lock mutex
   cfilter = *pFilter;
   while (cfilter != NULL)
   {
      if (cfilter->tide == teid)
      {
         if (cfilter->vers == vers)
         {
            break;
         }
         DBGLOG(DF_CACHE, "version changed TEid=0x%x old=%d new=%d", teid, cfilter->vers, vers)
         // version changed - mark as stale
         cfilter->vers = STALE_VERSION;
         ctable->changed[index] |= STALE_BIT;
      }
      cfilter = cfilter->next;
   }
   if (cfilter == NULL)
   {
      if (cache->filtHead != NULL || AllocFiltBlock( cache ))
      {
         cfilter = cache->filtHead;
         cache->filtHead = cfilter->next;
         cfilter->next = NULL;
         cfilter->tide = teid;
         cfilter->vers = vers;
         cfilter->node = NULL;
         cfilter->ctndx = ctable - cache->ctables;
         cfilter->next = *pFilter;
         *pFilter = cfilter;
      }
      else
      {
         ERRLOG("mem fail")
      }
   }
   // unlock mutex
   return cfilter;
}

/**
 * @brief   Allocates space in cache for section data
 *          If allocation is made, *phBuffer has holds cache location. If section
 *          already allocated, or insufficient space, then *phBuffer is NULL.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   H_CacheTable   ctable         Cache table handle
 * @param   void**         phBuffer    Pointer to SFM section buffer handle
 * @return  void.
 */
void SFMCacheAddBlock( H_SfmInstance sfm, U8BIT size1, U8BIT bknum, void **phBuffer )
{
   H_SfmCache cache = sfm->cache;
   H_CacheFilter cfilter;
   H_CacheNode cnode;
   U8BIT *buff;

   cfilter = *phBuffer;
   if (cfilter != NULL)
   {
      // check whether already had section
      for (cnode = cfilter->node; cnode; cnode = cnode->next)
      {
         if (cnode->bknum == bknum)
         {
            break;
         }
      }
      if (cnode != NULL)
      {
         // found current version of block already cached
         cnode = NULL;
      }
      else
      {
         buff = SBM_AllocateBuffer( cache->sbm, size1 );
         if (buff == NULL)
         {
            DBGLOG(DF_CACHE, "Cache cannot store sz1=%d allocated=%d (512 byte blks)", ALLOCSIZE(size1), cache->allocated)
            cnode = NULL;
         }
         else
         {
            cnode = AddCacheNode( cache, cfilter, buff );
            if (cnode == NULL)
            {
               ERRLOG("mem fail")
               SBM_ReleaseBuffer(cache->sbm, buff);
            }
            else
            {
               cnode->bknum = bknum;
               cnode->report = 0;
            #ifndef NDEBUG
               cache->allocated += ALLOCSIZE(size1);
            #endif
            }
         }
      }
      *phBuffer = cnode;
   }
}

/**
 * @brief   Get section data buffer pointer in SFM's cache for the handle.
 *          This should only be called after SFM_RequireSection returned with
 *          SFM_UPDATE_CACHE. This may be called from 'interrupt' environment.
 *          The returned pointer may be used by client to copy section data
 *          into SFM's cache buffer.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   void*          hBuffer        Section buffer handle returned
 *                                        by SFM_RequireSection
 * @return  U8BIT*         Pointer to buffer to store section data
 */
U8BIT* SFM_CacheBuffer( H_SfmInstance sfm, void *hBuffer )
{
   U8BIT *buff;
   if (!CacheValidNode( sfm->cache, hBuffer ))
   {
      ERRLOG("Invalid pointer!")
      buff = NULL;
   }
   else
   {
      buff = ((H_CacheNode)hBuffer)->sbuf;
   }
   return buff;
}

/**
 * @brief
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   U8BIT*         pSection    Pointer to whole section data buffer
 * @param   void*          hBuffer        Section buffer handle returned
 *                                        by SFM_RequireSection
 * @return  void.
 */
void SFM_CacheReleaseBuffer( H_SfmInstance sfm, U8BIT *pSection, void *hBuffer )
{
   H_CacheFilter cfilter = (H_CacheFilter)hBuffer;
   H_CacheNode *ppNode, cnode;
   ppNode = &cfilter->node;
   while (*ppNode != NULL)
   {
      cnode = *ppNode;
      if (cnode->sbuf == pSection)
      {
         if (cnode->bknum != pSection[6])
         {
            ERRLOG("Mis-matched of buffer data with node (0x%x,0x%x)", cnode->bknum, pSection[6])
         }
         else
         {
            *ppNode = cnode->next;
            FreeCacheNode( sfm->cache, cnode );
         }
         break;
      }
      ppNode = &(cnode->next);
   }
}

/**
 * @brief
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   void*          hCache         Handle to cache buffer given to the
 *                                        F_CacheMatch callback function
 * @return  BOOLEAN
 */
BOOLEAN SFMCacheValidHandle( H_SfmInstance sfm, void *ptr )
{
   H_FiltBlock memFilters;
   PU8BIT pval, mval;
   U32BIT size;
   BOOLEAN result;

   pval = (PU8BIT)ptr;
   result = FALSE;
   size = sizeof(S_FiltBlock) + (NUM_FILTERS * sizeof(S_CacheFilter));
   memFilters = sfm->cache->memFilters;
   while (memFilters != NULL)
   {
      mval = (PU8BIT)memFilters;
      if (pval > mval && pval < mval + size)
      {
         result = TRUE;
         break;
      }
      memFilters = memFilters->next;
   }
   return result;
}

/**
 * @brief   Tells SFM Cache to update DSM-CC with cached buffer (reported to
 *          F_CacheMatch callback funtion.
 *          This also allows SFM cache to clean up completed buffers
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   U8BIT*         pSection    Pointer to whole section data buffer
 * @param   H_CacheFilter  cfilter     Hande given to F_CacheMatch
 * @return  void.
 */
void SFMCacheProcessSection( H_SfmInstance sfm, U8BIT *pSection, H_CacheFilter cfilter )
{
   H_CacheNode *ppNode, cnode;
   ppNode = &cfilter->node;
   while (*ppNode != NULL)
   {
      cnode = *ppNode;
      if (cnode->sbuf == pSection)
      {
         if (cnode->bknum != pSection[6])
         {
            ERRLOG("Mis-matched of buffer data with node (0x%x,0x%x)", cnode->bknum, pSection[6])
         }
         else
         {
            SFMFilterProcessSection( sfm, pSection, SFMFilterDsmResource(sfm, cfilter->dsmref));
            *ppNode = cnode->next;
            FreeCacheNode( sfm->cache, cnode );
         }
         break;
      }
      ppNode = &(cnode->next);
   }
}

/**
 * @brief   Search cache for section data
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   H_CacheTable   ctable         Cache table handle
 * @param   S_TableExt     tex            Table Extension ID and mask
 * @return  void.
 */
void SFMCacheSearch( H_SfmInstance sfm, H_CacheTable ctable, S_TableExt tex,
   U16BIT dsmref, E_SFM_STATUS status )
{
   H_CacheFilter cfilter;
   switch (tex.mask)
   {
      case  0xFFFE:
         cfilter = FindFilter( sfm, ctable, tex.id ^ 0x0001 );
         if (cfilter != NULL)
         {
            cfilter->dsmref = dsmref;
            ReportSections( sfm, cfilter, status );
         }
      case  0xFFFF:
         cfilter = FindFilter( sfm, ctable, tex.id );
         if (cfilter != NULL)
         {
            cfilter->dsmref = dsmref;
            ReportSections( sfm, cfilter, status );
         }
         break;

      default:
         // this wild card could match lots of them !
      {
         U16BIT ndx;
         tex.id &= tex.mask;
         // lock mutex ?
         for (ndx = 0; ndx != TABLE_SIZE; ndx++)
         {
            cfilter = ctable->filters[ndx];
            while (cfilter != NULL)
            {
               if ((cfilter->tide & tex.mask) == tex.id &&
                   cfilter->vers != STALE_VERSION)
               {
                  cfilter->dsmref = dsmref;
                  ReportSections( sfm, cfilter, status );
               }
               cfilter = cfilter->next;
            }
         }
      }
      break;
   }
   CachePurgeExpired( sfm );
}

