/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Section Filter Manasger (SFM): filters
 * @file    sfm_filter.c
 * @date    16-10-2013
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

#include "sfm_main.h"
#include "sfm_cache.h"
#include "sfm_filter.h"
#include "cldsmcc.h"

/*---- Constant definitions for this file-------------------------------------*/

#define INVALID_NDX        0xFF  // must be U8BIT max value
#define INVALID_TID        0xFF
#define EXACT_MATCH        16
#define NEAR_MATCH         15
#define PRIORITY_SHIFT     5
#define PRI_EXACT_MASK     ((1 << PRIORITY_SHIFT) - 1)

/* Number of entries in Hash Table - probably best to be prime number */
#define HASH_SIZE          17
#define TOTAL_DSM_LISTS    (HASH_SIZE + 1)

/*---- Local typedef structs for this file------------------------------------*/

typedef struct s_BitCnt
{
   U16BIT zeros;
   U16BIT ones;
} S_BitCnt;

typedef struct s_Table
{
   U8BIT id;
   U8BIT mask;
} S_Table;

typedef struct s_DsmResource
{
   H_DsmResource next;
   S_TableExt tex;
   U8BIT tid;
   U8BIT priority;
   U8BIT dmxPidIndex;
   U8BIT accounted;
   H_DscSFRef dsmSfRef;
} S_DsmResource;

typedef struct s_PidTable
{
   S_DsmResource *dsmlists[TOTAL_DSM_LISTS];
   S_BitCnt bitcnt[16];
   H_CacheTable cachetable;
   U16BIT total;
} S_PidTable;

typedef struct s_DmxSecFilter
{
   SECFILT sfid;
   S_TableExt tex;
   S_Table tbl;
} S_DmxSecFilter;

typedef struct s_DmxPidFilter
{
   PIDFILT pfid;
   U16BIT pid;
   S_PidTable pt3B;
   S_PidTable pt3C;
   S_PidTable pt3D;
   U16BIT numDsmRes;
   U16BIT pidcounter;
   S_DmxSecFilter *dmxSecArray;
   BOOLEAN updating;
} S_DmxPidFilter;


/*---- Local (static) variable declarations for this file---------------------*/
/*   (internal variables declared static to make them local) */

/*---- LOCAL function definitions for this file ------------------------------*/
/*   (internal functions declared static to make them local) */

static U8BIT ExactPriority( U16BIT mask )
{
   U8BIT exactness, cnt;
   FUNCTION_START(ExactPriority)
   switch (mask)
   {
      case 0xFFFF:
         exactness = EXACT_MATCH;
         break;
      case 0xFFFE:
         exactness = NEAR_MATCH;
         break;

      default:
         /* for exactness, count number of bits set in mask */
         for (cnt = 16, exactness = 0; cnt--; mask >>= 1)
         {
            if (mask & 1)
            {
               exactness++;
            }
         }
   }
   FUNCTION_FINISH(ExactPriority)
   return exactness;
}

static U8BIT FindPidFilter( S_DmxPidFilter *pDmxPid, U8BIT max, U16BIT pid )
{
   U8BIT cnt;
   FUNCTION_START(FindPidFilter)
   for (cnt = max; cnt--; )
   {
      if (pDmxPid[cnt].pid == pid)
      {
         break;
      }
   }
   if (cnt == 0xFF)   // i.e. INVALID_NDX
   {
      U16BIT least = 0xFFFF;
      while (max--)
      {
         if (pDmxPid[max].pfid == INVALID_PIDFILT &&
             pDmxPid[max].updating == FALSE &&
             least >= pDmxPid[max].pidcounter)
         {
            least = pDmxPid[max].pidcounter;
            cnt = max;
         }
      }
   }
   FUNCTION_FINISH(FindPidFilter)
   return cnt;
}

static S_DmxPidFilter* RetrievePidFilter( S_DmxPidFilter *pDmxPid, U8BIT cnt, PIDFILT pfid )
{
   FUNCTION_START(RetrievePidFilter)
   while (cnt--)
   {
      if (pDmxPid->pfid == pfid)
      {
         return pDmxPid;
      }
      pDmxPid++;
   }
   FUNCTION_FINISH(RetrievePidFilter)
   return NULL;
}

static void IncBitcnt( S_BitCnt *bitcnt, S_TableExt tex )
{
   U16BIT mbit;
   for (mbit = 0x8000; mbit; mbit >>= 1, bitcnt++)
   {
      if (tex.mask & mbit)
      {
         if (tex.id & mbit)
         {
            bitcnt->ones++;
         }
         else
         {
            bitcnt->zeros++;
         }
      }
      else
      {
         bitcnt->zeros++;
         bitcnt->ones++;
      }
   }
}

static void DecBitcnt( S_BitCnt *bitcnt, S_TableExt tex )
{
   U16BIT mbit;
   for (mbit = 0x8000; mbit; mbit >>= 1, bitcnt++)
   {
      if (tex.mask & mbit)
      {
         if (tex.id & mbit)
         {
            bitcnt->ones--;
         }
         else
         {
            bitcnt->zeros--;
         }
      }
      else
      {
         bitcnt->zeros--;
         bitcnt->ones--;
      }
   }
}

static S_TableExt TableExtension( S_BitCnt *bitcnt )
{
   U16BIT mbit;
   S_TableExt tex = { 0, 0 };
   for (mbit = 0x8000; mbit; mbit >>= 1, bitcnt++)
   {
      if (!bitcnt->ones)
      {
         tex.mask |= mbit;
      }
      else if (!bitcnt->zeros)
      {
         tex.mask |= mbit;
         tex.id |= mbit;
      }
   }
   return tex;
}

static S_TableExt FindTexIdOne( S_DsmResource **ppDsmRes, U16BIT mbit )
{
   S_DsmResource *pDsmRes;
   S_TableExt tex = {0, 0};
   U16BIT hcnt;
   for (hcnt = TOTAL_DSM_LISTS; hcnt != 0; hcnt--, ppDsmRes++)
   {
      for (pDsmRes = *ppDsmRes; pDsmRes != NULL; pDsmRes = pDsmRes->next)
      {
         if (!pDsmRes->accounted && (pDsmRes->tex.id & mbit) == mbit)
         {
            pDsmRes->accounted = 1;
            return pDsmRes->tex;
         }
      }
   }
   ASSERT(0)
   return tex;
}

static S_TableExt FindTexIdZero( S_DsmResource **ppDsmRes, U16BIT mbit )
{
   S_DsmResource *pDsmRes;
   S_TableExt tex = {0, 0};
   U16BIT hcnt;
   for (hcnt = TOTAL_DSM_LISTS; hcnt != 0; hcnt--, ppDsmRes++)
   {
      for (pDsmRes = *ppDsmRes; pDsmRes != NULL; pDsmRes = pDsmRes->next)
      {
         if (!pDsmRes->accounted && (pDsmRes->tex.id & mbit) == 0)
         {
            pDsmRes->accounted = 1;
            return pDsmRes->tex;
         }
      }
   }
   ASSERT(0)
   return tex;
}

void ClearTexIdAccounts( S_DsmResource **ppDsmRes )
{
   S_DsmResource *pDsmRes;
   U16BIT hcnt;
   FUNCTION_START(ClearTexIdAccounts)
   for (hcnt = TOTAL_DSM_LISTS; hcnt != 0; hcnt--, ppDsmRes++)
   {
      for (pDsmRes = *ppDsmRes; pDsmRes != NULL; pDsmRes = pDsmRes->next)
      {
         pDsmRes->accounted = 0;
      }
   }
   FUNCTION_FINISH(ClearTexIdAccounts)
}

static S_DmxSecFilter* SetupSecArray( S_PidTable *pPidTbl, S_DmxSecFilter *pTmpSec, U16BIT scnt, S_Table tbl )
{
   S_TableExt tex;
   S_BitCnt bitscopy[16];
   S_BitCnt *bitcnt;
   U16BIT mbit;
   ASSERT(scnt != 0)
   if (scnt == 1)
   {
      pTmpSec->tex = TableExtension( pPidTbl->bitcnt );
   }
   else
   {
      memcpy(bitscopy, pPidTbl->bitcnt, 16 * sizeof(S_BitCnt));

      for (mbit = 0x8000, bitcnt = bitscopy; mbit; mbit >>= 1, bitcnt++)
      {
         if (bitcnt->ones == 1)
         {
            tex = FindTexIdOne( pPidTbl->dsmlists, mbit );
            DecBitcnt( bitscopy, tex );
            pTmpSec->tbl = tbl;
            pTmpSec->tex = tex;
            scnt--;
            pTmpSec++;
            if (scnt == 1)
               break;
            mbit = 0x8000;
            bitcnt = bitscopy;
         }
         else if (bitcnt->zeros == 1)
         {
            tex = FindTexIdZero( pPidTbl->dsmlists, mbit );
            DecBitcnt( bitscopy, tex );
            pTmpSec->tbl = tbl;
            pTmpSec->tex = tex;
            scnt--;
            pTmpSec++;
            if (scnt == 1)
               break;
            mbit = 0x8000;
            bitcnt = bitscopy;
         }
      }
      ClearTexIdAccounts( pPidTbl->dsmlists );
      pTmpSec->tex = TableExtension( bitscopy );
   }
   pTmpSec->tbl = tbl;
   pTmpSec++;
   return pTmpSec;
}

/**
 * @brief   Setup Demux Filter requirements in temporary section filter array
 * @param   S_DmxPidFilter *  pDmxPid        Pointer to PID filter data
 * @param   S_DmxSecFilter*   pTmpSec        Pointer to temporary array
 * @param   U16BIT            scTotal        total section filters per PID
 * @return  void
 */
static void SetupSecFilterArray( S_DmxPidFilter *pDmxPid, S_DmxSecFilter *pTmpSec, U16BIT scTotal )
{
   S_Table tbl = { 0xFF, 0xFF };
   U16BIT numb, numc, numd, totalbc;
   FUNCTION_START(SetupSecFilterArray)

   numb = pDmxPid->pt3B.total;
   numc = pDmxPid->pt3C.total;
   numd = pDmxPid->pt3D.total;
   switch (scTotal)
   {
      case 1:
         if (numb)
         {
            if (numc)
            {
               pTmpSec->tex.id = 0;
               pTmpSec->tex.mask = 0;
               pTmpSec->tbl.id = 0x38;
               pTmpSec->tbl.mask = 0xF8;
            }
            else if (numd)
            {
               pTmpSec->tex.id = 0;
               pTmpSec->tex.mask = 0;
               pTmpSec->tbl.id = 0x39;
               pTmpSec->tbl.mask = 0xF9;
            }
            else
            {
               pTmpSec->tex = TableExtension(pDmxPid->pt3B.bitcnt);
               pTmpSec->tbl.id = 0x3B;
               pTmpSec->tbl.mask = 0xFF;
            }
         }
         else if (numc)
         {
            if (numd)
            {
               pTmpSec->tex.id = 0;
               pTmpSec->tex.mask = 0;
               pTmpSec->tbl.id = 0x3C;
               pTmpSec->tbl.mask = 0xFC;
            }
            else
            {
               pTmpSec->tex = TableExtension(pDmxPid->pt3C.bitcnt);
               pTmpSec->tbl.id = 0x3C;
            }
         }
         else if (numd)
         {
            pTmpSec->tex = TableExtension(pDmxPid->pt3D.bitcnt);
            pTmpSec->tbl.id = 0x3D;
            pTmpSec->tbl.mask = 0xFF;
         }
         else
         {
            pTmpSec->tbl.id = INVALID_TID;
         }
         break;

      case 2:
         if (numb)
         {
            pTmpSec->tex = TableExtension(pDmxPid->pt3B.bitcnt);
            pTmpSec->tbl.id = 0x3B;
            pTmpSec->tbl.mask = 0xFF;
            pTmpSec++;
         }
         if (numc)
         {
            if (numd)
            {
               pTmpSec->tex.id = 0;
               pTmpSec->tex.mask = 0;
               pTmpSec->tbl.id = 0x3C;
               pTmpSec->tbl.mask = 0xFC;
            }
            else
            {
               pTmpSec->tex = TableExtension(pDmxPid->pt3C.bitcnt);
               pTmpSec->tbl.id = 0x3C;
               pTmpSec->tbl.mask = 0xFF;
            }
         }
         else if (numd)
         {
            pTmpSec->tex = TableExtension(pDmxPid->pt3D.bitcnt);
            pTmpSec->tbl.id = 0x3D;
            pTmpSec->tbl.mask = 0xFF;
         }
         else
         {
            pTmpSec->tbl.id = INVALID_TID;
         }
         break;

      default:
         if (numd)
         {
            pTmpSec->tex = TableExtension( pDmxPid->pt3D.bitcnt );
            tbl.id = 0x3D;
            pTmpSec->tbl = tbl;
            pTmpSec++;
            scTotal--;
         }
         totalbc = numb + numc;
         if (totalbc > scTotal)
         {
            if (numb == 0)
            {
               numc = scTotal;
            }
            else if (numc == 0)
            {
               numb = scTotal;
            }
            else
            {
               numb = 1 + ((numb * scTotal) / totalbc);
               if (numb == totalbc)
                  numb = totalbc - 1;
            }
            numc = scTotal - numb;
            scTotal = 0;
         }
         else
         {
            scTotal -= numb + numc;
         }
         if (numb)
         {
            tbl.id = 0x3B;
            pTmpSec = SetupSecArray( &pDmxPid->pt3B, pTmpSec, numb, tbl );
         }
         if (numc)
         {
            tbl.id = 0x3C;
            pTmpSec = SetupSecArray( &pDmxPid->pt3C, pTmpSec, numc, tbl );
         }
         while (scTotal--)
         {
            pTmpSec->tbl.id = INVALID_TID;
            pTmpSec++;
         }
   }
}

/**
 * @brief   Compare new temporary array with current active array of section filters
 * @param   S_DmxSecFilter*   pActive  Array of active section filters
 * @param   S_DmxSecFilter*   pTmpSec  Array of new temporary section filters
 * @param   U16BIT            secCount count of items in array
 * @return  TRUE - if array has changed, FALSE - if the same
 */
static BOOLEAN CompareSecFilters( S_DmxSecFilter *pActive, S_DmxSecFilter *pTmpSec, U16BIT secCount )
{
   BOOLEAN result = FALSE;
   FUNCTION_START(CompareSecFilters)
   for (; secCount--; pActive++, pTmpSec++)
   {
      if (pTmpSec->tbl.id != pActive->tbl.id)
      {
         result = TRUE;
         break;
      }
      if (pTmpSec->tbl.id == INVALID_TID)
      {
         break;
      }
      if (pTmpSec->tex.mask != pActive->tex.mask ||
          pTmpSec->tex.id != pActive->tex.id ||
          pTmpSec->tbl.mask != pActive->tbl.mask)
      {
         result = TRUE;
         break;
      }
   }
   FUNCTION_FINISH(CompareSecFilters)
   return result;
}

/**
 * @brief   Compare temporary array with active array and deactivate (remove)
            section filters that are no longer required
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   S_DmxSecFilter*   pDmxSec        Pointer to section filter array
                                             for relevant pid filter
 * @param   U16BIT            secCount       count of items in array
 * @return  void
 */
static void RemoveOldActives( H_SfmInstance sfm, S_DmxSecFilter *pDmxSec, U16BIT secCount )
{
   S_DmxSecFilter *pTmpSec;
   U16BIT stmp, scnt;
   FUNCTION_START(RemoveOldActives)
   for (scnt = secCount; scnt--; pDmxSec++)
   {
      if (pDmxSec->sfid != INVALID_SECFILT)
      {
         pTmpSec = sfm->dmxSecArray;
         for (stmp = secCount; stmp != 0; stmp--, pTmpSec++)
         {
            if (pTmpSec->tex.mask == pDmxSec->tex.mask &&
                pTmpSec->tex.id == pDmxSec->tex.id &&
                pTmpSec->tbl.mask == pDmxSec->tbl.mask &&
                pTmpSec->tbl.id == pDmxSec->tbl.id)
            {
               pTmpSec->sfid = pDmxSec->sfid;
               break;
            }
         }
         if (stmp == 0)
         {
            DBGLOG(DF_FILTER, "%p", pDmxSec)
            DBGLOG((DF_FILTER|DF_HWSF), "tid=0x%x tidMask=0x%x tide=0x%x tideMask=0x%x",
               pDmxSec->tbl.id, pDmxSec->tbl.mask, pDmxSec->tex.id, pDmxSec->tex.mask)
            STB_DMXReleaseSectFilter( sfm->demux, pDmxSec->sfid );
            pDmxSec->sfid = INVALID_SECFILT;
            pDmxSec->tbl.id = INVALID_TID;
         }
      }
   }
   FUNCTION_FINISH(RemoveOldActives)
}

/**
 * @brief   Utility function to setup a section filter
 * @param   dmx demux reference
 * @param   pfid PID filter Id
 * @param   pfid Handle to allocated PID filter
 * @param   tid  Table Id(s) required
 * @param   tidMask Mask for tid byte
 * @param   tidExt Table Id Extension(s)
 * @param   tidExtMask Mask for tidExt bytes
 * @return  SECFILT  Handle to the allocated section filter.
 *                   INVALID_SECFILT - failure.
 */
static SECFILT SectionFilterAdd( H_SfmInstance sfm, PIDFILT pfid, U8BIT tid,
   U8BIT tidMask, U16BIT tidExt, U16BIT tidExtMask )
{
   U8BIT match[SECFILT_MASK_SIZE];
   U8BIT mmask[SECFILT_MASK_SIZE];
   SECFILT sfid;
   U16BIT i;
   sfid = STB_DMXGrabSectFilter(sfm->demux, pfid);
   if (sfid == INVALID_SECFILT)
   {
      ERRLOG("Failed to grab Section Filter on PFID 0x%x", pfid)
   }
   else
   {
      match[0] = tid;
      match[1] = (U8BIT)(tidExt >> 8);
      match[2] = (U8BIT)(tidExt & 0x00ff);
      mmask[0] = tidMask;
      mmask[1] = (U8BIT)(tidExtMask >> 8);
      mmask[2] = (U8BIT)(tidExtMask & 0x00ff);
      for (i = 3; i != SECFILT_MASK_SIZE; i++)
      {
         match[i] = 0;
         mmask[i] = 0;
      }
      DBGLOG((DF_FILTER|DF_HWSF), "pf=%x tid=%x tidm=%x xid=%x xidm=%x", pfid, tid, tidMask, tidExt, tidExtMask)
      STB_DMXSetupSectFilter(sfm->demux, sfid, match, mmask, 0, TRUE /*crc checked*/);
   }
   return sfid;
}

/**
 * @brief   Check temporary array for inactive, but required, section filters
 *          and request them from platform.
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   PIDFILT       pfid           PID filter Id for platform
 * @param   U16BIT            secCount       count of items in temp array
 * @return  void
 */
static void RequestNewActives( H_SfmInstance sfm, PIDFILT pfid,
   S_DmxSecFilter *pDmxSec, U16BIT secCount )
{
   FUNCTION_START(RequestNewActives)
   for (; secCount--; pDmxSec++)
   {
      if (pDmxSec->sfid == INVALID_SECFILT && pDmxSec->tbl.id != INVALID_TID)
      {
         DBGLOG(DF_FILTER, "%p", pDmxSec)
         pDmxSec->sfid =
            SectionFilterAdd( sfm, pfid, pDmxSec->tbl.id,
               pDmxSec->tbl.mask, pDmxSec->tex.id, pDmxSec->tex.mask );
         DBGLOG(DF_FILTER, "tid=0x%x tidMask=0x%x tide=0x%x tideMask=0x%x", pDmxSec->tbl.id,
            pDmxSec->tbl.mask, pDmxSec->tex.id, pDmxSec->tex.mask)
      }
   }
   FUNCTION_FINISH(RequestNewActives)
}

/**
 * @brief   deactivate (remove) all section filters that are still active
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   S_DmxSecFilter*   pDmxSec        Pointer to section filter array
                                             for relevant pid filter
 * @param   U16BIT            secCount       count of items in array
 * @return  void
 */
static void RemoveAllActives( H_SfmInstance sfm, S_DmxSecFilter *pDmxSec, U16BIT secCount )
{
   FUNCTION_START(RemoveAllActives)
   for (; secCount--; pDmxSec++)
   {
      if (pDmxSec->sfid != INVALID_SECFILT)
      {
         DBGLOG(DF_FILTER, "%p", pDmxSec)
         DBGLOG(DF_FILTER, "tid=0x%x tidMask=0x%x tide=0x%x tideMask=0x%x",
            pDmxSec->tbl.id, pDmxSec->tbl.mask, pDmxSec->tex.id, pDmxSec->tex.mask)
         STB_DMXReleaseSectFilter( sfm->demux, pDmxSec->sfid );
         pDmxSec->sfid = INVALID_SECFILT;
         pDmxSec->tbl.id = INVALID_TID;
      }
   }
   FUNCTION_FINISH(RemoveAllActives)
}

/**
 * @brief   copy details into active section filter array for PID and
            clear temporary section filter array
 * @param   S_DmxSecFilter*   pTmpSec        Pointer to temporary array
 * @param   U16BIT            scnt           count of items in array
 * @return  void
 */
static void CopyAndClearTempArray( S_DmxSecFilter *pDmxSec, S_DmxSecFilter *pTmpSec, U16BIT scnt )
{
   FUNCTION_START(CopyAndClearTempArray)
   for (; scnt--; pTmpSec++, pDmxSec++)
   {
      *pDmxSec = *pTmpSec;
      pTmpSec->sfid = INVALID_SECFILT;
      pTmpSec->tbl.id = INVALID_TID;
   }
   FUNCTION_FINISH(CopyAndClearTempArray)
}

/**
 * @brief   Update demux section filters (adding and removing section filters
            as required by Dsmcc resource filters. This is done on the relevant
            PID filter, and then this PID filter is started.
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   S_DmxPidFilter*   pDmxPid        Pointer to Demux PID filter details
 * @return  void
 */
static void UpdateDemux( H_SfmInstance sfm, S_DmxPidFilter *pDmxPid, U16BIT secCount )
{
   FUNCTION_START(UpdateDemux)
   SetupSecFilterArray( pDmxPid, sfm->dmxSecArray, secCount );
   if (CompareSecFilters(pDmxPid->dmxSecArray, sfm->dmxSecArray, secCount))
   {
      STB_DMXStopPIDFilter( sfm->demux, pDmxPid->pfid );
      RemoveOldActives( sfm, pDmxPid->dmxSecArray, secCount );
      RequestNewActives( sfm, pDmxPid->pfid, sfm->dmxSecArray, secCount );
      CopyAndClearTempArray( pDmxPid->dmxSecArray, sfm->dmxSecArray, secCount );
      STB_DMXStartPIDFilter( sfm->demux, pDmxPid->pfid );
   }
   FUNCTION_FINISH(UpdateDemux)
}

/**
 * @brief   Insert Dsmcc Filter resource into list in demux Pid filter
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   S_DmxPidFilter*   pDmxPid        Pointer to Demux PID filter details
 * @param   S_DsmResource*    pDsmRes        Pointer to Dsmcc filter resource
 * @return  void
 */
static void InsertDsmResource( H_SfmInstance sfm, S_PidTable *pPidTbl, S_DsmResource *pDsmRes )
{
   H_DsmResource *phDsmRes;
   FUNCTION_START(InsertDsmResource)

   /* remove from free list */
   ASSERT( sfm->freeList == pDsmRes )
   sfm->freeList = pDsmRes->next;

   if (pDsmRes->tex.mask == 0xFFFF)
   {
      phDsmRes = pPidTbl->dsmlists + (pDsmRes->tex.id % HASH_SIZE);
   }
   else
   {
      phDsmRes = pPidTbl->dsmlists + HASH_SIZE;
      /* ensure tableIdExt has bits zero where mask is zero */
      //pDsmRes->tex.id &= pDsmRes->tex.mask;
   }
   pPidTbl->total++;
   IncBitcnt( pPidTbl->bitcnt, pDsmRes->tex );
   sfm->setup.mutexLock( sfm->setup.sfmMutex );
   while (*phDsmRes && (*phDsmRes)->priority > pDsmRes->priority)
   {
      phDsmRes = &(*phDsmRes)->next;
   }
   /* insert before existing one of same priority */
   pDsmRes->next = *phDsmRes;
   *phDsmRes = pDsmRes;
   sfm->setup.mutexUnlock( sfm->setup.sfmMutex );
   if (pPidTbl->cachetable != NULL)
   {
      U32BIT dsmref;
      dsmref = pDsmRes - sfm->dmxResFirst;
      switch (pDsmRes->priority >> PRIORITY_SHIFT)
      {
         case SF_PRIORITY_LOW:
            SFMCacheSearch( sfm, pPidTbl->cachetable, pDsmRes->tex, (U16BIT)dsmref, SFM_UPDATE_LOW );
            break;
         case SF_PRIORITY_HIGH:
            SFMCacheSearch( sfm, pPidTbl->cachetable, pDsmRes->tex, (U16BIT)dsmref, SFM_UPDATE_HIGH );
            break;
         default:
         case SF_PRIORITY_DIRECT:
            /*must not take from cache*/
            break;
      }
   }
   FUNCTION_FINISH(InsertDsmResource)
}

/**
 * @brief   Remove Dsmcc Filter resource from list in demux Pid filter resource
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   S_DmxPidFilter*   pDmxPid        Pointer to Demux PID filter details
 * @param   S_DsmResource*    pDsmRes        Pointer to Dsmcc filter resource
 * @return  void
 */
static void RemoveDsmResource( H_SfmInstance sfm, S_PidTable *pPidTbl,
   S_DsmResource *pDsmRes )
{
   H_DsmResource *phDsmRes;
   FUNCTION_START(RemoveDsmResource)
   ASSERT( pDsmRes != NULL )
   if (pDsmRes->tex.mask == 0xFFFF)
   {
      phDsmRes = pPidTbl->dsmlists + (pDsmRes->tex.id % HASH_SIZE);
   }
   else
   {
      phDsmRes = pPidTbl->dsmlists + HASH_SIZE;
   }
   pPidTbl->total--;
   DecBitcnt( pPidTbl->bitcnt, pDsmRes->tex );
   sfm->setup.mutexLock( sfm->setup.sfmMutex );
   while (*phDsmRes)
   {
      if (*phDsmRes == pDsmRes)
      {
         *phDsmRes = pDsmRes->next;
         break;
      }
      phDsmRes = &((*phDsmRes)->next);
   }
   sfm->setup.mutexUnlock( sfm->setup.sfmMutex );
   /* return to free list */
   pDsmRes->next = sfm->freeList;
   pDsmRes->tid = INVALID_TID;
   sfm->freeList = pDsmRes;
   FUNCTION_FINISH(RemoveDsmResource)
}

/*---- GLOBAL function definitions for this file -----------------------------*/


/**
 * @brief   Create Filter resources for Section Filter Manager instance
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  BOOLEAN                       TRUE if success
 */
BOOLEAN SFMFilterCreateResources( H_SfmInstance sfm )
{
   S_DmxPidFilter *pDmxPid;
   S_DmxSecFilter *pDmxSec;
   S_DsmResource *pDsmRes;
   U16BIT dmxCount, dsmCount, secCount, cnt;
   U32BIT size;
   BOOLEAN result;

   FUNCTION_START(SFMFilterCreateResources)
   dmxCount = sfm->setup.maxPidFilters;
   dsmCount = sfm->setup.maxAvailableSectionFilters;
   secCount = sfm->setup.maxSecFiltersPerPid;
   ASSERT( dmxCount != 0 && secCount != 0 );
   size = secCount * sizeof(S_DmxSecFilter);
   size *= dmxCount + 1;
   size += dmxCount * sizeof(S_DmxPidFilter);
   size += (dsmCount * sizeof(S_DsmResource));
   pDmxPid = sfm->setup.memAlloc( size );
   if (pDmxPid == NULL)
   {
      result = FALSE;
   }
   else
   {
      result = TRUE;
      memset(pDmxPid, 0, size);
      sfm->dmxPidArray = pDmxPid;
      pDmxSec = (S_DmxSecFilter *)(pDmxPid + dmxCount);
      /* Initialise global demux section filter array -
       * for temporary use in UpdateDemux */
      sfm->dmxSecArray = pDmxSec;
      for (cnt = secCount; cnt--; pDmxSec++)
      {
         pDmxSec->sfid = INVALID_SECFILT;
         pDmxSec->tbl.id = INVALID_TID;
      }
      /* Initialise Demux PID filter array */
      while (dmxCount--)
      {
         pDmxPid->pid = INVALID_PID;
         pDmxPid->pfid = INVALID_PIDFILT;
         pDmxPid->dmxSecArray = pDmxSec;
         for (cnt = secCount; cnt--; pDmxSec++)
         {
            pDmxSec->sfid = INVALID_SECFILT;
            pDmxSec->tbl.id = INVALID_TID;
         }
         pDmxPid++;
      }
      /* Initialise Dsmcc filter resource list */
      pDsmRes = (S_DsmResource *)pDmxSec;
      sfm->dmxResFirst = pDsmRes;
      sfm->freeList = pDsmRes;
      while (--dsmCount)
      {
         pDsmRes->next = pDsmRes + 1;
         pDsmRes->tid = INVALID_TID;
         pDsmRes->dmxPidIndex = INVALID_NDX;
         pDsmRes++;
      }
      pDsmRes->tid = INVALID_TID;
      pDsmRes->dmxPidIndex = INVALID_NDX;
      pDsmRes->next = NULL;
      sfm->dmxResLast = pDsmRes;
   }
   FUNCTION_FINISH(SFMFilterCreateResources)
   return result;
}

/**
 * @brief   Initialise caching references
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  void
 */
void SFMFilterInitCaching( H_SfmInstance sfm )
{
   S_DmxPidFilter *pDmxPid;
   H_SfmCache cache;
   U16BIT dmxCount;

   FUNCTION_START(SFMFilterInitCaching)
   pDmxPid = sfm->dmxPidArray;
   cache = sfm->cache;
   if (cache != NULL)
   {
      for (dmxCount = sfm->setup.maxPidFilters; dmxCount--; pDmxPid++)
      {
         pDmxPid->pt3B.cachetable = SFMCacheGetTable( cache );
         pDmxPid->pt3C.cachetable = SFMCacheGetTable( cache );
      }
   }
   FUNCTION_FINISH(SFMFilterInitCaching)
}

/**
 * @brief   Destroy Filter resources for Section Filter Manager instance
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  void
 */
void SFMFilterDestroyResources( H_SfmInstance sfm )
{
   S_DmxPidFilter *pDmxPid;
   U16BIT total;

   FUNCTION_START(SFMDestroyDemuxResources)
   pDmxPid = sfm->dmxPidArray;
   for (total = sfm->setup.maxPidFilters; total--; pDmxPid++)
   {
      if (pDmxPid->pfid != INVALID_PIDFILT)
      {
         STB_DMXStopPIDFilter( sfm->demux, pDmxPid->pfid );
         STB_DMXReleasePIDFilter( sfm->demux, pDmxPid->pfid );
      }
   }
   sfm->setup.memFree( sfm->dmxPidArray );
   FUNCTION_FINISH(SFMDestroyDemuxResources)
}

/**
 * @brief   Add DSM-CC section filter.
 *          Called by DSM-CC component.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_RequireSection. It assumes that this function
 *          is entered on the same task as SFM_DelectionFilter.
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   P_SecFilter   pFilter        Pointer to DSM-CC filter details
 * @param   H_DscSFRef      dsmSfRef       DSM-CC SF reference handle
 * @return  void*                            SFM Filter handle
 */
void* SFM_DsmccFilterAdd( H_SfmInstance sfm, P_SecFilter pFilter,
   H_DscSFRef dsmSfRef )
{
   S_DmxPidFilter *pDmxPid;
   S_DsmResource *pDsmRes;
   U8BIT dmxPidIndex;

   FUNCTION_START(SFM_DsmccFilterAdd)
   ASSERT( pFilter->priority < 8 )
   pDsmRes = sfm->freeList;
   if (pDsmRes == NULL)
   {
      ERRLOG("Eek! Dsmcc filter Resources Gone!")
   }
   else
   {
      dmxPidIndex = FindPidFilter( sfm->dmxPidArray, sfm->setup.maxPidFilters, pFilter->pid );
      if (dmxPidIndex == INVALID_NDX)
      {
         ERRLOG("Eek! Demux Pid Filters Gone!")
         pDsmRes = NULL;
      }
      else /* Todo?: assert that DSM-CC is not adding duplicate section filter */
      {
         pDmxPid = sfm->dmxPidArray + dmxPidIndex;
         pDmxPid->pid = pFilter->pid;
         pDsmRes->dsmSfRef = dsmSfRef;
         pDsmRes->priority = (pFilter->priority << PRIORITY_SHIFT) |
            ExactPriority( pFilter->tableIdExtMask );
         pDsmRes->dmxPidIndex = dmxPidIndex;
         pDsmRes->tid = pFilter->tableId;
         pDsmRes->tex.id = pFilter->tableIdExt;
         pDsmRes->tex.mask = pFilter->tableIdExtMask;

         DBGLOG(DF_FILTER, "ndx=%d PID=0x%x PFID=0x%x tid=0x%x tide=0x%x",
            dmxPidIndex, pFilter->pid, pDmxPid->pfid, pDsmRes->tid, pDsmRes->tex.id)

         switch (pDsmRes->tid)
         {
            case 0x3B: InsertDsmResource( sfm, &pDmxPid->pt3B, pDsmRes ); break;
            case 0x3C: InsertDsmResource( sfm, &pDmxPid->pt3C, pDsmRes ); break;
            case 0x3D: InsertDsmResource( sfm, &pDmxPid->pt3D, pDsmRes ); break;
            default:
               ASSERT(0);
         }
         pDmxPid->numDsmRes++;
         if (!pDmxPid->updating)
         {
            pDmxPid->updating = TRUE;
            sfm->setup.updateFilter(sfm, pDmxPid);
         }
      }
   }
   FUNCTION_FINISH(SFM_DsmccFilterAdd)
   return pDsmRes;
}

/**
 * @brief   Delete DSM-CC section filter.
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_RequireSection.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   void*          filterHandle   SFM Filter handle
 * @param   H_DscSFRef   dsmSfRef       DSM-CC reference handle
 * @return
 */
void SFM_DsmccFilterDelete( H_SfmInstance sfm, void *filterHandle,
   H_DscSFRef dsmSfRef )
{
   S_DsmResource *pDsmRes;
   S_DmxPidFilter *pDmxPid;
   FUNCTION_START(SFM_DsmccFilterDelete)
   pDsmRes = (S_DsmResource *)filterHandle;
   ASSERT( pDsmRes->dmxPidIndex != INVALID_NDX )
   ASSERT( pDsmRes->dsmSfRef == dsmSfRef )
   if (pDsmRes->dmxPidIndex != INVALID_NDX &&
       pDsmRes->dsmSfRef == dsmSfRef)
   {
      pDmxPid = sfm->dmxPidArray + pDsmRes->dmxPidIndex;
      ASSERT( pDmxPid->numDsmRes != 0 )
      if (pDmxPid->numDsmRes == 0)
      {
         ERRLOG("Eek! already have zero Dsmcc filter resources allocated!")
      }
      else
      {
         DBGLOG(DF_FILTER, "num=%d PID=0x%x PFID=0x%x tid=0x%x tide=0x%x",
            pDmxPid->numDsmRes, pDmxPid->pid, pDmxPid->pfid, pDsmRes->tid, pDsmRes->tex.id)

         ASSERT( pDmxPid->pid != INVALID_PID )
         switch (pDsmRes->tid)
         {
            case 0x3B: RemoveDsmResource( sfm, &pDmxPid->pt3B, pDsmRes ); break;
            case 0x3C: RemoveDsmResource( sfm, &pDmxPid->pt3C, pDsmRes ); break;
            case 0x3D: RemoveDsmResource( sfm, &pDmxPid->pt3D, pDsmRes ); break;
            default:
               ASSERT(0);
         }
         pDmxPid->numDsmRes--;
         if (!pDmxPid->updating)
         {
            pDmxPid->updating = TRUE;
            sfm->setup.updateFilter(sfm, pDmxPid);
         }
      }
   }
   FUNCTION_FINISH(SFM_DsmccFilterDelete)
}

/**
 * @brief   Update SFM PID filter to configure HW for latest requirements
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   H_DmxPidFilter pPF            Handle to SFM pid filter
 * @return  void.
 */
void SFM_FilterUpdate( H_SfmInstance sfm, H_DmxPidFilter pPF )
{
   FUNCTION_START(SFM_FilterUpdate)
   ASSERT( pPF->updating != FALSE )
   pPF->updating = FALSE;
   if (pPF->pfid != INVALID_PIDFILT)
   {
      if (pPF->numDsmRes != 0)
      {
         UpdateDemux( sfm, pPF, sfm->setup.maxSecFiltersPerPid );
      }
      else
      {
         STB_DMXStopPIDFilter( sfm->demux, pPF->pfid );
         RemoveAllActives( sfm, pPF->dmxSecArray, sfm->setup.maxSecFiltersPerPid );

         DBGLOG((DF_FILTER|DF_HWSF), "DELETE PID=0x%x PFID=0x%x", pPF->pid, pPF->pfid)
         STB_DMXReleasePIDFilter( sfm->demux, pPF->pfid );
         pPF->pfid = INVALID_PIDFILT;
         pPF->pidcounter = sfm->pidcounter;
         sfm->pidcounter++;
      }
   }
   else if (pPF->numDsmRes != 0)
   {
      pPF->pfid = STB_DMXGrabPIDFilter( sfm->demux, pPF->pid, sfm->setup.hwFilterCallback );
      if (pPF->pfid == INVALID_PIDFILT)
      {
         ERRLOG("Demux PID filter failed to be added! (0x%x)", pPF->pid)
      }
      else
      {
         DBGLOG((DF_FILTER|DF_HWSF), "ADDDED PID=0x%x PFID=0x%x", pPF->pid, pPF->pfid)
         SetupSecFilterArray( pPF, pPF->dmxSecArray, sfm->setup.maxSecFiltersPerPid );
         RequestNewActives( sfm, pPF->pfid, pPF->dmxSecArray, sfm->setup.maxSecFiltersPerPid );
         STB_DMXStartPIDFilter( sfm->demux, pPF->pfid );
      }
   }
   FUNCTION_FINISH(SFM_FilterUpdate)
}

/**
 * @brief   Change DSM-CC section filter priority
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_RequireSection.
 *          This function should be in same tthread as add and remove.
 * @param   H_SfmInstance     sfm            SFM instance handle.
 * @param   void*             filterHandle   SFM Filter handle
 * @param   H_DscSFRef      dsmSfRef       DSM-CC SF reference handle
 * @param   E_SFPriority priority       New filter priority
 * @return  void
 */
void SFM_DsmccFilterPriorityChange( H_SfmInstance sfm, void *filterHandle,
   H_DscSFRef dsmSfRef,
   E_SFPriority priority )
{
   S_DsmResource *pDsmRes;
   FUNCTION_START(SFM_DsmccFilterPriorityChange)
   pDsmRes = (S_DsmResource *)filterHandle;
   ASSERT( pDsmRes->dmxPidIndex != INVALID_NDX )
   ASSERT( pDsmRes->dsmSfRef == dsmSfRef )
   if (pDsmRes->dmxPidIndex != INVALID_NDX &&
       pDsmRes->dsmSfRef == dsmSfRef)
   {
      /* Only need to lock mutex when actually changing list data */
      sfm->setup.mutexLock( sfm->setup.sfmMutex );
      pDsmRes->priority = (pDsmRes->priority & PRI_EXACT_MASK) |
         priority << PRIORITY_SHIFT;
      sfm->setup.mutexUnlock( sfm->setup.sfmMutex );
   }

   FUNCTION_FINISH(SFM_DsmccFilterPriorityChange)
}

/**
 * @brief   This function performs minimal checking of section header data to
 *          find out whether SFM requires this section.
 *          Number of header bytes required is SECTION_HEADER_LENGTH.
 *          It assumes CRC has been checked and is passing valid DSM-CC sections
 *          So the first byte of data must be table id 0x3B, 0x3C or 0x3D.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_AddSectionFilter and SFM_DelSectionFilter.
 *          It does NOT call any DSM-CC core functions.
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   PIDFILT    pfid        PID filter with incoming section
 * @param   U8BIT*         pHeader     Pointer to section header (min 8 bytes)
 * @param   void**         phBuffer    Pointer to SFM section buffer handle
 * @return  E_SFM_STATUS - status of required update
 */
E_SFM_STATUS SFMFilterRequiresSection( H_SfmInstance sfm, PIDFILT pfid,
   U16BIT teid, U8BIT tid, U8BIT vers,
   void **phBuffer )
{
   S_DmxPidFilter *pDmxPid;
   S_DsmResource *pDsmRes;
   S_PidTable *pPidTbl;
   U16BIT hash;
   U8BIT priority = 0xFF;
   E_SFM_STATUS result;
   FUNCTION_START(SFM_FilterRequiresSection)
   DBGLOG(DF_HWSF, "pfid=%x tid=%x xid=%x", pfid, tid, teid)
   pDmxPid = RetrievePidFilter( sfm->dmxPidArray, sfm->setup.maxPidFilters, pfid );
   if (pDmxPid == NULL)
   {
      /* This section is for another SFM instance, and not this one */
      /* DBGLOG(DF_MAIN,"Unrecognised filter ID 0x%x", pfid) */
      *phBuffer = NULL;
      result = SFM_IGNORE;
   }
   else
   {
      switch (tid)
      {
         case 0x3B: pPidTbl = &pDmxPid->pt3B; break;
         case 0x3C: pPidTbl = &pDmxPid->pt3C; break;
         case 0x3D: pPidTbl = &pDmxPid->pt3D; break;
         default:
            ERRLOG("Unsupported table id 0x%x", tid) *
            phBuffer = NULL;
            return SFM_ERROR;
      }
      hash = teid % HASH_SIZE;
      sfm->setup.mutexLock( sfm->setup.sfmMutex );
      /* check exact match list */
      pDsmRes = pPidTbl->dsmlists[hash];
      while (pDsmRes != NULL)
      {
         if (pDsmRes->tex.id == teid)
         {
            priority = pDsmRes->priority;
            break;
         }
         pDsmRes = pDsmRes->next;
      }
      if (pDsmRes == NULL)
      {
         /* Try wild card list */
         pDsmRes = pPidTbl->dsmlists[HASH_SIZE];
         while (pDsmRes != NULL)
         {
            if (pDsmRes->tex.id == (teid & pDsmRes->tex.mask))
            {
               priority = pDsmRes->priority;
               break;
            }
            pDsmRes = pDsmRes->next;
         }
      }
      sfm->setup.mutexUnlock( sfm->setup.sfmMutex );
      if (pDsmRes != NULL)
      {
         switch (priority >> PRIORITY_SHIFT)
         {
            case SF_PRIORITY_LOW:
            case SF_PRIORITY_CLEAR:
               *phBuffer = pDsmRes;
               result = SFM_UPDATE_LOW;
               DBGLOG(DF_MAIN, "pfid=%x tid=%x xid=%x LOW priority", pfid, tid, teid)
               break;
            case SF_PRIORITY_HIGH:
            case SF_PRIORITY_DIRECT:
               *phBuffer = pDsmRes;
               result = SFM_UPDATE_HIGH;
               DBGLOG(DF_MAIN, "pfid=%x tid=%x xid=%x HIGH priority", pfid, tid, teid)
               break;
            default:
               *phBuffer = NULL;
               ERRLOG("Bad priority value 0x%x", priority)
               result = SFM_ERROR;
         }
      }
      else if (pPidTbl->cachetable != NULL)
      {
         *phBuffer = SFMCacheRetrieveFilter( sfm, pPidTbl->cachetable, teid, vers );
         result = SFM_UPDATE_CACHE;
      }
      else
      {
         DBGLOG(DF_FILTER, "Drop sec Tid=%x TEid=0x%x ver=%d", tid, teid, vers);
         *phBuffer = NULL;
         result = SFM_IGNORE;
      }
   }
   FUNCTION_FINISH(SFM_FilterRequiresSection)
   return result;
}

/**
 * @brief   Get DSM-CC filter request and pass section buffer to DSM-CC with
 *          it's filter handle
 *          This should only be called after calling SFM_FilterRequiresSection
 *          with it returning SFM_UPDATE_LOW or SFM_UPDATE_HIGH
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   U8BIT*         pSection    Pointer to whole section data buffer
 * @param   void*          hBuffer     SFM section handle returned by
 *                                     SFM_FilterRequiresSection
 * @return  void.
 */
void SFMFilterProcessSection( H_SfmInstance sfm, U8BIT *pSection, H_DsmResource pDsmRes )
{
   FUNCTION_START(SFM_FilterProcessSection)
   if (pDsmRes->tid == *pSection)
   {
      U16BIT txid = (U16BIT)((pSection[3] << 8) | pSection[4]);
      if (pDsmRes->tex.id == txid ||
          pDsmRes->tex.id == (txid & pDsmRes->tex.mask))
      {
         CDSM_SysProcessPrivateSection( sfm->dsmcc, pSection, pDsmRes->dsmSfRef );
      }
   }
   FUNCTION_FINISH(SFM_FilterProcessSection)
}

BOOLEAN SFMFilterValidHandle( H_SfmInstance sfm, void *hBuffer )
{
   H_DsmResource pDsmRes;
   BOOLEAN result;
   pDsmRes = hBuffer;
   if (pDsmRes >= sfm->dmxResFirst && pDsmRes <= sfm->dmxResLast)
   {
      result = TRUE;
   }
   else
   {
      result = FALSE;
   }
   return result;
}

H_DsmResource SFMFilterDsmResource( H_SfmInstance sfm, U16BIT dsmref )
{
   return sfm->dmxResFirst + dsmref;
}

