/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Section Filter Manager (SFM): filters
 * @file    sfm_filter.h
 * @date    16-10-2013
 * @author  Adam Sturtridge
 */
#ifndef _SFM_FILTER_H
#define _SFM_FILTER_H

#include "sfm_main.h"


/**
 * @brief   Create Filter resources for Section Filter Manager instance
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  BOOLEAN                       TRUE if success
 */
BOOLEAN SFMFilterCreateResources( H_SfmInstance sfm );

/**
 * @brief   Initialise caching references
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  void
 */
void SFMFilterInitCaching( H_SfmInstance sfm );

/**
 * @brief   Destroy Filter resources for Section Filter Manager instance
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @return  void
 */
void SFMFilterDestroyResources( H_SfmInstance sfm );


/**
 * @brief   This function performs minimal checking of section header data to
 *          find out whether SFM requires this section.
 *          Number of header bytes required is SECTION_HEADER_LENGTH.
 *          It assumes CRC has been checked and is passing valid DSM-CC sections
 *          So the first byte of data must be table id 0x3B, 0x3C or 0x3D.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_AddSectionFilter and SFM_DelSectionFilter.
 *          It does NOT call any DSM-CC core functions.
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   PIDFILT    pfid        PID filter with incoming section
 * @param   U8BIT          tableId     Table Id of incoming section
 * @param   U16BIT         tableIdExt  Table Id Extension of incoming section
 * @return  E_SFM_STATUS - status of required update
 */
E_SFM_STATUS SFMFilterRequiresSection( H_SfmInstance sfm, PIDFILT pfid,
   U16BIT teid, U8BIT tid, U8BIT vers,
   void **phBuffer );

/**
 * @brief   Get DSM-CC filter request and pass section buffer to DSM-CC with
 *          it's filter handle
 *          This should only be called after calling SFM_FilterRequiresSection
 *          with it returning SFM_UPDATE_LOW or SFM_UPDATE_HIGH
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   U8BIT*         pSection    Pointer to whole section data buffer
 * @param   void*          hBuffer     SFM section handle returned by
 *                                     SFM_FilterRequiresSection
 * @return  void.
 */
void SFMFilterProcessSection( H_SfmInstance sfm, U8BIT *pSection,
   H_DsmResource pDsmRes );

BOOLEAN SFMFilterValidHandle( H_SfmInstance sfm, void *ptr );

H_DsmResource SFMFilterDsmResource( H_SfmInstance sfm, U16BIT dsmref );

#endif /*_SFM_FILTER_H*/
