/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Section Filter Manager: main functions
 * @file    sfm_main.c
 * @date    16th October 2013
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

#include "sfm_main.h"
#include "sfm_filter.h"
#include "sfm_cache.h"

/*---- Constant definitions for this file-------------------------------------*/

#define KILOBYTES(n)    (n << 10)
#define MEGABYTES(n)    (n << 20)

#define INITERR(x, ...)     if (pSetup->errPrintf) \
      pSetup->errPrintf( "ERROR: " x "\n", ##__VA_ARGS__);

#define INITWRN(x, ...)     if (pSetup->errPrintf) \
      pSetup->errPrintf( "Warning: " x "\n", ##__VA_ARGS__);

/*---- Local typedef structs for this file------------------------------------*/


/*---- Local (static) variable declarations for this file---------------------*/
/*   (internal variables declared static to make them local) */

/*---- LOCAL function definitions for this file ------------------------------*/
/*   (internal functions declared static to make them local) */



/*---- GLOBAL function definitions for this file -----------------------------*/



/**
 * @brief   Create Section Filter Manager instance, using setup structure.
 * @param   S_SfmSetup     setup          setup parameters
 * @return  SFM instance.  NULL is failure.
 */
H_SfmInstance SFM_CreateInstance( S_SfmSetup *pSetup )
{
   S_SfmInstance *sfm = NULL;
   S_SfmSetup *sptr;

   FUNCTION_START(SFM_CreateInstance)
   if (pSetup != NULL)
   {
      if (pSetup->memAlloc == NULL)
      {
         INITERR("memAlloc NULL")
      }
      else if (pSetup->memFree == NULL)
      {
         INITERR("memFree NULL")
      }
      else if (pSetup->mutexLock == NULL)
      {
         INITERR("mutexLock NULL")
      }
      else if (pSetup->mutexUnlock == NULL)
      {
         INITERR("mutexUnlock NULL")
      }
      else if (pSetup->hwFilterCallback == NULL)
      {
         INITERR("hwFilterCallback NULL")
      }
      else if (pSetup->sfmMutex == NULL)
      {
         INITERR("sfmMutex NULL")
      }
      else
      {
         if (pSetup->maxPidFilters < 4)
         {
            INITWRN("maxPidFilters set to 4")
            pSetup->maxPidFilters = 4;
         }
         else if (pSetup->maxPidFilters > 32)
         {
            INITWRN("maxPidFilters set to 32")
            pSetup->maxPidFilters = 32;
         }
         if (pSetup->maxSecFiltersPerPid == 0)
         {
            INITWRN("maxSecFiltersPerPid set to 1")
            pSetup->maxSecFiltersPerPid = 1;
         }
         if (pSetup->maxAvailableSectionFilters == 0 ||
             pSetup->maxAvailableSectionFilters > NUM_SECTION_FILTERS_MAXIMUM)
         {
            INITWRN("maxAvailableSectionFilters set to %d", NUM_SECTION_FILTERS_MAXIMUM)
            pSetup->maxAvailableSectionFilters = NUM_SECTION_FILTERS_MAXIMUM;
         }
         else if (pSetup->maxAvailableSectionFilters < NUM_SECTION_FILTERS_MINIMUM)
         {
            INITWRN("maxAvailableSectionFilters set to %d", NUM_SECTION_FILTERS_MINIMUM)
            pSetup->maxAvailableSectionFilters = NUM_SECTION_FILTERS_MINIMUM;
         }
         sfm = pSetup->memAlloc( sizeof(S_SfmInstance));
         if (sfm == NULL)
         {
            INITERR("memory fail")
         }
         else
         {
            sptr = &sfm->setup;
            memcpy( sptr, pSetup, sizeof(S_SfmSetup));
            sptr++;
            memset(sptr, 0, sizeof(S_SfmInstance) - sizeof(S_SfmSetup));
            if (!SFMFilterCreateResources( sfm ))
            {
               INITERR("PID resources fail")
               sfm->setup.memFree( sfm );
               sfm = NULL;
            }
            sfm->cache = SFMCacheCreate( sfm );
            if (sfm->cache == NULL)
            {
               INITWRN("Cache initialisation failed - zero cache")
            }
            SFMFilterInitCaching( sfm );
         }
      }
   }
   FUNCTION_FINISH(SFM_CreateInstance)
   return sfm;
}

/**
 * @brief   Destroy Section Filter Manager instance, and return mutexes
 *          so that they may be destroyed by client.
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   void**         pSfmMtx     Pointer to sfmMutex provided in setup
 * @param   void**         pCchMtx     Pointer to cacheMutex provided in setup
 * @param   void**         pBufMtx     Pointer to bufferMutex provided in setup
 * @return void
 */
void SFM_DestroyInstance( H_SfmInstance sfm, void **pSfmMtx,
   void **pCchMtx, void **pBufMtx )
{
   FUNCTION_START(SFM_DestroyInstance)

   if (pSfmMtx != NULL)
   {
      *pSfmMtx = sfm->setup.sfmMutex;
   }
   if (pCchMtx != NULL)
   {
      *pCchMtx = sfm->setup.cacheMutex;
   }

   SFMCacheDestroy( sfm, pBufMtx );

   SFMFilterDestroyResources( sfm );

   sfm->setup.memFree( sfm );

   FUNCTION_FINISH(SFM_DestroyInstance)
}

/**
 * @brief   This allows controlling software to reconfigure SFM debug output.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   F_SfmPrintf    errPrintf      Error print function
 * @param   F_SfmPrintf    dbgPrintf      Debug print function
 * @param   U32BIT         dbgState       State to filter debug printing
 * @return
 */
void SFM_SetDebugConfig( H_SfmInstance sfm, F_Printf errPrintf,
   F_Printf dbgPrintf, U32BIT dbgState )
{
   FUNCTION_START(SFM_SetDebugConfig)
   ASSERT(sfm != NULL)
   sfm->setup.errPrintf = errPrintf;
   sfm->setup.dbgPrintf = dbgPrintf;
   sfm->setup.dbgState = dbgState;
   FUNCTION_FINISH(SFM_SetDebugConfig)
}

/**
 * @brief   This allows controlling software to reconfigure SFM debug output.
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   U32BIT         dbgState       State to filter debug printing
 * @return
 */
void SFM_SetDebugState( H_SfmInstance sfm, U32BIT dbgState )
{
   sfm->setup.dbgState = dbgState;
}

/**
 * @brief   Set instance handle for DSM-CC that SFM is supporting.
 * @param   H_SfmInstance       sfm       SFM instance handle.
 * @param   H_DsmCoreInst   dsmcc     DSM-CC instance being supported
 * @return
 */
void SFM_SetDsmInstance( H_SfmInstance sfm, H_DsmCoreInst dsmcc )
{
   FUNCTION_START(SFM_SetDsmInstance)
   sfm->dsmcc = dsmcc;
   FUNCTION_FINISH(SFM_SetDsmInstance)
}

/**
 * @brief   Set handle to be passed to F_DvpFilterSetup and F_DvpFilterRemove
 * @param   H_SfmInstance  sfm            SFM instance handle.
 * @param   DMXREF       demux          Demux handle used by platform
 * @return
 */
void SFM_SetDemuxHandle( H_SfmInstance sfm, DMXREF demux )
{
   FUNCTION_START(SFM_SetDemuxHandle)
   sfm->demux = demux;
   FUNCTION_FINISH(SFM_SetDemuxHandle)
}

/*------------- Functions process section data from platform -----------------*/

/**
 * @brief   This function performs minimal checking of section header data to
 *          find out whether SFM requires this section.
 *          Number of header bytes required is SECTION_HEADER_LENGTH.
 *          It assumes CRC has been checked and is passing valid DSM-CC sections
 *          So the first byte of data must be table id 0x3B, 0x3C or 0x3D.
 *          Note for multi-tasking: this function uses mutex locking to protect
 *          shared data with SFM_AddSectionFilter and SFM_DelSectionFilter.
 *          It does NOT call any DSM-CC core functions.
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   PIDFILT        pfid        PID filter with incoming section
 * @param   U8BIT*         pHeader     Pointer to section header (min 8 bytes)
 * @param   void**         phBuffer    Pointer to SFM section buffer handle
 * @return  E_SFM_STATUS - status of required update
 */
E_SFM_STATUS SFM_RequireSection( H_SfmInstance sfm, PIDFILT pfid,
   U8BIT *pHeader, void **phBuffer )
{
   E_SFM_STATUS result;
   FUNCTION_START(SFM_RequireSection)

   // dsmcc sections always have indicator for current/next set to current ...
   ASSERT((pHeader[5] & 0x1) == 0x1 )

   result = SFMFilterRequiresSection( sfm, pfid,
         (U16BIT)((pHeader[3] << 8) | pHeader[4]),       // table id extension
         pHeader[0],                                     // table id
         (pHeader[5] >> 1) & 0x1F,                       // version (used by cache)
         phBuffer );

   DBGLOG(DF_HWSF, "pfid=%x, sz=0x%x t=0x%x e=0x%x, %d, %d RESULT=%d", pfid,
      (((int)(pHeader[1] & 0x0f) << 8) | pHeader[2]) + 3, *pHeader,
      ((pHeader[3] << 8) | pHeader[4]), pHeader[6] + 1, pHeader[7] + 1, result)

   if (result == SFM_UPDATE_CACHE)
   {
      SFMCacheAddBlock( sfm, pHeader[1], pHeader[6], phBuffer );
      if (*phBuffer == NULL)
      {
         result = SFM_IGNORE;
      }
   }
   FUNCTION_FINISH(SFM_RequireSection)
   return result;
}

/**
 * @brief   Process required section buffer.
 *          This should only be called after calling SFM_RequireSection and
 *          with it returning SFM_UPDATE_...
 *          This function calls DSM-CC core functions.
 * @param   H_SfmInstance  sfm         SFM instance handle.
 * @param   U8BIT*         pSection    Pointer to whole section data buffer
 * @param   void*          hBuffer     SFM section handle returned by
 *                                     SFM_RequireSection
 * @return  void.
 */
void SFM_ProcessSection( H_SfmInstance sfm, U8BIT *pSection, void *hBuffer )
{
   FUNCTION_START(SFM_AddSectionFilter)
   if (SFMFilterValidHandle( sfm, hBuffer ))
   {
      SFMFilterProcessSection( sfm, pSection, hBuffer );
   }
   else if (SFMCacheValidHandle( sfm, hBuffer ))
   {
      SFMCacheProcessSection( sfm, pSection, hBuffer );
   }
   else
   {
      ERRLOG("Invalid buffer handle")
   }
   FUNCTION_FINISH(SFM_AddSectionFilter)
}

