/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Dsmcc's Service Information Query API
 * @file    dsiq.h
 * @date    28 September 2013
 * @author  Adam Sturtridge
 */
#ifndef _DSIQ_H_
#define _DSIQ_H_

#include "cldsmtypes.h"
#include "dmxtypes.h"
#include "stdfuncs.h"
#include "siqfuncs.h"
#include "dsm_debug.h"

#define UNKNOWN_AIT_TYPE   0x7FFF
#define UNKNOWN_AIT_VERS   0x7F

typedef enum
{
   SIQ_ERROR,
   SIQ_IGNORE,
   SIQ_UPDATE
} E_SIQ_STATUS;

/*------------------- Functions supplied to SIQ at setup ---------------------*/

/**
 * @brief   SIQ requests table data - normally PMT, but could be UNT.
 *          Table data should be given to appropriate function e.g. SIQ_ProcessPMT.
 *          It should always supply it after this function is called,
 *          and then, whenever the version changes.
 *          This state should continue until F_DvpCancelTable is called.
 * @param   ref External reference or handle
 * @param   pid PID for this table
 * @param   xid Table extension ID
 * @param   tid Table ID (e.g. PMT is 0x02)
 * @return  zero is failure, non-zero is success
 */
typedef U32BIT (*F_DvpRequestTable)( void *ref, U16BIT pid, U16BIT xid, U8BIT tid );

/**
 * @brief   Cancel request for Table data made by F_DvpRequestTable()
 * @param   ref External reference or handle
 * @param   rid Request Id returned by F_DvpRequestTable()
 * @return  void
 */
typedef void (*F_DvpCancelTable)( void *ref, U32BIT rid );


/*---------------- Setup structure for SIQ initialisation --------------------*/

typedef struct
{
   /* First five functions here are manditory */
   F_MemAlloc memAlloc;
   F_MemFree memFree;

   F_DvpRequestTable dvpRequestTable;
   F_DvpCancelTable dvpCancelTable;
   void *pmtref;

   F_PmtParseInit parsePmtInit;
   F_ParseDataBcastId parseDataBroadcastId;
   F_PmtParseDone parsePmtDone;

   F_ParseSsuSelectorBytes parseSsuSelectorBytes;

   /* notifyAitInfo is optional and may be set to NULL */
   F_NotifyAitInfo notifyAitInfo;

   /* errPrintf and dbgPrintf functions are only used in a debug build.
    * These are optional and may be set to NULL */
   F_Printf errPrintf;
   F_Printf dbgPrintf;

   /* dbgState controls type of debug output for dbgPrintf
    * For valid debug filter states see above */
   U32BIT dbgState;
} S_SiqSetup;


typedef struct s_SiqCachePat *H_SiqCachePat;
typedef struct s_SiqCachePmt *H_SiqCachePmt;
typedef struct s_SiqPmtTable *H_SiqPmtTable;

/*----------------------- Functions that configure SIQ -----------------------*/

/**
 * @brief   Create SIQ instance.
 * @param   S_SiqSetup          setup     setup parameters
 * @return  SIQ instance. NULL is failure.
 */
H_SiqInstance SIQ_CreateInstance( S_SiqSetup *pSetup );

/**
 * @brief   Destroy instance created by SIQ_CreateInstance
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @return
 */
void SIQ_DestroyInstance( H_SiqInstance siq );

/**
 * @brief   Set Debug config for SIQ.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   F_Printf       errPrintf      Error print function
 * @param   F_Printf       dbgPrintf      Debug print function
 * @param   U32BIT         dbgState       State to filter debug printing
 * @return
 */
void SIQ_SetDebugConfig( H_SiqInstance siq, F_Printf errPrintf,
   F_Printf dbgPrintf, U32BIT dbgState );

/**
 * @brief   Set Debug state for SIQ.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U32BIT         dbgState       State to filter debug printing
 * @return
 */
void SIQ_SetDebugState( H_SiqInstance siq, U32BIT dbgState );

/**
 * @brief   Set instance handle for DSM-CC that SIQ is supporting.
 * @param   H_SiqInstance       siq       SIQ instance handle.
 * @param   H_DsmCoreInst       dsm       DSM-CC instance being supported
 * @return
 */
void SIQ_SetDsmInstance( H_SiqInstance siq, H_DsmCoreInst dsm );


/*------------------ Functions that update SIQ from DVB ----------------------*/

/**
 * @brief   Provide SIQ with latest PAT data.
 *          Calling this function is optional. However, if not used, then the
 *          platform must use SIQ_ServiceRemoved and SIQ_TransportChanged.
 * @param   H_SiqInstance siq SIQ instance handle.
 * @param   U8BIT* dptr PAT section data
 * @return  none.
 */
void SIQ_UpdatePat( H_SiqInstance siq, U8BIT *dptr );

/**
 * @brief   Inform SIQ when tuning to another transport stream.
 *          Not required if PAT is supplied to SIQ.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U16BIT         origNetworkId  original network Id for transport stream
 * @param   U16BIT         transportId    Id of new transport stream
 * @return  void
 */
void SIQ_TransportChanged( H_SiqInstance siq, U16BIT origNetworkId, U16BIT transportId );

/**
 * @brief   Ask SIQ whether PMT section data is required.
 *          It assumes CRC has been checked and is passing valid PMT sections
 *          So the first byte of data must be PMT table id (i.e. 0x02).
 *          If SIQ does not receive PAT then this function adds SIQ knowledge
 *          of the service.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U8BIT*         pmt            raw PMT section data header
 * @return  E_SIQ_STATUS
 */
E_SIQ_STATUS SIQ_RequirePmt( H_SiqInstance siq, U8BIT *pmt,
   U16BIT *pSid, U8BIT *pVer );

/**
 * @brief   Inform SIQ when a service has been removed from PAT.
 *          Not required if PAT is supplied to SIQ.
 *          This may initiate calls to DSM-CC callback functions for Load
 *          Carousel or Load Object, where states are 'aborted'.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U16BIT         serviceId      Service Id of removed service
 * @return  void
 */
void SIQ_ServiceRemoved( H_SiqInstance siq, U16BIT serviceId );

/**
 * @brief   Provide PMT section data to SIQ.
 *          It assumes CRC has been checked and is passing valid PMT sections
 *          So the first byte of data must be PMT table id (i.e. 0x02).
 *          If SIQ does not receive PAT then this function adds SIQ knowledge
 *          of the service.
 * @param   H_SiqInstance  siqInstance    SIQ instance handle.
 * @param   U8BIT*         pmt            raw PMT section data
 * @return  void
 */
H_SiqPmtTable SIQ_ParsePmt( H_SiqInstance siq, U8BIT *pmt );

/**
 * @brief   Provide PMT section data to SIQ.
 *          It assumes CRC has been checked and is passing valid PMT sections
 *          So the first byte of data must be PMT table id (i.e. 0x02).
 *          If SIQ does not receive PAT then this function adds SIQ knowledge
 *          of the service.
 *          This may initiate calls to F_ParseDataBcastId(), and may be
 *          followed by DSM-CC callback functions for Load Carousel or Load
 *          Object, where states can be any (including 'aborted').
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U8BIT*         pmt            raw PMT section data
 * @return  E_SIQ_STATUS
 */
void SIQ_ProcessPmt( H_SiqInstance siq, H_SiqPmtTable hNewTable,
   U16BIT serviceId, U8BIT version );

/**
 * @brief   Clear cached PMT information.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U16BIT         serviceId      Service whose PMT needs to be cleared
 * @return  void
 */
void SIQ_CacheClearPmt( H_SiqInstance siq, U16BIT serviceId );


/*----------------- Functions provided to DSM-CC component -------------------*/

/**
 * @brief   Start DSM-CC query request.
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @return
 */
E_DscError SIQ_RequestQueryStart( H_SiqInstance siq, P_SIQueryRequest pQuery,
   H_SIQueryRef clDsmSIQueryRef, void *clDsmSIUserData,
   P_SIQueryResult pResult );

/**
 * @brief   Stop DSM-CC query request.
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @return
 */
void SIQ_RequestQueryStop(  H_SiqInstance siq,
   void *queryHandle, H_SIQueryRef clDsmSIQueryRef );

/**
 * @brief   Subscribe a component to receive notification of any changes or
 *          alterations to the service information of a specified service.
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U16BIT         serviceId      Service to be monitored.
 * @return  E_DscError
 */
E_DscError SIQ_ServiceInfoChangeSubscribe( H_SiqInstance siq, U16BIT serviceId );

/**
 * @brief   Unsubscribe component to previously subscribed service
 *          change or alteration updates.
 *          Called by DSM-CC component, and given in clDsmSysCreate setup.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U16BIT         serviceId      Service being monitored.
 * @return  void
 */
void SIQ_ServiceInfoChangeUnsubscribe( H_SiqInstance siq, U16BIT serviceId );

#endif /*_DSIQ_H_*/

