/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Service Information Query: cache functionality
 * @file    siq_cache.c
 * @date    29-09-2013
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

/* third party header files */
/* SIQ header files */
#include "siq_debug.h"
#include "siq_cache.h"
#include "siq_main.h"

#include "cldsmcc.h"


/*---constant definitions for this file--------------------------------------*/

#define SIQ_DEFAULT_PMT_TOTAL    64
#define VERSION_CHANGED          0xff
#define PMT_TID                  0x02

/*---local typedef structs for this file-------------------------------------*/

typedef struct s_SiqCachePat
{
   struct s_SiqCachePat *next;
   U16BIT transId;
   U16BIT serviceTotal;
   U8BIT version;
} S_SiqCachePat;

typedef struct s_SiqCachePmt
{
   H_SiqPmtTable table;
   U32BIT requestId;
   U16BIT serviceId;
   U16BIT pmtpid;
   U8BIT subscribed;
   U8BIT querying;
   U8BIT refresh;
   U8BIT version;
} S_SiqCachePmt;


/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

/*------------Local function definitions for this file-----------------------*/
/*   (internal functions declared static to make them local) */


static S_SiqCachePat* FindPat( S_SiqInstance *siq, U16BIT transId )
{
   S_SiqCachePat *pPat = siq->patList;
   while (pPat != NULL && pPat->transId != transId)
   {
      pPat = pPat->next;
   }
   return pPat;
}

static void ClearPmts( S_SiqInstance *siq, S_SiqCachePmt *pPmt, U16BIT total )
{
   DBGLOG(DS_CACHE, " ")
   while (total--)
   {
      if (pPmt->table != NULL)
      {
         DBGLOG(DS_CACHE, "pPmt=%p serviceId=%d", pPmt, pPmt->serviceId )
         SIQ_PmtDestroyTable( siq, pPmt->table );
         pPmt->table = NULL;
      }
      pPmt++;
   }
}

static void RemovePat( S_SiqInstance *siq, S_SiqCachePat *pPat )
{
   S_SiqCachePat **ppPat;
   ppPat = &(siq->patList);
   while (*ppPat != NULL)
   {
      if (pPat == *ppPat)
      {
         //ClearPmts( siq, (S_SiqCachePmt*)(pPat + 1), pPat->serviceTotal );
         *ppPat = pPat->next;
         siq->setup.memFree( pPat );
         break;
      }
      ppPat = &((*ppPat)->next);
   }
}

static void ClearPmtArray( S_SiqInstance *siq )
{
   S_SiqCachePmt *pPmt;
   U16BIT total;

   FUNCTION_START(ClearPmtArray)
   pPmt = siq->pmtArray;
   if (pPmt == NULL)
   {
      ERRLOG("")
      ASSERT( siq->pmtTotal == 0 )
   }
   else if (siq->pmtCount != 0)
   {
      total = siq->pmtTotal;
      while (total--)
      {
         if (pPmt->table != NULL)
         {
            DBGLOG(DS_CACHE, "pPmt=%p serviceId=%d", pPmt, pPmt->serviceId )
            SIQ_PmtDestroyTable( siq, pPmt->table );
            pPmt->table = NULL;
            siq->pmtCount--;
            if (siq->patList == NULL)
            {
               /* Default Pmt array */
               pPmt->serviceId = 0;
            }
            if (siq->pmtCount == 0)
            {
               break;
            }
         }
         pPmt++;
      }
   }
   FUNCTION_FINISH(ClearPmtArray)
}

static S_SiqCachePmt* GetCachePmt( S_SiqInstance *siq, U16BIT serviceId )
{
   S_SiqCachePmt *pPmt, *result;
   U16BIT total;

   FUNCTION_START(GetCachePmt)
   result = NULL;
   pPmt = siq->pmtArray;
   total = siq->pmtTotal;
   if (pPmt == NULL)
   {
      ERRLOG("")
      ASSERT( total == 0 )
   }
   else
   {
      while (total--)
      {
         if (pPmt->serviceId == serviceId)
         {
            result = pPmt;
            break;
         }
         pPmt++;
      }
      if (result == NULL && siq->patList == NULL)
      {
         /* Only executed when not receiving PAT data */
         pPmt = siq->pmtArray;
         total = siq->pmtTotal;
         while (total--)
         {
            if (pPmt->serviceId == 0)
            {
               result = pPmt;
               pPmt->serviceId = serviceId;
               break;
            }
            pPmt++;
         }
      }
   }
   FUNCTION_FINISH(GetCachePmt)
   return result;
}

/**
 * @brief   Inform SIQ when a service has been removed from PAT.
 *          Not required if PAT is supplied to SIQ.
 * @param   H_SiqInstance  siqInstance    SIQ instance handle.
 * @param   U16BIT         serviceId      Service Id of removed service
 * @return  void
 */
static void ServiceRemoved( H_SiqInstance siq, S_SiqCachePmt *pCachePmt )
{
   if (pCachePmt->table != NULL)
   {
      DBGLOG(DS_CACHE, "pPmt=%p serviceId=%d", pCachePmt, pCachePmt->serviceId )
      SIQ_PmtDestroyTable( siq, pCachePmt->table );
      pCachePmt->table = NULL;
      siq->pmtCount--;
   }
   if (pCachePmt->subscribed)
   {
      CDSM_SysProcessSIChangeEvent( siq->dsmInstance, SICHANGE_SERVICE_DELETED, pCachePmt->serviceId, 0 );
      pCachePmt->subscribed = 0;
   }
   if (pCachePmt->requestId)
   {
      siq->setup.dvpCancelTable( siq->setup.pmtref, pCachePmt->requestId ) ;
      pCachePmt->requestId = 0;
      pCachePmt->refresh = 0;
   }
   /* Delete outstanding queries for this service ? */
   SIQ_DeleteQueriesOnService( siq, pCachePmt->serviceId );
   if (siq->patList == NULL)
   {
      /* Remove from default Pmt array */
      pCachePmt->serviceId = 0;
   }
}

/*------------GLOBAL function definitions for this file----------------------*/

BOOLEAN SIQ_CacheInit( S_SiqInstance *siq )
{
   BOOLEAN result = TRUE;
   /* add default PMT array */
   siq->pmtArray = siq->setup.memAlloc( SIQ_DEFAULT_PMT_TOTAL * sizeof(S_SiqCachePmt));
   if (siq->pmtArray == NULL)
   {
      result = FALSE;
   }
   else
   {
      memset( siq->pmtArray, 0, SIQ_DEFAULT_PMT_TOTAL * sizeof(S_SiqCachePmt));
      siq->pmtTotal = SIQ_DEFAULT_PMT_TOTAL;
   }
   return result;
}

void SIQ_CacheExit( S_SiqInstance *siq )
{
   S_SiqCachePat *pPat, *pNxt;
   S_SiqCachePmt *pPmt;
   U16BIT total;

   FUNCTION_START(SIQ_CacheExit)
   pPmt = siq->pmtArray;
   total = siq->pmtTotal;
   while (total--)
   {
      if (pPmt->subscribed)
      {
         ERRLOG("Leaked SI change subscribe srv_id=%d", pPmt->serviceId)
      }
      if (pPmt->requestId)
      {
         siq->setup.dvpCancelTable( siq->setup.pmtref, pPmt->requestId );
         pPmt->requestId = 0;
      }
      if (pPmt->table != NULL)
      {
         SIQ_PmtDestroyTable(siq, pPmt->table);
         pPmt->table = NULL;
      }
      pPmt++;
   }
   pPat = siq->patList;
   if (pPat != NULL)
   {
      DBGLOG(DS_CACHE, " ")
      do
      {
         ClearPmts( siq, (S_SiqCachePmt *)(pPat + 1), pPat->serviceTotal );
         pNxt = pPat->next;
         siq->setup.memFree( pPat );
         pPat = pNxt;
      }
      while (pPat != NULL);
   }
   else if (siq->pmtArray != NULL)
   {
      siq->setup.memFree(siq->pmtArray);
   }
   FUNCTION_FINISH(SIQ_CacheExit)
}

/**
 * @brief   Provide SIQ with latest PAT data.
 *          Calling this function is optional. However, if not used, then the
 *          platform must use SIQ_ServiceRemoved and SIQ_TransportChanged.
 * @param   H_SiqInstance siq SIQ instance handle.
 * @param   U8BIT* dptr PAT section data
 * @return  none.
 */
void SIQ_UpdatePat( H_SiqInstance siq, U8BIT *dptr )
{
   S_SiqCachePat *pPat, *pCachePat;
   S_SiqCachePmt *newPmts;
   U16BIT id, len, total;
   U8BIT version;
   len = ((dptr[1] & 0x0F) << 8 | dptr[2]);
   if (len < 9)
   {
      ERRLOG("PAT length too small %d", len)
   }
   else
   {
      id = dptr[3] << 8 | dptr[4];
      version = (dptr[5] >> 1) & 0x1f;
      pCachePat = FindPat( siq, id );

      DBGLOG(DS_PAT, " trns_id=0x%x, pat=%p version=%u len=%u", id, pCachePat, version, len)

      if (pCachePat == NULL || pCachePat->version != version)
      {
         len -= 9;   // ignore section header (5 bytes) and CRC-32 (4 bytes)
         len >>= 2;  // the number of program elements: (program number(2 bytes) + PMT PID(2 bytes)
         dptr += 8;  // point to the beginning of the program elements
         pPat = siq->setup.memAlloc( sizeof(S_SiqCachePat) + (len * sizeof(S_SiqCachePmt) + sizeof(S_SiqCachePmt)/*fixed SSU PMT*/));
         if (pPat != NULL)
         {
            memset( pPat, 0, sizeof(S_SiqCachePat) + (len * sizeof(S_SiqCachePmt)));
            pPat->transId = id;
            pPat->version = version;
            newPmts = (S_SiqCachePmt *)(pPat + 1);
            total = 0;
            for (; len--; dptr += 4)
            {
               id = (dptr[0] << 8) | dptr[1];
               if (id != 0)
               {
                  newPmts[total].serviceId = id;
                  newPmts[total].pmtpid = ((dptr[2] & 0x1f) << 8) | dptr[3];
                  DBGLOG(DS_PAT, "sid=%x pmtpid=%x", id, newPmts[total].pmtpid)
                  total++;
               }
            }
            // add fixed SSU PMT
            {
               newPmts[total].serviceId = 0xFFFF;  // program number 0xFFFF is reserved for indicating the NIT PID
               newPmts[total].pmtpid = 7902;  // fixed SSU PMT PID
               DBGLOG(DS_PAT, "sid=%x pmtpid=%x", id, newPmts[total].pmtpid)
               total++;
            }
            pPat->serviceTotal = total;
            if (pCachePat != NULL)
            {
               S_SiqCachePmt *pCachePmt;
               /* Pat version change */
               for (pCachePmt = (S_SiqCachePmt *)(pCachePat + 1), len = pCachePat->serviceTotal; len--; pCachePmt++)
               {
                  U16BIT ndx;
                  id = INVALID_PID;
                  for (ndx = 0; ndx < pPat->serviceTotal; ndx++)
                  {
                     if (pCachePmt->serviceId == newPmts[ndx].serviceId)
                     {
                        id = newPmts[ndx].pmtpid;
                        newPmts[ndx] = *pCachePmt;
                        pCachePmt->table = NULL;
                        newPmts[ndx].pmtpid = id;
                        break;
                     }
                  }
                  if (id == INVALID_PID)
                  {
                     /* Service not found in new PAT */
                     ServiceRemoved( siq, pCachePmt );
                  }
               }
               RemovePat(siq, pCachePat);
            }
            else if (siq->patList == NULL && siq->pmtArray != NULL)
            {
               /* First time received a PAT - clear default PMT list */
               siq->setup.memFree( siq->pmtArray );
            }
            pPat->next = siq->patList;
            siq->patList = pPat;
            pCachePat = pPat;
         }
      }

      siq->pmtArray = (H_SiqCachePmt)(pCachePat + 1);
      siq->pmtTotal = pCachePat->serviceTotal;
      DBGLOG(DS_PAT, "PAT-transId=%x SIQ-transId=%x; pmtTotal=%u", pCachePat->transId, siq->transportId, siq->pmtTotal)
   }
}

/**
 * @brief   Inform SIQ when a demux is tuning to another transport stream.
 *          Not required if PAT is supplied to SIQ.
 * @param   H_SiqInstance  siqInstance    SIQ instance handle.
 * @param   U16BIT         transportId    Id of new transport stream
 * @return  void
 */
void SIQ_TransportChanged( H_SiqInstance siq, U16BIT origNetworkId, U16BIT transportId )
{
   if (siq->origNetworkId != origNetworkId ||
       siq->transportId != transportId)
   {
      siq->origNetworkId = origNetworkId;
      siq->transportId = transportId;
      SIQ_CacheClear( siq );
   }
}

/**
 * @brief   Inform SIQ when a service has been removed from PAT.
 *          Not required if PAT is supplied to SIQ.
 * @param   H_SiqInstance  siqInstance    SIQ instance handle.
 * @param   U16BIT         serviceId      Service Id of removed service
 * @return  void
 */
void SIQ_ServiceRemoved( H_SiqInstance siq, U16BIT serviceId )
{
   S_SiqCachePmt *pCachePmt;
   pCachePmt = GetCachePmt( siq, serviceId );
   if (pCachePmt == NULL)
   {
      ERRLOG("No Cache slot for serviceId=0x%x", serviceId)
   }
   else
   {
      ServiceRemoved( siq, pCachePmt );
   }
}

/**
 * @brief   Ask SIQ whether PMT section data is required.
 *          It assumes CRC has been checked and is passing valid PMT sections
 *          So the first byte of data must be PMT table id (i.e. 0x02).
 *          If SIQ does not receive PAT then this function adds SIQ knowledge
 *          of the service.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U8BIT*         pmt            raw PMT section data header
 * @return  E_SIQ_STATUS
 */
E_SIQ_STATUS SIQ_RequirePmt( H_SiqInstance siq, U8BIT *pmt,
   U16BIT *pSid, U8BIT *pVer )
{
   S_SiqCachePmt *pCachePmt;
   U8BIT version;
   U16BIT serviceId;
   E_SIQ_STATUS status;

   if (pmt == NULL)
   {
      ERRLOG("NULL PMT data")
      status = SIQ_ERROR;
   }
   else if (*pmt != PMT_TID)
   {
      ERRLOG("Bad PMT data! (table id=0x%x)", *pmt )
      status = SIQ_ERROR;
   }
   else
   {
      serviceId = pmt[3] << 8 | pmt[4];
      pCachePmt = GetCachePmt( siq, serviceId );
      if (pCachePmt == NULL)
      {
         ERRLOG("No Cache slot for serviceId=0x%x", serviceId)
         status = SIQ_ERROR;
      }
      else
      {
         if (siq->patList == NULL)
         {
            DBGLOG(DS_PAT, "Not received PAT")
         }
         version = (pmt[5] >> 1) & 0x1f;
         if (pCachePmt->table != NULL && pCachePmt->version == version)
         {
            DBGLOG(DS_CACHE, "Ignore PMT srv_id=0x%x vers=%d", serviceId, version)
            status = SIQ_IGNORE;
         }
         else
         {
            DBGLOG(DS_CACHE, "new PMT srv_id=0x%x vers=%d, was v=%d tbl=%p", serviceId, version, pCachePmt->version, pCachePmt->table)
            status = SIQ_UPDATE;
            *pSid = serviceId;
            *pVer = version;
         }
      }
   }
   return status;
}

/**
 * @brief   Provide PMT section data to SIQ.
 *          It assumes CRC has been checked and is passing valid PMT sections
 *          So the first byte of data must be PMT table id (i.e. 0x02).
 *          If SIQ does not receive PAT then this function adds SIQ knowledge
 *          of the service.
 * @param   H_SiqInstance  siqInstance    SIQ instance handle.
 * @param   U8BIT*         pmt            raw PMT section data
 * @return  void
 */
H_SiqPmtTable SIQ_ParsePmt( H_SiqInstance siq, U8BIT *pmt )
{
   return SIQ_PmtCreateTable( siq, pmt );
}

/**
 * @brief   Provide PMT section data to SIQ.
 *          It assumes CRC has been checked and is passing valid PMT sections
 *          So the first byte of data must be PMT table id (i.e. 0x02).
 *          If SIQ does not receive PAT then this function adds SIQ knowledge
 *          of the service.
 * @param   H_SiqInstance  siqInstance    SIQ instance handle.
 * @param   U8BIT*         pmt            raw PMT section data
 * @return  void
 */
void SIQ_ProcessPmt( H_SiqInstance siq, H_SiqPmtTable hNewTable,
   U16BIT serviceId, U8BIT version )
{
   H_SiqCachePmt pCachePmt;
   H_SiqPmtTable hOldTable;

   pCachePmt = GetCachePmt( siq, serviceId );
   if (pCachePmt == NULL)
   {
      ERRLOG("No Cache PMT!")
   }
   else
   {
      hOldTable = pCachePmt->table;
      DBGLOG(DS_CACHE, "new PMT srv_id=0x%x vers=%d old=%p requestId=%x",
         serviceId, version, hOldTable, pCachePmt->requestId )
      DBGLOG(DS_CACHE, "pCachePmt=%p newTable=%p", pCachePmt, hNewTable )
      pCachePmt->table = hNewTable;
      pCachePmt->version = version;
      if (hOldTable != NULL)
      {
         ASSERT( !pCachePmt->querying );
         if (pCachePmt->subscribed)
         {
            SIQ_PmtCheckTableChange( siq, hOldTable, hNewTable, serviceId );
         }
         SIQ_PmtDestroyTable( siq, hOldTable );
      }
      else
      {
         siq->pmtCount++;
         if (pCachePmt->requestId)
         {
            ASSERT( hOldTable == NULL );
            pCachePmt->querying = 0;
            SIQ_QueryPmtReceive( siq, serviceId, hNewTable );
            if (!pCachePmt->subscribed)
            {
               siq->setup.dvpCancelTable( siq->setup.pmtref, pCachePmt->requestId );
               pCachePmt->requestId = 0;
            }
         }
      }
   }
}

H_SiqPmtTable SIQ_CacheRetrievePmt( S_SiqInstance *siq, U16BIT serviceId, U32BIT *gettingPmt )
{
   S_SiqCachePmt *pCachePmt;
   H_SiqPmtTable result;

   FUNCTION_START(SIQ_CacheRetrievePmt)
   pCachePmt = GetCachePmt( siq, serviceId );
   if (pCachePmt == NULL)
   {
      result = NULL;
      *gettingPmt = 0;
   }
   else
   {
      result = pCachePmt->table;
      if (result != NULL)
      {
         *gettingPmt = 0;
      }
      else
      {
         if (!pCachePmt->requestId)
         {
            pCachePmt->querying = 1;
            pCachePmt->requestId = siq->setup.dvpRequestTable(siq->setup.pmtref, pCachePmt->pmtpid, serviceId, PMT_TID );
         }
         *gettingPmt = pCachePmt->requestId;
      }
   }
   FUNCTION_FINISH(SIQ_CacheRetrievePmt)
   return result;
}

/**
 * @brief   Clear cached PMT information.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U16BIT         serviceId      Service whose PMT needs to be cleared
 * @return  void
 */
void SIQ_CacheClearPmt( H_SiqInstance siq, U16BIT serviceId )
{
   S_SiqCachePmt *pPmt;
   U16BIT total;

   FUNCTION_START(SIQ_CacheClearPmt)
   DBGLOG(DS_CACHE, " serviceId=%d", serviceId )
   pPmt = siq->pmtArray;
   total = siq->pmtTotal;
   if (pPmt == NULL)
   {
      ERRLOG("")
      ASSERT( total == 0 )
   }
   while (total--)
   {
      if (pPmt->serviceId == serviceId)
      {
         DBGLOG(DS_CACHE, "pPmt=%p serviceId=%d", pPmt, pPmt->serviceId )
         SIQ_PmtDestroyTable( siq, pPmt->table );
         pPmt->table = NULL;
         pPmt->refresh = 1;
         siq->pmtCount--;
         if (siq->patList == NULL)
         {
            /* Remove from default PMT array */
            pPmt->serviceId = 0;
         }
         break;
      }
      pPmt++;
   }
   FUNCTION_FINISH(SIQ_CacheClearPmt)
}

void SIQ_CacheClear( S_SiqInstance *siq )
{
   ClearPmtArray(siq);
   if (siq->patList != NULL)
   {
      DBGLOG((DS_CACHE|DS_PAT), "old pmtTot=%u", siq->pmtTotal)
      siq->pmtArray = NULL;
      siq->pmtTotal = 0;
   }
}

/**
 * @brief    Subscribe a component to receive notification of any changes or
 *           alterations to the service information of a specified service.
 * @param    H_DsmCoreInst   instance          DMSM-CC instance handle.
 * @param    U16BIT              service_id        Service to monitor.
 * @return   E_DscError
 */
E_DscError SIQ_ServiceInfoChangeSubscribe( S_SiqInstance *siq, U16BIT serviceId )
{
   S_SiqCachePmt *pCachePmt;
   E_DscError err;

   FUNCTION_START(SIQ_ServiceInfoChangeSubscribe)
   pCachePmt = GetCachePmt( siq, serviceId );
   if (pCachePmt == NULL)
   {
      err = CLDSM_ERR_SI_SUBSCRIBE_FAILURE;
      ERRLOG("no pmt, serviceId=%x", serviceId)
   }
   else
   {
      if (pCachePmt->refresh)
      {
         pCachePmt->refresh = 0;
         if (pCachePmt->requestId)
         {
            siq->setup.dvpCancelTable( siq->setup.pmtref, pCachePmt->requestId );
            pCachePmt->requestId = 0;
         }
      }
      if (!pCachePmt->requestId)
      {
         pCachePmt->requestId = siq->setup.dvpRequestTable( siq->setup.pmtref, pCachePmt->pmtpid, serviceId, PMT_TID );
      }
      if (pCachePmt->requestId)
      {
         err = CLDSM_OK;
         pCachePmt->subscribed++;
      }
      else
      {
         err = CLDSM_ERR_SI_SUBSCRIBE_FAILURE;
         ERRLOG("rqst fail, serviceId=%x", serviceId)
      }
   }
   FUNCTION_FINISH(SIQ_ServiceInfoChangeSubscribe)
   return err;
}

/**
 * @brief    Unsubscribe component to previously subscribed service
 *           change or alteration updates.
 * @param    H_DsmCoreInst   instance          DMSM-CC instance handle.
 * @param    U16BIT              service_id        Service to monitor.
 * @return   n/a
 */
void SIQ_ServiceInfoChangeUnsubscribe( S_SiqInstance *siq, U16BIT serviceId )
{
   S_SiqCachePmt *pCachePmt;

   FUNCTION_START(SIQ_ServiceInfoChangeSubscribe)
   pCachePmt = GetCachePmt( siq, serviceId );
   if (pCachePmt != NULL && pCachePmt->subscribed)
   {
      pCachePmt->subscribed--;
      ASSERT(pCachePmt->requestId != 0)
      if (!pCachePmt->subscribed && !pCachePmt->querying)
      {
         /* Tell DVB stack to stop sending PMT data */
         siq->setup.dvpCancelTable( siq->setup.pmtref, pCachePmt->requestId );
         pCachePmt->requestId = 0;
      }
   }
   FUNCTION_FINISH(SIQ_ServiceInfoChangeSubscribe)
}

