/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Service Information Query: main functions
 * @file    siq_main.c
 * @date    28th September 2013
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

/* third party header files */
#include "cldsmcc.h"

/* SIQ header files */
#include "siq_cache.h"
#include "siq_debug.h"
#include "siq_main.h"
#include "siq_pmt.h"

/*---constant definitions for this file--------------------------------------*/

#define NEWLOG(x, ...)     if (pSetup->errPrintf) \
      pSetup->errPrintf( x "\n", ##__VA_ARGS__);

/*---local typedef structs for this file-------------------------------------*/

typedef struct s_SiqRequests
{
   struct s_SiqRequests *next;
   S_SIQueryRequest query;
   H_SIQueryRef dsmQueryRef;
   void *dsmUserData;
} S_SiqRequests;


/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

/*------------Local function definitions for this file-----------------------*/
/*   (internal functions declared static to make them local) */


/*------------GLOBAL function definitions for this file----------------------*/


/**
 * @brief
 * @param    S_SiqSetup          setup   setup parameters
 * @return
 */
H_SiqInstance SIQ_CreateInstance( S_SiqSetup *pSetup )
{
   S_SiqInstance *siq = NULL;
   S_SiqSetup *iptr;

   FUNCTION_START(SIQ_CreateInstance)
   if (pSetup != NULL)
   {
      if (pSetup->memAlloc == NULL)
      {
         NEWLOG("memAlloc NULL")
      }
      else if (pSetup->memFree == NULL)
      {
         NEWLOG("memFree NULL")
      }
      else if (pSetup->dvpRequestTable == NULL)
      {
         NEWLOG("dvpRequestTable NULL")
      }
      else if (pSetup->dvpCancelTable == NULL)
      {
         NEWLOG("dvpCancelTable NULL")
      }
      else if (pSetup->parseDataBroadcastId == NULL &&
               pSetup->parseSsuSelectorBytes == NULL)
      {
         NEWLOG("parseDataBroadcastId or parseSsuSelectorBytes is NULL")
      }
      else
      {
         siq = pSetup->memAlloc( sizeof(S_SiqInstance));
         if (siq != NULL)
         {
            iptr = &siq->setup;
            memcpy( iptr, pSetup, sizeof(S_SiqSetup));
            iptr++;
            memset(iptr, 0, sizeof(S_SiqInstance) - sizeof(S_SiqSetup));

            if (!SIQ_CacheInit( siq ))
            {
               pSetup->memFree( siq );
               siq = NULL;
            }
         }
      }
   }
   FUNCTION_FINISH(SIQ_CreateInstance)
   return siq;
}

/**
 * @brief   Set Debug config for SIQ.
 * @param   H_SiqInstance  siq           SIQ instance handle.
 * @param   F_SiqPrintf    errPrintf      Error print function
 * @param   F_SiqPrintf    dbgPrintf      Debug print function
 * @param   U32BIT         dbgState       State to filter debug printing
 * @return
 */
void SIQ_SetDebugConfig( H_SiqInstance siq, F_Printf errPrintf,
   F_Printf dbgPrintf, U32BIT dbgState )
{
   ASSERT(siq != NULL)
   siq->setup.errPrintf = errPrintf;
   siq->setup.dbgPrintf = dbgPrintf;
   siq->setup.dbgState = dbgState;
}

/**
 * @brief   Set Debug state for SIQ.
 * @param   H_SiqInstance  siq            SIQ instance handle.
 * @param   U32BIT         dbgState       State to filter debug printing
 * @return
 */
void SIQ_SetDebugState( H_SiqInstance siq, U32BIT dbgState )
{
   ASSERT(siq != NULL)
   siq->setup.dbgState = dbgState;
}

/**
 * @brief   Destroy instance created by SIQ_CreateInstance
 * @param   H_SiqInstance       siqInstance    SIQ instance handle.
 * @return
 */
void SIQ_DestroyInstance( H_SiqInstance siq )
{
   S_SiqRequests *pRequest, *pr;

   pRequest = siq->queryHead;
   while (pRequest != NULL)
   {
      ERRLOG("Leaked Query kind=%d srv_id=%d", pRequest->query.kind, pRequest->query.serviceId)
      pr = pRequest;
      pRequest = pRequest->next;
      siq->setup.memFree( pr );
   }

   SIQ_CacheExit( siq );

   siq->setup.memFree( siq );
}

/**
 * @brief
 * @param    void*               siqInstance    SIQ instance handle.
 * @param    H_DsmCoreInst   dsmInstance    DSM-CC instance being supported
 * @return
 */
void SIQ_SetDsmInstance( H_SiqInstance siq, H_DsmCoreInst dsmInstance )
{
   FUNCTION_START(SIQ_SetDsmInstance)
   siq->dsmInstance = dsmInstance;
   FUNCTION_FINISH(SIQ_SetDsmInstance)
}

/**
 * @brief   Start query request.
 *          Called by DSM-CC component.
 * @param   H_SiqInstance siqInstance SIQ instance handle.
 * @return
 */
E_DscError SIQ_RequestQueryStart( S_SiqInstance *siq, P_SIQueryRequest pQuery,
   H_SIQueryRef dsmQueryRef, void *dsmUserData,
   P_SIQueryResult pResult )
{
   S_SiqRequests *pRequest;
   H_SiqPmtTable hpmt;
   E_DscError err;
   U32BIT gettingPmt;
   /* no need to check for duplicates requests - dsmcc lib does that */
   hpmt = SIQ_CacheRetrievePmt( siq, pQuery->serviceId, &gettingPmt );
   if (hpmt != NULL)
   {
      SIQ_PmtProcessQuery( siq, pQuery, hpmt, pResult );
      if (pResult->kind == SIRESULT_FAILURE)
      {
         DBGLOG(DS_QUERY, "SIQ_PmtProcessQuery(kind=%d) failed", pQuery->kind)
         err = CLDSM_ERR_SI_QUERY_FAILED;
      }
      else
      {
         err = CLDSM_OK;
      }
   }
   else if (!gettingPmt)
   {
      ERRLOG("Eek! SIQ_CacheRetrievePmt(serv_id=%d) failed, and not getting PMT either! Q-kind=%d", pQuery->serviceId, pQuery->kind)
      err = CLDSM_ERR_INVALID_SIQUERY_STATUS;
   }
   else
   {
      pRequest = siq->setup.memAlloc( sizeof(S_SiqRequests));
      if (pRequest == NULL)
      {
         err = CLDSM_ERR_ALLOC_FAILED;
      }
      else
      {
         DBGLOG(DS_QUERY, "Pending Query request serviceId=%d kind=%d", pQuery->serviceId, pQuery->kind)
         pRequest->query = *pQuery;
         pRequest->dsmQueryRef = dsmQueryRef;
         pRequest->dsmUserData = dsmUserData;

         pRequest->next = siq->queryHead;
         siq->queryHead = pRequest;

         pResult->kind = SIRESULT_PENDING;
         pResult->data.queryHandle = pRequest;
         err = CLDSM_PENDING;
      }
   }
   return err;
}

/**
 * @brief   Stop query request.
 *          Called by DSM-CC component.
 * @param   H_SiqInstance siqInstance SIQ instance handle.
 * @return
 */
void SIQ_RequestQueryStop(  S_SiqInstance *siq,
   void *queryHandle, H_SIQueryRef dsmQueryRef )
{
   S_SiqRequests **ppRequest, *pRequest;

   ppRequest = &(siq->queryHead);
   while (*ppRequest != NULL)
   {
      if (*ppRequest == queryHandle)
      {
         pRequest = *ppRequest;
         ASSERT( pRequest->dsmQueryRef == dsmQueryRef );
         *ppRequest = pRequest->next;
         siq->setup.memFree( pRequest );
         break;
      }
      ppRequest = &((*ppRequest)->next);
   }
}

void SIQ_DeleteQueriesOnService(S_SiqInstance *siq, U16BIT serviceId)
{
   S_SiqRequests **ppRequest, *pRequest;

   ppRequest = &(siq->queryHead);
   while (*ppRequest != NULL)
   {
      pRequest = *ppRequest;
      if (pRequest->query.serviceId == serviceId)
      {
         *ppRequest = pRequest->next;
         siq->setup.memFree( pRequest );
      }
      else
      {
         ppRequest = &((*ppRequest)->next);
      }
   }
}

/**
 * @brief
 * @param    S_SiqInstance* siq    Pointer to SIQ instance
 * @return
 */
void SIQ_QueryPmtReceive(S_SiqInstance *siq, U16BIT serviceId, H_SiqPmtTable hpmt)
{
   S_SiqRequests **ppRequest, *pRequest;
   S_SIQueryResult result;

   //DBGLOG(DS_QUERY,"serviceId=%d",serviceId)
   pRequest = siq->queryHead;
   ppRequest = &(siq->queryHead);
   while (pRequest != NULL)
   {
      DBGLOG(DS_QUERY, "pRequest=%p ppRequest=%p", pRequest, ppRequest)
      if (pRequest->query.serviceId == serviceId)
      {
         *ppRequest = pRequest->next;
         SIQ_PmtProcessQuery( siq, &pRequest->query, hpmt, &result );
         if (CDSM_SysProcessSIQueryEvent( siq->dsmInstance,
                pRequest->dsmQueryRef, pRequest->dsmUserData, &result ) != CLDSM_OK)
         {
            ERRLOG("clDsmSysProcessSIQueryEvent: failed")
         }
         siq->setup.memFree( pRequest );
      }
      else
      {
         ppRequest = &(pRequest->next);
         DBGLOG(DS_QUERY, "ppRequest=%p", ppRequest)
      }
      pRequest = *ppRequest;
   }
}

