/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Service Information Query: PMT parsing functionality
 * @file    siq_pmt.c
 * @date    30th Sepember 2013
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

/* compiler library header files */

/* third party header files */

/* OBS header files */
#include "siq_debug.h"
#include "siq_main.h"
#include "siq_pmt.h"

#include "cldsmcc.h"

/*---constant definitions for this file--------------------------------------*/

#define DBC_ID_SSU            0x000A
#define DBC_ID_MHEG5          0x0106
#define DBC_ID_OAD            0x0111
#define DBC_ID_HBBTV          0x0123
#define DBC_ID_MHP0           0x00F0
#define DBC_ID_MHP1           0x00F1

#define DVB_OUI               0x00015A

/*---local typedef structs for this file-------------------------------------*/

typedef enum
{
   ST_VIDEO1 = 0x01,
   ST_VIDEO2 = 0x02,
   ST_AUDIO1 = 0x03,
   ST_AUDIO2 = 0x04,
   ST_PRIVATE = 0x05,
   ST_PES_PRV = 0x06,
   ST_MHEG = 0x07,
   ST_DATA_A = 0x0a,
   ST_DATA_B = 0x0b,
   ST_DATA_C = 0x0c,
   ST_DATA_D = 0x0d,
   ST_AUX = 0x0e,
   ST_AAC = 0x0f,
   ST_HEAAC = 0x11,
   ST_H264 = 0x1b
} E_STREAM_TYPE;

typedef enum
{
   SUT_PROPRIETORY,
   SUT_SIMPLE_SSU,
   SUT_WITH_UNT,
   SUT_RETURN_UNT
} E_SsuUpdType;

enum
{
   APP_TYP_MHEG5 = 0x0008,
   APP_TYP_HBBTV = 0x0010,
};

enum
{
   CAROUSEL_ID_DESCRIPTOR        = 0x13,
   DEFERRED_ASSOC_DESCRIPTOR     = 0x15,
   STREAM_ID_DESCRIPTOR          = 0x52,
   DATA_BROADCAST_ID_DESCRIPTOR  = 0x66,
   APP_SIGNALLING_DESCRIPTOR     = 0x6f
};

typedef struct s_SsuInfo *P_SsuInfo;

typedef struct s_SsuInfo
{
   P_SsuInfo next;
   U32BIT oui;
   E_SsuUpdType type;
   BOOLEAN hasVers;
   U8BIT version;
   U8BIT selectorLen;
} S_SsuInfo;

typedef struct s_PmtStream
{
   U16BIT pid;
   U16BIT numtags;
   U32BIT carouselId;
   U16BIT bootPref;
   U16BIT dbcId;
   U8BIT *cTags;
   P_SsuInfo ssuInfo;
} S_PmtStream;

typedef struct s_Deferred
{
   U16BIT transportId;
   U16BIT serviceId;
   U16BIT origNetId;
   U16BIT numAssocTags;
   U16BIT *assocTags;
} S_Deferred;

typedef struct s_SiqPmtTable
{
   U8BIT num_deferred;
   U8BIT num_streams;
   S_Deferred *deferreds;
   S_PmtStream *streams;
} S_SiqPmtTable;

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local) */


/*---------------------- Local function definitions -------------------------*/


/**
 * @brief   Search for AIT info - i.e. application signalling descriptor as
 *          defined by ETSI TS-102809, section 5.3.5.1.
 *          and then call callback function
 * @param
 * @return
 */
static void ParseForAitInfo( S_SiqInstance *siq, U16BIT sid,
   U16BIT pid, U8BIT *dptr, U8BIT *dend )
{
   U8BIT dlen, dtag;
   FUNCTION_START(ParseForAitInfo)
   while (dptr < dend)
   {
      dtag = *dptr++;
      dlen = *dptr++;
      DBGLOG(DS_PMT, " dtag=0x%x dlen=%d, data=0x%x 0x%x 0x%x 0x%x", dtag, dlen, dptr[0], dptr[1], dptr[2], dptr[3])
      if (dtag == APP_SIGNALLING_DESCRIPTOR)
      {
         if (dlen == 0)
         {
            siq->setup.notifyAitInfo( sid, pid, UNKNOWN_AIT_TYPE, UNKNOWN_AIT_VERS );
         }
         else
         {
            while (dlen >= 3)
            {
               siq->setup.notifyAitInfo( sid, pid, ((dptr[0] & 0x7f) << 8) + dptr[1], dptr[2] & 0x1f );
               dptr += 3;
               dlen -= 3;
            }
         }
      }
      dptr += dlen;
   }
   FUNCTION_FINISH(ParseForAitInfo)
}

void ParseSystemSoftwareUpdateInfo( S_SiqInstance *siq, S_PmtStream *pStream, U8BIT *dptr, U8BIT dlen, U8BIT **pxtr_ptr )
{
   P_SsuInfo pSsuInfo;
   U32BIT oui;
   U8BIT slen;
   FUNCTION_START(ParseSystemSoftwareUpdateInfo)
   ASSERT(*pxtr_ptr)
   pSsuInfo = (P_SsuInfo)*pxtr_ptr;
   oui = *dptr++;
   oui = (oui << 8) | *dptr++;
   oui = (oui << 8) | *dptr++;
   pSsuInfo->oui = oui;
   pSsuInfo->type = (E_SsuUpdType)(*dptr & 0x0f);
   dptr++;
   pSsuInfo->hasVers = (*dptr & 0x20)? TRUE : FALSE;
   pSsuInfo->version = *dptr & 0x1F;
   dptr++;
   slen = *dptr++;
   pSsuInfo->selectorLen = slen;
   if (slen)
   {
      /* store selector bytes for later */
      memcpy(pSsuInfo+1,dptr,slen);
   }
   pSsuInfo->next = pStream->ssuInfo;
   pStream->ssuInfo = pSsuInfo;
   slen += sizeof(S_SsuInfo) + 3;
   slen &= ~(0x03);
   *pxtr_ptr += slen;
   FUNCTION_FINISH(ParseSystemSoftwareUpdateInfo)
}

/**
 * @brief   Search for user's prefered Data Broadcast Id
 * @param
 * @return  U16BIT - preference value
 */
static BOOLEAN ParseForDataBroadcastIds( S_SiqInstance *siq, H_PmtRef pmtref,
   S_PmtStream *pStream, U8BIT *dptr, U8BIT *dend, U8BIT **pxtr_ptr )
{
   U16BIT prefer, preferred;
   U8BIT dlen, dtag;
   U16BIT dbcId;
   BOOLEAN found;
   FUNCTION_START(ParseForDataBroadcastIds)
   found = FALSE;
   preferred = 0;
   while (dptr < dend)
   {
      dtag = *dptr++;
      dlen = *dptr++;
      if (dtag == DATA_BROADCAST_ID_DESCRIPTOR)
      {
         dbcId = dptr[0] << 8 | dptr[1];
         switch (dbcId)
         {
            case DBC_ID_SSU:
            {
               pStream->dbcId = dbcId;
               ParseSystemSoftwareUpdateInfo( siq, pStream, dptr+3, *(dptr+2), pxtr_ptr );
               found = TRUE;
               break;
            }
            case DBC_ID_MHEG5:
            case DBC_ID_OAD:
            case DBC_ID_HBBTV:
            case DBC_ID_MHP0:
            case DBC_ID_MHP1:
            {
               if (pStream->carouselId != INVALID_CAROUSEL_ID && siq->setup.parseDataBroadcastId != NULL)
               {
                  DBGLOG(DS_PMT, "numtags=%d dbcId=%x",pStream->numtags,dbcId)
                  prefer = (U16BIT)siq->setup.parseDataBroadcastId( pmtref, pStream->carouselId, dptr, dlen );
                  if (preferred < prefer)
                  {
                     preferred = prefer;
                     pStream->bootPref = preferred;
                     pStream->dbcId = dbcId;
                     found = TRUE;
                  }
               }
               break;
            }
            default:
            {
               DBGLOG(DS_PMT, "Unknown data broadcast Id 0x%x")
               break;
            }
         }
      }
      dptr += dlen;
   }
   FUNCTION_FINISH(ParseForDataBroadcastIds)
   return found;
}

/**
 * @brief   Get extra size required by Data broadcast descriptors
 * @param
 * @return  U16BIT - size
 */
static BOOLEAN GetDataBroadcastExtraSize( U8BIT *dptr, U8BIT *dend, U16BIT *psz )
{
   U16BIT size = 0;
   U16BIT dbcId;
   U8BIT dlen, dtag, slen;
   BOOLEAN found = FALSE;
   while (dptr < dend)
   {
      dtag = *dptr++;
      dlen = *dptr++;
      if (dtag == DATA_BROADCAST_ID_DESCRIPTOR)
      {
         dbcId = dptr[0] << 8 | dptr[1];
         switch (dbcId)
         {
            case DBC_ID_SSU:
            {
               slen = sizeof(S_SsuInfo) + dptr[8] + 3;
               slen &= ~(0x03);
               size +=  slen;
               found = TRUE;
               break;
            }
            case DBC_ID_MHEG5:
            case DBC_ID_OAD:
            case DBC_ID_HBBTV:
            case DBC_ID_MHP0:
            case DBC_ID_MHP1:
            {
               found = TRUE;
               break;
            }
            default:
               break;
         }
      }
      dptr += dlen;
   }
   return found;
}

/**
 * @brief   Check that PMT stream has Stream Id descriptor
 * @param
 * @return  BOOLEAN - TRUE if found
 */
static U16BIT CountStreamIdDesc( U8BIT *dptr, U8BIT *dend )
{
   U16BIT count = 0;
   U8BIT dlen, dtag;
   while (dptr < dend)
   {
      dtag = *dptr++;
      dlen = *dptr++;
      if (dtag == STREAM_ID_DESCRIPTOR)
      {
         count++;
      }
      dptr += dlen;
   }
   return count;
}

/**
 * @brief   Get component tag from Stream Id descriptor
 * @param
 * @return  BOOLEAN - TRUE if found
 */
static U16BIT GetComponentTags( U8BIT *dptr, U8BIT *dend, U8BIT *ctags )
{
   U16BIT count = 0;
   U8BIT dlen, dtag;
   while (dptr < dend)
   {
      dtag = *dptr++;
      dlen = *dptr++;
      if (dtag == STREAM_ID_DESCRIPTOR)
      {
         ctags[count] = *dptr;
         count++;
      }
      dptr += dlen;
   }
   return count;
}

/**
 * @brief   Get Carouel Id from descriptor for the PMT stream
 * @param
 * @return  U32BIT - carousel id (or invalid if not found)
 */
static U32BIT GetCarouselInfo( U8BIT *dptr, U8BIT *dend /*, formatId?*/)
{
   U32BIT carouselId = INVALID_CAROUSEL_ID;
   U8BIT dlen, dtag;
   while (dptr < dend)
   {
      dtag = *dptr++;
      dlen = *dptr++;
      if (dtag == CAROUSEL_ID_DESCRIPTOR)
      {
         carouselId = *dptr;
         dptr++;
         carouselId <<= 8;
         carouselId |= *dptr;
         dptr++;
         carouselId <<= 8;
         carouselId |= *dptr;
         dptr++;
         carouselId <<= 8;
         carouselId |= *dptr;
         /* format ID is optional in HbbTv, and not required by MHEG *
         dptr++;
         stream_ptr->format_id = *dptr;
         if ( stream_ptr->format_id == 0x01 )
         {}
         */
         break;
      }
      dptr += dlen;
   }
   return carouselId;
}

/**
 * @brief   Get size required to store deferred information
 * @param
 * @return  U16BIT - size required
 */
static U16BIT DeferredSize( U8BIT *dptr, U8BIT *dend, U8BIT *pNum )
{
   U16BIT size = 0;
   U8BIT dlen, dtag, num = 0;
   while (dptr < dend)
   {
      dtag = *dptr++;
      dlen = *dptr++;
      if (dtag == DEFERRED_ASSOC_DESCRIPTOR)
      {
         /* add space for assoc tag array with assocTag_bytes (i.e. *dptr) */
         size += sizeof(S_Deferred) + *dptr;
         num++;
      }
      dptr += dlen;
   }
   *pNum = num;
   return size;
}

/**
 * @brief   Parse for deferred services in this PMT stream
 * @param
 * @return  void
 */
static U16BIT ParseDeferreds( U8BIT *dptr, U8BIT *dend, S_Deferred *pDeferred, U16BIT *atags )
{
   U16BIT size = 0;
   U16BIT i;
   U8BIT dlen, dtag;
   while (dptr < dend)
   {
      dtag = *dptr++;
      dlen = *dptr++;
      if (dtag == DEFERRED_ASSOC_DESCRIPTOR && dlen > 8)
      {
         pDeferred->numAssocTags = *dptr / 2;
         pDeferred->assocTags = atags;
         dptr++;
         for (i = pDeferred->numAssocTags; i--; atags++)
         {
            *atags = (dptr[0] << 8) | dptr[1];
            size += 2;
            dptr += 2;
            dlen -= 2;
         }
         pDeferred->transportId = (dptr[0] << 8) | dptr[1];
         dptr += 2;
         pDeferred->serviceId = (dptr[0] << 8) | dptr[1];
         dptr += 2;
         pDeferred->origNetId = (dptr[0] << 8) | dptr[1];
         dptr += 2;
         dlen -= 7;
         pDeferred++;
      }
      dptr += dlen;
   }
   return size;
}

/*---global function definitions---------------------------------------------*/

/**
 * @brief    Create PMT structure for monitoring (life-cycle, NB_Info)
 * @param    U8BIT* data                  Raw PMT section
 * @return   S_SiqPmtTable structure
 */
H_SiqPmtTable SIQ_PmtCreateTable( S_SiqInstance *siq, U8BIT *data )
{
   S_SiqPmtTable *pmt_table;
   S_PmtStream *stream_ptr;
   H_PmtRef pmtref;
   U8BIT *dptr, *section_end, *dloop_end, *ctags, *xtr_ptr;
   U8BIT num_streams, num_deferred;
   U8BIT stream_type;
   U16BIT section_len, dloop_len;
   U16BIT pid, size, sid, cnt, xtr_size;

   FUNCTION_START(PMT_CreateTable)

   dptr = data;
   section_len = (dptr[1] << 8 | dptr[2]) & 0xfff;
   dloop_len = (dptr[10] << 8 | dptr[11]) & 0xfff;

   sid = dptr[3] << 8 | dptr[4]; /* service id*/

   DBGLOG(DS_PMT, "sid=0x%x tid=0x%x sec_len=%d first desc len=%d", sid,
      *data, section_len, dloop_len )

   section_end = data + section_len - 4; // -4 for crc
   /* Search for deferred service assoc in first descriptor loop */
   dptr = data + 12;
   dloop_end = dptr + dloop_len;

   size = sizeof(S_SiqPmtTable) + DeferredSize( dptr, dloop_end, &num_deferred );
   xtr_size = 0;
   DBGLOG(DS_PMT, "PMT size=%u num_deferred=%u", size, num_deferred)

   dptr = dloop_end;

   /* count number of streams of interest */
   num_streams = 0;
   while (dptr < section_end)
   {
      stream_type = *dptr;
      // use 'pid' to temporarily hold desc loop length
      pid = ((dptr[3] & 0x0f) << 8) | dptr[4];
      dptr += 5;
      dloop_end = dptr + pid;
      switch (stream_type)
      {
         case ST_MHEG: case ST_DATA_A:
            ERRLOG(" Unhandled stream type=0x%x", stream_type)
         default:
         case ST_PRIVATE:
            break;

         case ST_PES_PRV:
            /* Telefonica SSU */
            DBGLOG(DS_PMT, "TF SSU")
            num_streams++;
            size += sizeof(S_PmtStream);
            break;

         case ST_DATA_B:
         case ST_DATA_C:
         case ST_DATA_D:
         {
            U16BIT dbsz;
            cnt = CountStreamIdDesc(dptr, dloop_end);
            if (GetDataBroadcastExtraSize(dptr, dloop_end, &dbsz) || cnt)
            {
               DBGLOG(DS_PMT, "S-typ %u cnt=%u dbsz=%u", stream_type, cnt, dbsz)
               size += sizeof(S_PmtStream) + cnt;
               xtr_size += dbsz;
               num_streams++;
            }
         }
         break;
      }
      dptr = dloop_end;
   }
   DBGLOG(DS_PMT, "PMT size=%u xtr_size=%u num_streams=%u", size, xtr_size, num_streams)
   size += xtr_size;

   pmt_table = (S_SiqPmtTable *)siq->setup.memAlloc( size );
   if (pmt_table == NULL)
   {
      ERRLOG("Memory failure")
   }
   else
   {
      memset( pmt_table, 0, size );
      pmt_table->deferreds = (S_Deferred *)(pmt_table + 1);
      pmt_table->num_deferred = num_deferred;
      stream_ptr = (S_PmtStream *)(pmt_table->deferreds + num_deferred);
      pmt_table->streams = stream_ptr;
      pmt_table->num_streams = num_streams;
      xtr_ptr= (U8BIT*)(stream_ptr + num_streams);
      ctags  = xtr_ptr + xtr_size;

      if (siq->setup.parsePmtInit)
      {
         /* Inform client about to start */
         pmtref = siq->setup.parsePmtInit( sid );
      }
      else
      {
         U_PARAM pr;
         pr.ptr = NULL;
         pr.u32 = sid;
         pmtref = pr.ptr;
      }

      /* Search for deferred service assoc in first descriptor loop */
      dptr = data + 12;
      dloop_end = dptr + dloop_len;

      ctags += ParseDeferreds( dptr, dloop_end, pmt_table->deferreds, (U16BIT *)ctags );

      dptr = dloop_end;

      /*process second descriptor loop*/
      while (dptr < section_end)
      {
         stream_type = *dptr;
         pid = ((dptr[1] & 0x1f) << 8) | dptr[2];
         dloop_len = ((dptr[3] & 0x0f) << 8) | dptr[4];
         dptr += 5;
         dloop_end = dptr + dloop_len;
         switch (stream_type)
         {
            default:
               DBGLOG(DS_PMT, "; strm typ=0x%x", stream_type)
               break;

            case ST_PRIVATE:
               if (siq->setup.notifyAitInfo)
               {
                  DBGLOG(DS_PMT, " bytes=%d, data=0x%x 0x%x 0x%x 0x%x", dloop_end - dptr, dptr[0], dptr[1], dptr[2], dptr[3])
                  ParseForAitInfo(siq, sid, pid, dptr, dloop_end);
               }
               break;

            case ST_PES_PRV:
               /* Telefonica SSU */
               DBGLOG(DS_PMT, "TF SSU data")
               if (num_streams != 0)
               {
                  stream_ptr->dbcId = DBC_ID_SSU;
                  stream_ptr->carouselId = INVALID_CAROUSEL_ID;
                  stream_ptr->pid = pid;
                  stream_ptr++;
                  num_streams--;
               }
               break;

            case ST_DATA_B:
            case ST_DATA_C:
            case ST_DATA_D:
               DBGLOG(DS_PMT, "[%d] stream typ=0x%x pid=0x%x", num_streams, stream_type, pid)
               if (num_streams != 0)
               {
                  cnt = GetComponentTags(dptr, dloop_end, ctags);
                  if (cnt != 0)
                  {
                     stream_ptr->numtags = cnt;
                     stream_ptr->cTags = ctags;
                     DBGLOG(DS_PMT, "cnt=%u, ctags[0]=%u", cnt, ctags[0])
                  }
                  else
                  {
                     DBGLOG(DS_PMT, "no ctags")
                  }
                  stream_ptr->carouselId = GetCarouselInfo( dptr, dloop_end );
                  if (ParseForDataBroadcastIds(siq, pmtref, stream_ptr, dptr, dloop_end, &xtr_ptr) || cnt)
                  {
                     ctags += cnt;
                     stream_ptr->pid = pid;
                     DBGLOG(DS_PMT, "crsl-id=0x%x bootPref=0x%x", stream_ptr->carouselId, stream_ptr->bootPref)
                     stream_ptr++;
                     num_streams--;
                  }
                  else
                  {
                     DBGLOG(DS_PMT, "no recognised dbcId and no ctags")
                  }
               }
            }
         dptr = dloop_end;
      }
      if (siq->setup.parsePmtDone)
      {
         /* Inform client have finished */
         siq->setup.parsePmtDone( pmtref );
      }
      DBGLOG(DS_PMT, "table=%p ctags=%p", pmt_table, ctags)
   }
   FUNCTION_FINISH(PMT_CreateTable)
   return pmt_table;
}

/**
 * @brief   Destroy PMT structure created by PMT_CreateTable
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   H_SiqPmtTable     table       Handle to SIQ's PMT table
 * @return  void.
 */
void SIQ_PmtDestroyTable( S_SiqInstance *siq, H_SiqPmtTable table )
{
   if (table != NULL)
   {
      DBGLOG(DS_PMT, "table=%p", table )
      siq->setup.memFree( table );
   }
}

/**
 * @brief   find PID from assoc tag or component tag
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   H_SiqPmtTable     hpmt        Handle to SIQ's PMT table
 * @param   U16BIT            assocTag    Association Tag value
 * @param   P_SIQueryResult   pResult Pointer to DSM-CC Query Result
 * @return   void.
 */
static void PmtDsmccPid( S_SiqInstance *siq, S_SiqPmtTable *pmt,
   U16BIT assocTag, P_SIQueryResult pResult )
{
   S_PmtStream *sptr;
   U16BIT cnt, i;
   FUNCTION_START(PmtDsmccPid);
   sptr = pmt->streams;
   for (cnt = pmt->num_streams; cnt--; sptr++)
   {
      for (i = 0; i != sptr->numtags; i++)
      {
         if (sptr->cTags[i] == (assocTag & 0xFF))
         {
            pResult->kind = SIQUERY_PID;
            pResult->data.pid = sptr->pid;
            return;
         }
      }
   }
   FUNCTION_FINISH(PmtDsmccPid);
}

/**
 * @brief   find PID from assoc tag or component tag
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   H_SiqPmtTable     hpmt        Handle to SIQ's PMT table
 * @param   U16BIT            assocTag    Association Tag value
 * @param   P_SIQueryResult   pResult Pointer to DSM-CC Query Result
 * @return   void.
 */
static void PmtCarouselInfo( S_SiqInstance *siq, S_SiqPmtTable *pmt,
   U8BIT cTag, P_SIQueryResult pResult )
{
   S_PmtStream *sptr;
   U16BIT cnt, i;
   FUNCTION_START(PmtCarouselInfo);
   sptr = pmt->streams;
   for (cnt = pmt->num_streams; cnt--; sptr++)
   {
      if (sptr->dbcId != DBC_ID_SSU)
      {
         for (i = 0; i != sptr->numtags; i++)
         {
            if (sptr->cTags[i] == cTag)
            {
               pResult->kind = SIQUERY_CAROUSEL_INFO;
               pResult->data.carouselInfo.pid = sptr->pid;
               pResult->data.carouselInfo.associationTag = cTag;
               pResult->data.carouselInfo.carouselId = sptr->carouselId;
               pResult->data.carouselInfo.next = NULL;
               return;
            }
         }
      }
   }
   FUNCTION_FINISH(PmtCarouselInfo);
}

/**
 * @brief   Retrieve pid and association tag for Carousel
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   H_SiqPmtTable     hpmt        Handle to SIQ's PMT table
 * @param   U32BIT            carouselId  Carousel Id
 * @param   P_SIQueryResult   pResult Pointer to DSM-CC Query Result
 * @return  void.
 */
static void PmtCarousel(S_SiqInstance *siq, S_SiqPmtTable *pmt,
   U32BIT carouselId, P_SIQueryResult pResult)
{
   S_PmtStream *sptr;
   U32BIT i;
   FUNCTION_START(PmtCarousel)
   sptr = pmt->streams;
   for (i = 0; i != pmt->num_streams; i++, sptr++)
   {
      if (sptr->carouselId == carouselId)
      {
         pResult->kind = SIQUERY_CAROUSEL;
         pResult->data.carouselInfo.pid = sptr->pid;
         if (sptr->numtags == 0)
         {
            pResult->data.carouselInfo.associationTag = INVALID_ASSOCIATION_TAG;
         }
         else
         {
            pResult->data.carouselInfo.associationTag = sptr->cTags[0];
         }
         pResult->data.carouselInfo.carouselId = carouselId;
         pResult->data.carouselInfo.next = NULL;
         DBGLOG(DS_QUERY, "  strm[%d] assocTag=0x%x crslid=0x%x", i, pResult->data.carouselInfo.associationTag, carouselId)
         break;
      }
   }
   FUNCTION_FINISH(PmtCarousel);
}

/**
 * @brief   Retrieve pid and association tag for unknown carousel
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   H_SiqPmtTable     pmt        Handle to SIQ's PMT table
 * @param   P_SIQueryResult   pResult   Pointer to DSM-CC Query Result
 * @return  void.
 */
static void PmtFindCarousel(S_SiqInstance *siq, H_SiqPmtTable pmt,
   P_SIQueryResult pResult)
{
   S_PmtStream *sptr;
   U32BIT i;
   FUNCTION_START(PmtFindCarousel)
   sptr = pmt->streams;
   for (i = 0; i != pmt->num_streams; i++, sptr++)
   {
      if (sptr->carouselId != INVALID_CAROUSEL_ID)
      {
         pResult->kind = SIQUERY_FIND_CAROUSEL;
         pResult->data.carouselInfo.pid = sptr->pid;
         if (sptr->numtags == 0)
         {
            pResult->data.carouselInfo.associationTag = INVALID_ASSOCIATION_TAG;
         }
         else
         {
            pResult->data.carouselInfo.associationTag = sptr->cTags[0];
         }
         pResult->data.carouselInfo.carouselId = sptr->carouselId;
         pResult->data.carouselInfo.next = NULL;
         DBGLOG(DS_QUERY, "  strm[%d] assocTag=0x%x crslid=0x%x", i, pResult->data.carouselInfo.associationTag, sptr->carouselId)
         break;
      }
   }
   FUNCTION_FINISH(PmtFindCarousel);
}

/**
 * @brief   Retrieve PID and association tag for prefered Boot Carousel
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   H_SiqPmtTable     hpmt        Handle to SIQ's PMT table
 * @param   U16BIT            serviceId   Service Id
 * @param   P_SIQueryResult pResult Pointer to DSM-CC Query Result
 * @return  void.
 */
static void PmtBootCarousel(S_SiqInstance *siq, S_SiqPmtTable *pmt,
   P_SIQueryResult pResult)
{
   S_PmtStream *sptr;
   U16BIT i, bootPref = 0;
   FUNCTION_START(PmtBootCarousel)
   sptr = pmt->streams;
   for (i = 0; i != pmt->num_streams; i++, sptr++)
   {
      if (sptr->dbcId != DBC_ID_SSU && bootPref < sptr->bootPref)
      {
         bootPref = sptr->bootPref;
         ASSERT( sptr->carouselId != INVALID_CAROUSEL_ID );
         pResult->kind = SIQUERY_BOOT_CAROUSEL;
         pResult->data.carouselInfo.pid = sptr->pid;
         if (sptr->numtags == 0)
         {
            pResult->data.carouselInfo.associationTag = INVALID_ASSOCIATION_TAG;
         }
         else
         {
            pResult->data.carouselInfo.associationTag = sptr->cTags[0];
         }
         pResult->data.carouselInfo.carouselId = sptr->carouselId;
         pResult->data.carouselInfo.next = NULL;
         DBGLOG(DS_QUERY, "  strm[%d] assocTag=0x%x crslid=0x%x", i, pResult->data.carouselInfo.associationTag, sptr->carouselId)
      }
   }
   FUNCTION_FINISH(PmtBootCarousel)
}

/**
 * @brief   Retrieve pid and association tag for Update Carousel
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   H_SiqPmtTable     hpmt        Handle to SIQ's PMT table
 * @param   U32BIT            carouselId  Carousel Id
 * @param   P_SIQueryResult   pResult Pointer to DSM-CC Query Result
 * @return  void.
 */
static void PmtSsuCarousel(S_SiqInstance *siq, S_SiqPmtTable *pmt,
   U32BIT oui, P_SIQueryResult pResult)
{
   S_PmtStream *sptr;
   P_SsuInfo ssuInfo;
   U32BIT i;
   P_CarouselInfo pCrslInfo;
   FUNCTION_START(PmtSsuCarousel)
   sptr = pmt->streams;
   pCrslInfo = NULL;
   for (i = 0; i != pmt->num_streams; i++, sptr++)
   {
      ssuInfo = sptr->ssuInfo;
      while (ssuInfo != NULL)
      {
         /* Call client function to parse selector bytes - which can determine whether
          * target is for correct model/version etc */
         if ((ssuInfo->oui == oui || ssuInfo->oui == DVB_OUI) &&
             (!ssuInfo->selectorLen || siq->setup.parseSsuSelectorBytes((U8BIT*)(ssuInfo+1),ssuInfo->selectorLen)))
         {
            if (pCrslInfo == NULL)
            {
               pResult->kind = SIQUERY_SSU_CAROUSEL;
               pCrslInfo = &pResult->data.carouselInfo;
            }
            else
            {
               pCrslInfo->next = siq->setup.memAlloc( sizeof(S_CarouselInfo) );
               if (pCrslInfo->next == NULL)
               {
                  ERRLOG("mem fail for extra carousel info")
                  return;
               }
               pCrslInfo = pCrslInfo->next;
            }
            pCrslInfo->pid = sptr->pid;
            pCrslInfo->carouselId = ssuInfo->oui;
            pCrslInfo->next = NULL;
            if (sptr->numtags == 0)
            {
               pCrslInfo->associationTag = INVALID_ASSOCIATION_TAG;
            }
            else
            {
               pCrslInfo->associationTag = sptr->cTags[0];
            }
            DBGLOG(DS_QUERY, "  strm[%d] ctag=0x%x crslid=0x%x", i, pCrslInfo->associationTag, oui)
         }
         ssuInfo = ssuInfo->next;
      }
   }
   if (pResult->kind == SIRESULT_FAILURE && pmt->num_streams == 1)
   {
      sptr = pmt->streams;
      if (sptr->dbcId == DBC_ID_SSU && sptr->carouselId == INVALID_CAROUSEL_ID && sptr->ssuInfo == NULL)
      {
         pResult->kind = SIQUERY_SSU_CAROUSEL;
         pResult->data.carouselInfo.pid = sptr->pid;
         pResult->data.carouselInfo.carouselId = oui;
         pResult->data.carouselInfo.associationTag = INVALID_ASSOCIATION_TAG;
         pResult->data.carouselInfo.next = NULL;
      }
   }
   FUNCTION_FINISH(PmtSsuCarousel);
}

/**
 * @brief   Parse PMT for Deferred Service details
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   H_SiqPmtTable     hpmt        Handle to SIQ's PMT table
 * @param   U16BIT            assocTag    Association Tag value
 * @param   P_SIQueryResult pResult Pointer to DSM-CC Query Result
 * @return  void.
 */
static void PmtDeferredService(S_SiqInstance *siq, S_SiqPmtTable *pmt, U16BIT assocTag,
   P_SIQueryResult pResult)
{
   S_Deferred *ds;
   U16BIT *atags, ndx;
   U8BIT num_deferred;

   FUNCTION_START(PmtDeferredService)
   num_deferred = pmt->num_deferred;
   for (ds = pmt->deferreds; num_deferred--; ds++)
   {
      atags = ds->assocTags;
      for (ndx = ds->numAssocTags; ndx--; atags++)
      {
         if (*atags == assocTag)
         {
            pResult->kind = SIQUERY_DEFERRED_SERVICE;
            pResult->data.deferredService.transport_stream_id = ds->transportId;
            pResult->data.deferredService.service_id = ds->serviceId;
            pResult->data.deferredService.original_network_id = ds->origNetId;
            num_deferred = 0; //break out of outer loop
            break;
         }
      }
   }
   FUNCTION_FINISH(PmtDeferredService)
}

/**
 * @brief   Retrieve Information required by Query request from SIQ's PMT table
 * @param   S_SiqInstance*    siq        SIQ instance handle.
 * @param   P_SIQueryRequest    pQuery      Pointer to DSM-CC Query request.
 * @param   H_SiqPmtTable     hpmt        Handle to SIQ's PMT table
 * @param   P_SIQueryResult pResult Pointer to DSM-CC Query Result
 * @return  void
 */
void SIQ_PmtProcessQuery(S_SiqInstance *siq, P_SIQueryRequest pQuery, H_SiqPmtTable hpmt,
   P_SIQueryResult pResult )
{
   pResult->kind = SIRESULT_FAILURE;
   switch (pQuery->kind)
   {
      case SIQUERY_PID:
         DBGLOG(DS_QUERY, "PID serviceId=0x%x assocTag=0x%x", pQuery->serviceId, pQuery->associationTag)
         PmtDsmccPid(siq, hpmt, pQuery->associationTag, pResult);
         break;

      case SIQUERY_BOOT_CAROUSEL:
         DBGLOG(DS_QUERY, "Boot Carousel serviceId=0x%x", pQuery->serviceId)
         PmtBootCarousel(siq, hpmt, pResult);
         break;

      case SIQUERY_FIND_CAROUSEL:
         DBGLOG(DS_QUERY, "Find Carousel serviceId=0x%x", pQuery->serviceId)
         PmtFindCarousel(siq, hpmt, pResult);
         break;

      case SIQUERY_CAROUSEL_INFO:
         DBGLOG(DS_QUERY, "Carousel id=0x%x serviceId=0x%x", (int)pQuery->dataId, pQuery->serviceId)
         PmtCarouselInfo(siq, hpmt, (U8BIT)pQuery->dataId, pResult);
         break;

      case SIQUERY_CAROUSEL:
         DBGLOG(DS_QUERY, "Carousel id=0x%x serviceId=0x%x", (int)pQuery->dataId, pQuery->serviceId)
         PmtCarousel(siq, hpmt, pQuery->dataId, pResult);
         break;

      case SIQUERY_DEFERRED_SERVICE:
         DBGLOG(DS_QUERY, "Deferred serviceId=0x%x assocTag=0x%x", pQuery->serviceId, pQuery->associationTag)
         PmtDeferredService(siq, hpmt, pQuery->associationTag, pResult);
         break;

      case SIQUERY_SSU_CAROUSEL:
         DBGLOG(DS_QUERY, "Carousel id=0x%x serviceId=0x%x", (int)pQuery->dataId, pQuery->serviceId)
         PmtSsuCarousel(siq, hpmt, pQuery->dataId, pResult);
         break;

      case SIQUERY_SSU_PID:
         /* Special case of ssu PID being given to DSMCC API (and held in service id field) */
         pResult->kind = SIQUERY_SSU_PID;
         pResult->data.carouselInfo.pid = pQuery->serviceId;
         pResult->data.carouselInfo.carouselId = INVALID_CAROUSEL_ID;
         pResult->data.carouselInfo.associationTag = INVALID_ASSOCIATION_TAG;
         break;

      default:
      case SIQUERY_PIDLIST:
         ERRLOG("NOT supported %d", pQuery->kind);
         break;
   }
}

/**
 * @brief   For each carousel Id in old table, search through new table to
 *          see if it still exists. If not, tell DSM-CC.
 *          Then tell DSM-CC that PMT has been updated.
 * @param   S_SiqInstance* siq        SIQ instance handle.
 * @param   H_SiqPmtTable  hOldTable
 * @param   H_SiqPmtTable  hNewTable
 * @param   U16BIT         serviceId
 * @return  none.
 */
void SIQ_PmtCheckTableChange( S_SiqInstance *siq, H_SiqPmtTable pOldTable,
   H_SiqPmtTable pNewTable, U16BIT serviceId )
{
   S_PmtStream *pOldStream, *pNewStream;
   U16BIT i, j;
   U32BIT carouselId;

   FUNCTION_START(SIQ_PmtCheckTableChange)
   ASSERT(pOldTable != NULL)
   ASSERT(pNewTable != NULL)

   pOldStream = pOldTable->streams;
   for (i = pOldTable->num_streams; i--; pOldStream++)
   {
      carouselId = pOldStream->carouselId;
      pNewStream = pNewTable->streams;
      for (j = pNewTable->num_streams; j--; pNewStream++)
      {
         if (carouselId == pNewStream->carouselId)
         {
            /* found carousel in new one as well - OK */
            carouselId = INVALID_CAROUSEL_ID;
            break;
         }
      }
      if (carouselId != INVALID_CAROUSEL_ID)
      {
         CDSM_SysProcessSIChangeEvent( siq->dsmInstance, SICHANGE_CAROUSEL_DELETED, serviceId, carouselId );
      }
   }

   CDSM_SysProcessSIChangeEvent( siq->dsmInstance, SICHANGE_SERVICE_UPDATED, serviceId, 0 );

   FUNCTION_FINISH(SIQ_PmtCheckTableChange)
}

