/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   API functions for DSM-CC client requests
 * @file    dsm_client.c
 * @date    01/12/2004
 * @author  Ocean Blue
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

#include "stb_os.h"
#include "cldsmcc.h"
#include "fs_types.h"
#include "dsm_client.h"
#include "dm_debug.h"
#include "dsm_main.h"

/*---constant definitions for this file--------------------------------------*/

/* For a object type */
#define LOAD_FLAGS_REQUEST_OBJECT_TYPE  0x00100000

/*---local typedef structs for this file-------------------------------------*/

typedef struct s_DsmControl
{
   S_DsmApi api;
} S_DsmControl;

typedef struct s_DsmRefs
{
   H_DsmCoreInst cldsm;
   void *apimutex;
   H_DsmObject dsmobj;
   F_MemFree memFree;
} S_DsmRefs;

typedef struct s_ObjUserData
{
   H_DsmControl dsmctrl;
   union
   {
      F_DSM_CBLOAD load;
      F_DSM_TYPE type;
   } cb;
   U32BIT flags;
   U32BIT size;
   void *client_ud;
} S_ObjUserData;


/*---local (static) variable declarations for this file----------------------*/

static U8BIT dummy_data[2] = {"\0\0"};

/*---local function definitions----------------------------------------------*/

static void UnloadObject( H_DsmControl dsmctrl, H_DsmObject dsmObj )
{
   E_DscError cldsmErr;
   DBGPRINT(DM_CLIENT, "( 0x%x )", dsmObj)
   STB_OSMutexLock(dsmctrl->api.mutex);;
   cldsmErr = CDSM_UnloadObject( dsmctrl->api.cldsm, dsmObj, RST_MODE_FORCE);
   STB_OSMutexUnlock(dsmctrl->api.mutex);;
   if (cldsmErr != CLDSM_OK)
   {
      ERRPRINT("clDsmUnloadObject: %d", cldsmErr)
   }
}

/**
 * @brief End the file load request. This function should be called either during
 *       an asynchronous file load or after an asynchronous file load completes
 *       indicate that the requested file is no longer requred.
 * @param obj DSM object handle
 */
static void DsmccUnloadObject(void *obj)
{
   H_DsmObject dsmObj = obj;
   H_DsmCoreInst cldsm;
   H_DsmControl dsmctrl;
   E_DscError cldsmErr;
   cldsm = CDSM_ObjectGetInstance(dsmObj);
   if (cldsm == NULL)
   {
      ERRPRINT("invalid object: %p", dsmObj)
   }
   else
   {
      dsmctrl = CDSM_SysGetControl(cldsm);
      if (dsmctrl == NULL)
      {
         ERRPRINT("invalid instance: %p", cldsm)
      }
      else
      {
         cldsmErr = CDSM_CloseObject(dsmctrl->api.cldsm, dsmObj);
         if (cldsmErr)
         {
            DBGPRINT(DM_CLIENT, "%d", cldsmErr)
         }
         UnloadObject( dsmctrl, dsmObj );
      }
   }
}

static void DsmccRefsFree( void *dsmRef )
{
   S_DsmRefs *refs = dsmRef;
   E_DscError cldsmErr;
   STB_OSMutexLock(refs->apimutex);;
   cldsmErr = CDSM_UnloadObject( refs->cldsm, refs->dsmobj, RST_MODE_FORCE);
   STB_OSMutexUnlock(refs->apimutex);;
   if (cldsmErr != CLDSM_OK)
   {
      ERRPRINT("clDsmUnloadObject: err=%d, %p, %p", cldsmErr, refs, refs->dsmobj)
   }
   refs->memFree(dsmRef);
}

static
E_FsStatus StreamObjectQueryService( H_DsmControl dsmctrl, S_CONTENT *pContent, F_DSM_CBLOAD cb_func )
{
   E_DscError cldsmErr;
   S_DvbLocator *pDeferredService;
   E_FsStatus ackStatus;
   cldsmErr = CDSM_StreamGetDeferredService( dsmctrl->api.cldsm, pContent->fs_handle, pContent, cb_func, &pDeferredService );
   switch (cldsmErr)
   {
      case CLDSM_PENDING:
         ackStatus = FS_STATUS_PENDING;
         break;

      default:
      /*pDeferredService should be current service*/
      /*fall through*/
      case CLDSM_OK:
         pContent->size = sizeof(S_DvbLocator);
         pContent->data = (U8BIT *)pDeferredService;
         ackStatus = FS_STATUS_OK;
         break;
   }
   return ackStatus;
}

static
E_FsStatus DirectoryListing( H_DsmControl dsmctrl, H_DsmObject objectHandle,
   S_CONTENT *pContent, BOOLEAN incKind )
{
   E_DscError err;
   S_DsmRefs *refs;
   U8BIT *data;
   void *entry;
   U16BIT count, totalLen;
   E_FsStatus ackStatus = FS_STATUS_ERROR;
   DBGPRINT(DM_CLIENT, "%p", objectHandle)
   err = CDSM_DirEntrySizes(objectHandle, &count, &totalLen);
   DBGPRINT(DM_CLIENT, " err=%d cnt=%d ttl=%d", err, count, totalLen)
   if (!err && totalLen && count)
   {
      if (incKind)
      {
         totalLen += count; /* add 'count' - for kind byte on each entry */
      }
      DBGPRINT(DM_CLIENT, " ttl=%d", totalLen)
      refs = dsmctrl->api.memAlloc( sizeof(S_DsmRefs) + totalLen );
      if (refs == NULL)
      {
         ERRPRINT("malloc failed %d", sizeof(S_DsmRefs) + totalLen)
      }
      else
      {
         DBGPRINT(DM_CLIENT, "%p", refs)
         refs->cldsm = dsmctrl->api.cldsm;
         refs->apimutex = dsmctrl->api.mutex;
         refs->memFree = dsmctrl->api.memFree;
         refs->dsmobj = objectHandle;
         err = CDSM_DirEntryFirst(objectHandle, &entry);
         if (!err)
         {
            pContent->fs_handle = refs;
            pContent->destroy = DsmccRefsFree;
            data = (U8BIT *)(refs + 1);
            pContent->data = data;
            pContent->size = totalLen;
            count--;
            if (incKind)
            {
               *data = (U8BIT)(OBJECT_TYPE_UNKNOWN + (int)CDSM_DirEntryKind(entry));
               data++;
            }
            data += CDSM_DirEntryNameCopy(entry, data) - 1;
            while (count && CDSM_DirEntryNext(entry, &entry) == CLDSM_OK)
            {
               count--;
               *data++ = ',';
               if (incKind)
               {
                  *data = (U8BIT)(OBJECT_TYPE_UNKNOWN + (int)CDSM_DirEntryKind(entry));
                  data++;
               }
               data += CDSM_DirEntryNameCopy(entry, data) - 1;
            }
            *data = 0;
            ackStatus = FS_STATUS_OK;
         }
         else
         {
            ERRPRINT("err=%d", err)
            dsmctrl->api.memFree(refs);
         }
      }
   }
   else
   {
      ERRPRINT("err=%d", err)
   }
   return ackStatus;
}

static
E_FsStatus StreamEventListing( H_DsmControl dsmctrl, H_DsmObject objectHandle, S_CONTENT *pContent )
{
   E_DscError err;
   U8BIT *data;
   U16BIT size;
   E_FsStatus ackStatus = FS_STATUS_ERROR;

   err = CDSM_StreamEventNameList( dsmctrl->api.cldsm, objectHandle, &data, &size );
   if (err == CLDSM_OK)
   {
      DBGPRINT(DM_CLIENT, "%p %p %d", objectHandle, data, size)
      pContent->size = size;
      pContent->data = data;
      ackStatus = FS_STATUS_OK;
   }
   else
   {
      ERRPRINT("err=%d", err)
      ackStatus = FS_STATUS_ERROR;
   }
   return ackStatus;
}

static
E_FsStatus StreamEventXmldata( H_DsmControl dsmctrl, H_DsmObject objectHandle, S_CONTENT *pContent )
{
   E_DscError err;
   U8BIT *data;
   U16BIT size;
   E_FsStatus ackStatus = FS_STATUS_ERROR;

   err = CDSM_StreamEventXmlData( dsmctrl->api.cldsm, objectHandle, &data, &size );
   if (err == CLDSM_OK)
   {
      DBGPRINT(DM_CLIENT, "%p %p %d", objectHandle, data, size)
      pContent->size = size;
      pContent->data = data;
      ackStatus = FS_STATUS_OK;
   }
   else
   {
      ERRPRINT("err=%d", err)
      ackStatus = FS_STATUS_ERROR;
   }
   return ackStatus;
}

/**
 * @brief   Read a loaded object from DSM-CC. The object can be either a stream
 *          object or a file object. The object contents will be loaded (if
 *          possible)
 * @param   pUserData pointer to object user data
 * @param   objectHandle Handle to the object (returned by clDsmLoadObject)
 * @param   pContent Pointer to allocated memory containg the file contents
 * @return  Status of the object load, with one of the following values:
 *          FS_STATUS_ERROR - Object load error
 *          FS_STATUS_OK - Object loaded sucessfully
 *          FS_STATUS_NONE - request flag not relevant
 *          FS_STATUS_PENDING - deferred service pending
 */
static E_FsStatus ReadObject( H_ObjUserData pUserData,
   H_DsmObject objectHandle, S_CONTENT *pContent )
{
   H_DsmControl dsmctrl = pUserData->dsmctrl;
   U32BIT flags = pUserData->flags;
   E_DscError cldsmErr;
   E_DsmObjectKind kind;
   E_FsStatus ackStatus;

   ASSERT( pContent->size == 0 );
   ASSERT( pContent->data == NULL );
   ASSERT( pContent->fs_handle == objectHandle );

   /* Read DSM object */
   cldsmErr = CDSM_OpenObject( dsmctrl->api.cldsm, objectHandle, &kind );
   if (cldsmErr == CLDSM_ERR_END_OF_DATA && kind == FILE_OBJ)
   {
      /* Zero-length file object */
      DBGPRINT(DM_CLIENT, " zero length file")
      pContent->size = 0;
      pContent->data = dummy_data;
      ackStatus = FS_STATUS_OK;
   }
   else if (cldsmErr)
   {
      ERRPRINT("clDsmOpenObject err=%d", cldsmErr)
      ackStatus = FS_STATUS_ERROR;
   }
   else if ((flags & LOAD_FLAGS_REQUEST_FILE_ONLY_BIT) && (kind != FILE_OBJ))
   {
      ackStatus = FS_STATUS_NONE;
   }
   else
   {
      switch (kind)
      {
         case STREAM_OBJ_WITH_EVENTS:
            if (flags & LOAD_FLAGS_REQUEST_EVENT_LIST_BIT)
            {
               ackStatus = StreamEventListing( dsmctrl, objectHandle, pContent );
               break;
            }
            else if (flags & LOAD_FLAGS_REQUEST_STREAM_OBJ_XML_BIT)
            {
               ackStatus = StreamEventXmldata( dsmctrl, objectHandle, pContent );
               break;
            }
         case STREAM_OBJ:
            if (flags & LOAD_FLAGS_REQUEST_DEFERRED_SERVICE_BIT)
            {
               ackStatus = StreamObjectQueryService( dsmctrl, pContent, pUserData->cb.load );
            }
            else
            {
               ackStatus = FS_STATUS_NONE;
            }
            break;

         case FILE_OBJ:
            cldsmErr = CDSM_FileDirect( objectHandle, &pContent->data, &pContent->size );
            if (cldsmErr)
            {
               ERRPRINT("clDsmFiledirect: %d", cldsmErr)
               pContent->data = NULL;
               ackStatus = FS_STATUS_ERROR;
            }
            else
            {
               DBGPRINT(DM_CLIENT, "file len=%d", pContent->size)
               ackStatus = FS_STATUS_OK;
            }
            break;

         case DIRECTORY_OBJ:
         case SERVICE_GATEWAY_OBJ:
            if (flags & LOAD_FLAGS_REQUEST_SERVICE_CONTEXT_BIT)
            {
               cldsmErr = CDSM_ObjectGetServiceContext( objectHandle, pUserData->size,
                     &pContent->data, &pContent->size );
               if (cldsmErr == CLDSM_OK || cldsmErr == CLDSM_ERR_END_OF_DATA)
               {
                  DBGPRINT(DM_CLIENT, "SRV CTXT len=%d", pContent->size)
                  ackStatus = FS_STATUS_OK;
               }
               else
               {
                  ERRPRINT("%d", cldsmErr)
                  ackStatus = FS_STATUS_ERROR;
               }
            }
            else if (flags & LOAD_FLAGS_REQUEST_DIRECTORY_LIST_BIT)
            {
               ackStatus = DirectoryListing( dsmctrl, objectHandle, pContent, TRUE );
               DBGPRINT(DM_CLIENT, "DIR stat=%d len=%d", ackStatus, pContent->size)
            }
            else
            {
               ackStatus = DirectoryListing( dsmctrl, objectHandle, pContent, FALSE );
               DBGPRINT(DM_CLIENT, "DIR stat=%d len=%d", ackStatus, pContent->size)
            }
            break;

         default:
            ERRPRINT("kind=%d", kind)
            ackStatus = FS_STATUS_ERROR;
      }
      if (kind != FILE_OBJ || cldsmErr)
      {
         cldsmErr = CDSM_CloseObject( dsmctrl->api.cldsm, objectHandle );
         if (cldsmErr)
         {
            ERRPRINT("%d", cldsmErr)
         }
      }
   }
   return ackStatus;
}

/**
 * @brief   Get the type of loaded object from DSM-CC, and then unload it.
 * @param   objectHandle Handle to the object (returned by clDsmLoadObject)
 * @return  Status of the object load, with one of the following values:
 *          FS_STATUS_ERROR - Object load error
 *          FS_STATUS_OK - Object loaded sucessfully
 */
static E_FsStatus GetDsmObjectType( H_ObjUserData pUserData,
   H_DsmObject objectHandle, E_DsmObjectType *pType )
{
   H_DsmControl dsmctrl = pUserData->dsmctrl;
   E_DscError cldsmErr;
   E_DsmObjectKind kind;
   E_FsStatus ackStatus;

   /* Read DSM object */
   cldsmErr = CDSM_OpenObject( dsmctrl->api.cldsm, objectHandle, &kind );
   if (cldsmErr == CLDSM_ERR_END_OF_DATA && kind == FILE_OBJ)
   {
      /* Zero-length file object */
      *pType = OBJECT_TYPE_FILE;
      ackStatus = FS_STATUS_OK;
      DBGPRINT(DM_CLIENT, " zero length file")
   }
   else if (cldsmErr)
   {
      *pType = OBJECT_TYPE_UNKNOWN;
      ackStatus = FS_STATUS_ERROR;
      ERRPRINT("clDsmOpenObject err=%d", cldsmErr)
   }
   else
   {
      ackStatus = FS_STATUS_OK;
      switch (kind)
      {
         case STREAM_OBJ_WITH_EVENTS:
         {
            *pType = OBJECT_TYPE_STREAM_WITH_EVENTS;
            break;
         }
         case STREAM_OBJ:
         {
            *pType = OBJECT_TYPE_STREAM;
            break;
         }
         case FILE_OBJ:
         {
            *pType = OBJECT_TYPE_FILE;
            break;
         }
         case DIRECTORY_OBJ:
         {
            *pType = OBJECT_TYPE_DIRECTORY;
            break;
         }
         case SERVICE_GATEWAY_OBJ:
         {
            *pType = OBJECT_TYPE_ROOT_DIR;
            break;
         }
         default:
            *pType = OBJECT_TYPE_UNKNOWN;
            ackStatus = FS_STATUS_ERROR;
            ERRPRINT("kind=%d", kind)
      }
      cldsmErr = CDSM_CloseObject( dsmctrl->api.cldsm, objectHandle );
      if (cldsmErr)
      {
         ERRPRINT("%d", cldsmErr)
      }
   }
   cldsmErr = CDSM_UnloadObject( dsmctrl->api.cldsm, objectHandle, RST_MODE_FORCE);
   if (cldsmErr)
   {
      ERRPRINT("%d", cldsmErr)
   }
   return ackStatus;
}

/*---global function definitions---------------------------------------------*/

/**
 * @brief   Callback function provided to DSM-CC library. Object load events are
 *          passed to this callback by the DSM-CC library.
 * @param   objectHandle Handle to the object (returned by clDsmLoadObject)
 * @param   status Status of the load request for the specified object.
 * @param   engineLoadRef Reference from MHEG5 engine
 * @param   objLoadUserData2 DsmCmd_t cammand value (passing in as void*)
 * @return  void
 */
void DsmccObjectLoadEventFunc( H_DsmObject objectHandle, E_ObjLoadStatus status,
   H_ObjUserData pUserData )
{
#ifndef NDEBUG
   H_DsmControl dsmctrl = pUserData->dsmctrl;
#endif
   E_FsStatus ackStatus;
   S_CONTENT content;
   E_DsmObjectType type;

   DBGPRINT(DM_CLIENT, "( 0x%x, %d, 0x%x )", objectHandle, status, pUserData)

   if (pUserData != NULL)
   {
      content.size = 0;
      content.data = NULL;
      content.user_data = pUserData->client_ud;
      switch (status)
      {
         case OBJ_LOAD_COMPLETED:
         {
            content.destroy = DsmccUnloadObject;
            content.fs_handle = objectHandle;
            DBGPRINT(DM_CLIENT, "object loaded; flgs=%x", pUserData->flags)
            if (pUserData->flags & LOAD_FLAGS_REQUEST_EXISTANCE_BIT)
            {
               ackStatus = FS_STATUS_OK;
            }
            else if (pUserData->flags == LOAD_FLAGS_REQUEST_OBJECT_TYPE)
            {
               ackStatus = GetDsmObjectType( pUserData, objectHandle, &type );
            }
            else
            {
               ackStatus = ReadObject( pUserData, objectHandle, &content );
            }
         }
         break;

         case OBJ_LOAD_ABORTED_UNLOAD:
            /* object already unloaded, so destroy function should be NULL */
            content.destroy = NULL;
            content.fs_handle = NULL;
            ackStatus = FS_STATUS_ABORT;
            break;

         default:
            /* load failed */
            ERRPRINT("status %d", status)
            content.destroy = NULL;
            content.fs_handle = NULL;
            ackStatus = FS_STATUS_ERROR;
      }

      if (ackStatus != FS_STATUS_PENDING)
      {
         if (pUserData->flags == LOAD_FLAGS_REQUEST_OBJECT_TYPE)
         {
            pUserData->cb.type( content.user_data, type );
         }
         else
         {
            pUserData->cb.load( ackStatus, &content );
         }
      }
      else
      {
         ERRPRINT("status PENDING!")
      }
   }
}

void DsmccStreamEventFunc( H_DsmEvent eventHandle,
   E_SENotifyStatus status, void *userData1, void *userData2,
   U8BIT *namePtr, U8BIT *dataPtr, U8BIT nameLen, U8BIT dataLen )
{
   F_DSM_EVENT cb_func = (F_DSM_EVENT)userData1;
   if (status == SEN_TRIGGERED)
   {
      cb_func( userData2, namePtr, dataLen, dataPtr );
   }
}

/**
 * @brief   Perform a file load request which is executed synchronously or
 *          asynchronously. See load flags above for modifying requests.
 *
 *          Synchronous behaviour:
 *           If DSM-CC is able to determine that the file does not exist, the
 *           function returns with the FS_STATUS_ERROR.
 *
 *           If DSM-CC has the requested file in cache, it can read the file
 *           contents and pass a pointer to it in the pFileData return parameter,
 *           along with the file length in the pFileSize returned parameter. Once
 *           the contents of the file has been used, resource allocated to it
 *           must be released by calling the dsmFDataAck function. The function
 *           return value will be set to FS_STATUS_OK.
 *
 *          Asynchronous behaviour:
 *           If DSM-CC does not have the file in cache it will perform an
 *           asynchronous load. This is indicated by returning FS_STATUS_PENDING.
 *           When the load is completed, the 'cb_func' is called
 *
 *          If the path parameter points to a stream object within the DSM-CC
 *          carousel this will be loaded. The referenced multiplex will be located
 *          and a reference to this will be passed back to the MHEG-5 engine.
 *
 * @param   dsmctrl DSMCC controling Instance
 * @param   path Path of Object (directory, file or stream)
 * @param   flags Load flags containing cache priority, and other control bits
 * @param   cb_func Call-back Function
 * @param   pContent Pointer to the file content details.
 *          Note: When content data for object is no longer required, the destroy
 *          function in S_CONTENT must be called to free any allocated resource.
 * @return  E_FsStatus
 *          FS_STATUS_OK
 *          The Object has been sucessfully loaded
 */
E_FsStatus DSMCC_ClientLoadObject( H_DsmControl dsmctrl,
   U8BIT *path, U32BIT flags, F_DSM_CBLOAD cb_func, S_CONTENT *pContent )
{
   E_DscError cldsmErr = CLDSM_OK;
   E_ObjLoadStatus objStatus;
   H_DsmObject fileHandle = NULL;
   E_FsStatus ackStatus;
   E_CacheRules cachingRules;
   S_ObjUserData oud;

   DBGPRINT(DM_CLIENT, "( %s, 0x%x, 0x%x )", path, cb_func, pContent )

   cachingRules = (E_CacheRules)(flags & LOAD_FLAGS_CACHE_PRIORITY_MASK);

   oud.dsmctrl = dsmctrl;
   oud.flags = flags;
   oud.cb.load = cb_func;
   oud.size = pContent->size;
   oud.client_ud = pContent->user_data;
   pContent->size = 0;
   pContent->data = NULL;

   STB_OSMutexLock(dsmctrl->api.mutex);

   if (cachingRules)
   {
      if ((cachingRules & 1) == 0)
      {
         cachingRules >>= 1;
         /* Cache priority is even - this indicates that the object is static */
         cachingRules |= CACHE_RULES_STATIC;
      }
      else if (cachingRules != 1)
      {
         cachingRules >>= 1;
      }
   }
   else
   {
      cldsmErr = CDSM_SetObjectCachingRules( dsmctrl->api.cldsm, path, CACHE_RULES_FROM_STREAM );
   }
   cldsmErr = CDSM_LoadObject( dsmctrl->api.cldsm, path, 0,
         cachingRules, &oud, sizeof(S_ObjUserData),
         &objStatus, &fileHandle );

   pContent->destroy = DsmccUnloadObject;
   pContent->fs_handle = (void *)fileHandle;

   STB_OSMutexUnlock(dsmctrl->api.mutex);

   if (cldsmErr)
   {
      ERRPRINT(" cldsmErr=%d", cldsmErr)
      pContent->destroy = NULL;
      pContent->fs_handle = NULL;
      ackStatus = FS_STATUS_ERROR;
   }
   else
   {
      DBGPRINT(DM_CLIENT, " status=%d hdl=%p", objStatus, fileHandle )
      switch (objStatus)
      {
         case OBJ_LOAD_IN_PROGRESS:
            /* File load will return asynchronously */
            ackStatus = FS_STATUS_PENDING;
            break;

         case OBJ_LOAD_COMPLETED:
            if (flags & LOAD_FLAGS_REQUEST_EXISTANCE_BIT)
            {
               ackStatus = FS_STATUS_OK;
               break;
            }
            STB_OSMutexLock(dsmctrl->api.mutex);
            ackStatus = ReadObject( &oud, fileHandle, pContent );
            STB_OSMutexUnlock(dsmctrl->api.mutex);
            if (ackStatus != FS_STATUS_ERROR)
            {
               break;
            }
         default:
            ackStatus = FS_STATUS_ERROR;
            UnloadObject(dsmctrl, fileHandle);
            pContent->size = 0;
            pContent->destroy = NULL;
            pContent->fs_handle = NULL;
      }
   }
   return ackStatus;
}

/**
 * @brief   Request object type for given path. This may be executed synchronously
 *          or asynchronously - see comment for DSMCC_ClientLoadObject()
 * @param   dsmctrl DSMCC controling Instance
 * @param   path Path of Object (directory, file or stream)
 * @param   cbfunc Call-back Function
 * @param   userData User data to be passed back to client in callback function.
 * @param   pType  Pointer to object type - return value is valid when FS_STATUS_OK.
 * @return  E_FsStatus, FS_STATUS_OK - The Object has been sucessfully found
 */
E_FsStatus DSMCC_ClientObjectType( H_DsmControl dsmctrl, U8BIT *path,
   F_DSM_TYPE cbfunc, void *userData, E_DsmObjectType *pType )
{
   E_DscError cldsmErr = CLDSM_OK;
   E_ObjLoadStatus objStatus;
   H_DsmObject fileHandle = NULL;
   E_FsStatus ackStatus;
   S_ObjUserData oud;

   DBGPRINT(DM_CLIENT, "( %s, %p, %p )", path, cbfunc, pType )

   oud.dsmctrl = dsmctrl;
   oud.cb.type = cbfunc;
   oud.flags = LOAD_FLAGS_REQUEST_OBJECT_TYPE;
   oud.size = 0;
   oud.client_ud = userData;

   STB_OSMutexLock(dsmctrl->api.mutex);

   cldsmErr = CDSM_LoadObject( dsmctrl->api.cldsm, path, 0,
         CACHE_RULES_DEFAULT, &oud, sizeof(S_ObjUserData),
         &objStatus, &fileHandle );

   STB_OSMutexUnlock(dsmctrl->api.mutex);

   if (cldsmErr)
   {
      ERRPRINT(" cldsmErr=%d", cldsmErr)
      ackStatus = FS_STATUS_ERROR;
      *pType = OBJECT_TYPE_UNKNOWN;
   }
   else
   {
      DBGPRINT(DM_CLIENT, " status=%d hdl=%p", objStatus, fileHandle )
      switch (objStatus)
      {
         case OBJ_LOAD_IN_PROGRESS:
            /* object type will return asynchronously */
            ackStatus = FS_STATUS_PENDING;
            break;

         case OBJ_LOAD_COMPLETED:
            STB_OSMutexLock(dsmctrl->api.mutex);
            ackStatus = GetDsmObjectType( &oud, fileHandle, pType );
            STB_OSMutexUnlock(dsmctrl->api.mutex);
            if (cldsmErr)
            {
               ERRPRINT(" cldsmErr=%d", cldsmErr)
            }
            break;

         default:
            ackStatus = FS_STATUS_ERROR;
            *pType = OBJECT_TYPE_UNKNOWN;
            UnloadObject(dsmctrl, fileHandle);
      }
   }
   return ackStatus;
}

/**
 * @brief   Provide a hint that the specified file may be required in the future.
 *          If possible this should be loaded into the DSM-CC cache.
 * @param   dsmctrl DSMCC Instance
 * @param   path Path of Object (directory, file or stream)
 * @return  void
 */
void DSMCC_ClientPreloadHint( H_DsmControl dsmctrl, U8BIT *path )
{
   E_DscError cldsmErr;

   STB_OSMutexLock(dsmctrl->api.mutex);

   cldsmErr = CDSM_PrefetchObject( dsmctrl->api.cldsm, path, 0 /* timeout */ );

   STB_OSMutexUnlock(dsmctrl->api.mutex);

   if (cldsmErr != CLDSM_OK)
   {
      ERRPRINT("ERROR: clDsmPrefetchObject returned %d", cldsmErr)
   }
}

/**
 * @brief   Client request to subscribe to DSMCC Stream Event, specified by name.
 *          DSMCC notifies when the named stream event occurs, by calling cb_func
 *          (a callback function defined by the Client).
 *          Before calling this function the Client must request that the stream object of
 *          interest be loaded using DSMCC_ClientLoadObject() function. This action will
 *          deliver a handle to the object stream - this is passed as the streamObject
 *          argument to this function. Do not have to wait until load is completed before
 *          subscribing.
 * @param   dsmctrl DSMCC Instance
 * @param   streamObject DSMCCs file system handle
 * @param   eventName Name of stream event required by the Client.
 * @param   cb_func Callback funcrtion for client.
 * @param   userData Optional data to be passed back to client. It is stored
 *          and returned unchanged.
 * @param   pEventHandle Handle to the stream event returned by subscribe.
 *          The caller may use this handle when calling DSMCC_ClientEventUnsubscribe
 * @return  E_FsStatus
 *          FS_STATUS_OK
 *          The event has been successfully subscribed to.
 */
E_FsStatus DSMCC_ClientEventSubscribeName( H_DsmControl dsmctrl,
   FS_HANDLE streamObject, U8BIT *eventName,
   F_DSM_EVENT cb_func, void *userData,
   H_DsmEvent *pEventHandle )
{
   E_DscError err;
   E_FsStatus status;

   if (cb_func == NULL || pEventHandle == NULL)
   {
      status = FS_STATUS_ERROR;
   }
   else
   {
      STB_OSMutexLock(dsmctrl->api.mutex);

      err = CDSM_StreamEventSubscribe( dsmctrl->api.cldsm, streamObject, eventName,
            cb_func, userData, pEventHandle );
      STB_OSMutexUnlock(dsmctrl->api.mutex);
      if (err == CLDSM_OK)
      {
         DBGPRINT(DM_CLIENT, " name=%s", eventName)
         status = FS_STATUS_OK;
      }
      else
      {
         status = FS_STATUS_ERROR;
      }
   }
   return status;
}

/**
 * @brief   Client request to subscribe to DSMCC Stream Event, specified by event ID
 *          and association tag of stream object.
 *          DSMCC notifies when the stream event occurs, by calling cb_func
 *          (a callback function defined by the Client).
 * @param   dsmctrl DSMCC Instance
 * @param   associationTag association tag.
 * @param   evemntId Id of stream event.
 * @param   cb_func Callback funcrtion for client.
 * @param   userData Optional data to be passed back to client. It is stored
 *          and returned unchanged.
 * @param   pEventHandle Handle to the stream event returned by subscribe.
 *          The caller may use this handle when calling DSMCC_ClientEventUnsubscribe
 * @return  E_FsStatus
 *          FS_STATUS_OK
 *          The event has been successfully subscribed to.
 */
E_FsStatus DSMCC_ClientEventSubscribeId( H_DsmControl dsmctrl,
   U16BIT associationTag, U16BIT eventId,
   F_DSM_EVENT cb_func, void *userData,
   H_DsmEvent *pEventHandle )
{
   E_DscError err;
   E_FsStatus status;
   if (cb_func == NULL || pEventHandle == NULL)
   {
      status = FS_STATUS_ERROR;
   }
   else
   {
      STB_OSMutexLock(dsmctrl->api.mutex);

      err = CDSM_SpecialEventSubscribe( dsmctrl->api.cldsm, associationTag, eventId,
            cb_func, userData, pEventHandle );
      STB_OSMutexUnlock(dsmctrl->api.mutex);
      if (err == CLDSM_OK)
      {
         DBGPRINT(DM_CLIENT, " id=0x%x", eventId)
         status = FS_STATUS_OK;
      }
      else
      {
         status = FS_STATUS_ERROR;
      }
   }
   return status;
}

/**
 * @brief   Client request to UN-subscribe to DSMCC Stream Event.
 * @param   dsmctrl DSMCC Instance
 * @param   event Handle to previously subscribed event
 * @return  void
 */
void DSMCC_ClientEventUnsubscribe( H_DsmControl dsmctrl, H_DsmEvent eventHandle)
{
   STB_OSMutexLock(dsmctrl->api.mutex);

   CDSM_StreamEventUnsubscribe( dsmctrl->api.cldsm, eventHandle );

   STB_OSMutexUnlock(dsmctrl->api.mutex);
}

/**
 * @brief   Client request to retrieve MHEG5 File system acceleration file groups
 *          The client must release the array of group data, by calling
 *          DSMCC_ClientUnloadFileGroups, when finished with the data
 * @param   dsmctrl DSMCC Instance
 * @param   hOC Handle to carousel
 * @param   total Pointer to return number of file groups in the carousel info
 * @param   pGroups Pointer to return an array of file group data.
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_ClientLoadFileGroups( H_DsmControl dsmctrl, H_ObjCarousel hOC,
   U16BIT *total, S_CarouselInfoFileGroup **pGroups )
{
   E_DscError err;
   BOOLEAN result;

   STB_OSMutexLock(dsmctrl->api.mutex);

   err = CDSM_CarouselLoadFileGroups( dsmctrl->api.cldsm, hOC, total, pGroups );

   STB_OSMutexUnlock(dsmctrl->api.mutex);

   if (err == CLDSM_OK)
   {
      result = TRUE;
   }
   else
   {
      ERRPRINT("err=%d", err)
      result = FALSE;
   }

   return result;
}

/**
 * @brief   Client request to release file group data returned from
 *          DSMCC_ClientLoadFileGroups
 * @param   dsmctrl DSMCC Instance
 * @param   hOC Handle to carousel
 * @param   groups Array of file group data.
 * @return  void
 */
void DSMCC_ClientUnloadFileGroups( H_DsmControl dsmctrl, H_ObjCarousel hOC,
   S_CarouselInfoFileGroup *groups )
{
   E_DscError err;

   STB_OSMutexLock(dsmctrl->api.mutex);

   err = CDSM_CarouselUnloadFileGroups( dsmctrl->api.cldsm, hOC, groups );

   STB_OSMutexUnlock(dsmctrl->api.mutex);

   if (err != CLDSM_OK)
   {
      ERRPRINT("err=%d", err)
   }
}

