/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DSM-CC control Manager
 * @file    dsm_control.c
 * @date    17 October 2013
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>
#include <stdlib.h>

#include "stb_os.h"

#include "cldsmcc.h"
#ifndef NDEBUG
#include "cldsmdbg.h"
#endif
#include "dsfm.h"
#include "dsiq.h"
#include "sbm.h"
#include "stbsipmt.h"
#include "dm_debug.h"
#include "dsm_control.h"
#include "dsm_main.h"
#include "dsm_si.h"
#include "dsmcc_version.h"

/*---constant definitions for this file--------------------------------------*/

#define MAIN_STACK_SIZE    0x2000
#define TIMER_RESOLUTION   100
#define RESTART_TIME       200

#define START_MAGIC        0xDEED
#define CLDSM_MAGIC        0xBADE
#define ERA_BITS           4
#define ERA_SHIFT          (32 - ERA_BITS)
#define ERA_LAST           ((1 << ERA_BITS) - 1)
#define ERA_FIRST          0x00

#if defined(NDEBUG) && !defined(DSM_DBG_MASK)
#define DSM_DBG_MASK 0
#endif

#define GETENV_DSMDBG(mskitem) \
   strcpy(mskptr,#mskitem); \
   value = getenv(envname); \
   if (value != NULL) { \
      if (atoi(value) == 1) mask |= mskitem; \
      else mask &= ~mskitem; }

/*---local typedef structs for this file-------------------------------------*/

typedef enum
{
   QP_TOP,
   QP_HIGH,
   QP_LOW,
   QP_TOTAL
} E_Q_PRIORITY;

typedef enum
{
   DSM_MSG_PAT_DATA,
   DSM_MSG_PMT_DATA    = 0x02,
   /* above items are equal to Table ID,
    * and below start with values above max Table Id */
   DSM_MSG_NEW_SECTION = 0x100,
   DSM_MSG_CACHE_SECTION,
   DSM_MSG_UPDATE_FILTER,
   DSM_MSG_PMT_REMOVED,
   DSM_MSG_CLOSE_DOWN
} E_DsmMsgType;

typedef enum
{
   BL_WAITING,
   BL_BOOTING,
   BL_LOADED,
   BL_FAILED
} E_STATUS;

typedef struct s_PmtInfo
{
   U16BIT xid;
   U8BIT version;
   U8BIT reserved;
} S_TableInfo;

typedef struct s_DsmMessage
{
   E_DsmMsgType type;
   U8BIT *pSection;
   union
   {
      void *hBuffer;
      S_TableInfo tinfo;
   } u;
} S_Message;

typedef struct s_timer
{
   struct s_timer *next;
   U32BIT trigger;
   U_PARAM handle;
   U_PARAM usrData;
} S_Timer;

typedef struct s_msgqueue
{
   void *mutex;
   S_Message *begin;
   S_Message *write;
   S_Message *read;
   S_Message *ends;
} S_MsgQueue;

typedef struct s_DsmControl
{
   S_DsmApi api;

   void *timerMtx;
   void *task;
   void *q_count_sema;

   S_MsgQueue queues[QP_TOTAL];
   S_Timer *timers;

   H_DsmControl next;
   H_SiqInstance siq;
   H_SfmInstance sfm;
   H_SbmInstance sbm;
   H_DsiInstance dsi;

#ifndef NDEBUG
   U32BIT debugMask;
#endif
   H_DsmCarousel carousel;
   U32BIT couId;
   U32BIT restart;
   U16BIT counter;

   U16BIT transId;
   U16BIT networkId;
   U16BIT serviceId;

   U8BIT sikind;
   U8BIT status;
   DMXREF dmxref;
} S_DsmControl;

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

#ifndef NDEBUG
unsigned int gDebugState = DSM_DBG_MASK;
#endif

S_DsmControl *instanceHead = NULL;

static BOOLEAN DsmStart( H_DsmControl dsmctrl, U16BIT serviceId, U32BIT couId, U8BIT sikind );
static void DsmRestart( H_DsmControl dsmctrl, U16BIT serviceId );

/*------------------- Local function definitions ----------------------------*/

static BOOLEAN QueueWrite( S_MsgQueue *queue, S_Message *pmsg )
{
   BOOLEAN result;
   S_Message *next;
   STB_OSMutexLock(queue->mutex);
   next = queue->write;
   next++;
   if (next == queue->ends)
   {
      next = queue->begin;
   }
   if (next == queue->read)
   {
      result = FALSE;
   }
   else
   {
      *(queue->write) = *pmsg;
      queue->write = next;
      result = TRUE;
   }
   STB_OSMutexUnlock(queue->mutex);
   return result;
}

static BOOLEAN QueueRead( S_MsgQueue *queue, S_Message *pmsg )
{
   BOOLEAN result;
   S_Message *next;
   STB_OSMutexLock(queue->mutex);
   if (queue->read == queue->write)
   {
      result = FALSE;
   }
   else
   {
      next = queue->read;
      *pmsg = *next;
      next++;
      if (next == queue->ends)
      {
         next = queue->begin;
      }
      queue->read = next;
      result = TRUE;
   }
   STB_OSMutexUnlock(queue->mutex);
   return result;
}

static U16BIT CheckTimers( S_DsmControl *dsmctrl, U32BIT now )
{
   S_Timer *pTimer;
   U16BIT restartid;
   U16BIT timeout;
   U8BIT era;
   restartid = 0;
   era = now >> ERA_SHIFT;
   timeout = TIMEOUT_NEVER;
   switch (era)
   {
      case ERA_LAST:
         STB_OSMutexLock(dsmctrl->timerMtx);
         pTimer = dsmctrl->timers;
         while (pTimer != NULL && pTimer->trigger <= now &&
                (pTimer->trigger >> ERA_SHIFT) != ERA_FIRST)
         {
            dsmctrl->timers = pTimer->next;
            STB_OSMutexUnlock(dsmctrl->timerMtx);
            if ((pTimer->handle.u32 >> 16) == CLDSM_MAGIC)
            {
               STB_OSMutexLock(dsmctrl->api.mutex);
               CDSM_SysProcessTimerEvent(dsmctrl->api.cldsm, pTimer->usrData.ptr, TIMER_TRIGGERED, pTimer->handle.ptr);
               STB_OSMutexUnlock(dsmctrl->api.mutex);
            }
            else if ((pTimer->handle.u32 >> 16) == START_MAGIC)
            {
               restartid = pTimer->usrData.u16;
            }
            dsmctrl->api.memFree(pTimer);
            STB_OSMutexLock(dsmctrl->timerMtx);
            pTimer = dsmctrl->timers;
         }
         STB_OSMutexUnlock(dsmctrl->timerMtx);
         if (pTimer != NULL)
         {
            timeout = (U16BIT)((S32BIT)pTimer->trigger - (S32BIT)now);
         }
         break;

      case ERA_FIRST:
         STB_OSMutexLock(dsmctrl->timerMtx);
         pTimer = dsmctrl->timers;
         while (pTimer != NULL &&
                (pTimer->trigger <= now || (pTimer->trigger >> ERA_SHIFT) == ERA_LAST))
         {
            dsmctrl->timers = pTimer->next;
            STB_OSMutexUnlock(dsmctrl->timerMtx);
            if ((pTimer->handle.u32 >> 16) == CLDSM_MAGIC)
            {
               STB_OSMutexLock(dsmctrl->api.mutex);
               CDSM_SysProcessTimerEvent(dsmctrl->api.cldsm, pTimer->usrData.ptr, TIMER_TRIGGERED, pTimer->handle.ptr);
               STB_OSMutexUnlock(dsmctrl->api.mutex);
            }
            else if ((pTimer->handle.u32 >> 16) == START_MAGIC)
            {
               restartid = pTimer->usrData.u16;
            }
            dsmctrl->api.memFree(pTimer);
            STB_OSMutexLock(dsmctrl->timerMtx);
            pTimer = dsmctrl->timers;
         }
         STB_OSMutexUnlock(dsmctrl->timerMtx);
         if (pTimer != NULL)
         {
            timeout = (U16BIT)(pTimer->trigger - now);
         }
         break;

      default:
         STB_OSMutexLock(dsmctrl->timerMtx);
         pTimer = dsmctrl->timers;
         while (pTimer != NULL && pTimer->trigger <= now)
         {
            dsmctrl->timers = pTimer->next;
            STB_OSMutexUnlock(dsmctrl->timerMtx);
            if ((pTimer->handle.u32 >> 16) == CLDSM_MAGIC)
            {
               STB_OSMutexLock(dsmctrl->api.mutex);
               CDSM_SysProcessTimerEvent(dsmctrl->api.cldsm, pTimer->usrData.ptr, TIMER_TRIGGERED, pTimer->handle.ptr);
               STB_OSMutexUnlock(dsmctrl->api.mutex);
            }
            else if ((pTimer->handle.u32 >> 16) == START_MAGIC)
            {
               restartid = pTimer->usrData.u16;
            }
            dsmctrl->api.memFree(pTimer);
            STB_OSMutexLock(dsmctrl->timerMtx);
            pTimer = dsmctrl->timers;
         }
         STB_OSMutexUnlock(dsmctrl->timerMtx);
         if (pTimer != NULL)
         {
            timeout = (U16BIT)(pTimer->trigger - now);
         }
   }
   if (restartid != 0)
   {
      DsmRestart(dsmctrl, (U16BIT)restartid);
   }
   return timeout;
}

static void DsmMain(void *arg)
{
   S_DsmControl *dsmctrl = (S_DsmControl *)arg;
   S_MsgQueue *queues = dsmctrl->queues;
   void *apimutex = dsmctrl->api.mutex;
   S_Message msg;
   U32BIT si_time, now;
   U16BIT sema_timeout;
   BOOLEAN closing = FALSE;

   FUNCTION_START(DsmMain);

   sema_timeout = TIMEOUT_NEVER;
   si_time = 0xFFFFFFFF;

   while (!closing)
   {
      if (STB_OSSemaphoreWaitTimeout(dsmctrl->q_count_sema, sema_timeout))
      {
         if (QueueRead(queues + QP_TOP, &msg) ||
             QueueRead(queues + QP_HIGH, &msg) ||
             QueueRead(queues + QP_LOW, &msg))
         {
            switch (msg.type)
            {
               case DSM_MSG_PAT_DATA:
               {
                  STB_OSMutexLock(apimutex);
                  SIQ_UpdatePat( dsmctrl->siq, msg.pSection );
                  STB_OSMutexUnlock(apimutex);
                  SBM_ReleaseBuffer( dsmctrl->sbm, msg.pSection );
                  if (dsmctrl->dsi)
                  {
                     if (dsmctrl->status == (U8BIT)BL_WAITING)
                     {
                        DsmStart( dsmctrl, dsmctrl->serviceId, dsmctrl->couId, dsmctrl->sikind );
                     }
                     si_time = DSI_CheckRequests( dsmctrl->dsi, STB_OSGetClockMilliseconds());
                  }
                  break;
               }
               case DSM_MSG_PMT_DATA:
               {
                  H_SiqPmtTable table;
                  table = SIQ_ParsePmt( dsmctrl->siq, msg.pSection );
                  SBM_ReleaseBuffer( dsmctrl->sbm, msg.pSection );
                  STB_OSMutexLock(apimutex);
                  SIQ_ProcessPmt( dsmctrl->siq, table, msg.u.tinfo.xid, msg.u.tinfo.version );
                  STB_OSMutexUnlock(apimutex);
                  if (dsmctrl->dsi)
                  {
                     si_time = DSI_CheckRequests( dsmctrl->dsi, STB_OSGetClockMilliseconds());
                  }
                  break;
               }
               case DSM_MSG_PMT_REMOVED:
               {
                  STB_OSMutexLock(apimutex);
                  SIQ_ServiceRemoved( dsmctrl->siq, msg.u.tinfo.xid );
                  STB_OSMutexUnlock(apimutex);
                  break;
               }
               case DSM_MSG_NEW_SECTION:
               case DSM_MSG_CACHE_SECTION:
               {
                  STB_OSMutexLock(apimutex);
                  SFM_ProcessSection( dsmctrl->sfm, msg.pSection, msg.u.hBuffer );
                  STB_OSMutexUnlock(apimutex);
                  if (msg.type == DSM_MSG_NEW_SECTION)
                  {
                     SBM_ReleaseBuffer( dsmctrl->sbm, msg.pSection );
                  }
                  break;
               }
               case DSM_MSG_UPDATE_FILTER:
               {
                  SFM_FilterUpdate( dsmctrl->sfm, msg.u.hBuffer );
                  break;
               }
               case DSM_MSG_CLOSE_DOWN:
                  closing = TRUE;
            }
         }
      }
      now = STB_OSGetClockMilliseconds();
      if (now > si_time)
      {
         ASSERT(dsmctrl->dsi);
         si_time = DSI_CheckRequests( dsmctrl->dsi, now );
      }
      sema_timeout = CheckTimers(dsmctrl, now);
      if (now + sema_timeout > si_time)
      {
         sema_timeout = (U16BIT)(si_time - now);
      }
   }
   /* flush queues */
   while (STB_OSSemaphoreWaitTimeout( dsmctrl->q_count_sema, 0 ))
   {
      if (QueueRead(queues + QP_TOP, &msg) ||
          QueueRead(queues + QP_HIGH, &msg) ||
          QueueRead(queues + QP_LOW, &msg))
      {
         switch (msg.type)
         {
            case DSM_MSG_NEW_SECTION:
            case DSM_MSG_PMT_DATA:
            case DSM_MSG_PAT_DATA:
            {
               SBM_ReleaseBuffer( dsmctrl->sbm, msg.pSection );
               break;
            }
            case DSM_MSG_PMT_REMOVED:
            case DSM_MSG_CACHE_SECTION:
            case DSM_MSG_CLOSE_DOWN:
            case DSM_MSG_UPDATE_FILTER:
            {
               break;
            }
         }
      }
   }
   dsmctrl->task = NULL;
   FUNCTION_FINISH(DsmMain);
}

static void ProcessPmt( S_DsmControl *dsmctrl, U16BIT serviceId, U8BIT *pmt )
{
   S_Message msg;
   U16BIT len;
   if (pmt == NULL)
   {
      msg.type = DSM_MSG_PMT_REMOVED;
      msg.u.tinfo.xid = serviceId;
      if (!QueueWrite(dsmctrl->queues + QP_TOP, &msg))
      {
         ERRPRINT("failed to add to Q")
      }
      else
      {
         STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
      }
   }
   else if (SIQ_RequirePmt(dsmctrl->siq, pmt, &msg.u.tinfo.xid, &msg.u.tinfo.version) == SIQ_UPDATE)
   {
      ASSERT(msg.u.tinfo.xid == serviceId);
      msg.pSection = SBM_AllocateBuffer( dsmctrl->sbm, pmt[1] );
      if (msg.pSection == NULL)
      {
         ERRPRINT("failed to alloc buffer")
      }
      else
      {
         msg.type = DSM_MSG_PMT_DATA;
         len = ((pmt[1] << 8 | pmt[2]) & 0xfff) + 3;
         memcpy(msg.pSection, pmt, len);
         if (!QueueWrite(dsmctrl->queues + QP_TOP, &msg))
         {
            ERRPRINT("failed to add to Q")
            SBM_ReleaseBuffer( dsmctrl->sbm, msg.pSection );
         }
         else
         {
            STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
         }
      }
   }
}

static void DvbPmtCallback( U16BIT serviceId, U8BIT *pmt )
{
   S_DsmControl *dsmctrl;
   FUNCTION_START(UpdateFilter)
   dsmctrl = instanceHead;
   while (dsmctrl != NULL)
   {
      if (dsmctrl->status != (U8BIT)BL_WAITING &&
          dsmctrl->dsi == NULL && dsmctrl->siq && dsmctrl->task)
      {
         /* This instance is using pmt updates from DVB */
         ProcessPmt(dsmctrl, serviceId, pmt);
      }
      dsmctrl = dsmctrl->next;
   }
   FUNCTION_FINISH(UpdateFilter)
}

static BOOLEAN UsingPmtsFromDvb( void )
{
   S_DsmControl *dsmctrl;
   FUNCTION_START(UsingPmtsFromDvb)
   dsmctrl = instanceHead;
   while (dsmctrl != NULL)
   {
      if (dsmctrl->dsi == NULL)
      {
         /* This instance is using pmt updates from DVB */
         return TRUE;
      }
      dsmctrl = dsmctrl->next;
   }
   FUNCTION_FINISH(UsingPmtsFromDvb)
   return FALSE;
}

static void ProcessDsi( DMXREF demux, U16BIT nbytes, PIDFILT pfid )
{
   S_Message emsg;
   S_DsmControl *dsmctrl;
   U8BIT header[SECTION_HEADER_LENGTH];
   U16BIT sectionLen;
   if (STB_DMXCopyPIDFilterSect(demux, header, SECTION_HEADER_LENGTH, pfid))
   {
      sectionLen = (((header[1] & 0x0F) << 8) | header[2]) + 3;
      if (nbytes != sectionLen)
      {
         ERRPRINT("Length mis-match for section (bytes=%#x != data1=%#x,data2=%#x)",
            nbytes, header[1], header[2] )
      }
      else
      {
         dsmctrl = instanceHead;
         while (dsmctrl != NULL)
         {
            if (dsmctrl->dsi != NULL && demux == dsmctrl->dmxref)
            {
               if (DSI_RequireTable(dsmctrl->dsi, header, &emsg.u.tinfo.xid, &emsg.u.tinfo.version))
               {
                  emsg.type = (E_DsmMsgType)*header;
                  emsg.pSection = SBM_AllocateBuffer( dsmctrl->sbm, header[1] );
                  if (emsg.pSection == NULL)
                  {
                     ERRPRINT("failed to alloc buffer")
                  }
                  else
                  {
                     if (!STB_DMXCopyPIDFilterSect(demux, emsg.pSection, sectionLen, pfid) ||
                         !QueueWrite(dsmctrl->queues + QP_TOP, &emsg))
                     {
                        ERRPRINT("failed to add to Q")
                        SBM_ReleaseBuffer( dsmctrl->sbm, emsg.pSection );
                     }
                     else
                     {
                        STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
                     }
                  }
               }
            }
            dsmctrl = dsmctrl->next;
         }
      }
   }
}

/**
 * @brief   This function is called when DVP filter reports an incoming section.
 *          It will copy and examine the section header to determine whether the
 *          section is required. If so, then it will copy entire section buffer.
 *          It uses STB_DMXCopyPIDFilterSect to copy section buffer data.
 *          It assumes CRC has been checked and valid DSM-CC sections are passed
 *          So the first byte of copied data must be table id 0x3B, 0x3C or 0x3D.
 *          Voyager expects this to be done in context of DVP thread.
 * @param   U8BIT       demux    Demux path index
 * @param   U16BIT      nbytes   Number of section data bytes
 * @param   U16BIT      pfid     Hardware PID filter id of incoming section
 * @return  void
 */
static void ProcessSection( DMXREF demux, U16BIT nbytes, PIDFILT pfid )
{
   S_Message emsg;
   E_SFM_STATUS status;
   S_MsgQueue *queue;
   S_DsmControl *dsmctrl;
   U8BIT header[SECTION_HEADER_LENGTH];
   U16BIT sectionLen;
   FUNCTION_START(ProcessSection)
   if (STB_DMXCopyPIDFilterSect(demux, header, SECTION_HEADER_LENGTH, pfid))
   {
      sectionLen = (((header[1] & 0x0F) << 8) | header[2]) + 3;
      if (*header < 0x3B || *header > 0x3D)
      {
         ERRPRINT("Bad table id=0x%x", *header)
      }
      else if (nbytes < sectionLen)
      {
         ERRPRINT("Demux not providing enough bytes for section (bytes=%#x != data1=%#x,data2=%#x)",
            nbytes, header[1], header[2] )
      }
      else
      {
         dsmctrl = instanceHead;
         while (dsmctrl != NULL)
         {
            if (demux == dsmctrl->dmxref)
            {
               status = SFM_RequireSection( dsmctrl->sfm, pfid, header, &emsg.u.hBuffer );
               switch (status)
               {
                  case SFM_UPDATE_CACHE:
                     emsg.pSection = SFM_CacheBuffer( dsmctrl->sfm, emsg.u.hBuffer );
                     if (emsg.pSection == NULL)
                     {
                        ERRPRINT("SFM_CacheBuffer failed for %#x", emsg.u.hBuffer)
                     }
                     else
                     {
                        STB_DMXCopyPIDFilterSect(demux, emsg.pSection, sectionLen, pfid);
                        if (nbytes != sectionLen && CDSM_UtilCalculateCRC(emsg.pSection, sectionLen))
                        {
                           SFM_CacheReleaseBuffer( dsmctrl->sfm, emsg.pSection, emsg.u.hBuffer );
                        }
                     }
                  /* fall through */

                  case SFM_ERROR:
                  case SFM_IGNORE:
                     queue = NULL;
                     break;

                  case SFM_UPDATE_LOW:
                     queue = &dsmctrl->queues[QP_LOW];
                     break;

                  case SFM_UPDATE_HIGH:
                     queue = &dsmctrl->queues[QP_HIGH];
                     break;
               }
               if (queue != NULL)
               {
                  emsg.pSection = SBM_AllocateBuffer( dsmctrl->sbm, header[1] );
                  if (emsg.pSection == NULL)
                  {
                     ERRPRINT("Memory failure for size=%#x", sectionLen)
                  }
                  else
                  {
                     emsg.type = DSM_MSG_NEW_SECTION;
                     if (!STB_DMXCopyPIDFilterSect(demux, emsg.pSection, sectionLen, pfid) ||
                         (nbytes != sectionLen && CDSM_UtilCalculateCRC(emsg.pSection, sectionLen)) ||
                         !QueueWrite(queue, &emsg))
                     {
                        SBM_ReleaseBuffer( dsmctrl->sbm, emsg.pSection );
                     }
                     else
                     {
                        STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
                     }
                  }
               }
            }
            dsmctrl = dsmctrl->next;
         }
      }
   }
   FUNCTION_FINISH(ProcessSection)
}

/**
 * @brief   Report to SFM client that there is an update to be processed. This
 *          is called in the normal DSM-CC thread environment, and is due to
 *          SFM finding section in cache. The client may place the request on a
 *          queue that will subsequently cause SFM_ProcessSection to be called.
 * @param   H_SfmInstance  sfm            SFM dsmctrl handle.
 * @param   U8BIT*         pSection       Pointer to section data buffer
 * @param   void*          hBuffer        Section buffer handle
 * @param   E_SFM_STATUS   status         Status of update, either:
 *                                        SFM_UPDATE_LOW or SFM_UPDATE_HIGH
 * @return  void
 */
static void CacheMatch( H_SfmInstance sfm, U8BIT *pSection,
   void *hBuffer, E_SFM_STATUS status )
{
   S_Message emsg;
   S_DsmControl *dsmctrl;
   FUNCTION_START(CacheMatch)
   dsmctrl = instanceHead;
   while (dsmctrl != NULL)
   {
      if (dsmctrl->sfm == sfm)
      {
         emsg.pSection = pSection;
         emsg.u.hBuffer = hBuffer;
         emsg.type = DSM_MSG_CACHE_SECTION;
         switch (status)
         {
            default:
               break;

            case SFM_UPDATE_LOW:
               if (!QueueWrite(dsmctrl->queues + QP_LOW, &emsg))
               {
                  ERRPRINT("")
               }
               else
               {
                  STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
               }
               break;

            case SFM_UPDATE_HIGH:
               if (!QueueWrite(dsmctrl->queues + QP_HIGH, &emsg))
               {
                  ERRPRINT("")
               }
               else
               {
                  STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
               }
               break;
         }
         break;
      }
      dsmctrl = dsmctrl->next;
   }
   FUNCTION_FINISH(CacheMatch)
}

/**
 * @brief   Report to SFM client that there is a filter update to be processed.
 *          The client must place the request on a queue that will subsequently
 *          cause SFM_FilterUpdate to be called.
 * @param   H_SfmInstance  sfm            SFM dsmctrl handle.
 * @param   H_DmxPidFilter hPF            Handle to SFM pid filter
 * @return  void
 */
static void UpdateFilter( H_SfmInstance sfm, H_DmxPidFilter hPF )
{
   S_Message emsg;
   S_DsmControl *dsmctrl;
   FUNCTION_START(UpdateFilter)
   dsmctrl = instanceHead;
   while (dsmctrl != NULL)
   {
      if (dsmctrl->sfm == sfm)
      {
         emsg.pSection = NULL;
         emsg.u.hBuffer = hPF;
         emsg.type = DSM_MSG_UPDATE_FILTER;
         if (!QueueWrite(dsmctrl->queues + QP_TOP, &emsg))
         {
            ERRPRINT("")
         }
         else
         {
            STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
         }
         break;
      }
      dsmctrl = dsmctrl->next;
   }
   FUNCTION_FINISH(UpdateFilter)
}

U32BIT StubRequestTable( void *pmtref, U16BIT pid, U16BIT xid, U8BIT tid )
{
   return 0x55000000 | (U32BIT)(tid << 16) | xid;
}

static void StubCancelTable( void *ref, U32BIT rid )
{
}

/**
 * @brief      Create SIQ instance
 * @return     H_SiqInstance - instance of SIQ - NULL is failure
 */
static H_SiqInstance CreateSiq( S_DsmccConfig *config, void *pmtref )
{
   S_SiqSetup setup;

   setup.memAlloc = config->memAlloc;
   setup.memFree = config->memFree;

   if (!(config->controlFlags & GET_PMT_DIRECT_FROM_HW_FLAG))
   {
      setup.dvpRequestTable = StubRequestTable;
      setup.dvpCancelTable = StubCancelTable;
      if (!UsingPmtsFromDvb())
      {
         STB_SIRegisterPmtObserver( DvbPmtCallback );
      }
   }
   else
   {
      setup.dvpRequestTable = (F_DvpRequestTable)DSI_RequestTable;
      setup.dvpCancelTable = (F_DvpCancelTable)DSI_CancelTable;
   }
   setup.pmtref = pmtref;

   setup.parsePmtInit = config->parsePmtInit;
   setup.parsePmtDone = config->parsePmtDone;
   setup.parseDataBroadcastId = config->parseDataBroadcastId;
   setup.parseSsuSelectorBytes = config->parseSsuSelectorBytes;

   setup.notifyAitInfo = config->notifyAitInfo;

#ifndef NDEBUG
   setup.errPrintf = (F_Printf)STB_SPDebugNoCnWrite;
   if (gDebugState & DS_ALL)
   {
      setup.dbgPrintf = (F_Printf)STB_SPDebugNoCnWrite;
      setup.dbgState = gDebugState & DS_ALL;
   }
   else
#endif
   {
   #if DSM_DP_LEVEL >= 2
      setup.dbgPrintf = (F_Printf)STB_SPDebugNoCnWrite;
   #else
      setup.dbgPrintf = DBG_DebugPrintf;
   #endif
      setup.dbgState = 0;
   }

   return SIQ_CreateInstance( &setup );
}

/**
 * @brief   Destroy SIQ
 * @return  void.
 */
static void DestroySiq( H_SiqInstance siq )
{
   SIQ_DestroyInstance( siq );
}

/**
 * @brief   Create SFM dsmctrl
 * @return  H_SfmInstance - dsmctrl of SFM - NULL is failure
 */
static H_SfmInstance CreateSfm( S_DsmccConfig *config )
{
   S_SfmSetup setup;
   H_SfmInstance sfm;

   setup.sfmMutex = STB_OSCreateMutex();
   if (setup.sfmMutex == NULL)
   {
      ERRPRINT("")
      sfm = NULL;
   }
   else
   {
      setup.cacheMutex = STB_OSCreateMutex();
      if (setup.cacheMutex == NULL)
      {
         STB_OSDeleteMutex( setup.sfmMutex );
         ERRPRINT("")
         sfm = NULL;
      }
      else
      {
         setup.bufferMutex = STB_OSCreateMutex();
         if (setup.bufferMutex == NULL)
         {
            STB_OSDeleteMutex( setup.cacheMutex );
            STB_OSDeleteMutex( setup.sfmMutex );
            ERRPRINT("")
            sfm = NULL;
         }
         else
         {
            setup.memAlloc = config->memAlloc;
            setup.memFree = config->memFree;

            setup.mutexLock = STB_OSMutexLock;
            setup.mutexUnlock = STB_OSMutexUnlock;

            setup.hwFilterCallback = ProcessSection;

            setup.updateFilter = UpdateFilter;

            setup.maxPidFilters = 4;
            setup.maxSecFiltersPerPid = 1;

            setup.sectionBuffCacheSize = config->sectionBuffCacheSize;

            if (setup.sectionBuffCacheSize)
            {
               setup.cacheMatch = CacheMatch;
            }
            setup.maxAvailableSectionFilters = NUM_SECTION_FILTERS_DEFAULT;

         #ifndef NDEBUG
            setup.errPrintf = (F_Printf)STB_SPDebugNoCnWrite;
            if (gDebugState & DF_ALL)
            {
               setup.dbgPrintf = (F_Printf)STB_SPDebugNoCnWrite;
               setup.dbgState = gDebugState & DF_ALL;
            }
            else
         #endif
            {
            #if DSM_DP_LEVEL >= 2
               setup.dbgPrintf = (F_Printf)STB_SPDebugNoCnWrite;
            #else
               setup.dbgPrintf = DBG_DebugPrintf;
            #endif
               setup.dbgState = 0;
            }

            sfm = SFM_CreateInstance( &setup );
         }
      }
   }
   return sfm;
}

/**
 * @brief   Destroy SFM
 * @return  void.
 */
static void DestroySfm( H_SfmInstance sfm )
{
   void *sfmMutex;
   void *cacheMutex;
   void *bufferMutex;
   if (sfm != NULL)
   {
      SFM_DestroyInstance( sfm, &sfmMutex, &cacheMutex, &bufferMutex );
      STB_OSDeleteMutex( bufferMutex );
      STB_OSDeleteMutex( cacheMutex );
      STB_OSDeleteMutex( sfmMutex );
   }
}

static E_DscError StartTimer( H_DsmControl dsmctrl, U16BIT magic,
   U32BIT timePeriod, void *clDsmTmrUserData, void **pTimerHandle )
{
   S_Timer *pNewTimer, **ppTimer;
   U32BIT trigger, tval;
   U16BIT counter;
   E_DscError err;
   U8BIT era;

   pNewTimer = dsmctrl->api.memAlloc(sizeof(S_Timer));
   if (pNewTimer == NULL)
   {
      *pTimerHandle = NULL;
      err = CLDSM_ERR_ALLOC_FAILED;
   }
   else
   {
      trigger = STB_OSGetClockMilliseconds() + timePeriod;
      pNewTimer->trigger = trigger;
      pNewTimer->usrData.ptr = clDsmTmrUserData;
      era = trigger >> ERA_SHIFT;
      STB_OSMutexLock(dsmctrl->timerMtx);
      dsmctrl->counter++;
      counter = dsmctrl->counter;
      pNewTimer->handle.u32 = (((U32BIT)magic) << 16) | (U32BIT)counter;
      counter = 0;
      ppTimer = &dsmctrl->timers;
      while (*ppTimer != NULL)
      {
         tval = (*ppTimer)->trigger;
         if (tval > trigger &&
             (era != ERA_FIRST || (tval >> ERA_SHIFT) != ERA_LAST))
         {
            break;
         }
         counter++;
         ppTimer = &((*ppTimer)->next);
      }
      pNewTimer->next = *ppTimer;
      *ppTimer = pNewTimer;
      STB_OSMutexUnlock(dsmctrl->timerMtx);
      if (counter == 0)
      {
         STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
      }
      *pTimerHandle = pNewTimer->handle.ptr;
      err = CLDSM_OK;
   }
   return err;
}

static E_DscError StartTimerFunc( H_DsmControl dsmctrl,
   U32BIT timePeriod, void *clDsmTmrUserData, void **pTimerHandle )
{
   return StartTimer(dsmctrl, CLDSM_MAGIC, timePeriod, clDsmTmrUserData, pTimerHandle);
}

static void StopTimer( H_DsmControl dsmctrl, U32BIT handle )
{
   S_Timer *pTimer, **ppTimer;
   STB_OSMutexLock(dsmctrl->timerMtx);
   ppTimer = &dsmctrl->timers;
   while (*ppTimer != NULL)
   {
      pTimer = *ppTimer;
      if (handle == pTimer->handle.u32)
      {
         *ppTimer = pTimer->next;
         dsmctrl->api.memFree(pTimer);
         break;
      }
      ppTimer = &(pTimer->next);
   }
   STB_OSMutexUnlock(dsmctrl->timerMtx);
}

static void StopTimerFunc( H_DsmControl dsmctrl, void *timerHandle )
{
   U_PARAM th;
   th.ptr = timerHandle;
   StopTimer( dsmctrl, th.u32 );
}

static void dsmccErrorFunc( E_DscError err, void *args )
{
}

static H_DsiInstance CreateDsi( S_DsmControl *dsmctrl, S_DsmccConfig *config)
{
   H_DsiInstance dsi;
   dsi = DSI_CreateInstance(dsmctrl, ProcessDsi, config);
   dsmctrl->dsi = dsi;
   return dsi;
}

void DestroyDsi( S_DsmControl *dsmctrl )
{
   if (dsmctrl->dsi != NULL)
   {
      DSI_DestroyInstance( dsmctrl->dsi );
      dsmctrl->dsi = NULL;
   }
}

static H_DsmCoreInst CreateDsmcc( S_DsmControl *dsmctrl, S_DsmccConfig *config )
{
   S_DsmSetup setup;
   void *memContext = NULL;
   E_DscError cldsmErr = CLDSM_OK;
   H_DsmCoreInst cldsm;

#ifndef NDEBUG
   CDSM_SetPrintFuncs( DBG_ErrorPrintf, DBG_WarnPrintf, DBG_DebugPrintf, DBG_InfoPrintf );
   CDSM_SetPrintState( gDebugState );
#endif

   setup.siqInstance = CreateSiq( config, dsmctrl->dsi );
   if (setup.siqInstance == NULL)
   {
      cldsm = NULL;
   }
   else
   {
      setup.sfmInstance = CreateSfm( config );
      if (setup.sfmInstance == NULL)
      {
         if (!(config->controlFlags & GET_PMT_DIRECT_FROM_HW_FLAG) &&
             !UsingPmtsFromDvb())
         {
            STB_SIUnregisterPmtObserver( DvbPmtCallback );
         }
         DestroySiq( setup.siqInstance );
         cldsm = NULL;
      }
      else
      {
         /* Set up the callback functions provided by the DSM-CC component. */
         setup.allocFunc = config->memAlloc;
         setup.freeFunc = config->memFree;

         setup.errorFunc = dsmccErrorFunc;

         setup.startTimerFunc = StartTimerFunc;
         setup.stopTimerFunc = StopTimerFunc;

         setup.addSectionFilterFunc = SFM_DsmccFilterAdd;
         setup.delSectionFilterFunc = SFM_DsmccFilterDelete;
         setup.sfPriorityChangeFunc = SFM_DsmccFilterPriorityChange;

         setup.startSIQueryFunc = SIQ_RequestQueryStart;
         setup.stopSIQueryFunc = SIQ_RequestQueryStop;
         setup.subscribeSIChangeFunc = SIQ_ServiceInfoChangeSubscribe;
         setup.unsubscribeSIChangeFunc = SIQ_ServiceInfoChangeUnsubscribe;

         setup.notifyCarouselLoadEventFunc = config->carouselLoad;
         setup.notifyObjectLoadEventFunc = DsmccObjectLoadEventFunc;
         setup.notifyStreamEventFunc = DsmccStreamEventFunc;
         setup.progressFunc = NULL;
         setup.dsmControl = dsmctrl;

         setup.ssuFuncs = config->ssuFuncs;

         /* Set other values to suggested defaults */
         setup.memMgrSetup = NULL;
         setup.maxAvailableSectionFilters = NUM_SECTION_FILTERS_DEFAULT;
         setup.timeUnitResolution = TIMER_RESOLUTION;
         setup.storeSIQueryResults = TRUE;
         setup.turboCaching = (config->controlFlags & DISABLE_AUTO_CACHING_FLAG) ? FALSE : TRUE;
         setup.multipleSsuDownloads = FALSE;
         setup.notifyDeferredServiceFunc = NULL;
         setup.maxMemorySize = DSMCC_DEFAULT_CACHE_SIZE;
         /* Create an dsmctrl of the DSM-CC component. */
         cldsmErr = CDSM_SysCreate( &setup, &cldsm, &memContext );

         if (cldsmErr == CLDSM_OK)
         {
            SIQ_SetDsmInstance( setup.siqInstance, cldsm );
            SFM_SetDsmInstance( setup.sfmInstance, cldsm );
            dsmctrl->api.cldsm = cldsm;
            dsmctrl->sfm = setup.sfmInstance;
            dsmctrl->siq = setup.siqInstance;
#ifndef NDEBUG
            dsmctrl->debugMask = gDebugState;
            CDSM_SetDebugState(cldsm, gDebugState & DD_ALL);
#endif
         }
         else
         {
            DestroySfm( setup.sfmInstance );
            if (!(config->controlFlags & GET_PMT_DIRECT_FROM_HW_FLAG) &&
                !UsingPmtsFromDvb())
            {
               STB_SIUnregisterPmtObserver( DvbPmtCallback );
            }
            DestroySiq( setup.siqInstance );
            cldsm = NULL;
         }
      }
   }
   return cldsm;
}

static void DestroyDsmcc( S_DsmControl *dsmctrl )
{
   E_DscError err = CLDSM_OK;
   H_SiqInstance siqInstance;
   H_SfmInstance sfmInstance;

   err = CDSM_SysReset( dsmctrl->api.cldsm, RST_MODE_FORCE );
   if (err)
   {
      ERRPRINT("err=%d", err)
   }
   err = CDSM_SysDestroy(dsmctrl->api.cldsm, &siqInstance, &sfmInstance);
   if (err == CLDSM_OK)
   {
      dsmctrl->api.cldsm = NULL;

      ASSERT(dsmctrl->sfm == sfmInstance);
      dsmctrl->sfm = NULL;
      DestroySfm( sfmInstance );

      ASSERT(dsmctrl->siq == siqInstance);
      dsmctrl->siq = NULL;
      DestroySiq( siqInstance );

      if (dsmctrl->dsi != NULL)
      {
         DestroyDsi( dsmctrl );
      }
      else if (!UsingPmtsFromDvb())
      {
         STB_SIUnregisterPmtObserver( DvbPmtCallback );
      }
   }
   else
   {
      ERRPRINT("err=%d", err)
   }
}

/**
 * @brief   Destroy Section Buffer manager
 * @return  void.
 */
static void DestroySecBuff( H_SbmInstance sbm )
{
   void *bufferMutex;
   if (sbm != NULL)
   {
      SBM_DestroyInstance( sbm, &bufferMutex );
      STB_OSDeleteMutex( bufferMutex );
   }
}

/**
 * @brief   Create Section Buffer dsmctrl
 * @return  H_SbmInstance - dsmctrl of SBQ - NULL is failure
 */
static H_SbmInstance CreateSecBuff( S_DsmControl *dsmctrl, U16BIT size )
{
   S_SbmSetup setup;
   H_SbmInstance sbm;
   setup.mtx_sem = STB_OSCreateMutex();
   if (setup.mtx_sem == NULL)
   {
      ERRPRINT("")
      sbm = NULL;
   }
   else
   {
      setup.memAlloc = dsmctrl->api.memAlloc;
      setup.memFree = dsmctrl->api.memFree;
      setup.lock = STB_OSMutexLock;
      setup.unlock = STB_OSMutexUnlock;
      setup.bufferSize = ((size + 15) >> 4);
      sbm = SBM_CreateInstance( &setup );
      if (sbm == NULL)
      {
         STB_OSDeleteMutex(setup.mtx_sem);
      }
      else
      {
         dsmctrl->sbm = sbm;
      }
   }
   return sbm;
}

static void DestoryResources( S_DsmControl *dsmctrl )
{
   int i;
   if (dsmctrl->sbm != NULL)
   {
      DestroySecBuff( dsmctrl->sbm );
      dsmctrl->sbm = NULL;
   }
   for (i = 0; i != QP_TOTAL; i++)
   {
      if (dsmctrl->queues[i].mutex != NULL)
      {
         STB_OSDeleteMutex(dsmctrl->queues[i].mutex);
      }
   }
   if (dsmctrl->q_count_sema != NULL)
   {
      STB_OSDeleteSemaphore(dsmctrl->q_count_sema);
      dsmctrl->q_count_sema = NULL;
   }
   if (dsmctrl->timerMtx != NULL)
   {
      STB_OSDeleteMutex(dsmctrl->timerMtx);
      dsmctrl->timerMtx = NULL;
   }
   if (dsmctrl->api.mutex != NULL)
   {
      STB_OSDeleteMutex(dsmctrl->api.mutex);
      dsmctrl->api.mutex = NULL;
   }
}

static BOOLEAN CreateResources( S_DsmControl *dsmctrl, U16BIT size )
{
   S_Message *pmsg;
   int i;
   BOOLEAN result;
   dsmctrl->api.mutex = STB_OSCreateMutex();
   if (dsmctrl->api.mutex == NULL)
   {
      ERRPRINT("")
      result = FALSE;
   }
   else
   {
      dsmctrl->timerMtx = STB_OSCreateMutex();
      if (dsmctrl->timerMtx == NULL)
      {
         STB_OSDeleteMutex(dsmctrl->api.mutex);
         dsmctrl->api.mutex = NULL;
         ERRPRINT("")
         result = FALSE;
      }
      else
      {
         dsmctrl->q_count_sema = STB_OSCreateCountSemaphore(0);
         if (dsmctrl->q_count_sema == NULL)
         {
            ERRPRINT("")
            result = FALSE;
         }
         else
         {
            result = TRUE;
            pmsg = (S_Message *)(dsmctrl + 1);
            for (i = 0; i != QP_TOTAL; i++)
            {
               dsmctrl->queues[i].mutex = STB_OSCreateMutex();
               if (dsmctrl->queues[i].mutex == NULL)
               {
                  ERRPRINT("")
                  result = FALSE;
                  break;
               }
               dsmctrl->queues[i].begin = pmsg;
               dsmctrl->queues[i].write = pmsg;
               dsmctrl->queues[i].read = pmsg;
               pmsg += size * (i + 1);
               dsmctrl->queues[i].ends = pmsg;
            }
            if (i == QP_TOTAL && !CreateSecBuff(dsmctrl, size))
            {
               result = FALSE;
            }
         }
         if (!result)
         {
            DestoryResources(dsmctrl);
         }
      }
   }
   return result;
}

static BOOLEAN DsmStart( H_DsmControl dsmctrl, U16BIT serviceId, U32BIT couId, U8BIT sikind )
{
   E_DscError err;
   BOOLEAN result = FALSE;

   dsmctrl->carousel = NULL;
   dsmctrl->restart = 0;
   dsmctrl->counter = 0;
   dsmctrl->sikind = sikind;
   dsmctrl->status = (U8BIT)BL_BOOTING;
   STB_OSMutexLock(dsmctrl->api.mutex);
   if (sikind == SIQUERY_SSU_PID)
   {
      err = CDSM_SysSetCurrService(dsmctrl->api.cldsm, dsmctrl->networkId, dsmctrl->transId, 0);
   }
   else
   {
      err = CDSM_SysSetCurrService(dsmctrl->api.cldsm, dsmctrl->networkId, dsmctrl->transId, serviceId);
   }
   if (err == CLDSM_OK)
   {
      err = CDSM_LoadCarousel( dsmctrl->api.cldsm, serviceId, couId,
            (E_SIQueryKind)sikind, &dsmctrl->carousel );
      if (err == CLDSM_OK)
      {
         DBGPRINT(DM_CONTROL, "OK %p", dsmctrl)
         result = TRUE;
         dsmctrl->status = (U8BIT)BL_LOADED;
      }
      else if (err != CLDSM_ERR_SI_QUERY_FAILED || couId != INVALID_CAROUSEL_ID)
      {
         ERRPRINT("err=%d", err)
         CDSM_SysSetCurrService( dsmctrl->api.cldsm, 0, 0, 0 );
         dsmctrl->status = (U8BIT)BL_FAILED;
      }
   }
   else
   {
      ERRPRINT("err=%d", err)
   }
   STB_OSMutexUnlock(dsmctrl->api.mutex);
   if (err == CLDSM_ERR_SI_QUERY_FAILED && couId == INVALID_CAROUSEL_ID)
   {
      U_PARAM th;
      DBGPRINT(DM_CONTROL, "Schedule re-start %p", dsmctrl)
      dsmctrl->status = (U16BIT)BL_FAILED;
      th.u16 = serviceId;
      StartTimer(dsmctrl, START_MAGIC, RESTART_TIME, th.ptr, &th.ptr);
      dsmctrl->restart = th.u32;
      result = TRUE;
   }
   return result;
}

static void DsmRestart( H_DsmControl dsmctrl, U16BIT serviceId )
{
   E_DscError err;
   U16BIT sid;
   dsmctrl->restart = 0;
   STB_OSMutexLock(dsmctrl->api.mutex);
   sid = CDSM_SysCurrServiceId(dsmctrl->api.cldsm);
   if (sid != serviceId)
   {
      err = CLDSM_ERR_NO_CURRENT_SERVICE_SET;
   }
   else if (CDSM_CurrentCarousel(dsmctrl->api.cldsm) != NULL)
   {
      err = CLDSM_ERR_CAROUSELS_STILL_LOADED;
   }
   else
   {
      err = CDSM_LoadCarousel( dsmctrl->api.cldsm, serviceId, INVALID_CAROUSEL_ID,
            SIQUERY_BOOT_CAROUSEL, &dsmctrl->carousel );
      if (err == CLDSM_OK)
      {
         DBGPRINT(DM_CONTROL, "srvId=%d", serviceId)
         dsmctrl->status = (U8BIT)BL_LOADED;
      }
      else if (err == CLDSM_ERR_SI_QUERY_FAILED)
      {
         DBGPRINT(DM_CONTROL, "srvId=%d err=%d", serviceId, err)
      }
      else
      {
         ERRPRINT("srvId=%d err=%d", serviceId, err)
      }
   }
   STB_OSMutexUnlock(dsmctrl->api.mutex);
   if (err == CLDSM_ERR_SI_QUERY_FAILED)
   {
      U_PARAM th;
      th.u16 = serviceId;
      StartTimer(dsmctrl, START_MAGIC, RESTART_TIME, th.ptr, &th.ptr);
      dsmctrl->restart = th.u32;
   }
}

static BOOLEAN DsiStart( H_DsmControl dsmctrl, U16BIT serviceId, U32BIT couId, U8BIT sikind )
{
   BOOLEAN result;

   DBGPRINT(DM_CONTROL, "%p sid=%d", dsmctrl, serviceId)

   ASSERT(dsmctrl->dsi);

   dsmctrl->serviceId = serviceId;
   dsmctrl->couId = couId;
   dsmctrl->sikind = sikind;
   dsmctrl->status = (U8BIT)BL_WAITING;
   result = DSI_Start( dsmctrl->dsi, dsmctrl->transId, dsmctrl->dmxref );
   if (result)
   {
      U32BIT count = 600;
      /* Wait for boot process to start loading carousel (for upto 3 secs) */
      while (dsmctrl->status == (U8BIT)BL_WAITING || dsmctrl->status == (U8BIT)BL_BOOTING)
      {
         STB_OSTaskDelay(5);
         count--;
         if (!count)
         {
            ERRPRINT("DSI Timeout")
            DSI_Stop( dsmctrl->dsi );
            break;
         }
      }
      result = (dsmctrl->status == (U8BIT)BL_LOADED) ? TRUE : FALSE;
   }
   return result;
}

#ifndef NDEBUG
/**
 * @brief   Set the DSMCC debug mask for instance.
 * @param   instance DSM control instance handle
 * @param   mask mask value
 * @return  void
 */
static void DebugInstanceSetMask(H_DsmControl dsmctrl, U32BIT mask)
{
   FUNCTION_START(DebugInstanceSetMask)
   if (dsmctrl->debugMask != mask)
   {
      ERRPRINT("dsmctrl %p; mask %x",dsmctrl,mask)
      dsmctrl->debugMask = mask;
      CDSM_SetDebugState(dsmctrl->api.cldsm, mask & DD_ALL);
   #if DSM_DP_LEVEL >= 2
      SFM_SetDebugState(dsmctrl->sfm, mask & DF_ALL);
      SIQ_SetDebugState(dsmctrl->siq, mask & DS_ALL);
   #else
      SFM_SetDebugConfig(dsmctrl->sfm,
         (F_Printf)STB_SPDebugNoCnWrite,
         (F_Printf)STB_SPDebugNoCnWrite,
         mask & DF_ALL);
      SIQ_SetDebugConfig(dsmctrl->siq,
         (F_Printf)STB_SPDebugNoCnWrite,
         (F_Printf)STB_SPDebugNoCnWrite,
         mask & DS_ALL);
   #endif
   }
   FUNCTION_FINISH(DebugInstanceSetMask)
}

U32BIT DebugInstanceGetMask(H_DsmControl dsmctrl)
{
   FUNCTION_START(DebugInstanceGetMask)
   FUNCTION_FINISH(DebugInstanceGetMask)
   return dsmctrl->debugMask;
}
#endif



/*------------------- Global function definitions ---------------------------*/

/**
 * @brief   Open DSM-CC and create controling instance.
 * @param   S_DsmccConfig *config Configuration of control instance.
 * @return  void
 */
H_DsmControl DSMCC_Open( S_DsmccConfig *config )
{
   S_DsmControl *dsmctrl;
   U16BIT i, msz, size;
   size = config->sectionBuffPoolSize;
   size *= sizeof(S_Message);
   for (i = 1, msz = 0; i != (QP_TOTAL + 1); i++)
   {
      msz += size * i;
   }
   dsmctrl = config->memAlloc(sizeof(S_DsmControl) + msz);
   if (dsmctrl != NULL)
   {
      memset(dsmctrl, 0, sizeof(S_DsmControl));
      dsmctrl->api.memAlloc = config->memAlloc;
      dsmctrl->api.memFree = config->memFree;
      if (!CreateResources(dsmctrl, config->sectionBuffPoolSize))
      {
         ERRPRINT("")
         config->memFree( dsmctrl );
         dsmctrl = NULL;
      }
      else if ((config->controlFlags & GET_PMT_DIRECT_FROM_HW_FLAG) &&
               !CreateDsi(dsmctrl, config))
      {
         ERRPRINT("")
         DestoryResources( dsmctrl );
         config->memFree( dsmctrl );
         dsmctrl = NULL;
      }
      else if (!CreateDsmcc(dsmctrl, config))
      {
         ERRPRINT("")
         if (config->controlFlags & GET_PMT_DIRECT_FROM_HW_FLAG)
         {
            DestroyDsi( dsmctrl );
         }
         else if (!UsingPmtsFromDvb())
         {
            STB_SIUnregisterPmtObserver( DvbPmtCallback );
         }
         DestoryResources( dsmctrl );
         config->memFree( dsmctrl );
         dsmctrl = NULL;
      }
      else
      {
         dsmctrl->task = STB_OSCreateTask( DsmMain, (void *)dsmctrl, MAIN_STACK_SIZE,
               (U8BIT)config->taskPriority, (U8BIT *)"DsmMain" );
         if (dsmctrl->task)
         {
            DBGPRINT(DM_CONTROL, "OK %p", dsmctrl)
            dsmctrl->next = instanceHead;
            instanceHead = dsmctrl;
         }
         else
         {
            ERRPRINT("")
            DestroyDsmcc( dsmctrl );
            DestoryResources( dsmctrl );
            config->memFree( dsmctrl );
            dsmctrl = NULL;
         }
      }
   }
   return dsmctrl;
}

/**
 * @brief   Close DSM-CC controling instance, and destroy all resources.
 * @param   dsmctrl DSM control instance handle
 * @return  void
 */
void DSMCC_Close(H_DsmControl dsmctrl)
{
   S_Message emsg;
   S_DsmControl **pDsmctrl;
   if (dsmctrl != NULL)
   {
      pDsmctrl = &instanceHead;
      while (*pDsmctrl != NULL)
      {
         if (*pDsmctrl == dsmctrl)
         {
            DBGPRINT(DM_CONTROL, "%p", dsmctrl)

            /* Stop Task */
            emsg.type = DSM_MSG_CLOSE_DOWN;
            while (!QueueWrite(dsmctrl->queues + QP_TOP, &emsg))
            {
               STB_OSTaskDelay(7);
            }
            STB_OSSemaphoreSignal(dsmctrl->q_count_sema);
            while (dsmctrl->task != NULL)
            {
               STB_OSTaskDelay(7);
            }
            *pDsmctrl = dsmctrl->next;
            DestroyDsmcc( dsmctrl );
            DestoryResources( dsmctrl );
            dsmctrl->api.memFree( dsmctrl );
            break;
         }
         pDsmctrl = &((*pDsmctrl)->next);
      }
   }
}

/**
 * @brief   Start DSM-CC controling instance. This will start search for a boot
 *          carousel on service specified by locator. The search depends on the
 *          value given 'boot_carousel_id' parameter.
 *          When boot carousel id is set to INVALID_CAROUSEL_ID, DSM-CC will
 *          attempt to start on a carousel which the client recognises in callback
 *          function 'parseDataBroadcastId'. Even when the PMT does not contain
 *          a recognised data broadcast id, this function will return TRUE, and the
 *          DSM-CC will keep trying to load a boot carousel.
 *          When boot carousel id is set to a specific value or UNKNOWN_CAROUSEL_ID
 *          DSMCC_Start() will return FALSE if a carousel with the required ID is
 *          not present.
 *          When boot carousel id is set to UNKNOWN_CAROUSEL_ID, DSM-CC will attempt
 *          to start on whatever carousel is signalled in PMT for that service.
 * @param   dsmctrl DSM control instance handle
 * @param   locator DVB location of service.
 * @param   boot_carousel_id carousel id - INVALID_CAROUSEL_ID,
 *          or UNKNOWN_CAROUSEL_ID, or a specific value.
 * @param   DMXREF dmxref demux reference to be used in section filter API
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_Start( H_DsmControl dsmctrl, S_DvbLocator *locator,
   U32BIT boot_carousel_id, DMXREF dmxref )
{
   DSMCC_SetTunedTransportInfo( dsmctrl, locator->original_network_id,
      locator->transport_stream_id, dmxref );
   return DSMCC_StartFs( dsmctrl, locator->service_id, boot_carousel_id );
}

/**
 * @brief   Inform DSMCC instance of the tuned transport details. This must be
 *          called before calling DSMCC_StartFs() or DSMCC_StartSsu()
 * @param   dsmctrl DSM control instance handle
 * @param   onet_id Original Network Id.
 * @param   tran_id Transport ID for the transport stream.
 * @param   DMXREF dmxref demux reference to be used in section filter API
 * @return  BOOLEAN - TRUE if success
 */
void DSMCC_SetTunedTransportInfo( H_DsmControl dsmctrl, U16BIT onet_id,
   U16BIT tran_id, DMXREF dmxref )
{
   DBGPRINT(DM_CONTROL, "%p onet=%x tran=%x dmx=%x",
      dsmctrl, onet_id, tran_id, dmxref)

   SFM_SetDemuxHandle( dsmctrl->sfm, dmxref );
   dsmctrl->dmxref = dmxref;
   dsmctrl->transId = tran_id;
   dsmctrl->networkId = onet_id;

   SIQ_TransportChanged( dsmctrl->siq, onet_id, tran_id );
}

/**
 * @brief   Start DSM-CC controlling instance. This must follow call to
 *          DSMCC_SetTunedTransportInfo.
 *          The function will start search for an Object Carousel on the specified
 *          service. The search depends on the value given 'carousel_id' parameter.
 *          When carousel id is set to INVALID_CAROUSEL_ID, DSM-CC will attempt
 *          to start on a carousel which the client recognises in callback
 *          function 'parseDataBroadcastId'. Even when the PMT does not contain
 *          a recognised data broadcast id, this function will return TRUE, and the
 *          DSM-CC will keep searching.
 *          When the carousel id is set to a specific value or UNKNOWN_CAROUSEL_ID
 *          and a carousel of the required ID is not present, either this function
 *          will return FALSE or 'carouselLoad' callback will be passed a result of
 *          OC_LOAD_ABORTED_ERROR.
 *          When carousel id is set to UNKNOWN_CAROUSEL_ID, DSM-CC will attempt
 *          to start on whatever carousel is signalled in PMT for that service.
 * @param   dsmctrl DSM control instance handle
 * @param   service_id Service Id.
 * @param   carousel_id Carousel Id
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_StartFs( H_DsmControl dsmctrl, U16BIT service_id, U32BIT carousel_id )
{
   BOOLEAN result;
   U8BIT sikind;
   DBGPRINT(DM_CONTROL, "%p sid=%d cid=%d", dsmctrl, service_id, carousel_id)
   switch (carousel_id)
   {
      case INVALID_CAROUSEL_ID:
         sikind = (U8BIT)SIQUERY_BOOT_CAROUSEL;
         break;

      case UNKNOWN_CAROUSEL_ID:
         sikind = (U8BIT)SIQUERY_FIND_CAROUSEL;
         break;

      default:
         sikind = (U8BIT)SIQUERY_CAROUSEL;
   }
   if (service_id == CDSM_SysCurrServiceId( dsmctrl->api.cldsm ))
   {
      ERRPRINT("%p already on service Id=%d", dsmctrl, service_id)
      result = FALSE;
   }
   else if (dsmctrl->dsi)
   {
      result = DsiStart( dsmctrl, service_id, carousel_id, sikind );
   }
   else
   {
      result = DsmStart( dsmctrl, service_id, carousel_id, sikind );
   }
   return result;
}

/**
 * @brief   Start DSM-CC controling instance. This must follow call to
 *          DSMCC_SetTunedTransportInfo.
 *          The function will start loading Update Carousel for SSU on specified service
 * @param   dsmctrl DSM control instance handle
 * @param   service_id Service Id.
 * @param   oui Manufacturer's OUI
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_StartSsu( H_DsmControl dsmctrl, U16BIT service_id, U32BIT oui )
{
   BOOLEAN result;

   DBGPRINT(DM_CONTROL, "%p sid=%d oui=%x", dsmctrl, service_id, oui)
   if (service_id == CDSM_SysCurrServiceId( dsmctrl->api.cldsm ))
   {
      ERRPRINT("%p already on service Id=%d", dsmctrl, service_id)
      result = FALSE;
   }
   else if (dsmctrl->dsi)
   {
      result = DsiStart( dsmctrl, service_id, oui, (U8BIT)SIQUERY_SSU_CAROUSEL );
   }
   else
   {
      result = DsmStart( dsmctrl, service_id, oui, (U8BIT)SIQUERY_SSU_CAROUSEL );
   }
   return result;
}

/**
 * @brief   Start DSM-CC controling instance. This must follow call to
 *          DSMCC_SetTunedTransportInfo. The function will start loading Update Carousel
 *          for SSU on a single specified PID.
 * @param   dsmctrl DSM control instance handle
 * @param   pid PID on which to find all SSU data
 * @param   oui Manufacturer's OUI
 * @return  BOOLEAN - TRUE if success
 */
BOOLEAN DSMCC_StartSsuWithPid( H_DsmControl dsmctrl, U16BIT pid, U32BIT oui )
{
   BOOLEAN result;

   DBGPRINT(DM_CONTROL, "%p pid=%d oui=%x", dsmctrl, pid, oui)
   if (dsmctrl->dsi)
   {
      result = DsiStart( dsmctrl, pid, oui, (U8BIT)SIQUERY_SSU_PID );
   }
   else
   {
      result = DsmStart( dsmctrl, pid, oui, (U8BIT)SIQUERY_SSU_PID );
   }
   return result;
}

/**
 * @brief   Stop DSM-CC controling instance. Causes all carousels to unload
 * @param   H_DsmControl  dsmctrl DSM control instance handle
 * @param   E_DsmRstMode mode Mode of operation when unloading
 * @return  void
 */
void DSMCC_Stop( H_DsmControl dsmctrl, E_DsmRstMode mode )
{
   E_DscError err;

   STB_OSMutexLock(dsmctrl->api.mutex);

   DBGPRINT(DM_CONTROL, "%p sid=%d boot_crsl=%p curr_crsl=%p", dsmctrl,
      CDSM_SysCurrServiceId(dsmctrl->api.cldsm),
      dsmctrl->carousel,
      CDSM_CurrentCarousel(dsmctrl->api.cldsm))
   if (dsmctrl->carousel != NULL)
   {
      //is dsmctrl->carousel == CDSM_CurrentCarousel(dsmctrl->api.cldsm) ?
      err = CDSM_UnloadCarousel( dsmctrl->api.cldsm, CDSM_CurrentCarousel(dsmctrl->api.cldsm), mode );
      if (err != CLDSM_OK)
      {
         ERRPRINT("err=%d", err)
      }
      dsmctrl->carousel = NULL;
   }
   if (dsmctrl->status != (U8BIT)BL_WAITING && dsmctrl->restart != 0)
   {
      StopTimer( dsmctrl, dsmctrl->restart );
      dsmctrl->restart = 0;
   }
   err = CDSM_SysSetCurrService( dsmctrl->api.cldsm, 0, 0, 0 );
   if (err != CLDSM_OK)
   {
      ERRPRINT("err=%d", err)
   }
   err = CDSM_SysReset( dsmctrl->api.cldsm, mode );
   if (err != CLDSM_OK)
   {
      ERRPRINT("err=%d", err)
   }
   STB_OSMutexUnlock(dsmctrl->api.mutex);

   dsmctrl->status = (U8BIT)BL_WAITING;
   if (dsmctrl->dsi)
   {
      DSI_Stop( dsmctrl->dsi );
   }
   SFM_CacheClear( dsmctrl->sfm );
}

/**
 * @brief   Reboot DSM-CC controling instance. Causes all carousels to unload and
 *          initiate search for boot carousel.
 * @param   H_DsmControl instance DSM control instance handle
 * @return  void
 */
void DSMCC_Reboot( H_DsmControl dsmctrl )
{
   H_ObjCarousel hOC;
   E_DscError err;
   U16BIT serviceId;

   if (dsmctrl->status != (U16BIT)BL_WAITING && dsmctrl->restart != 0)
   {
      StopTimer( dsmctrl, dsmctrl->restart );
      dsmctrl->restart = 0;
   }
   STB_OSMutexLock(dsmctrl->api.mutex);
   serviceId = CDSM_SysCurrServiceId( dsmctrl->api.cldsm );
   if (serviceId == 0)
   {
      err = CLDSM_ERR_NO_CURRENT_SERVICE_SET;
      ERRPRINT("%p serviceId=%d", dsmctrl, serviceId)
   }
   else
   {
      hOC = CDSM_CurrentCarousel(dsmctrl->api.cldsm);
      DBGPRINT(DM_CONTROL, "%p sid=%d c_crsl=%p", dsmctrl, serviceId, hOC)
      if (hOC != NULL)
      {
         err = CDSM_UnloadCarousel( dsmctrl->api.cldsm, hOC, RST_MODE_FORCE );
         if (err != CLDSM_OK)
         {
            ERRPRINT("err=%d", err)
         }
         SFM_CacheClear(dsmctrl->sfm);
      }
      err = CDSM_LoadCarousel( dsmctrl->api.cldsm, serviceId, INVALID_CAROUSEL_ID,
            SIQUERY_BOOT_CAROUSEL, &dsmctrl->carousel );
      if (err != CLDSM_OK)
      {
         ERRPRINT("err=%d", err)
      }
   }
   STB_OSMutexUnlock(dsmctrl->api.mutex);
   if (err == CLDSM_ERR_SI_QUERY_FAILED)
   {
      U_PARAM th;
      dsmctrl->status = (U16BIT)BL_FAILED;
      th.u16 = serviceId;
      StartTimer(dsmctrl, START_MAGIC, RESTART_TIME, th.ptr, &th.ptr);
      dsmctrl->restart = th.u32;
   }
}

/**
 * @brief   Get currently loaded carousel handle
 * @param   H_DsmControl  dsmctrl DSM control instance handle
 * @return  H_ObjCarousel carousel handle - NULL if none loaded
 */
H_ObjCarousel DSMCC_CurrentCarousel( H_DsmControl dsmctrl )
{
   H_ObjCarousel hOC;
   STB_OSMutexLock(dsmctrl->api.mutex);
   hOC = CDSM_CurrentCarousel(dsmctrl->api.cldsm);
   STB_OSMutexUnlock(dsmctrl->api.mutex);
   return hOC;
}

/**
 * @brief   Get currently loaded carousel ID
 * @param   H_DsmControl  dsmctrl DSM control instance handle
 * @return  U32BIT carousel id - INVALID_CAROUSEL_ID if none loaded
 */
U32BIT DSMCC_CurrentCarouselId( H_DsmControl dsmctrl )
{
   U32BIT cid = 0;
   STB_OSMutexLock(dsmctrl->api.mutex);
   if (CDSM_GetCarouselId( dsmctrl->api.cldsm, CDSM_CurrentCarousel(dsmctrl->api.cldsm), &cid ) != CLDSM_OK)
   {
      ERRPRINT("")
   }
   STB_OSMutexUnlock(dsmctrl->api.mutex);
   return cid;
}

/**
 * @brief   Set current carousel.
 *          This does not unload any previously loaded carousels
 * @param   H_DsmControl  dsmctrl DSM control instance handle
 * @param   H_ObjCarousel hOC Handle to carousel
 * @return  BOOLEAN - TRUE if success (i.e. had valid carousel handle)
 */
BOOLEAN DSMCC_SetCurrentCarousel(H_DsmControl dsmctrl, H_ObjCarousel hOC)
{
   BOOLEAN result;
   STB_OSMutexLock(dsmctrl->api.mutex);
   if (CDSM_SetCurrentCarousel( dsmctrl->api.cldsm, hOC ) == CLDSM_OK)
   {
      DBGPRINT(DM_CONTROL, "%p c_crsl=%p", dsmctrl, hOC)
      result = TRUE;
   }
   else
   {
      ERRPRINT("")
      result = FALSE;
   }
   STB_OSMutexUnlock(dsmctrl->api.mutex);
   return result;
}

/**
 * @brief   Unload specified Carousel
 * @param   H_DsmControl  dsmctrl DSM control instance handle
 * @param   H_ObjCarousel hOC Handle to carousel
 * @param   E_DsmRstMode mode Mode of operation when unloading
 * @return  void
 */
void DSMCC_UnloadCarousel(H_DsmControl dsmctrl, H_ObjCarousel hOC, E_DsmRstMode mode)
{
   E_DscError err;
   DBGPRINT(DM_CONTROL, "crsl=%p mode=%d", hOC, mode)
   STB_OSMutexLock(dsmctrl->api.mutex);
   err = CDSM_UnloadCarousel( dsmctrl->api.cldsm, hOC, mode );
   STB_OSMutexUnlock(dsmctrl->api.mutex);
   if (err != CLDSM_OK)
   {
      ERRPRINT("Unload Carousel err %d", err)
   }
   else
   {
      SFM_CacheClear(dsmctrl->sfm);
   }
}

/**
 * @brief   Unload SSU module. To be called once the client has finished saving
 *          this modules data. This is needed to
 * @param   dsmctrl DSM control instance handle.
 * @param   hCarousel Handle to the DSMCC Update Carousel
 * @param   moduleRef Reference to a module
 * @return  void
 */
void DSMCC_SsuUnloadModule( H_DsmControl dsmctrl, H_DsmCarousel hCarousel,
   U32BIT moduleRef )
{
   E_DscError err;
   DBGPRINT(DM_CONTROL, "crsl=%p", hCarousel)
   STB_OSMutexLock(dsmctrl->api.mutex);
   err = CDSM_UnloadModule( dsmctrl->api.cldsm, hCarousel, moduleRef );
   STB_OSMutexUnlock(dsmctrl->api.mutex);
   if (err != CLDSM_OK)
   {
      ERRPRINT("Unload Carousel err %d", err)
   }
}


/**
 * @brief   Set limit of DSMCC dynamic memory usage for file system caching
 *          This function overrides the default size is 6 MB.
 *          If this size is less than the currently allocated memory size,
 *          then it will release file system data until limit is reached - but
 *          only releasing data not currently in used by the client.
 * @param   dsmctrl DSM control instance handle.
 * @param   cacheSize Size in bytes (minimum of 1 MB)
 * @return  void
 */
void DSMCC_SetFileSystemCacheLimit( H_DsmControl dsmctrl, U32BIT cacheSize )
{
   STB_OSMutexLock(dsmctrl->api.mutex);
   CDSM_SysSetMemoryMax( dsmctrl->api.cldsm, cacheSize );
   STB_OSMutexUnlock(dsmctrl->api.mutex);
}

/**
 * @brief   Clear cached PMT for particular service
 * @param   H_DsmControl   dsmctrl        DSM control instance handle.
 * @param   U16BIT         serviceId      Service whose PMT needs to be cleared
 * @return  void
 */
void DSMCC_SiqCacheClearPmt( H_DsmControl dsmctrl, U16BIT serviceId )
{
   DBGPRINT(DM_CONTROL, "sid=%d", serviceId)
   STB_OSMutexLock(dsmctrl->api.mutex);
   SIQ_CacheClearPmt( dsmctrl->siq, serviceId );
   STB_OSMutexUnlock(dsmctrl->api.mutex);
}

/**
 * @brief   Returns version major.minor.patch as string
 * @return  a version string
 */
const char* DSMCC_VersionString(void)
{
   return DSMCC_VER_STRING;
}

/**
 * @brief   Find DSMCC instance that has beed started on specified service
 * @param   serviceId Service on which DSMCC is started
 * @param   dmxref demux reference used for HW section filtering
 * @return  H_DsmControl instance handle, or NULL if not found
 */
H_DsmControl DSMCC_FindInstance( U16BIT serviceId, DMXREF dmxref  )
{
   S_DsmControl *dsmctrl;
   FUNCTION_START(DSMCC_FindInstance)
   dsmctrl = instanceHead;
   while (dsmctrl != NULL)
   {
      if (dsmctrl->dmxref == dmxref && CDSM_SysCurrServiceId(dsmctrl->api.cldsm) == serviceId)
      {
         break;
      }
      dsmctrl = dsmctrl->next;
   }
   FUNCTION_FINISH(DSMCC_FindInstance)
   return dsmctrl;
}

/**
 * @brief    Get the DSMCC debug mask.
 * @return   Debug mask
 */
U32BIT DSMCC_DebugGetMask(void)
{
#ifndef NDEBUG
   return gDebugState;
#else
   return 0;
#endif
}

/**
 * @brief   Set the DSMCC debug mask.
 * @param   mask mask value
 * @return  void
 */
void DSMCC_DebugSetMask(U32BIT mask)
{
#ifndef NDEBUG
   S_DsmControl *dsmctrl;
   FUNCTION_START(DSMCC_DebugSetMask)

   gDebugState = mask;
   CDSM_SetPrintState( gDebugState & DD_ALL );

   dsmctrl = instanceHead;
   while (dsmctrl != NULL)
   {
      DebugInstanceSetMask(dsmctrl, mask);
      dsmctrl = dsmctrl->next;
   }
   FUNCTION_FINISH(DSMCC_DebugSetMask)
#else
   USE_UNWANTED_PARAM(mask);
#endif
}


/**
 * @brief   Get the DSMCC debug mask for instance.
 * @param   dsmctrl DSM control instance handle
 * @return  Debug mask
 */
U32BIT DSMCC_DebugInstanceGetMask(H_DsmControl instance)
{
#ifndef NDEBUG
   S_DsmControl *dsmctrl;
   dsmctrl = instanceHead;
   while (dsmctrl != NULL)
   {
      if (dsmctrl == instance)
      {
         return dsmctrl->debugMask;
      }
      dsmctrl = dsmctrl->next;
   }
#else
   USE_UNWANTED_PARAM(instance);
#endif
   return 0;
}

/**
 * @brief   Set the DSMCC debug mask.
 * @param   instance DSM control instance handle
 * @param   mask mask value
 * @return  void
 */
void DSMCC_DebugInstanceSetMask(H_DsmControl instance, U32BIT mask)
{
#ifndef NDEBUG
   S_DsmControl *dsmctrl;
   FUNCTION_START(DSMCC_DebugInstanceSetMask)

   ERRPRINT("(instance %p, mask %x)",instance, mask)
   dsmctrl = instanceHead;
   while (dsmctrl != NULL)
   {
      if (dsmctrl == instance)
      {
         DebugInstanceSetMask(dsmctrl, mask);
         break;
      }
      dsmctrl = dsmctrl->next;
   }
   FUNCTION_FINISH(DSMCC_DebugInstanceSetMask)
#else
   USE_UNWANTED_PARAM(instance);
   USE_UNWANTED_PARAM(mask);
#endif
}


/**
 * @brief   Enable DSMCC debug for an instance, by reading environment variables
 *          with name format: <rootname>_<maskname>. E.g. DSMOTA_DS_PMT
 * @param   instance DSM control instance handle
 * @param   rootname Root name for environment variables for this instance
 * @return  void
 */
void DSMCC_DebugInstanceEnable(H_DsmControl dsmctrl, const char *rootname)
{
#if defined(NDEBUG) && defined(DEBUG_USE_GETENV)
   char envname[64];
   char *mskptr;
   char *value;
   U32BIT mask;

   strcpy(envname,rootname);
   mskptr = envname + strlen(rootname);

#ifndef NDEBUG
   mask = dsmctrl->debugMask;
#endif
   /* Query environment variables "<rootname>_<maskname>" */
   GETENV_DSMDBG(DD_GEN    )
   GETENV_DSMDBG(DD_OC     )
   GETENV_DSMDBG(DD_SF     )
   GETENV_DSMDBG(DD_TIMER  )
   GETENV_DSMDBG(DD_SE     )
   GETENV_DSMDBG(DD_LM     )
   GETENV_DSMDBG(DD_OBJ    )
   GETENV_DSMDBG(DD_DC     )
   GETENV_DSMDBG(DD_QUERY  )
   GETENV_DSMDBG(DD_MOD    )
   GETENV_DSMDBG(DD_ALL    )
   GETENV_DSMDBG(DS_MAIN   )
   GETENV_DSMDBG(DS_QUERY  )
   GETENV_DSMDBG(DS_CACHE  )
   GETENV_DSMDBG(DS_PMT    )
   GETENV_DSMDBG(DS_PAT    )
   GETENV_DSMDBG(DS_FUNC   )
   GETENV_DSMDBG(DS_ALL    )
   GETENV_DSMDBG(DF_MAIN   )
   GETENV_DSMDBG(DF_FILTER )
   GETENV_DSMDBG(DF_CACHE  )
   GETENV_DSMDBG(DF_HWSF   )
   GETENV_DSMDBG(DF_ALL    )
   GETENV_DSMDBG(DM_SI     )
   GETENV_DSMDBG(DM_CONTROL)
   GETENV_DSMDBG(DM_CLIENT )
   DBGPRINT(DM_CONTROL, "mask=%x",mask)
   DebugInstanceSetMask(dsmctrl, mask);
#else  /*DEBUG_USE_GETENV*/
   USE_UNWANTED_PARAM(dsmctrl);
   USE_UNWANTED_PARAM(rootname);
#endif /*DEBUG_USE_GETENV*/
}

